/* 
 * File:   SPECTRAL_MIDI.h
 * Author: Jeremy Leach 2019
 */

#ifndef SPECTRAL_MIDI_H
#define	SPECTRAL_MIDI_H

#ifdef	__cplusplus 
extern "C" {
#endif

#include "mcc_generated_files/mcc.h"
#include "SPECTRAL_General.h"
    
/******************************************************************************/
/***** Constants **************************************************************/
/******************************************************************************/
#define max_midi_channels 6

#define max_tone_processors 6
#define instances_per_tone_processor 3

#define midi_sense_delay_hundreths 1
  
#define midi_buffer_size 64
#define midi_buffer_total_size 68
#define max_midi_buffer_count 63
    
#define iat_index_NULL 255
#define volume_update_10ms 10

#define patches_per_patchset 18
#define performances_per_patchset 8
#define performance_size 31
#define msg_performance_size 34
#define msg_select_performance_size 3

/******************************************************************************/
/***** Enumerations ***********************************************************/
/******************************************************************************/
 enum midi_statuses
{
    STATUS_note_off            = 0x80, //Note Off.              Expected bytes = 2.
    STATUS_note_on             = 0x90, //Note On.               Expected bytes = 2.
    STATUS_aftertouch          = 0xa0, //After touch.           Expected bytes = 2.
    STATUS_control_change      = 0xb0, //Control Change.        Expected bytes = 2.
    STATUS_program_change      = 0xc0, //Program Change.        Expected bytes = 1.
    STATUS_sysex_start         = 0xf0, //Start of system message.
    STATUS_sysex_end           = 0xf7, //End of system message.
    STATUS_channel_aftertouch  = 0xd0, //Channel After touch(pressure).   Expected bytes = 1.
    STATUS_pitch_bend          = 0xe0, //Pitch bend
    STATUS_active_sensing      = 0xfe, //Active sensing.
};

enum midi_cc
{
   cc_bank_MSB = 0,
   cc_modulation = 1,
   cc_breath_controller = 2,
   cc_foot_controller = 4,
   cc_portamento_time = 5,
   cc_channel_volume = 7,
   cc_expression = 11,
   cc_effect_controller1 = 12,
   cc_effect_controller2 = 13,
   cc_general_purpose16 = 16,
   cc_general_purpose17 = 17,
   cc_bank_LSB = 32,
   cc_sustain_pedal = 64,
   cc_portamento_onoff = 65,
   cc_sound_controller1 = 70,
   cc_sound_controller2 = 71,
   cc_portamento_amount = 84,
   cc_effect1_depth = 91,
   cc_effect2_depth = 92,
   cc_effect3_depth = 93,
   cc_effect4_depth = 94,
   cc_effect5_depth = 95,
   cc_all_notes_off = 123
};

/******************************************************************************/
/***** Types ******************************************************************/
/******************************************************************************/
typedef union
{    
    uint8_t data[5];
    struct
    {    
    uint8_t status;
    uint8_t data1;
    uint8_t data2;
    };
}midi_event;

typedef union
{   
uint8_t data[midi_buffer_total_size];
struct  {
        uint8_t buf[midi_buffer_size];
        uint8_t head;
        uint8_t tail;
        uint8_t midi_running_status; 
        uint8_t count;
        };
} fifo_midi_buffer;

typedef struct
{
    uint8_t midi_channel;
    uint8_t note_number;
    uint8_t flag_held_on_by_sustain;
    uint8_t allocation_id;
} tone_instance_allocation;

typedef struct
{
    uint8_t processor_id;
    uint8_t instance_id;
    uint8_t iat_index; 
    uint8_t flag_null;
} tone_instance;

typedef struct  {
    uint8_t flag_sustain_on;
    uint8_t midi_channel_patch_setting;
    uint8_t last_midi_note_number;
    uint8_t total_load;
    uint8_t max_instances;
} channel_info;

typedef struct { 
    uint8_t module_patch_id;
    uint8_t volume;
    uint8_t note_range_low;
    uint8_t note_range_high; 
} performance_channel_t;

typedef union
{
uint8_t data[performance_size];
struct {
    performance_channel_t pc[max_midi_channels];
    uint8_t haas_delay_ms;   
    uint8_t processor_channel_allocation[max_tone_processors];
};
} performance_t;

typedef union 
{
uint8_t data[msg_performance_size];
struct  
{
uint8_t message_recipient;
uint8_t message_type;
uint8_t performance_id;
performance_t performance;
};
} msg_performance;

typedef union 
{
uint8_t data[3];
struct  
{
uint8_t message_recipient;
uint8_t message_type;
uint8_t performance_id;
};
} msg_select_performance;

/******************************************************************************/
/***** Methods ****************************************************************/
/******************************************************************************/  
    uint8_t GetPatchForChannel(uint8_t midi_channel);
    uint8_t get_channel_max_instances(uint8_t c,uint8_t for_performance_id);
    void SPECTRAL_MIDI_Initialise(uint8_t for_performance_id);
    tone_instance FindAllocatedToneInstance(uint8_t midi_channel,uint8_t note_number);
    tone_instance AllocateToneInstance(uint8_t midi_channel,uint8_t note_number);
    uint8_t fMidiBufferHasData(volatile fifo_midi_buffer *f);
    uint8_t ReadByteFromMidiBuffer(volatile fifo_midi_buffer *f);
    uint8_t WaitForMidiByte();
    void Get2EventDataBytes(midi_event *e,uint8_t first_byte);
    void Get1EventDataBytes(midi_event *e,uint8_t first_byte);
    void ProcessNoteOrSustain(midi_event e,uint8_t status_without_channel,uint8_t midi_channel);
    void ProcessReceivedMidi();
    void Rx_MIDI_Event_InterruptHandler();
    void ProcessMIDI();
    void MIDI_TMR1_InterruptHandler();
            
#ifdef	__cplusplus 
}
#endif

#endif	/* SPECTRAL_MIDI_H */

