﻿using Spectral1.DATA_ACCESS;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;

namespace Spectral1.PARSER
{
    public partial class c_lexer
    {
        #region "====================== DECLARATIONS ============================"
        List<c_lexer_token_definition> _tokenDefinitions;

        public enum TokenType
        {
            Comment,
            NotDefined,
            OpenParenthesis,
            HarmonicPhaseShapeType,
            OpenCurlyParenthesis,
            CloseParenthesis,
            CloseCurlyParenthesis,
            Comma,
            Dot,
            Colon,
            SemiColon,
            Equals,
            NotEquals,
            Integer,
            Percent,
            Decimal,
            NoteValue,
            And,
            Or,
            SequenceTerminator,
            Define,
            //------------------
            All,
            //-------------------
            Slope,
            dbplus3,
            dbplus6,
            dbplus12,
            dbplus24,
            dbminus3,
            dbminus6,
            dbminus12,
            dbminus24,
            //------------------
            Direction,
            Up,
            Down,
            //------------------
            WaveSet,
            NoteSector,
            CurrentNoteSector,
            NoteSectorFrom,
            NoteSectorTo,
            IntensityLayer,
            CurrentIntensityLayer,
            IntensityLayerFrom,
            IntensityLayerTo,
            CurrentWaveformBlock,
            Waveform,
            CurrentWaveform,
            WaveformFrom,
            WaveformTo,
            WaveformBlock,
            WaveformName,
            NameString,
            HarmonicID,
            HarmonicIDFrom,
            HarmonicIDTo,
            //-----------------
            CopyTo,
            CopyToRange,
            CopyAcrossNoteSectors,
            CopyAcrossIntensityLayers,
            CopyAcrossAllDimensions,
            //-----------------
            MorphAcrossNoteSectors,
            MorphAcrossIntensityLayers,
            MorphWaveformBlocks,
            MorphWaveforms,
            MorphHarmonicLevels,
            //-----------------
            ShapeTheHarmonics,
            //-----------------
            SetCurrentIntensityLayer,
            SetCurrentNoteSector,
            SetCurrentWaveform,
            SetName,
            SetHarmonicLevels,
            SetHarmonicLevelsFromCSV,
            SetHarmonicLevelsFromInstrumentType,
            SetHarmonicLevelsFromLevelShapeType,
            SetHarmonicLevelsFromWaveFormShape,
            SetHarmonicLevel,
            SetHarmonicPhases,
            SetHarmonicPhasesFromCSV,
            SetHarmonicPhasesFromInstrumentType,
            SetHarmonicPhasesFromWaveFormShape,
            SetHarmonicPhasesFromPhaseShapeType,
            SetHarmonicPhase,
            //--------------------------
            MorphType,
            Linear,
            Exponential,
            //--------------------------
            HarmonicLevelShapeType,
            AllZero,
            AllMax,
            LinearDecreasing,
            ExponentialAll,
            ExponentialOdd,
            //--------------------------
            Phase,
            Level,
            LevelCSV,
            PhasesCSV,
            Inverted,
            AllInPhase,
            InvertedEven,
            InvertedOdd,
            Inverted3711,
            //---------------------------
            WaveformShape,
            Triangle,
            Square,
            Sine,
            Saw,
            //---------------------------
            InstrumentType,
            Trumpet,
            Violin,
            Flute,
            Cello,
            Guitar,
            MutedGuitar,
            Clarinet,
            //==================================
            Patch,
            General,
            ADSR,
            //-----------------------------------
            SetPortamento,
            SetScalingSplit,
            SetPatchGain,
            SetDetuning,
            SetActiveOscillators,
            SetDetuningMode,
            SetOscillatorDetuning,
            SetLFO,
            SetEnvelopeGainController,
            ConfigureSection,
            ConfigureEnvelope,
            //-----------------------------------
            Enabled,
            PortamentoAmount,
            Note,
            Gain,
            DetuningType,
            Detuning,
            OscillatorCount,
            DetuningMode,
            Oscillator,
            LFOType,
            WaveType,
            Frequency,
            FrequencyCC,
            DepthCC,
            Envelope,
            InitialLevel,
            GainCC,
            Section,
            Duration,
            EndKSU,
            EndKSL,
            Sample,
            SampleMode,
            EnvelopeType,
            Target,
            TargetKSU,
            TargetKSL,
            LinearDelta,
            LinearDeltaKSU,
            LinearDeltaKSL,
            ExpMult,
            ExpMultKSU,
            ExpMultKSL,

            //--------------------------------
            Random,
            Regular,

            True,
            False,

            Cents,
            Hz,

            Tremolo,
            Vibrato,
            TimbreLFO,

            Modulation,
            BreathController,
            FootController,
            Expression,
            Velocity,
            None,

            Attack,
            Decay,
            Sustain,
            Release,

            KeyHammer1,
            KeyHammer2,
            KeyHammer3,
            KeyHammer4,
            BowScrape1,
            BowScrape2,
            KeyClick,
            KeyUp,
            KSVLong,
            KSLong,
            KSStd,
            KSShort,
            KSVShort,
            KSVLongF,
            KSLongF,
            KSStdF,
            KSShortF,
            KSVShortF,

            OneShot,
            Looped,

            Amplitude,
            NoiseGain,
            NoiseCutoffFrequency,
            TimbreMorph,
            SampleGain,
            PitchShift,
            TremoloDepth,
            VibratoDepth,
            TimbreLFODepth,

            LoadPreset,
            PresetName,
            BulkValue,

            //==================================
            amplitude_HammeredString,
            amplitude_SlowAttack,
            amplitude_VerySlowAttack,
            amplitude_Organlike,
            tremolo_DeepAndSlow,
            tremolo_ShallowAndSlow,
            tremolo_DeepAndFast,
            tremolo_ShallowAndFast,
            tremolo_BulkSetLinearDSR,
            tremolo_BulkSetExpDSR,
            tremolo_Clear,
            vibrato_DeepAndSlow,
            vibrato_ShallowAndSlow,
            vibrato_DeepAndFast,
            vibrato_ShallowAndFast,
            vibrato_BulkSetLinearADSR,
            vibrato_BulkSetExpADSR,
            vibrato_Clear,
            timbre_FastSweepUp,
            timbre_SlowSweepUp,
            timbre_ExpVeryFastSweepUp,
            timbre_FastSweepDown,
            timbre_SlowSweepDown,
            timbre_ExpVeryFastSweepDown,
            timbre_ExpFastSweepUp,
            timbre_ExpSlowSweepUp,
            timbre_ExpFastSweepDown,
            timbre_ExpSlowSweepDown,
            timbre_BulkSetLinearADSR,
            timbre_BulkSetExpADSR,
            timbre_Clear,
            noise_QuickSoftPuff,
            noise_SlowSoftPuff,
            noise_QuickHardPuff,
            noise_SlowHardPuff,
            noise_Clear,
            noise_co_apply_lpf_exp_fast,
            noise_co_apply_lpf_exp_slow,
            noise_co_BulkSetLinearADSR,
            noise_co_BulkSetExpADSR,
            noise_co_Clear,
            pitch_fast_linear_drop,
            pitch_slow_linear_drop,
            pitch_fast_linear_rise,
            pitch_slow_linear_rise,
            pitch_fast_exp_drop,
            pitch_slow_exp_drop,
            pitch_fast_exp_rise,
            pitch_slow_exp_rise,
            pitch_BulkSetLinearADSR,
            pitch_BulkSetExpADSR,
            pitch_Clear,
            sample_AttackOnLevel100,
            sample_AttackOnLevel75,
            sample_AttackOnLevel50,
            sample_AttackOnLevel25,
            sample_SustainOnLevel25,
            sample_ReleaseOnLevel25,
            sample_AttackOnLevel10,
            sample_SustainOnLevel10,
            sample_ReleaseOnLevel10,
            sample_AttackOnLevel5,
            sample_SustainOnLevel5,
            sample_ReleaseOnLevel5,
            sample_KS_100,
            sample_KS_75,
            sample_KS_50,
            sample_KS_25,
            sample_Clear,
            timbre_lfo_DeepAndSlow,
            timbre_lfo_ShallowAndSlow,
            timbre_lfo_DeepAndFast,
            timbre_lfo_ShallowAndFast,
            timbre_lfo_BulkSetLinearDSR,
            timbre_lfo_BulkSetExpDSR,
            timbre_lfo_Clear,

            //==================================
            Filter

        }
        #endregion

        #region "====================== METHODS - GENERAL ============================"
        public c_lexer()
        {
            initialise_tokenizer();
        }

        public List<c_lexer_token> Tokenize(ref RichTextBox rtb)
        {
            string lqlText = rtb.Text;
            var tokens = new List<c_lexer_token>();
            string remainingText = lqlText;
            int rtb_char_index = 0;
            int rtb_match_start;
            int rtb_match_end;

            while (!string.IsNullOrWhiteSpace(remainingText))
            {
                var match = find_match(remainingText);
                if (match.IsMatch)
                {
                    
                    rtb_match_start = rtb.Find(match.Value, rtb_char_index, RichTextBoxFinds.None);
                    c_lexer_token new_token = new c_lexer_token(match.TokenType, match.Value, rtb_char_index);

                    tokens.Add(new_token);


                    if (rtb_match_start > -1)
                    {
                        rtb_match_end = rtb_match_start + match.Value.Length - 1;
                        rtb.Select(rtb_match_start, rtb_match_end - rtb_match_start + 1);

                        if (is_name_string(new_token))
                        {
                            rtb.SelectionColor = System.Drawing.Color.Yellow;
                        }
                        else if (new_token.TokenType == TokenType.Comment)
                        {
                            rtb.SelectionColor = System.Drawing.Color.Salmon;
                        }
                        else if (is_object_token(new_token))
                        {
                            rtb.SelectionColor = System.Drawing.Color.LawnGreen;
                        }
                        else if (is_function_token(new_token))
                        {
                            rtb.SelectionColor = System.Drawing.Color.Orange;
                        }
                        else if (is_parameter_name_token(new_token))
                        {
                            rtb.SelectionColor = System.Drawing.Color.Turquoise;
                        }
                        else
                        {
                            rtb.SelectionColor = System.Drawing.Color.WhiteSmoke;
                        }

                        rtb_char_index = rtb_match_end + 1;
                    }


                    remainingText = match.RemainingText;
                }
                else
                {
                    remainingText = remainingText.Substring(1);
                }
            }

            tokens.Add(new c_lexer_token(TokenType.SequenceTerminator, string.Empty,0));

            return tokens;
        }
        #endregion

        #region "====================== METHODS - TOKEN LOGIC ===================================="
        public bool is_name_string(c_lexer_token token)
        {
            return token.TokenType == TokenType.NameString;
        }

        public bool is_definition_object_token(c_lexer_token token)
        {
            return token.TokenType == TokenType.WaveSet
                   || token.TokenType == TokenType.Patch
                   || token.TokenType == TokenType.Filter;
        }

        public bool is_object_token(c_lexer_token token)
        {
            return is_definition_object_token(token) ||
                is_waveset_definition_object_token(token) ||
                is_patch_definition_object_token(token);
        }

        public bool is_parameter_name_token(c_lexer_token token)
        {
            return token.TokenType == TokenType.HarmonicID
                   || token.TokenType == TokenType.HarmonicIDFrom
                   || token.TokenType == TokenType.HarmonicIDTo
                   || token.TokenType == TokenType.HarmonicLevelShapeType
                   || token.TokenType == TokenType.HarmonicPhaseShapeType
                   || token.TokenType == TokenType.InstrumentType
                   || token.TokenType == TokenType.IntensityLayer
                   || token.TokenType == TokenType.IntensityLayerFrom
                   || token.TokenType == TokenType.IntensityLayerTo
                   || token.TokenType == TokenType.Level
                   || token.TokenType == TokenType.LevelCSV
                   || token.TokenType == TokenType.MorphType
                   || token.TokenType == TokenType.NoteSector
                   || token.TokenType == TokenType.NoteSectorFrom
                   || token.TokenType == TokenType.NoteSectorTo
                   || token.TokenType == TokenType.Phase
                   || token.TokenType == TokenType.PhasesCSV
                   || token.TokenType == TokenType.Slope
                   || token.TokenType == TokenType.Waveform
                   || token.TokenType == TokenType.WaveformFrom
                   || token.TokenType == TokenType.WaveformTo
                   || token.TokenType == TokenType.WaveformName
                   || token.TokenType == TokenType.WaveformShape
                   || token.TokenType == TokenType.Enabled
                    || token.TokenType == TokenType.PortamentoAmount
                    || token.TokenType == TokenType.Note
                    || token.TokenType == TokenType.Gain
                    || token.TokenType == TokenType.DetuningType
                    || token.TokenType == TokenType.Detuning
                    || token.TokenType == TokenType.OscillatorCount
                    || token.TokenType == TokenType.DetuningMode
                    || token.TokenType == TokenType.Oscillator
                    || token.TokenType == TokenType.LFOType
                    || token.TokenType == TokenType.WaveType
                    || token.TokenType == TokenType.Frequency
                    || token.TokenType == TokenType.FrequencyCC
                    || token.TokenType == TokenType.DepthCC
                    || token.TokenType == TokenType.Envelope
                    || token.TokenType == TokenType.InitialLevel
                    || token.TokenType == TokenType.GainCC
                    || token.TokenType == TokenType.Section
                    || token.TokenType == TokenType.Duration
                    || token.TokenType == TokenType.EndKSU
                    || token.TokenType == TokenType.EndKSL
                    || token.TokenType == TokenType.Sample
                    || token.TokenType == TokenType.SampleMode
                    || token.TokenType == TokenType.EnvelopeType
                    || token.TokenType == TokenType.Target
                    || token.TokenType == TokenType.TargetKSU
                    || token.TokenType == TokenType.TargetKSL
                    || token.TokenType == TokenType.LinearDelta
                    || token.TokenType == TokenType.LinearDeltaKSU
                    || token.TokenType == TokenType.LinearDeltaKSL
                    || token.TokenType == TokenType.ExpMult
                    || token.TokenType == TokenType.ExpMultKSU
                    || token.TokenType == TokenType.ExpMultKSL 
                    || token.TokenType == TokenType.PresetName
                    || token.TokenType == TokenType.BulkValue
                    ;
        }

        public static bool is_function_token(c_lexer_token token)
        {
            return is_waveset_function_token(token) ||
                            is_intensity_layer_function_token(token) ||
                            is_note_sector_function_token(token) ||
                            is_waveform_block_function_token(token) ||
                            is_waveform_function_token(token)||
                            //--------------------------
                            is_patch_general_function_token(token)||
                            is_patch_adsr_function_token(token);
        }

        public static bool is_waveform_shape_type_token(c_lexer_token token)
        {
            return token.TokenType == TokenType.Triangle ||
                token.TokenType == TokenType.Square ||
                token.TokenType == TokenType.Saw||
                token.TokenType == TokenType.Sine
                ;

        }

        #endregion


        #region "========================== PRIVATE METHODS ============================"
        private c_lexer_token_match find_match(string lqlText)
        {
            foreach (var tokenDefinition in _tokenDefinitions)
            {
                var match = tokenDefinition.Match(lqlText);
                if (match.IsMatch)
                    return match;
            }

            return new c_lexer_token_match() { IsMatch = false };
        }


        private void initialise_tokenizer()
        {
            _tokenDefinitions = new List<PARSER.c_lexer_token_definition>();

            //-------------------------------
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.Comment, @"^(//((?!\*/).)*)(?!\*/)[^\r\n]"));

            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.OpenParenthesis, "^\\("));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.CloseParenthesis, "^\\)"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.OpenCurlyParenthesis, "^\\{"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.CloseCurlyParenthesis, "^\\}"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.Comma, "^,"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.Dot, "^\\."));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.Colon, "^:"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.SemiColon, "^;"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.Equals, "^equals"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.NotEquals, "^notequals"));

            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.dbplus3, "^+3db"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.dbplus6, "^+6db"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.dbplus12, "^+12db"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.dbplus24, "^+24db"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.dbminus3, "^-3db"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.dbminus6, "^-6db"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.dbminus12, "^-12db"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.dbminus24, "^-24db"));

            //=====================================================================
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.amplitude_SlowAttack, "^amplitude_HammeredString"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.amplitude_SlowAttack, "^amplitude_SlowAttack"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.amplitude_VerySlowAttack, "^amplitude_VerySlowAttack"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.amplitude_Organlike, "^amplitude_Organlike"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.tremolo_DeepAndSlow, "^tremolo_DeepAndSlow"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.tremolo_ShallowAndSlow, "^tremolo_ShallowAndSlow"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.tremolo_DeepAndFast, "^tremolo_DeepAndFast"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.tremolo_ShallowAndFast, "^tremolo_ShallowAndFast"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.tremolo_BulkSetLinearDSR, "^tremolo_BulkSetLinearDSR"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.tremolo_BulkSetExpDSR, "^tremolo_BulkSetExpDSR"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.tremolo_Clear, "^tremolo_Clear"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.vibrato_DeepAndSlow, "^vibrato_DeepAndSlow"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.vibrato_ShallowAndSlow, "^vibrato_ShallowAndSlow"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.vibrato_DeepAndFast, "^vibrato_DeepAndFast"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.vibrato_ShallowAndFast, "^vibrato_ShallowAndFast"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.vibrato_BulkSetLinearADSR, "^vibrato_BulkSetLinearADSR"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.vibrato_BulkSetExpADSR, "^vibrato_BulkSetExpADSR"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.vibrato_Clear, "^vibrato_Clear"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.timbre_FastSweepUp, "^timbre_FastSweepUp"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.timbre_SlowSweepUp, "^timbre_SlowSweepUp"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.timbre_ExpVeryFastSweepUp, "^timbre_ExpVeryFastSweepUp"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.timbre_FastSweepDown, "^timbre_FastSweepDown"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.timbre_SlowSweepDown, "^timbre_SlowSweepDown"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.timbre_ExpVeryFastSweepDown, "^timbre_ExpVeryFastSweepDown"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.timbre_ExpFastSweepUp, "^timbre_ExpFastSweepUp"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.timbre_ExpSlowSweepUp, "^timbre_ExpSlowSweepUp"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.timbre_ExpFastSweepDown, "^timbre_ExpFastSweepDown"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.timbre_ExpSlowSweepDown, "^timbre_ExpSlowSweepDown"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.timbre_BulkSetLinearADSR, "^timbre_BulkSetLinearADSR"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.timbre_BulkSetExpADSR, "^timbre_BulkSetExpADSR"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.timbre_Clear, "^timbre_Clear"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.noise_QuickSoftPuff, "^noise_QuickSoftPuff"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.noise_SlowSoftPuff, "^noise_SlowSoftPuff"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.noise_QuickHardPuff, "^noise_QuickHardPuff"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.noise_SlowHardPuff, "^noise_SlowHardPuff"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.noise_Clear, "^noise_Clear"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.noise_co_apply_lpf_exp_fast, "^noise_co_apply_lpf_exp_fast"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.noise_co_apply_lpf_exp_slow, "^noise_co_apply_lpf_exp_slow"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.noise_co_BulkSetLinearADSR, "^noise_co_BulkSetLinearADSR"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.noise_co_BulkSetExpADSR, "^noise_co_BulkSetExpADSR"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.noise_co_Clear, "^noise_co_Clear"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.pitch_fast_linear_drop, "^pitch_fast_linear_drop"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.pitch_slow_linear_drop, "^pitch_slow_linear_drop"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.pitch_fast_linear_rise, "^pitch_fast_linear_rise"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.pitch_slow_linear_rise, "^pitch_slow_linear_rise"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.pitch_fast_exp_drop, "^pitch_fast_exp_drop"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.pitch_slow_exp_drop, "^pitch_slow_exp_drop"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.pitch_fast_exp_rise, "^pitch_fast_exp_rise"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.pitch_slow_exp_rise, "^pitch_slow_exp_rise"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.pitch_BulkSetLinearADSR, "^pitch_BulkSetLinearADSR"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.pitch_BulkSetExpADSR, "^pitch_BulkSetExpADSR"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.pitch_Clear, "^pitch_Clear"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.sample_AttackOnLevel100, "^sample_AttackOnLevel100"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.sample_AttackOnLevel75, "^sample_AttackOnLevel75"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.sample_AttackOnLevel50, "^sample_AttackOnLevel50"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.sample_AttackOnLevel25, "^sample_AttackOnLevel25"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.sample_SustainOnLevel25, "^sample_SustainOnLevel25"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.sample_ReleaseOnLevel25, "^sample_ReleaseOnLevel25"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.sample_AttackOnLevel10, "^sample_AttackOnLevel10"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.sample_SustainOnLevel10, "^sample_SustainOnLevel10"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.sample_ReleaseOnLevel10, "^sample_ReleaseOnLevel10"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.sample_AttackOnLevel5, "^sample_AttackOnLevel5"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.sample_SustainOnLevel5, "^sample_SustainOnLevel5"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.sample_ReleaseOnLevel5, "^sample_ReleaseOnLevel5"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.sample_KS_100, "^sample_KS_100"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.sample_KS_75, "^sample_KS_75"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.sample_KS_50, "^sample_KS_50"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.sample_KS_25, "^sample_KS_25"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.sample_Clear, "^sample_Clear"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.timbre_lfo_DeepAndSlow, "^timbre_lfo_DeepAndSlow"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.timbre_lfo_ShallowAndSlow, "^timbre_lfo_ShallowAndSlow"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.timbre_lfo_DeepAndFast, "^timbre_lfo_DeepAndFast"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.timbre_lfo_ShallowAndFast, "^timbre_lfo_ShallowAndFast"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.timbre_lfo_BulkSetLinearDSR, "^timbre_lfo_BulkSetLinearDSR"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.timbre_lfo_BulkSetExpDSR, "^timbre_lfo_BulkSetExpDSR"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.timbre_lfo_Clear, "^timbre_lfo_Clear"));


            //=====================================================================

            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.NoteValue, "^^([CDEFGAB]#?)(-?[0-8])"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.Percent, @"^[%$][-+]?\d+([,.]\d{1,3})?|^[-+]?\d+([,.]\d{1,3})?[$%]"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.Decimal, @"^-{0,1}\d*\.\d+"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.Integer, "^[-+]?[0-9]\\d*"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.Define, "^define"));

            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.AllZero, "^allzero"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.AllMax, "^allmax"));

            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.Direction, "^direction"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.Up, "^up"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.Down, "^down"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.Slope, "^slope"));

            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.WaveSet, "^waveset"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.CurrentNoteSector, "^currentnotesector"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.NoteSectorFrom, "^notesectorfrom"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.NoteSectorTo, "^notesectorto"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.NoteSector, "^notesector"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.CurrentIntensityLayer, "^currentintensitylayer"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.IntensityLayerFrom, "^intensitylayerfrom"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.IntensityLayerTo, "^intensitylayerto"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.IntensityLayer, "^intensitylayer"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.CurrentWaveformBlock, "^currentwaveformblock"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.CurrentWaveform, "^currentwaveform"));

            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.WaveformShape, "^waveformshape"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.WaveformName, "^waveformname"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.WaveformFrom, "^waveformfrom"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.WaveformTo, "^waveformto"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.WaveformBlock, "^waveformblock"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.Waveform, "^waveform"));

            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.HarmonicIDFrom, "^harmonicidfrom"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.HarmonicIDTo, "^harmonicidto"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.HarmonicID, "^harmonicid"));

            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.CopyToRange, "^copytorange"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.CopyTo, "^copyto"));

            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.CopyAcrossNoteSectors, "^copyacrossnotesectors"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.CopyAcrossIntensityLayers, "^copyacrossintensitylayers"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.CopyAcrossAllDimensions, "^copyacrossalldimensions"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.MorphAcrossNoteSectors, "^morphacrossnotesectors"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.MorphAcrossIntensityLayers, "^morphacrossintensitylayers"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.MorphWaveformBlocks, "^morphwaveformblocks"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.MorphWaveforms, "^morphwaveforms"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.MorphHarmonicLevels, "^morphharmoniclevels"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.ShapeTheHarmonics, "^shapetheharmonics"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.SetCurrentIntensityLayer, "^setcurrentintensitylayer"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.SetCurrentNoteSector, "^setcurrentnotesector"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.SetCurrentWaveform, "^setcurrentwaveform"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.SetName, "^setname"));

            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.HarmonicLevelShapeType, "^harmoniclevelshapetype"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.SetHarmonicLevelsFromCSV, "^setharmoniclevelsfromcsv"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.SetHarmonicLevelsFromLevelShapeType, "^setharmoniclevelsfromlevelshapetype"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.SetHarmonicLevelsFromInstrumentType, "^setharmoniclevelsfrominstrumenttype"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.SetHarmonicLevelsFromWaveFormShape, "^setharmoniclevelsfromwaveformshape"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.SetHarmonicLevels, "^setharmoniclevels"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.SetHarmonicLevel, "^setharmoniclevel"));

            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.HarmonicPhaseShapeType, "^harmonicphaseshapetype"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.SetHarmonicPhasesFromCSV, "^setharmonicphasesfromcsv"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.SetHarmonicPhasesFromPhaseShapeType, "^setharmonicphasesfromphaseshapetype"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.SetHarmonicPhasesFromInstrumentType, "^setharmonicphasesfrominstrumenttype"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.SetHarmonicPhasesFromWaveFormShape, "^setharmonicphasesfromwaveformshape"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.SetHarmonicPhases, "^setharmonicphases"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.SetHarmonicPhase, "^setharmonicphase"));

            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.MorphType, "^morphtype"));

            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.Exponential, "^exponential"));

            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.LinearDecreasing, "^lineradescreasing"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.ExponentialAll, "^exponentialall"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.ExponentialOdd, "^exponentialodd"));

            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.LevelCSV, "^levelcsv"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.Level, "^level"));

            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.PhasesCSV, "^phasescsv"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.Phase, "^phase"));

            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.AllInPhase, "^allinphase"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.InvertedEven, "^invertedeven"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.InvertedOdd, "^invertedodd"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.Inverted3711, "^inverted3711"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.Inverted, "^inverted"));

            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.Triangle, "^triangle"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.Square, "^square"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.Sine, "^sine"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.Saw, "^saw"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.InstrumentType, "^instrumenttype"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.Trumpet, "^trumpet"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.Violin, "^violin"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.Flute, "^flute"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.Cello, "^cello"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.Guitar, "^guitar"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.MutedGuitar, "^mutedguitar"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.Clarinet, "^clarinet"));

            //======PATCH ================================================================
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.Patch, "^patch"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.General, "^general"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.ADSR, "^adsr"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.SetPortamento, "^setportamento"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.SetScalingSplit, "^setscalingsplit"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.SetPatchGain, "^setpatchgain"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.SetActiveOscillators, "^setactiveoscillators"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.SetDetuningMode, "^setdetuningmode"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.SetOscillatorDetuning, "^setoscillatordetuning"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.SetLFO, "^setlfo"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.SetEnvelopeGainController, "^setenvelopegaincontroller"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.ConfigureSection, "^configuresection"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.ConfigureEnvelope, "^configureenvelope"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.Enabled, "^enabled"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.PortamentoAmount, "^portamentoamount"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.Note, "^note"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.DetuningType, "^detuningtype"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.OscillatorCount, "^oscillatorcount"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.DetuningMode, "^detuningmode"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.SetDetuning, "^setdetuning"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.Detuning, "^detuning"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.Oscillator, "^oscillator"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.LFOType, "^lfotype"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.WaveType, "^wavetype"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.FrequencyCC, "^frequencycc"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.Frequency, "^frequency"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.DepthCC, "^depthcc"));
          
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.InitialLevel, "^initiallevel"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.GainCC, "^gaincc"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.Gain, "^gain"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.Section, "^section"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.Duration, "^duration"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.EndKSU, "^endksu"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.EndKSL, "^endksl"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.SampleMode, "^samplemode"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.EnvelopeType, "^envelopetype"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.Envelope, "^envelope"));
            
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.TargetKSU, "^targetksu"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.TargetKSL, "^targetksl"));

            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.Target, "^target"));

            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.LinearDeltaKSU, "^lineardeltaksu"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.LinearDeltaKSL, "^lineardeltaksl"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.LinearDelta, "^lineardelta"));

            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.Linear, "^linear"));

            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.ExpMultKSU, "^expmultksu"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.ExpMultKSL, "^expmultksl"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.ExpMult, "^expmult"));

            //-----------------------------------------------
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.Amplitude, "^amplitude"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.NoiseGain, "^noisegain"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.NoiseCutoffFrequency, "^noisecutofffrequency"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.TimbreMorph, "^timbremorph"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.SampleGain, "^samplegain"));

            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.Sample, "^sample"));

            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.PitchShift, "^pitchshift"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.TremoloDepth, "^tremolodepth"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.VibratoDepth, "^vibratodepth"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.TimbreLFODepth, "^timbrelfodepth"));

            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.Random, "^random"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.Regular, "^regular"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.True, "^true"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.False, "^false"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.Cents, "^cents"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.Hz, "^hz"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.Tremolo, "^tremolo"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.Vibrato, "^vibrato"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.TimbreLFO, "^timbrelfo"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.Modulation, "^modulation"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.BreathController, "^breathcontroller"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.FootController, "^footcontroller"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.Expression, "^expression"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.None, "^none"));

            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.KeyHammer1, "^keyhammer1"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.KeyHammer2, "^keyhammer2"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.KeyHammer3, "^keyhammer3"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.KeyHammer4, "^keyhammer4"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.BowScrape1, "^bowscrape1"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.BowScrape2, "^bowscrape2"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.KeyClick, "^keyclick"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.KeyUp, "^keyup"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.KSVLongF, "^ksvlongf"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.KSLongF, "^kslongf"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.KSStdF, "^ksstdf"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.KSShortF, "^ksshortf"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.KSVShortF, "^ksvshortf"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.KSVLong, "^ksvlong"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.KSLong, "^kslong"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.KSStd, "^ksstd"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.KSShort, "^ksshort"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.KSVShort, "^ksvshort"));


            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.OneShot, "^oneshot"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.Looped, "^looped"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.ExpMultKSU, "^expmultksu"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.ExpMultKSL, "^expmultksl"));

            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.LoadPreset, "^loadpreset"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.PresetName, "^presetname"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.BulkValue, "^bulkvalue"));

            //==============================================
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.Filter, "^filter"));

            //-----------------------------------------------

            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.NameString, "^'[a-z A-Z0-9]+'"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.All, "^all"));

            //------------------------------------------------
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.Velocity, "^velocity"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.Attack, "^attack"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.Decay, "^decay"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.Sustain, "^sustain"));
            _tokenDefinitions.Add(new c_lexer_token_definition(TokenType.Release, "^release"));
        }
        #endregion

    }
}
