#include "util.h"
#include "oled.h"
#include "flash.h"

unsigned long places(signed char n){
//as fast as switch version, but only~15 words larger than loop    

    const unsigned long r[]={1UL,10UL,100UL,1000UL,10000UL,100000UL,
    1000000UL,10000000UL,100000000UL,1000000000UL};
    return r[n];


//use ample RAM to generate sequence once at startup
//still uses too much flash to generate
/*
    static unsigned long r[10];
    if(r[0]!=1){
        char i;
        r[0]=1;
        for(i=1;i<10;i++){r[i]=r[i-1]*10;}
    }
    return r[n];
*/       
/*
//different implementation is 78 words larger than below, but much faster
    switch(n){      
        case 0:return 1UL;
        case 1:return 10UL;
        case 2:return 100UL;
        case 3:return 1000UL;
        case 4:return 10000UL;
        case 5:return 100000UL;
        case 6:return 1000000UL;
        case 7:return 10000000UL;
        case 8:return 100000000UL;
        default:return 1000000000UL;
    }
*/
/*
//slow version for not much flash, assuming long mult is used elsewhere
    unsigned long r=1;
    while(n--){r=r*10;}
    return r;    
*/
}

char getDigit(signed char p){  //get digit at pth decimal place (0 = ones)
    unsigned long n=t;
    signed char res,pstart,zf;
    zf=0;
    pstart=9;
    while(pstart>=p){
        res=0;
        while(n>=places(pstart)){
            n=n-places(pstart);
            res++;
            zf=1;   //non zero digit detected
        }
        pstart=pstart-1;
    }
    if((zf==0)&&(res==0)&&(p!=0)){return F_SPACE;}  //zero blanking, ones position is never blanked
    return res;
}

void getV(void){                //common ADC sample code
    LATA5=1;
    LATA4=0;
    __delay_us(200);
    vf=getADC();
    //set reverse
    LATA5=0;
    LATA4=1;
    __delay_us(200);
    vr=getADC();
    //set normal
    LATA5=1;
    LATA4=0;    
    vr=(ADC_OS*1023)-vr;    
}

long getR(void){                         //get resistance, negative => invalid
    //getV();
    long temp;
    if(abs(vf-vr)>R_TOLERANCE){return -1;}  //check if same behaviour in reverse
    vf=vf+vr;                               //combine/average
    if(vf>(R_HIGH_ADC*ADC_OS*2)){return -1;}//if high, return error
    temp=(rtest*vf)/((ADC_OS*1023*2)-vf)-rlead;    
    if(temp<0){return 0;}else{return temp;}    
}

long getDF(void){                         //get diode FV, result in mV, negative value for invalid
    //getV();
    if(vr>D_HIGH_ADC*ADC_OS){       //appears open circuit reversed
        if(vf<D_HIGH_ADC*ADC_OS){   //conducting in forward
            return (vf*bvolt)/(ADC_OS*1023);
        }
    }
    return -1;
}

long getDR(void){
    int temp;          //swap
    temp=vf;
    vf=vr;
    vr=temp;
    return getDF();     //use forward function
}

int getCLO(void){      //use shared capacitance method to get lo-range capacitance
    //DUT is normally fully charged via RA5, RA4 is at 0V
    unsigned char chs;
    int adc;
    int res;
    chs=ADCON0;                 //save channel
    ADCON0bits.CHS=0b011011;    //VSS=> discharge sample and hold cap
    ADCON0bits.ADGO=1;          //start
    while(ADCON0bits.ADGO){}    //wait till done
    //ADRES; //don't need to do anything with this read of GND
    TRISA5=1;                   //float pin to allow capacitance to be read
    ADCON0=chs;                 //change to test channel
    ADCON0bits.ADGO=1;          //start
    while(ADCON0bits.ADGO){}    //wait till done
    adc=ADRES;                  //this is our reading
    TRISA5=0;                   //restore pin for normal operation
    if(adc<SCAP_ADCLO){return -1;}
    if(adc>SCAP_ADCHI){return -1;}
    res = ((adc*SCAP_SAMPLEC)/(1023-adc))-SCAP_STRAY;
    if(res>999){return -1;}
    return res;
}

long getC(void){
    unsigned int j,k;
    long ret;
    k=1023;
    unsigned int n=C_COUNT;
    unsigned int n2;
    LATA5=1;
    LATA4=0;
    ADCON0bits.ADGO=1;          //start
    while(ADCON0bits.ADGO){}    //wait till done
    if(ADRES<512){sleepDelay();}    //help to charge up cap if needed
    LATA5=0;
    ADCON0bits.ADGO=1;          //start
    while(ADCON0bits.ADGO){}    //wait till done
    j=ADRES;
    while((k&512)&&(n)){
        n--;
        ADCON0bits.ADGO=1;          //start
        while(ADCON0bits.ADGO){}    //wait till done
        k=ADRES;
    }
    LATA5=1;
    LATA4=0;
    n2=C_COUNT-n;   //use calculation later and free n
    //if j<512, probably invalid (too fast discharge)
    //if j==k, very large value
    if(j<512){return -1;}   //first reading not valid, low value
    if(j<=k){return -2;}    //no difference, high value, avoid div/0
    ret = (((unsigned long)j*(n2)*(C_FACTOR/64))/(j-k))*64;  //2 step C_FACTOR to avoid overflow
    return ret;
}

void getFVR(void){
    unsigned char chs;
    int res;
    FVRCON=0x81;            //FVR on with ADFVR=1.024V
    chs=ADCON0;             //save
    ADCON0bits.CHS=0b011110;//FVR
    while(!FVRCONbits.FVRRDY){}        //wait till ready    
    res=getADC();
    FVRCON=0;               //shut down FVR
    ADCON0=chs;             //restore
    if(res<2800){
        bvolt=6000;         //6V=> limit of PIC    
    }else{
        bvolt=(1L<<24)/res;     //16384*1024mV
    }
}

void showD(void){
    OLEDchar4(F_SPACE);
    OLEDchar4(lzFix(getDigit(3)));
    OLEDchar4(F_POINT);                                
    OLEDchar4(lzFix(getDigit(2)));
    OLEDchar4(F_VOLT);                
}

void showCLO(void){
    //returned in pF
    if(t<10000L){
        OLEDchar4(getDigit(3));
        OLEDchar4(getDigit(2));
        OLEDchar4(getDigit(1));
        OLEDchar4(getDigit(0));
        OLEDchar4(F_PICO);                                            
        OLEDchar4(F_FARAD);    
    }else{
        showC();
    }
}

void showC(void){
    //returned in pF, but ~1n is smallest, read down to 0.1 nF
    if(t<100000L){
        OLEDchar4(getDigit(4));
        OLEDchar4(lzFix(getDigit(3)));
        OLEDchar4(F_POINT);                                    
        OLEDchar4(lzFix(getDigit(2)));
        OLEDchar4(F_NANO);                                            
    }else if(t<1000000L){
        OLEDchar4(F_SPACE);
        OLEDchar4(getDigit(5));
        OLEDchar4(getDigit(4));
        OLEDchar4(lzFix(getDigit(3)));
        OLEDchar4(F_NANO);                                            
    }else if(t<100000000L){
        OLEDchar4(getDigit(7));
        OLEDchar4(lzFix(getDigit(6)));
        OLEDchar4(F_POINT);                                    
        OLEDchar4(lzFix(getDigit(5)));
        OLEDchar4(F_MICRO);                                            
    }else{
        OLEDchar4(getDigit(9));
        OLEDchar4(getDigit(8));
        OLEDchar4(getDigit(7));
        OLEDchar4(lzFix(getDigit(6)));
        OLEDchar4(F_MICRO);                                            
    }
    OLEDchar4(F_FARAD);            
}

void showR(void){
    if(t<2000L){
        OLEDchar4(F_SPACE);
        OLEDchar4(getDigit(3));
        OLEDchar4(getDigit(2));
        OLEDchar4(getDigit(1));
        OLEDchar4(getDigit(0));
    }else if(t<10000L){
        OLEDchar4(getDigit(3));
        OLEDchar4(F_POINT);                                
        OLEDchar4(getDigit(2));
        OLEDchar4(getDigit(1));
        OLEDchar4(F_KILO);                
    }else if(t<100000L){
        OLEDchar4(getDigit(4));
        OLEDchar4(getDigit(3));
        OLEDchar4(F_POINT);                                
        OLEDchar4(getDigit(2));
        OLEDchar4(F_KILO);                
    }else if(t<1000000L){
        OLEDchar4(F_SPACE);
        OLEDchar4(getDigit(5));
        OLEDchar4(getDigit(4));
        OLEDchar4(getDigit(3));
        OLEDchar4(F_KILO);                
    }else{
        OLEDchar4(getDigit(6));
        OLEDchar4(F_POINT);                                
        OLEDchar4(getDigit(5));
        OLEDchar4(getDigit(4));
        OLEDchar4(F_MEGA);                
    }    
    OLEDchar4(F_OHM);                
}

char lzFix(char c){ //suppress leading zero suppression for display reasons
    if(c==F_SPACE){return 0;}
    return c;
}

void setHand(void){                      //check and set handedness
    delay200();
    if(getADC()<(512*ADC_OS)){textFlip=TEXT_LH;}       //flip
    OLEDflip(textFlip);                         //set hardware
    OLEDclear();
    col=0;
    page=0;
    if(textFlip==TEXT_RH){        
        OLEDchar4(F_R);
    }else{
        OLEDchar4(F_L);
    }
    OLEDchar4(F_SPACE);
    OLEDchar4(F_H);
    OLEDchar4(F_A);
    OLEDchar4(F_N);
    OLEDchar4(F_D);
    delay2000();    
    showRelease();    
}

void setTestR(void){                //set and save series test resistance value
    unsigned int rval=rtest;
    char updone=0;
    char downdone=0;
    int j;          //timer/counter
    OLEDclear();
    col=0;
    page=0;
    OLEDchar1(F_C);
    OLEDchar1(F_A);
    OLEDchar1(F_L);
    OLEDchar1(F_SPACE);
    OLEDchar1(F_R);
    while((!updone)||(!downdone)){
        if(!updone){
            updone=1;
            page=0;
            col=45;
            OLEDchar1(F_PLUS);
            j=23;
            while(j){
                j--;
                if(getADC()<(512*ADC_OS)){  //if tips closed
                    j=23;                   //reset timer
                    rval=rval+1;            //increase
                    updone=0;
                    downdone=0;
                }
                if(rval>R_TEST_UPPER){rval=R_TEST_UPPER;}
                showCount(j);
                //value is in t
                t=rval;
                showOhms();
                sleepDelay();
            }
        }
        if(!downdone){
            downdone=1;
            page=0;
            col=45;
            OLEDchar1(F_MINUS);
            j=23;
            while(j){
                j--;
                if(getADC()<(512*ADC_OS)){  //if tips closed
                    j=23;                   //reset timer
                    rval=rval-1;            //increase
                    updone=0;
                    downdone=0;
                }
                if(rval<R_TEST_LOWER){rval=R_TEST_LOWER;}
                showCount(j);
                //value is in t
                t=rval;
                showOhms();
                sleepDelay();                
            }
        }
    }    
    t=rval;
    if(saveQuery()){
        rtest=rval;         //keep value
        SaveCalToFlash();   //save to flash
    }
    loadCalFromFlash();
    showLoad();
    t=rtest;
    showOhms();//value is in t
    delay2000();    //allow time to show

}

void setContR(void){                    //set and save contact/lead resistance
    int j,a;        //timer/counter
    long rTemp,rAcc;//working variables
    char rOK=1;     //contact test OK    
    rTemp=rlead;    //save
    showHold();
    j=23;
    while(j){
        j--;
        if(getADC()<(512*ADC_OS)){j=0;}//tips closed
        showCount(j);
        sleepDelay();
    }   
    OLEDclear();
    if(getADC()<(512*ADC_OS)){  //if tips closed
        rAcc=0;
        rlead=0;        //temporarily set so that we can calibrate it out
        j=R_LEAD_OS;
        while(rOK && j){
            j--;
            getV();
            a=getR();
            if((a>=R_LEAD_LOWER)&&(a<=R_LEAD_UPPER)){
                rAcc=rAcc+a;
                t=a;
                showCount(j);
                showOhms();
            }else{
                rOK=0;
            }
        }
        rlead=rTemp;            //restore saved
        showRelease();
        if(rOK){
            rAcc=rAcc/R_LEAD_OS;
            t=rAcc;            
            if(saveQuery()){
                rlead=rAcc;         //keep value
                SaveCalToFlash();   //save to flash
            }
        }
    }
    loadCalFromFlash();
    //value is in t
    showLoad();
    t=rlead;
    showOhms();
    delay2000();    //allow time to show
}

char promptCal(void){   //prompt to check if calibration wanted
    char j;
    OLEDclear();
    col=5;
    page=0;
    OLEDchar4(F_C);
    OLEDchar4(F_A);
    OLEDchar4(F_L);
    OLEDchar4(F_QUERY);
    j=23;
    while(j){
        j--;
        showCount(j);        
        sleepDelay();                
        sleepDelay();                
        if(getADC()<(512*ADC_OS)){
            showRelease();
            return 1;
        }
    }
    return 0;        
}

void doCalibration(void){           //wrap up all the calibration features
    setHand();                      //check and set handedness
    if(promptCal()){                //check if calibration needed to save time
        setTestR();                     //set and save series test resistance value
        setContR();                     //set and save contact/lead resistance
    }
    //reset for main program
    OLEDclear();
    col=0;
    page=0;
}

void sleepDelay(void){                  //nominally 200ms for legible display updates
    WDTCONbits.PS=8;    //256ms nominal
    WDTCONbits.SEN=1;   //enable
    SLEEP();            //sleep
    NOP();              //ensure next instruction is executed
    WDTCONbits.SEN=0;   //disable
}

void loadCalFromFlash(void){            //custom wrapper for this program
    rtest=NVMimage[0];      //load
    rlead=NVMimage[1];      //load
    //validate
    if(rtest<R_TEST_LOWER){rtest=R_TEST_LOWER;}
    if(rtest>R_TEST_UPPER){rtest=R_TEST_UPPER;}
    if(rlead<R_LEAD_LOWER){rlead=R_LEAD_LOWER;}
    if(rlead>R_LEAD_UPPER){rlead=R_LEAD_UPPER;}
}

void SaveCalToFlash(void){              //custom wrapper for this program
    flashshadow[0]=(rtest>>0)&0xFF;
    flashshadow[1]=(rtest>>8)&0xFF;
    flashshadow[2]=(rlead>>0)&0xFF;
    flashshadow[3]=(rlead>>8)&0xFF;
    eraseflash(LAST_FLASH_PAGE);
    writeflash(LAST_FLASH_PAGE);
}

void showCount(char j){     //helper to catch duplicated code
    col=54;          //show time
    page=0;
    OLEDchar1((j>>2)&7);    
}

void showOhms(void){    //to help with repeated code, uses global t
    col=0;          //show values
    page=2;
    OLEDchar2(getDigit(4));
    OLEDchar2(getDigit(3));
    OLEDchar2(getDigit(2));
    OLEDchar2(getDigit(1));
    OLEDchar2(getDigit(0));
    OLEDchar2(F_OHM);               
}

void showHold(void){
    OLEDclear();
    col=12;
    page=1;
    OLEDchar2(F_H);
    OLEDchar2(F_O);
    OLEDchar2(F_L);
    OLEDchar2(F_D);    
}

void showRelease(void){ //also waits for release
    OLEDclear();
    col=0;
    page=1;
    OLEDchar2(F_R);
    OLEDchar2(F_E);
    OLEDchar2(F_L);
    OLEDchar2(F_E);
    OLEDchar2(F_A);
    OLEDchar2(F_S);
    OLEDchar2(F_E);    
    while(getADC()<(512*ADC_OS)){}//wait for release
    delay200();
}

void showLoad(void){
    OLEDclear();    //show loaded value
    col=0;
    page=0;
    OLEDchar1(F_L);
    OLEDchar1(F_O);
    OLEDchar1(F_A);
    OLEDchar1(F_D);
    OLEDchar1(F_SPACE);
    OLEDchar1(F_R);    
}

char saveQuery(void){       //prompt for save and return result
    char j;
    OLEDclear();
    col=0;
    page=0;
    OLEDchar1(F_S);
    OLEDchar1(F_A);
    OLEDchar1(F_V);
    OLEDchar1(F_E);
    OLEDchar1(F_QUERY);
    j=23;
    while(j){
        j--;
        showCount(j);        
        showOhms();        //value is in t
        sleepDelay();                
        if(getADC()<(512*ADC_OS)){
            OLEDclear();
            col=0;
            page=0;
            OLEDchar4(F_S);
            OLEDchar4(F_A);
            OLEDchar4(F_V);
            OLEDchar4(F_E);
            OLEDchar4(F_SPACE);
            OLEDchar4(F_R);
            delay2000();    //allow time to show
            return 1;
        }
    }
    return 0;    
}

