//Digital Lighting Controller Translator
//Receives new (c2020) DLC data and outputs old (c2010) data for slaves
//PIC16F1705
//local         mains detect on RC4
//              status LED (active high) on RA4
//upstream      incoming data on RC5
//downstream    chain sense on RC3 (AN7)
//              latch on RA2
//              data on RC0
//              clock on RC1
//              clear on RC2

#include "config.h"
#include <xc.h>
#include "io.h"
#include "tmr2.h"
#include "uart.h"

#define _XTAL_FREQ (32000000UL)
#define T2_MAX 680
#define INCOMING_SIZE 32
#define PWM_STEPS 20
#define CHAIN_LENGTH 8
#define OFFSET (1)


const unsigned char onPoints[PWM_STEPS]={   //1st is always off
    70+OFFSET,65+OFFSET,61+OFFSET,57+OFFSET,54+OFFSET,
    51+OFFSET,48+OFFSET,45+OFFSET,42+OFFSET,39+OFFSET,
    36+OFFSET,33+OFFSET,30+OFFSET,27+OFFSET,24+OFFSET,
    20+OFFSET,16+OFFSET,11+OFFSET,6+OFFSET,0+OFFSET
};

unsigned char SPIbm[PWM_STEPS][4]={        //what is sent at each stage above, 1st should be 0 always
    {0,0,0,0},{0,0,0,0},{0,0,0,0},{0,0,0,0},
    {0,0,0,0},{0,0,0,0},{0,0,0,0},{0,0,0,0},
    {0,0,0,0},{0,0,0,0},{0,0,0,0},{0,0,0,0},
    {0,0,0,0},{0,0,0,0},{0,0,0,0},{0,0,0,0},
    {0,0,0,0},{0,0,0,0},{0,0,0,0},{0,0,0,0}    
};

//ADC points for chain length sense divider
const int chainPoint[CHAIN_LENGTH]={163,411,547,633,693,736,770,796};
//bitmasks for setting/clearing
const unsigned char bitmask[8]={1,2,4,8,16,32,64,128};
const unsigned char bitunmask[8]={254,253,251,247,239,223,191,127};

char checkChain(void);
void setBits(unsigned char p, unsigned char d);

void main(void){
    unsigned char d;                        //received data byte
    char frame;                             //frame error flag
    static unsigned char byteCount=126;     //#of bytes into frame, >63 => error
    static unsigned char frameType=0xFF;    //first byte of frame
    char last_AC_IN;
    unsigned char tcycles=0;    //raw counter
    unsigned char tact=0;       //adjusted control duty cycle
    unsigned char tup=60;
    unsigned char tdown=156;    //these are typical values, so will allow operation without AC present
    unsigned char toffset=9;    
    unsigned int ledCount=8192; //timing of LED operation
    unsigned char LEDstate=1;   //which sets output behaviour   
    unsigned char dataOK=0;     //flag
    unsigned char ACOK=0;       //flag
    unsigned char chainOK=0;    //flag    
    OSCCONbits.IRCF=0b1110;                 //32MHz when 4xPLL is on
    ioInit();
    tmr2Init();
    uartInit();
    last_AC_IN=AC_IN;
    while(1){
        //check mains phase
        if(last_AC_IN != AC_IN){    //this is true least
            last_AC_IN = AC_IN;     //capture state now
            //STATUS_LED=last_AC_IN;
            if(last_AC_IN){         //positive going should be delayed
                tup=tcycles;
                if((tup<65)&&(tup>55)){ACOK=1;} //reasonable timing
            }else{
                tdown=156;       //fixed for 50Hz
                tcycles=0;
                toffset=(tdown-tup-tup)>>2;
            }
        }
        if(TMR2IF){             //don't check TMR2IE, assume it's enabled
            TMR2IF=0;           //clear flag
            tcycles++;
            tact++;            
            if(tcycles==(tup+toffset)){tact=0;}       //synchronise
            if(tact>(tdown>>1)){tact=tact-(tdown>>1);}    //avoid missed cycles
                 if(tact==onPoints[ 0]){sendBuffer(&SPIbm[ 0][0]);}
            else if(tact==onPoints[ 1]){sendBuffer(&SPIbm[ 1][0]);}
            else if(tact==onPoints[ 2]){sendBuffer(&SPIbm[ 2][0]);}
            else if(tact==onPoints[ 3]){sendBuffer(&SPIbm[ 3][0]);}
            else if(tact==onPoints[ 4]){sendBuffer(&SPIbm[ 4][0]);}
            else if(tact==onPoints[ 5]){sendBuffer(&SPIbm[ 5][0]);}
            else if(tact==onPoints[ 6]){sendBuffer(&SPIbm[ 6][0]);}
            else if(tact==onPoints[ 7]){sendBuffer(&SPIbm[ 7][0]);}
            else if(tact==onPoints[ 8]){sendBuffer(&SPIbm[ 8][0]);}
            else if(tact==onPoints[ 9]){sendBuffer(&SPIbm[ 9][0]);}
            else if(tact==onPoints[10]){sendBuffer(&SPIbm[10][0]);}
            else if(tact==onPoints[11]){sendBuffer(&SPIbm[11][0]);}
            else if(tact==onPoints[12]){sendBuffer(&SPIbm[12][0]);}
            else if(tact==onPoints[13]){sendBuffer(&SPIbm[13][0]);}
            else if(tact==onPoints[14]){sendBuffer(&SPIbm[14][0]);}
            else if(tact==onPoints[15]){sendBuffer(&SPIbm[15][0]);}
            else if(tact==onPoints[16]){sendBuffer(&SPIbm[16][0]);}
            else if(tact==onPoints[17]){sendBuffer(&SPIbm[17][0]);}
            else if(tact==onPoints[18]){sendBuffer(&SPIbm[18][0]);}
            else if(tact==onPoints[19]){sendBuffer(&SPIbm[19][0]);}
            //manage status LED
            ledCount=(ledCount+1)&16383;
            if(ledCount==0){
                chainOK=checkChain();
                if(dataOK==0){LEDstate=1;}
                if(chainOK==0){LEDstate=2;}
                if(ACOK==0){LEDstate=3;}
                if(dataOK && chainOK && ACOK){LEDstate=0;}
                dataOK=0;   //clear and allow to be set for next cycle                
                ACOK=0;   //clear and allow to be set for next cycle
            }
            if(LEDstate==0){
                STATUS_LED=1;
            }else{
                if(((ledCount>>12)<LEDstate) && ((ledCount&2048))){
                    STATUS_LED=1;
                }else{
                    STATUS_LED=0;
                }
            }
        }
        if(RCIF){   //don't need to check RCIE as we assume it's enabled
            if(OERR){CREN=0;CREN=1;}   //reset overflow error (serious, but better than locking up)
            RCIF=0;           //clear flag
            frame=FERR;
            d=RC1REG;
            if(frame){
                if(d==0){               //start of frame
                    byteCount=0;        //reset
                    frameType=0xFF;
                }else{                  //data error
                    byteCount=126;      
                }                
            }else{                      //normal data
                if(byteCount==0){       //first byte of frame
                    frameType=d;
                }else{                  //control bytes
                    if(frameType==0){
                        if((byteCount>0)&&(byteCount<=INCOMING_SIZE)){
                            //din[byteCount-1]=d;
                            setBits(byteCount-1,d); //set up output bitmaps
                        }
                    }
                }
                if(byteCount<126){byteCount++;}     //keep count
                if(byteCount==2){dataOK=1;}         //have received a valid packet
            }
        }
    }
    return;
}

char checkChain(void){
    int v;
    char n=0;
    v=getADC(CHAIN_SENSE);
    while(n<CHAIN_LENGTH){
        if(v<chainPoint[n]){return n;}
        n++;
    }
    return n;
}

void setBits(unsigned char p, unsigned char d){
    unsigned char bi,by;
    bi=p&7;
    by=(p>>3)&3;
    //d=d>>4;//gives 0-15
    d=d>>1;
    d=d+(d>>2);
    d=d>>3; //gives 0-19
//    if( 0>d){SPIbm[ 0][by]=SPIbm[ 0][by]|bitmask[bi];}else{SPIbm[ 0][by]=SPIbm[ 0][by]&bitunmask[bi];}
    SPIbm[ 0][by]=SPIbm[ 0][by]&bitunmask[bi];//above result is constant
    if( 1<=d){SPIbm[ 1][by]=SPIbm[ 1][by]|bitmask[bi];}else{SPIbm[ 1][by]=SPIbm[ 1][by]&bitunmask[bi];}
    if( 2<=d){SPIbm[ 2][by]=SPIbm[ 2][by]|bitmask[bi];}else{SPIbm[ 2][by]=SPIbm[ 2][by]&bitunmask[bi];}
    if( 3<=d){SPIbm[ 3][by]=SPIbm[ 3][by]|bitmask[bi];}else{SPIbm[ 3][by]=SPIbm[ 3][by]&bitunmask[bi];}
    if( 4<=d){SPIbm[ 4][by]=SPIbm[ 4][by]|bitmask[bi];}else{SPIbm[ 4][by]=SPIbm[ 4][by]&bitunmask[bi];}
    if( 5<=d){SPIbm[ 5][by]=SPIbm[ 5][by]|bitmask[bi];}else{SPIbm[ 5][by]=SPIbm[ 5][by]&bitunmask[bi];}
    if( 6<=d){SPIbm[ 6][by]=SPIbm[ 6][by]|bitmask[bi];}else{SPIbm[ 6][by]=SPIbm[ 6][by]&bitunmask[bi];}
    if( 7<=d){SPIbm[ 7][by]=SPIbm[ 7][by]|bitmask[bi];}else{SPIbm[ 7][by]=SPIbm[ 7][by]&bitunmask[bi];}
    if( 8<=d){SPIbm[ 8][by]=SPIbm[ 8][by]|bitmask[bi];}else{SPIbm[ 8][by]=SPIbm[ 8][by]&bitunmask[bi];}
    if( 9<=d){SPIbm[ 9][by]=SPIbm[ 9][by]|bitmask[bi];}else{SPIbm[ 9][by]=SPIbm[ 9][by]&bitunmask[bi];}
    if(10<=d){SPIbm[10][by]=SPIbm[10][by]|bitmask[bi];}else{SPIbm[10][by]=SPIbm[10][by]&bitunmask[bi];}
    if(11<=d){SPIbm[11][by]=SPIbm[11][by]|bitmask[bi];}else{SPIbm[11][by]=SPIbm[11][by]&bitunmask[bi];}
    if(12<=d){SPIbm[12][by]=SPIbm[12][by]|bitmask[bi];}else{SPIbm[12][by]=SPIbm[12][by]&bitunmask[bi];}
    if(13<=d){SPIbm[13][by]=SPIbm[13][by]|bitmask[bi];}else{SPIbm[13][by]=SPIbm[13][by]&bitunmask[bi];}
    if(14<=d){SPIbm[14][by]=SPIbm[14][by]|bitmask[bi];}else{SPIbm[14][by]=SPIbm[14][by]&bitunmask[bi];}
    if(15<=d){SPIbm[15][by]=SPIbm[15][by]|bitmask[bi];}else{SPIbm[15][by]=SPIbm[15][by]&bitunmask[bi];}
    if(16<=d){SPIbm[16][by]=SPIbm[16][by]|bitmask[bi];}else{SPIbm[16][by]=SPIbm[16][by]&bitunmask[bi];}
    if(17<=d){SPIbm[17][by]=SPIbm[17][by]|bitmask[bi];}else{SPIbm[17][by]=SPIbm[17][by]&bitunmask[bi];}
    if(18<=d){SPIbm[18][by]=SPIbm[18][by]|bitmask[bi];}else{SPIbm[18][by]=SPIbm[18][by]&bitunmask[bi];}
    if(19<=d){SPIbm[19][by]=SPIbm[19][by]|bitmask[bi];}else{SPIbm[19][by]=SPIbm[19][by]&bitunmask[bi];}
}