'''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
'
' Orrery V1
' Kenneth Horton, August 2021
'
' Requires MMBasic 5.2 or later and an ILI9341 based LCD panel with touch
' Plus optionally a DS3231 real time clock
'
'''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''

CPU 48
' Option LCDPANEL ILI9341, L, 2, 23, 6
' Option TOUCH 7,15
' GUI CALIBRATE

Option Explicit
Option Default none 
Option Autorun On

' Earth Mercury conjunction 25/10/2020 18:16 GMT
' Earth Venus conjunction 03/06/2020 17:37 GMT
' Winter solstice 21/12/2020 10:02 GMT
' Earth Mars opposition 13/10/2020 19:20 GMT
' Earth Jupiter opposition 14/07/2020 08:00 GMT
' Earth Saturn opposition 09/07/2019 16:53 GMT
' Earth Uranus opposition 31/10/2020 15:18 GMT
' Earth Neptune opposition 11/09/2020 20:10 GMT
' Total solar eclipse 14/12/2020 13:33 GMT
' Time between high tides 12 hours 25 minutes 25 seconds = 44725 seconds - Gives 74.5 10 minute steps
' Screen 320 x 240

' Define some general colours
const orange = RGB(255,223,0)
const chartreuse = RGB(239,255,0)
const spring_green = RGB(0,255,239)
const azure = RGB(0,239,255)
const violet = RGB(239,0,255)
const rose = RGB(255,0,239)
const bright_blue = RGB(159,159,255)

' Define the colours for various elements of the display
Const c.Sun = RGB(yellow)
Const c.Mercury =  RGB(240,240,240)
Const c.Venus =  RGB(255,255,128)
Const c.Earth =  RGB(200,200,255)
Const c.Mars =  RGB(255,128,128)
Const c.Jupiter =  RGB(255,255,192)
Const c.Saturn =  RGB(255,255,128)
Const c.Uranus =  RGB(240,255,255)
Const c.Neptune =  RGB(200,200,255)
Const c.Moon =  RGB(yellow)
Const c.button = RGB(cyan)
Const c.save = RGB(white)
Const c.delete = RGB(magenta)
Const c.entry = RGB(yellow)
Const c.ghosttext = RGB(210,210,210)
Const c.yes = RGB(192,255,192)
Const c.No = RGB(255,192,192)

Const False = 0
Const True = 1

' Define the radius of the sun, moon and planet orbits
Const Sun_radius = 8
Const Mercury_radius =  18
Const Venus_radius =  30
Const Earth_radius =  48
Const Mars_radius =  66
Const Jupiter_radius =  78
Const Saturn_radius =  90
Const Uranus_radius =  102
Const Neptune_radius =  114
Const Moon_radius =  9

' Define the rotational periods
Const Mercury_period = 87.9691 	' days
Const Venus_period = 224.701 	' days
Const Earth_period = 365.25636 	' days
Const Mars_period = 686.973 	' days
Const Jupiter_period = 4332.589 ' days
Const Saturn_period = 10759.22 	' days
Const Uranus_period = 30687.15 	' days
Const Neptune_period = 60190.03 ' days
Const Moon_period = 27.3217 	' days
Const Tide_period = 44725	' seconds 12:25:25 (may need tweaking by a couple of seconds)

' Define the starting positions
Const Mercury_initial = 73.999	' positions in degrees as at 31/12/1899
Const Venus_initial = 252.971	' relative to 0 degrees being the Earth's northern winter solstice
Const Earth_initial = 10.44
Const Mars_initial = 213.486
Const Jupiter_initial = 144.097
Const Saturn_initial = 176.246
Const Uranus_initial = 152.185
Const Neptune_initial = 357.189
Const Moon_initial = 181.501

Const moon_phase = 13

' Variables to be retained during power down
Dim Integer Var_initialised = false ' Flag to indicated valid saved entries
Dim Integer show_time		' Show/hide time switch
Dim Integer show_moon		' Show/hide moon icon switch
Dim Integer show_earth		' Show/hide earth axis icon switch
Dim Integer show_tide		' Show/hide tide icon switch
Dim Integer Mode		' Real-time / demonstration mode switch
Dim Float Oval			' Circle / oval orbits 1 = circle > 1 = oval
Dim Integer show_symbols	' text / astronomical symbols switch
Dim Integer High_tide(5)	' High tide array
Dim Integer century		' most significant 2 digits of the year
Dim Integer brightness		' LCD brightness
Dim Integer Time_zone		' Offset into Time_zone_table for current time zone

' Variables used to plot planet and moon positions
Dim float Mercury_position
Dim float Venus_position
Dim float Earth_position
Dim float Mars_position
Dim float Jupiter_position
Dim float Saturn_position
Dim float Uranus_position
Dim float Neptune_position
Dim float Moon_position

' General variables
Dim Integer Date_time		' Number of seconds since 01/01/1900 00:00
Dim integer Earth_axis = 0	' The earth's tilt for earth ICON
Dim string old_time length 10	' Used to detect time change
Dim integer startup = True	' Set false once everything has initialised 
Dim Integer Day, Month, Year	' Date used for demonstration mode
Dim Integer key_coord(17, 5)	' Used for key input
Dim String key_caption(17)	' Button text
Dim Integer BP,BP1		' Used for key input
Dim Integer TimeOut		' Used to exit setup screen if no button presses
Dim Integer Time_zone_table(50,1) ' Table holding time zones
Dim Integer Time_zone_GMT	' Offset into Time_zone_table for GMT
Dim Integer Time_zone_entries	' Number of entries in Time_zone_table

' Load the time zone table. On exit:
' Time_zone_table will contain a list of time zones
' Time_zone_GMT will point to the GMT entry
' Time_zone_entries - 1 will point to the last entry
Time_zone_entries = 0
do 
	read Time_zone_table(Time_zone_entries, 0), Time_zone_table(Time_zone_entries, 1)
	if (Time_zone_table(Time_zone_entries, 0) = 0) and (Time_zone_table(Time_zone_entries, 1) = 0) then
		Time_zone_GMT = Time_zone_entries
	end if
	if Time_zone_table(Time_zone_entries, 0) > 14 then
		exit do
	end if
	Time_zone_entries = Time_zone_entries + 1
Loop 

' Restore saved variables after power down
Var Restore

if Var_initialised = false then			' It is set to false if this is the first time ever run
	' This is the first time ever run, so set up variables to defaults
	Var_initialised = True
	show_time = True
	show_moon = True
	show_earth = True
	show_tide = True
	Mode = 0
	Oval = 1
	show_symbols = False
	High_tide(1) = 14			' Liverpool 14/12/20 10:26
	High_tide(2) = 12
	High_tide(3) = 2020
	High_tide(4) = 10
	High_tide(5) = 26
	High_tide(0) = dateconv(High_tide(1),High_tide(2),High_tide(3),High_tide(4),High_tide(5),0)
	century = 20
	brightness = 70
	Time_zone = Time_zone_GMT
	VAR Clear				' Clear any old saved variables
	Save_VAR				' and save
end if

' Try to read the real time clock, but ignore error if not present
on error skip
RTC gettime

' Turn on backlight
PWM 2, 250, brightness

' Draw the fixed parts of the display that don't change
DrawFIxedItems

' Main loop
do
	if (mode = 0) and ((show_tide = True) or (show_time = True)) then
		' Real-time mode and tide or time displayed, so update every minute
		do while (mid$(old_time$,4,2) = mid$(time$,4,2)) and (startup = False)
			' Loop until the minute changes but monitor for screen touches
			if Touch(x) <> -1 then
				Do While Touch(x) <> -1 : Loop	' wait for touch to be released 
				DrawSetupScreen			' Paint the set-up screen and do set-up
				DoSetUp
				DrawFIxedItems			' Re-draw fixed items and exit do
				exit do
			end if
		loop
		
		' Convert date and time to no of seconds since midnight 01/01/1900
		Date_time = dateconv(val(left$(date$,2)),val(mid$(date$,4,2)),Century * 100 + val(mid$(date$,9,2)),val(left$(time$,2)),val(mid$(time$,4,2)),val(mid$(time$,7,2)))
		' Display the date
		text 0,226,Left$(date$,6) + str$(Century,2,0,"0") + right$(date$,2),"LT",1,1,rgb(white)

	else if (mode = 0) and (show_tide = False) and (show_time = False) then
		' Real-time mode and tide and time NOT displayed, so update every hour
		do while (left$(old_time$, 2) = left$(time$, 2)) and (startup = False)
			' Loop until the hour changes but monitor for screen touches
			if Touch(x) <> -1 then
				Do While Touch(x) <> -1 : Loop	' wait for touch to be released 
				DrawSetupScreen			' Paint the set-up screen and do set-up
				DoSetUp
				DrawFIxedItems			' Re-draw fixed items and exit do
				exit do
			end if
		loop
		' Convert date and time to no of seconds since midnight 01/01/1900
		Date_time = dateconv(val(left$(date$,2)),val(mid$(date$,4,2)),Century * 100 + val(mid$(date$,9,2)),val(left$(time$,2)),val(mid$(time$,4,2)),val(mid$(time$,7,2)))
		' Display the date
		text 0,226,Left$(date$,6) + str$(Century,2,0,"0") + right$(date$,2),"LT",1,1,rgb(white)

	else
		' Demonstration mode so update every second
		do while (right$(old_time$, 1) = right$(time$, 1)) and (startup = False)
			if Touch(x) <> -1 then
				Do While Touch(x) <> -1 : Loop	' wait for touch to be released 
				DrawSetupScreen			' Paint the set-up screen and do set-up
				DoSetUp
				DrawFIxedItems			' Re-draw fixed items and exit do
				exit do
			end if
		loop
		Increment_date					' Advance the date by 1 day
		' Convert date and time to no of seconds since midnight 01/01/1900
		Date_time = dateconv(day, month, year, 0, 0, 0)
		' Display the date
		text 0,226,Str$(day,2,0,"0") + "/" + Str$(month,2,0,"0") + "/" + Str$(year,4,0,"0"),"LT",1,1,rgb(white)
	end if
	' Set CPU to fast so we update the screen as quickly as possible
	cpu 48

	if (mode = 0) and (left$(old_time$, 2) <> left$(time$, 2)) then
		' Change of hour, update time from real-time clock if present
		on error skip
		RTC gettime
	end if

	' Show the time if enabled (real-time only)
	if (show_time = true) and (mode = 0) then text 0,212,left$(time$,5),"LT",1,1,rgb(white)

	' Remove the planets and moon from the display
	' This is a lot quicker than CLS and is thus more restful on the eye!
	Unplot_planet Mercury_radius,Mercury_position
	Unplot_planet Venus_radius,Venus_position
	Unplot_planet Earth_radius,Earth_position
	Unplot_planet Mars_radius,Mars_position
	Unplot_planet Jupiter_radius,Jupiter_position
	Unplot_planet Saturn_radius,Saturn_position
	Unplot_planet Uranus_radius,Uranus_position
	Unplot_planet Neptune_radius,Neptune_position
	Unplot_moon Earth_radius,Moon_radius,Earth_position,Moon_position

	' Calculate the new positions
	Mercury_position = Planet_position(Mercury_initial, Mercury_period)
	Venus_position = Planet_position(Venus_initial, Venus_period)
	Earth_position = Planet_position(Earth_initial, Earth_period)
	Mars_position = Planet_position(Mars_initial, Mars_period)
	Jupiter_position = Planet_position(Jupiter_initial, Jupiter_period)
	Saturn_position = Planet_position(Saturn_initial, Saturn_period)
	Uranus_position = Planet_position(Uranus_initial, Uranus_period)
	Neptune_position = Planet_position(Neptune_initial, Neptune_period)
	Moon_position = Planet_position(Moon_initial, Moon_period)

	' Redraw the planets and moon
	Plot_planet Mercury_radius,Mercury_position,"m",1,c.Mercury
	Plot_planet Venus_radius,Venus_position,"V",2,c.Venus
	Plot_planet Earth_radius,Earth_position,"E",4,c.Earth
	Plot_planet Mars_radius,Mars_position,"M",5,c.Mars
	Plot_planet Jupiter_radius,Jupiter_position,"J",6,c.Jupiter
	Plot_planet Saturn_radius,Saturn_position,"S",7,c.Saturn
	Plot_planet Uranus_radius,Uranus_position,"U",8,c.Uranus
	Plot_planet Neptune_radius,Neptune_position,"N",10,c.Neptune
	Plot_moon Earth_radius,Moon_radius,Earth_position,Moon_position,c.Moon

	' Refresh the moon ICON if selected
	if show_moon = True then Plot_Moon_phase Earth_position, Moon_position

	' Refresh the earth ICON if selected
	if show_earth = True then Plot_Sun_latitude

	' Refresh the tide ICON if selected
	if (show_tide = True) and (mode = 0) then Plot_tide

	' Startup is set to true the first time through the loop
	if (startup = True) then
		startup = False			' Set it to false
		' Invalidate old time - This ensures we go round the loop again immediately
		old_time = "        "
	else
		old_time = time$		' Save the time
		' Slow the CPU - 20MHz is the minimum for touch to work!
		cpu 20
	end if
loop


Sub DrawFIxedItems
	cls

	' Draw fixed items
	circle 160,120,Sun_radius,0,1,c.Sun,c.Sun

	if show_moon = True then text 0,43,"Moon","LT",1,1,rgb(white)

	if show_earth = True then 
		text 287,42,"Tilt","LT",1,1,rgb(white)
		line 265,19,278,19,1,rgb(yellow)	' Earth arrow
		line 274,15,278,19,1,rgb(yellow)
		line 274,23,278,19,1,rgb(yellow)
	end if

	if mode = 0 then
		' Real-time mode
		if show_tide = True then text 235,226,"Tide","LT",1,1,rgb(white)
	else
		' Demonstration mode reset starting date to current date
		day = Val(Left$(date$, 2))
		month = Val(mid$(date$, 4, 2))
		Year = century * 100 + Val(right$(date$, 2))
		Date_time = dateconv(day, month, year, 0, 0, 0)
	end if
End Sub


Function dateconv(dd as integer, mm as integer, yy as integer, hh as integer, mi as integer, ss as integer) as integer
	' This converts a date and time to the number of seconds since midnight 1 January 1900 (A very big number)
	local integer x_year
	x_year = yy - 1900		' set the base as 1900
	dateconv = x_year * 365		' Number of days in the year, but don't forget leap years
	if mm < 3 then			' January or February so not a leap year yet!
		x_year = x_year - 1	' Only a leap year from March onwards to the following February
	end if
	' Full leap year formula - NO millennium bug here!!!
	dateconv = dateconv + (x_year \ 4) + ((x_year + 300) \ 400) - (x_year \ 100)
 
	' Now add on the days from the start of the month
	select case mm
		case 2
			dateconv = dateconv + 31
		case 3
			dateconv = dateconv + 59
		case 4
			dateconv = dateconv + 90
		case 5
			dateconv = dateconv + 120
		case 6
			dateconv = dateconv + 151
		case 7
			dateconv = dateconv + 181
		case 8
			dateconv = dateconv + 212
		case 9
			dateconv = dateconv + 243
		case 10
			dateconv = dateconv + 273
		case 11
			dateconv = dateconv + 304
		case 12
			dateconv = dateconv + 334
	end select

	' Now add the day of the month and multiply by the number of seconds in the day.
	' Add on the seconds since midnight and we have a big number!
	dateconv = ((dateconv + dd) * 86400) + (hh * 3600) + (mi * 60) + ss
end Function


sub Plot_planet(radius as integer, position as float, ident as string, symbol as integer, col as integer)
	' Draw the planet's orbit and place the letter / symbol at the appropriate point in colour col
	circle 160,120,radius,1,Oval,col,-1
	if show_symbols = false then
		text cint(160 - Sin(Rad(position)) * radius * Oval),cint(120 - Cos(Rad(position)) * radius),ident,"CM",1,1,RGB(white)
	else
		text cint(160 - Sin(Rad(position)) * radius * Oval),cint(120 - Cos(Rad(position)) * radius),chr$(symbol),"CM",2,1,RGB(white)
	end if
end sub


sub Unplot_planet(radius as integer, position as float)
	' Overwrite the planet's letter / symbol with one of background colour
	' This effectively erases it!
	if show_symbols = false then
		text cint(160 - Sin(Rad(position)) * radius * Oval),cint(120 - Cos(Rad(position)) * radius)," ","CM",1,1,RGB(black)
	else
		text cint(160 - Sin(Rad(position)) * radius * Oval),cint(120 - Cos(Rad(position)) * radius),chr$(15),"CM",2,1,RGB(black)
	end if
end sub


sub Plot_moon(e_radius as integer, m_radius as integer, e_position as float, m_position as float, col as integer)
	' Draw the moon's orbit and place a dot at the appropriate point in colour col
	Local float x1,y1
	x1 = 160 - Sin(Rad(e_position)) * e_radius * Oval
	y1 = 120 - Cos(Rad(e_position)) * e_radius 
	circle cint(x1),cint(y1),m_radius,1,Oval,col,-1
	circle cint(x1 - Sin(Rad(m_position)) * m_radius * Oval),cint( y1 - Cos(Rad(m_position)) * m_radius),2,0,1,col,col
end sub


sub Unplot_moon(e_radius as integer, m_radius as integer, e_position as float, m_position as float)
	' Draw the moon's orbit and place a dot at the appropriate point in the background colour
	' This effectively erases it!
	Plot_moon(e_radius, m_radius, e_position, m_position, RGB(black))
end sub


Function Planet_position(Initial_position as float, Planet_period as float) as float
	' Calculate the planet's current position based upon the starting position on 01/01/1900
	' by using the planet's orbital speed and no. of seconds since 01/01/1900
	' For time zones other than GMT, it is necessary to adjust Date_time.
	Local integer adjustment = Time_zone_table(Time_zone, 0) * 3600 + Time_zone_table(Time_zone, 1) * 60
	Planet_position = Initial_position + (((Date_time - adjustment)* 360) / (86400 * Planet_period))
	Planet_position = Planet_position - (Planet_position \ 360 * 360)
end function

Sub Plot_Moon_phase (Earth_pos as integer, Moon_pos as integer)
	' Draw moon ICON top left of screen
	Local integer EM = cint(Earth_pos - Moon_pos)

	if EM < 0 then EM = EM + 360

	if (EM <= moon_phase) or (EM >= 360 - moon_phase) then
		' Full moon
		circle 19,19,18,0,1,c.Sun,c.Sun

	else if (EM > 180 + moon_phase) and (EM < 360 - moon_phase) then
		' Crescent moon
		circle 19,19,18,0,1,c.Sun,c.Sun
		circle 18 - cint((EM - 180) / 5) ,19,17 + cint(EM - 180) / 20)),0,1,RGB(black),RGB(black)

	else if (EM < 180 - moon_phase) and (EM > moon_phase) then
		' Crescent moon
		circle 19,19,18,0,1,c.Sun,c.Sun
		circle 20 + cint((180 - EM) / 5) ,19,17 + cint(180 - EM) / 20)),0,1,RGB(black),RGB(black)

	else
		' New moon
		circle 19 ,19,18,0,1,RGB(black),RGB(black)
	end if

	' Draw outline
	circle 19 ,19,19,1,1,RGB(gray),-1
end sub


Sub Plot_Sun_latitude
	' Draw the earth ICON top right of screen
	' Paint over existing axis in the background colour
	' This effectively erases it!
 	line 308 - cint(Earth_axis / 11.25), 1, 292 + cint(Earth_axis / 11.25), 37 ,1 ,RGB(black)
	' Re-draw the earth
	circle 300 ,19,18,0,1,RGB(blue),RGB(blue)
	Earth_axis = Earth_position
	if Earth_axis > 180 then Earth_axis = 360 - Earth_axis
	' Draw the axis North to South pole
	line 308 - cint(Earth_axis / 11.25), 1, 292 + cint(Earth_axis / 11.25), 37 ,1 ,RGB(white)
end sub


Sub Plot_tide()
	' Draw tide ICON bottom left of screen
	Local height as float				' Height 0 - 39
	Local direction as integer

	' Calculate the height and direction of the tide
	height = (Date_time - High_tide(0)) mod Tide_period 
	if height < 0 then height = Tide_period + height
	direction = int(40 - (height / 558.9))
	height = int(abs(40 - (height / 558.9)))

	' Draw the sea
	box 287,193,32,46,1,RGB(black),RGB(black)			' Delete previous graphic
	gui bitmap 287,234 - height,&HFFFF7E7E1818,16,3,1,RGB(blue)	' Draw waves
	gui bitmap 303,234 - height,&HFFFF7E7E1818,16,3,1,RGB(blue)
	box 287,237 - height,32,3 + height,1,RGB(blue),RGB(blue)	' Draw sea
	triangle 319,239, 299,239, 319,192, orange,orange		' Draw shore

	if (height <= 1) or (height >= 38) then				' Slack tide no arrow
		line 275,210,275,235,1,rgb(black)
		line 271,231,275,235,1,rgb(black)
		line 279,231,275,235,1,rgb(black)
		line 275,210,271,214,1,rgb(black)
		line 275,210,279,214,1,rgb(black)
	else if direction > 0 then 					' Tide falling downward arrow
		line 275,210,275,235,1,rgb(yellow)
		line 271,231,275,235,1,rgb(yellow)
		line 279,231,275,235,1,rgb(yellow)
		line 275,210,271,214,1,rgb(black)
		line 275,210,279,214,1,rgb(black)
	else if direction < 0 then
		line 275,210,275,235,1,rgb(yellow)			' Tide rising upward arrow
		line 271,231,275,235,1,rgb(black)
		line 279,231,275,235,1,rgb(black)
		line 275,210,271,214,1,rgb(yellow)
		line 275,210,279,214,1,rgb(yellow)
	end if
end sub


Sub test
	' This is a test routine just to verify starting positions on 31/12/1899
	' and make sure the calculations work correctly.
	' The errors are in (fractions of) degrees.
	' NOTE: Time zone must be set to GMT before running test!
	' Mercury:  304.156 304.152  Error:  0.00390625
	' Venus:    162.195 162.199  Error:  0.00390625
	' Earth:   -0.00390625
	' Mars:     292.371 292.371  Error:  0
	' Jupiter:  202.215 202.215  Error:  0.000244141
	' Saturn:   196.918 196.918  Error:  0.000244141
	' Uranus:   309.941 309.941  Error:  0
	' Neptune:  260.863 260.863  Error:  0.000366211
	' Moon:     173.25 353.238   Error:  0.0117188

	Date_time = dateconv(25,10,2020,18,16,0)
	Mercury_position = Planet_position(Mercury_initial, Mercury_period)
	Earth_position = Planet_position(Earth_initial, Earth_period)
	print "Mercury: ",Mercury_position,Earth_position,"  Error: ",abs(Mercury_position- Earth_position)

	Date_time = dateconv(3,6,2020,17,37,0)
	Venus_position = Planet_position(Venus_initial, Venus_period)
	Earth_position = Planet_position(Earth_initial, Earth_period)
	print "Venus:   ",Venus_position,Earth_position,"  Error: ",abs(Venus_position- Earth_position)

	Date_time = dateconv(21,12,2020,10,02,0)
	Earth_position = Planet_position(Earth_initial, Earth_period)
	print "Earth:   ",Earth_position

	Date_time = dateconv(13,10,2020,19,20,0)
	Mars_position = Planet_position(Mars_initial, Mars_period)
	Earth_position = Planet_position(Earth_initial, Earth_period)
	print "Mars:    ",Mars_position,Earth_position,"  Error: ",abs(Mars_position- Earth_position)

	Date_time = dateconv(14,7,2020,8,0,0)
	Jupiter_position = Planet_position(Jupiter_initial, Jupiter_period)
	Earth_position = Planet_position(Earth_initial, Earth_period)
	print "Jupiter: ",Jupiter_position,Earth_position,"  Error: ",abs(Jupiter_position- Earth_position)

	Date_time = dateconv(9,7,2019,16,53,0)
	Saturn_position = Planet_position(Saturn_initial, Saturn_period)
	Earth_position = Planet_position(Earth_initial, Earth_period)
	print "Saturn:  ",Saturn_position,Earth_position,"  Error: ",abs(Saturn_position- Earth_position)

	Date_time = dateconv(31,10,2020,15,18,0)
	Uranus_position = Planet_position(Uranus_initial, Uranus_period)
	Earth_position = Planet_position(Earth_initial, Earth_period)
	print "Uranus:  ",Uranus_position,Earth_position,"  Error: ",abs(Uranus_position- Earth_position)

	Date_time = dateconv(11,9,2020,20,10,0)
	Neptune_position = Planet_position(Neptune_initial, Neptune_period)
	Earth_position = Planet_position(Earth_initial, Earth_period)
	print "Neptune: ",Neptune_position,Earth_position,"  Error: ",abs(Neptune_position- Earth_position)

	Date_time = dateconv(14,12,2020,13,33,0)
	Moon_position = Planet_position(Moon_initial, Moon_period)
	Earth_position = Planet_position(Earth_initial, Earth_period)
	print "Moon:    ",Moon_position,Earth_position,"   Error: ",180 - abs(Moon_position- Earth_position)
end sub


Sub DoSetUp
TimeOut = Timer		' Used to exit setup screen if no key presses
do
	if Timer - TimeOut > 60000 then Exit Sub 	' Test for timeout
	BP = CheckButtonPress(1, 10)			' Get button press
	Select Case BP
		Case 1	' Set date, time , time zone
			CheckButtonRelease BP
			GetDateTime 1
			GetDateTime 0
			DrawTimeZoneScreen
			do
				if Timer - TimeOut > 60000 then Exit Sub
				BP1 = CheckButtonPress(1, 3)	' Get button press
				Select Case BP1
					Case 1	' Increase time zone
						CheckButtonRelease BP1
						Time_zone = Time_zone + 1
						if Time_zone_table(Time_zone, 0) > 14 then Time_zone = 0
						DrawTimeZoneScreen

					Case 2	' Decrease time zone
						CheckButtonRelease BP1
						Time_zone = Time_zone - 1
						if Time_zone < 0 then Time_zone = Time_zone_entries - 1
						DrawTimeZoneScreen

					Case 3	' Exit time zone screen
						CheckButtonRelease BP1
						Exit do
				End Select
			Loop
			DrawSetupScreen

		Case 2	' Alter display brightness
			brightness = brightness + 10
			if brightness > 100 then brightness = 10
			PWM 2, 250, brightness
			DrawSetupScreen

		Case 3	' Local tide screen
			CheckButtonRelease BP
			DrawTideScreen
			do
				if Timer - TimeOut > 60000 then Exit Sub
				BP1 = CheckButtonPress(1, 3)
				Select Case BP1
					Case 1	' Set high tide date and time
						CheckButtonRelease BP1
						GetDateTime 3
						DrawTideScreen

					Case 2	' Show / hide tide display
						CheckButtonRelease BP1
						if Show_tide = 0 then
							Show_tide = 1
						else
							Show_tide = 0
						end if
						DrawTideScreen

					Case 3	' Exit high tide screen
						CheckButtonRelease BP1
						Exit do
				End Select
			Loop
			DrawSetupScreen

		Case 4	' Real-time / demonstration mode
			CheckButtonRelease BP
			if mode = 0 then
				mode = 1
			else
				mode = 0
			end if
			DrawSetupScreen

		Case 5	' Show / hide moon
			CheckButtonRelease BP
			if show_moon = false then
				show_moon = true
			else
				show_moon = false
			end if
			DrawSetupScreen

		Case 6	' Circular / elliptical orbits
			CheckButtonRelease BP
			if oval = 1 then
				oval = 1.14
			else
				oval = 1
			end if
			DrawSetupScreen

		Case 7	' Show / hide time display
			CheckButtonRelease BP
			if show_time = false then
				show_time = true
			else
				show_time = false
			end if
			DrawSetupScreen

		Case 8	' Show planets as letters or symbols
			CheckButtonRelease BP
			if show_symbols = false then
				show_symbols = true
			else
				show_symbols = false
			end if
			DrawSetupScreen

		Case 9	' Show / hide earth
			CheckButtonRelease BP
			if show_earth = false then
				show_earth = true
			else
				show_earth = false
			end if
			DrawSetupScreen

		Case 10	' Exit setup screen
			CheckButtonRelease BP
			Save_VAR
			startup = True
			exit sub
	End select
loop
end sub


Sub DrawSetupScreen
	' Paint the buttons on the setup screen
	CLS
	DrawButton 1, 0, 0, 0, 155, 38, RGB(yellow), "Set date"
	DrawButton 2, 0, 164, 0, 155, 38, RGB(yellow), "LCD " + str$(brightness,3) +"%"
	DrawButton 3, 0,  0, 48, 155, 38, RGB(yellow), "Set tide"

	if mode = 0 then
		DrawButton 4, 0, 164, 48, 155, 38, RGB(yellow), "Real time"
	else
		DrawButton 4, 0, 164, 48, 155, 38, RGB(yellow), "Fast mode"
	end if

	if show_moon = True then
		DrawButton 5, 0, 0, 96, 155, 38, c.yes, "Show moon"
	else
		DrawButton 5, 0, 0, 96, 155, 38, c.no, "Hide moon"
	end if

	if oval = 1 then
		DrawButton 6, 0, 164, 96, 155, 38, RGB(yellow), "Circle"
	else
		DrawButton 6, 0, 164, 96, 155, 38, RGB(yellow), "Oval"
	end if

	if show_time = True then
		DrawButton 7, 0, 0, 144, 155, 38, c.yes, "Show time"
	else
		DrawButton 7, 0, 0, 144, 155, 38, c.no, "Hide time"
	end if

	if show_symbols = True then
		DrawButton 8, 0, 164, 144, 155, 38, RGB(yellow), "Symbols"
	else
		DrawButton 8, 0, 164, 144, 155, 38, RGB(yellow), "Text"
	end if

	if show_earth = True then
		DrawButton 9, 0, 0, 192, 170, 38, c.yes, "Show earth"
	else
		DrawButton 9, 0, 0, 192, 170, 38, c.no, "Hide earth"
	end if

	DrawButton 10, 0, 179, 192, 140, 38, RGB(white), "Save"
  End If
End Sub


Sub DrawTideScreen
	' Paint the tide setup screen
	CLS
	Text 160,0,"Set local high tide","CT",1,2,RGB(white)
	DrawButton 1, 0, 0, 40, 192, 38, RGB(yellow), "Date & time"

	if Show_tide = True then
		DrawButton 2, 0, 0, 88, 155, 38, c.yes, "Show tide"
	else
		DrawButton 2, 0, 0, 88, 155, 38, c.no, "Hide tide"
	end if
	DrawButton 3, 0, 0, 136, 155, 38, RGB(white), "Save"
	Text 160,200,Str$(High_tide(1),2,0,"0") + "/" + Str$(High_tide(2),2,0,"0") + "/" + Str$(High_tide(3),4,0,"0") + " " + Str$(High_tide(4),2,0,"0") + ":" + Str$(High_tide(5),2,0,"0"),"CT",1,2,RGB(white)
End sub


Sub DrawTimeZoneScreen
	' Paint the tide setup screen
	CLS
	Text 160,0,"Set your time zone","CT",1,2,RGB(white)
	Text 160,40, "GMT " + str$(Time_zone_table(Time_zone, 0),-3,0,"0") + ":" + str$(Time_zone_table(Time_zone, 1),2,0,"0"),"CT",1,2,RGB(white)
	DrawButton 1, 0, 0, 80, 155, 38, RGB(yellow), "Increase"
	DrawButton 2, 0, 0, 130, 155, 38, RGB(yellow), "Decrease"
	DrawButton 3, 0, 0, 180, 155, 38, RGB(white), "Save"
end sub


Sub Save_VAR
	' Save all the variables that need to be preserved during power down
	VAR Save Var_initialised, show_time, show_moon, show_earth, show_tide, Mode, Oval, show_symbols, High_tide(), century, brightness, Time_zone
End Sub


Sub GetDateTime dt As Integer
	' menu to get a date or time
	' dt = 0 Set time; dt = 1 Set date; dt = 2 Set high tide time; dt = 3 Set high tide date
	Static Integer digit, i, b, d, m, y, c, hh, mm
	Local String SCap(9) LENGTH 8 = ("7","8","9","4","5","6","1","2","3","0")
	Local String str Length 12, strold Length 12
	Const bh = MM.VRes\5, bw = MM.HRes\5

	If (dt = 0) or (dt = 1)  Then
		on error skip
		RTC gettime
	end if

	If (dt = 1) or (dt = 3)  Then strold = "dd/mm/yyyy" Else strold = "hh:mm"
	str = strold
	digit = 1

	CLS
	For i = 0 To 8
		DrawButton i, 0, bw + bw * (i Mod 3) + 2, bh + bh * (i \ 3) + 2, bw - 4, bh - 4, c.button, SCap(i)
	Next i
	DrawButton 9, 0, bw*2 + 2, bh*4 + 2, bw - 4, bh - 4, c.button, "0"
	DrawButton 10, 0, 0, bh*4 + 2, bw * 2 - 4, bh - 4, c.delete, "Delete"
	DrawButton 11, 0, bw*3 + 2, bh*4 + 2, bw * 2 - 4, bh - 4, c.save, "Save"
	DisplayDateTimeStr str

	Do
		' Flash the cursor
		If (Timer > 700) and (digit <= len(strold)) Then Line (13 - len(strold)) * 8 + digit * 24, 40, (16 - len(strold)) * 8 + digit * 24, 40, 2, c.entry : Timer = 0
		b = CheckButtonPress(0, 11)
		If (Timer > 500) Or (b >= 0) Then Line (13 - len(strold)) * 8 + digit * 24, 40,(16 - len(strold)) * 8 + digit * 24, 40, 2, 0
			Select Case b
				Case 0 To 9	' Digits 0 - 9 in order SCap
					If digit <= len(strold) Then
						str = Left$(str, digit-1) + SCap(b) + Mid$(str, digit+1, 9)
						digit = digit + 1
						If digit = 3 Or digit = 6 Then digit = digit + 1
						DisplayDateTimeStr str
					End If
					Pause 150
					CheckButtonRelease b
				Case 10	' Delete button
					If digit > 1 Then
						digit = digit - 1
						If digit = 3 Or digit = 6 Then digit = digit - 1
						str = Left$(str, digit-1) + Mid$(strold, digit)
						DisplayDateTimeStr str
					End If
					Pause 150
					CheckButtonRelease b
				Case 11	' Save button
					Pause 150
					CheckButtonRelease b
					If (dt = 1) or (dt = 3) Then
						d = Val(Left$(str, 2))
						m = Val(Mid$(str, 4, 2))
						c = Val(Mid$(str, 7, 2))
						y = Val(Right$(str, 2))
						' Validate date format
						If (c < 19) Or (m < 1) Or (m > 12) Or (d < 1) Or (d > 31) Then
							MessageBox "Invalid", "Date"
						Else
							if dt = 1 then	' Save date
								Date$ = Str$(d) + "/" + Str$(m) + "/" + Str$(y)
								Century = c
								on error skip
								RTC settime y, m, d, val(left$(time$,2)), val(mid$(time$,4,2)), 0
							Else		' Save high tide date
								High_tide(1) = d
								High_tide(2) = m
								High_tide(3) = c * 100 + y
								GetDateTime 2
							End if
						End If
					Else
						hh = Val(Left$(str, 2))
						mm = Val(Right$(str, 2))
						' Validate time format
						If (Mid$(str, 5, 1) = "m") Or (mm > 59) Or (hh > 23) Then
							MessageBox "Invalid", "Time"
						Else
							if dt = 0 then	' Save time
								Time$ = Str$(hh) + ":" + Str$(mm) + ":" + "00"
								on error skip
								RTC settime val(mid$(date$,9,2)), val(mid$(date$,4,2)), val(left$(date$,2)), hh, mm, 0
							Else		' Save high tide time
								High_tide(4) = hh
								High_tide(5) = mm
								High_tide(0) = dateconv(High_tide(1),High_tide(2),High_tide(3),High_tide(4),High_tide(5),0)
							End if
						End if
					End If
					Exit sub
			End Select
		End If
	Loop
End Sub


'''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
' Draw buttons and get button presses
'
' The subroutine DrawButton will draw a button (normally used when drawing
' the screen for input).
'
' The function CheckButtonPress() will check if a button has been touched.
' If it has it will set it to selected (reverse video) and return with the
' button's number.
'
' The subroutine CheckButtonRelease will wait for the touch to be released
' and will then draw the button as normal.
'
' These routines use the global arrays key_coord() and key_caption() to
' track the coordinates and size of each button and save its caption.
'
'''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''

' draw a button
Sub DrawButton n As Integer, bmode As Integer, x As Integer, y As Integer, w As Integer, h As Integer, c As Integer, s As String
	Local Integer bc, fc

	If bmode = 0 Then
		key_coord(n,0) = x : key_coord(n,1) = y : key_coord(n,2) = w : key_coord(n,3) = h
		key_coord(n,4) = c : key_caption(n) = s
	EndIf

	If bmode > 1 Then
		bc = key_coord(n,4) : fc = 0    ' draw in reverse video if it is being touched
	Else
		bc = 0 : fc = key_coord(n,4)    ' a normal (untouched) button
	EndIf

	RBox key_coord(n,0), key_coord(n,1), key_coord(n,2), key_coord(n,3), , key_coord(n,4), bc)
	Text key_coord(n,0) + key_coord(n,2)/2, key_coord(n,1) + key_coord(n,3)/2, key_caption(n), CM,1 ,2, fc, bc
End Sub


' check if a button has been touch and animate the button's image
' returns the button's number
Function CheckButtonPress(startn As Integer, endn As Integer) As Integer
	Local Integer xt, yt, n

	CheckButtonPress = -1
	If Touch(x) <> -1 Then
		' we have a touch
		xt = Touch(x)
		yt = Touch(y)
		' scan the array key_coord() to see if the touch was within the
		' boundaries of a button
		For n = startn To endn
				If xt > key_coord(n,0) And xt < key_coord(n,0) + key_coord(n,2) And yt > key_coord(n,1) And yt < key_coord(n,1) + key_coord(n,3) Then
				' we have a button press
				' draw the button as pressed
				DrawButton n, 2
			CheckButtonPress = n
			Exit For
			EndIf
    		Next n
  	EndIf
End Function


' wait for the touch to be released and then draw the button as normal
Sub CheckButtonRelease n As Integer
	' if a button is currently down check if it has been released
	 Do While Touch(x) <> -1 : Loop   ' wait for the button to be released
	DrawButton n, 1                  ' draw the button as normal (ie, not pressed)
End Sub


'''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
' this handy routine draws a message box with an OK button
' then waits for the button to be touched
'''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
Sub MessageBox s1 As String, s2 As String
	Local Integer w
	If Len(s1) > Len(s2) Then w = Len(s1) Else w = Len(s2)
	w = w * 8     ' get the width of the text (used for the box width)

	' draw the box and the message in it
	RBox MM.HRes/2 - w - 20, 60, w * 2 + 40, 130, , c.entry, 0
	Text MM.HRes/2, 70, s1, CT, 1, 2, RGB(white)
	Text MM.HRes/2, 100, s2, CT, 1, 2, RGB(white)

	' draw the OK button
	RBox 110, 140, 100, 34, , c.button
	Text MM.HRes/2, 157, "OK", CM, 1, 2, c.button

	' wait for the button to be touched
	Do While Not (Touch(x) > 110 And Touch(x) < 210 And Touch(y) > 140 And Touch(y) < 180) : Loop

	' draw the OK button as depressed
	RBox 110, 140, 100, 34, , c.button, c.button
	Text MM.HRes/2, 157, "OK", CM, 1, 2, 0, c.button

	' wait for the touch to be removed
	Do While Touch(x) <> -1 : Loop
End Sub


' utility sub to display formatted date or time
Sub DisplayDateTimeStr s As String
	Local Integer i, j = (13 - len(s)) * 8
	For i = 1 To len(s)
		If i = 3 Or i = 6 Then
			Text j + 24 * i, 8, Mid$(s, i, 1), , 1, 2, RGB(white), 0
		ElseIf Mid$(s, i, 1) > "9" Then
			Text j + 24 * i, 8, Mid$(s, i, 1), , 1, 2, c.ghosttext, 0
		Else
			Text j + 24 * i, 8, Mid$(s, i, 1), , 1, 2, c.entry, 0
		EndIf
	Next i
End Sub


Sub Increment_date
	' Increments the date in demonstration mode
	day = day + 1
	if (day = 31) and ((month = 4) or (month = 6) or (month = 9) or (month = 11)) then
		' Month roll over for April, June, September, November
		month = month + 1
		day = 1
	end if

	if (day = 32) and ((month = 1) or (month = 3) or (month = 5) or (month = 7) or (month = 8) or (month = 10) or (month = 12)) then
		' Month roll over for January, March, May, July, August, October, December
		month = month + 1
		day = 1
	end if

	' Test for leap years with full formula
	if ((year mod 4 = 0) and ( year mod 100 <> 0)) or ( year mod 400 = 0)  then
		if (day = 30) and (month = 2)  then
			' Leap year February has 29 days
			month = 3
			day = 1
		end if
	else
		if (day = 29) and (month = 2) then
			' Not a leap year February has 28 days
			month = 3
			day = 1
		end if
	end if

	if month > 12 then
		' Year roll over
		month = 1
		year = year + 1
	end if
end sub


' Time Zones
Data -12, 00
Data -11, 00
Data -10, 00
Data -09, 30
Data -09, 00
Data -08, 30
Data -08, 00
Data -07, 00
Data -06, 00
Data -05, 00
Data -04, 00
Data -03, 30
Data -03, 00
Data -02, 30
Data -02, 00
Data -01, 30
Data -01, 00
Data 00, 00
Data 01, 00
Data 02, 00
Data 03, 00
Data 03, 30
Data 04, 00
Data 04, 30
Data 05, 00
Data 05, 30
Data 05, 45
Data 06, 00
Data 06, 30
Data 06, 45
Data 07, 00
Data 07, 30
Data 08, 00
Data 09, 00
Data 09, 30
Data 10, 00
Data 10, 30
Data 11, 00
Data 12, 00
Data 12, 45
Data 13, 00
Data 13, 45
Data 14, 00
Data 99, 99


' Zodiacfont.bas
' Font type    : Full (15 characters)
' Font size    : 12x12 pixels
' Memory usage : 274 bytes
' &H01  Mercury
' &H02  Venus
' &H03  Earth
' &H04  Earth (alternative)
' &H05  Mars
' &H06  Jupiter
' &H07  Saturn
' &H08  Uranus
' &H09  Uranus (alternative)
' &H010 Neptune 
' &H011 Square
' &H012 Neptune (alternative)
' &H013 Sun
' &H014 Pluto
' &H015 Blank

DefineFont #2
  0F010C0C 28710400 222004A1 88401002 07200007 00002000 02C21870 20011440
  00078C21 20000720 07200000 01072000 0124208C 18021240 000070C0 24810A70
  7F212422 212221F4 70800A24 00070000 81070530 0124208C 18021240 000070C0
  22210C02 200A2421 200212A0 02F07F42 70000200 C32C7002 01122002 20042220
  00000882 A8000720 208C0107 12420124 70C01802 22230600 E23F2222 22222222
  05883065 04002000 22124221 0A242122 00027080 20800F20 40FF0700 14400114
  01144001 40011440 0440FF17 30844224 52A00446 F07F2106 20000520 08700000
  24200481 01144201 08042120 07007080 048440F0 7F084440 04400004 FF074000
  00000000 00000000 00000000 00000000 00000000
End DefineFont


CSUB Triangle integer, integer, integer, integer, integer, integer, integer, integer
00000008
27BDFFF8 AFBF0004 00852023 03E42021 ACC40000 8FBF0004 03E00008 27BD0008
27BDFF98 AFBF0064 AFBE0060 AFB7005C AFB60058 AFB50054 AFB40050 AFB3004C
AFB20048 AFB10044 AFB00040 00808021 00A09821 00C09021 00E0A021 8FB50078
00002021 3C059D00 24A50078 27A60018 0411FFE3 00000000 8FA30018 8FA20080
5040000E 3C029D00 5200000C 3C029D00 1240000A 3C029D00 12A00008 00000000
52600007 8C4200BC 12800003 8FA2007C 54400008 8E110000 3C029D00 8C4200BC
3C049D00 24840490 0040F809 00832021 8E110000 8E770000 8E560000 8E9E0000
8EB50000 AFB50034 8FA3007C 8C750000 8FA30080 8C620000 7C42B800 AFA20038
8FA20084 10400006 3C109D00 8C540000 2402FFFF 16820020 03D7102A 3C109D00
24120001 AFB20010 8FA30038 AFA30014 8E020050 02202021 02E02821 02C03021
0040F809 03C03821 AFB20010 8FA20038 AFA20014 8E020050 02C02021 03C02821
8FA60034 0040F809 02A03821 AFB20010 8FA30038 AFA30014 8E020050 8FA40034
02A02821 02203021 0040F809 02E03821 100000B4 8FBF0064 10400008 02BE102A
02E01021 03C0B821 0040F021 02201021 02C08821 0040B021 02BE102A 10400008
03D7102A 03C01021 02A0F021 0040A821 02C01021 8FB60034 AFA20034 03D7102A
10400006 02E01021 03C0B821 0040F021 02201021 02C08821 0040B021 16F5001D
03D51026 02D1102A 14400006 02203821 0236102A 10400005 8FA30034 10000002
02C03821 02C08821 8FA30034 0071102A 14400003 00E3102A 10000002 0062380B
8FB10034 3C029D00 8C420048 00F73821 AFB40010 8C420000 02202021 02E02821
02203021 0040F809 00F13823 1000007D 8FBF0064 0002102B 03C21023 AFA20028
0057102A 1440002B 02E08021 02D11023 AFA2002C 8FA30034 00711823 AFA30030
00009021 00009821 03D71023 AFA20020 02B71823 AFA30024 8FA30020 0263001A
006001F4 00002012 00912021 8FA30024 0243001A 006001F4 00003012 00D13021
00C4102A 10400004 3C039D00 00801021 00C02021 00403021 8C620048 AFB40010
8C420000 02002821 0040F809 02003821 26100001 8FA2002C 02629821 8FA30030
02439021 8FA30028 0070102A 1040FFE3 8FA30020 02B0102A 1440002B 8FA20034
00561023 AFA20028 021E9023 72429002 8FA20034 00511023 AFA2002C 02179823
70539802 02BE1023 AFA20020 02B71823 AFA30024 8FA30020 0243001A 006001F4
00002012 00962021 8FA30024 0263001A 006001F4 00003012 00D13021 00C4102A
10400004 3C039D00 00801021 00C02021 00403021 8C620048 AFB40010 8C420000
02002821 0040F809 02003821 26100001 8FA20028 02429021 8FA3002C 02B0102A
1040FFE4 02639821 8FA20038 1054001C 3C109D00 24120001 AFB20010 AFA20014
8E020050 02202021 02E02821 02C03021 0040F809 03C03821 AFB20010 8FA30038
AFA30014 8E020050 02C02021 03C02821 8FA60034 0040F809 02A03821 AFB20010
8FA20038 AFA20014 8E020050 8FA40034 02A02821 02203021 0040F809 02E03821
8FBF0064 8FBE0060 8FB7005C 8FB60058 8FB50054 8FB40050 8FB3004C 8FB20048
8FB10044 8FB00040 03E00008 27BD0068 61766E49 2064696C 6D6D6F43 0D646E61
0000000A
End CSUB

