//##############################################################################
//    filename:        Inductor_Tester_Meas.h
//##############################################################################
//    header file LCD library for the Inductor Tester Measurement code ...
//##############################################################################
//      Author:            P.Prosser
//      Company:           
//      Adapted:            
//      Revision:           0.0 (for XC32)
//      Date:         	    Aug 2024
//     	Compiler            
//      MPLAB X             
//      MCC                 
//##############################################################################
//
//==============================================================================
//      LCD library FUNCTIONS:
//

#ifndef Inductor_Tester_Meas_H
#define Inductor_Tester_Meas_H

#include <stdint.h>
#include <stdbool.h>

#include "Inductor_Tester.h"

#define Discharge_Time_Ms 250

#define Discharge_Inactive DUT_Discharge_Clear()
#define Discharge_Active DUT_Discharge_Set()
#define Pulser_D_Hi  FF_D_Clear()  /* This is inverted by the transistor */
#define Pulser_D_Lo  FF_D_Set()  /* This is inverted by the transistor */
#define Pulser_CLK_Hi FF_CLK_Clear() /* This is inverted by the transistor */
#define Pulser_CLK_Lo FF_CLK_Set() /* This is inverted by the transistor */
#define Discharge_Recovery_Time_Ms 75
#define Short_Duration 3
#define Short_Latency 20

#define Pulser_Time 4  /* need to slow the micro down as the BJT inverter is not fast. a 1us puilse is too narrow. use 4* this */

/* current set */
#define Zero_Current_Output 0.0
#define DUT_Measurement_Channel 1
#define Current_Cutoff_Channel 0
#define Current_Sense_Gain 20
#define Current_Cuttoff_Voltage_Gain (4.7/(4.7+2.7)) //voltage divider on the comparator
#define Current_Sense_Resistance 0.005
#define Current_Sense_Resistance_High_Gain 1.005
#define Current_Scaling_From_Low_Gain_ADC ((float) (LTestData.ADC_Full_Scale_V/(float)ADC_MAX_COUNT)/((float)Current_Sense_Gain*(float)Current_Sense_Resistance))
#define Current_Scaling_From_High_Gain_ADC ((float) (LTestData.ADC_Full_Scale_V/(float)ADC_MAX_COUNT)/((float)Current_Sense_Resistance_High_Gain))

/* ADC Inputs*/
/*  Voltage Sense = AN0 = ADC0, SH0ALT = 00                            */
/*  Resistance Low Gain = ADC0, SH0ALT = 01                            */
/*  Resistance Hi Gain  = ADC0, SH0ALT = 11                            */
/*  Inductor Current MEasurement = ADC1, SH1ALT = 00                   */
#define ADC0_Voltage_Sense_Channel_Select 0b00
#define ADC0_Resistance_Low_Gain_Channel_Select 0b01
#define ADC0_Resistance_Hi_Gain_Channel_Select 0b11
#define ADC1_Inductor_Current_Channel_Select 0b00
#define ADC1_Inductor_Current_High_Gain_Channel_Select 0b01

/* Measurement Parameters*/
#define Test_Current_High_Range DAC_ISet_10mA_Cal  /* 10mA */
#define Test_Current_Medium_Range 0.1  /* 100mA */
#define Test_Current_Low_Range 1.0  /* 1A */
#define Sample_Rate_High_Speed 3750000
#define Sample_Rate_Medium_Speed 375000
#define Sample_Rate_Low_Speed 37500
//#define 
/* Settling Times */
#define Resistance_Settle_High_Res CORETIMER_DelayMs (200) //2V, 20000uF, 0.01A test current = 0.01s
#define Resistance_Settle_Medium_Res CORETIMER_DelayMs (200) //2V, 20000uF, 0.1A test current = 0.01s
#define Resistance_Settle_MediumLow_Res CORETIMER_DelayMs (200) //2V, 20000uF, 0.1A test current = 0.01s
#define Resistance_Settle_Low_Res CORETIMER_DelayMs (200) //2V, 20000uF, 0.1A test current = 0.01s
#define Resistance_Settle_LowLow_Res CORETIMER_DelayMs (200) //2V, 20000uF, 0.1A test current = 0.01s
#define Resistance_Settle CORETIMER_DelayMs (20) //10V, 1H, 0.1A test current = 0.01s
#define Resistance_Post_Meas_Settle CORETIMER_DelayMs (1)
/* Settling Times  for FAST meas*/
#define Resistance_Fast_Settle CORETIMER_DelayMs (20) //Allow the current source to stabilise
#define Resistance_Fast_Post_Meas_Channel_Sel_Settle CORETIMER_DelayMs (1)
#define Restsiance_Fast_Delay_To_Meas CORETIMER_DelayUs (30)

/* one full buffer is 128 samples / 3750000 seconds = 34us Use 140 which is 4 buffers*/
#define Capacitance_Settle CORETIMER_DelayMs (20) //let current source settle
#define ADC0_Wait_1_Buffer CORETIMER_DelayUs (140)
#define ADC1_Wait_1_Buffer CORETIMER_DelayUs (140)
#define Inductor_Test_Wait_One_Buffer     bufferA_Full = false; bufferB_Full = false; do {} while ((!bufferA_Full) && (!bufferB_Full));
#define Resistance_Range_Crossover 0.08
#define High_Resistance_Threshold (Resistance_Range_Crossover*LTestData.ADC_Full_Scale_V/LTestData.ISet_CAL_10mA)
#define High_Resistance_Max (0.99* LTestData.ADC_Full_Scale_V/LTestData.ISet_CAL_10mA)
#define Medium_Resistance_Threshold (Resistance_Range_Crossover*LTestData.ADC_Full_Scale_V/LTestData.ISet_CAL_100mA)
#define MediumLow_Resistance_Threshold (Resistance_Range_Crossover*LTestData.ADC_Full_Scale_V/(LTestData.ISet_CAL_10mA * Current_Sense_Gain))
#define Low_Resistance_Threshold (Resistance_Range_Crossover*LTestData.ADC_Full_Scale_V/(LTestData.ISet_CAL_100mA* Current_Sense_Gain))
#define LowLow_Resistance_Threshold (Resistance_Range_Crossover*LTestData.ADC_Full_Scale_V/(LTestData.ISet_CAL_100mA* Current_Sense_Gain))

/* Measurement Parameters */
#define Rail_Sense_Top_Res  10000 /* ohms */
#define Rail_Sense_Bottom_Res  4700 /* ohms */
#define Test_Rail_Voltage_Scaling ((float)Rail_Sense_Bottom_Res/( (float)Rail_Sense_Bottom_Res + (float)Rail_Sense_Top_Res)) 
#define Inductance_Current_Settle CORETIMER_DelayMs (20) //Allow the current source to stabilise
#define Inductance_Current_Decay_Time CORETIMER_DelayMs (50) //Allow the current source to stabilise
#define ADC_RBFIE_mask 0x03000000   // bits 25 and 24 of register ADCDSTAT


void Discharge_DUT();
void Initialise_Pulser();
void Pulse_Start();
void Pulse_Start_With_Timeout(int Max_Pulse_Time);
void Clear_Test_Current();
void Set_Test_Current(float Current_Set);
void Set_Cutoff_Current(float Current_Set) ;
void Clear_Cutoff_Current();
void Select_Input_To_Read(int Input_To_Read);
void Enable_ADC_0();
void Disable_ADC_0();
void Enable_ADC_1();
void Disable_ADC_1();
void Select_ADC0_Input_To_Read(int Input_To_Read);
void Select_ADC1_Input_To_Read(int Input_To_Read);
void Measure_DUT_Voltage_High_Gain();
void Measure_DUT_Voltage_Low_Gain();
void Measure_Test_Voltage_Rail();
void Measure_Current_Sense_Voltage();
float ADC0_Low_Gain();
float ADC0_High_Gain();
float Current_Sense_Voltage();
float Measure_High_Resistance();
float Measure_Medium_Resistance();
float Measure_MedLow_Resistance();
float Measure_Low_Resistance();
float Measure_LowLow_Resistance();
float Measure_High_Resistance_Fast();
float Measure_Medium_Resistance_Fast();
float Measure_MedLow_Resistance_Fast();
float Measure_Low_Resistance_Fast();
float Measure_LowLow_Resistance_Fast();
float ADC0_Low_Gain_Quick();
float Current_Sense_Voltage_Quick();
void Short_DUT_1ms(); 
void Short_DUT(); 
void Un_Short_DUT(); 
void ADC_Sample_Rate();
void ADC_Cap_Buffer_Capture(long int Sample_Rate, float Test_Current, uint16_t *Meas_Ptr, int *Sample_Cnt , bool* Overflow);
void ADC_Cap_Buffer_Find_Thresholds(uint16_t *Meas_Ptr, int* First_Sample_Meas, int* Final_Sample_Meas);
void ADC_Ind_Buffer_Capture(long int Sample_Rate, uint16_t *Meas_Ptr, uint16_t *Meas_Ptr_1);
void ADC_Ind_Buffer_Find_Thresholds(uint16_t *Meas_Ptr, uint16_t *Meas_Ptr_1, int* First_Sample_Meas, int* Final_Sample_Meas);
void Update_Current_Sense_Voltage();
bool DUT_Is_A_Capacitor();
void ADC_Ind_Sat_Buffer_Capture(float Cutoff_Current, long int Sample_Rate, uint16_t *Meas_Ptr, uint16_t *Meas_Ptr_1);
void ADC_Ind_Sat_Buffer_Find_Thresholds(float Test_Current, uint16_t *Meas_Ptr, uint16_t *Meas_Ptr_1, int* First_Sample_Meas, int* Final_Sample_Meas);


#endif