//interface for HD44780 LCD using 4 bit mode
#include "hd44780.h"

void LCD_init(void){
    //turn on power
    TRISDbits.TRISD6=0; //LCD_PWR
    LCD_PWR=1;
    //set up pins and do LCD initialisation
    LCD_EN=0;
    LCD_RS=0;           //default to commands for init
    TRISDbits.TRISD7=0; //D7
    TRISBbits.TRISB0=0; //D6
    TRISBbits.TRISB1=0; //D5
    TRISBbits.TRISB2=0; //D4
    TRISBbits.TRISB3=0; //EN
    TRISBbits.TRISB4=0; //RS    
    __delay_ms(10);        //powerup delay
    LCD_D7=0;
    LCD_D6=0;
    LCD_D5=1;
    LCD_D4=0;   //0=> 4-bit mode
    LCD_EN=1;
    LCD_EN=0;   //clock
    __delay_ms(1);
    LCD_RS=1;           //default to data
    LCD_cmd(0b00000001);         //clear
    __delay_ms(10);
    LCD_cmd(0b00000110);         //increment on write, no shift
    LCD_cmd(0b00001100);         //display on, cursor off
    LCD_cmd(0b00101000);         //4 bit mode, 2-line font
    LCD_cmd(0b10000000);         //start of data ram (ready for output)
}

void LCD_deinit(void){      //for low power mode
    LCD_EN=0;
    LCD_RS=0;
    LCD_D7=0;
    LCD_D6=0;
    LCD_D5=0;
    LCD_D4=0;   //all low to avoid parasitic drain
    LCD_PWR=0;  //off    
}

void LCD_data(unsigned char d){
    LCD_D7=(d&128)?1:0;
    LCD_D6=(d&64)?1:0;
    LCD_D5=(d&32)?1:0;
    LCD_D4=(d&16)?1:0;
    LCD_EN=1;
    LCD_EN=0;   //clock
    LCD_D7=(d&8)?1:0;
    LCD_D6=(d&4)?1:0;
    LCD_D5=(d&2)?1:0;
    LCD_D4=(d&1)?1:0;   //needs 4 bytes less than the obvious (d&1)
    LCD_EN=1;
    LCD_EN=0;   //clock    
    __delay_ms(1);      //all commands/data take max 37us except return home (0x2)
}

void LCD_cmd(unsigned char c){
    LCD_RS=0;           //clear for command
    LCD_data(c);
    LCD_RS=1;           //default to data    
}

void LCD_print(const char *c){
    while(*c){
        LCD_data(*c++);
    }    
}

void LCD_setpos(unsigned char x){
    LCD_cmd((x & 0x7F)|0x80);
}

void LCD_cgram(const unsigned char* p){
    unsigned char i,j;
    LCD_cmd(0b01000000);    //start of CGRAM
    for(i=0;i<64;i++){
        j=p[i];
        LCD_data(j);     //fill entire CGRAM
    }
    LCD_cmd(0b10000000);    //start of data ram (ready for output)    
}

unsigned long places(signed char n){
    unsigned long r=1;
    while(n--){r=r*10;}
    return r;    
}

char getDigit(unsigned long n, signed char p){  //get digit at pth decimal place (0 = ones)
    signed char res,pstart,zf;
    zf=0;
    pstart=9;
    while(pstart>=p){
        res=0;
        while(n>=places(pstart)){
            n=n-places(pstart);
            res++;
            zf=1;   //non zero digit detected
        }
        pstart=pstart-1;
    }
    if((zf==0)&&(res==0)&&(p!=0)){return F_SPACE;}  //zero blanking, ones position is never blanked
    return res+'0';
}

char lzFix(char c){ //suppress leading zero suppression for display reasons
    if(c==F_SPACE){return '0';}
    return c;
}

void LCD_ulong(unsigned long n, char p){    //output unsigned long to p places using getDigit
    while(p--){
        LCD_data(getDigit(n,p));
    }
}

void LCD_clr(void){                 //separate function to include necessary delay, also does a home
    LCD_cmd(LCD_CLR);
    __delay_ms(2);
}