//USB Cable Tester
//Set PCB version in io.h #define PCB_REVC or #define PCB_REVD

#include "config.h"
#include <xc.h>
#include <string.h> //for strlen
#include "io.h"
#include "hd44780.h"
#include "util.h"
#include "graphics.h"
#include "uart.h"
#include "menu.h"
#include "nvm.h"
#include "tmr0.h"
#include "usbtest.h"

extern unsigned long t;
unsigned char matrix[PIN_COUNT][PIN_COUNT];
unsigned char dfpcheck[DFP_PIN_COUNT];
unsigned char ufpcheck[UFP_PIN_COUNT];
const unsigned char lines[]={LCD_LINE0,LCD_LINE1,LCD_LINE2,LCD_LINE3};
unsigned char sig[SIG_SIZE];            //working sig of current cable
unsigned char sig_match;
long powerV[4];

extern unsigned int menuResult;                 //set by menuService
extern unsigned int calValues[MENU_COUNT];      //used by menu/cal
volatile unsigned char t0cnt=0;                 //by tmr0
unsigned char t0last;                           //for checking time
unsigned char plast=0;                          //interval for power checking
unsigned int idleTime=0;                        //for checking idle timeout
char msg[MSG_SIZE]="";
int msgPtr=0;   
char scrollCnt=0;

__EEPROM_DATA(0,4,220,0,50,0,2,0); // cal values: 1024mV FVR, 220m# shunt, 50m# contacts, scroll div

#define CAL_TMOUT (30)

unsigned char sigAll[SIG_SIZE]= //this will hold the OR of all valid connections
    {0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00};
//eg 0x80,0x04,0x00,0x20,0x01,0x00,0x02,0x20,0x11,0x08,0x80,0x44,0x02,0x20,0x11,0x08,0x80,0x44

//use sigs[0] instead of sigNone
/*
const unsigned char sigNone[SIG_SIZE]= //blank signature
    {0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00};
*/

unsigned char sigDelta[SIG_SIZE]=   //contains sig of differences, ie all 0's for good cable
    {0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00};

const unsigned char sigs[SIG_COUNT][SIG_SIZE]={
    {0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00},    //no connection
    {0x80,0x04,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00},    //power only
    {0x80,0x04,0x00,0x20,0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00},    //USB2.0    
    //3.0: 1 TX/RX SS pair
    {0x80,0x04,0x00,0x20,0x01,0x00,0x02,0x00,0x10,0x08,0x00,0x40,0x00,0x00,0x00,0x00,0x00,0x00},    //USB3.0
    {0x80,0x04,0x00,0x20,0x01,0x00,0x00,0x20,0x01,0x00,0x80,0x04,0x00,0x00,0x00,0x00,0x00,0x00},    //USB3.0
    {0x80,0x04,0x00,0x20,0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x02,0x00,0x10,0x08,0x00,0x40},    //USB3.0
    {0x80,0x04,0x00,0x20,0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x20,0x01,0x00,0x80,0x04},    //USB3.0
    //3.1: 2 TX/RX SS pairs
    {0x80,0x04,0x00,0x20,0x01,0x00,0x02,0x00,0x10,0x08,0x00,0x40,0x00,0x20,0x01,0x00,0x80,0x04},    //3.1
    {0x80,0x04,0x00,0x20,0x01,0x00,0x00,0x20,0x01,0x00,0x80,0x04,0x02,0x00,0x10,0x08,0x00,0x40},    //3.1
    //same as above 3.0, but lacking 2.0 DP/DM
    //3.0: 1 TX/RX SS pair
    {0x80,0x04,0x00,0x00,0x00,0x00,0x02,0x00,0x10,0x08,0x00,0x40,0x00,0x00,0x00,0x00,0x00,0x00},    //USB3.0-2.0
    {0x80,0x04,0x00,0x00,0x00,0x00,0x00,0x20,0x01,0x00,0x80,0x04,0x00,0x00,0x00,0x00,0x00,0x00},    //USB3.0-2.0
    {0x80,0x04,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x02,0x00,0x10,0x08,0x00,0x40},    //USB3.0-2.0
    {0x80,0x04,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x20,0x01,0x00,0x80,0x04},    //USB3.0-2.0
    //3.1: 2 TX/RX SS pairs
    {0x80,0x04,0x00,0x00,0x00,0x00,0x02,0x00,0x10,0x08,0x00,0x40,0x00,0x20,0x01,0x00,0x80,0x04},    //3.1-2.0
    {0x80,0x04,0x00,0x00,0x00,0x00,0x00,0x20,0x01,0x00,0x80,0x04,0x02,0x00,0x10,0x08,0x00,0x40}     //3.1-2.0
};

const char sigStrings[SIG_COUNT][15]={
    "No cable      ",
    "Power only    ",
    "USB 2.0       ",
    "USB 3.2x1     ",
    "USB 3.2x1     ",
    "USB 3.2x1     ",
    "USB 3.2x1     ",
    "USB 3.2x2     ",
    "USB 3.2x2     ",
    "USB 3.2x1-2.0 ",
    "USB 3.2x1-2.0 ",
    "USB 3.2x1-2.0 ",
    "USB 3.2x1-2.0 ",
    "USB 3.2x2-2.0 ",
    "USB 3.2x2-2.0 "
};

    
int main(void){    
    unsigned char mStat,uStat,dStat,delta,i,j;
    unsigned int v;
    int m;      //for char array message indexing
    int cableV0,cableV1,cableV2;
    char sd,sp,sm;
    ioinit();
    LCD_init();
    uartInit();
    tmr0init();
    LCD_cgram(graphics);
    setIO();
    setPWM(0);         //shut off
    loadCal();         //load cal values from EEPROM
    GIE=1;
    LCD_clr();
    LCD_print("    SILICON CHIP    ");
    LCD_cmd(LCD_SETPOS+LCD_LINE1);
    LCD_print("  USB CABLE TESTER  ");
    LCD_cmd(LCD_SETPOS+LCD_LINE3);
    LCD_print("ESC on UART for cal.");
/*
    //not used?
    for(i=0;i<SIG_SIZE;i++){
        for(j=0;j<SIG_COUNT;j++){
            sigAll[i]=sigAll[i] | sigs[j][i];
        }
    }
*/
    while(t0cnt<CAL_TMOUT){        //timeout for calibration
        LCD_cmd(LCD_SETPOS+LCD_LINE0+19);        
        LCD_data(((CAL_TMOUT-t0cnt)>>2)+'0');
        while(uartAvailable()){
            if(uartReceive()==ESC_KEY){
                doCalibration();
                t0cnt=CAL_TMOUT;    //force timeout
            }
        }
        __delay_ms(20);        
    }
    uartDeinit();                   //free pins
    t0last=t0cnt;                   //start tracking time
    setTimeout();
    LCD_clr();
    while(1){
        mStat=usbScan();
        uStat=doUFP();
        dStat=doDFP();       
        if(mStat==0){       //idle, no cable present
            msg[0]=0;       //reset message
            msgPtr=0;       //and pointer when no cable
            plast=3;        //set timer to low value to allow quick response
            v=getBatt();
            LCD_cmd(LCD_SETPOS+LCD_LINE0);
            LCD_print("  USB Cable Tester  ");
            if(uStat){
                setTimeout();
                if(dStat){      //both have issues
                    if(t0cnt&8){    //~0.5Hz
                        reportDFP();
                    }else{
                        reportUFP();
                    }
                }else{          //only UFP has issues
                    reportUFP();
                }
            }else{
                if(dStat){      //only DFP has issues
                    setTimeout();
                    reportDFP();                    
                }else{          //no issues, no cable plugged in
                    LCD_cmd(LCD_SETPOS+LCD_LINE1);
                    LCD_print("  Ready for cable.  ");                    
                    LCD_cmd(LCD_SETPOS+LCD_LINE2);
                    LCD_print("                    ");                    
                    LCD_cmd(LCD_SETPOS+LCD_LINE3);
                    LCD_print("Battery: ");
                    LCD_data(getDigit(v,3));
                    LCD_data('.');
                    LCD_data(getDigit(v,2));
                    LCD_data('V');
                    LCD_data(' ');
                    LCD_ulong(idleTime>>2,5);
                    LCD_data('s');
                }                
            }
        }else{              //cable detected, run full tests
            setTimeout();
            getSig(&matrix[0][0],sig);
            delta=closest_match(sig);   //match index is now in sig_match            
            setSigPlus(sigs[sig_match]);
//            sp=match_sig(sigDelta,sigNone); //use signone as reference to count bits
            sp=match_sig(sigDelta,sigs[0]); //use sigs[0] as reference to count bits
            setSigMinus(sigs[sig_match]);
//            sm=match_sig(sigDelta,sigNone); //use signone as reference to count bits
            sm=match_sig(sigDelta,sigs[0]); //use sigs[0] as reference to count bits
            LCD_cmd(LCD_SETPOS+LCD_LINE0);
            LCD_print("CABLE INSERTED:");
            if(delta==0){   //perfect match for a cable type
                LCD_print("OK   ");
            }else{
                LCD_print("Fault");                
            }
            LCD_cmd(LCD_SETPOS+LCD_LINE1);
            if(sig_match<SIG_COUNT){
                LCD_print(sigStrings[sig_match]);                
            }else{
                LCD_print("Error         ");       //same length as sigStrings
            }
            LCD_ulong(sp,2);
            LCD_data('+');
            LCD_ulong(sm,2);
            LCD_data('-');
            //check internal DFP/UFP connections
            if((strlen(msg)==0)||(msgPtr>strlen(msg))){  //check and update message
                for(m=0;m<MSG_SIZE;m++){msg[m]=0;}       //clear
                msgPtr=0;
                if(sp){ //add additions
                    appendArray(msg,"Shorts:",MSG_SIZE);
                    setSigPlus(sigs[sig_match]);
                    for(i=0;i<12;i++){
                        if(checkAdd(sigDelta,i)){
                            appendArrayS(msg,npins[i],MSG_SIZE);
                            appendArray(msg,",",MSG_SIZE);
                        }
                    }
                }
                if(sm){ //add subtractions
                    appendArray(msg,"Opens:",MSG_SIZE);
                    setSigMinus(sigs[sig_match]);
                    for(i=0;i<12;i++){
                        if(checkAdd(sigDelta,i)){
                            appendArrayS(msg,npins[i],MSG_SIZE);
                            appendArray(msg,",",MSG_SIZE);
                        }
                    }
                }
                if(uStat){
                    if(dStat){
                        appendArray(msg,"Check DFP and UFP.",MSG_SIZE);                                                        
                    }else{
                        appendArray(msg,"Check UFP.",MSG_SIZE);                                                        
                    }
                }else{
                    if(dStat){
                        appendArray(msg,"Check DFP.",MSG_SIZE);                                                        
                    }else{
                        if(!msg[0]){    //supply default message if nothing else noted
                            appendArray(msg,"UFP and DFP OK.",MSG_SIZE);                                                        
                        }
                    }                
                }                        
            }else{
                scrollCnt++;
                if(scrollCnt>=calValues[3]){
                    msgPtr++;
                    scrollCnt=0;
                }
            }
            LCD_cmd(LCD_SETPOS+LCD_LINE2);
            if(strlen(msg)<21){
                for(m=0;m<20;m++){      //avoid scroll on short messages
                    LCD_data(msg[m]);                
                }   
                while(msgPtr<20){msgPtr++;} //refresh quickly if scrolling not needed
            }else{
                for(m=msgPtr;m<msgPtr+20;m++){
                    LCD_data(msg[m]);                
                }                
            }
            LCD_cmd(LCD_SETPOS+LCD_LINE3);
            //check lead resistance
            if(plast==0){
                plast=20;    //reset interval
                setPWR();        
                RLY_CTL=1;
                __delay_ms(5);
                cableV0=getmV(calValues[1]/10);    //tenth of shunt value => 100mA
                cableV1=getmV(calValues[1]/2);     //half of shunt value => 500mA
                cableV2=getmV(calValues[1]);       //shunt value => 1A
                __delay_ms(5);
                RLY_CTL=0;
                setIO();  
                if(cableV0>0){
                    cableV0=cableV0-calValues[2]/10;
                    if(cableV0<0){cableV0=0;}
                }
                if(cableV1>0){
                    cableV1=cableV1-calValues[2]/2;
                    if(cableV1<0){cableV1=0;}
                }
                if(cableV2>0){
                    cableV2=cableV2-calValues[2];
                    if(cableV2<0){cableV2=0;}
                }
                if(cableV2>=0){   //valid
                    LCD_ulong(cableV2,4);
                    LCD_print("mV at 1A: ");
                    LCD_ulong(cableV2,4);       //1A => ohms == V
                    LCD_print("m\x1");
                }else if(cableV1>=0){           //valid
                    LCD_ulong(cableV1,4);
                    LCD_print("mV at 500mA:");
                    cableV1=cableV1*2;          //500mA => ohms is double V
                    LCD_data(getDigit(cableV1,3));
                    LCD_data('.');
                    LCD_data(getDigit(cableV1,2));
                    LCD_data('\x1');                    
                }else if(cableV0>=0){   //valid
                    LCD_ulong(cableV0,4);
                    LCD_print("mV at 100mA:");
                    cableV0=cableV0*10;          //100mA => ohms 10xV
                    LCD_data(getDigit(cableV0,3));
                    LCD_data('.');
                    LCD_data(getDigit(cableV0,2));
                    LCD_data('\x1');                    
                }else{
                    LCD_print("High resistance     ");                    
                }
            }
        }
        while(t0last==t0cnt){}   //wait for tick
        while(t0last!=t0cnt){
            if(idleTime){idleTime--;}
            if(plast){plast--;}
            t0last++;
        }
        if(idleTime==0){    //timeout sleep
            doSleep();
            setTimeout();   //clear timeout           
        }
    }
    return 0;
}

void __interrupt() isr(void){
    char d;
    if(TMR0IE && TMR0IF){
        TMR0IF=0;               //clear
        t0cnt++;
    }
    if(RCIF && RCIE){
        if(OERR){CREN=0;CREN=1;}   //reset overflow error (serious, but better than locking up)
        RCIF=0;           //clear flag
        d=RC1REG;        //do read
        if(((rx_head+1)%BUFSIZE)!=rx_tail){ //avoid overflow by dropping
            rxbuf[rx_head]=d;
            rx_head=(rx_head+1)%BUFSIZE;
        }
    }
    /* //TX not used
    if(TXIF && TXIE){
       if(tx_head!=tx_tail){    //data in buffer
           TX1REG=txbuf[tx_tail];
           tx_tail=(tx_tail+1)%BUFSIZE;
       }else{
           TXIE=0;              //no more data, shut down
       }
    }
    */
}
