; iDoorBell.ASM: intelligent door bell
; In brief : The sequence of mixing short and long pulses goes to the digital
; input pin RA0 which decodes and compares it to the secret code stored in 
; the microcontroller. If equal, then the digital output pin RA1 goes high 
; level (for some millisseconds, just to actvate the 12V relay then release it)
; 

controller	equ	10F320
;controller	equ	10F322

	if	controller == 10F320
	List P = 10F320
#include <p10f320.inc> 			
	Errorlevel -302		; avoid warning #302: Register in operand not in bank0. 
	endif

	if	controller == 10F322
	List P = 10F322
#include <p10f322.inc> 			
	Errorlevel -302		; avoid warning #302: Register in operand not in bank0. 
	endif

    __config _FOSC_INTOSC & _MCLRE_OFF & _LVP_OFF & _WDTE_OFF & _BOREN_OFF

;----------------------------- define hardware --------------------------------
;internal 4 MHz
#define	 DOORBTN  PORTA, 0		; RA0 input, receives the secret code
#define	 BUZZER	  PORTA, 1		; RA1 output, drives T1, then 12V relay
;#define	 LED1	  PORTA, 2	; RA2 output, drives LED1, optional debugging

#define	 SECRETCODE	 b'00111000'; pseudo "S.O.S." morse code (incomplet)

;----------------------------- define constants -------------------------------

;OPTION_REG: p.103
#define  OPTIONVAL	b'01000111'	; pullup ON, <5>=0:RA2 I/O, TMR0 prescale(1:256)
;#define	 OPTIONVAL	b'11000011'	; pullup OFF, <5>=0:RA2 I/O, TMR0  1:16

;INTCON: p.46
#define	 INTERMASK	b'00100000'	; <5>=1: enables TMR0 interrupts
;#define	INTERMASK	b'00110000'	; use TMR0 and GP2/INT interruption

;----------------------------- define variables -------------------------------

	CBLOCK	0x40	; SRAM = [0x40...0x7F] = 64 bytes (PIC10F320/322)

;context
	w_tmp
	STATUS_tmp
;delay
	dureeP, cpt1, cpt2, delayT1on	; counters
;pulses (the "secret" code should not contain more than 8 pushes)
	codeReceived		; bit0=>pulse0; bit1=>pulse1; ... ; bit7=>pulse7 
	theSecretCode
	nextpulse
	
	ENDC
;9 bytes used

;------------------------------------------------------------------------------
; macro to put a literal value (constant) into a register
;------------------------------------------------------------------------------
MOVLF	macro	literal, file	; literal -> file
	MOVLW	literal
	MOVWF	file
	endm

;-----------------------------------------------------------------------------
; macro to save context
;-----------------------------------------------------------------------------
SAVE_CONTEXT	macro
	MOVWF	w_tmp  			; save W register
	SWAPF	STATUS, w		; swap status, then put result into w
	MOVWF	STATUS_tmp		; save status swapped
	endm

;-----------------------------------------------------------------------------
; macro to restore context
;-----------------------------------------------------------------------------
RESTORE_CONTEXT	macro
	SWAPF	STATUS_tmp, w	; swap old status, then put result into w
	MOVWF   STATUS			; restore status
	SWAPF   w_tmp, f		; Invert nibbles in old W (without modifying Z)
	SWAPF   w_tmp, w  		; Invert again nibbles in W
							; (so, W is restored without modifying STATUS)
	endm


;//////////////////////////////////////////////////////////////////////////////
;	MAIN PROGRAM
;//////////////////////////////////////////////////////////////////////////////
	ORG		0x00			; reset vector
	GOTO	Init
	ORG		0X04			; interrupt vector
	GOTO	IntTMR0

;//////////////////////////////////////////////////////////////////////////////

#include "IF8B16B3.INC"

;------------------------------------------------------------------------------
; Delay approx. 100 ms (@ 4 MHz)
;------------------------------------------------------------------------------
Delay100ms
	MOVLW	.130		; 130 * (768 + 3) = 100023 us # 100 ms
	MOVWF	cpt2
Delay
	DECFSZ	cpt1, f		; 3*256 = 768 us internal delay loop 
	GOTO	Delay		;
	DECFSZ	cpt2, f
	GOTO	Delay
	RETLW	0

;------------------------------------------------------------------------------
Init
;set GPIOs
	movlw	b'11111001'
	movwf	TRISA			; RA1,RA2 outputs ; RA0,RA3 inputs
	movlw	b'00000000'
	movwf	PORTA			; RA1,RA2 Low
	clrf	ANSELA			; use all analog as digital I/O
	;CLRF	WPUA			; all pull-up individually disabled
	movlw	b'00001000'
	movwf	WPUA			; GP3 pullup individually enabled (to lower Idd)

;DS41585A.PDF(p.28/210): The HFINTOSC consists of a primary & secondary clock. 
;The secondary clock starts first with rapid start-up time, but low accuracy. 
;The primary clock follows with slower start-up time and higher accuracy. 

;set freq. to 4 MHz, then wait until sec. clock ready and pri. clock stable
	movlw	b'01010000'		; OSCCON<6:4>=101 => 4 MHz
	;movlw	b'01010000'		; OSCCON<6:4>=100 => 2 MHz
	;movlw	b'00110000'		; OSCCON<6:4>=011 => 1 MHz
	iorwf	OSCCON, F
	btfss	OSCCON, HFIOFR	; HFIOFR=1 => secondary clock is ready
	goto	$-1
	btfss	OSCCON, HFIOFS	; HFIOFS=1 => primary clock is stable
	goto	$-1

;init OPTION_REG & INTCON
	movlw	OPTIONVAL		; <7>=0: pullup ON, <5>=0: RA2 I/O, ...
	movwf	OPTION_REG		; ... TMR0 prescale (1:256) => 65.536 ms overflow
	movlw	INTERMASK
	movwf	INTCON			; enable TMR0 interrupt
	
;init variables	
	clrf	dureeP			; clear timing of valid puls
	clrf	cpt1			; clear counter used for delays 
	clrf	codeReceived
	clrf	nextpulse
	clrf	delayT1on
	movlw	SECRETCODE
	movwf	theSecretCode
;-----
	bcf		INTCON, TMR0IF	; clear TMR0 interrupt flag
	bsf		INTCON, GIE		; set general interrupt enable

loop
	nop
	goto	loop			; infinite loop (waiting for TMR0 interrupt)

;----------------------------- Interrupt routine -----------------------------
; each 65.536 ms
IntTMR0
	SAVE_CONTEXT
	
	SIregNUL  delayT1on, openT1 ; if (delayT1on == 0) goto openT1
	decf	delayT1on, f		; else, T1 still on ( > 100 ms to unlock ? ) 
	goto	endIntTMR0

openT1:
	movlw	b'00000000'
	movwf	PORTA			; RA1 low => T1 opens => 12V relay de-activated

;pushbutton ON ?
	btfss	DOORBTN		; DOORBTN pushed ?	
	goto	pulses		; no, it is released, then figure out short or long pulse ? 
;DOORBTN pushed
	;bsf		LED1		; optional debugging
	incf	dureeP, f
	goto	endIntTMR0


pulses:
	SIregLTval8b	dureeP, .16, shortpulse	; if (dureeP < 16) goto shortpulse 
longpulse:
	bsf	  STATUS, C			; Carry = 1
	RLF	  codeReceived, f	; rotate left, then set bit0
	goto  nextP

shortpulse:
; et si dureeP == 0 ?
	SIregNUL  dureeP, idleState ; if (dureeP == 0) goto idleState

	bcf	  STATUS, C				; Carry = 0
	RLF	  codeReceived, f		; rotate left, then clear bit0
nextP:
	incf  nextpulse, f
	SIregLTval8b	nextpulse, .8, idleState
	;we got more than 8 pulses, then look if valid code received
	SIregaNEregb8b	codeReceived, theSecretCode, notOk 
Ok
	movlw	b'00000010'
	movwf	PORTA			; RA1 high => T1 closes => 12V relay activated
	movlw	.2
	movwf	delayT1on
notOk:
	clrf	dureeP			; clear timing of valid pulse
	clrf	cpt1			; clear counter used for delays 
	clrf	codeReceived
	clrf	nextpulse
	;bcf		LED1			; optional debugging
	goto	endIntTMR0

idleState:
	clrf	dureeP
	;bcf		LED1			; optional debugging

endIntTMR0:
	bcf		INTCON, TMR0IF	; clear TMR0 interrupt flag
	RESTORE_CONTEXT
	RETFIE  				; return from interrupt
;-----------------------------------------------------------------------------
	END
