//basic code to drive the LEDs in the MIDI Switch Matrix
//MIDI SYSEX packets:
//packets starting 0xF0 and ending with 0xF7 are SYSEX messages (Manufacturer specific etc)
//format is 0xF0 [MAN ID] ... 0xF7
//0x7D is MAN ID for prototyping, test, private use and experimentation
//so we need to send 0xF0 0x7D [data] 0xF7
//we will use 'S','C' to identify our data, followed by LED data as 0b0SLLLLLL
//S= set/clear, L=LED number
//any number of LED data can follow, as we use any byte with MSB set as SYSEX terminator
//state machine:
//state 0: waiting for 0xF0
//state 1: waiting for 0x7D
//state 2: waiting for 'S'
//state 3: waiting for 'C'
//state 4: expecting LED data (stay in state 4) or anything with MSB set (go to state 0)

//TimerOne library from https://github.com/PaulStoffregen/TimerOne
#include <TimerOne.h>

//same pins as switches
//rows
byte rowPins[8]={2,3,4,5,6,7,8,9};
//columns
byte columnPins[8]={A0,A1,A2,A3,A4,A5,11,10};

byte LED[128]={     //set array as 128 to allow all combinations of seven bits
  0,0,0,0,0,0,0,0,  //we're only going to use 64 elements
  0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0
};
  
void setup() {
  int i;
  for(i=0;i<8;i++){   //set pin modes
    pinMode(rowPins[i],OUTPUT);
    pinMode(columnPins[i],INPUT);
    digitalWrite(columnPins[i],LOW);    //toggle drive with pinMode
  }
  Timer1.initialize(200);  //update rate= frame rate * 8 columns
  Timer1.attachInterrupt(LEDupdate); 
  Serial1.begin(31250);   //midi baud rate
  Serial.begin(115200);   //for debugging
}

void loop() {
  static byte state=0;  
  int data;
  if(Serial1.available()){
    data=Serial1.read();
    Serial.print("0x");
    Serial.println(data,HEX);
    switch(state){
      case 0: if(data==0xF0){state=1;} break;
      case 1: if(data==0x7D){state=2;} break;
      case 2: if(data=='S'){state=3;} break;
      case 3: if(data=='C'){state=4;} break;
      case 4:
        if(data&0x80){        //any byte with MSB set
          state=0;
        }else{
          if(data&0x40){
            LED[data&0x3F]=1;
            Serial.print(data&0x3F);
            Serial.println(" ON");
          }else{
            LED[data&0x3F]=0;
            Serial.print(data&0x3F);
            Serial.println(" OFF");
          }
        }
        break;
    }
  }
}

void LEDupdate(void){   //timer ISR
  static byte col=0;
  byte i;
  for(i=0;i<8;i++){   //set column drivers off
    pinMode(columnPins[i],INPUT);      
  }
  for(i=0;i<8;i++){
    digitalWrite(rowPins[i],LED[col+i*8]);  //set row drivers
  } 
  pinMode(columnPins[col],OUTPUT);  //enable active column
  col=(col+1)&7;
}
