//this version outputs 16 notes on four channels to allow use of multi-channel synths

//MIDI synth output on D13 using TIMER4
//TimerOne library from https://github.com/PaulStoffregen/TimerOne
#include <TimerOne.h>

//OCR4A=duty for tone generation, updated from WAV by timerone interrupt
#define SAMPLE_RATE (50000L)
//256 wav values for ease of calculation
unsigned char WAVdata[256]={128,131,134,137,140,143,146,149,152,155,158,161,164,167,170,172,175,178,181,184,186,189,192,194,197,199,202,204,207,209,211,214,216,218,220,222,224,226,228,230,231,233,235,236,238,239,240,242,243,244,245,246,247,248,249,249,250,251,251,252,252,252,252,252,253,252,252,252,252,252,251,251,250,249,249,248,247,246,245,244,243,242,240,239,238,236,235,233,231,230,228,226,224,222,220,218,216,214,211,209,207,204,202,199,197,194,192,189,186,184,181,178,175,172,170,167,164,161,158,155,152,149,146,143,140,137,134,131,128,124,121,118,115,112,109,106,103,100,97,94,91,88,85,83,80,77,74,71,69,66,63,61,58,56,53,51,48,46,44,41,39,37,35,33,31,29,27,25,24,22,20,19,17,16,15,13,12,11,10,9,8,7,6,6,5,4,4,3,3,3,3,3,3,3,3,3,3,3,4,4,5,6,6,7,8,9,10,11,12,13,15,16,17,19,20,22,24,25,27,29,31,33,35,37,39,41,44,46,48,51,53,56,58,61,63,66,69,71,74,77,80,83,85,88,91,94,97,100,103,106,109,112,115,118,121,124};

//switch pins:
//rows (pullups)
byte rowPins[8]={2,3,4,5,6,7,8,9};
//columns
byte columnPins[8]={A0,A1,A2,A3,A4,A5,11,10};

//MIDI notes:
//f=440 x 2^((n−69)/12)
//28 => 60=middle C is centre of 64 note range
#define START_NOTE 28

//integer versions of MIDI notes
int notes[128]={8,9,9,10,10,11,12,12,13,14,15,15,16,17,18,19,21,22,23,24,26,28,29,31,33,35,37,39,41,44,46,49,52,55,58,62,65,69,73,78,82,87,92,98,104,110,117,123,131,139,147,156,165,175,185,196,208,220,233,247,262,277,294,311,330,349,370,392,415,440,466,494,523,554,587,622,659,698,740,784,831,880,932,988,1047,1109,1175,1245,1319,1397,1480,1568,1661,1760,1865,1976,2093,2217,2349,2489,2637,2794,2960,3136,3322,3520,3729,3951,4186,4435,4699,4978,5274,5588,5920,6272,6645,7040,7459,7902,8372,8870,9397,9956,10548,11175,11840,12544};
int lastKey=-1;   //to note changes
int currentKey=-1;  //for ISR to play
long currentF=0;    //long to allow scaling later

//MIDIUSB library from
//https://github.com/arduino-libraries/MIDIUSB
#include "MIDIUSB.h"

//calculates channel from note 0-15=>0, 16-31=>1, 32-47=>2, 48-63=>3
int channel(int note){
  int r;
  r=(note-START_NOTE)/16;
  if(r<0){r=0;}
  if(r>3){r=3;}
  return r;
}

//also use hardwired velocity
#define MIDI_VELOCITY 64

//for 'hardware' MIDI output
//from: https://github.com/FortySevenEffects/arduino_midi_library
#include <MIDI.h>
MIDI_CREATE_INSTANCE(HardwareSerial,Serial1,hmidi);

void setup() {
  byte i;
  timer4_init();      //set up fast PWM for audio
  OCR4A=WAVdata[0];   
  for(i=0;i<8;i++){   //set pin modes
    pinMode(rowPins[i],INPUT_PULLUP);
    pinMode(columnPins[i],INPUT);
  }
  hmidi.begin();
  Serial.begin(115200);
  Timer1.initialize(1000000L/SAMPLE_RATE); //micros between callbacks
  Timer1.attachInterrupt(toneUpdate); 
}

void loop() {
  int key;
  key=scanPins();
  if(key!=lastKey){
    if(key>-1){ //keydown
      if(lastKey>-1){     //switch between keys without a gap, keyup+keydown
        Serial.print("UP:");
        Serial.println(lastKey);
        noteOff(channel(lastKey+START_NOTE),lastKey+START_NOTE,MIDI_VELOCITY);
        MidiUSB.flush();        
        hmidi.sendNoteOff(lastKey+START_NOTE,0,channel(lastKey+START_NOTE)+1);
        Serial.print("DOWN:");
        Serial.println(key);
        noteOn(channel(key+START_NOTE),key+START_NOTE,MIDI_VELOCITY);
        MidiUSB.flush();        
        hmidi.sendNoteOn(key+START_NOTE,MIDI_VELOCITY,channel(key+START_NOTE)+1);
      }else{              //normal keydown
        Serial.print("DOWN:");
        Serial.println(key);        
        noteOn(channel(key+START_NOTE),key+START_NOTE,MIDI_VELOCITY);
        MidiUSB.flush();        
        hmidi.sendNoteOn(key+START_NOTE,MIDI_VELOCITY,channel(key+START_NOTE)+1);
      }      
    }else{                //normal keyup
      Serial.print("UP:");
      Serial.println(lastKey);      
      noteOff(channel(lastKey+START_NOTE),lastKey+START_NOTE,MIDI_VELOCITY);
      MidiUSB.flush();        
      hmidi.sendNoteOff(lastKey+START_NOTE,0,channel(lastKey+START_NOTE)+1);
    }
    currentKey=key;
    lastKey=key;
    if(currentKey>-1){
      currentF=notes[currentKey+START_NOTE];
    }else{
      currentF=0;
    }
  }
  delay(1);  //crude debouncing  
}

void timer4_init(){
  pinMode(13,OUTPUT);
  TCCR4A=0;
  TCCR4B=1; //187kHz for 48MHz PLL
  TCCR4C=0;
  TCCR4D=0;
  OCR4C=255;  //MAX
  PLLFRQ=(PLLFRQ&0xCF)|0x30;
  TCCR4A=0x82; //start?
  OCR4A=0; 
}


int scanPins(){ //returns 0-63 or -1 if none
  byte i,j;
  int key=-1;     //what we'll return
  int count=0;    //if more than one, then not valid (only 1KRO)
  //assume all row pins are input_pullup and columns are floating as per setup()
  for(j=0;j<8;j++){
    pinMode(columnPins[j],OUTPUT);
    digitalWrite(columnPins[j],LOW);
    for(i=0;i<8;i++){
      if(digitalRead(rowPins[i])==0){
        key=i*8+j;
        count++;
      }
    }
    pinMode(columnPins[j],INPUT);
  }
  if(count!=1){key=-1;}
  return key;  
}

void toneUpdate(void){
  static unsigned long t=0; //pointer to generate tones from array
  t=t+currentF*((65536*256)/SAMPLE_RATE);   //nominal multiplier should be 419 for default values
  OCR4A=WAVdata[(t>>16)&0xFF];  
}

//these helpers taken from MIDIUSB example MIDIUSB_write
void noteOn(byte channel, byte pitch, byte velocity) {
  midiEventPacket_t noteOn = {0x09, 0x90 | channel, pitch, velocity};
  MidiUSB.sendMIDI(noteOn);
}

void noteOff(byte channel, byte pitch, byte velocity) {
  midiEventPacket_t noteOff = {0x08, 0x80 | channel, pitch, velocity};
  MidiUSB.sendMIDI(noteOff);
}
