#include "util.h"
#include "io.h"
#include "oled.h"
#include "flash.h"

const unsigned int tcon[]={ //raw divider ADC values mapped to tenths of a K
    0    ,0    ,0    ,2533 ,2589 ,2636 ,2677 ,2713 ,
    2747 ,2780 ,2810 ,2840 ,2869 ,2897 ,2925 ,2953 ,
    2981 ,3010 ,3040 ,3070 ,3102 ,3136 ,3172 ,3211 ,
    3253 ,3300 ,3353 ,3416 ,3495 ,3593 ,9990 ,9990 ,
    9990   //extra value to avoid index overlow
};

const char stateString[][11]={
    "IDLE      ",
    "NO BATTERY",
    "FLOAT     ",
    "BULK/ABS  ",
    "STORAGE   ",
    "PWR FAULT ",
    "POWER OFF "    
};

const char calString[][11]={
    "**********",
    "BATTERY V ",
    "SUPPLY V  ",
    "OUTPUT V  ",
    "OUTPUT I  ",
    "LOW V BAT ",
    "LOW V SUP ",
    "LOW V OUT ",
    "BULK START",
    "BULK END  ",
    "BULK BOOST",
    "STORE DROP",
    "BULK TIME ",
    "STORE TIME",
    "STORE DLY ",
    "TEMP COEFF",
    "USE EDITS ",
    "SAVE FLASH"
};

struct flashshadow f;   //live in RAM
const struct flashshadow hef __at(FLASHPAGESIZE*HEF_PAGE)={FLASH_DEFAULTS};//in writable flash
const struct flashshadow fbackup={FLASH_DEFAULTS};  //non-writable backup

char digits[5]; //for decoding to decimal
int pwm=NULL_PWM;
unsigned int vbat,vin,vout,iout;    //all in mV/mA
int temp;   //tenths of a degC
volatile unsigned char counter;
volatile unsigned int seconds=0;
volatile unsigned char flag1, flag2;    //flag1 ~ 20Hz for debouncing, flag2 ~1Hz for screen updates
volatile unsigned char s1down=0;
volatile unsigned char s2down=0;
volatile unsigned char s3down=0;
volatile unsigned char s1Last=0;
volatile unsigned char s2Last=0;
volatile unsigned char s3Last=0;

void getData(void){ //get updated V/A values via ADC
    //vbat,vin,vout,iout,temp
    unsigned int t;
    unsigned char osctemp=OSCCON;
    OSCCONbits.IRCF = 0b1111; //use 16MHz HFINTOSC
    t=getADC(VBATSENSE);    //raw ADC
    vbat=adjust(t,f.vbatcal);
    t=getADC(VOUTSENSE);    //raw ADC
    vout=adjust(t,f.voutcal);   
    t=getADC(VINSENSE);    //raw ADC
    vin=adjust(t,f.vincal);
    t=getADC(ISENSE);    //raw ADC
    iout=adjust(t,f.ioutcal);
    t=getADC(TSENSE);    //raw ADC
    temp=getTemp(t)-2731;
    OSCCON=osctemp;
}

void normalData(void){
    OLEDsetpage(2);
    OLEDsetcolumn(48);
    showV(vbat);
    OLEDsetpage(3);
    OLEDsetcolumn(48);
    showV(vin);
    OLEDsetpage(4);
    OLEDsetcolumn(48);
    showV(vout);
    OLEDsetpage(5);
    OLEDsetcolumn(48);
    showAA(iout);
    OLEDsetpage(6);
    OLEDsetcolumn(48);
    if((temp<-TEMP_MIN)||(temp>TEMP_MAX)){
        OLEDchararray("T ERR");        
    }else{
        snum(temp,0);
        OLEDchar(digits[3]);
        OLEDchar(digits[2]);
        OLEDchar(digits[1]);
        OLEDchar(DEGREES);            
        OLEDchar('C');    
    }
}

void snum(int s, char p){    //set up signed number for display
    if(s<0){
        s=abs(s);
        toDec(s);
        doZB(p);
        if(digits[1]==' '){digits[1]='-';return;}
        if(digits[2]==' '){digits[2]='-';return;}
        if(digits[3]==' '){digits[3]='-';return;}
        if(digits[4]==' '){digits[4]='-';return;}
    }else{
        toDec(s);
        doZB(p);        
    }
}

void unum(unsigned int s, char p){  //set up unsigned number
    toDec(s);
    doZB(p);
}

unsigned int adjust(unsigned int raw,unsigned int scale){
    //scale is effectively a 8.8 fixed point multiplier presented as uint
    unsigned long temp;
    temp=((long)raw)*((long)scale);
    temp=temp>>8;
    return temp;
}

void doZB(char p){  //leading zero blank except places at or lower than p
    if(digits[4]=='0'){  //always blank leading
        digits[4]=' ';
        if((digits[3]=='0')&(p<3)){
            digits[3]=' ';            
            if((digits[2]=='0')&(p<2)){
                digits[2]=' ';            
                if((digits[1]=='0')&(p<1)){
                    digits[1]=' ';            
                }
            }
        }
    }
}

void toDec(unsigned int n){
    digits[0]='0';
    digits[1]='0';
    digits[2]='0';
    digits[3]='0';
    digits[4]='0';
    unsigned int p=10000;
    char d=4;
    while(n){
        if(n>=p){
            n=n-p;
            digits[d]++;
        }else{
            p=p/10;
            d--;
        }
    }
}

int getTemp(unsigned int v){    //returns 1/10 of K
    int i,o;
    long a,b;
    i=(v>>5)&0x1F;  //index into array
    o=v&0x1F;       //error offset
    a=tcon[i];
    b=tcon[i+1];
    return ((b*o)+(a*(0x20-o)))>>5;
}

void normalTitle(void){
    OLEDsetpage(2);
    OLEDsetcolumn(0);
    OLEDchararray("Battery:        BULK");
    OLEDsetpage(3);
    OLEDsetcolumn(0);
    OLEDchararray("Supply:         S1:");
    OLEDsetpage(4);
    OLEDsetcolumn(0);
    OLEDchararray("Output:         Start");
    OLEDsetpage(5);
    OLEDsetcolumn(0);
    OLEDchararray("Current:        S2:");    
    OLEDsetpage(6);
    OLEDsetcolumn(0);
    OLEDchararray("Temp:           Stop");        
}

void flasherror(void){           //display message
    OLED2chararray("FLASH",0,2,ffont);    //double size
    OLED2chararray("ERROR",68,2,ffont);    //double size
    OLEDsetpage(5);
    OLEDsetcolumn(18);
    OLEDchararray("Defaults loaded.");    
}

void showV(unsigned int v){ //assumes v is in mV
    unum(v,3);
    OLEDchar(digits[4]);
    OLEDchar(digits[3]);
    OLEDchar('.');
    OLEDchar(digits[2]);
    OLEDchar('V');   
}

void showVV(unsigned int v){ //assumes v is in mV, shows 2dp
    unum(v,3);
    OLEDchar(digits[4]);
    OLEDchar(digits[3]);
    OLEDchar('.');
    OLEDchar(digits[2]);
    OLEDchar(digits[1]);
    OLEDchar('V');   
}

void showAA(unsigned int v){ //assumes v is in mA, shows 2dp
    unum(v,3);
    OLEDchar(digits[3]);
    OLEDchar('.');
    OLEDchar(digits[2]);
    OLEDchar(digits[1]);
    OLEDchar('A');   
}

void showU(unsigned int v){ //full normal 5 digit unsigned
    unum(v,0);
    OLEDchar(digits[4]);
    OLEDchar(digits[3]);
    OLEDchar(digits[2]);
    OLEDchar(digits[1]);
    OLEDchar(digits[0]);
}

void showS(int v){ //full normal 5 digit signed with 2dp
    snum(v,2);
    OLEDchar(digits[4]);
    OLEDchar(digits[3]);
    OLEDchar(digits[2]);
    OLEDchar('.');    
    OLEDchar(digits[1]);
    OLEDchar(digits[0]);    
}

void showPC(unsigned int v){ //one dp ie xxxx.x%
    unum(v,0);
    OLEDchar(digits[4]);
    OLEDchar(digits[3]);
    OLEDchar(digits[2]);
    OLEDchar(digits[1]);
    OLEDchar('.');   
    OLEDchar(digits[0]);
    OLEDchar('%');   
}

void showT(unsigned int v){ //hhhh:mm
    unsigned int h,m;
    m=v;
    h=0;
    while(m>=60){
        h++;
        m=m-60;
    }    
    unum(h,0);
    //OLEDchar(digits[3]);
    OLEDchar(digits[2]);
    OLEDchar(digits[1]);
    OLEDchar(digits[0]);
    OLEDchar(':');   
    unum(m,1);
    OLEDchar(digits[1]);
    OLEDchar(digits[0]);
}

signed char buttonAdj(unsigned char n){
    if(S1){return n;}
    if(S2){return -n;}
    return 0;
}

void dispSleep(unsigned char n){     //save spurious writes
    static unsigned char oldn=0;
    if(n!=oldn){
        OLEDsetcontrast(n);
        if(n){
            OLEDsendCommand(0xAF);      //display on
        }else{
            OLEDsendCommand(0xAE);      //display off
        }
    
    }    
    oldn=n;
}

void __interrupt() isr_isr(void){
    unsigned char stemp;
    if(PIR1bits.TMR1IF && PIE1bits.TMR1IE){    //Timer1 interrupt
        TMR1H=249;               //set counter for short overflow
        PIR1bits.TMR1IF=0;       //clear flag
        counter++;
        stemp=S1;
        if(stemp&&(!s1Last)){
            s1down=1;
        }
        s1Last=stemp;

        stemp=S2;
        if(stemp&&(!s2Last)){
            s2down=1;
        }
        s2Last=stemp;

        stemp=S3;
        if(stemp&&(!s3Last)){
            s3down=1;
        }
        s3Last=stemp;
        //flag1=1;
        if(counter>PRESCALE){
            flag2=1;
            counter=0;
            seconds++;
        }
    }
}
