//  A development of Tim Blythman's "Clayton's GPS Time Source" Silicon Chip April 2018 which should be read with this.
// By the addition of an I2C connected 26 x 2 LCD display and a push button with its associated pull down resistor an ESP8266 board 
//  becomes a clock which can display the time in any country or Time Zone.  A table provides the data to ensure that the time is correct
//  as Daylight Saving Time comes and goes in both the northern and southern hemispheres.
// The table currently has 38 entries serving most of the time zones in the world, many more can be added.
// Connect an I2C enabled 16 x 2 (or 20 x 4) LCD display, SCL to D1 on the ESP 8266 and SDA to D2.  Connect a 10K resistor from D6 to D7
//  and a momentary push button between 3v3 and D6
//
// Connects to WiFi, requests NTP data and sends out as valid GPS sentences (RMC, GSA and GGA), with offset added to millis()
// Also shows Wifi status via custom ESP82 sentence
// Implements 64bit millis() for long term installation
// Settings menu with save to EEPROM with defaults (baudrate, WIFI SSID, WIFI PASS, NTP Server, State)
// uses IP API to get approx latitude/longitude
// LED status- on solid while connecting to Wi-Fi, flickers during NMEA data transmit
// working with analog clock and HiVis Clock
// V9 Add 1PPS feature
// V10 change srollover to 64 bit to prevent glitching
// V11 general tidyup (variable location/scope)

#include <EEPROM.h>
#define BAUDRATE_DEFAULT 9600
#include <ESP8266WiFi.h>
#include <ESP8266HTTPClient.h>
#include <WiFiUdp.h>
#include <LiquidCrystal_I2C.h>
// set the LCD I2C address to 0x27
// and for two lines of 16 characters
LiquidCrystal_I2C lcd(0x27,20,4);
HTTPClient http;
WiFiUDP udp;
IPAddress hostIP;
//GPIO4 is D2 on D1 Mini
#define PPSPIN 4
#define inpin 12    //button to advance countries
#define gndpin 13   //pseudo ground

//default settings
char ssid[80] = "SSID";                 //needs to be changed in settings
char pass[80] = "Password";               //needs to be changed in settings
char host[80] = "pool.ntp.org";           //should work OK
char state[80]= "5";                      //NSW
char ipapi[]="http://ip-api.com/line?fields=lat,lon";
char hex[]="0123456789ABCDEF";            //for writing out checksum
unsigned long baudrate=9600;

unsigned long long boott=(43099ULL*86400ULL)<<32;    //1/1/2018
unsigned long ts=0;
unsigned long dsn;              //day serial number
int date,mon,yr;                //calculated from daysn
bool ntplock=false;             //set after we get a valid NTP return
bool ntphit=false;              //an attempt was made to get NTP
bool passes = false;            // startup control
bool DSTflag = false;           //set when in DST
bool DateOrder;                 // 0 = dd/mm/yy, 1 = mm/dd/yy
bool SH;                        // southern hemisphere
unsigned long long srollover;   //for detecting when a new second rolls over
long lastadjust=0;              //track last adjustment, signed to see if +/-
unsigned long lastNTP;          //last time NTP was attempted
unsigned long long tnow;        //64bit time to work with
unsigned long tshort;           //32 bit time in seconds
unsigned long DSTevent;         //date & time of next DST event
signed long StateOffset;        //to be configured from EE at startup
unsigned long DSToffset;        //to be configured from EE at startup
unsigned long DSToffsetCopy = 3600;    //to be configured from EE at startup
char DSTkey;
char StateName[8] = "";
int DSTstartday, DSTstartweekday, DSTstartmonth, DSTendday, DSTendweekday, DSTendmonth, DSTstartHour, DSTendHour, MaxTable;
unsigned char DST_Pars [10];           //Data from the DST parameters table

void setup() {
  char sentence[80]="";

  lcd.begin();         // initialise LCD
//  wire.begin();        //wire.begin(0,2); needed for esp-01
  lcd.backlight();    // enable backlight, hopefully
  pinMode(inpin,INPUT);
  pinMode(gndpin,OUTPUT);           //pseudo ground
  digitalWrite(gndpin,LOW);
  pinMode(LED_BUILTIN,OUTPUT);      //active low on some devices
  digitalWrite(LED_BUILTIN,LOW);
  pinMode(PPSPIN,OUTPUT);
  pinMode(PPSPIN,LOW);
  int httpCode;               //for HTTP call
  int httptries=10;           //retries for IP API
  unsigned long long ttemp;        //for storing millis() while working
  delay(1000);
  loadconfig();               //load from EEPROM, including baudrate
  Serial.begin(baudrate);
  WiFi.mode(WIFI_STA);
  WiFi.begin(ssid, pass);
  while(WiFi.status()!=WL_CONNECTED){
    ttemp=millis64();
    if((ttemp%1000)<srollover){       //do this each second to let connected device know we're awake
      togglePPS();
      doupdate();
      doRMC();
      doGGA();
      doGSA();
      doESPstat();
    }
    checkinput();
    srollover=ttemp%1000;
    yield();                          //to stop watchdog resets
  }
  digitalWrite(LED_BUILTIN,HIGH);     //LED off now Wi-Fi connected
  while(httptries>0){                 //use IP API to find lat/lon
    ttemp=millis64();
    http.begin(ipapi);                //URL
    httpCode=http.GET();              //fetch
    if(httpCode==200){                //valid data returned?
      setlatlon(http.getString());    //set lat/lon from data
      httptries=0;                    //got our data, so continue
    }
    http.end();
    httptries--;
    if((ttemp%1000)<srollover){       //do this each second to let connected device know we're awake
      togglePPS();
      doupdate();
      doRMC();
      doGGA();
      doGSA();
      doESPstat();
    }
    srollover=ttemp%1000;
    yield();                          //to stop watchdog resets
  }

  udp.begin(2390);
  getNTP();                           //try to get time
  lastNTP=millis();
  DST_data (DSTkey, 0);              //From EE
  Serial.println(StateName);
  DST_data (0, 1);                   //Dummy, print table
}

void loop() {
  char PrintThis[8] = "";
  char PrintLine[12] = "";
  int i;
  unsigned long long ttemp;
  if((millis()-lastNTP>3600000UL)||(!ntplock&&(millis()-lastNTP>60000UL))){   //by default, try every hour, or every minute if not locked on
    getNTP();
    lastNTP=millis();
  }
  ttemp=millis64();
  if((ttemp%1000)<srollover){           //output sentences each second, will glitch to (1)296ms every 71 days
    digitalWrite(LED_BUILTIN,LOW);      //LED on
    togglePPS();
    doupdate();
/*    doRMC();
    doGGA();
    doGSA();  */
    doESPstat();
    doLCD();
    digitalWrite(LED_BUILTIN,HIGH);     //LED off
  }
  checkinput();
  srollover=ttemp%1000;
  if ((digitalRead(inpin) == 1)  && (passes == true))  { //button has been pressed
    if (DSTkey < (MaxTable - 1)) {
      DSTkey++;
    }
    else  {
      DSTkey = 0;
    }
    passes = false;                     //force change
    DST_data (DSTkey, 0);              //From Table
    itoa(DSTkey,PrintThis,10);
    sprintf(PrintLine,"%02d, %s",DSTkey,StateName);
    Serial.println(PrintLine);
    EEPROM.begin(1024);
    for(i=0;i<6;i++){
      EEPROM.write(244+i,PrintThis[i]);             //save to EEPROM
    }
    EEPROM.write(244 + 6,0);
    EEPROM.end();
  }  
}

void getNTP(){
  byte NTPpacket[48]="";
  unsigned long long ts3,ts4;
  unsigned long long newboott,ts1,ts2;           //to measure adjustment, outgoing/return timestamps
  int i;
  ntphit=true;            //flag that we're trying to do an update
  for(i=0;i<48;i++){      //clear packet contents 
    NTPpacket[i]=0;
  }
  NTPpacket[0] = 0xE3;    //set default values for outgoing packet to server
  NTPpacket[2] = 0x06;
  NTPpacket[3] = 0xEC;
  NTPpacket[12]  = 0x31;
  NTPpacket[13]  = 0x4E;
  NTPpacket[14]  = 0x31;
  NTPpacket[15]  = 0x34;
  udp.beginPacket(host,123);    //NTP requests are to port 123
  udp.write(NTPpacket,48);      //buffer 48 bytes
  udp.endPacket();              //send 48 bytes
  ts1=millis64();               //outgoing timestamp
  while((millis64()-ts1<500UL)&(!udp.parsePacket())){}      //wait up to a 0.5s for packet to return
  ts2=millis64();               //return timestamp
  udp.read(NTPpacket,48);       //save the returned packet
  ts3=(((unsigned long long)NTPpacket[32])<<56)|(((unsigned long long)NTPpacket[33])<<48)|(((unsigned long long)NTPpacket[34])<<40)|(((unsigned long long)NTPpacket[35])<<32)|(((unsigned long long)NTPpacket[36])<<24)|(((unsigned long long)NTPpacket[37])<<16)|(((unsigned long long)NTPpacket[38])<<8)|(((unsigned long long)NTPpacket[39]));
  ts4=(((unsigned long long)NTPpacket[40])<<56)|(((unsigned long long)NTPpacket[41])<<48)|(((unsigned long long)NTPpacket[42])<<40)|(((unsigned long long)NTPpacket[43])<<32)|(((unsigned long long)NTPpacket[44])<<24)|(((unsigned long long)NTPpacket[45])<<16)|(((unsigned long long)NTPpacket[46])<<8)|(((unsigned long long)NTPpacket[47]));
  if((ts3!=0)&&(ts4!=0)){                           //if valid packet returned
    newboott=ts3-NTPfromms(ts1+(ts2-ts1)/2);        //calculate time offset
    lastadjust=msfromNTP(((long long)newboott-(long long)boott));    //cast to signed so we can check for +/- adjustment
    boott=newboott;                                 //update time offset
    ntplock=true;                                   //fix is good
  }
}

void dodsn(unsigned long dsn){
  byte dim[]={0,31,28,31,30,31,30,31,31,30,31,30,31};   //days in month
  yr=1900;
  mon=1;
  date=dsn+1;                       //1/1/1900 has timestamp 0, ie date of 0 is 1
  while(date>365){                  //count years             
    if(((yr%4)==0)&&(yr!=1900)){
      date--;
    }
    yr++;
    date=date-365;
  }
  if(((yr%4)==0)&&(yr!=1900)){dim[2]++;}  //if a leap year, Feb has 29
  while(date>dim[mon]){                   //count months
    date=date-dim[mon];
    mon++;
  }
  if((date==0)&&(mon==1)){                //glitch on 31/12 of leap years
    yr--;
    date=31;
    mon=12;
  }
}

unsigned long long NTPfromms(unsigned long long t){      //converts a time in ms to time in 64 bit NTP, max precision, will rollover after 194 years
  return ((t<<22)/1000)<<10;
}

unsigned long msfromNTP(unsigned long long t){      //converts a 64 bit NTP time into ms
  return ((t*1000)>>32);                            //resolution will be lost however we do this
}

int seconds(unsigned long t){
  return t%60;
}

int minutes(unsigned long t){
  return (t/60)%60;
}

int hours(unsigned long t){
  return (t/3600)%24;
}

void doESPstat(){                       //prints a sentence that looks like GPS data, but gives us WIFI info
  char sentence[80]="";                 //WiFi.status(),WiFi.localIP(),ntp data received, have we just tried to hit NTP?, last adjustment value in ms
  sprintf(sentence,"$ESP82,%02d,%03d.%03d.%03d.%03d,%01d,%01d,%02d,%s,%11u,%11u",WiFi.status(),WiFi.localIP()[0],WiFi.localIP()[1],WiFi.localIP()[2],WiFi.localIP()[3],ntplock,ntphit,DSTkey,StateName,tshort,DSTevent);
  Serial.println(sentence);
  ntphit=false;                         //clear this flag
}

void doupdate(){                                  //update sentence values
  char sentence[20]="";
  tnow=boott+NTPfromms(millis64());               //get current time
  tshort=(tnow)>>32;                              //get integer seconds part
  tshort = tshort + StateOffset + DSToffset;      // daylight saving time in country
  dodsn(tshort/86400);                            //calculate date
  if ((passes == false) && (ntplock == true))  {  // First time here with valid tshort
    if ( DSTstartmonth == 0 ) {                   // State does not do daylight saving
      passes = true;                              //don't come here again for this state
      DSTevent = 4294967295;                      // a long time in the future
      DSTflag = false;                            //All is well
      DSToffsetCopy = 0;
      DSToffset = 0;
    }
    else  {
    // if this is startup then we need the time of the next DST event in seconds after 1900 to be compatible with tshort
    //unsigned long GetSecs (byte month, byte sun, byte weekday, int year, byte hour)
    // Returns number of seconds from 1/1/1900 to "hour" in "sunth" "weekday" of month, weekday 0 = Sun
      passes = true;                              //done with first time
      if (SH == true)   {                         // southern hemisphere
        DSTevent = GetSecs ( DSTendmonth, DSTendday, DSTendweekday, yr, DSTendHour); //03:00 first Sunday in April in yr
        if (tshort <= DSTevent) {                 //we are in Autumn
          DSToffset = DSToffsetCopy;
          DSTflag = true;                          //All is well
        }
        else {                                      //winter or spring
          DSTevent = GetSecs ( DSTstartmonth, DSTstartday, DSTstartweekday, yr, DSTstartHour); //02:00 first Sunday in October in yr.
          if (tshort <= DSTevent) { //we are in Winter
            DSToffset = 0;
            DSTflag = false;                        //All is well
          }
          else {
            DSToffset = DSToffsetCopy;     
            DSTflag = true;                         //summer late in year
           DSTevent = GetSecs ( DSTendmonth, DSTendday, DSTendweekday, yr + 1, DSTendHour); //03:00 first Sunday in April next yr
          } 
        }
      }
      else  {                                       // northern hemisphere
        DSTevent = GetSecs ( DSTstartmonth, DSTstartday, DSTstartweekday, yr, DSTstartHour); //02:00 first Sunday in April in yr.
        if (tshort <= DSTevent) { //we are in spring
          DSToffset = 0;
          DSTflag = false;                          //All is well
        }
        else {                                   //summer or autumn
          DSTevent = GetSecs ( DSTendmonth, DSTendday, DSTendweekday, yr, DSTendHour); //03:00 first Sunday in October in yr
          if (tshort <= DSTevent) { //we are in summer
            DSToffset = DSToffsetCopy;
            DSTflag = true;                         //All is well
          }
          else {
            DSToffset = 0;     
            DSTflag = false;                        //winter late in year
            DSTevent = GetSecs ( DSTstartmonth, DSTstartday, DSTstartweekday, yr + 1, DSTstartHour); //02:00 first Sunday in April next yr.
          } 
        }
      }
    }
  }
//  lcd.setCursor(0,2);             // set cursor 3rd line
//  sprintf(sentence,"%16u",DSTevent);  
//  lcd.print(sentence);            //  

  if (tshort >= DSTevent) {         //Is it time to start or end DST?
    if (DSTflag == true)  {
      DSTflag = false;
      DSToffset = 0;                //next second will be - 1 hour
      if (SH == true) {             //southern hemisphere
      DSTevent = GetSecs ( DSTstartmonth, DSTstartday, DSTstartweekday, yr, DSTstartHour); //02:00 first Sunday in October in yr.      
      }
      else  {                       //northern
        DSTevent = GetSecs ( DSTstartmonth, DSTstartday, DSTstartweekday, yr + 1, DSTstartHour); //02:00 first Sunday in April next yr.      
      }
    }
    else {                          //start of DST
      DSTflag = true;
      DSToffset = DSToffsetCopy;    //next second will be + 1 hour
      if (SH == true) {
        DSTevent = GetSecs ( DSTendmonth, DSTendday, DSTendweekday, yr + 1, DSTendHour); //03:00 first Sunday in April next yr 
      }
      else  {                       //northern
        DSTevent = GetSecs ( DSTendmonth, DSTendday, DSTendweekday, yr, DSTendHour); //03:00 first Sunday in October this yr 
      }
    }
  }
}

void doRMC(){                             //output an RMC sentence
  char sentence[82]="";                   //standard RMC: time, fix, coordinates, date, checksum
  byte checksum=0x00;
  sprintf(sentence,"$GPRMC,%02d%02d%02d.%03d,%c,%s,0.00,000.00,%02d%02d%02d,,,*" ,hours(tshort),minutes(tshort),seconds(tshort),((unsigned long)(tnow))/4294968UL,ntplock?'A':'V',state,date,mon,yr%100);  
  for(int i=1;i<strlen(sentence)-1;i++){
    checksum=checksum^sentence[i];
  }
  sentence[strlen(sentence)]=hex[(checksum>>4)&0xF];
  sentence[strlen(sentence)]=hex[checksum&0xF];
  Serial.println(sentence);      
}
void doLCD(){
  char const * wdays[] = {"Sun", "Mon", "Tue", "Wed", "Thu", "Fri", "Sat"};
  char sentence[16]="";   
  byte  LCDday, LCDmon, LCDyr;                   //Day of week, Sat = 0, Fri = 6
  if (mon == 1 || mon == 2) {
    LCDmon = mon + 12; 
    LCDyr = yr -1+ 2000;
  }
  else {
    LCDmon = mon;
    LCDyr = yr + 2000;
  }
  LCDday = (date + 13*(LCDmon+1)/5 + (LCDyr % 100)*5/4 + LCDyr/400 + LCDyr/20 + 6) % 7; //Sun = 0, Sat = 6
  lcd.setCursor(0,0);             // set cursor to start of line 1
  sprintf(sentence,"%02d:%02d.%02d %s",hours(tshort),minutes(tshort),seconds(tshort),StateName);
  lcd.print(sentence);            // print top line
  lcd.setCursor(0,1);             // set cursor to start of line 1
  sprintf(sentence,"%02d/%02d/20%02d %s ",date,mon,yr%100,wdays[LCDday]);  
  lcd.print(sentence);            // print 2nd line
  DSTflag ? lcd.print ("D") : lcd.print (" "); 
  lcd.setCursor(0,2);             // set cursor 3rd line
  sprintf(sentence,"%16u",tshort);  
  lcd.print(sentence);            //  
  lcd.setCursor(0,3);             // set cursor to start of line 4
  sprintf(sentence,"%16u",DSTevent);  
  lcd.print(sentence);            //  
}

void doGSA(){                           //output a GSA sentence
  char sentence[82]="";
  byte checksum=0x00;
  sprintf(sentence,"$GPGSA,A,%c,,,,,,,,,,,,,1.00,1.00,1.00,*",ntplock?'3':'1');
  for(int i=1;i<strlen(sentence)-1;i++){
    checksum=checksum^sentence[i];
  }
  sentence[strlen(sentence)]=hex[(checksum>>4)&0xF];
  sentence[strlen(sentence)]=hex[checksum&0xF];
  Serial.println(sentence);        
}

void doGGA(){                           //output a GGA sentence
  char sentence[82]="";
  byte checksum=0x00;
  sprintf(sentence,"$GPGGA,%02d%02d%02d.%03d,%s,%c,04,1.0,0.0,M,0.0,M,,*" ,hours(tshort),minutes(tshort),seconds(tshort),((unsigned long)(tnow))/4294968UL,state,ntplock?'8':'0');
  for(int i=1;i<strlen(sentence)-1;i++){
    checksum=checksum^sentence[i];
  }
  sentence[strlen(sentence)]=hex[(checksum>>4)&0xF];
  sentence[strlen(sentence)]=hex[checksum&0xF];
  Serial.println(sentence);          
}

unsigned long long millis64(){
  static unsigned long lo,hi;   //hi and lo 32 bits
  unsigned long newlo=millis(); //check millis()
  if(newlo<lo){hi++;}           //rollover has happened
  lo=newlo;                     //save for next check
  return ((unsigned long long)(hi) << 32) | ((unsigned long long)(lo));  //return 64 bit result
}

void checkinput(){              //see if user wants to change settings
  if(Serial.available()){
    if(Serial.read()=='~'){
      delay(100);
      while(Serial.available()){Serial.read();}   //purge stream before input
      doconfig();
    }
  }
}

void doconfig(){  
  bool done=false;
  int d,i;
  EEPROM.begin(1024);
  showmenu();
  while(!done){
    if(Serial.available()){
      d=Serial.read();
      if(d>31){Serial.println((char)d);}  
      switch(d){
        case '1':
          baudrate=4800;
          EEPROM.put(0,baudrate);
          delay(100);
          while(Serial.available()){Serial.read();}   //purge stream before input
          showmenu();
          break;
        case '2':
          baudrate=9600;
          EEPROM.put(0,baudrate);
          delay(100);
          while(Serial.available()){Serial.read();}   //purge stream before input
          showmenu();
          break;
        case '3':
          delay(100);
          while(Serial.available()){Serial.read();}   //purge stream before input
          Serial.println(F("Enter SSID:"));
          getstr(ssid);                               //get input
          for(i=0;i<80;i++){
            EEPROM.write(4+i,ssid[i]);                //save to EEPROM
          }
          showmenu();
          break;
        case '4':
          delay(100);
          while(Serial.available()){Serial.read();}   //purge stream before input
          Serial.println(F("Enter password:"));
          getstr(pass);                               //get input
          for(i=0;i<80;i++){
            EEPROM.write(84+i,pass[i]);               //save to EEPROM
          }
          showmenu();
          break;
        case '5':
          delay(100);
          while(Serial.available()){Serial.read();}   //purge stream before input
          Serial.println(F("Enter NTP Server:"));
          getstr(host);                               //get input
          for(i=0;i<80;i++){
            EEPROM.write(164+i,host[i]);              //save to EEPROM
          }
          showmenu();
          break;
        case '6':
          delay(100);
          while(Serial.available()){Serial.read();}   //purge stream before input
          Serial.println("Enter number of required country");
          DST_data (0, 1);                     //Print table
          getstr(state);                        //get input
          for(i=0;i<80;i++){
            EEPROM.write(244+i,state[i]);       //save to EEPROM
          }
          showmenu();
          if (atoi(state) < MaxTable)  {
            DSTkey = atoi(state);
          }
          else  {
            DSTkey = 5;                         //NSW
          }
          DST_data (DSTkey, 0);                //From EE
          passes = false;                       //force change
          break;
        case '9':
          delay(100);
          while(Serial.available()){Serial.read();}   //purge stream before input
          Serial.println(F("Exit config, saved"));
          EEPROM.end();
          done=true;
          break;
      }
    }
  }  
}

void loadconfig(){
  int i;
  Serial.begin(9600);
  EEPROM.begin(1024);
  EEPROM.get(0,baudrate);
  Serial.println(baudrate);
  if((baudrate!=4800)&&(baudrate!=9600)){     //sanity check for corrupt EEPROM
    baudrate=BAUDRATE_DEFAULT;                //load sane defaults
    EEPROM.put(0,baudrate);                   //save safe defaults      
    for(i=0;i<80;i++){
      EEPROM.write(4+i,ssid[i]);
      EEPROM.write(84+i,pass[i]);
      EEPROM.write(164+i,host[i]);
      EEPROM.write(244+i,state[i]);
    }
    Serial.println(F("Defaults written to EEPROM"));        
  }else{
    for(i=0;i<80;i++){                        //load from eeprom
      ssid[i]=EEPROM.read(4+i);
      pass[i]=EEPROM.read(84+i);
      host[i]=EEPROM.read(164+i);
      state[i]=EEPROM.read(244+i);
      }
      DSTkey = atoi(state);
    }
  ssid[79]=0;           //null terminate in case we have garbage
  pass[79]=0;
  host[79]=0;
  state[79]=0;
//  }
  EEPROM.end();
  delay(100);
  Serial.end();
}

void showmenu(){
  Serial.println();
  Serial.println(F("NTP GPS Source Setup:"));
  Serial.print(F("Current Baudrate:"));
  Serial.println(baudrate);
  Serial.println(F("1.Set 4800 Baudrate"));
  Serial.println(F("2.Set 9600 Baudrate"));
  Serial.print(F("3.Set SSID. Current:"));
  Serial.println(ssid);
  Serial.print(F("4.Set Password. Current:"));
  Serial.println(pass);
  Serial.print(F("5.Set NTP Server. Current:"));
  Serial.println(host);
  Serial.print(F("6.Set State. Current:"));
  Serial.println(state);
  Serial.println(F("9.Exit and save"));
  Serial.println(F("Enter a number:"));
}

void getstr(char *s){
  char buf[100]="";
  bool done=false;
  int i,d;
  while(!done){
    if(Serial.available()){
      d=Serial.read();
      if(d>31){
        Serial.write(d);
        buf[strlen(buf)]=d;   
      }      
      if((d==13)||(strlen(buf)>78)){                //jump out
        done=true;
        Serial.println();
        delay(100);
        while(Serial.available()){Serial.read();}   //purge stream before input
      }
    }
  }
  for(i=0;i<80;i++){
    s[i]=buf[i];
  }
  s[79]=0;        //null terminate
}

  void setlatlon(String a){               //extract latitude/longitude in RMC format from string in lat(CR)lon format. Works to nearest minute as this is <2km
/*  int lonindex;
  float lat,lon; 
  int lati,latf,loni,lonf;              //latitude/longitude integer/fractional parts
  char ns='N';
  char ew='E';                          //for positive values
  lat=a.toFloat();                      //latitude is first line
  lonindex=a.indexOf('\n')+1;           //look for /n
  lon=a.substring(lonindex).toFloat();  //longitude is on second line
  if((lat==0)||(lon==0)){return;}       //if either are zero, data is probably wrong
  if(lat<0){                            //check if negative and change compass direction
    ns='S';
    lat=-lat;
  }
  if(lon<0){                            //check if negative and change compass direction
    ew='W';
    lon=-lon;
  }
  lati=lat;
  latf=(lat-lati)*60;                   //counts up to 59 minutes
  loni=lon;
  lonf=(lon-loni)*60;                   //counts up to 59 minutes  
  sprintf(dummy,"%02d%02d.000,%c,%03d%02d.000,%c",lati,latf,ns,loni,lonf,ew);   */
}   

// ***************************************************************************/
unsigned long GetSecs (byte month, byte sun, byte weekday, int year, byte hour) {
// Returns number of seconds from 1/1/1900 to "hour" in "sunth" "weekday" of month
// weekday 0 = Sun

  byte day;
  unsigned int j;
  unsigned long yearsecs = 0;
  byte months[] = {0, 31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31};
  
  if (sun < 7)  { //not the last sunday in the month
    day = GetDay (month, 1, year); //day of first of month
    if (day == weekday) {
      day = ((sun -1) * 7 + 1);
    }
    else  {
      if (day > weekday)  {
        day = ((8 + weekday - day) + (sun - 1) * 7);
      }
      else  {
        day = ((1 + weekday - day) + (sun - 1) * 7);
      }
    }
  }
  else  { //last sunday  
    if ((month == 2) && ((year & 3ul) == 0))  { //feb in leap year
      day = GetDay (month, 29, year);
      if (weekday <= day) {
        day = (29 + weekday - day); //last sunday
      }
      else  {
        day = (29 - 7 + weekday - day);
      }
    }
    else  {
      day = GetDay (month, months[month], year);
    }
    if (weekday <= day) {
      day = (months[month] + weekday - day); //last sunday
    }
    else  {
      day = (months[month] - 7 + weekday - day);
    }
  }
  for (j = 1900; j < year; j++) {
    if (((j & 3ul) > 0) ||( j == 1900))   { //not leap year
      yearsecs = yearsecs + (31536000);
    }
    else  {
      yearsecs = yearsecs + (31622400); //seconds in a leap year
    }
  }
  for (j = 1; j < month; j++) {
    yearsecs = yearsecs + (months[j] * 86400ul);
    if ((j == 2) && (year & 3ul) == 0)  {
      yearsecs = yearsecs + 86400;
    }
  }
  yearsecs = yearsecs + ((day - 1) * 86400ul);
  yearsecs = yearsecs + (hour * 3600ul);
  return yearsecs;
}

byte GetDay (byte month, byte date, int year) { // Returns dayofweek for a date, sun = 0
  int t[] = {0, 3, 2, 5, 0, 3, 5, 1, 4, 6, 2, 4};
  year -= month < 3;
  return ((year + year/4 - year/100 + year/400 + t[month-1] + date) % 7);
}

void togglePPS(){
pinMode(PPSPIN,HIGH);
delay(10);
pinMode(PPSPIN,LOW);
}

void DST_data (byte dTZptr, byte action )
{
  typedef struct _WorldDST 
  {
    char WorldName[8];
    unsigned char SUTCoffset;  //Note * 15 minutes. byte has 0-96, need -48 to +48, so subtract 48
              //  Also Display now in first bit, 0 = dd/mm/yyyy, 1 = mm/dd/yyyy
    unsigned char SDSMSS;    // daylight saving start month & Sunday; MMMM.DSSS  SSS = 1st to 4th, 7 = last
              //  Also (in bit 3) msb of DSO -1, 0 = 15 / 45 min, 8 = 30 min / 1hr
    unsigned char SDSMSE;    // daylight saving end month & Sunday; MMMM.DSSS  SSS = 1st to 4th, 7 = last
              //  Also (in bit 3) lsb of DSO -1, 0 = 15 / 30 min, 8 = 45 min / 1hr
    unsigned char SDSSDH;    // daylight saving start day & hour; DDDH.HHHH  Sun = 0
    unsigned char SDSEDH;    // daylight saving end day & hour; DDDH.HHHH  Sun = 0
  } WorldDST;
  WorldDST TZ1[43]=   {
//     Disp    Off SM      Sun msb  EM    Sun lsb  SD     Hr   ED     Hr
   "Suva   ", 0*128 + 96, 10*16 + 4 + 8,  1*16 + 3 + 8,  0*32 + 02,  0*32 + 03, //FST, Fiji, + 12 has DST from 0200 4th sun oct to 0300 3rd sun in Jan
   "NZ     ", 0*128 + 96,  9*16 + 7 + 8,  4*16 + 1 + 8,  0*32 + 02,  0*32 + 03, //NZ last sunday in september to first sunday in april
   "L How I", 0*128 + 90, 10*16 + 1 + 8,  4*16 + 1 + 0,  0*32 + 02,  0*32 + 03, //LHT GMT + 10.5, DST + .5, first sunday in october to first sunday in april
   "Norfk I", 0*128 + 92, 10*16 + 1 + 8,  4*16 + 1 + 8,  0*32 + 02,  0*32 + 03, //NFT GMT + 11,DST +1, first sunday in october to first sunday in april
   "QLD Aus", 0*128 + 88,  0*16 + 0 + 8,  0*16 + 0 + 8,  0*32 + 00,  0*32 + 00, //Australian Eastern Std Time, Brisbane, does not have daylight saving  
   "NSW Aus", 0*128 + 88, 10*16 + 1 + 8,  4*16 + 1 + 8,  0*32 + 02,  0*32 + 03, //AEDT, Sydney, +10, first sunday in october to first sunday in april
   "SA Aus ", 0*128 + 86, 10*16 + 1 + 8,  4*16 + 1 + 8,  0*32 + 02,  0*32 + 03, //CDT, Adelaide, +9.5, first sunday in october to first sunday in april
   "S'Pore ", 0*128 + 80,  0*16 + 0 + 8,  0*16 + 0 + 8,  0*32 + 00,  0*32 + 00, //SGT, Singapore, + 8, No DST
   "WA Aus ", 0*128 + 80,  0*16 + 0 + 8,  0*16 + 0 + 8,  0*32 + 00,  0*32 + 00, //WST, Perth, + 8, No DST
   "Xmas I ", 0*128 + 76,  0*16 + 0 + 8,  0*16 + 0 + 8,  0*32 + 00,  0*32 + 00, //CXT, Christmas Island + 7, No DST
   "Cocos I", 0*128 + 74,  0*16 + 0 + 8,  0*16 + 0 + 8,  0*32 + 00,  0*32 + 00, //CCT, + 7.5, No DST
   "Bangkok", 0*128 + 76,  0*16 + 0 + 8,  0*16 + 0 + 8,  0*32 + 00,  0*32 + 00, //ICT, Thailand, + 7, No DST
   "Dhaka  ", 0*128 + 72,  0*16 + 0 + 8,  0*16 + 0 + 8,  0*32 + 00,  0*32 + 00, //BST, Bangladesh, + 6, No DST
   "Delhi  ", 0*128 + 70,  0*16 + 0 + 8,  0*16 + 0 + 8,  0*32 + 00,  0*32 + 00, //IST,  India, + 5 1/2, No DST
   "Karachi", 0*128 + 68,  0*16 + 0 + 8,  0*16 + 0 + 8,  0*32 + 00,  0*32 + 00, //PKT, Pakistan, + 5, No DST
   "Muscat ", 0*128 + 64,  0*16 + 0 + 8,  0*16 + 0 + 8,  0*32 + 00,  0*32 + 00, //GST, Oman, + 4, No DST
   "Riyadh ", 0*128 + 60,  0*16 + 0 + 8,  0*16 + 0 + 8,  0*32 + 00,  0*32 + 00, //AST, Saudi Arabia, + 3, No DST
   "Gaza   ", 0*128 + 56,  3*16 + 1 + 8, 10*16 + 4 + 8,  0*32 + 00,  5*32 + 00, //Gaza strip, DST ends midnight Thu/Fri 24th October
   "Amman  ", 0*128 + 56,  0*16 + 0 + 8,  0*16 + 0 + 8,  0*32 + 00,  0*32 + 00, //EET,  Jordan, + 2, no DST
   "Namibia", 0*128 + 52,  9*16 + 1 + 8,  4*16 + 1 + 8,  0*32 + 02,  0*32 + 02, //NAM,  Namibia, + 1, first out of dst
   "Berlin ", 0*128 + 52,  3*16 + 7 + 8, 10*16 + 7 + 8,  0*32 + 02,  0*32 + 03, //CET,  Germany, +1, 0200 last sun in march to 0300 last sun in October
   "UK     ", 0*128 + 48,  3*16 + 7 + 8, 10*16 + 7 + 8,  0*32 + 02,  0*32 + 02, //GMT,  London has DST from last sunday in march to last sunday in october
   "Antarct", 0*128 + 48,  3*16 + 1 + 8, 10*16 + 7 + 8,  0*32 + 01,  0*32 + 02, //GMT,  Troll Station start 1/3/15 01:00 end ??
   "Cblanca", 0*128 + 48,  4*16 + 7 + 8,  7*16 + 3 + 8,  0*32 + 02,  5*32 + 03, //GMT,  Morocco has DST from last sunday in april 0200 to 3rd friday in july 0300
   "Sc'sund", 0*128 + 44,  3*16 + 7 + 8, 10*16 + 7 + 8,  0*32 + 00,  0*32 + 01, //EGT (East G'nland), Scoresbysund, GMT - 1, 0000 last sun in mar to 0100 last sun in Oct
   "Fern de", 0*128 + 40,  0*16 + 0 + 8,  0*16 + 0 + 8,  0*32 + 00,  0*32 + 00, //FNT Fernando de Noronha, Pernambuco, GMT - 2, no DST
   "Nuuk   ", 0*128 + 36,  3*16 + 7 + 8, 10*16 + 7 + 8,  6*32 + 22,  6*32 + 23, //WGT (West G'nland), Nuuk, GMT - 3, 2200 last sat in mar to 2300 last sat in Oct
   "Brazil ", 0*128 + 36, 10*16 + 3 + 8,  2*16 + 4 + 8,  0*32 + 00,  0*32 + 00, //BRT,  Brazil, GMT -3, midnight 3rd sat / sun Oct to midnight 3rd sat / sun in Feb
//              Disp    Off SM     Sun msb  EM    Sun lsb  SD     Hr   ED     Hr
   "P'guay ", 0*128 + 32, 10*16 + 1 + 8,  3*16 + 4 + 8,  0*32 + 00,  0*32 + 00,   //PYT,  Paraguay, Asuncion, GMT -4, midnight 1st Sat/Sun Oct to midnight 4th sat / sun March
   "Stanley", 0*128 + 36,  0*16 + 0 + 8,  0*16 + 0 + 8,  0*32 + 00,  0*32 + 00, //FKST, Stanley, Falkland Islands, GMT - 4, No change to DST, DST all year
   "NewYork", 1*128 + 28,  3*16 + 2 + 8, 11*16 + 1 + 8,  0*32 + 02,  0*32 + 02, //EST,  New York, GMT - 5, 2nd Sun in March to 1st Sun in November
   "Mexico ", 1*128 + 28,  3*16 + 2 + 8, 10*16 + 7 + 8,  0*32 + 02,  0*32 + 02, //EST,  Mexico, GMT - 5, 2nd Sun in March??? to last Sunday in October
   "Chicago", 1*128 + 24,  3*16 + 2 + 8, 11*16 + 1 + 8,  0*32 + 02,  0*32 + 02, //CDT,  Chicago, Illinois, -6
   "E/Isle ", 0*128 + 24,  4*16 + 0 + 8,  9*16 + 0 + 8,  9*32 + 00,  2*32 + 02, //EAST, Chile, Easter Island, -6, last Sun April 00 to 2nd Sun Sep 00
   "Tucson ", 1*128 + 20,  0*16 + 0 + 8,  0*16 + 0 + 8,  0*32 + 00,  0*32 + 00, //MST,  Mountain Standard Time - Tucson, -7, No DST
   "Seattle", 1*128 + 16,  3*16 + 2 + 8, 11*16 + 1 + 8,  0*32 + 02,  0*32 + 02, //PST,  Seattle/Vancouver, GMT - 8, 2nd Sun in March to 1st Sun in November
   "Alaska ", 1*128 + 12,  3*16 + 2 + 8, 11*16 + 1 + 8,  0*32 + 02,  0*32 + 02, //AKST, Anchorage, GMT - 9, 2nd Sun in March to 1st Sun in November
   "Hawaii ", 1*128 +  8,  0*16 + 0 + 8,  0*16 + 0 + 8,  0*32 + 00,  0*32 + 00, //HAST, Hawaii-Aleutian Standard Time, GMT - 10, No DST
   "Alofi  ", 0*128 +  4,  0*16 + 0 + 8,  0*16 + 0 + 8,  0*32 + 00,  0*32 + 00, //NUT,  Niue Time, GMT - 11, No DST
   " End  2", 0*128 +  0,  0*16 + 0 + 8,  0*16 + 0 + 8,  0*32 + 00,  0*32 + 00, //Terminator, First char is space
   " End  1", 0*128 +  0,  0*16 + 0 + 8,  0*16 + 0 + 8,  0*32 + 00,  0*32 + 00, //Terminator, First char is space
   " End   ", 0*128 +  0,  0*16 + 0 + 8,  0*16 + 0 + 8,  0*32 + 00,  0*32 + 00  //Terminator, First char is space
  };
  byte counter;
  char letter;
  char sentence[80]="";
  char Name[8]="";
  if (action == 0) {               //return parameters for pointer
    for (counter = 0; counter < 7; counter++) { //Copy Time Zone identifier to TZone
      letter = TZ1[dTZptr].WorldName[counter];
      StateName[counter] = letter;
    } 
    StateOffset      = ((TZ1[dTZptr].SUTCoffset & 0x7F) -48) * 900;    //(NB seconds)
    DateOrder        = (TZ1[dTZptr].SUTCoffset & 0x80) /128;   //(0 = dd/mm/yyyy, 1 = mm/dd/yyyy)
    DSTstartmonth    = (TZ1[dTZptr].SDSMSS & 0xF0) /16;      //(daylight saving start month)
    DSTendmonth      = (TZ1[dTZptr].SDSMSE & 0xF0) /16;      //(daylight saving end month)
    DSTstartday      = TZ1[dTZptr].SDSMSS & 0x07;    //(daylight saving start Sunday [7 = last])
    DSTendday        = TZ1[dTZptr].SDSMSE & 0x07;    //(daylight saving end Sunday [7 = last]))
    DSTstartweekday  = TZ1[dTZptr].SDSSDH /32;     //(daylight saving start day [0 = Sunday])
    DSTendweekday    = TZ1[dTZptr].SDSEDH /32;     //(daylight saving end day [0 = Sunday]))
    DSToffset        = (((TZ1[dTZptr].SDSMSS & 0x08)/8 *2 + (TZ1[dTZptr].SDSMSE & 0x08)/8) + 1) * 900;   //daylight saving offset - in seconds
    DSTstartHour     = TZ1[dTZptr].SDSSDH & 0x1F;    //(daylight saving start hour)
    DSTendHour       = TZ1[dTZptr].SDSEDH & 0x1F;    //(daylight saving end hour) 
    DSToffsetCopy = DSToffset;
    if ( DSTendmonth < DSTstartmonth )  {
      SH = true;
    }
    else  {
      SH = false;      
    }
  }
  else    {     //print to serial all country names and their pointer
    counter = 0;
    letter = TZ1[counter].WorldName[0];
    while (letter != 0x20)  {
      sprintf(sentence, "%02d = %s",counter,TZ1[counter].WorldName);
      Serial.println(sentence);
      counter++;
      letter = TZ1[counter].WorldName[0];
    }
    MaxTable = counter;
  }
}
