'----------------------------------------------------------------------------------
' Tiny85 DDS PWM Audio Oscillator
' Software by: Andrew Woodfield
' Date: Feb 2020
' File Name: AudioGen_64x32_72x40_PencoderV04.bas
'----------------------------------------------------------------------------------
' Program Description:

' An Attiny85 is connected via I2C to a 0.49" 64 x 32 bit I2C OLED display
' with its integrated SSD1306 display controller and a standard rotary encoder.
'
' The encoder and its integrated shaft push switch are connected to the ATtiny85
' via a single ADC input on Pin 5 to allow the frequency of a variable audio sine
' tone from 1Hz to 9999Hz to be selected in 1, 10, 100 or 1000Hz steps.
'
' The PWM output is low pass filtered and level controlled via a variable resistor.
'
' Note: Only a PULSE-type rotary encoder will work with this software
'----------------------------------------------------------------------------------
' Fuse settings
'
' LOCK Byte: 0ffh (No locks)
' EXTd Byte: 0ffh (BOD disabled)
'
' HIGH Byte: 05fh
'   RSTDISBL 0 RST disabled
'       DWEN 1 DW not enabled
'      SPIEN 0 SPI programming enabled
'      WDTON 1 Watchdog timer off
'     EESAVE 1 EEPROM not preserved in erase
'  BODLEVEL2 1 Brown Out Detector disabled
'  BODLEVEL1 1
'  BODLEVEL0 1
'
' LOW Byte:  0efh
'     CKDIV8 1 NOT divided by 8
'      CKOUT 1 Not enabled
'       SUT1 1 Slow rising power
'       SUT0 0
'     CKSEL3 1 8 MHz EXT crystal
'     CKSEL2 1
'     CKSEL1 1
'     CKSEL0 1
'
'------------------------------------------------------------------
' Compiler Directives

$regfile = "attiny85.dat"
$crystal = 8000000                                          '8MHz EXTERNAL crystal

$hwstack = 64
$swstack = 64
$framesize = 64

' Port configuration

Config PortB = input                                        'set all to inputs first

Config Scl = Portb.2                                        'i2c interface pins...
Config Sda = PortB.0                                        '...for OLED display

config PORTB.1 = output                                     'set up PWM output pin

' Variable declarations

dim involts as word                                         'input voltage from resistor divider
dim oldvalue as word                                        'previous adc value
dim freqvalue as word                                        'frequency (Range: 0 - 9999)
dim tempvalue as word                                       'saves adc value during display routine
dim v4 as word , v3 as word , v2 as word , v1 as word       'four digits for freqvalue "JKLM" Hz (where f4=J)

dim tempdata as byte , page as byte , J as byte , K as byte 'general use

dim send as byte                                            'holds value sent during i2c data write routine

dim digitpointer as byte                                    'points to location to display current digit
dim digitvalue as word                                      'value of current digit to be displayed

dim loopcounter as word                                     ' counter
dim innercounter as byte                                    'test loop counter
dim startcolumn as byte                                     'start and end columns for horizontal addressing mode
dim endcolumn as byte
dim baseaddr as byte                                        'column addr for page addressing mode

dim delta as word                                           'decimal value for oscillator offset for display update
dim ddsdelta as word                                        'hex value for oscillator offset for dds freq update
dim deltapointer as byte                                    'pointer to delta value

dim ddsfreq as dword                                        '32 bit word but only 24 bits are used in this application
Dim ddslow As Byte At ddsfreq Overlay
Dim ddsmid As Byte At ddsfreq + 1 Overlay
Dim ddshigh As Byte At ddsfreq + 2 Overlay
Dim ddsmax As Byte At ddsfreq + 3 Overlay                   'just here for completeness

dim changeflag as bit                                       'dds update flag


' Subroutine declarations

Declare Sub CLRoled
Declare Sub Oled_init
declare sub dump_graphic
declare sub showfreqvalue
declare Sub display_16x32char



'main code starts here

I2cinit                                                     'initialise the i2c interface

waitms 100                                                  'recommended by display vendors

Call Oled_init                                              'initialise oled display

CLRoled                                                     'clean up display

Restore Splash                                              'reset data read pointer to start of graphics data table
Call Dump_graphic()                                         'send the data to the screen
wait 2                                                      'and wait so we can see it

CLRoled                                                     'clean up display

'initialise adc
config adc = single , prescaler = auto , reference = avcc   'starts automatically with supply rail reference

'configure 8-bit counter timer 0 for Fast PWM operation
TCCR0A = &H23
TCCR0B = &H09                                               'set Fast PWM Mode 7
OCR0B = &H3F                                                'set PWM for mid-range 'average zero' output
OCR0A = &H7F                                                'set TOP for Mode 7 Fast PWM


Enable Interrupts                                           'enable global ints
Enable Pcint0                                               'enable pin change interrupts
On Pcint0 int_encoder                                       'vector for pin change
Pcmsk = &B00100000                                          'and enable pcint0 for pinb.5 (RST pin)


delta = 1                                                   'start with frequency increment of 1
ddsdelta = 21                                               'and dds increment of 21 for 24-bit dds
deltapointer = 1                                            'deltapointer=1-4 for delta=1, 10, 100 and 1000
freqvalue = 1000                                            'set to 1000Hz at startup
changeflag = 0                                              'initialise flag


'set up the stored audio tone (1000Hz with 8MHz clock and 24 bit DDS requires DDS=0x0051EC)

ddshigh = &H00
ddsmid = &H51
ddslow = &HEC


entrypoint:                                                 'simple read and write routine which runs continuously

         showfreqvalue                                      'calculate and display the freqvalue on the oled

'setup adder registers from stored DDS tone value

         R5 = ddshigh                                       'set dds adder value
         R4 = ddsmid                                        ' 24 bit phase acc
         R3 = ddslow


    $asm

         cbi $11,0                                          'clear bit0 of GPIO0 for use in next interrupt routine

'point Z register pair to the start of the 7-bit quadrature wave table for DDS loop byte output

         ldi   r31,high(sinewave*2)                         'setup Z pointer hi
         ldi   r30,low(sinewave*2)                          'setup Z pointer lo       '

'clear 24-bit dds accumulator

         clr   r30                                          'low byte of Z-pointer
         clr   r2
         clr   r1

' main dds loop
'
'        r30,r2,r1 is the phase accumulator
'          r5,r4,r3 is the adder value determining frequency
'
'        add value to accumulator
'        load byte from ROM table
'        output byte to pwm oscillator
'        repeat (forever)
'
'
ddsloop1:
         ADD   r1 , r3                                      '1 cycle
         adc   r2,r4                                        '1
         adc   r30,r5                                       '1 - r30 is low order byte of 16-bit Z-pointer
         lpm                                                '3 - get byte (from quadtable) @ Z-pointer (r31,r30) into r0
         !out OCR0B , r0                                    '1 Fast PWM
         sbis $11,0                                         '1 if false, 2 if true
         rjmp  ddsloop1                                     '2 => 10 cycles
         rjmp  entrypoint

     $end asm



end                                                         'never reach this point...



'===================================================================================================================
'interrupt routines
'-------------------------------------------------------------------------------------------------------------------

int_encoder:                                                'pin change interrupt vector routine for one-pin rotary encoder

   involts = getadc(0)
   if involts < 90 then
      incr deltapointer
      if deltapointer = 5 then deltapointer = 1
      select case deltapointer
         case 1 :
            delta = 1
            ddsdelta = 21
         case 2 :
            delta = 10
            ddsdelta = 210
         case 3 :
            delta = 100
            ddsdelta = 2097
         case 4 :
            delta = 1000
            ddsdelta = 20972
      end select
   end if

   if involts > 140 and involts < 170 then
      freqvalue = freqvalue + delta
      ddsfreq = ddsfreq + ddsdelta
      set changeflag
   end if

   if involts > 250 and involts < 320 then
      freqvalue = freqvalue - delta
      ddsfreq = ddsfreq - ddsdelta
      set changeflag
   end if


   if changeflag = 1 then                                   'display and update dds if freqvalue changed

      reset changeflag                                      'reset flag for next time

      $asm
      sbi $11,0                                             'set bit0 of GPIO0 to flag mainline code
      $end asm

      if freqvalue > 64500 then                              'has freqvalue gone negative?
         freqvalue = 1
         ddsfreq = 21
      end if
      if freqvalue > 9999 then                               'check limits
         freqvalue = 9999
         ddsfreq = 209694
      end if

   end if

Return



'===================================================================================================================
'subroutines
'-------------------------------------------------------------------------------------------------------------------

Sub Oled_init

   I2cstart
   I2cwbyte &H78
   I2cwbyte &HAE                                            'DISPLAYOFF
   I2cwbyte &HD5                                            'SETDISPLAYCLOCKDIV
   I2cwbyte &HF0                                            'ratio was 0x80 but 64x32 jitter less with F0

'   I2cwbyte &H00                                            'SETMULTIPLEX ***some vendors say this command pair is required
'   I2cwbyte &H1F                                            '  3f 128x64  ***but 64x32 oled displays do not like this command!!
   I2cwbyte &HD3                                            'SETDISPLAYOFFSET
   I2cwbyte &H00

   I2cwbyte &H40                                            'SETSTARTLINE
   I2cwbyte &H8D                                            'CHARGEPUMP
   I2cwbyte &H14                                            'vccstate 14
   I2cwbyte &H20                                            'MEMORYMODE
   I2cwbyte &H00                                            '
   I2cwbyte &HA1                                            'A0 upside down SEGREMAP
   I2cwbyte &HC8                                            'COMSCANDEC

   I2cwbyte &HDA                                            'SETCOMPINS
   I2cwbyte &H12                                            ' 02 128x32  12  128x64

'   I2cwbyte &HAD                                            'INT IREF SETTING  ***some vendors suggest this four-byte sequence
'   I2cwbyte &H30                                            ' value            ***but they do not work well with 64x32 oleds

'   I2cwbyte &H81                                            'SETCONTRAST
'   I2cwbyte 255                                             'contrast value 1-->256

   I2cwbyte &HD9                                            'SETPRECHARGE
   I2cwbyte &HF1                                            'vccstate  f1
   I2cwbyte &HDB                                            'SETVCOMDETECT

   I2cwbyte &H40
   I2cwbyte &HA4                                            'DISPLAYALLON_RESUME
   I2cwbyte &HA6                                            'NORMALDISPLAY

   I2cwbyte &HAF
   I2cstop

End Sub


'----------------------------------------------------------------------------------------------------

Sub CLRoled                                                 'clear OLED display/memory

   Page = &HB0                                              'point to row 0

   for J = 0 to 7                                           '8 rows in the 128 x 64 pixel SSD1306

      I2cstart
      I2cwbyte &H78                                         'send device address
      I2cwbyte &H00                                         'command stream follows
      tempdata = page + J
      I2cwbyte tempdata                                     'set memory pointer to select desired row
      I2cstop

      I2cstart                                              'For each of the 128 columns in each 8 pixel rows in the 132 x 64 SH1106 screen MEMORY...
      I2cwbyte &H78
      I2cwbyte &H40                                         'next bytes will be data
      For K = 0 to 127
         I2cwbyte &H00                                      'clear 8 bits in each bit-wide column
      Next K
      I2cstop

   next J

End Sub

'-----------------------------------------------------------------------------------------------------
Sub Dump_graphic

  'Function    : Dump graphic to LCD
  'Inputs      : None
  'Outputs     : None
  'Assumptions : The i2c bus needs to be initalised before calling this routine
  '
  '              The graphic data table must be selected with RESTORE before calling this routine
  '
  '              The display comprises 4 x 8-bit high 'rows' or 'pages' of 64 (bit-width) columns
  '              which are displayed CENTRED from the internal 128 bytes/row SSD1306 graphics memory
  '              The 4 pages (rows) are displayed starting from page/row 3 in display memory


Page = 180                                                  'point to row 3 (Range 0-7, where row 0 = &HB0)

   for J = 0 to 3                                           '4 rows in the 64 x 32 pixel SSD1306 screen

      I2cstart
      I2cwbyte &H78                                         'send device address
      I2cwbyte &H00                                         'one command is to follow
      tempdata = page + J
      I2cwbyte tempdata                                     'set memory pointer to select desired row
      I2cwbyte &H00                                         'reset column pointer to start at column 32
      I2cwbyte &H12
      I2cstop

      I2cstart                                              'For each of the 64 columns in each 8-pixel rows in the 64 x 32 screen MEMORY...
      I2cwbyte &H78
      I2cwbyte &H40                                         'next bytes will be data
      For K = 0 to 63
         read tempdata                                      'then get data from table
         I2cwbyte tempdata                                  'clear 8 bits in each bit-wide column
      Next K
      I2cstop

   next J

End Sub



'--------------------------------------------------------------------------------------------------

sub showfreqvalue

      tempvalue = freqvalue                                 'copy freqvalue to preserve original (Range: 0 - 9999)
      v4 = tempvalue \ 1000                                 'extract msd (digit4)
      tempvalue = tempvalue mod 1000

      v3 = tempvalue \ 100
      tempvalue = tempvalue mod 100

      v2 = tempvalue \ 10
      v1 = tempvalue mod 10

      'load digit#4 into LH char location
      digitpointer = 3
      digitvalue = v4
      display_16x32char

      digitpointer = 4
      digitvalue = v3
      display_16x32char

      digitpointer = 5
      digitvalue = v2
      display_16x32char

      digitpointer = 6
      digitvalue = v1
      display_16x32char

      'dp and units characters were written once only at very start of code

end sub

'--------------------------------------------------------------------------------------------------

Sub display_16x32char
  'Function    : Dump four 16-byte lines (i.e. 16 columns x 32 rows of pixels/bits) of graphics to OLED
  '              starting in line 1 and from various columns across the display. The 64 pixel wide
  '              display allows up to 4 chars to be displayed
  'Inputs      : None
  'Outputs     : None

'Four full-height chars can fit across 64x32 OLED using a 16x32 font
'
'The graphics data for digits 0-9 is located in 'numbertable' with the graphics data for the special
'space and dp characters immediately following the digit data. This allows the two special characters
'to be referenced as follows:
'
' Special Char    digitvalue
'     space          &hA
'      dp            &hB
'
'
  digitvalue = digitvalue * 64                              'point to start of table graphics data for this number
  startcolumn = digitpointer - 1
  startcolumn = startcolumn * 16
  endcolumn = startcolumn + 15                              'and determine end

  I2cstart
  I2cwbyte &H78                                             'oled i2c address
  I2cwbyte &H00                                             'command mode - stream
  I2cwbyte &H20                                             'select horizontal addressing mode  (was page addressing mode)
  I2cwbyte &H00
  I2cstop

  I2cstart
  I2cwbyte &H78
  I2cwbyte &H00
  I2cwbyte &H21                                             'define start and end columns
  I2cwbyte startcolumn
  I2cwbyte endcolumn
  I2cstop

  I2cstart
  I2cwbyte &H78
  I2cwbyte &H00
  I2cwbyte &H22                                             'Select page start and end (Note: Always page 4 to page 7 for 64x32 oled)
  I2cwbyte 4
  I2cwbyte 7
  I2cstop

  I2cstart
  I2cwbyte &H78
  I2cwbyte &H80
  I2cstop

  I2cstart
  I2cwbyte &H78
  I2cwbyte &H40                                             'and get ready to send the data...

  For loopcounter = 1 To 64                                 '64 bytes per char for 16x32 pixel chars
    send = lookup(digitvalue , numbertable)                 'read data from table
    I2cwbyte Send                                           'send to display
    incr digitvalue                                         'and point to the next table data
  Next loopcounter

  I2cstop
End Sub

'--------------------------------------------------------------------------------------------------


'-----------------------------------------------------------------------------
'LCD graphics data lookup tables
'-----------------------------------------------------------------------------
splash:                                                     '64x32 audio osc power-on screen

data &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &hC0 , &hE0 , &hE0 , &h60 , &h60 , &hE0 , &hC0 , &h00,
data &h00 , &h20 , &hE0 , &hE0 , &hE0 , &h20 , &h00 , &hE0 , &hE0 , &h20 , &h00 , &h20 , &h20 , &hE0 , &hE0 , &h20,
data &h20 , &hE0 , &hE0 , &hC0 , &h00 , &h00 , &h20 , &hE0 , &hE0 , &hE0 , &h20 , &h00 , &h00 , &h80 , &hC0 , &hE0,
data &h60 , &h20 , &h60 , &hE0 , &hC0 , &h80 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00,
data &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h20 , &h3F , &h3F , &h3F , &h24 , &h04 , &h3F , &h3F , &h20,
data &h00 , &h00 , &h1F , &h3F , &h2F , &h30 , &h30 , &h1F , &h0F , &h00 , &h00 , &h20 , &h20 , &h3F , &h3F , &h20,
data &h20 , &h3F , &h3F , &h1F , &h00 , &h00 , &h20 , &h3F , &h3F , &h3F , &h20 , &h00 , &h00 , &h0F , &h1F , &h3F,
data &h30 , &h20 , &h30 , &h3F , &h1F , &h0F , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00,
data &h00 , &h00 , &h00 , &h1C , &h0C , &h04 , &h04 , &hFC , &hFC , &h04 , &h0C , &h1C , &h00 , &hC0 , &hE0 , &h20,
data &hE0 , &hC0 , &h00 , &h20 , &hE0 , &hE0 , &h20 , &hE0 , &hC0 , &h00 , &h00 , &hC0 , &hE0 , &hE0 , &hA0 , &hE0,
data &h40 , &h00 , &h00 , &h00 , &h00 , &hF0 , &hF8 , &hFC , &h0C , &h04 , &h0C , &hF8 , &hF0 , &h00 , &h00 , &hC0,
data &hE0 , &hA0 , &h60 , &h00 , &h00 , &h00 , &hC0 , &hE0 , &h60 , &hE0 , &hC0 , &h00 , &h00 , &h00 , &h00 , &h00,
data &h00 , &h00 , &h00 , &h00 , &h04 , &h04 , &h04 , &h07 , &h07 , &h04 , &h04 , &h00 , &h00 , &h03 , &h07 , &h04,
data &h07 , &h03 , &h00 , &h04 , &h07 , &h07 , &h00 , &h07 , &h07 , &h00 , &h00 , &h03 , &h07 , &h07 , &h04 , &h04,
data &h02 , &h00 , &h00 , &h00 , &h00 , &h01 , &h03 , &h07 , &h04 , &h04 , &h06 , &h03 , &h01 , &h00 , &h00 , &h02,
data &h06 , &h05 , &h07 , &h03 , &h00 , &h00 , &h03 , &h07 , &h04 , &h06 , &h02 , &h00 , &h00 , &h00 , &h00 , &h00

'16x32 pixel font data for numbers, space and dp
'
numbertable:                                                '0-9
'hex=&h30, ascii="0"
data &hF0 , &hF8 , &hFC , &hFC , &h3C , &h1C , &h1C , &h1C , &h1C , &h3C , &hFC , &hFC , &hF8 , &hF0 , &h00 , &h00,
data &hFF , &hFF , &hFF , &hFF , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &hFF , &hFF , &hFF , &hFF , &h00 , &h00,
data &hFF , &hFF , &hFF , &hFF , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &hFF , &hFF , &hFF , &hFF , &h00 , &h00,
data &h0F , &h1F , &h3F , &h3F , &h3C , &h38 , &h38 , &h38 , &h38 , &h3C , &h3F , &h3F , &h1F , &h0F , &h00 , &h00
'
'hex=&h31, ascii="1"
'
data &h00 , &h00 , &h80 , &hC0 , &hE0 , &hF0 , &hF8 , &hFC , &hFC , &hFC , &h00 , &h00 , &h00 , &h00 , &h00 , &h00,
data &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &hFF , &hFF , &hFF , &hFF , &h00 , &h00 , &h00 , &h00 , &h00 , &h00,
data &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &hFF , &hFF , &hFF , &hFF , &h00 , &h00 , &h00 , &h00 , &h00 , &h00,
data &h00 , &h00 , &h30 , &h30 , &h30 , &h30 , &h3F , &h3F , &h3F , &h3F , &h30 , &h30 , &h30 , &h30 , &h00 , &h00
'
'hex=&h32, ascii="2"
'
data &hF0 , &hF8 , &hFC , &h3C , &h1C , &h0C , &h0C , &h0C , &h0C , &h1C , &hFC , &hFC , &hF8 , &hF0 , &h00 , &h00,
data &h01 , &h01 , &h01 , &h00 , &h00 , &h00 , &h80 , &hC0 , &hE0 , &hF0 , &hFF , &h7F , &h3F , &h1F , &h00 , &h00,
data &hE0 , &hF0 , &hF8 , &hFC , &h3E , &h1F , &h0F , &h07 , &h03 , &h01 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00,
data &h3F , &h3F , &h3F , &h3F , &h30 , &h30 , &h30 , &h30 , &h30 , &h30 , &h3E , &h3E , &h3E , &h3E , &h00 , &h00
'
'hex=&h33, ascii="3"
'
data &hF0 , &hF8 , &hFC , &h3C , &h1C , &h1C , &h1C , &h1C , &h1C , &h1C , &hFC , &hFC , &hF8 , &hF0 , &h00 , &h00,
data &h00 , &h00 , &h00 , &h00 , &hC0 , &hC0 , &hC0 , &hC0 , &hC0 , &hE0 , &hFF , &hFF , &h3F , &h1F , &h00 , &h00,
data &h00 , &h00 , &h00 , &h00 , &h03 , &h03 , &h03 , &h03 , &h03 , &h07 , &hFF , &hFF , &hFC , &hF8 , &h00 , &h00,
data &h0F , &h1F , &h3F , &h3C , &h38 , &h38 , &h38 , &h38 , &h38 , &h38 , &h3F , &h3F , &h1F , &h0F , &h00 , &h00
'
'hex=&h34, ascii="4"
'
data &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &hC0 , &hF0 , &hFC , &hFC , &hFC , &hFC , &h00 , &h00 , &h00 , &h00,
data &h00 , &h00 , &hC0 , &hF0 , &hFC , &h7F , &h1F , &h07 , &hFF , &hFF , &hFF , &hFF , &h00 , &h00 , &h00 , &h00,
data &hFC , &hFF , &hFF , &hF7 , &hF1 , &hF0 , &hF0 , &hF0 , &hFF , &hFF , &hFF , &hFF , &hF0 , &hF0 , &h00 , &h00,
data &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h30 , &h30 , &h3F , &h3F , &h3F , &h3F , &h30 , &h30 , &h00 , &h00
'
'hex=&h35, ascii="5"
'
data &hFC , &hFC , &hFC , &hFC , &h1C , &h1C , &h1C , &h1C , &h1C , &h1C , &h1C , &h1C , &h1C , &h1C , &h00 , &h00,
data &hFF , &hFF , &hFF , &hFF , &hE0 , &hE0 , &hE0 , &hE0 , &hE0 , &hE0 , &hE0 , &hE0 , &hC0 , &h80 , &h00 , &h00,
data &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h01 , &hFF , &hFF , &hFF , &hFF , &h00 , &h00,
data &h0E , &h1E , &h3E , &h3C , &h38 , &h30 , &h30 , &h30 , &h30 , &h38 , &h3F , &h3F , &h1F , &h0F , &h00 , &h00
'
'hex=&h36, ascii="6"
'
data &hF0 , &hF8 , &hFC , &h3C , &h1C , &h1C , &h1C , &h1C , &h1C , &h1C , &h3C , &h78 , &h70 , &h60 , &h00 , &h00,
data &hFF , &hFF , &hFF , &hE0 , &hE0 , &hE0 , &hE0 , &hE0 , &hE0 , &hE0 , &hE0 , &hE0 , &hC0 , &h80 , &h00 , &h00,
data &hFF , &hFF , &hFF , &hFF , &h01 , &h00 , &h00 , &h00 , &h00 , &h01 , &hFF , &hFF , &hFF , &hFF , &h00 , &h00,
data &h0F , &h1F , &h3F , &h3F , &h38 , &h30 , &h30 , &h30 , &h30 , &h38 , &h3F , &h3F , &h1F , &h0F , &h00 , &h00
'
'hex=&h37, ascii="7"
'
data &h00 , &h7C , &h7C , &h7C , &h1C , &h1C , &h1C , &h1C , &h1C , &h1C , &hFC , &hFC , &hFC , &h00 , &h00 , &h00,
data &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &hC0 , &hFC , &hFF , &h3F , &h03 , &h00 , &h00 , &h00,
data &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &hF8 , &hFF , &hFF , &h07 , &h00 , &h00 , &h00 , &h00 , &h00,
data &h00 , &h00 , &h00 , &h00 , &h30 , &h30 , &h3F , &h3F , &h3F , &h30 , &h30 , &h00 , &h00 , &h00 , &h00 , &h00
'
'hex=&h38, ascii="8"
'
data &hF0 , &hF8 , &hFC , &hFC , &h3C , &h1C , &h1C , &h1C , &h1C , &h3C , &hFC , &hFC , &hF8 , &hF0 , &h00 , &h00,
data &h1F , &h3F , &hFF , &hFF , &hF0 , &hE0 , &hE0 , &hE0 , &hE0 , &hF0 , &hFF , &hFF , &h3F , &h1F , &h00 , &h00,
data &hFE , &hFF , &hFF , &hFF , &h03 , &h01 , &h01 , &h01 , &h01 , &h03 , &hFF , &hFF , &hFF , &hFE , &h00 , &h00,
data &h0F , &h1F , &h3F , &h3F , &h38 , &h30 , &h30 , &h30 , &h30 , &h38 , &h3F , &h3F , &h1F , &h0F , &h00 , &h00
'
'hex=&h39, ascii="9"
'
data &hF0 , &hF8 , &hFC , &hFC , &h3C , &h1C , &h1C , &h1C , &h1C , &h3C , &hFC , &hFC , &hF8 , &hF0 , &h00 , &h00,
data &h3F , &h7F , &hFF , &hFF , &hE0 , &hC0 , &hC0 , &hC0 , &hC0 , &hE0 , &hFF , &hFF , &hFF , &hFF , &h00 , &h00,
data &h00 , &h00 , &h00 , &h01 , &h01 , &h01 , &h01 , &h01 , &h01 , &h01 , &hFF , &hFF , &hFF , &hFF , &h00 , &h00,
data &h0F , &h1F , &h3F , &h38 , &h38 , &h30 , &h30 , &h30 , &h38 , &h3C , &h3F , &h1F , &h0F , &h07 , &h00 , &h00
'
spacechar:
'hex=&h20, ascii=" "
'
data &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 ,
data &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 ,
data &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 ,
data &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00
'
dp:
'hex=&h2E, ascii="."
'
data &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00,
data &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00,
data &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00,
data &h00 , &h00 , &h00 , &h00 , &h00 , &h3C , &h3C , &h3C , &h3C , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00
'


!.org &H0800                                                'NOTE!!! Bascom compiler locates this table at 2x 0800h!!! i.e. 1000h
sinewave:                                                   '256 step sinewave table for PWM version using PB1 output

data &h40 , &h41 , &h43 , &h44 , &h46 , &h47 , &h49 , &h4A , &h4C , &h4E , &h4F , &h51 , &h52 , &h54 , &h55 , &h57
data &h58 , &h59 , &h5B , &h5C , &h5E , &h5F , &h60 , &h62 , &h63 , &h64 , &h66 , &h67 , &h68 , &h69 , &h6A , &h6C
data &h6D , &h6E , &h6F , &h70 , &h71 , &h72 , &h73 , &h74 , &h75 , &h76 , &h76 , &h77 , &h78 , &h79 , &h79 , &h7A
data &h7B , &h7B , &h7C , &h7C , &h7D , &h7D , &h7E , &h7E , &h7E , &h7F , &h7F , &h7F , &h7F , &h7F , &h7F , &h7F
data &h7F , &h7F , &h7F , &h7F , &h7F , &h7F , &h7F , &h7F , &h7E , &h7E , &h7E , &h7D , &h7D , &h7C , &h7C , &h7B
data &h7B , &h7A , &h79 , &h79 , &h78 , &h77 , &h76 , &h76 , &h75 , &h74 , &h73 , &h72 , &h71 , &h70 , &h6F , &h6E
data &h6D , &h6C , &h6A , &h69 , &h68 , &h67 , &h66 , &h64 , &h63 , &h62 , &h60 , &h5F , &h5E , &h5C , &h5B , &h59
data &h58 , &h57 , &h55 , &h54 , &h52 , &h51 , &h4F , &h4E , &h4C , &h4A , &h49 , &h47 , &h46 , &h44 , &h43 , &h41
data &h40 , &h3E , &h3C , &h3B , &h39 , &h38 , &h36 , &h35 , &h33 , &h31 , &h30 , &h2E , &h2D , &h2B , &h2A , &h28
data &h27 , &h26 , &h24 , &h23 , &h21 , &h20 , &h1F , &h1D , &h1C , &h1B , &h19 , &h18 , &h17 , &h16 , &h15 , &h13
data &h12 , &h11 , &h10 , &h0F , &h0E , &h0D , &h0C , &h0B , &h0A , &h09 , &h09 , &h08 , &h07 , &h06 , &h06 , &h05
data &h04 , &h04 , &h03 , &h03 , &h02 , &h02 , &h01 , &h01 , &h01 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00
data &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h01 , &h01 , &h01 , &h02 , &h02 , &h03 , &h03 , &h04
data &h04 , &h05 , &h06 , &h06 , &h07 , &h08 , &h09 , &h09 , &h0A , &h0B , &h0C , &h0D , &h0E , &h0F , &h10 , &h11
data &h12 , &h13 , &h15 , &h16 , &h17 , &h18 , &h19 , &h1B , &h1C , &h1D , &h1F , &h20 , &h21 , &h23 , &h24 , &h26
data &h27 , &h28 , &h2A , &h2B , &h2D , &h2E , &h30 , &h31 , &h33 , &h35 , &h36 , &h38 , &h39 , &h3B , &h3C , &h3E



