// <editor-fold defaultstate="collapsed" desc="Microchip preamble">
/**
  Generated main.c file from MPLAB Code Configurator

  @Company
    Microchip Technology Inc.

  @File Name
    main.c

  @Summary
    This is the generated main.c using PIC24 / dsPIC33 / PIC32MM MCUs.

  @Description
    This source file provides main entry point for system initialization and application code development.
    Generation Information :
        Product Revision  :  PIC24 / dsPIC33 / PIC32MM MCUs - 1.145.0
        Device            :  PIC24FJ256GA704
    The generated drivers are tested against the following:
        Compiler          :  XC16 v1.36b
        MPLAB 	          :  MPLAB X v5.25
*/

/*
    (c) 2019 Microchip Technology Inc. and its subsidiaries. You may use this
    software and any derivatives exclusively with Microchip products.

    THIS SOFTWARE IS SUPPLIED BY MICROCHIP "AS IS". NO WARRANTIES, WHETHER
    EXPRESS, IMPLIED OR STATUTORY, APPLY TO THIS SOFTWARE, INCLUDING ANY IMPLIED
    WARRANTIES OF NON-INFRINGEMENT, MERCHANTABILITY, AND FITNESS FOR A
    PARTICULAR PURPOSE, OR ITS INTERACTION WITH MICROCHIP PRODUCTS, COMBINATION
    WITH ANY OTHER PRODUCTS, OR USE IN ANY APPLICATION.

    IN NO EVENT WILL MICROCHIP BE LIABLE FOR ANY INDIRECT, SPECIAL, PUNITIVE,
    INCIDENTAL OR CONSEQUENTIAL LOSS, DAMAGE, COST OR EXPENSE OF ANY KIND
    WHATSOEVER RELATED TO THE SOFTWARE, HOWEVER CAUSED, EVEN IF MICROCHIP HAS
    BEEN ADVISED OF THE POSSIBILITY OR THE DAMAGES ARE FORESEEABLE. TO THE
    FULLEST EXTENT ALLOWED BY LAW, MICROCHIP'S TOTAL LIABILITY ON ALL CLAIMS IN
    ANY WAY RELATED TO THIS SOFTWARE WILL NOT EXCEED THE AMOUNT OF FEES, IF ANY,
    THAT YOU HAVE PAID DIRECTLY TO MICROCHIP FOR THIS SOFTWARE.

    MICROCHIP PROVIDES THIS SOFTWARE CONDITIONALLY UPON YOUR ACCEPTANCE OF THESE
    TERMS.
*/
// </editor-fold>

// <editor-fold defaultstate="collapsed" desc="My description">
/*
 * This Test_1_Sensors application was the first test of the latest board with the LoRa module.
 * the significant points about it are-
 * It uses the timer function with the modularised 1sec Flag, 10 sec etc and the second incrementer.
 * see the int1.c for the details. The variables are declared external in the tmr1.h file
 * You just have to include the tmr1.h file to use it. You can see it used in the AM2303 reader
 * See this in tmr1.c-
 * int dSec ;
 * bool flag_for_dSec = false;
 * bool flag_for_1S = false;   
 * bool flag_for_10S = false;
 * bool flag_for_30S = false;
 * bool flag_for_60S = false;
 * unsigned long sec_incrementer = 10000;  // A variable that gets incremented every second continuously.
 * unsigned long sec_snapshot;   
 * And also see the TMR1_CallBack routine where all the testing is done.
 *  
 * This application has-
 * the LED display multiplexer implemented and working. tmr2
 * the RTCC being displayed on the LED display 
 * the RTCC clock out on one of the LED pins. You can scope the 32KHz.
 * the RTCC working.
 * the RS232 running at 460800 baud
 * the AM2302 reader with the test for the existence of the module.
 * the OLED working.
 * the analogue voltage reader functioning.
 * the LM35 temperature sensor working. 
 * the BME280 sensor working.
 * the beeper working. Note that the particular beeper put on the board has its own internal oscillator. 
 * ____________________________________
 * NOTE: for Test_3_Buttons
 * In this version I added the button interrupt and reading stuff. 
 * Buttons 2 and 3 increment and decrement a value called set_point. 
 * It's got a repeating loop in there too.
 * This still has all the LED display and temperature stuff still in.
 * I wanted to muck around with the automatic tire pressure pump control and gauge
 * So I thought I'd leave this now and create a new application without all the unnecessary stuff
 * 
 * _____________________________________
 * NOTE: for Tire Control
 * 
 * This next comments are cut and pasted from the Fence RGB controller.
 * ESP8266 Server - RGB controller.
 * I used these to get the OC1 PWM going for the motor controller.
 * 
// in ESP8266_server_8_RGB_Controller   ---

// Search for this    red_LED_RPOR     =   OC1_funct;   
// the OC initialisation for RPOR is in user.c  under void InitApp(void)
    
//  #define red_LED_RPOR    (RPOR10bits.RP20R)      // RC4 = RP20  
//  #define green_LED_RPOR  (RPOR10bits.RP21R)      // RC5 = RP21
//  #define blue_LED_RPOR   (RPOR11bits.RP22R)      // RC6 = RP22
//  #define OC1_funct   13      // Will be different for the different PIC versions.
//  #define OC2_funct   14      // (Page 174 of PIC24FJ128GA204 manual)
//  #define OC3_funct   15      // (Page 138 of PIC24FJ256GA704 manual)

// Also look in RGB_driver.c    under    void T3_OC_init(void)
// for where the OC itself is initialised.

 * 
 * Note how the button interrupt handling is handled in main.c

    BTN2_SetInterruptHandler(&BTN2_CallBack);   // assign the handler to the IOC. See the pin_manager.h
    BTN3_SetInterruptHandler(&BTN3_CallBack);   // assign the handler to the IOC. See the pin_manager.h
    
 * the actual handler has to be written and pointed to. 

 * Had success getting it going. This version just increments a value and 
 * varies the PWM according to a set point value.
 * 
 * Developed this on the latest MOTE board, but it's not really appropriate
 * for this application
 * wanted to move to the previous board that   
    
 * 
*/
// </editor-fold>

// <editor-fold defaultstate="collapsed" desc="Global defines">
#define CPU_CLOCK          (32000000)       // 16MHz Primary 
#define TCY_CLOCK          (CPU_CLOCK/2)
#define FCY                (CPU_CLOCK/2)
// </editor-fold>

// <editor-fold defaultstate="collapsed" desc="Includes">
#include "mcc_generated_files/system.h"
#include "mcc_generated_files/pin_manager.h"
#include "mcc_generated_files/adc1.h"
#include "SSD1306_OLED.h"
#include "mcc_generated_files/spi1.h"
#include "mcc_generated_files/tmr1.h"
#include "mcc_generated_files/tmr2.h"
#include "mcc_generated_files/oc1.h"
#include "RS232.h"

#include <string.h>
#include <stdio.h>
#include <stdbool.h>
#include <stdint.h>
#include <stdlib.h>
#include <libpic30.h>       // for __delay_ms()   NOTE: you have to define FCY for this to work AND you have to define it BEFORE including this 
#include <math.h>
#include <float.h>
#include <time.h>            /* for asctime, tm */


// </editor-fold>

// <editor-fold defaultstate="collapsed" desc="Global variables">
uint16_t    looper;
char txt[100];
extern LCD_text_structure LCD_text_buffer;  // This is in the OLED library 

typedef enum
{
	HONEYWELL	=	0,      // The e-bay clones
	MPX5700                 // From RS Components
} TXD_TYPE;

typedef enum
{
	INIT	=	0,      // The initial status
	SERVICE,            // Service Mode
	WAITING,            // Waiting for a button to be pressed
	SET_PRESS,          // Action. There's a difference between the set point and the measured point
	ADJUSTING,          // Action. There's a difference between the set point and the measured point
	FINISHED,           // The set point and the measured point are now the same   
    OVER_PRESS          // An over pressure situation has occurred 
} MASTER_STATE;

const char *master_state_text[] = { "Initialise", "Service", "Inactive and waiting", "Setting Pressure", "Adjusting",  "Finished", "Over pressure"  };

typedef enum
{
	SM_DP_VC	=	1,      // Service Mode, Display Pressure, Valve closed. (Default, un-energised)
	SM_DP_VO,               // Service Mode, Display Pressure, Valve OPEN. (energised)
	SM_DT,                  // Service Mode, Display Temperature
	SM_DP_VC_PO             // Service Mode, Display Pressure, Valve closed, pump on.
} SERVICE_STATE;

const char *service_state_text[] = { "Dummy","Display pressure, Valve closed", "Display pressure, Valve open", "Display Temp, Valve closed", "Display Temp, Valve closed, Pump on" };


    MASTER_STATE	master_state;
    SERVICE_STATE   service_state;
    TXD_TYPE        txd_type;
    
double  psi;
double  temperature;

int16_t  gap            =   0;      // The difference between actual pressure measured and the set point
// Note: This was a bug here. As the gap has to go negative, it can't be an unsigned integer. Was having so much trouble with this
// until I realized this and changed to an integer. The others can be unsigned as they only go positive.
// In fact, as a precaution, as in speed, it's better they are unsigned integers to prevent -ve going.
uint16_t speed          =   0;      // The representation of the value in the OC1  OC1_PrimaryValueSet()
uint16_t set_point      =   32;     // The pressure set point value
uint16_t last_set_point =   32;     // A pressure set point value for use for re-run prevention

#define over_pressure   35
    

// </editor-fold>

// <editor-fold defaultstate="collapsed" desc="Analog Variables">
int conversion_count;
double          real_volts;
const double    conversion_count_hi_trig_pt = 13.3;
const double    conversion_count_lo_trig_pt = 11.7;    // These values are legacy from the VSR

// For these values, see page 300 of the PIC24FJ256GA705 FAMILY manual

#define AIR_SENSE   0b01010     // RC0 = AN10   // On 2nd Arduino bd, this is the SPI line
#define LM35_ch     0b01100     // RC2 = AN12   // On 2nd Arduino bd, this is the Bluetooth STATE line
#define POT         0b00110     // RB14 = AN6   // On 2nd Arduino bd, this is actually the POT
// The air valve is on RB5  On 2nd Arduino bd, this is the XBEE Din line

#define AVdd    0b11110     // This is the measurement of the AVdd. If the AVdd is connected to 3.3V, it should be 1023.

// </editor-fold>

// <editor-fold defaultstate="collapsed" desc="LED variables">
char    clock_disp_buffer [20];     // This will be the build buffer for whatever clock required.

bool    temp_humid_flipflop = false;    // flips between a temperature and a Humidity display
bool    mux_time_up =   false;          // This is the MUX interrupt flag interrupt 

extern  const uint8_t LED_7_SEG_table[];

extern  uint8_t led_char_disp_buffer [6];   // This will be the build buffer for whatever clock required. Just contains characters.
extern  bool    dp0;                // There are 6 displays. Each one could have dp set.
extern  bool    dp1;                // Interrupt checks these bits and sets (OR x80H) the dp bit to a ONE if this is non-zero
extern  bool    dp2;
extern  bool    dp3;
extern  bool    dp4;
extern  bool    dp5;
extern  bool    colon1;                // this is bits 6 and 7 of the led_disp_element_mux'er
extern  bool    colon2;                // Interrupt checks these bits and if set, outputs a 6 or a 7 to the mux port.
// </editor-fold>

// <editor-fold defaultstate="collapsed" desc="Button variables">

bool    btn1_pressed    =   false;
bool    btn2_pressed    =   false;
bool    btn3_pressed    =   false;
bool    btn4_pressed    =   false;
bool    runonce1        =   false;
bool    runonce2        =   false;
bool    runonce3        =   false;
bool    runonce4        =   false;

clock_t     t_pressed;      // Time of pressing
clock_t     t_of_action;    // Time of action

// </editor-fold>

void    VT100_show_control_vals(void);  // This declaration is just here because of the forward reference

// <editor-fold defaultstate="collapsed" desc="Get the A2D Conversion">
int get_conversion_count_MCC (ch)
{
//    printf ("Starting sampling\r\n");
    ADC1_Initialize();
    ADC1_ChannelSelect(ch);
    ADC1_Start();
    //Provide Delay
    __delay_ms(50);
    ADC1_Stop();
//    printf ("Stopped sampling\r\n");
//    printf ("Starting Conversion\r\n");
    while(!ADC1_IsConversionComplete())
    {
        ADC1_Tasks();   
    }
//    printf ("Conversion complete\r\n");
    return ADC1_ConversionResultGet();
}
// </editor-fold>

// <editor-fold defaultstate="collapsed" desc="Set speed">
void	set_speed (uint16_t s)
{
	speed = s;
	OC1_PrimaryValueSet (s);
}
// </editor-fold>

// <editor-fold defaultstate="collapsed" desc="Test motor incrementally">
void    test_motor(void)
{
        set_speed(0);
        printf ("0\r\n");
        __delay_ms(1000);
        set_speed(256);
        printf ("256\r\n");
        __delay_ms(1000);
        set_speed(512);
        printf ("512\r\n");
        __delay_ms(1000);
        set_speed(768);
        printf ("768\r\n");
        __delay_ms(1000);
        set_speed(1022);
        printf ("1022\r\n");
        __delay_ms(1000);
        set_speed(1023);
        printf ("1023\r\n");
        __delay_ms(1000);
        set_speed(1024);
        printf ("1024\r\n\r\n");
        __delay_ms(1000);
        set_speed(0);
        printf ("0\r\n");
        __delay_ms(1000);
    }
// </editor-fold>
    
// <editor-fold defaultstate="collapsed" desc="Test the Analog read">
void    test_analog(void)
    {
        conversion_count = get_conversion_count_MCC (POT);
        printf ("POT conversion value is %d\r\n",conversion_count);
        real_volts = conversion_count ;
        real_volts = real_volts * 3.3 / 1024 ;
        printf ("POT = %2.2fV\r\n\n",real_volts);
        
        conversion_count = get_conversion_count_MCC (LM35_ch);
        printf ("LM35 conversion value is %d\r\n",conversion_count);
        real_volts = conversion_count ;
/*
        printf ("NOTE: The voltage divider down scales the 5V to 3.3V\r\n");
        printf ("So although the A2D reference volts is 3.3V, use 5V as the value.\r\n");
        printf ("Use 1.7K and 3.3K as the divider resistors.\r\n");
        printf ("There are 5V / 1024 = 0.00488V per count.\r\n");
        printf ("So multiply the count by 5/1024\r\n");
*/
        real_volts = real_volts * 5 / 10.24 ;        // LM35 OUTPUT is 0 mV + 10.0 mV/C
        printf ("Temperature = %2.2f Deg\r\n\n",real_volts);
        
        conversion_count = get_conversion_count_MCC (AIR_SENSE);
        printf ("Air sensor conversion count is %d\r\n",conversion_count);
        real_volts = conversion_count ;
        real_volts = real_volts * 5 / 1024 ;
        printf ("Air sensor voltage = %2.2fV\r\n\n",real_volts);

        psi = real_volts - 0.5;     // There is an offset of 0.5 of a volt in the txd
        if (psi < 0) psi = 0;       // It is possible for it to be < zero
        psi = psi / 0.04;           // There are 4V / 100 = 40 mV per PSI
        printf ("Corresponding PSI = %2.1f\r\n\n",psi);

        sprintf (LCD_text_buffer.line1,"%2.1f psi",psi);
        OLED_clear();
        OLED_text(0,10,LCD_text_buffer.line1,3); 
        OLED_write(); 

}


// </editor-fold>

// <editor-fold defaultstate="collapsed" desc="Get the LM35 temperature and return as double">
double get_temp  (void)
{
        conversion_count = get_conversion_count_MCC (LM35_ch);
//        printf ("LM35 conversion value is %d\r\n",conversion_count);
        real_volts = conversion_count ;
/*
        printf ("NOTE: The voltage divider down scales the 5V to 3.3V\r\n");
        printf ("So although the A2D reference volts is 3.3V, use 5V as the value.\r\n");
        printf ("Use 1.7K and 3.3K as the divider resistors.\r\n");
        printf ("There are 5V / 1024 = 0.00488V per count.\r\n");
        printf ("So multiply the count by 5/1024\r\n");
*/
        real_volts = real_volts * 5 / 10.24 ;        // LM35 OUTPUT is 0 mV + 10.0 mV/C
//        printf ("Temperature = %2.2f Deg\r\n\n",real_volts);
        temperature = real_volts;
        return temperature;
    
}
// </editor-fold>

// <editor-fold defaultstate="collapsed" desc="Get the pressure and return as double">
double get_psi_con1  (TXD_TYPE txd)     // This is the function for the transducer connected to connector # 1
{
        conversion_count = get_conversion_count_MCC (AIR_SENSE);    // AIR_SENCE is the PIC analog channel that the connector is joined to
//    printf ("Air sensor conversion count is %d\r\n",conversion_count);
        real_volts = conversion_count ;         
        real_volts = real_volts * 5 / 1024 ;    // 5V divided by the resolution of the AtoD 
//    printf ("Air sensor voltage = %2.2fV\r\n\n",real_volts);

    if (txd  == HONEYWELL)              // If the transducer is a Honeywell type
    {
        psi = real_volts - 0.5;     // There is an offset of 0.5 of a volt in the txd
        if (psi < 0) psi = 0;       // It is possible for it to be < zero
        psi = psi / 0.04;           // There are 4V / 100 = 40 mV per PSI
    }

    else if (txd  == MPX5700)       // If the transducer is a Honeywell type
    {
        real_volts = (real_volts - 0.2);        // You'll see that at 0 pressure, the voltage is about 0.2V
        if (real_volts < 0) real_volts = 0;     // Don't want it to go below 0V
        psi = real_volts * 22.554;              // See the oneNote description of this calculation. Search for "Freescale MPX5700"
    }

    return psi;

}
// </editor-fold>

// <editor-fold defaultstate="collapsed" desc="Put the PSI onto the OLED">
void OLED_display_psi (double psi_x)
{
    sprintf (LCD_text_buffer.line1,"%2.1f psi",psi_x);
    OLED_clear();
    OLED_text(0,10,LCD_text_buffer.line1,3); 
    OLED_write(); 
}
// </editor-fold>

// <editor-fold defaultstate="collapsed" desc="Display the PSI and Set point">
void OLED_display_psi_and_setpoint (void)
{
    OLED_clear();
    sprintf (LCD_text_buffer.line1,"%2.1f psi",get_psi_con1(txd_type));
    OLED_text(0,0,LCD_text_buffer.line1,3); 
    sprintf (LCD_text_buffer.line1,"%d psi",set_point);
    OLED_text(0,40,LCD_text_buffer.line1,3); 
    OLED_write(); 
}
// </editor-fold>

// <editor-fold defaultstate="collapsed" desc="Button 1 callback">
void BTN1_CallBack(void)
{
    t_pressed   =   clock();
//    if ((t_of_action - t_pressed) > 200)   btn1_pressed =  true;
    btn1_pressed =  true;
}
// </editor-fold>

// <editor-fold defaultstate="collapsed" desc="Button 2 callback">
void BTN2_CallBack(void)
{
    t_pressed   =   clock();
//    if ((t_of_action - t_pressed) > 200)   btn2_pressed =  true;
    btn2_pressed =  true;
}
// </editor-fold>

// <editor-fold defaultstate="collapsed" desc="Button 3 callback">
void BTN3_CallBack(void)
{
    t_pressed   =   clock();
//    if ((t_of_action - t_pressed) > 200)   btn3_pressed =  true;
    btn3_pressed =  true;
}
// </editor-fold>

// <editor-fold defaultstate="collapsed" desc="Button 4 callback">
void BTN4_CallBack(void)
{
    t_pressed   =   clock();
//    if ((t_of_action - t_pressed) > 200)   btn4_pressed =  true;
    btn4_pressed =  true;
}
// </editor-fold>

// <editor-fold defaultstate="collapsed" desc="Button 3 action">

	void	check_button3_action(void)
	{                                               // NOTE! that (!BTN3_GetValue) means that the button is PRESSED.
		if 	( (!runonce1) && (!BTN3_GetValue()))    // If haven't done this before and a button is pressed
			{	
                set_point--;
                LED1_Toggle() ; 		// This is just a test confirmation that button pressed
				runonce3 = true;		// Don't want to perpetually run this
                btn3_pressed = false;
//                t_of_action = clock();
//                printf("t_of_action = %ld\r\n", t_of_action);
//                __delay_ms(100);        // The de-bounce value 
			}

// If has already run once and button has been released, allow to run again by resetting 'runonce'
// If it hasn't been released, then 'runonce' will stay set, therefore preventing the commands above 
// from executing again.
        sec_snapshot = sec_incrementer; // Snapshot of current. Make sure interrupts on for this to work

        while (!BTN3_GetValue())
        {
            if ((sec_incrementer-1) >= sec_snapshot) // something has happened. Is it a timeout? If yes, bomb-out.
            {
                set_point--;
                __delay_ms(150);
                //sec_snapshot = sec_incrementer; // Snapshot of current. Make sure interrupts on for this to work

                OLED_display_psi_and_setpoint();
                VT100_show_control_vals();

//                sprintf (LCD_text_buffer.line1,"%d psi",set_point);
//                OLED_clear();
//                OLED_text(0,10,LCD_text_buffer.line1,4); 
//                OLED_write(); 
            }
        }
       
		if ((runonce1) && (BTN3_GetValue()))  runonce3 = false;     // If button is released 
	}

// </editor-fold>

// <editor-fold defaultstate="collapsed" desc="Button 4 action">

	void	check_button4_action(void)
	{                                               // NOTE! that (!BTN2_GetValue) means that the button is PRESSED.
		if 	( (!runonce4) && (!BTN4_GetValue()))    // If haven't done this before and a button is pressed
			{	
                set_point++;
                LED1_Toggle() ; 		// This is just a test confirmation that button pressed
				runonce4 = true;		// Don't want to perpetually run this
                btn4_pressed = false;
//                t_of_action = clock();
//                printf("t_of_action = %ld\r\n", t_of_action);
//                __delay_ms(100);        // The de-bounce value 
			}

        sec_snapshot = sec_incrementer; // Snapshot of current. Make sure interrupts on for this to work

        while (!BTN4_GetValue())        // Stay in this loop while the button is pressed
        {
            if ((sec_incrementer-1) >= sec_snapshot) // Has been pressed for longer than one second?
            {
                set_point++;        // Up the set point
                __delay_ms(150);    // this is the repeat delay
                //sec_snapshot = sec_incrementer; // Can do it this way, but limited to 1S. Using the __delay allows for faster

                OLED_display_psi_and_setpoint();
                VT100_show_control_vals();

                //                sprintf (LCD_text_buffer.line1,"%d psi",set_point);
//                OLED_clear();
//                OLED_text(0,10,LCD_text_buffer.line1,4); 
//                OLED_write(); 
            }
        }
       
// If has already run once and button has been released, allow to run again by resetting 'runonce'
// If it hasn't been released, then 'runonce' will stay set, therefore preventing the commands above 
// from executing again.

		if ((runonce4) && (BTN4_GetValue()))  runonce4 = false;     // If button is released 
	}

// </editor-fold>
    
// <editor-fold defaultstate="collapsed" desc="Button 3 Debounce">

	void	button3_debounce(void)
	{                                               // NOTE! that (!BTN3_GetValue) means that the button is PRESSED.
		if 	( (!runonce1) && (!BTN3_GetValue()))    // If haven't done this before and a button is pressed
			{	
				runonce3 = true;		// Don't want to perpetually run this
                __delay_ms(200);        // The de-bounce value 
                btn3_pressed = false;
			}

		while (  (runonce1) && (!BTN3_GetValue())   );  // Do nothing while button is held down
        runonce3 = false;     // If button is released  //  Button must have been released
	}

// </editor-fold>

// <editor-fold defaultstate="collapsed" desc="Beeper">
    // The first parameter is the number of beeps.
    // The second parameter is the on time of each individual beeps in mS
    // The third parameter is the inter-beep delay in mS
    
void beep (uint8_t count, uint16_t timeon, uint16_t timeoff)
{
    uint8_t looper_c;
    
    for (looper_c = 0; looper_c < count; looper_c++)
    {
        BUZER_SetHigh();
        __delay_ms (timeon);
        BUZER_SetLow();
        if (looper_c != (count - 1)) __delay_ms (timeoff);
        
    }
}
// </editor-fold>

// <editor-fold defaultstate="collapsed" desc="VT100 setup routines">
    void VT100_show_temp(double temp)
    {
        VT100_goto_xy(8, 41);           // Go to the Temp location inside the box
//        printf ("     ");               // Blank out the temperature field
        VT100_goto_xy(8, 41);          // Go to the Temp location inside the box
        VT100_show_float(temp);         // Display the temperature at the current cursor location
        VT100_set_special_graphics();   // Set special graphics set for the degree symbol
        printf("f");                    // f in special char set is the degrees symbol
        VT100_set_std_char_set();       // Restore the standard character set
    }

    void VT100_show_control_vals()
    {
        VT100_goto_xy(14, 12);          // Go to the Temp location inside the box
        VT100_goto_xy(14, 12);          // Go to the Temp location inside the box
        printf ("%d psi  ",set_point);
        VT100_goto_xy(16, 12);          // Go to the set_point location inside the box
        VT100_goto_xy(16, 12);          // Go to the set_point location inside the box
        printf ("%d psi  ",gap);
        VT100_goto_xy(18, 12);          // Go to the gap location inside the box
        VT100_goto_xy(18, 12);          // Go to the gap location inside the box
        printf ("%d    ",speed);
    }
    
    void VT100_show_p1(double p1)
    {
        VT100_goto_xy(18, 39);   // Go to the P1 location inside the box
        VT100_show_float(p1);         // Display the temperature at the current cursor location
//        put_str_to_console(" PSI");       // f in special char set is the degrees symbol
        printf(" PSI");       // f in special char set is the degrees symbol
    }

    void VT100_show_p2(double p2)
    {
        VT100_goto_xy(18, 53);          // Go to the P2 location inside the box
        VT100_show_float(p2);           // Display the temperature at the current cursor location
//        put_str_to_console(" PSI");       // f in special char set is the degrees symbol
        printf(" PSI");       // f in special char set is the degrees symbol
    }

    void VT100_show_master_state()
    {
//            VT100_goto_xy(26,1);
//            VT100_clr_line_to_right(); // Get rid of previous trailing characters
//            VT100_goto_xy(27,1);
//            VT100_clr_line_to_right(); // Get rid of previous trailing characters
        put_string_to_U1_port (CH_back_white);
        put_string_to_U1_port (CH_fore_red);
        put_string_to_U1_port (CH_BOLD);              // Check if this makes the white whiter

        VT100_text_at_xy ( 8, 57, "                      ");
        VT100_text_at_xy ( 9, 57, "                      ");
        VT100_text_at_xy (10, 57, "                      ");
        VT100_text_at_xy (11, 57, "                      ");
//        printf ("Master state = %d, corresponding to- %s",master_state,master_state_text[master_state]);
        VT100_goto_xy(9,58);
        printf ("%s",master_state_text[master_state]);

        VT100_attrib_off();
    }
    
    void    VT100_over_press()
    {
        VT100_goto_xy(9,57);
        put_string_to_U1_port (CH_BOLD);              // Check if this makes the white whiter
        put_string_to_U1_port (CH_back_red);
        put_string_to_U1_port (CH_fore_white);
        VT100_back_blink ();
        printf ("  Releasing pressure  ");
        VT100_attrib_off();
    }
    
    void VT100_show_service_state()
    {
        VT100_goto_xy(25,1);
        VT100_clr_line_to_right(); // Get rid of previous trailing characters
        printf ("Service state = %d, corresponding to- %s",service_state,service_state_text[service_state]);
    }
    
    void VT100_close_valve(void)
    {
        VT100_set_special_graphics();    // Special graphic character set
        VT100_text_at_xy(18, 29,"qvq"); 
        VT100_set_std_char_set();       // Standard character set
        VT100_text_at_xy(19, 28,"   "); 
        VT100_text_at_xy(22, 28,"Closed"); 
    }

    void VT100_open_valve(void)
    {
        VT100_text_at_xy(18, 29,"   "); 
        VT100_set_special_graphics();    // Special graphic character set
        VT100_text_at_xy(18, 30,"x"); 
        VT100_set_std_char_set();       // Standard character set. The "V" is a V
        VT100_text_at_xy(19, 30,"V"); 
        VT100_text_at_xy(22, 28,"Open  "); 
    }

    void    VT100_draw_system(void)
    {
        VT100_set_std_char_set();   // Select the standard character set
        VT100_set_80_collumn();     // 80 columns  NOTE: Doing this clears the screen
        VT100_clear();              // Clear the screen
        VT100_home();               // Cursor home
        VT100_cursor_off();         // Cursor off
        VT100_set_special_graphics();
        printf("lqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqwqqqqqqqqqqqqqqqqqqqqqqqqk");
        printf("x                                                     x                        x");
        printf("x lqqqqqqqqqqqqqqqqqqqk                               x                        x");
        printf("x x         lqqqk     x                               x                        x");
        printf("x x         x   tqqqqqnqqqqqqk                        x                        x");
        printf("x x  lqqqqqqj   mqqk  x      x       lqqqqqqqqqk      mqqqqqqqqqqqqqqqqqqqqqqqqu");
        printf("x x  x             x  x      x       x         x                               x");
        printf("x x  x  _       _  x  x      x       x         x                               x");
        printf("x x  mqq=qqqqqqq=qqj  x      x       x         x                               x");
        printf("x mqqqqqqqqqqqqqqqqqqqj      x       mqqqqqqqqqj                               x");
        printf("x                            x                                                 x");
        printf("x lqqqqqqqqqqqqqqqqqqqk      x                                                 x");
        printf("x x                   x      tqqqqqqqqqqqqwqqqqqqqqqqqqwqqqqqqqqqqqqqk         x");
        printf("x x                   x      x            x            x             x         x");
        printf("x x                   x      x            x            x             x         x");
        printf("x x                   x  lqqqnqqqk   lqqqqvqqqqk  lqqqqvqqqqk        aa        x");
        printf("x x                   x  x   x   x   x         x  x         x      aa  aa      x");
        printf("x x                   x  x  qvq  x   x         x  x         x    aa      aa    x");
        printf("x x                   x  x       x   x         x  x         x   aa   aa   aa   x");
        printf("x mqqqqqqqqqqqqqqqqqqqj  mqqqqqqqj   mqqqqqqqqqj  mqqqqqqqqqj    aa      aa    x");
        printf("x                                                                  aa  aa      x");
        printf("x                                                                    aa        x");
        printf("x                                                                              x");
        printf("mqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqj");

        VT100_set_std_char_set();
        VT100_text_at_xy (21, 4, "Control parameters");
        VT100_text_at_xy (21, 28, "Valve");
        VT100_text_at_xy (21, 40, "Meter 1");
        VT100_text_at_xy (21, 53, "Meter 2");
        VT100_text_at_xy (11, 41, "System");
        VT100_text_at_xy (12, 38, "Temperature");
        VT100_underline();
        VT100_text_at_xy (14, 4,  "Set Pt");
        VT100_text_at_xy (16, 4,  "Gap");
        VT100_text_at_xy (18, 4,  "Speed");
        VT100_attrib_off();
        put_string_to_U1_port (CH_back_red);
        put_string_to_U1_port (CH_fore_white);
        put_string_to_U1_port (CH_BOLD);              // Check if this makes the white whiter
        VT100_text_at_xy ( 2, 57, "                      ");
        VT100_text_at_xy ( 3, 57, "  Automatic Tire Pump ");
        VT100_text_at_xy ( 4, 57, "      Controller      ");
        VT100_text_at_xy ( 5, 57, "                      ");
        VT100_attrib_off();
        put_string_to_U1_port (CH_back_blue);
        put_string_to_U1_port (CH_fore_white);
        VT100_text_at_xy ( 7, 8, "           ");
        VT100_text_at_xy ( 8, 8, "   Pump    ");
        put_string_to_U1_port (CH_back_default);
        put_string_to_U1_port (CH_fore_default);
        VT100_show_control_vals();
    }
    
// </editor-fold>

// <editor-fold defaultstate="collapsed" desc="Initialise the Application itself">
void    init_app(void)          // This is where the test for the button 4 hold down takes place.
{
    BTN3_SetInterruptHandler(&BTN3_CallBack);   // assign the handler to the IOC. See the pin_manager.h
    BTN4_SetInterruptHandler(&BTN4_CallBack);   // assign the handler to the IOC. See the pin_manager.h

    OC1_Start();            // So the next command will work          
    set_speed(0);           // Do this first to stop the pump

    __delay_ms(500);        // 100 wasn't enough
    InitializeDisplay();    // Initialise the OLED display
    __delay_ms(500);        // 300 wasn't enough
    OLED_clear();           // Do this here otherwise end up with garbage

//    txd_type  = HONEYWELL;   // This is the e-bay clones that screw in to 1/4 BSB fitting
    txd_type    = MPX5700;   // This is the PCB mounted transducer connected by tube

    VT100_draw_system();

    state_int_start();      // This is a define in the appropriate TMR .h file.
    OC_tmr_start();         // This is a define in the appropriate TMR .h file.
    if (!BTN4_GetValue())     // NOTE: can't use "if (btn3_pressed)" as it's interrupt driven. Have to directly read it.
    {
        beep (4,50,50);
        while (!BTN4_GetValue());   // Just wait until it's released before continuing
        btn4_pressed    = false;    // And, just in case, clear any interrupt triggered flags
        master_state = SERVICE;     // User requested service mode from the start
        service_state = SM_DP_VC;   // confirm the service mode is in the first setting
        VT100_show_service_state();
    }
    else
    {
        beep (2,60,60);
        master_state = WAITING;     // Service mode bypassed. Go straight to 'normal' wait state.
        VT100_show_master_state();
        VT100_show_control_vals();

        VT100_goto_xy(10,60);
        put_string_to_U1_port (CH_back_white);
        put_string_to_U1_port (CH_fore_blue);
        put_string_to_U1_port (CH_BOLD);              // Check if this makes the white whiter
        printf ("Press up or down");
        VT100_goto_xy(11,60);
        printf ("to start a pump.");
        VT100_attrib_off();

    }
}
// </editor-fold>

// <editor-fold defaultstate="collapsed" desc="Put the Temp onto the OLED">
void OLED_display_temp (double temp_x)
{
    OLED_clear();
    sprintf (LCD_text_buffer.line1,"%2.1f",temp_x);
    OLED_text(0,10,LCD_text_buffer.line1,3); 
    sprintf (LCD_text_buffer.line1,"deg");
    OLED_text(70,10,LCD_text_buffer.line1,2); 
    OLED_write(); 
}
// </editor-fold>

// <editor-fold defaultstate="collapsed" desc="Service mode handler">
void    service(void)
{
    switch (service_state)
    {
        case SM_DP_VC:
            if (flag_for_1S)      // Decimal seconds 5 = 0.5 of second
                {
                    flag_for_1S = false;
                    LED2_Toggle() ;
                    VT100_show_p1(get_psi_con1(txd_type));
                    OLED_display_psi(psi);
                    VT100_show_control_vals();
                    VT100_show_temp(get_temp());
                }
            if (btn3_pressed)
                {
                    button3_debounce();
                    service_state   = SM_DP_VO;
                    AIR_VALVE_SetHigh() ;       // Energise the valve to release air
                    VT100_open_valve() ;        // Show open
                    VT100_show_service_state();
                }
            break;
                
        case SM_DP_VO:
            if (flag_for_1S)      // Decimal seconds 5 = 0.5 of second
                {
                    flag_for_1S = false;
                    LED2_Toggle() ;
                    VT100_show_p1(get_psi_con1(txd_type));
                    OLED_display_psi(psi);
                    VT100_show_control_vals();
                    VT100_show_temp(get_temp());
                }
            if (btn3_pressed)
                {
                    button3_debounce();
                    service_state   = SM_DP_VC_PO;  // Skip SM_DT - it's redundant. Always show temp
                    AIR_VALVE_SetLow() ;    // Ensure the valve is de-energised
                    VT100_close_valve() ;   // Show closed
                    VT100_goto_xy(18, 40);   // Go to the P1 location inside the box
                    printf ("        ");      // Blank out the pressure field
                    VT100_show_service_state();
                }
            break;
                
        case SM_DT:                 // Service Mode, Display temperature
            if (flag_for_1S)      // Decimal seconds 5 = 0.5 of second
                {
                    flag_for_1S = false;
                    LED2_Toggle() ;
                    VT100_show_temp(get_temp());
                    OLED_display_temp(temperature);
                }
            if (btn3_pressed)
                {
                    button3_debounce();
                    service_state   = SM_DP_VC_PO;
                    AIR_VALVE_SetLow() ;        // Ensure the valve is de-energised
                    VT100_close_valve() ;       // Show closed
                    set_speed(400);   // Drive pump at 400
                    VT100_show_service_state();
                }
        break;

        case SM_DP_VC_PO:               // Service Mode, Display pressure, Valve closed, pump on
            if (flag_for_1S)      // Decimal seconds 5 = 0.5 of second
                {
                    flag_for_1S = false;
                    LED2_Toggle() ;
                    VT100_show_p1(get_psi_con1(txd_type));
                    OLED_display_psi(psi);
                    VT100_show_control_vals();
                    VT100_show_temp(get_temp());
                }
            if (btn3_pressed)
                {
                    button3_debounce();
                    service_state   = SM_DP_VC;
                    set_speed(0);     // Pump off
                    AIR_VALVE_SetLow() ;        // Ensure the valve is de-energised
                    VT100_close_valve() ;       // Show closed
                    VT100_show_service_state();
                }
        break;
        }
    if (btn4_pressed) asm("reset");
}
// </editor-fold>

// <editor-fold defaultstate="collapsed" desc="Waiting">
void waiting(void)
{
    if (flag_for_1S)      // Decimal seconds 5 = 0.5 of second
    {
        flag_for_1S = false;
        LED2_Toggle() ;
        VT100_show_p1(get_psi_con1(txd_type));
        gap = set_point - (uint16_t) ceil(get_psi_con1(txd_type)) + 1 ;
        OLED_display_psi_and_setpoint();
        VT100_show_temp(get_temp());
        VT100_show_control_vals();
    }

    if (btn3_pressed || btn4_pressed)
    {
        btn3_pressed    = false;
        btn4_pressed    = false;
        master_state    = SET_PRESS;
        sec_snapshot_2  = sec_incrementer; // Take time snapshot at time button first pressed

        VT100_show_master_state();
    }
}
// </editor-fold>

// <editor-fold defaultstate="collapsed" desc="Set Pressure">
void set_press (void)
{
    while ((sec_incrementer-5) <= sec_snapshot_2)
    {
        if (btn3_pressed)
        {
            btn3_pressed = false;
            sec_snapshot_2    = sec_incrementer; // If a button pressed, reset the timer to extend further
            check_button3_action();
        }
        if (btn4_pressed)
        {
            btn4_pressed = false;
            sec_snapshot_2    = sec_incrementer; // If a button pressed, reset the timer to extend further
            check_button4_action();
        }
        if (last_set_point != set_point)
        {
            last_set_point = set_point;
            gap = set_point - (uint16_t) ceil(get_psi_con1(txd_type));
            OLED_display_psi_and_setpoint();
            VT100_show_control_vals();
            VT100_show_temp(get_temp());
        }
    }
// The waiting for adjustment has finished. Go to the adjusting state
    master_state    = ADJUSTING;
    speed = 0;
    VT100_show_master_state();
}
// </editor-fold>

// <editor-fold defaultstate="collapsed" desc="Adjusting">
void adjusting (void)
{
    if (flag_for_1S)      // Decimal seconds 5 = 0.5 of second
    {
        flag_for_1S = false;
        LED2_Toggle() ;
        VT100_show_p1(get_psi_con1(txd_type));
        OLED_display_psi_and_setpoint();
        VT100_show_temp(get_temp());
        VT100_show_control_vals();
    }
    if (flag_for_dSec)      // decimal Second - Tenth of a second.
    {
        flag_for_dSec = false;
        gap = set_point - (int16_t) ceil(get_psi_con1(txd_type)) ;

        if ((gap > 8) & (speed != 1023))    // Do an incremental speed up by 64 a time until ceiling of 1023
        {
            if (speed > 1023) speed = 1023; // Make sure doesn't go over the max
            if (speed < 1023) speed = speed + 64;   // if less, pump up
            if (speed < 400) speed = 400;       // This is for the first time round, when it would be 0
        }

        if ((gap <= 8) & (gap > 0))       // Do an incremental slow down by 64 a time until floor of 400
        {
//            speed = (gap * 64) - 1;
            if (speed > 464) speed = speed - 64;         // Don't randomly subtract. Could go -ve = flat out
            if (speed < 400) speed = 400;       // Don't reduce less that minimum speed
        }

        if (gap == 0)
        {
            __delay_ms(1000);   // Let the pump run a bit longer to accommodate expansion
            VT100_show_p1(get_psi_con1(txd_type));
            VT100_show_control_vals();
            __delay_ms(1000);   // Let the pump run a bit longer to accommodate expansion
            VT100_show_p1(get_psi_con1(txd_type));
            VT100_show_control_vals();
            speed = 0;
            master_state    = FINISHED;
            VT100_show_master_state();
            btn3_pressed    = false;
            btn4_pressed    = false;
        }
        
        if (gap < 0)
        {
            VT100_open_valve();     // Show open on the terminal graphic
            AIR_VALVE_SetHigh() ;   // Ensure the valve is energised = OPEN to atmosphere
            if  ( abs(gap) > 6 ) __delay_ms(200) 
            else if      (( abs(gap) <= 6 ) & ( abs(gap) > 4 )) __delay_ms(140)
            else if ( abs(gap) <= 4 )  __delay_ms(80);
            AIR_VALVE_SetLow() ;   // Ensure the valve is de-energised = CLOSED to atmosphere
            VT100_close_valve();     // Show open on the terminal graphic
            __delay_ms(250);    
        }
        set_speed (speed);
    }

}
// </editor-fold>

// <editor-fold defaultstate="collapsed" desc="Finished">
void finished(void)
    {
        beep (1,1000,0);
        __delay_ms(2000);   // This just allows the "Finished" statue to show for a sec
        master_state    = WAITING;  // Now change and exit
        VT100_show_master_state();
    }
// </editor-fold>

// <editor-fold defaultstate="collapsed" desc="Over Pressure">
void over_press()
    {
        set_speed(0);           // Set speed to 0
        VT100_open_valve();     // Show open on the terminal graphic
        AIR_VALVE_SetHigh() ;   // Ensure the valve is energised = OPEN to atmosphere
        VT100_over_press();     // Display the warning on the terminal
        while (get_psi_con1(txd_type) > 5);     // Leave it open until pressure reduced to safe level 
        // __delay_ms(20000) ;     // Leave it open for 20 seconds
        AIR_VALVE_SetLow() ;    // Don't want to over heat the valve, so de-energise now = CLOSED to atmosphere
        VT100_close_valve();     // Show open on the terminal graphic
        if (master_state == SERVICE)    // Was bug here. MUST use == otherwise doesn't compare correctly
        {                       // If it was in service mode when pressure violation occurred, go back to service mode
            service_state   = SM_DP_VC;     // goto first service mode  
            VT100_show_service_state();     // Because it must have been in service mode
        }
        else    // was in anything but service mode
        {
            master_state = WAITING; // goto first mode  
            VT100_show_master_state();  // Because it must have been in one of the main modes modes
        }    
    }
// </editor-fold>

// <editor-fold defaultstate="collapsed" desc="Main">
int main(void)
{
    // initialize the device
    SYSTEM_Initialize();

    master_state = INIT;    // Tell the master state loop that state is initialise
      
    while (1)
    {
    if (get_psi_con1(txd_type) > over_pressure)      // continuously check for over pressure 
        {
            master_state = OVER_PRESS;  // and set corresponding state
        }   

    switch (master_state)
        {
            case INIT:
                init_app();     // init_app to distinguish it from system init
                break;
            case SERVICE:
                service();
                break;
            case WAITING:
                waiting();
                break;
            case SET_PRESS:
                set_press();
                break;
            case ADJUSTING:
                adjusting();
                break;
            case FINISHED:
                finished();
                break;
            case OVER_PRESS:
                over_press();
                break;
        }
    }    // Main Switch loop
// </editor-fold>
    return 1;
}
/**
 End of File
*/

