/* RF_Power_Meter_sketch.ino -- a sketch for the 
   RF Power Meter using an Arduino Nano MCU
   with an AD8318-based logarithmic detector module,
   a 24-bit LTC2400 ADC (with LT1019 voltage reference)
   and a 16x2 LCD module with an I2C serial driver backpack.
   
   Written by Jim Rowe for Silicon Chip
   Version 1.0, revised 17/04/2020 at 8:50 am
      
   Note: the following SPI interface connections
   are assumed by the Arduino SPI library:
   LTC2400  <---------> Arduino Nano
   pin 5 (CSbar) <-  from   DigIO pin 10 (SS-bar)
   pin 6 (SDO)   ->  to     DigIO pin 12 (MISO)
   pin 7 (SCK)   <-  from   DigIO pin 13 (SCK)
   Note also that the I2C interface library used to drive
   the LCD display uses the A4 pin of the Arduino for the
   SDA line, and the A5 pin for the SCK line.
   Also note that if the LCD serial 'backpack' uses a PCF8574T
   device, its address will be 0x27 -- not the address of 0x3F
   shown below, which is the address for a PCF8574AT device.
   FURTHER NOTE: if you are using a Chinese LCD module with a
   serial (I2C) backpack, and are also using an Arduino IDE
   V 1.8.12 (or later), make sure you use this library to
   drive the serial LCD: Arduino-LiquidCrystal-I2C-library
   written by joaopedrosgs and fdebrabander.
   It can be downloaded as a zip file from:
   https://github.com/fdebrabander/Arduino-LiquidCrystal-I2C-library
   Other libraries gave trouble, but this one works correctly.
*/
#include <SPI.h>
#include <Wire.h> 
#include <LiquidCrystal_I2C.h>

// first set the LCD address to 0x3F, with 2 x 16 char lines
LiquidCrystal_I2C lcd(0x3F,16,2); // (or 0x27 for a PCF8574T)

long inSample;  // the incoming 32-bit words from the ADC
float AvgMeas;  // the average of 10 measurements
float DetSlope; // slope of AD8318 Vout between -55dBm & 0dBm
float DetSlopP; // slope of AD8318 Vout between 0dBm & +5dBm
float DetSlopN; // slope of AD8318 Vout between -65dBm & -55dBm
unsigned int AttVal = 0; // external input attenuation, in dB
String LCDLine1;  // for 1st line of LCD display
String LCDLine2;  // for 2nd line of LCD display
String AttStrng = "00";  // string to show current input
                         // attenuation level (initially zero)
byte S1 = 0;    // S1 = 0 if not pressed, 1 if pressed
byte S2 = 0;    // S2 = 0 if not pressed, 1 if pressed
byte S3 = 0;    // S3 = 0 if not pressed, 1 if pressed

const float Von65dBm = 2.0451; // AD8318's Vout for -65dBm RFin
    // (change for calibration if it has been measured)
const float Von55dBm = 1.8812; // AD8318's Vout for -55dBm RFin
    // (change for calibration if it has been measured)
const float Vo0dBm = 0.5634;   // AD8318's Vout for 0dBm RFin
    // (change for calibration if it has been measured)
const float Vo5dBm = 0.5186;   // AD8318's Vout for +5dBm RFin
    // (change for calibration if it has been measured)
const float Vref = 2.5000; // Vref = ADC reference voltage
   // (change for calibration if LT1019 output measured)
const byte SSPin = 10;    // digIO pin 10 for CSbar/SS-bar
const byte MISOPin = 12;  // digIO pin 12 for MISO
const byte SCKPin = 13;   // digIO pin 13 for SCK
const int SetASw = 7;  // label D7 SetASw (S1: set atten SW)
const int IncrSw = 3;  // label D3 IncrSw (S3: increment)
const int DecrSw = 5;  // label D5 DecrSw (S2: decrement)

void setup()
{
  // start serial comms to PC at 115,200 baud, 8N1
  Serial.begin(115200);
  // then send start message to PC (via IDE's Serial Monitor)
  Serial.println("Silicon Chip Digital RF Power Meter");
  Serial.println(" ");
   
  lcd.begin();                   // initialise LCD
  LCDLine1 = "Silicon Chip  ";   // set line 1 and
  LCDLine2 = "RF Power Meter ";  // line 2 initial messages
  DisplayonLCD();                // & send both lines to LCD
  
  pinMode(SSPin, OUTPUT);       // make D10 an output (CS-bar)
  digitalWrite(SSPin, HIGH);    // and initialise to HIGH
  pinMode(MISOPin, INPUT);      // make D12 an input (MISO)
  pinMode(SCKPin, OUTPUT);      // make D13 an output (SCK)
  digitalWrite(SCKPin,LOW);     // and set low (for Ext SCK)
  
  pinMode(SetASw, INPUT_PULLUP);  // set D7 an input w/pullup
  pinMode(IncrSw, INPUT_PULLUP);  // & D5 an input w/pullup
  pinMode(DecrSw, INPUT_PULLUP);  // & D3 an input w/pullup
  
  // then work out the slope of the AD8318 output voltage
  // by subtracting the 0dBm value from the -55dBm value
  // and dividing by 55, giving a figure in V/dBm
  DetSlope = (Von55dBm - Vo0dBm)/55.00;
  // also work out the slope at each end
  DetSlopP = (Vo0dBm - Vo5dBm)/5.00; // at bottom end
  DetSlopN = (Von65dBm - Von55dBm)/10.00; // and at top end
  
  // next initialise SPI port for max speed, data MSB first,
  // data mode 0 -- with SCK idle low (CPOL = 0),
  //                MISO read on rising edge (CPHI = 0)
  SPI.beginTransaction(SPISettings(2000000, MSBFIRST, SPI_MODE0));
  
  delay(2000);  // delay 2s to allow init screen to be seen
} // end of setup function

 // ==========================================================
 // main loop() function begins here
 // ==========================================================
void loop()
{
  CheckS1();    // before proceeding, check S1
  if(S1 == 1)   // if it has been pressed,
  {
    AttenSel(); // go to AttenSel
  }
  // but if not pressed, take the average of 10 new samples
  // (returned as a 24-bit FP number)
  AvgMeas = Get10MeasAvg(); 
  // then process & display result on LCD, also
  // send to IDE's Serial Monitor for logging
  CalcnShow();
}      // end of main loop

// ===========================================================
// CalcnShow() function: to process the average of 10 readings
// from the LTC2400, calculate the equivalent dBm & mV figs,
// display them on the LCD display & also send back to the PC.
// ===========================================================
long CalcnShow()
{
  float ADCInV;  // the measured ADC input volts
  float dBmRdg;  // the corresponding dBm reading
  float PowExp;  // the power exponent
  float PowLvl;  // the corresponding power reading in mW
  float milliV;  // and the corresponding mV reading
  float microV;   // or microvolts reading
  
  // first calculate the ADC input voltage using the equation
  // ADCInV = Vref * sample / 16777215 (FFFFFF = 16777215)
  ADCInV = Vref * (AvgMeas / 16777215.0);
  
  // next work out the dBm reading
  if (ADCInV > Von55dBm)  // if ADCInV is > Von55dBm
    {
      dBmRdg = -(55 + (ADCInV - Von55dBm)/DetSlopN);
      // (because slope below -55dBm = DetSlopN V/dBm
    }
  else if (ADCInV < Vo0dBm)   // or if it's < Vo0dBm
    {
      dBmRdg = (Vo0dBm - ADCInV)/DetSlopP;
      // (because slope above 0dBm is = DetSlopP V/dBm
    }
  else  // must be in linear range between 0dBm & -55dBm
    {
      dBmRdg = -(ADCInV - Vo0dBm)/DetSlope;
      // because slope in this region is DetSlope V/dBm
    }
  dBmRdg = dBmRdg + AttVal; // make correction for AttVal
  LCDLine1 = "RF Pwr= " + String(dBmRdg,1) + "dBm  ";

  PowExp = dBmRdg/10.0; // now find the power exponent
  // and work out the power level in uW
  if (PowExp < 0)   // now if PowExp is negative,
  {
    PowLvl = 1/(pow(10.0,-PowExp)); 
  }
  else
  {
    PowLvl = pow(10.0,PowExp);
  }
 PowLvl = PowLvl * 1000.0;  // convert into mW 
  // then find the voltage level in millivolts or
  // microvolts, and set up line 2
  milliV = sqrt(PowLvl * 50);   // 50 ohms impedance)
  if (milliV < 1)
  {
    microV = 1000.0 * milliV;   // convert to uV
    LCDLine2 = "=" + String(microV, 1) + "uV At=" + AttStrng + "dB ";
  }
  else
  {
    LCDLine2 = "=" + String(milliV,1) + "mV At=" + AttStrng + "dB ";
  }
  // now display lines on LCD
  DisplayonLCD();
  // & also send them back to the PC (IDE Serial Monitor)
  Serial.println(LCDLine1 + " " + LCDLine2); 
} // and return
// ========================================================
// Get10MeasAvg() function -- take 10 measurements and get
// their average value
// ========================================================
float Get10MeasAvg()
  {
  int x;          // counter
  long NxtSample; // next sample from ADC
  
  AvgMeas = 0;   // first zero the average measurement      
  for(x = 0; x < 10; x++)
    {
       NxtSample = TakeMeas();  // take a sample measurement
       AvgMeas = AvgMeas + NxtSample; // then add to AvgMeas
    }
  AvgMeas = AvgMeas / 10.000;  // work out 10-sample average
  return(AvgMeas);    // & return with the average    
  }
// ========================================================
// TakeMeas() function: to take a 32-bit measurement from
// the LTC2400, convert to 24 bits and return
// ========================================================
long TakeMeas()
{
   inSample = SpiRead(); // get a 32-bit sample
   inSample = 0;         // then discard (previous sample)
   inSample = SpiRead(); // now get a new 32-bit sample
   inSample >>= 4;       // shift out the 4 'sub-LSB' bits    
   inSample &= 0xFFFFFF; // now mask off the status bits,
                         // leaving just the 24 data bits
   return(inSample);     // & return with the 24-bit sample
}
// ========================================================
// SpiRead() function: to fetch a four-byte sample from the
// LTC2400 via the SPI interface
// ========================================================
long SpiRead()
{
  long result = 0; // result declared a 32-bit long
  long b = 0;      // likewise for b
  digitalWrite(SCKPin,LOW); // ensure LTC2400 set for Ext SCK
  digitalWrite(SSPin,LOW);  // pull SS low to start transfer

  // wait until a sample is available (when MISO pin goes low)
  while( digitalRead(MISOPin) )
    ;

  b = SPI.transfer(0xff); // get B3, most significant byte
  result = b<<8;   // shift left by 8 bits, pass to result 
  b = SPI.transfer(0xff); // get B2, next significant byte
  result |= b;            // bitwise OR with result
  result = result<<8;     // shift left by 8 bits again
  b = SPI.transfer(0xff); // get B1, next significant byte
  result |= b;            // bitwise OR with result
  result = result<<8;     // shift left by 8 bits once more
  b = SPI.transfer(0xff); // get B0, least significant byte
  result |= b;            // bitwise OR with result, which
                       // should give result = [B3,B2,B1,B0]
  digitalWrite(SSPin, HIGH); // pull SS high again to end txfr
  return(result);          // and return with 32-bit result
}  // end of SpiRead function
 // =========================================================
 // DisplayonLCD() -- function to display 2 lines of 16 chars
 // on LCD (LCDLine1, LCDLine2)
 // =========================================================
 void DisplayonLCD()
 {
  lcd.setCursor(0,0);   // set LCD cursor to start of line 1
  lcd.print(LCDLine1);  // display line 1 message
  lcd.setCursor(0,1);   // then set cursor to start of line 2
  lcd.print(LCDLine2);  // then display line 2 message
 }  // and return
 // =========================================================
 // AttenSel() -- function to respond if S1 pressed, to  
 // allow attenuation figure and calibration levels to be set
 // =========================================================
 void AttenSel()
 {
   LCDLine1 = "Set Extern Input"; // define line 1 message
   LCDLine2 = "Attenuation: " + AttStrng + "dB" ; // then line 2
   DisplayonLCD();  // & send both lines to LCD
   SelectAtten();   // then go select atten value
  }  // then return
 // =========================================================
 // CheckS1() -- function to check if switch S1 is pressed
 // =========================================================
 void CheckS1()
 {
   if(digitalRead(SetASw) == LOW) // if S1 is pressed
    {
      S1 = 1;   // set S1 to 1
    }
   else
   {
      S1 = 0;   // otherwise set it to 0
   }
 }    // then leave
 // =========================================================
 // CheckS2() -- function to check if switch S2 is pressed
 // =========================================================
 void CheckS2()
 {
   if(digitalRead(DecrSw) == LOW) // if S2 pressed
    {
      S2 = 1;   // set S2 to 1
    }
   else
   {
      S2 = 0;   // otherwise set it to 0
   }
 }    // then leave
 // =========================================================
 // CheckS3() -- function to check if switch S3 is pressed
 // =========================================================
 void CheckS3()
 {
   if(digitalRead(IncrSw) == LOW) // if S3 pressed
    {
      S3 = 1;   // set S3 to 1
    }
   else
   {
      S3 = 0;   // otherwise set it to 0
   }
 }    // then leave
 // =========================================================
 // SelectAtten() -- function to allow selecting external
 // attenuation value using the S2 and S3 buttons
 // =========================================================
void SelectAtten()
  {
  do
    {
      CheckS2();     // first see if S2 (decrement) is pressed 
      if(S2 == 1)    // if so,
      {
        if(AttVal > 0)   // and if AttVal is greater than zero
        {
          AttVal = AttVal - 1; // decrement AttVal
          UpdAtStrng();   // update AtStrng
          // define line 1 string
          LCDLine1 = "Set Extern Input";
          // then line 2
          LCDLine2 = "Attenuation: " + AttStrng + "dB " ;
          DisplayonLCD();  // & send both lines to LCD
          S2 = 0;  // then clear S2 
        }
      }
      CheckS3();    // now see if S3 (increment) is pressed
      if(S3 == 1)   // if so,
      {
        AttVal = AttVal + 1;   // increment AttVal
        UpdAtStrng();   // update AtStrng
        // define line 1 string
        LCDLine1 = "Set Extern Input";
        // then line 2
        LCDLine2 = "Attenuation: " + AttStrng + "dB " ;
        DisplayonLCD();  // & send both lines to LCD
        S3 = 0;     // and clear S3
      }
      delay(300);   // delay to allow all switch lines to rise
      CheckS1();    // now see if S1 has been pressed again 
    } while (S1 == 0);   // keep looping if not
    S1 = 0;  // but if it was pressed, clear S1
  }   // and return
// ==========================================================
// UpdAtStrng() -- a function to update AttStrng when AttVal
// has been changed
// ==========================================================
void UpdAtStrng()
{
   if (AttVal == 0)
   {
      AttStrng = "00";
   }
   else
   {
      AttStrng = String(AttVal); // update AttStrng
   }
}
// ==========================================================
//    end of code
