// Silicon Chip Ol'Timer II
// 16F1455 based word clock
// USB interface based on Signal 11's M-Stack and CDC-ACM demo
// Virtual serial port provides menu/setup
// WS2812B string attached to RC5
// DS3231 on I2C bus for timekeeping
// ISR at end of main must call usb_service() and handle any other IRQ
// The defines below set the main functions

//try to exclude USB:
#define USE_USB_INTERFACE

//use USB debug info, see also extern debug variable in rtc.c/rtc.h
//bitmaps.h and util.c
#define USE_USB_DEBUG

//manual touchpad controls
#define USE_TOUCHPADS

//ADC channels for touch pads
//RevA PCB
//#define DOWN_BUTTON 6
//#define UP_BUTTON 7
//#define SET_BUTTON 3

//RevB PCB
#define DOWN_BUTTON 4
#define UP_BUTTON 5
#define SET_BUTTON 6

//bitmap for touchpads
#define DOWN_BUTTON_BV 1
#define UP_BUTTON_BV 2
#define SET_BUTTON_BV 4

#define TOUCH_ON_THRESHOLD 110
#define TOUCH_OFF_THRESHOLD 100

#ifdef USE_TOUCHPADS
//for onboard setup menu
typedef enum{
    MENU_OFF=0,
    SET_HOUR,
    SET_MINUTES,
    SET_SECONDS,
    SET_HOURS_RED,
    SET_HOURS_GREEN,
    SET_HOURS_BLUE,
    SET_MINUTES_RED,
    SET_MINUTES_GREEN,
    SET_MINUTES_BLUE,
    SET_OTHER_RED,
    SET_OTHER_GREEN,
    SET_OTHER_BLUE,
    SET_BRIGHTNESS_LOW,
    SET_BRIGHTNESS_HIGH,
    SET_SAVE            
}settings_t;
settings_t settings=MENU_OFF;       //state machine for onboard setup
//hold off (in 1/183s T1 ticks) for touch
//used to create time based debounce
char touchHoldOff=0;
#define TOUCH_DEBOUNCE_DELAY 30
//called when buttons pressed
void set_button(void);
void up_button(void);
void down_button(void);
//some failsafe colours for setting mode in case internal colours corrupted
#define SETTINGS_RED 31
#define SETTINGS_GREEN 31
#define SETTINGS_BLUE 31
#define BUTTON_STEP 15
#define BUTTON_MIN (BUTTON_STEP-1)
#define BUTTON_MAX (256-BUTTON_STEP)
#endif

#define _XTAL_FREQ (48000000L)
#include "usb.h"
#include <xc.h>
#include <string.h>
#include "usb_config.h"
#include "usb_ch9.h"
#include "usb_cdc.h"
#include "hardware.h"
#include "io.h"
#include "timer1.h"
#include "util.h"
#include "rtc.h"
#include "bitmaps.h"
#include "flash.h"

#ifdef MULTI_CLASS_DEVICE
static uint8_t cdc_interfaces[] = { 0 };
#endif

//if space available, increase this:
//see also util.h
#define BUFSIZE 160
char txbuf[BUFSIZE]="";
char rxbuf[BUFSIZE]="";
int txptr=0;
int rxptr=0;
//char txptr=0;
//char rxptr=0;
char sBuf[7]="";               //current settings command string
char sPtr=0;
volatile char sCtr=0;           //seconds counter for backup timing

//some simple colour configs, order is GRB
#define WHITE {255,255,255}
#define RED {0,255,0}
#define GREEN {255,0,0}
#define BLUE {0,0,255}
#define YELLOW {255,255,0}

volatile unsigned char RGBdata[64][3];       //laid out for easy linear access as needed, colour order is GRB

unsigned char hColour[3]=YELLOW;
unsigned char mColour[3]=GREEN;
unsigned char oColour[3]=BLUE;
unsigned char brightness=10;
unsigned int rawLDR;
unsigned char brightnessL=10;   //1st brightness value for 2 point interpolation
unsigned char brightnessH=50;   //2nd brightness value for 2 point interpolation
unsigned char bChanged=1;       //flag to update display immediately, which we do at startup

char hours=0;
char minutes=0;
char seconds=0;

//buffer for writing/reading flash
extern unsigned char flashshadow[FLASHPAGESIZE];

typedef enum{
    IDLE,
    MENU,
    TIME,
    MINUTES,
    HOURS,
    OTHER
}setmode_t;

//line coding struct/functions
//#define USE_LINE_CODING

#ifdef USE_LINE_CODING
static struct cdc_line_coding line_coding =
{
	115200,
	CDC_CHAR_FORMAT_1_STOP_BIT,
	CDC_PARITY_NONE,
	8,
};
#endif

#ifdef USE_USB_DEBUG
char debug=0;
#endif

int main(void)
{
    unsigned char i;
    unsigned char bIndex;
    unsigned char hTemp;    //adjusted hour
#ifdef USE_TOUCHPADS
    unsigned char tTemp;    //for touch value
    unsigned char touchMap=0; //bitmap of buttons
    unsigned char touchLast=0;//bitmap of buttons    
#endif
    setmode_t sMode=IDLE;   //setting state machine
#ifdef USE_USB_INTERFACE
    hardware_init();
#endif
#ifndef USE_USB_INTERFACE
    //this is set by the USB startup code if used
    OSCCONbits.IRCF = 0b1111; //use 16MHz HFINTOSC
#endif
    ioinit();
    timer1init();
#ifdef MULTI_CLASS_DEVICE
	cdc_set_interface_list(cdc_interfaces, sizeof(cdc_interfaces));
#endif
#ifdef USE_USB_INTERFACE
	usb_init();
#endif
    loadSettings();
    getRTC();
    while (1) {
#ifdef USE_USB_INTERFACE
        for(i=0;i<rxptr;i++){       //do something to received data
            switch(rxbuf[i]){
                case '+':
                case '=':
                    if(brightnessL<=250){brightnessL=brightnessL+1;bChanged=1;}
                    break;
                case '-':
                case '_':
                    if(brightnessL>2){brightnessL=brightnessL-1;bChanged=1;}
                    break;
                case '.':
                case '>':
                    if(brightnessH<=250){brightnessH=brightnessH+1;bChanged=1;}
                    break;
                case ',':
                case '<':
                    if(brightnessH>2){brightnessH=brightnessH-1;bChanged=1;}
                    break;
#ifdef USE_USB_DEBUG
                case 'q': if(debug){debug=0;}else{debug=1;}break;   //toggle debug output
#endif
                case 27: sMode=MENU;showMenu();sPtr=0;
#ifdef USE_USB_DEBUG
                    debug=0;
#endif
                    break;       //reset to menu on ESC
                case '0' ... '9':
                    if((sPtr<6)&&(sMode>MENU)){
                        //somehow these three lines are optimal for the compiler
                        sBuf[sPtr]=rxbuf[i];
                        sPtr++;
                        tx(rxbuf[i]);
                    }else if(sMode==MENU){
                        switch(rxbuf[i]){
                            case '1': sMode=TIME;txA("Time (HHMMSS):\r\n");break;
                            case '2': sMode=MINUTES;txA("Minutes");txA(" hex (RRGGBB):\r\n");break; //split string to reuse duplicated parts, about 8 words per call
                            case '3': sMode=HOURS;txA("Hours");txA(" hex (RRGGBB):\r\n");break;
                            case '4': sMode=OTHER;txA("AM/PM");txA(" hex (RRGGBB):\r\n");break;
                            case '5': saveSettings();txA("Save done\r\n");break;
                        }
                    }
                    break;
                case 'A' ... 'F':
                case 'a' ... 'f':
                    if((sPtr<6)&&(sMode>TIME)){
                        sBuf[sPtr]=rxbuf[i];
                        sPtr++;
                        tx(rxbuf[i]&223);       //toUpper
                    }
                    break;
                case 8:             //backspace
                case 127:           //delete
                    if((sPtr>0)&&(sMode>MENU)){
                        sPtr--;
                        //sBuf[sPtr]=0;     //optional as we only check if string is full (saves 6 words)
                        tx(8);tx(32);tx(8);                        
                    }
                    break;
                case 13:            //enter
                    if(sPtr==6){        //enough characters entered
                        switch(sMode){
                            case TIME:
                                setTime();
                                sMode=MENU;
                                if(setRTC()){
                                    txA("\r\nRTC set.\r\n");
                                }else{
                                    txA("\r\nRTC nack.\r\n");                                    
                                }
                                showMenu();
                                sPtr=0;
                                bChanged=1;
                                break;       //reset to menu
                            case MINUTES: setMC();sMode=MENU;txA("\r\nColour set.\r\n");showMenu();sPtr=0;bChanged=1;break;       //reset to menu
                            case HOURS: setHC();sMode=MENU;txA("\r\nColour set.\r\n");showMenu();sPtr=0;bChanged=1;break;       //reset to menu
                            case OTHER: setOC();sMode=MENU;txA("\r\nColour set.\r\n");showMenu();sPtr=0;bChanged=1;break;       //reset to menu
                        }
                    }
                    break;                                
            }
        }
        rxptr=0;                //mark rx buffer empty, data all processed
#endif
#ifdef USE_TOUCHPADS
        if(touchHoldOff==0){        //wait for timeout
            tTemp=getTouch(DOWN_BUTTON)/4;
            if(tTemp>TOUCH_ON_THRESHOLD){touchMap=touchMap|DOWN_BUTTON_BV;}  //down
            if(tTemp<TOUCH_OFF_THRESHOLD){touchMap=touchMap&(~DOWN_BUTTON_BV);}  //down
            tTemp=getTouch(UP_BUTTON)/4;
            if(tTemp>TOUCH_ON_THRESHOLD){touchMap=touchMap|UP_BUTTON_BV;}  //up
            if(tTemp<TOUCH_OFF_THRESHOLD){touchMap=touchMap&(~UP_BUTTON_BV);}  //up
            tTemp=getTouch(SET_BUTTON)/4;
            if(tTemp>TOUCH_ON_THRESHOLD){touchMap=touchMap|SET_BUTTON_BV;}  //set
            if(tTemp<TOUCH_OFF_THRESHOLD){touchMap=touchMap&(~SET_BUTTON_BV);}  //set
            if(touchMap&(~touchLast)&DOWN_BUTTON_BV){               //down pressed
                down_button();
                touchHoldOff=TOUCH_DEBOUNCE_DELAY;
                bChanged=1;
            }
            if(touchMap&(~touchLast)&UP_BUTTON_BV){               //up pressed
                up_button();
                touchHoldOff=TOUCH_DEBOUNCE_DELAY;
                bChanged=1;
            }
            if(touchMap&(~touchLast)&SET_BUTTON_BV){               //set pressed
                set_button();
                touchHoldOff=TOUCH_DEBOUNCE_DELAY;
                bChanged=1;
            }
            touchLast=touchMap;
        }
//            if(touchMap&DOWN_BUTTON_BV){RGBdata[0][1]=50;}
//            if(touchMap&UP_BUTTON_BV){RGBdata[1][1]=50;}
//            if(touchMap&SET_BUTTON_BV){RGBdata[2][1]=50;}
#endif
        if(sCtr||bChanged){
            bChanged=0;         //clear flag
#ifdef USE_TOUCHPADS
            if(settings==MENU_OFF){ //only update from RTC if menu off
#endif
                if(sCtr){
//                    Debugging for touch buttons  
//                    int t=getTouch(DOWN_BUTTON)/4;
//                    tx('T');
//                    tx(toHex[(t>>4)&0xF]);
//                    tx(toHex[(t>>0)&0xF]);
//                    tx('\r');
//                    tx('\n');
                    sCtr--;
                    if(!getRTC()){  //accumulate seconds if RTC fail
                        seconds++;
                        validateTimes();
                        /*
                        if(seconds>59){
                            seconds=seconds-60;
                            minutes++;
                            if(minutes>59){
                                minutes=minutes-60;
                                hours++;
                                if(hours>23){hours=hours-24;}
                            }
                        }
                        */
                    }
                }
#ifdef USE_TOUCHPADS
            }
#endif
            //probe LDR and set brightness variable, used in setMap below
            rawLDR=getADC(ADCRA4)/4;        //so we don't need to use longs
            brightness=((rawLDR*brightnessH)+((256-rawLDR)*brightnessL))/256;
            /*
            tx('B');
            tx(toHex[(brightness>>4)&0xF]);    
            tx(toHex[(brightness>>0)&0xF]);    
            tx('\r');
            tx('\n');
            */
            //update display
            for(i=0;i<64;i++){      //clear
                RGBdata[i][0]=0;
                RGBdata[i][1]=0;
                RGBdata[i][2]=0;
            }
//if touchpads enabled, need to override display while setting
#ifdef USE_TOUCHPADS
            if(settings==MENU_OFF){
#endif
                //normal clock display
                //hours
                hTemp=hours;
                if(minutes>=45){hTemp=hTemp+1;}     //quarter to next hour
                if(hTemp>23){hTemp=hTemp-24;}       //loop back
                if(hTemp==0){bIndex=11;setMap(bIndex,hColour[0],hColour[1],hColour[2]);}
                else if((hTemp>0)&&(hTemp<13)){bIndex=hTemp-1;setMap(bIndex,hColour[0],hColour[1],hColour[2]);}
                else{bIndex=hTemp-13;setMap(bIndex,hColour[0],hColour[1],hColour[2]);}
                //AM/PM
                if(hTemp<12){setMap(12,oColour[0],oColour[1],oColour[2]);}
                else{setMap(13,oColour[0],oColour[1],oColour[2]);}
                //minutes
                if(minutes<15){setMap(17,mColour[0],mColour[1],mColour[2]);}
                else if(minutes<30){setMap(15,mColour[0],mColour[1],mColour[2]);}
                else if(minutes<45){setMap(14,mColour[0],mColour[1],mColour[2]);}
                else {setMap(16,mColour[0],mColour[1],mColour[2]);}
#ifdef USE_TOUCHPADS
            }else{      //in settings mode
                switch(settings){
                case SET_HOUR:
                    RGBdata[7][1]=SETTINGS_RED; //letter H in red
                    RGBdata[7][0]=SETTINGS_GREEN; //letter H in green
                    RGBdata[7][2]=SETTINGS_BLUE; //letter H in blue
                    putDigits(toBCD(hours),0,SETTINGS_RED,SETTINGS_GREEN,SETTINGS_BLUE);                    
                    break;
                case SET_MINUTES:
                    RGBdata[21][1]=SETTINGS_RED; //letter M in red
                    RGBdata[21][0]=SETTINGS_GREEN; //letter M in green
                    RGBdata[21][2]=SETTINGS_BLUE; //letter M in blue
                    putDigits(toBCD(minutes),0,SETTINGS_RED,SETTINGS_GREEN,SETTINGS_BLUE);                    
                    break;
                case SET_SECONDS:
                    RGBdata[55][1]=SETTINGS_RED; //letter S in red
                    RGBdata[55][0]=SETTINGS_GREEN; //letter S in green
                    RGBdata[55][2]=SETTINGS_BLUE; //letter S in blue
                    putDigits(toBCD(seconds),0,SETTINGS_RED,SETTINGS_GREEN,SETTINGS_BLUE);                    
                    break;
                case SET_HOURS_RED:
                    RGBdata[7][1]=SETTINGS_RED; //letter H in solid red
                    RGBdata[63][1]=hColour[1]; //demo in red
                    RGBdata[63][0]=hColour[0]; //demo in green
                    RGBdata[63][2]=hColour[2]; //demo in blue                    
                    putDigits(hColour[1],0,SETTINGS_RED,SETTINGS_GREEN,SETTINGS_BLUE);                    
                    break;
                case SET_HOURS_GREEN:
                    RGBdata[7][0]=SETTINGS_GREEN; //letter H in solid green
                    RGBdata[63][1]=hColour[1]; //demo in red
                    RGBdata[63][0]=hColour[0]; //demo in green
                    RGBdata[63][2]=hColour[2]; //demo in blue                    
                    putDigits(hColour[0],0,SETTINGS_RED,SETTINGS_GREEN,SETTINGS_BLUE);                    
                    break;
                case SET_HOURS_BLUE:
                    RGBdata[7][2]=SETTINGS_BLUE; //letter H in solid blue
                    RGBdata[63][1]=hColour[1]; //demo in red
                    RGBdata[63][0]=hColour[0]; //demo in green
                    RGBdata[63][2]=hColour[2]; //demo in blue                    
                    putDigits(hColour[2],0,SETTINGS_RED,SETTINGS_GREEN,SETTINGS_BLUE);                    
                    break;
                case SET_MINUTES_RED:
                    RGBdata[21][1]=SETTINGS_RED; //letter M in solid red
                    RGBdata[63][1]=mColour[1]; //demo in red
                    RGBdata[63][0]=mColour[0]; //demo in green
                    RGBdata[63][2]=mColour[2]; //demo in blue                    
                    putDigits(mColour[1],0,SETTINGS_RED,SETTINGS_GREEN,SETTINGS_BLUE);                    
                    break;
                case SET_MINUTES_GREEN:
                    RGBdata[21][0]=SETTINGS_GREEN; //letter M in solid green
                    RGBdata[63][1]=mColour[1]; //demo in red
                    RGBdata[63][0]=mColour[0]; //demo in green
                    RGBdata[63][2]=mColour[2]; //demo in blue                    
                    putDigits(mColour[0],0,SETTINGS_RED,SETTINGS_GREEN,SETTINGS_BLUE);                    
                    break;
                case SET_MINUTES_BLUE:
                    RGBdata[21][2]=SETTINGS_BLUE; //letter M in solid blue
                    RGBdata[63][1]=mColour[1]; //demo in red
                    RGBdata[63][0]=mColour[0]; //demo in green
                    RGBdata[63][2]=mColour[2]; //demo in blue                    
                    putDigits(mColour[2],0,SETTINGS_RED,SETTINGS_GREEN,SETTINGS_BLUE);                    
                    break;
                case SET_OTHER_RED:
                    RGBdata[62][1]=SETTINGS_RED; //letter O in solid red
                    RGBdata[63][1]=oColour[1]; //demo in red
                    RGBdata[63][0]=oColour[0]; //demo in green
                    RGBdata[63][2]=oColour[2]; //demo in blue                    
                    putDigits(oColour[1],0,SETTINGS_RED,SETTINGS_GREEN,SETTINGS_BLUE);                    
                    break;
                case SET_OTHER_GREEN:
                    RGBdata[62][0]=SETTINGS_GREEN; //letter O in solid green
                    RGBdata[63][1]=oColour[1]; //demo in red
                    RGBdata[63][0]=oColour[0]; //demo in green
                    RGBdata[63][2]=oColour[2]; //demo in blue                    
                    putDigits(oColour[0],0,SETTINGS_RED,SETTINGS_GREEN,SETTINGS_BLUE);                    
                    break;
                case SET_OTHER_BLUE:
                    RGBdata[62][2]=SETTINGS_BLUE; //letter O in solid blue
                    RGBdata[63][1]=oColour[1]; //demo in red
                    RGBdata[63][0]=oColour[0]; //demo in green
                    RGBdata[63][2]=oColour[2]; //demo in blue                    
                    putDigits(oColour[2],0,SETTINGS_RED,SETTINGS_GREEN,SETTINGS_BLUE);                    
                    break;
                case SET_BRIGHTNESS_LOW:
                    putDigit(digits[18],10,brightnessL,brightnessL,brightnessL);    //L
                    putDigit(digits[0],42,brightnessL,brightnessL,brightnessL);    //0
                    RGBdata[8][1]=brightnessL;   //red
                    RGBdata[16][0]=brightnessL;  //green
                    RGBdata[24][2]=brightnessL;  //blue
                    RGBdata[32][1]=brightnessL;  //
                    RGBdata[32][0]=brightnessL;  //yellow
                    RGBdata[40][0]=brightnessL;  //
                    RGBdata[40][2]=brightnessL;  //cyan
                    RGBdata[48][1]=brightnessL;  //
                    RGBdata[48][2]=brightnessL;  //magenta
                    RGBdata[56][1]=brightnessL;  //
                    RGBdata[56][0]=brightnessL;  //
                    RGBdata[56][2]=brightnessL;  //white
                    break;
                case SET_BRIGHTNESS_HIGH:
                    putDigit(digits[16],10,brightnessH,brightnessH,brightnessH);    //H
                    putDigit(digits[17],42,brightnessH,brightnessH,brightnessH);    //I
                    RGBdata[8][1]=brightnessH;   //red
                    RGBdata[16][0]=brightnessH;  //green
                    RGBdata[24][2]=brightnessH;  //blue
                    RGBdata[32][1]=brightnessH;  //
                    RGBdata[32][0]=brightnessH;  //yellow
                    RGBdata[40][0]=brightnessH;  //
                    RGBdata[40][2]=brightnessH;  //cyan
                    RGBdata[48][1]=brightnessH;  //
                    RGBdata[48][2]=brightnessH;  //magenta
                    RGBdata[56][1]=brightnessH;  //
                    RGBdata[56][0]=brightnessH;  //
                    RGBdata[56][2]=brightnessH;  //white
                    break;
                case SET_SAVE:
                    setMap(18,0,255,0);    //red save icon
                    break;


                }            
            }
#endif

#ifdef USE_USB_INTERFACE            
#ifdef USE_USB_DEBUG
            if(debug){dumpLEDtoUSB();}
#endif
#endif
            sendLEDdata();
        }
#ifdef USE_USB_INTERFACE            
		
        //data is sent from txbuf, size is txptr
        if (usb_is_configured() &&
		    !usb_in_endpoint_halted(2) &&
		    !usb_in_endpoint_busy(2)) {
			int i,k;
			unsigned char *buf = usb_get_in_buffer(2);
            k=txptr;
            if(k){
                if(k>32){k=32;}     //limit data sent            
                for (i = 0; i < k; i++) {buf[i] = txbuf[i];}
                for(i=0;i<txptr;i++){txbuf[i]=txbuf[i+k];}  //move buffer up
                txptr=txptr-k;
                usb_send_in_buffer(2, k);
            }
		}

		//data is received from buffer and placed in rxbuf, size is rxptr
		//silently dumps any overflow; can be checked below
        if (usb_is_configured() &&
		    !usb_out_endpoint_halted(2) &&
		    usb_out_endpoint_has_data(2)) {
			const unsigned char *out_buf;
			size_t out_buf_len;
			int i,k;
			out_buf_len = usb_get_out_buffer(2, &out_buf);
            k=rxptr+out_buf_len;
            if(k>BUFSIZE-2){k=BUFSIZE-2;}
            for(i=rxptr;i<k;i++){rxbuf[i]=out_buf[i-rxptr];}
            rxptr=k;
            usb_arm_out_endpoint(2);
		}
#endif
    }
}

#ifdef USE_USB_INTERFACE            

/* Callbacks. These function names are set in usb_config.h. */
void app_set_configuration_callback(uint8_t configuration)
{

}

uint16_t app_get_device_status_callback()
{
	return 0x0000;
}

void app_endpoint_halt_callback(uint8_t endpoint, bool halted)
{

}

int8_t app_set_interface_callback(uint8_t interface, uint8_t alt_setting)
{
	return 0;
}

int8_t app_get_interface_callback(uint8_t interface)
{
	return 0;
}

void app_out_transaction_callback(uint8_t endpoint)
{

}

void app_in_transaction_complete_callback(uint8_t endpoint)
{

}

int8_t app_unknown_setup_request_callback(const struct setup_packet *setup)
{
	/* To use the CDC device class, have a handler for unknown setup
	 * requests and call process_cdc_setup_request() (as shown here),
	 * which will check if the setup request is CDC-related, and will
	 * call the CDC application callbacks defined in usb_cdc.h. For
	 * composite devices containing other device classes, make sure
	 * MULTI_CLASS_DEVICE is defined in usb_config.h and call all
	 * appropriate device class setup request functions here.
	 */
	return process_cdc_setup_request(setup);
}

int16_t app_unknown_get_descriptor_callback(const struct setup_packet *pkt, const void **descriptor)
{
	return -1;
}

void app_start_of_frame_callback(void)
{

}

void app_usb_reset_callback(void)
{

}

/* CDC Callbacks. See usb_cdc.h for documentation. */

int8_t app_send_encapsulated_command(uint8_t interface, uint16_t length)
{
	return -1;
}

int16_t app_get_encapsulated_response(uint8_t interface,
                                      uint16_t length, const void **report,
                                      usb_ep0_data_stage_callback *callback,
                                      void **context)
{
	return -1;
}

int8_t app_set_comm_feature_callback(uint8_t interface,
                                     bool idle_setting,
                                     bool data_multiplexed_state)
{
	return -1;
}

int8_t app_clear_comm_feature_callback(uint8_t interface,
                                       bool idle_setting,
                                       bool data_multiplexed_state)
{
	return -1;
}

int8_t app_get_comm_feature_callback(uint8_t interface,
                                     bool *idle_setting,
                                     bool *data_multiplexed_state)
{
	return -1;
}

int8_t app_set_line_coding_callback(uint8_t interface,
                                    const struct cdc_line_coding *coding)
{
#ifdef USE_LINE_CODING
	line_coding = *coding;
#endif
	return 0;
}

int8_t app_get_line_coding_callback(uint8_t interface,
                                    struct cdc_line_coding *coding)
{
	/* This is where baud rate, data, stop, and parity bits are set. */
#ifdef USE_LINE_CODING
	*coding = line_coding;
#endif
	return 0;
}

int8_t app_set_control_line_state_callback(uint8_t interface,
                                           bool dtr, bool dts)
{
	return 0;
}

int8_t app_send_break_callback(uint8_t interface, uint16_t duration)
{
	return 0;
}
#endif


#ifdef _PIC14E
void interrupt isr()
{
	static unsigned char tmr1postacc=0;
#ifdef USE_USB_INTERFACE            
    usb_service();
#endif
    if((PIR1bits.TMR1IF)&&(PIE1bits.TMR1IE)){              //Timer1 interrupt
        PIR1bits.TMR1IF=0;            //clear flag
        tmr1postacc++;                //divide down to 1s
        if(tmr1postacc>182){            
            sCtr++;                   //global seconds counter
            tmr1postacc=0;
        }
#ifdef USE_TOUCHPADS
        if(touchHoldOff){touchHoldOff--;}
#endif
    }
}
#elif _PIC18

#ifdef __XC8
void interrupt high_priority isr()
{
	usb_service();
}
#elif _PICC18
#error need to make ISR
#endif

#endif

#ifdef USE_TOUCHPADS

void set_button(void){
    if(settings==SET_SECONDS){
        setRTC();           //update RTC when leaving seconds setting
    }
    if(settings<SET_SAVE){
        settings++;
    }else{
        settings=MENU_OFF;
    }
}

void up_button(void){
    switch(settings){
        //don't need to do anything with up/down buttons in idle mode
        case SET_HOUR:
            hours=hours+1;
            if(hours>23){hours=hours-24;}
            break;
        case SET_MINUTES:
            minutes=minutes+1;
            if(minutes>59){minutes=minutes-60;}
            break;
        case SET_SECONDS:
            seconds=seconds+1;
            if(seconds>59){seconds=seconds-60;}
            break;
        case SET_HOURS_RED:
            if(hColour[1]<BUTTON_MAX){hColour[1]=hColour[1]+BUTTON_STEP;}
            break;
        case SET_HOURS_GREEN:
            if(hColour[0]<BUTTON_MAX){hColour[0]=hColour[0]+BUTTON_STEP;}
            break;
        case SET_HOURS_BLUE:
            if(hColour[2]<BUTTON_MAX){hColour[2]=hColour[2]+BUTTON_STEP;}
            break;
        case SET_MINUTES_RED:
            if(mColour[1]<BUTTON_MAX){mColour[1]=mColour[1]+BUTTON_STEP;}
            break;
        case SET_MINUTES_GREEN:
            if(mColour[0]<BUTTON_MAX){mColour[0]=mColour[0]+BUTTON_STEP;}
            break;
        case SET_MINUTES_BLUE:
            if(mColour[2]<BUTTON_MAX){mColour[2]=mColour[2]+BUTTON_STEP;}
            break;
        case SET_OTHER_RED:
            if(oColour[1]<BUTTON_MAX){oColour[1]=oColour[1]+BUTTON_STEP;}
            break;
        case SET_OTHER_GREEN:
            if(oColour[0]<BUTTON_MAX){oColour[0]=oColour[0]+BUTTON_STEP;}
            break;
        case SET_OTHER_BLUE:
            if(oColour[2]<BUTTON_MAX){oColour[2]=oColour[2]+BUTTON_STEP;}
            break;
        case SET_BRIGHTNESS_LOW:
            if(brightnessL<255){brightnessL=brightnessL+1;}
            break;
        case SET_BRIGHTNESS_HIGH:
            if(brightnessH<255){brightnessH=brightnessH+1;}
            break;
        case SET_SAVE:
            saveSettings();
            settings=MENU_OFF;
            break;
    }        
}

void down_button(void){
    switch(settings){
        //don't need to do anything with up/down buttons in idle mode
        case SET_HOUR:
            hours=hours-1;
            if(hours>23){hours=hours+24;}   //unsigned chars
            break;
        case SET_MINUTES:
            minutes=minutes-1;
            if(minutes>59){minutes=minutes+60;}   //unsigned chars
            break;
        case SET_SECONDS:
            seconds=seconds-1;
            if(seconds>59){seconds=seconds+60;}   //unsigned chars
            break;
        case SET_HOURS_RED:
            if(hColour[1]>BUTTON_MIN){hColour[1]=hColour[1]-BUTTON_STEP;}
            break;
        case SET_HOURS_GREEN:
            if(hColour[0]>BUTTON_MIN){hColour[0]=hColour[0]-BUTTON_STEP;}
            break;
        case SET_HOURS_BLUE:
            if(hColour[2]>BUTTON_MIN){hColour[2]=hColour[2]-BUTTON_STEP;}
            break;
        case SET_MINUTES_RED:
            if(mColour[1]>BUTTON_MIN){mColour[1]=mColour[1]-BUTTON_STEP;}
            break;
        case SET_MINUTES_GREEN:
            if(mColour[0]>BUTTON_MIN){mColour[0]=mColour[0]-BUTTON_STEP;}
            break;
        case SET_MINUTES_BLUE:
            if(mColour[2]>BUTTON_MIN){mColour[2]=mColour[2]-BUTTON_STEP;}
            break;
        case SET_OTHER_RED:
            if(oColour[1]>BUTTON_MIN){oColour[1]=oColour[1]-BUTTON_STEP;}
            break;
        case SET_OTHER_GREEN:
            if(oColour[0]>BUTTON_MIN){oColour[0]=oColour[0]-BUTTON_STEP;}
            break;
        case SET_OTHER_BLUE:
            if(oColour[2]>BUTTON_MIN){oColour[2]=oColour[2]-BUTTON_STEP;}
            break;
        case SET_BRIGHTNESS_LOW:
            if(brightnessL>2){brightnessL=brightnessL-1;}
            break;
        case SET_BRIGHTNESS_HIGH:
            if(brightnessH>2){brightnessH=brightnessH-1;}
            break;
        case SET_SAVE:
            saveSettings();
            settings=MENU_OFF;
            break;
    }    
}
#endif
