//Silicon Chip Peltier Controller
//Pair of custom shields
//Fast PWM on D9/D10
//I2C LCD, IR receiver, LEDs, Temp and RPM sensing
//Timers:
//TMR0: millis()
//TMR1: PWM for Peltiers 9/10
//TMR2: IR remote/fan PWM, make do with on/off for fans

//alarm thresholds
//to ignore, set to high values
#define PELTIER_MAX 80
#define VARIANCE_ALERT 10

//Very Fast PWM using custom TOP
//300kHz
//#define PWM_MAX 53
//400kHz
#define PWM_MAX 39
//200kHz
//#define PWM_MAX 79
#define PWM_TOP (PWM_MAX-1)
//change the following to limit current if your PSU is small
#define MAX_DUTY_CYCLE 100

//For I2C LCD
#include <Wire.h>
#include <I2CLCD.h>
I2CLCD lcd;  //no address here, set in begin()

//for DS18B20's
#include <OneWire.h>
#include <DallasTemperature.h>

//FOR IR receiver
#include <IRremote.h>
IRrecv irrecv(1);
decode_results results;

#if !(defined(__AVR_ATmega328P__) || defined(__AVR_ATmega168__))
#error This sketch may give unexpected results for processors other than ATMega328/168
#endif

#include <util/atomic.h> // this library includes the ATOMIC_BLOCK macro.
#define RED_LED_PIN A3
#define BLUE_LED_PIN 2
#define GREEN_LED_PIN 13
#define BUZZER_PIN 12
#define PUMP1_PIN 7
#define PUMP2_PIN 8
#define FAN1_TACHO 4
#define FAN2_TACHO 5
#define FAN3_TACHO 6
#define FAN_PWM 3
//cannot use UART as TX in use by IR RX
#define IR_PIN 1
#define ANALOG_BANK_PIN 2
#define ANALOG_X_PIN A2
#define ANALOG_Y_PIN A1
#define ANALOG_Z_PIN A0
#define OVERSAMPLE 16
#define NTC_DIVIDER (4700.0)
#define TACHO_TIME 150
#define OFFSET_LIMIT 30

volatile unsigned long fan1PulseCount=0;
volatile unsigned long fan2PulseCount=0;
volatile unsigned long fan3PulseCount=0;
unsigned int fan1RPM=0;
unsigned int fan2RPM=0;
unsigned int fan3RPM=0;

enum LED {
  OFF=0,
  RED,
  GREEN,
  BLUE
};

enum TSensor{   //use these names in the array below
  NONE=0,
  NTC10K,
  NTC100K,
  DS18B20
};

char TempSensor[]={
  NONE,      //align CON# with index
  NTC10K,    //CON1
  NTC10K,    //CON2
  NTC10K,    //CON3
  NTC10K,    //CON4
  NTC10K,    //CON5
  DS18B20    //on PCB
};

//mode defs
#define SETPOINT 1
#define COOL 2
#define HEAT 3
#define PROFILE 4
#define FAULT 5

//globals
long pDrive=0;         //Peltier drive in %, neg is cool
long tempset=200;      //tenths of a degree
int runState=0;        //off

//use the following to run on power up (for example)
//long tempset=260;      //tenths of a degree
//int runState=SETPOINT; //target

//for running temp profiles
unsigned long pStart=0; //start time for profile run
int pIndex=0;          //index into profile steps

#define HOURS *3600L
#define MINUTES *60L
typedef struct {unsigned long time; int temp;} point;   //NB: temps are in tenths of a degree, time in seconds
point profile[]={
  {0,200},     //(time,temp) first point should be at time=0, times should increase
  {180,200},
  {181,220},
  {300,220},
  {301,200},
  {480,200}
};
int pEntries=sizeof(profile)/sizeof(point);

void setup() {
  int LCDaddress;
  LEDSetup();
  LCDaddress=lcd.findAddress(0x20);   //start at 0x20, PCF8574* should be 0x20-0x3F
  if((LCDaddress<0x20)||(LCDaddress>0x3F)){while(1){LEDSet(RED);delay(100);LEDSet(OFF);delay(100);}}        //no display
  lcd.begin(LCDaddress,20, 4); // initialize the lcd at the found address
  lcd.setBacklight(255);
  lcd.home(); lcd.clear();
  pinMode(PUMP1_PIN,OUTPUT);        //digital on/off only
  digitalWrite(PUMP1_PIN,LOW);
  pinMode(PUMP2_PIN,OUTPUT);
  digitalWrite(PUMP2_PIN,LOW);
  pinMode(FAN_PWM,OUTPUT);
  digitalWrite(FAN_PWM,LOW);
  pinMode(BUZZER_PIN,OUTPUT);
  digitalWrite(BUZZER_PIN,LOW);
  irrecv.enableIRIn(); // Start the receiver
  LEDSet(RED);          //start in off state
  pinMode(9,OUTPUT);    //Peltier controls
  pinMode(10,OUTPUT);
  digitalWrite(9,LOW);
  digitalWrite(10,LOW);   //startup failsafes  
}

void loop() {
  long pTarget;             //we'll do a slow ramp to avoid damage
  float pDiff;              //difference across peltiers
  static long pIoffset=0;   //integrating term
  int t;
  byte i;
  static LED leds=0;
  long t0,t1;
  float temps[7];     //7 for indexes 1-6
  unsigned long pTime;      //for profile
  t0=millis();
  for(i=1;i<7;i++){temps[i]=getTemp(i);}        //read everything once to minimise downtime
  t1=millis();
  tachoCheck();                     //get latest tacho readings
  //update profile if running
  if(runState==PROFILE){
    pTime=(millis()-pStart)/1000;   //seconds of running in profile
    long tTarget;
    while((pTime>=profile[pIndex+1].time)&&(pIndex<pEntries-1)){pIndex++;}   //move forward as necessary, should also avoid div/0 and non-monotonic errors
    if(pIndex>=pEntries-1){runState=0;}           //stop if end of profile reached 
    if(runState){
      tempset=(((profile[pIndex+1].time-pTime)*profile[pIndex].temp)+((pTime-profile[pIndex].time)*profile[pIndex+1].temp))/(profile[pIndex+1].time-profile[pIndex].time);
    }
  }  
  lcd.home();                       //display temperatures/fan speeds
  lcd.print(F("T1="));              //3cols    
  printN(&lcd,temps[1],2);          //2 cols   =5
  lcd.print(F(" T4="));             //4cols    =9
  printN(&lcd,temps[4],2);          //2 cols =11
  if(runState!=PROFILE){            //override part of display in profile mode
    lcd.print(F(" F1="));             //4cols  =15
    printN(&lcd,fan1RPM,4);           //4 cols =19
  }else{
    lcd.print(F(" t="));            //3cols =14
    printN(&lcd,pTime,5);           //5 cols =19    
  }
  lcd.setCursor(0,1);
  lcd.print(F("T2="));              //3cols    
  printN(&lcd,temps[2],2);          //2 cols   =5
  lcd.print(F(" T5="));             //4cols    =9
  printN(&lcd,temps[5],2);          //2 cols =11
  if(runState!=PROFILE){            //override part of display in profile mode
    lcd.print(F(" F2="));             //4cols  =15
    printN(&lcd,fan2RPM,4);           //4 cols =19
  }else{
    lcd.print(F(" step="));            //6cols =17
    printN(&lcd,pIndex,2);           //2 cols =19    
  }
  lcd.setCursor(0,2);
  lcd.print(F("T3="));              //3cols    
  printN(&lcd,temps[3],2);          //2 cols   =5
  lcd.print(F(" T6="));             //4cols    =9
  printN(&lcd,temps[6],2);          //2 cols =11
  if(runState!=PROFILE){            //override part of display in profile mode
    lcd.print(F(" F3="));             //4cols  =15
    printN(&lcd,fan3RPM,4);           //4 cols =19
  }else{
    lcd.print(F(" Tgt="));            //5cols =16
    printN(&lcd,profile[pIndex+1].temp/10,3);           //3 cols =19    
  }
  //set outputs  NB: tempset is in tenths of a degree!  
  pTarget=(tempset-(temps[1]*10))*5;     //rough P only algorithm
  if(tempset<(temps[1]*10)){pIoffset--;if(pIoffset<-OFFSET_LIMIT){pIoffset=-OFFSET_LIMIT;}}  //integrating term
  if(tempset>(temps[1]*10)){pIoffset++;if(pIoffset>OFFSET_LIMIT){pIoffset=OFFSET_LIMIT;}}
  pTarget=pTarget+pIoffset;
  if(pTarget<-MAX_DUTY_CYCLE){pTarget=-MAX_DUTY_CYCLE;}     //bounds checking
  if(pTarget>MAX_DUTY_CYCLE){pTarget=MAX_DUTY_CYCLE;}  
  //adjust output in case cooling at high dT is needed
  //wind back Peltiers is dT>30, as passive transfer will be significant and we won't need to dissipate I2R losses too
  if(temps[1]>temps[2]){        //difference across peltiers, take (max of T1 or T2)-T3
    pDiff=temps[1]-temps[3];
  }else{
    pDiff=temps[2]-temps[3];    
  }
  pDiff=pDiff-20;               //adjust Peltier output if more than 20degrees cooling is needed
  if(pDiff<0){pDiff=0;}         //ignore negative
  pTarget=pTarget+pDiff*3;      //apply adjustment of 3%/degree
  if(pTarget<-MAX_DUTY_CYCLE){pTarget=-MAX_DUTY_CYCLE;}     //bounds checking
  if(pTarget>MAX_DUTY_CYCLE){pTarget=MAX_DUTY_CYCLE;}  
  converge(&pDrive,pTarget,5);      //move pDrive towards pTarget
  if(((temps[2]>PELTIER_MAX)||(temps[3]>PELTIER_MAX))&&(runState!=OFF)){     //fault=shutdown
    runState=FAULT;
    LEDSet(RED);      //flash red
    for(int i=0;i<300;i++){            //no timers available, so manual tone
      digitalWrite(BUZZER_PIN,HIGH);
      delay(1);
      digitalWrite(BUZZER_PIN,LOW);
      delay(1);      
    }
  }
  if((runState==SETPOINT)&&(abs(temps[1]*10-tempset)>(VARIANCE_ALERT*10))){   //variance condition
    LEDSet(RED);      //flash red
    for(int i=0;i<50;i++){            //no timers available, so manual tone
      digitalWrite(BUZZER_PIN,HIGH);
      delay(1);
      digitalWrite(BUZZER_PIN,LOW);
      delay(1);      
    }
  }
  if(runState==OFF){pDrive=0;pIoffset=0;}      //fixed values for most modes
  if(runState==FAULT){pDrive=0;pIoffset=0;}      //fixed values for most modes
  if(runState==COOL){pDrive=-100;pIoffset=0;}  //override calculated control values
  if(runState==HEAT){pDrive=100;pIoffset=0;}
  setBipolar((pDrive*PWM_TOP)/100);   //scaled output
  //setBipolar(-(pDrive*PWM_TOP)/100);   //scaled output, use this one if output is backwards  
  checkIRdelay(500);             //non-blocking delay checks IR
  if(runState!=0){
    digitalWrite(PUMP1_PIN,HIGH);
    digitalWrite(PUMP2_PIN,HIGH);
    digitalWrite(FAN_PWM,HIGH);    
    if(pDrive<0){
      LEDSet(BLUE);         //blue LED for cool
    }else{
      LEDSet(GREEN);        //green LED otherwise
    }
  }else{
    digitalWrite(PUMP1_PIN,LOW);
    digitalWrite(PUMP2_PIN,LOW);
    digitalWrite(FAN_PWM,LOW);        
    LEDSet(RED);
  }
  lcd.setCursor(0,3);
  switch(runState){
    case OFF:      lcd.print(F("OFF: "));break;
    case SETPOINT: lcd.print(F("SET: "));break;
    case COOL:     lcd.print(F("COOL:"));break;
    case HEAT:     lcd.print(F("HEAT:"));break;
    case PROFILE:  lcd.print(F("PROF:"));break;
    case FAULT:    lcd.print(F("FLT: "));break;
    default:       lcd.print(F("???? "));break;    //ERROR!
  }
  if(tempset>99){lcd.write((tempset/100)%10+'0');}else{lcd.write(' ');}
  lcd.write((tempset/10)%10+'0');
  lcd.write('.');
  lcd.write((tempset)%10+'0');
  lcd.print(F(" PELT="));
  if(pDrive<0){
    lcd.write('-');
  }else if(pDrive>0){
    lcd.write('+');
  }else{
    lcd.write(' ');    
  }
  if(abs(pDrive)>99){lcd.write((abs(pDrive)/100)%10+'0');}else{lcd.write(' ');}
  if(abs(pDrive)>9){lcd.write((abs(pDrive)/10)%10+'0');}else{lcd.write(' ');}
  lcd.write((abs(pDrive))%10+'0');
}

void converge(long* d, long s, long inc){  
  long i = s - *d;
  inc=abs(inc);           //sanity check
  if(i<-inc){i=-inc;}
  if(i>inc){i=inc;}
  *d=*d+i;
}

void checkIRdelay(unsigned long t){
  unsigned long t0=millis();
  static unsigned long oldcode=0x400;     //not a valid code for either
  unsigned long code;  
  while(millis()-t0<t){
    if (irrecv.decode(&results)) {
      code=results.value;
      irrecv.resume();          // Receive the next value
      if(code!=oldcode){IRaction(code);oldcode=code;}      
    }    
  }
}

void IRaction(unsigned long code){     //interpret Jaycar XC3718 or Altronics A1012 with TV=089
  switch(code){
    case 0xFFC23D:    //XC3718 play/pause
    case 0x00000C:    //A1012 power
    case 0x00080C:
      if(runState){
        runState=OFF;
      }else{
        runState=SETPOINT;
      }
      break;
    case 0xFFE21D:    //XC3718 CH+
    case 0x000020:    //A1012 CH+
    case 0x000820:
      if(runState){
        runState=OFF;
      }else{
        runState=HEAT;
      }
      break;
    case 0xFFA25D:    //XC3718 CH-
    case 0x000021:    //A1012 CH-
    case 0x000821:
      if(runState){
        runState=OFF;
      }else{
        runState=COOL;
      }      
      break;
    case 0xFFA857:    //XC3718 VOL+
    case 0x000010:    //A1012 VOL+
    case 0x000810:
      tempset=tempset+1;
      if(tempset>999){tempset=999;}
      break;
    case 0xFFE01F:    //XC3718 VOL-
    case 0x000011:    //A1012 VOL-
    case 0x000811:
      tempset=tempset-1;      
      if(tempset<0){tempset=0;}
      break;
    case 0xFF6897:    //XC3718 0
    case 0x000000:    //A1012 0
    case 0x000800:
      if(runState==OFF){tempset=(tempset*10+0)%1000;}
      break;
    case 0xFF30CF:    //XC3718 1
    case 0x000001:    //A1012 1
    case 0x000801:
      if(runState==OFF){tempset=(tempset*10+1)%1000;}      
      break;
    case 0xFF18E7:    //XC3718 2
    case 0x000002:    //A1012 2
    case 0x000802:
      if(runState==OFF){tempset=(tempset*10+2)%1000;}
      break;
    case 0xFF7A85:    //XC3718 3
    case 0x000003:    //A1012 3
    case 0x000803:
      if(runState==OFF){tempset=(tempset*10+3)%1000;}
      break;
    case 0xFF10EF:    //XC3718 4
    case 0x000004:    //A1012 4
    case 0x000804:
      if(runState==OFF){tempset=(tempset*10+4)%1000;}
      break;
    case 0xFF38C7:    //XC3718 5
    case 0x000005:    //A1012 5
    case 0x000805:
      if(runState==OFF){tempset=(tempset*10+5)%1000;}
      break;
    case 0xFF5AA5:    //XC3718 6
    case 0x000006:    //A1012 6
    case 0x000806:
      if(runState==OFF){tempset=(tempset*10+6)%1000;}
      break;
    case 0xFF42BD:    //XC3718 7
    case 0x000007:    //A1012 7
    case 0x000807:
      if(runState==OFF){tempset=(tempset*10+7)%1000;}
      break;
    case 0xFF4AB5:    //XC3718 8
    case 0x000008:    //A1012 8
    case 0x000808:
      if(runState==OFF){tempset=(tempset*10+8)%1000;}
      break;
    case 0xFF52AD:    //XC3718 9
    case 0x000009:    //A1012 9
    case 0x000809:
      if(runState==OFF){tempset=(tempset*10+9)%1000;}
      break;   
    case 0xFF906F:    //XC3718 EQ
    case 0x000022:    //A1012 -/--
    case 0x000822:
      if(runState==OFF){runState=PROFILE;pStart=millis();pIndex=0;}   //activate and reset profile mode
      break;   
  }
}


char printN(Print* s, long n, char p){    //print n to print object s in p columns
  char i,sign,t,r;  
  boolean z=1;            //leading zeros
  long mx;                //highest displayable value
  sign=0;
  r=0;
  if(p<1){return;}        //nothing to do
  mx=1;
  for(i=0;i<p;i++){mx=mx*10;}
  if(n<0){sign=1;n=-n;mx=mx/10;s->write('-');r++;}   //flag sign, one less spot for digits
  if(n>=mx){n=mx-1;}                          //clip
  mx=mx/10;
  while(mx){
    t=n/mx;
    n=n-(t*mx);
    if((t==0)&&z&&(mx>9)){
      s->write(' ');
    }else{
      s->write(t+'0');
      z=0;
    }
    mx=mx/10;
    r++;    
  }
  return r;
}

void LEDSetup(){                  //set direction and state
  pinMode(RED_LED_PIN,OUTPUT);
  pinMode(BLUE_LED_PIN,OUTPUT);
  pinMode(GREEN_LED_PIN,OUTPUT);
  digitalWrite(RED_LED_PIN,LOW);
  digitalWrite(BLUE_LED_PIN,LOW);
  digitalWrite(GREEN_LED_PIN,LOW);  
}

void LEDSet(LED s){                 //simple interface to LEDs
  digitalWrite(RED_LED_PIN,LOW);
  digitalWrite(BLUE_LED_PIN,LOW);
  digitalWrite(GREEN_LED_PIN,LOW);
  switch(s){
    case RED: digitalWrite(RED_LED_PIN,HIGH); break;
    case GREEN: digitalWrite(GREEN_LED_PIN,HIGH); break;
    case BLUE: digitalWrite(BLUE_LED_PIN,HIGH); break;
  }
}

void tachoSetup(){                //set as inputs and enable Pin Change Interrupt
  pinMode(FAN1_TACHO,INPUT_PULLUP);
  pinMode(FAN2_TACHO,INPUT_PULLUP);
  pinMode(FAN3_TACHO,INPUT_PULLUP);
  pciSetup(FAN1_TACHO);
  pciSetup(FAN2_TACHO);
  pciSetup(FAN3_TACHO);
}

void tachoCheck(){                      //one-off fan speed check, does not require previous setup
  unsigned long t0;
  unsigned long f1Temp,f2Temp,f3Temp;  
  tachoSetup();
  t0=millis();  
  ATOMIC_BLOCK(ATOMIC_RESTORESTATE){    //snapshot/clear volatiles
    fan1PulseCount=0;
    fan2PulseCount=0;
    fan3PulseCount=0;
  }
  while(millis()-t0<TACHO_TIME){}       //wait for TACHO_TIME
  ATOMIC_BLOCK(ATOMIC_RESTORESTATE){    //snapshot volatiles
    f1Temp=fan1PulseCount;
    f2Temp=fan2PulseCount;
    f3Temp=fan3PulseCount;
  }
  pciRelease(FAN1_TACHO);
  pciRelease(FAN2_TACHO);
  pciRelease(FAN3_TACHO);
  fan1RPM=f1Temp*(30000L/TACHO_TIME);   //2 pulses/rev * 30000 ms/min is conversion
  fan2RPM=f2Temp*(30000L/TACHO_TIME);   //2 pulses/rev * 30000 ms/min is conversion
  fan3RPM=f3Temp*(30000L/TACHO_TIME);   //2 pulses/rev * 30000 ms/min is conversion  
}

void pciSetup(byte pin){      //from https://playground.arduino.cc/Main/PinChangeInterrupt
  *digitalPinToPCMSK(pin) |= bit (digitalPinToPCMSKbit(pin));   // enable pin
  PCIFR  |= bit (digitalPinToPCICRbit(pin));                    // clear any outstanding interrupt
  PCICR  |= bit (digitalPinToPCICRbit(pin));                    // enable interrupt for the group
}

void pciRelease(byte pin){      //clear mask only
  *digitalPinToPCMSK(pin) &= ~(bit(digitalPinToPCMSKbit(pin)));   // disable pin  
}

ISR (PCINT2_vect){                //handle pin change interrupt for D0 to D7 here
  static char oldState=0xFF;      //assume pullups for idle state
  char newState=PIND;             //port D is D0-D7
  if((newState^oldState)&(digitalPinToBitMask(FAN1_TACHO))){fan1PulseCount++;}    //count on change
  if((newState^oldState)&(digitalPinToBitMask(FAN2_TACHO))){fan2PulseCount++;}    //count on change
  if((newState^oldState)&(digitalPinToBitMask(FAN3_TACHO))){fan3PulseCount++;}    //count on change
  oldState=newState;
}

float getTemp(char n){      //get temperature
  float a,r;                                        //scratchpad analog, resistance
  float retVal=-999;                                //fail default
  int IOPin=-1;                                     //pin to use for IO
  boolean bankPinState=digitalRead(ANALOG_BANK_PIN);//save pin to restore (multiplexed with LEDs)
  boolean LEDPinState=digitalRead(RED_LED_PIN);     //other pin (avoid wrong transient output), pins are written the same to give no output
  switch(n){                                        //select bank/pin based on n
    case 1:
      IOPin=ANALOG_X_PIN;
      digitalWrite(ANALOG_BANK_PIN,LOW);
      digitalWrite(RED_LED_PIN,LOW);
      break;
    case 2:
      IOPin=ANALOG_Y_PIN;
      digitalWrite(ANALOG_BANK_PIN,LOW);
      digitalWrite(RED_LED_PIN,LOW);
      break;
    case 3:
      IOPin=ANALOG_Z_PIN;
      digitalWrite(ANALOG_BANK_PIN,LOW);
      digitalWrite(RED_LED_PIN,LOW);
      break;
    case 4:
      IOPin=ANALOG_X_PIN;
      digitalWrite(ANALOG_BANK_PIN,HIGH);
      digitalWrite(RED_LED_PIN,HIGH);
      break;
    case 5:
      IOPin=ANALOG_Y_PIN;
      digitalWrite(ANALOG_BANK_PIN,HIGH);
      digitalWrite(RED_LED_PIN,HIGH);
      break;
    case 6:
      IOPin=ANALOG_Z_PIN;
      digitalWrite(ANALOG_BANK_PIN,HIGH);
      digitalWrite(RED_LED_PIN,HIGH);
      break;
  }
  switch(TempSensor[n]){     //choose routine
    case NTC10K:
      a=0;
      analogRead(IOPin);          //dummy read
      for(int i=0;i<OVERSAMPLE;i++){
        a=a+analogRead(IOPin);
      }
      a=a/OVERSAMPLE;
      r=(a*NTC_DIVIDER)/(1024-a);
      retVal=1/(1/298.15+log(r/10000.0)/3100.0)-273.15;    //simplified Steinhart-Hart with R0=10k, beta=3100 at T0 25C
      break;
    case NTC100K:
      a=0;
      analogRead(IOPin);          //dummy read
      for(int i=0;i<OVERSAMPLE;i++){
        a=a+analogRead(IOPin);
      }
      a=a/OVERSAMPLE;
      r=(a*NTC_DIVIDER)/(1024-a);
      retVal=1/(1/298.15+log(r/100000.0)/4400.0)-273.15;    //simplified Steinhart-Hart with R0=100k, beta=4400 at T0 25C
      break;
    case DS18B20:
      OneWire oneWire(IOPin);
      DallasTemperature sensors(&oneWire);
      sensors.begin();
      sensors.setWaitForConversion(false);                  //don't wait between
      retVal=sensors.getTempCByIndex(0);                    //get previous result      
      sensors.requestTemperatures();                        //then start next
      break;
  }
  digitalWrite(ANALOG_BANK_PIN,bankPinState);               //restore bank pin
  digitalWrite(RED_LED_PIN,LEDPinState);                    //restore other LED pin
  return retVal;
}

void manualPWM10(int n){            //fast PWM on 10
  if(n<1){
    TCCR1A = (TCCR1A & 207)| (0<<4);  //COM1B, detach PWM
    digitalWrite(10,LOW);
  }else{
    if(n>PWM_MAX){n=PWM_MAX;}
    n=n-1;
    TCCR1A = (TCCR1A & 207)| (2<<4);  //COM1B
    //need WGM=14 for fast PWM with top=ICR1
    TCCR1A = (TCCR1A & 252)| (2<<0);  //WGM1=14=12|2
    TCCR1B = (TCCR1B & 231)| (12<<1); //WGM1=14
    TCCR1B = (TCCR1B & 248)| (1<<0);  //PS=1  
    ICR1 = PWM_TOP;                   //top
    OCR1B = n;                     //compare  
  }
}

void manualPWM9(int n){         //fast PWM on 9
  if(n<1){
    TCCR1A = (TCCR1A &  63)| (0<<6);  //COM1A, detach PWM
    digitalWrite(9,LOW);
  }else{
    if(n>PWM_MAX){n=PWM_MAX;}
    n=n-1;
    TCCR1A = (TCCR1A &  63)| (2<<6);  //COM1A
    //need WGM=14 for fast PWM with top=ICR1
    TCCR1A = (TCCR1A & 252)| (2<<0);  //WGM1=14=12|2
    TCCR1B = (TCCR1B & 231)| (12<<1); //WGM1=14
    TCCR1B = (TCCR1B & 248)| (1<<0);  //PS=1  
    ICR1 = PWM_TOP;                   //top
    OCR1A = n;                       //compare  
  }
}

void setBipolar(int entry){       //manages a pair of complementary outputs
  if(entry>0){
    manualPWM10(entry);
    manualPWM9(0);
  }else if(entry<0){
    manualPWM10(0);
    manualPWM9(abs(entry));        
  }else{
    manualPWM10(0);
    manualPWM9(0);        
  }  
}

