/* ****************************************************************************
 * File:      iodriver.h
 * Date:      21 Apr 2024
 * Author:    Andrew Levido
 *
 * Copyright 2024 Imbrius Pty Ltd - All Rights Reserved
 *
 *****************************************************************************/
#ifndef _iodriver_H
#define _iodriver_H

#ifdef __cplusplus
extern "C" {
#endif

/* Description -----------------------------------------------------------------
 * Implements a variety of I/O functions:
 *  - Reads Mode switches on initialisations and stores state wich is available 
 *    via API calls.
 *  - Polls and debounces ESTOP, RUN and REV inputs
 *  - Provides a mechanism for turning Red, Yellow and Green LEDS on, off or to 
 *    flash them at a defined rate for an arbitrary number of flashes.
 *  - Provides API calls to close or open SS_BYP relay and drive FLT_OUT and 
 *    ATS_OUT pins for user relay
 *  - Reads Speed, Ramp and Heatsink Temp analog inputs. Controls heatsink fan 
 *    based on temperature
 *
 *  - Requires IO_scan_ISR() be called every scan period (20ms is typical)
 */

/* Includes ------------------------------------------------------------------*/
#include "main.h"
#include <stdbool.h>

/* Defines -------------------------------------------------------------------*/

/* Typedefs & Enumerations ---------------------------------------------------*/

/* LED flash rate on and off times are equal and defined as below */
typedef enum {  IO_FLASH_FAST = 5,      //  5 x IO_SCAN_PERIOD = 100ms
                IO_FLASH_MED = 10,      // 10 x IO_SCAN_PERIOD = 200ms
                IO_FLASH_SLOW = 25      // 20 x IO_SCAN_PERIOD = 500ms
             } io_flash_rate_t;

/* Public Functions Declarations ---------------------------------------------*/
/** Initialise IO Driver
 *  Initialise driver memory and peripherals. Must be called before any other 
 *  functions. 
 */
void io_init(void);

/** Check 3-phase mode state
 *  Gets the state of the mode pin at the time io_init was last called. If it 
 *  was never called, it reads the state of the pin directly
 * 
 * @returns           True if 3Phase switch is closed, otherwise false
 */
bool io_is_mode_3ph(void);

/** Check pool pump mode state
 * Gets the state of the mode pin at the time io_init was last called. If it 
 * was never called, it reads the state of the pin directly
 * 
 * @returns           True if PPM switch closed. otherwise false
*/
bool io_is_mode_pool(void);

/** Check pool pump delay time
 * Gets the state of the mode pin at the time io_init was last called. If it 
 * was never called, it reads the state of the pin directly
 * 
 * @returns           True if PPT switch is closed, otherwise false
*/
bool io_is_mode_pool_long(void);

/** Check speed source
 * Gets the state of the mode pin at the time io_init was last called. If it 
 * was never called, it reads the state of the pin directly
 * 
 * @returns           True if external speed switch is closed, otherwise false
*/
bool io_is_mode_speed_ext(void);

/** Check relay mode
 *  Gets the state of the mode pin at the time io_init was last called. If it
 *  was never called, it reads the state of the pin directly
 * 
 * @returns           True if relay at speed switch is closed, otherwise false
*/
bool io_is_mode_relay_atspeed(void);

/** Check boost switch mode
 * Gets the state of the mode pin at the time io_init was last called. If it 
 * was never called, it reads the state of the pin directly
 * 
 * @returns           True if spare switch is closed, otherwise false
*/
bool io_is_mode_boost(void);

/** Get eStop debounced pin state
 * 
 * @returns           True if pin is high, false if low
 */
bool io_is_estop(void); 

/** Get Run debounced pin state
 * 
 * @returns           True if pin is high, false if low
 */
bool io_is_run(void); 

/** Get Reverse debounced pin state
 * 
 * @returns           True if pin is high, false if low
 */
bool io_is_reverse(void); 

/** Get Speed Demand
 * 
 * @returns           Speed demand 16-bit normalised 
 */
uint16_t io_get_s_demand(void); 

/** Get Ramp Demand
 * 
 * @returns           Ramp demand 16-bit normalised (reversed)
 */
uint16_t io_get_r_demand(void); 


/** Check Heatsink Temperature
 * 
 * Checks the heatsink temperature and controls the fan.
 * 
 * @returns true if overtemperature
 */
bool io_check_hs_temp(void);

/** Yellow LED on
 */
void io_ledy_on(void);

/** Yellow LED off
 */
void io_ledy_off(void);

/** Yellow LED flash
 * 
 * @param num         Number of flashes (1 - 255, 0 = flash forever)
 * @param rate        Enumerated rate type
 */
void io_ledy_flash(uint8_t num, io_flash_rate_t rate);

/** Green LED on
 */
void io_ledg_on(void);

/** Green LED off
 */
void io_ledg_off(void);

/** Green LED flash
 * 
 * @param num         Number of flashes (1- 255, 0 = flash forever)
 * @param rate        Enumerated rate type
 */
void io_ledg_flash(uint8_t num, io_flash_rate_t rate);

/** Red LED on
 */
void io_ledr_on(void);

/** Red LED off
 */
void io_ledr_off(void);

/** Red LED flash
 * 
 * @param num         Number of flashes (1- 255, 0 = flash forever)
 * @param rate        Enumerated rate type
 */
void io_ledr_flash(uint8_t num, io_flash_rate_t rate);

/** Set Soft Start Bypass Relay
 * 
 * @param close       True to close relay, false to open
 */
void io_set_ss_relay(bool close);

/** Set At Speed Output
 * 
 * @param close       True to close relay, false to open
 */
void io_set_ats_output(bool close);

/** Set Fault Output
 * 
 * @param close       True to close relay, false to open
 */
void io_set_flt_output(bool close);

/* Task Function -------------------------------------------------------------*/

/* Interrupt Service Routine Declarations ------------------------------------*/
/** Called from a timer interrupt every scan period */
void io_scan_ISR(void);

#ifdef __cplusplus
}
#endif

#endif

/* End iodriver */