/* ****************************************************************************
 * File:      state_machine.h
 * Date:      26 Apr 2024
 * Author:    Andrew Levido
 *
 * Copyright 2024 Imbrius Pty Ltd - All Rights Reserved
 *
 *****************************************************************************/
#ifndef _state_machine_H
#define _state_machine_H

#ifdef __cplusplus
extern "C" {
#endif

/* Description -----------------------------------------------------------------
 * Implements a very simple augmented state machine. Each state has three
 * function pointers: 
 * 
 * - a mandatory "tick" function that is called regularly when the state is 
 *   active. This function should implement the state logic and return a 
 *   pointer to the next state, or NULL if no state change occurs. Note 
 *   returning the current state from the tick function calls the state's entry
 *   function but does not change the state.
 * 
 * - an optional entry function which is called once when the state is 
 *   entered (ie a non-NULL state pointer is returned from the tick function).
 * 
 * - an optional exit function that is called once when the state is exited. 
 * 
 * The state objects and their functions must be statically allocated prior to
 * staring the state machine 
 * 
 * A timer mechanism is provided to states can implement time delays by calling
 * the timer functions from within the entry, tick and exit functions. The timer 
 * counts tick periods.
 * 
 * The state machine functionality is executed every time a tick event occurs
 * Ticks events occur at a rate determined when the state machine is started.
 * 
 * The user must call sm_1ms_clock() each millisecond from a suitable ISR. 
 * Typically this will be the 1ms system tick.
 */

/* Includes ------------------------------------------------------------------*/
#include <stdint.h>
#include <stdbool.h>
#include <ctype.h>

/* Defines -------------------------------------------------------------------*/

/* Typedefs & Enumerations ---------------------------------------------------*/
/* State type - user declares state objects of this type and provides the tick
 * function and (optionally) entry and exit functions */
typedef struct _sm_state {
  struct _sm_state * (*tick)(void);
  void (*entry)(void);
  void (*exit)(void);
} sm_state_t;

/* Public Functions Declarations ---------------------------------------------*/
/** Start State Machine 
 *  Starts the state machine processing. This function never returns.
 * 
 * @param initial_state     Pointer to SM initial state to enter
 * @param period            Tick period in ms
*/
void sm_start(sm_state_t * initial_state, uint32_t period);

/** Start a timer counting tick periods
 *  Sets the tick counter to a specific number. Counter decrements every tick
 *
 * @param delay_ticks       Number of ticks 
 */
void sm_start_timer(uint32_t delay_ticks);

/** Clear timer counting tick periods
 *  Sets tick counter to zero
 */
void sm_clear_timer(void);

/** Test if timer has reached zero
 *  This function will return true if the timer is not started, is cleared or
 *  has counted down to zero
 * 
 * returns                  True if counter = zero, else false
*/
bool sm_timer_expired(void);

/* Interrupt Service Routine Declarations ------------------------------------*/

/** Clock from which tick events are derived - should be called every 
 *  millisecond */
void sm_1ms_clock(void);

#ifdef __cplusplus
}
#endif

#endif

/* End state_machine */
