/* *****************************************************************************
 * File:      state_machine.c
 * Date:      26 Apr 2024
 * Author:    Andrew Levido
 *
 * Copyright 2024 Imbrius Pty Ltd - All Rights Reserved
 *
 ******************************************************************************/

/* Includes ------------------------------------------------------------------*/
#include "state_machine.h"

/* Defines -------------------------------------------------------------------*/

/* Typedefs & Enumerations ---------------------------------------------------*/

/* Global Variables ----------------------------------------------------------*/
static struct {
  sm_state_t * current_state;           // Pointer to current state object
  volatile bool tick_flag;              // Flag indicating a tick (set in ISR)
  volatile uint32_t tick_counter;       // Counter for timer (counts ticks)
  volatile uint32_t ms_counter;         // Counter for tick clock
  volatile uint32_t period;             // Tick period in ms (reload ms_counter)                   
} sm;

/* Private Functions ---------------------------------------------------------*/

/* Public Functions ----------------------------------------------------------*/
void sm_start(sm_state_t * initial_state, uint32_t period)
{
  sm_state_t * next_state;

  // Clear timer counter & set up tick period
  sm.period = period;
  sm.tick_counter = 0;

  // Set up Initial state & call entry function, if present
  sm.current_state = initial_state;
  if(sm.current_state->entry != NULL) { 
    sm.current_state->entry(); 
  }

  // clear tick flag and start monitoring for it to change
  sm.tick_flag = false;
  sm.ms_counter = period;
  while(1) {
    if(sm.tick_flag == true) {
      // Tick flag set, so clear it and increment timer counter if active
      sm.tick_flag = false;
      if(sm.tick_counter > 0) {
        sm.tick_counter--;
      }
      // Execute current state's tick function
      next_state = sm.current_state->tick();
      if(next_state != NULL) {
        // State change: exit current, switch, clear timer, enter next
        if(sm.current_state->exit != NULL) { 
          sm.current_state->exit(); 
        }
        sm.current_state = next_state;
        sm.tick_counter = 0;
        if(sm.current_state->entry != NULL) { 
          sm.current_state->entry(); 
        }
      }
    }
  }
}

void sm_1ms_clock(void)
{
  if(sm.ms_counter > 0) {
    sm.ms_counter--;
    if(sm.ms_counter == 0) {
      sm.ms_counter = sm.period;
      sm.tick_flag = true;
    }
  }
}

void sm_start_timer(uint32_t delay_ticks)
{
  sm.tick_counter = delay_ticks;
}

void sm_clear_timer(void)
{
  sm.tick_counter = 0;
}

bool sm_timer_expired(void)
{
   return(sm.tick_counter == 0 ? true : false);
}

/* Interrupt Service Routines ------------------------------------------------*/

/* End state_machine */