/* *****************************************************************************
 * File:      iodriver.c
 * Date:      21 Apr 2024
 * Author:    Andrew Levido
 *
 * Copyright 2024 Imbrius Pty Ltd - All Rights Reserved
 *
 ******************************************************************************/

/* Includes ------------------------------------------------------------------*/
#include "io_driver.h"

/* Defines -------------------------------------------------------------------*/
/* Switch de-bouncing parameters */
#define IO_DEBOUNCE_MASK      0b1111        // Check last four samples...
#define IO_DEBOUNCE_LO2HI     0b0111        // 1 low then 3 consecutive highs
#define IO_DEBOUNCE_HI2LO     0b1000        // 1 high then 3 consecutive lows

//#define IO_R_SCALE            1600          // (2^16 / 2^12) * 100


/* Table of temperature vs ADC reading (see '9048-Thermistor_Modelling'
   spreadsheet) for details 

      Temp     ADC Reading
      ------   ----------
         0  10895
         5  12551
        10  14386
        15  16410
        20  18600
        25  20953
        30  23441
        35  26037
        40  28706
        45  31430
        50  34110
        55  36800
        60  39338
        65  41793
        70  44128
        75  46249
        80  48278
        85  50084
        90  51681
        95  53198
       100  54504
       105  55675
*/
#define IO_T_OVERTEMP_ON      53198         // 95˚C
#define IO_T_OVERTEMP_OFF     44128         // 70˚C
#define IO_T_FAN_START        31430         // 45˚C
#define IO_T_FAN_STOP         28708         // 40˚C
 
/* Typedefs & Enumerations ---------------------------------------------------*/
/* Pin information */
typedef struct {
  GPIO_TypeDef * port;
  uint16_t pin;
  bool state;
  uint8_t debounce;
} pin_t;

/* LED information */
typedef struct {
  GPIO_TypeDef * port;
  uint16_t pin;
  bool state;
  bool flashing;
  uint8_t flashcount;
  uint16_t counter;
  uint16_t reload;
} led_t;

/* Global Variables ----------------------------------------------------------*/

// External referenced to ADC and DMA channel defined in main.c
extern ADC_HandleTypeDef hadc1;
extern DMA_HandleTypeDef hdma_adc1;

/*IO Driver state */
static struct {
  bool mode_3phase;               // State of 3-Phase mode pin on initialisation
  bool mode_poolpump;             // State of Pool mode pin on initialisation 
  bool mode_pool_long;            // State of Pool pump time pin on initialisation
  bool mode_speed_ext;            // State of Speed source pin on intialisation
  bool mode_relay_atspeed;        // State of Relay source pin on initialisation
  bool mode_boost;                // State of Boost mode pin on intialisation
  pin_t estop;                    // Estop pin data structure
  pin_t run;                      // Run pin data structure
  pin_t reverse;                  // Rev pin data structure
  led_t ledg;                     // Green LED data structure
  led_t ledy;                     // Yellow LED data structure
  led_t ledr;                     // Red LED data structure
  bool initialised;               // Flag indicating driver initialisation
  uint16_t adc_data[4];           // Raw ADC data (written by DMA)
  bool fan_on;                    //
  bool hs_overtemp;               //
} io;

/* Private Function Declarations  --------------------------------------------*/
/* Helper function to debounce pin & trigger appropriate callback on 
 * state change */
void io_pin_debounce(pin_t * pindata);

/* Helper function to manage LED state and flashing */
void io_led_manage(led_t * leddata);

/* Public Functions ----------------------------------------------------------*/
void io_init(void)
{
  // Set up pins
  io.estop.port = nESTOP_GPIO_Port;
  io.estop.pin = nESTOP_Pin;

  io.run.port = RUN_GPIO_Port;
  io.run.pin = RUN_Pin;

  io.reverse.port = REV_GPIO_Port;
  io.reverse.pin = REV_Pin;

  io.ledg.port = LED_GRN_GPIO_Port;
  io.ledg.pin = LED_GRN_Pin;

  io.ledy.port = LED_YEL_GPIO_Port;
  io.ledy.pin = LED_YEL_Pin; 

  io.ledr.port = LED_RED_GPIO_Port;
  io.ledr.pin = LED_RED_Pin; 

  // Read Mode pins
  io.mode_3phase = 
    HAL_GPIO_ReadPin(_3PH_SW_GPIO_Port, _3PH_SW_Pin) == GPIO_PIN_SET ? false : true;
  io.mode_poolpump = 
    HAL_GPIO_ReadPin(PPM_SW_GPIO_Port, PPM_SW_Pin) == GPIO_PIN_SET ? false : true;
  io.mode_pool_long =
    HAL_GPIO_ReadPin(PPT_SW_GPIO_Port, PPT_SW_Pin) == GPIO_PIN_SET ? false : true; 
  io.mode_speed_ext = 
    HAL_GPIO_ReadPin(SPD_SW_GPIO_Port, SPD_SW_Pin) == GPIO_PIN_SET ? false : true;
  io.mode_relay_atspeed = 
    HAL_GPIO_ReadPin(FLT_SW_GPIO_Port, FLT_SW_Pin) == GPIO_PIN_SET ? false : true;
  io.mode_boost =
    HAL_GPIO_ReadPin(BST_SW_GPIO_Port, BST_SW_Pin) == GPIO_PIN_SET ? false : true; 

  // Calibrate ADC
  HAL_ADC_Stop(&hadc1);
  HAL_ADCEx_Calibration_Start(&hadc1);

  // Initialise flags
  io.fan_on = false;
  io.hs_overtemp = false;
  
  // Flag driver is initialised
  io.initialised = true;
}

bool io_is_mode_3ph(void)
{
  if(io.initialised == false) {
    io.mode_3phase = 
    HAL_GPIO_ReadPin(_3PH_SW_GPIO_Port,_3PH_SW_Pin) == GPIO_PIN_SET ? false : true;
  }
  return(io.mode_3phase);
}

bool io_is_mode_pool(void)
{
  if(io.initialised == false) {
    io.mode_poolpump = 
    HAL_GPIO_ReadPin(PPM_SW_GPIO_Port, PPM_SW_Pin) == GPIO_PIN_SET ? false : true;
  }
  return(io.mode_poolpump);
}

bool io_is_mode_pool_long(void)
{
  if(io.initialised == false) {
    io.mode_pool_long = 
    HAL_GPIO_ReadPin(PPT_SW_GPIO_Port, PPT_SW_Pin) == GPIO_PIN_SET ? false : true; 
  }
  return(io.mode_pool_long);
}

bool io_is_mode_speed_ext(void)
{
  if(io.initialised == false) {
    io.mode_speed_ext = 
     HAL_GPIO_ReadPin(SPD_SW_GPIO_Port, SPD_SW_Pin) == GPIO_PIN_SET ? false : true;
  }
  return(io.mode_speed_ext);
}

bool io_is_mode_relay_atspeed(void)
{
  if(io.initialised == false) {
    io.mode_relay_atspeed = 
    HAL_GPIO_ReadPin(FLT_SW_GPIO_Port, FLT_SW_Pin) == GPIO_PIN_SET ? false : true;
  }
  return(io.mode_relay_atspeed);
}

bool io_is_mode_boost(void)
{
  if(io.initialised == false) {
    io.mode_boost = 
    HAL_GPIO_ReadPin(BST_SW_GPIO_Port, BST_SW_Pin) == GPIO_PIN_SET ? false : true; 
  }
  return(io.mode_boost);
}


bool io_is_estop(void)
{
  return(io.estop.state);
}

bool io_is_run(void)
{
  return(io.run.state);
}

bool io_is_reverse(void)
{
  return(io.reverse.state);
}

uint16_t io_get_s_demand(void)
{
  // Left shifted result already scaled
  if(io.mode_speed_ext == true) {
    return(io.adc_data[3] + 0x000f);
  }
  else {
    return(io.adc_data[0] + 0x000f);
  }
}

uint16_t io_get_r_demand(void)
{
  // left shifted data already scaled - only needs to be reversed
  return(0xffff - io.adc_data[1]);
}

bool io_check_hs_temp(void)
{
  if(io.hs_overtemp == false) {
    if(io.adc_data[2] >= IO_T_OVERTEMP_ON) {
      io.hs_overtemp = true;
    }
  }
  else {
    if(io.adc_data[2] <= IO_T_OVERTEMP_OFF) {
      io.hs_overtemp = false;
    }
  }

  if(io.fan_on == false) {
    if(io.adc_data[2] >= IO_T_FAN_START) {
      io.fan_on = true;
      HAL_GPIO_WritePin(FAN_GPIO_Port, FAN_Pin, GPIO_PIN_SET);
    }
  }
  else {
    if(io.adc_data[2] <= IO_T_FAN_STOP) {
      io.fan_on = false;
      HAL_GPIO_WritePin(FAN_GPIO_Port, FAN_Pin, GPIO_PIN_RESET);
    }
  }
  
  return(io.hs_overtemp);
}

void io_ledy_on(void)
{
  if(io.initialised == true) {
    io.ledy.state = 1;
    io.ledy.flashing = 0;
  }
}
void io_ledy_off(void)
{
  if(io.initialised == true) {
    io.ledy.state = 0;
    io.ledy.flashing = 0;
  }
}
void io_ledy_flash(uint8_t num, io_flash_rate_t rate)
{
  if(io.initialised == true) {
    io.ledy.state = 1;
    io.ledy.flashing = 1;
    io.ledy.flashcount = num;
    io.ledy.counter = rate;
    io.ledy.reload = rate;
  }
}

void io_ledg_on(void)
{
  if(io.initialised == true) {
    io.ledg.state = 1;
    io.ledg.flashing = 0;
  }
}
void io_ledg_off(void)
{
  if(io.initialised == true) {
    io.ledg.state = 0;
    io.ledg.flashing = 0;
  }
}

void io_ledg_flash(uint8_t num, io_flash_rate_t rate)
{
  if(io.initialised == true) {
    io.ledg.state = 1;
    io.ledg.flashing = 1;
    io.ledg.flashcount = num;
    io.ledg.counter = rate;
    io.ledg.reload = rate;
  }
}

void io_ledr_on(void)
{
  if(io.initialised == true) {
    io.ledr.state = 1;
    io.ledr.flashing = 0;
  }
}
void io_ledr_off(void)
{
  if(io.initialised == true) {
    io.ledr.state = 0;
    io.ledr.flashing = 0;
  }
}

void io_ledr_flash(uint8_t num, io_flash_rate_t rate)
{
  if(io.initialised == true) {
    io.ledr.state = 1;
    io.ledr.flashing = 1;
    io.ledr.flashcount = num;
    io.ledr.counter = rate;
    io.ledr.reload = rate;
  }
}

void io_set_ss_relay(bool close)
{
  HAL_GPIO_WritePin(SS_BYP_GPIO_Port, SS_BYP_Pin, close);
}

void io_set_ats_output(bool close)
{
  if(io.mode_relay_atspeed == true){
    HAL_GPIO_WritePin(OUT_RLY_GPIO_Port, OUT_RLY_Pin, close);
  }
}

void io_set_flt_output(bool close)
{
  if(io.mode_relay_atspeed == false){
    HAL_GPIO_WritePin(OUT_RLY_GPIO_Port, OUT_RLY_Pin, close);
  }
}

/* Private Function Definitions  ---------------------------------------------*/
void io_pin_debounce(pin_t * pindata)
{
  GPIO_PinState pinvalue;
  pinvalue = HAL_GPIO_ReadPin(pindata->port, pindata->pin);
  pindata->debounce = pindata->debounce << 1;
  pindata->debounce |= pinvalue == GPIO_PIN_SET ? 1 : 0;
  pindata->debounce &= IO_DEBOUNCE_MASK;

  if(pindata->debounce == IO_DEBOUNCE_LO2HI) {
    pindata->state = true;
  }
  else if(pindata->debounce == IO_DEBOUNCE_HI2LO) {
    pindata->state= false;
  }
}

void io_led_manage(led_t * leddata)
{
  if(leddata->flashing == true) {
    // flashing ...
    if(leddata->counter-- == 0) {
      // period counter expired
      leddata->counter = leddata->reload;
      if(leddata->state == true) {
        // Was on...
        leddata->state = false;
        if(leddata->flashcount > 0) {
          // and not infinite flashing...
          leddata->flashcount--;
          if(leddata->flashcount == 0) {
            // Last flash done...
            leddata->flashing = false;
          }
        }
      }
      else {
        // Was off...
        leddata->state = true;
      }
    }
  }
  HAL_GPIO_WritePin(leddata->port, leddata->pin, leddata->state);
}

/* Interrupt Service Routines and callbacks  ---------------------------------*/

/* IO Scan Timer ISR */
void io_scan_ISR(void)
{
  if(io.initialised != true) { return; }
  // Scan & debounce pins
  io_pin_debounce(&io.estop);
  io_pin_debounce(&io.run);
  io_pin_debounce(&io.reverse);

  // Manage LEDS
  io_led_manage(&io.ledg);
  io_led_manage(&io.ledy);
  io_led_manage(&io.ledr);
  
  // Read ADC
    HAL_ADC_Start_DMA(&hadc1, (uint32_t*)&io.adc_data, 4);
}

void HAL_ADC_ConvCpltCallback(ADC_HandleTypeDef * hadc)
{
  HAL_ADC_Stop_DMA(hadc);
}

/* End iodriver */