/* ****************************************************************************
 * File:      pwm.h
 * Date:      11 May 2024
 * Author:    Andrew Levido
 *
 * Copyright 2024 Imbrius Pty Ltd - All Rights Reserved
 *
 *****************************************************************************/
#ifndef _pwm_H
#define _pwm_H

#ifdef __cplusplus
extern "C" {
#endif

/* Description -----------------------------------------------------------------
 *
 * Driver for Motor-control PWM. Supports single-phase or three-phase operation.
 * Uses Timer 1 (Advanced Control Timer) and three channels, 1, 2 and 3 assigned
 * to U, V and W phases respectively. The external break input is used to detect 
 * faults, and one GPIO is used to control an active high enable signal.
 * 
 * Timer1 is set up for complimentary centre-aligned PWM with period and 
 * dead-time defined in  main.h (IMSC_PWM_RELOAD & IMSC_PWM_DEADTIME Fclk 
 * cycles). The break input has a filter defined by IMSC_PWM_FILT in main.h
 * 
 * The PWM channels are set up polarity & idle state to match the IGBT driver.
 */

/* Includes ------------------------------------------------------------------*/
#include "main.h"
#include <stdbool.h>

/* Defines -------------------------------------------------------------------*/
#define PWM_FMAX            50.0            // Hz

/* Typedefs & Enumerations ---------------------------------------------------*/

/* Public Functions Declarations ---------------------------------------------*/
/** Initialise PWM Driver
 * 
 * Starts the timer and enables 2 or 3 PWM channels but outputs are not enabled.
 * Must be called once before any other pwm function
 * 
 * @param three_phase             True for 3-phase operation, false for 1-phase
 * @param boost                   True for voltage boost
 */
void pwm_init(bool three_phase, bool boost);

/** Start PWM 
 * 
 * Initialises PWM waveform generation to single phase (channels U and V out of
 * phase by 180˚), three phase forward (U = 0˚, V = 120˚, W = 240˚) or three
 * phase reverse (U = 0˚, W = 120˚, V = 240˚). Enables the timer outputs and 
 * asserts the enable output
 */
void pwm_enable(void);

/** Stop PWM
 * 
 * Disables PWM outputs and deasserts the enable output if not in fault state
 */
void pwm_disable(void);

/** Sets the 3-phase rotation direction (ignored if 1=phase)
 * 
 * @param dir                     True for U-V-W, false for U-W-V
*/
void pwm_set_dir(bool dir);

/** Set the Motor speed
 * 
 * Sets the motor frequency such that full speed (2^16 - 1) is PWM_FMAX
 * Sets the motor voltage according to the characteristic defined above
 * 
 * @param speed                 Motor speed in normalised uint_16
*/
void pwm_set_speed(uint16_t speed);

/** Query if PWM Fault has occurred
 * 
 * @returns                     True if fault active, false if not
*/
bool pwm_is_fault(void);

/** Clear PWM Fault 
 * 
 * Clears PWM fault flag, only if PWM is disabled. 
*/
void pwm_clear_fault(void);

/* Interrupt Service Routine Declarations ------------------------------------*/

/* Timer 1 break ISR 
*/
void pwm_break_ISR(void);

/** Timer 1 update ISR 
*/
void pwm_update_ISR(void);

#ifdef __cplusplus
}
#endif

#endif

/* End pwm */