/* *****************************************************************************
 * File:      pwm.c
 * Date:      11 May 2024
 * Author:    Andrew Levido
 *
 * Copyright 2024 Imbrius Pty Ltd - All Rights Reserved
 *
 ******************************************************************************/

/* Includes ------------------------------------------------------------------*/
#include "pwm.h"
#include "debug.h"

/* Defines -------------------------------------------------------------------*/
/* PWM accumulator starting positions */
#define PWM_ACCU_0          0UL             // 2^32 * 0
#define PWM_ACCU_120        1431655765UL    // 2^32 * 1/3
#define PWM_ACCU_180        2147483448UL    // 2^32 * 1/2
#define PWM_ACCU_240        2863311531UL    // 2^32 * 2/3

/* Frequency scale factor 
 * - magic number = 2^17 * (IMSC_PWM_RELOAD + 1) * Tclk 
 *        4.19343 = 2^17 * 2048 * 15.625e-9
 * - the 0.5 is for correctly rounding float to integer
 * */
#define PWM_FSCALE          ((uint32_t)((PWM_FMAX * 4.19343) + 0.5))

/* For easy calculation, 100% speed is denoted by 65535 */
#define PWM_FULL_SCALE      65535

/* Boost mode minimum modulation depth */
#define PWM_MIN_MOD_DEPTH   (0.075 * PWM_FULL_SCALE)    // 7.5%

/* Typedefs & Enumerations ---------------------------------------------------*/
typedef union {
  int32_t accu;
  struct {
    uint32_t remainder : 24;
    uint32_t index     : 8;
  } __attribute__((packed));
} accumulator_t;

/* Global Variables ----------------------------------------------------------*/
extern TIM_HandleTypeDef htim1; 

static struct {
  accumulator_t u_phase;
  accumulator_t v_phase;
  accumulator_t w_phase;
  bool running;
  bool three_phase;
  bool initalised;
  bool rotation;
  bool fault;
  bool boost;
  uint16_t modulation_depth;
  uint32_t increment;
} pwm;

const int16_t single_phase_table[] = {									
0x0000,	0x0324,	0x0648,	0x096B,	0x0C8C,	0x0FAB,	0x12C8,	0x15E2,	0x18F9,	0x1C0C,
0x1F1A,	0x2224,	0x2528,	0x2827,	0x2B1F,	0x2E11,	0x30FC,	0x33DF,	0x36BA,	0x398D,
0x3C57,	0x3F17,	0x41CE,	0x447B,	0x471D,	0x49B4,	0x4C40,	0x4EC0,	0x5134,	0x539B,
0x55F6,	0x5843,	0x5A82,	0x5CB4,	0x5ED7,	0x60EC,	0x62F2,	0x64E9,	0x66D0,	0x68A7,
0x6A6E,	0x6C24,	0x6DCA,	0x6F5F,	0x70E3,	0x7255,	0x73B6,	0x7505,	0x7642,	0x776C,
0x7885,	0x798A,	0x7A7D,	0x7B5D,	0x7C2A,	0x7CE4,	0x7D8A,	0x7E1E,	0x7E9D,	0x7F0A,
0x7F62,	0x7FA7,	0x7FD9,	0x7FF6,	0x7FFE,	0x7FF6,	0x7FD9,	0x7FA7,	0x7F62,	0x7F0A,
0x7E9D,	0x7E1E,	0x7D8A,	0x7CE4,	0x7C2A,	0x7B5D,	0x7A7D,	0x798A,	0x7885,	0x776C,
0x7642,	0x7505,	0x73B6,	0x7255,	0x70E3,	0x6F5F,	0x6DCA,	0x6C24,	0x6A6E,	0x68A7,
0x66D0,	0x64E9,	0x62F2,	0x60EC,	0x5ED7,	0x5CB4,	0x5A82,	0x5843,	0x55F6,	0x539B,
0x5134,	0x4EC0,	0x4C40,	0x49B4,	0x471D,	0x447B,	0x41CE,	0x3F17,	0x3C57,	0x398D,
0x36BA,	0x33DF,	0x30FC,	0x2E11,	0x2B1F,	0x2827,	0x2528,	0x2224,	0x1F1A,	0x1C0C,
0x18F9,	0x15E2,	0x12C8,	0x0FAB,	0x0C8C,	0x096B,	0x0648,	0x0324,	0x0000,	0xFCDC,
0xF9B8,	0xF695,	0xF374,	0xF055,	0xED38,	0xEA1E,	0xE707,	0xE3F4,	0xE0E6,	0xDDDC,
0xDAD8,	0xD7D9,	0xD4E1,	0xD1EF,	0xCF04,	0xCC21,	0xC946,	0xC673,	0xC3A9,	0xC0E9,
0xBE32,	0xBB85,	0xB8E3,	0xB64C,	0xB3C0,	0xB140,	0xAECC,	0xAC65,	0xAA0A,	0xA7BD,
0xA57E,	0xA34C,	0xA129,	0x9F14,	0x9D0E,	0x9B17,	0x9930,	0x9759,	0x9592,	0x93DC,
0x9236,	0x90A1,	0x8F1D,	0x8DAB,	0x8C4A,	0x8AFB,	0x89BE,	0x8894,	0x877B,	0x8676,
0x8583,	0x84A3,	0x83D6,	0x831C,	0x8276,	0x81E2,	0x8163,	0x80F6,	0x809E,	0x8059,
0x8027,	0x800A,	0x8000,	0x800A,	0x8027,	0x8059,	0x809E,	0x80F6,	0x8163,	0x81E2,
0x8276,	0x831C,	0x83D6,	0x84A3,	0x8583,	0x8676,	0x877B,	0x8894,	0x89BE,	0x8AFB,
0x8C4A,	0x8DAB,	0x8F1D,	0x90A1,	0x9236,	0x93DC,	0x9592,	0x9759,	0x9930,	0x9B17,
0x9D0E,	0x9F14,	0xA129,	0xA34C,	0xA57E,	0xA7BD,	0xAA0A,	0xAC65,	0xAECC,	0xB140,
0xB3C0,	0xB64C,	0xB8E3,	0xBB85,	0xBE32,	0xC0E9,	0xC3A9,	0xC673,	0xC946,	0xCC21,
0xCF04,	0xD1EF,	0xD4E1,	0xD7D9,	0xDAD8,	0xDDDC,	0xE0E6,	0xE3F4,	0xE707,	0xEA1E,
0xED38,	0xF055,	0xF374,	0xF695,	0xF9B8,	0xFCDC,				
};																	
									
const int16_t three_phase_table[] = {									
0x0000,	0x0552,	0x0AA1,	0x0FEA,	0x152A,	0x1A5F,	0x1F85,	0x249B,	0x299D,	0x2E89,
0x335C,	0x3814,	0x3CAE,	0x412A,	0x4584,	0x49BB,	0x4DCE,	0x51BA,	0x557E,	0x591A,
0x5C8C,	0x5FD4,	0x62F0,	0x65E1,	0x68A6,	0x6B3F,	0x6DAB,	0x6FED,	0x7203,	0x73EE,
0x75B0,	0x7748,	0x78B9,	0x7A03,	0x7B27,	0x7C28,	0x7D06,	0x7DC4,	0x7E63,	0x7EE4,
0x7F4B,	0x7F98,	0x7FCE,	0x7FEF,	0x7FFD,	0x7FFA,	0x7FE9,	0x7FCA,	0x7FA1,	0x7F6E,
0x7F35,	0x7EF7,	0x7EB6,	0x7E73,	0x7E30,	0x7DEF,	0x7DB1,	0x7D77,	0x7D43,	0x7D14,
0x7CED,	0x7CCE,	0x7CB8,	0x7CAA,	0x7CA4,	0x7CAA,	0x7CB8,	0x7CCE,	0x7CED,	0x7D14,
0x7D43,	0x7D77,	0x7DB1,	0x7DEF,	0x7E30,	0x7E73,	0x7EB6,	0x7EF7,	0x7F35,	0x7F6E,
0x7642,	0x7505,	0x73B6,	0x7255,	0x70E3,	0x6F5F,	0x6DCA,	0x6C24,	0x6A6E,	0x68A7,
0x7E63,	0x7DC4,	0x7D06,	0x7C28,	0x7B27,	0x7A03,	0x78B9,	0x7748,	0x75B0,	0x73EE,
0x7203,	0x6FED,	0x6DAB,	0x6B3F,	0x68A6,	0x65E1,	0x62F0,	0x5FD4,	0x5C8C,	0x591A,
0x557E,	0x51BA,	0x4DCE,	0x49BB,	0x4584,	0x412A,	0x3CAE,	0x3814,	0x335C,	0x2E89,
0x299D,	0x249B,	0x1F85,	0x1A5F,	0x152A,	0x0FEA,	0x0AA1,	0x0552,	0x0000,	0xFAAE,
0xF55F,	0xF016,	0xEAD6,	0xE5A1,	0xE07B,	0xDB65,	0xD663,	0xD177,	0xCCA4,	0xC7EC,
0xC352,	0xBED6,	0xBA7C,	0xB645,	0xB232,	0xAE46,	0xAA82,	0xA6E6,	0xA374,	0xA02C,
0x9D10,	0x9A1F,	0x975A,	0x94C1,	0x9255,	0x9013,	0x8DFD,	0x8C12,	0x8A50,	0x88B8,
0x8747,	0x85FD,	0x84D9,	0x83D8,	0x82FA,	0x823C,	0x819D,	0x811C,	0x80B5,	0x8068,
0x8032,	0x8011,	0x8003,	0x8006,	0x8017,	0x8036,	0x805F,	0x8092,	0x80CB,	0x8109,
0x814A,	0x818D,	0x81D0,	0x8211,	0x824F,	0x8289,	0x82BD,	0x82EC,	0x8313,	0x8332,
0x8348,	0x8356,	0x835B,	0x8356,	0x8348,	0x8332,	0x8313,	0x82EC,	0x82BD,	0x8289,
0x824F,	0x8211,	0x81D0,	0x818D,	0x814A,	0x8109,	0x80CB,	0x8092,	0x805F,	0x8036,
0x8017,	0x8006,	0x8003,	0x8011,	0x8032,	0x8068,	0x80B5,	0x811C,	0x819D,	0x823C,
0x82FA,	0x83D8,	0x84D9,	0x85FD,	0x8747,	0x88B8,	0x8A50,	0x8C12,	0x8DFD,	0x9013,
0x9255,	0x94C1,	0x975A,	0x9A1F,	0x9D10,	0xA02C,	0xA374,	0xA6E6,	0xAA82,	0xAE46,
0xB232,	0xB645,	0xBA7C,	0xBED6,	0xC352,	0xC7EC,	0xCCA4,	0xD177,	0xD663,	0xDB65,
0xE07B,	0xE5A1,	0xEAD6,	0xF016,	0xF55F,	0xFAAE,				
};																								

/* Private Functions ---------------------------------------------------------*/

/* Public Functions ----------------------------------------------------------*/
void pwm_init(bool three_phase, bool boost)
{
  GPIO_InitTypeDef GPIO_InitStruct = {0};

  /* Clear enable pin just in case... */
  HAL_GPIO_WritePin(PWM_EN_GPIO_Port, PWM_EN_Pin, GPIO_PIN_RESET);

  /* Set phase mode and boost flags */
  if(three_phase == true) {
    dbg_msg("Initialise 3-Phase PWM\n");
    pwm.three_phase = true;
  }
  else {
    dbg_msg("Initialise 1-Phase PWM\n");
    pwm.three_phase = false;
  }
  pwm.boost = boost;

  /* Start the PWM timer, disable all channels & initilase compare registers to
    zero */
  HAL_TIM_Base_Start_IT(&htim1);
  __HAL_TIM_MOE_DISABLE_UNCONDITIONALLY(&htim1);

  __HAL_TIM_SET_COMPARE(&htim1, TIM_CHANNEL_1, 1024);
  __HAL_TIM_SET_COMPARE(&htim1, TIM_CHANNEL_2, 1024);
  __HAL_TIM_SET_COMPARE(&htim1, TIM_CHANNEL_3, 1024);

  /* Enable all three PWM channels if three-phase mode */
  if(pwm.three_phase == true) {
    HAL_TIM_PWM_Start(&htim1, TIM_CHANNEL_1);
    HAL_TIMEx_PWMN_Start(&htim1,TIM_CHANNEL_1);
    HAL_TIM_PWM_Start(&htim1, TIM_CHANNEL_2);
    HAL_TIMEx_PWMN_Start(&htim1,TIM_CHANNEL_2);
    HAL_TIM_PWM_Start(&htim1, TIM_CHANNEL_3); 
    HAL_TIMEx_PWMN_Start(&htim1,TIM_CHANNEL_3);
  }
  else {
    /* In three-phase mode, set W channel pins as outputs and clear them, then
       enable PWM on channels 1 and 2 */
    GPIO_InitStruct.Pin = PWM_WH_Pin;
    GPIO_InitStruct.Mode = GPIO_MODE_OUTPUT_PP;
    GPIO_InitStruct.Pull = GPIO_NOPULL;
    GPIO_InitStruct.Speed = GPIO_SPEED_FREQ_LOW;
    HAL_GPIO_Init(GPIOA, &GPIO_InitStruct);
    GPIO_InitStruct.Pin = PWM_WL_Pin;
    GPIO_InitStruct.Mode = GPIO_MODE_OUTPUT_PP;
    GPIO_InitStruct.Pull = GPIO_NOPULL;
    GPIO_InitStruct.Speed = GPIO_SPEED_FREQ_LOW;
    HAL_GPIO_Init(GPIOB, &GPIO_InitStruct);
    HAL_GPIO_WritePin(PWM_WL_GPIO_Port, PWM_WL_Pin, GPIO_PIN_RESET);
    HAL_GPIO_WritePin(PWM_WH_GPIO_Port, PWM_WH_Pin, GPIO_PIN_RESET);

    HAL_TIM_PWM_Start(&htim1, TIM_CHANNEL_1);
    HAL_TIMEx_PWMN_Start(&htim1,TIM_CHANNEL_1);
    HAL_TIM_PWM_Start(&htim1, TIM_CHANNEL_2);
    HAL_TIMEx_PWMN_Start(&htim1,TIM_CHANNEL_2);
  }
  pwm.initalised = true;
}

void pwm_enable(void)
{
  if(pwm.initalised != true) { return; }
  if(pwm.fault == true) { return; }

  /* Don't re-initialise if already running! This will cause the motor to
     skip horribly */
  if(pwm.running == true) { return; }

  // Initialise table indices
  if(pwm.three_phase == true) {
    if(pwm.rotation == false){
      pwm.u_phase.accu = PWM_ACCU_0;
      pwm.v_phase.accu = PWM_ACCU_120;
      pwm.w_phase.accu = PWM_ACCU_240;
    }
    else {
      pwm.u_phase.accu = PWM_ACCU_0;
      pwm.v_phase.accu = PWM_ACCU_240;
      pwm.w_phase.accu = PWM_ACCU_120;
    }
  }
  else {
    pwm.u_phase.accu = PWM_ACCU_0;
    pwm.v_phase.accu = PWM_ACCU_180;
    pwm.w_phase.accu = PWM_ACCU_0;
  }
  
  // Enable timer outputs and enable pin
  __HAL_TIM_MOE_ENABLE(&htim1);  
  pwm.running = true;
  HAL_GPIO_WritePin(PWM_EN_GPIO_Port, PWM_EN_Pin, GPIO_PIN_SET);

  dbg_msg("Enable PWM\n");
}

void pwm_disable(void)
{
  /* Clear enable pin and disable PWM channels */
  HAL_GPIO_WritePin(PWM_EN_GPIO_Port, PWM_EN_Pin, GPIO_PIN_RESET);
   __HAL_TIM_MOE_DISABLE_UNCONDITIONALLY(&htim1);

  pwm.running = false;

  dbg_msg("Disable PWM\n");
}

void pwm_set_dir(bool dir)
{
  if(pwm.initalised != true) { return; }
  if(pwm.three_phase == false) { return; }

  pwm.rotation = dir;
  dbg_printf("Reverse: %d\n", dir);
}

void pwm_set_speed(uint16_t speed)
{
  if(pwm.initalised != true) { return; }

  /* Calculate amount to increment accumulators (frequency) */
  pwm.increment = speed * PWM_FSCALE;

  /* Calculate modulation depth (voltage) */
  if((speed < PWM_MIN_MOD_DEPTH) && (pwm.boost == true)) {
    pwm.modulation_depth = PWM_MIN_MOD_DEPTH;
  }
  else {
    pwm.modulation_depth = speed;
  }
  
  dbg_printf("L: %d ", pwm.increment);
  dbg_printf("M: %u\n", (uint16_t)pwm.modulation_depth);
}

bool pwm_is_fault(void)
{
  return(pwm.fault);
}

void pwm_clear_fault(void)
{
  /* clear fault if not running and fault pin deasserted */
  if(pwm.initalised != true) { return; }
  if(pwm.running == true) { return; }
  if(HAL_GPIO_ReadPin(HOT_FLT_GPIO_Port, HOT_FLT_Pin) == GPIO_PIN_SET) {
    pwm.fault = false;
  }
}
/* Interrupt Service Routines ------------------------------------------------*/
void pwm_break_ISR(void)
{
  HAL_GPIO_WritePin(PWM_EN_GPIO_Port, PWM_EN_Pin, GPIO_PIN_RESET);
   __HAL_TIM_MOE_DISABLE_UNCONDITIONALLY(&htim1);

  pwm.fault = true;
  pwm.running = false;
}

void pwm_update_ISR(void)
{
  int32_t u, v, w;

  /* Skip if not actually running */
  if(pwm.running != true) { return; }

  if(pwm.three_phase == true) {
     /* Calculate & update duty cycle */
    u = three_phase_table[pwm.u_phase.index] * pwm.modulation_depth;
    u = (u >> 21) + 1024;
    v = three_phase_table[pwm.v_phase.index] * pwm.modulation_depth;
    v = (v >> 21) + 1024;
    w = three_phase_table[pwm.w_phase.index] * pwm.modulation_depth;
    w = (w >> 21) + 1024;
    __HAL_TIM_SET_COMPARE(&htim1, TIM_CHANNEL_1, u);
    __HAL_TIM_SET_COMPARE(&htim1, TIM_CHANNEL_2, v);
    __HAL_TIM_SET_COMPARE(&htim1, TIM_CHANNEL_3, w);

    /* Increment accumulators */
    pwm.u_phase.accu += pwm.increment;
    pwm.v_phase.accu += pwm.increment;
    pwm.w_phase.accu += pwm.increment;
  }
  else {
     /* Calculate & update duty cycle */
    u = single_phase_table[pwm.u_phase.index] * pwm.modulation_depth;
    u = (u >> 21) + 1024;
    v = single_phase_table[pwm.v_phase.index] * pwm.modulation_depth;
    v = (v >> 21) + 1024;
    __HAL_TIM_SET_COMPARE(&htim1, TIM_CHANNEL_1, u);
    __HAL_TIM_SET_COMPARE(&htim1, TIM_CHANNEL_2, v);

    /* Increment accumulators */
    pwm.u_phase.accu += pwm.increment;
    pwm.v_phase.accu += pwm.increment;
  }
}

/* End pwm */
