/* *****************************************************************************
 * File:      imsc.c
 * Date:      23 Apr 2024
 * Author:    Andrew Levido
 *
 * Copyright 2024 Imbrius Pty Ltd - All Rights Reserved
 *
 ******************************************************************************/

/* Includes ------------------------------------------------------------------*/
#include "imsc.h"
#include "io_driver.h"
#include "state_machine.h"
#include "pwm.h"
#include "debug.h"

/* Defines -------------------------------------------------------------------*/

/* State machine tick period (ms) */
#define IMSC_TICK_PERIOD          100   

/* IMSC timers in tick periods (milleconds / IMSC_TICK_PERIOD )*/        
#define IMSC_SS_BYPASS_DELAY      (2000 / IMSC_TICK_PERIOD)   // 2 seconds
#define IMSC_IDLE_DWELL_TIME      (1000 / IMSC_TICK_PERIOD)   // 1 second
#define IMSC_PPUMP_TIME           (30000 / IMSC_TICK_PERIOD)  // 30 seconds
#define IMSC_PPUMP_TIME_LONG      (300000 / IMSC_TICK_PERIOD)  // 300 seconds

/* Full scale */
#define IMSC_FULL_SCALE           65535

/* Minimum speed (100 full scale)  */
#define IMSC_MIN_SPEED            (0.01 * IMSC_FULL_SCALE )    // 1%

/* Speed hysteresis */
#define IMSC_SPEED_MARGIN         (0.005 * IMSC_FULL_SCALE)    // 0.5%
#define IMSC_SPEED_MARGIN_ATS     (0.010 * IMSC_FULL_SCALE)    // 1.0%

/* Ramp Time = (65535 * ∆t) / ((N >> SHIFT) + OFFS) 
 *
 *  Tmin = 6553.5 / 2157 = 3.04 s
 *  Tmax = 6553.5 / 109  = 60.12
 */
#define IMSC_RAMP_SHIFT             5         
#define IMSC_RAMP_OFFSET            109     
      
/* Pool pump speed */
#define IMSC_PP_SPEED               (IMSC_FULL_SCALE )         // 100%

/* Typedefs & Enumerations ---------------------------------------------------*/

/* Private Function Prototypes  ----------------------------------------------*/
/* Initialise state */
void st_initialise_entry(void);
void st_initialise_exit(void);
sm_state_t * st_initialise_tick(void);

/* Fault state */
void st_fault_entry(void);
void st_fault_exit(void);
sm_state_t * st_fault_tick(void);

/* Idle state */
void st_idle_entry(void);
void st_idle_exit(void);
sm_state_t * st_idle_tick(void);

/* Pool Pump state */
void st_ppump_entry(void);
void st_ppump_exit(void);
sm_state_t * st_ppump_tick(void);

/* Ramp up/down state */
void st_ramp_entry(void);
void st_ramp_exit(void);
sm_state_t * st_ramp_tick(void);

/* Up to speed state */
void st_atspeed_entry(void);
void st_atspeed_exit(void);
sm_state_t * st_atspeed_tick(void);

/* Global Variables ----------------------------------------------------------*/
/* IO Scan Timer */
extern TIM_HandleTypeDef htim17;

/* Create the state objects */

/* Initial State - entered on power-up*/
sm_state_t st_initialise = {
  .entry = st_initialise_entry,
  .exit = st_initialise_exit,
  .tick = st_initialise_tick,
};

/* Fault state - IGBT bridge trip, overvoltage or heatsink temperature */
sm_state_t st_fault = {
  .entry = st_fault_entry,
  .exit = st_fault_exit,
  .tick = st_fault_tick,
};

/* Poolpump state - Run at 100% for a period */
sm_state_t st_ppump = {
  .entry = st_ppump_entry,
  .exit = st_ppump_exit,
  .tick = st_ppump_tick,
};

/* Idle state  - Waiting for run and speed signals */
sm_state_t st_idle = {
  .entry = st_idle_entry,
  .exit = st_idle_exit,
  .tick = st_idle_tick,
};

/* Ramp State - Ramping to/from set speed*/
sm_state_t st_ramp = {
  .entry = st_ramp_entry,
  .exit = st_ramp_exit,
  .tick = st_ramp_tick,
};

/* At Speed State -  At set speed */
sm_state_t st_atspeed = {
  .entry = st_atspeed_entry,
  .exit = st_atspeed_exit,
  .tick = st_atspeed_tick,
};

/* Application data - data describing application state */
static struct {
  uint16_t speed_now;           // Current speed (normalised)
  bool dir_now;                 // Current phase rotation (3Ph) 
  bool igbt_fault;              // Fault cause was IGBT trip or overvoltage
  bool temperature_fault;       // Fault cause was heatsink overtemperature
  bool estop_cyle;              // Estop opened in fault state    
} imsc;

/* Private Functions ---------------------------------------------------------*/


/* Helpers */
bool imsc_is_fault(void)
{
 
  // Check for a IGBT trip or Overvoltage (from PWM driver)
  imsc.igbt_fault = pwm_is_fault();

  // Check for overtemperature fault 
  imsc.temperature_fault = io_check_hs_temp();

  // If either fault return true, esle false
  if((imsc.temperature_fault == true)||(imsc.igbt_fault == true)) {
    return(true);
  }
  return(false);
}

uint16_t imsc_speed_req(bool idle) 
{
  uint16_t speed_req;

  /* set speed request to user speed demand */
  speed_req = io_get_s_demand();

  /* If estop or run are open set it to zero */
  if((io_is_estop() == false) || io_is_run() == false) { speed_req = 0; }
  
  /* Or if running in 3ph mode & change in direction requested set it to zero */
  if((idle != true) && (io_is_mode_3ph() == true)) {
    if(imsc.dir_now != io_is_reverse()) { speed_req = 0; }
  }
  return(speed_req);
}

uint16_t imsc_ramp_delta(void)
{
  return ((io_get_r_demand() >> IMSC_RAMP_SHIFT) + IMSC_RAMP_OFFSET);
}

/* Public Functions ----------------------------------------------------------*/
void imsc_start(void)
{
  /* Start the timer for I/O timer driver */
  HAL_TIM_Base_Start_IT(&htim17);
  
  /* Start the state machine - function never returns */
  sm_start(&st_initialise, IMSC_TICK_PERIOD);
}

/* Initialise State (initial state) ............................................
 *
 *  Entry Function
 *   - initialise some flags
 *   - start IO driver (read mode switches)
 *   - start PWM (1-phase or 3-phase)
 *   - flash R, Y and G LEDs twice fast
 *   - start soft start bypass relay timer
 * 
 *  Tick Function
 *   - if timer expired 
 *      - If 1-phase + pool mode ==> Ppump state
 *      - else ==> Idle state
 *   - else ==> No state change
 * 
 *  Exit Function
 *   -  close ss bypass relay
 */
 void st_initialise_entry(void)
{
  dbg_msg("Initialise: Entry\n");

  imsc.igbt_fault = false;
  imsc.temperature_fault = false;
  imsc.estop_cyle = false;

  io_init();

  pwm_init(io_is_mode_3ph(), io_is_mode_boost());

  /* Start soft start bypass timer */
  sm_start_timer(IMSC_SS_BYPASS_DELAY);

  /* Flash LEDs to indicate reset */
  io_ledg_flash(2, IO_FLASH_FAST);
  io_ledy_flash(2, IO_FLASH_FAST);
  io_ledr_flash(2, IO_FLASH_FAST);
}

void st_initialise_exit(void)
{
  dbg_msg("Initialise: Exit\n");

  /* Close soft start bypass relay */
  io_set_ss_relay(true);
}

sm_state_t * st_initialise_tick(void)
{
  /* Check if bypass timer has expired. If single phase & pool mode --> Ppump
    else --> Idle*/
  if(sm_timer_expired()) {
    if((io_is_mode_pool() == true) && (io_is_mode_3ph() == false)) {
      return(&st_ppump);
    }
    else { return(&st_idle); }
  }
  else { return(NULL); }
}

/* Fault State  ...............................................................
 *
 *  Entry Function
 *   - shut down PWM and zero out speed
 *   - clear stop cycle flag
 *   - assert fault output
 *   - set Red LED on and Yellow LED if temperature
 * 
 *  Tick Function
 *   - if faults have resolved and estop is cycled  ==> Idle state
 *   - else stay in fault state
 * 
 *  Exit Function
 *   - turn off Red & Yellow LEDs
 *   - deassert fault output
 */
void st_fault_entry(void)
{
  dbg_msg( "Fault: Entry\n");

  pwm_disable();
  imsc.speed_now = 0;
  imsc.estop_cyle = false;
  io_set_flt_output(true);
  io_ledr_on();
  if(imsc.temperature_fault == true) {
    io_ledy_on();
  }
}

void st_fault_exit(void)
{
  dbg_msg("Fault: Exit\n");

  /* Red LED off and clear fault output*/
  io_ledr_off();
  io_ledy_off();
  io_set_flt_output(false);
}

sm_state_t * st_fault_tick(void)
{
  /* if no fault & estop cycled --> Idle */
  pwm_clear_fault();
  if(imsc_is_fault() == false) {
    if(imsc.estop_cyle == false) {
      if(io_is_estop() == false) {
        imsc.estop_cyle = true;
      }
    }
    else {
      if(io_is_estop() == true) {
        return(&st_idle);
      }
    }
  }
  return(NULL); 
}

/* Pool Pump State ............................................................
 *
 *  Entry Function
 *   - flash Green LED slowly
 *   - start pool pump timer
 *   - set speed to zero & start PWM
 * 
 *  Tick Function
 *   - if fault ==> fault state
 *   - if Estop open ==> Idle state
 *   - if run open 
 *      - if speed now > min speed ==> ramp
 *      - else ==> idle
 *   - if timer expired ==> ramp state
 *   - if speed now < pool pump speed, increment speed
 *   - stay in pool pump state
 * 
 *  Exit Function
 *   - turn Green LED off
 */
 void st_ppump_entry(void)
{
  dbg_msg("Pool Pump: Entry\n");
  
  io_ledg_flash(0, IO_FLASH_SLOW);
  if(io_is_mode_pool_long() == true) {
    sm_start_timer(IMSC_PPUMP_TIME_LONG);
  }
  else {
    sm_start_timer(IMSC_PPUMP_TIME);
  }
  imsc.speed_now = 0;
  pwm_enable();
}

void st_ppump_exit(void)
{
  dbg_msg("Pool Pump: Exit\n");

  io_ledg_off();
}

sm_state_t * st_ppump_tick(void)
{
  /* Check for faults --> Fault state */
  if(imsc_is_fault() == true) {
    return(&st_fault);
  }
  
  /* If ESTOP is open --> Idle */
  if(io_is_estop() == false) {
    return(&st_idle);
  }

  // If RUN opened --> ramp state
  if(io_is_run() == false) {
    if(imsc.speed_now > IMSC_MIN_SPEED) {
      return(&st_ramp);
    }
    else { return(&st_idle); }
  }

  // if timed out --> ramp state
  if(sm_timer_expired() == true) {
    return(&st_ramp);
  }

  // if not at fmax, increment
  if(imsc.speed_now < IMSC_PP_SPEED - IMSC_SPEED_MARGIN) {
    imsc.speed_now += imsc_ramp_delta();
    if(imsc.speed_now > IMSC_PP_SPEED) { imsc.speed_now = IMSC_PP_SPEED; }
    pwm_set_speed(imsc.speed_now);
    return(NULL);
  }
  // otherwise dwell
  return(NULL); 

}

/* Idle State  ...............................................................
 *
 *  Entry Function
 *   - stop PWM, set speed now to zero
 *   - turn Yellow LED on
 *   - start idle dwell timer
 * 
 *  Tick Function
 *   - if fault ==> Fault state
 *   - if dwell timer still running ==> stay in this state
 *   - if Estop is open ==> stay in this state
 *   - if speed request > min speed ==> ramp state
 * 
 *  Exit Function
 *   - turn yellow LED off
 *   - read reverse pin
 */
void st_idle_entry(void)
{
  dbg_msg("Idle: Entry\n");

  pwm_disable();
  imsc.speed_now = 0;

  /* Set Yellow LED on */
  io_ledy_on();

  /* Start idle dwell timer */
  sm_start_timer(IMSC_IDLE_DWELL_TIME);
}

void st_idle_exit(void)
{
  dbg_msg("Idle: Exit\n");

  /* Set direction flag to required input */
  imsc.dir_now = io_is_reverse();

  /* Clear yellow LED */
  io_ledy_off();
}

sm_state_t * st_idle_tick(void)
{
  /* Check for faults --> Fault state */
  if(imsc_is_fault() == true) {
    return(&st_fault);
  }
  /* Check dwell timer --> Keep waiting */
  if(sm_timer_expired() != true) {
    return(NULL); 
  }
  /* If ESTOP is open --> Keep waiting */
  if(io_is_estop() == false) {
    return(NULL);
  }
  /* If speed request > MIN_FREQ --> Ramp */
  if(imsc_speed_req(true) > IMSC_MIN_SPEED) {
    return(&st_ramp);
  }
  return(NULL);
}

/* Ramp State  ...............................................................
 *
 *  Entry Function
 *   - flash Green LED fast
 *   - set PWM speed & direction & enable
 * 
 *  Tick Function
 *   - if fault ==> Fault state
 *   - if Estop open ==> idle
 *   - get speed request
 *   - if speed_now <= speed_request - margin
 *     - increment speed_now, but limit to speed_request
 *     - set pwm speed to new speed_now ==> stay in this state
 *   - if speed_now >= speed_request + margin
 *     - decrement speed_now, but limit to speed_request
 *     - if speed_now < min_speed ==> idle state
 *     - set pwm speed to new speed_now ==> stay in this state
 *   - else ==> at_speed state
 * 
 *  Exit Function
 *   - turn Green LED off
 */
void st_ramp_entry(void)
{
  dbg_msg( "Ramp: Entry\n");

  /* Flash Green LED */
  io_ledg_flash(0,IO_FLASH_FAST);

  pwm_set_dir(imsc.dir_now);
  pwm_set_speed(imsc.speed_now);
  pwm_enable();
}

void st_ramp_exit(void)
{
  dbg_msg("Ramp: Exit\n");

  /* Green LED off */
  io_ledg_off();
}

sm_state_t * st_ramp_tick(void)
{
  uint16_t speed_req;
  uint16_t delta;
  uint16_t ramp;

  /* Check for faults --> Fault state */
  if(imsc_is_fault() == true) {
    return(&st_fault);
  }
  
  /* If ESTOP is open --> Idle */
  if(io_is_estop() == false) {
    return(&st_idle);
  }

  /* Update speed request */
  speed_req = imsc_speed_req(false);
  ramp = imsc_ramp_delta();

  
  if( speed_req > imsc.speed_now) {
    /* Request > Current Speed */
    delta = speed_req - imsc.speed_now;
    if(delta > IMSC_SPEED_MARGIN) {
      /* Delta is meaningful so increment current speed by smaller of delta or 
         ramp increment */
      if( delta > ramp) { delta = ramp; }
      imsc.speed_now += delta;
      if(imsc.speed_now > IMSC_FULL_SCALE - IMSC_SPEED_MARGIN) { 
        /* curent speed within spitting distance of full speed */
        return(&st_atspeed); 
      }
      /* Set speed and exit*/
      pwm_set_speed(imsc.speed_now);
      return(NULL);
    }
  }
  if( speed_req < imsc.speed_now ) {
    /* Request < Current speed */
    delta = imsc.speed_now - speed_req;
    if(delta > IMSC_SPEED_MARGIN) {
      /* Delta is meaningful so decrement current speed by smaller of delta or 
         ramp increment */
      if( delta > ramp) { delta = ramp; }
      imsc.speed_now -= delta;
      if(imsc.speed_now < IMSC_MIN_SPEED) { 
        /* current speed within spitting distance of zero speed */
        return(&st_idle); 
      }
      /* Set speed and exit*/
      pwm_set_speed(imsc.speed_now);
      return(NULL);    
    } 
  }
  /* If just right --> at speed */
  return(&st_atspeed); 
}

/* At Speed State  ...............................................................
 *
 *  Entry Function
 *   - assert at speed output
 *   - turn Green LED on
 * 
 *  Tick Function
 *   - if fault ==> Fault state
 *   - if Estop open ==> idle
 *   - if speed_now <= speed_request - margin ==> ramp state
 *   - if speed_now >= speed_request + margin ==> ramp state
 *   - else ==> stay in this state
 * 
 *  Exit Function
 *   - deassert at speed output
 *   - turn off Green LED
 */
void st_atspeed_entry(void)
{
  dbg_msg( "At Speed: Entry\n");

  /* Green LED on & set at speed output */
  io_ledg_on();
  io_set_ats_output(true);

}

void st_atspeed_exit(void)
{
  dbg_msg("At Speed: Exit\n");

  /* Green LED off and clear at speed output */
  io_ledg_off();
  io_set_ats_output(false);
}

sm_state_t * st_atspeed_tick(void)
{
  uint16_t speed_req;

  /* Check for faults --> Fault state */
  if(imsc_is_fault() == true) {
    return(&st_fault);
  }
  
  /* If ESTOP is open --> Idle */
  if(io_is_estop() == false) {
    return(&st_idle);
  }

  /* Update speed request */
  speed_req = imsc_speed_req(false);

  /* If too hot --> Ramp */
  if(imsc.speed_now >= speed_req + IMSC_SPEED_MARGIN_ATS) {
    return(&st_ramp);
  }

  /* If too cold --> Ramp */
  if(imsc.speed_now <= speed_req - IMSC_SPEED_MARGIN_ATS) {
    return(&st_ramp);
  }
 
  return(NULL); 
}

/* Interrupt Service Routines ------------------------------------------------*/

/* End imsc */