/* 
 * File:   sh1106_3_1.c
 * 
 * Prepared by:  S. Denholm
 *
 * Graphical display OLED 128x64.
 * Driver:      SH1106
 * Interface:   I2C
 * Version 3
 * 
 */

#include <xc.h>
#include <string.h>     // required for 'strlen' function
#include "my_i2c.h"
#include "my_delays.h"
#include "oled_sh1106.h"

//==============================================================================
// Declaration of global variables.
//==============================================================================
cuchar *font, *font2;   // pointer variables
uchar width, height, char_pages, min, max;
uchar Oled_RAM[128];
uchar cnt_width = 0;
uint cnt_character = 0;

//==============================================================================
// Initialization the OLED module.
//==============================================================================

void Oled_Init(void) {
    Delay_ms(500);
    Oled_Command(0xAE); // Set Display OFF
    Oled_Command(0x81);
    Oled_Command(0xCF); // Set Contrast Control (80%)
    Oled_Command(0xA4); // Entire Display ON set to Normal
    Oled_Command(0xA6); // Normal/Reverse set to Normal

    Oled_Command(0x20);
    Oled_Command(0x02); // Set Memory Addressing Mode to page addressing
    Oled_Command(0x00); // Set Lower Column
    Oled_Command(0x10); // Set Higher Column
    Oled_Command(0xB0); // Set Page Start for page addressing mode

    Oled_Command(0x40); // Set Display Start Line
    Oled_Command(0xA1); /* Set Segment Re-map
                         * Note: to flip display if required, set Seg Re-map to A0h and
                         * COM output scan direction to C0h */
    Oled_Command(0xA8);
    Oled_Command(0x3F); // Set Multiplex Ratio (multiplex ratio = 64))
    Oled_Command(0xC8); // Set COM Output scan direction (normal = C8h, display flipped = C0h)
    Oled_Command(0xD3);
    Oled_Command(0x00); // Set Display Offset (no offset)
    Oled_Command(0xDA);
    Oled_Command(0x12); // Set COM Pins Hardware Configuration
    Oled_Command(0xD5);
    Oled_Command(0x80); // Set Display Clock Divide Ration & Clock Frequency (set DR = 1 and Fosc = +15%)
    Oled_Command(0xD9);
    Oled_Command(0xF1); // Set Pre-charge Period
    Oled_Command(0xDB);
    Oled_Command(0x40); // Set VCOMH Deselect Level
    Oled_Command(0x8D);
    Oled_Command(0x14); // Charge Pump Setting

    Oled_Clear(0x00);   // Clear screen

    Oled_Command(0xAF); // Set Display ON
}

//==============================================================================
// Write a command to the OLED.
//==============================================================================

void Oled_Command(uchar cmd) {
    I2C_Start();            // send I2C start
    I2C_WriteByte(OLED_12C_ADDRESS); // send I2C module slave address
    I2C_WriteByte(0x00);    /* send the control byte -> Continuation bit Co = 0, Data/Command D/C = 0
                             *  If Co=0 transmission of data only
                             *  If Co=1 the next two bytes are a data byte and another control byte
                             *  If D/C=0 following data byte is a command
                             *  If D/C=1 following data byte is data to be stored in the OLED GDDRAM */
    I2C_WriteByte(cmd);     // send the command byte
    I2C_Stop();             // send I2C stop
}

//==============================================================================
// Write a byte at OLED RAM memory.
//==============================================================================

void Oled_WriteRam(uchar dat) {
    I2C_Start();
    I2C_WriteByte(OLED_12C_ADDRESS);
    I2C_WriteByte(0x40);    // Co = 0, D/C = 1 ie "b01000000"
    I2C_WriteByte(dat);     // send the data to be stored in OLED GDDRAM
    I2C_Stop();
}

//==============================================================================
// Fill the OLED memory. Use for clearing the screen.
// pattern = byte to fill memory.
//==============================================================================

void Oled_Clear(uchar pattern) {
    uchar i, j;
    uint k = 0;
    for (i = 0; i < 8; i++) {
        Oled_SetPointer(0, i);
        for (j = 0; j < 131; j++) // changed value 128 to 131 to fully clear display
        {
            Oled_WriteRam(pattern);
            Oled_RAM[k] = pattern;
            k++;
        }
    }
}

//==============================================================================
// Set the pointer in OLED frame memory where MCU can access.
// seg:     sets the segment.
// page:    sets the  page.
//==============================================================================

void Oled_SetPointer(uchar seg, uchar page) {
    uchar low_column, high_column;
    low_column = (seg & 0b00001111);        // low_column = b0000 + lower  4bit nibble of seg
    high_column = (seg & 0b11110000) >> 4;  // higher 4bit nibble of seg moved to the lower 4bit nibble in high_column
    high_column = high_column | 0b00010000; // load b0001 into the higher 4bit nibble
    page = (page & 0b00000111);             // convert page number to a binary value in the lower 3 bits of page
    page = (page | 0b10110000);             // load hex B (1011) into the higher 4bit nibble of page ie code for page address command
    Oled_Command(page);                     // Set display Page Start
    Oled_Command(low_column);               // Set the display Lower Column address (00h-0Fh))
    Oled_Command(high_column);              // Set the display Higher Column address (10h-1Fh))
}

//==============================================================================
// Set the font that will be used.
// _font:   the selected font (ie name as specified in the font header file).
// _width:  sets the width in units of pixels.
// _height: sets the height in units of pixels.
// _min:    sets the range minimum.
// _max:    sets the range maximum.
//==============================================================================

void Oled_SetFont(cuchar *_font, uchar _width, uchar _height, uchar _min, uchar _max) {
    font2 = _font;              // this will be the memory starting address of the specified font list
    width = _width;
    char_pages = _height / 8;   // the number of pages the character occupies
    min = _min;                 // the font ASCII code list start (this will normally be 32 ie "space")
    max = _max;                 // the font ASCII code list end (this will normally be 127 ie "DEL")
}

//==============================================================================
// Write a char on the OLED.
// c:       ASCII character to be written.
// seg:     sets the segment location (horizontal position). Valid values: 0..127
// page:    sets the page location (vertical position). Valid values: 0..7
//==============================================================================

void Oled_WriteChar(uchar c, uchar seg, uchar page) {
    uint i, j;
    uchar x_seg, y_page;
    x_seg = seg;
    y_page = page;
    font = font2;
    j = c - min;
    j = j * (width * char_pages);
    for (i = 0; i < j; i++) {   // i++      -> use value of i then increment i by 1
        font++;                 // font++   -> use value of font then increment font pointer by 1
    }
    for (i = 0; i < width; i++) {
        y_page = page;
        for (j = 0; j < char_pages; j++) {
            if (x_seg < 128) {
                Oled_SetPointer(x_seg, y_page);
                Oled_WriteRam(*font);
                Oled_RAM[x_seg + (y_page * 128)] = *font;
            }
            y_page++;
            font++;
        }
        x_seg++;
    }
}

//==============================================================================
// Print text (ie string constant) on the OLED.
// Text_line:   string of text to be displayed on the OLED
// *Text_line:  pointer to the text to be displayed
// seg:         sets the segment. Valid values: 0..127
// page:        sets the page. Valid values: 0..7
//==============================================================================

void Oled_ConstText(cschar *buffer, uchar seg, uchar page) {    /* '*buffer' is the pointer or memory address to
                                                                 * the string constant to be displayed on the OLED. */
    uchar x_seg = seg;
    x_seg = x_seg;
    while (*buffer) {
        Oled_WriteChar(*buffer, x_seg, page);
        x_seg = x_seg + width;
        buffer++;       /* increment the pointer to the next character in the string.
                         * Loop until the string null character '\0' is reached and 
                         * then move on.  */
    }
}

