/* Digital Capacitance Meter with OLED display (approx range 10pF - 10,000uF)
 * 
 * File:   main_3_1.c
 * 
 * Developed by: S.Denholm
 * Date:         March 2024
 * 
 * MICROCHIP MPLAB X IDE v6.15 & XC8 code compiler v2.45
 * 
 *                               PIC16F1847
 *                       +----------:_:----------+
 *    Cx Charge S1B <- 1 : RA2               RA1 : 18 <- BAT Monitor
 *       Cx Measure -> 2 : RA3/C1IN+ C12IN0-/RA0 : 17 <- Vref +3.16V
 *         Range_Hi -> 3 : RA4               RA7 : 16 -> not used
 *         MCLR/Vpp -> 4 : RA5        CLKOUT/RA6 : 15 -> TP2
 *              GND -> 5 : Vss               Vdd : 14 <- +5V
 *        Range_Mid -> 6 : RB0               RB7 : 13 <> ICSPDAT
 *    MSSP I2C SDA1 -> 7 : RB1/SDA1          RB6 : 12 <> ICSPCLK
 *         Range_Lo -> 8 : RB2               RB5 : 11 <- S3 Measure Start Button
 *  Cx Discharge Q4 <- 9 : RB3          SCL1/RB4 : 10 <- MSSP I2C SCL1
 *                       +-----------------------+
 *                                DIP-18
 */

#include <xc.h>
#include "config_settings.h"
#include <stdio.h>          // Types/macros/functions used for input/output operations eg. sprintf function    
#include <stdint.h>         // Types/macros used to define integer types
#include "my_i2c.h"         // Declaration of functions used in my_i2c source file
#include "oled_sh1106.h"    // Declaration of functions used in oled_sh1106 source file
#include "cap_meter_font.h" // Declaration of functions used in font source file
#include "my_delays.h"      // Declaration of functions used in my_delays source file

#define Charge_Cx       LATA2
#define Discharge_Cx    LATB3
#define Start_Btn       RB5
#define Range_Hi        RA4
#define Range_Mid       RB0
#define Range_Lo        RB2

//---- Global Variable Definitions --------
uint8_t measure_done;
uint8_t TMR1OF_count;
uint16_t TMR1_value;
uint32_t Measure_time;
uint32_t Cx_value;
uint32_t R_divide;
char buffer[20];

//----- Function Declarations -------------
void Battery_check(void);

//===========================  MAIN PROGRAM  ===================================

void main(void) {
    // Oscillator module configure
    OSCCON = 0x6A;  // 01101010b SPLLEN = disabled, IRCF = 4MHz, SCS = Internal Oscillator
    OSCTUNE = 0x00; // no oscillator tuning ``````````````````````````````````````````````````` 

    // GPIO configure
    TRISA = 0x3B;   // b0011 1011 -> Set RA2,6,7 for output. Set RA0,1,3,4,5 for input
    TRISB = 0xF7;   // b1111 0111 -> Set RB3 for output. Set RB0,1,2,4,5,6,7 for input

    LATA7 = 0;      // Unused pin so set output low

    ANSELA = 0x00;  // Initially set all PORTA pins for digital I/O
    ANSELB = 0x00;  // Initially set all PORTB pins for digital I/O
    ANSA0 = 1;      // Set pin RA0 for analog input (Comparator C1 input C12IN0-)
    ANSA1 = 1;      // Set pin RA1 for analog input (Battery voltage monitor)
    ANSA3 = 1;      // Set pin RA3 for analog input (Comparator C1 input C1IN+)

    // ADC configure
    ADCON0bits.CHS = 1;     // Select channel AN1 for battery voltage monitoring
    ADCON1bits.ADFM = 1;    // Right justified. The six most significant bits of ADRESH are set to '0' on conversion
    ADCON1bits.ADCS = 4;    // ADC conversion clock  =  Fosc/4
    ADCON1bits.ADNREF = 0;  // VREF- is connected to Vss
    ADCON1bits.ADPREF = 0;  // VREF+ is connected to Vdd

    // Timer 1 module configure
    T1CONbits.TMR1CS = 0;   // Timer1 clock source is instruction clock (FOSC/4)
    T1CONbits.T1CKPS = 0;   // 1:1 Prescale value ( 4MHz/4 -> this will give a 1us timer tick)
    T1GCONbits.TMR1GE = 1;  // Timer1 counting is controlled by the Timer1 gate function
    T1GCONbits.T1GPOL = 0;  // 0 = Timer1 gate is active-low (Timer1 counts when gate is low)
    T1GCONbits.T1GTM = 0;   // Timer1 Gate Toggle mode is disabled and toggle flip-flop is cleared
    T1GCONbits.T1GSS = 2;   // Comparator 1 optionally synchronized output (sync_C1OUT)
    TMR1ON = 1;             // Turn Timer1 on to allow comparator output to initiate

    // Comparator module configure
    CM1CON0bits.C1OE = 1;   // C1OUT is present on the C1OUT pin.
    CM1CON0bits.C1POL = 0;  // Comparator output is not inverted
    CM1CON0bits.C1SP = 1;   // Comparator operates in normal power, higher speed mode
    CM1CON0bits.C1HYS = 0;  // Comparator hysteresis disabled
    CM1CON0bits.C1SYNC = 1; /* Comparator output to Timer1 and I/O pin is synchronous to changes on Timer1 clock source.
                             * Output updated on the falling edge of Timer1 clock source. */
    CM1CON1bits.C1INTP = 1; // The C1IF interrupt flag will be set upon a positive going edge of the C1OUT bit
    CM1CON1bits.C1INTN = 0; // No interrupt flag will be set on a negative going edge of the CxOUT bit
    CM1CON1bits.C1PCH = 0;  // C1VP connects to C1IN+ pin
    CM1CON1bits.C1NCH = 0;  // C1VN connects to C12IN0- pin
    C1ON = 1;               // Turn Comparator1 on

    // Interrupts configure
    GIE = 0;
    PEIE = 0;
    TMR1IE = 1;             // Enables the Timer1 overflow interrupt

    // Local variable definitions
    unsigned int i, j, k, l = 0, page;
    unsigned int zero_offset = 275;     /* This offset adjustment compensates for some inherent MCU instruction cycle time
                                         * which becomes more dominant for measurement of small capacitances eg. <1nF.
                                         * It also compenstates for any stray capacitance for the meter as built.
                                         * The offset value was determined and adjusted as part of the meter calibration
                                         * process and then the source code was recomplied and uploaded to the MCU */

    // Variable initial conditions
    TMR1OF_count = 0;
    measure_done = 0;

    // Initiate the I2C interface & OLED Display
    I2C_Close();            // Close the  I2C Bus
    I2C_Init(0x09);         /* I2C 100kHz, 4MHz-CRYSTAL  Ref PIC16F1847 data sheet p265
                            *  Baud Rate Generator */
    Oled_Init();            // Initiate the OLED

    // WELCOME BANNER
    Oled_SetFont(Font_6x8, 6, 8, 32, 127);
    Oled_ConstText("      DIGITAL", 2, 1);
    Oled_ConstText("  CAPACITANCE METER", 2, 3);
    Oled_ConstText("  10pF to 10,000uF", 2, 5);
    Oled_ConstText(" S.Denholm April 2024", 2, 7);
    Delay_ms(3000);

    while (1) {
        // Battery Voltage Check
        Battery_check();

        // Range Series Resistance Divisor select for Cx Calculation
        if (Range_Hi) {
            R_divide = 500;
        }
        if (Range_Mid) {
            R_divide = 25000;
        }
        if (Range_Lo) {
            R_divide = 2000000;
        }

        Oled_Clear(0x00);       // Clear the OLED screen
        Oled_ConstText("   Connect Cx then", 2, 1);
        Oled_ConstText(" Press MEASURE button", 2, 4);
        Oled_ConstText("     to commence...", 2, 7);

        // Wait for press of the MEASURE Button
        while (Start_Btn == 1) {
           }

        // Commence measurement of Cx
        Oled_Clear(0x00);   // Clear the OLED screen
        Oled_ConstText("********************", 2, 1);
        Oled_ConstText(".... MEASURING .....", 2, 4);
        Oled_ConstText("********************", 2, 7);
        Charge_Cx = 1;      // Initially set pin RA2 high (ie Q1, Q2, Q3 off) to prevent Cx from charging
        Discharge_Cx = 1;   // Initially set pin RB3 high (ie Q4 on) to discharge Cx        
        Delay_ms(2000);     /* Time delay sufficient to ensure that Cx is fully discharged (ie allowing for up to 10,000uF + 20%)
                             * and that Comparator1 output is low for measurement pre-conditions */

        // Set the preconditions for measurement
        TMR1ON = 0;         // Turn Timer1 off
        TMR1H = 0;          // Clear Timer1 high byte register
        TMR1L = 0;          // Clear Timer1 low byte register
        TMR1OF_count = 0;   // Reset the Timer1 overflow counter
        measure_done = 0;   // Reset Cx "measurement done" flag
        Cx_value = 0;

        C1IF = 0;           // Ensure Comparator1 Interrupt Flag bit is cleared
        C1IE = 1;           // Enable Comparator1 interrupt

        TMR1IF = 0;         // Ensure Timer1 Overflow Interrupt Flag bit is cleared
        TMR1IE = 1;         // Enable Timer1 overflow interrupt

        PEIE = 1;           // Enable all peripheral interrupts      
        GIE = 1;            // Enable all active interrupts

        // Cx Measurement start
        Discharge_Cx = 0;   // Stop Cx discharging
        Charge_Cx = 0;      // Start Cx charging
        TMR1ON = 1;         // Start Timer1

        while (measure_done == 0) {     // Wait until a measurement event is captured
            if (TMR1OF_count > 100){    // Check if the measurement time is too long (say more than approx 6.5sec)
                                        // and if so terminate the measurement and exit the while loop      
                measure_done = 1;
            }
        }

        Oled_Clear(0x00);               // Clear the OLED screen
        Oled_ConstText("  Capacitor Cx Value", 2, 1);

        // Select the measurement range then calculate Cx and display the result
        if (Range_Hi == 1) {
            TMR1_value = TMR1H * 256 + TMR1L;
            Measure_time = (TMR1_value + TMR1OF_count * 65536);
            Cx_value = 10 * Measure_time / R_divide;

            if (95 <= Cx_value / 10 && Cx_value / 10 <= 12000) {
                sprintf(buffer, "     = %lu.%lu uF", Cx_value / 10, Cx_value % 10);
                Oled_ConstText(buffer, 2, 3);
            } else if (Cx_value / 10 < 95) {
                Oled_Clear(0x00);                   // Clear the OLED screen 
                Oled_ConstText("   Range too high!", 2, 2);
                Oled_ConstText(" Select a lower range", 2, 4);
            } else if (Cx_value / 10 > 12000) {     // Allow for capacitor values up to 10,000uF +20%
                Oled_Clear(0x00);                   // Clear the OLED screen 
                Oled_ConstText("   !! OVER RANGE !!", 2, 4);
            }
        }
        if (Range_Mid == 1) {
            TMR1_value = TMR1H * 256 + TMR1L;
            Measure_time = (TMR1_value + TMR1OF_count * 65536);
            Cx_value = 10 * Measure_time / R_divide;

            if (1 <= Cx_value / 10 && Cx_value / 10 <= 105) {
                sprintf(buffer, "     = %lu.%lu uF", Cx_value / 10, Cx_value % 10);
                Oled_ConstText(buffer, 2, 3);
            } else if (Cx_value / 10 < 1) {
                Oled_Clear(0x00);               // Clear the OLED screen 
                Oled_ConstText("   Range too high!", 2, 2);
                Oled_ConstText(" Select a lower range", 2, 4);
            } else if (Cx_value / 10 > 105) {
                Oled_Clear(0x00);               // Clear the OLED screen 
                Oled_ConstText("     Over Range!", 2, 2);
                Oled_ConstText(" Select a higher range", 2, 4);
            }
        }
        if (Range_Lo == 1) {
            TMR1_value = TMR1H * 256 + TMR1L;
            Measure_time = (TMR1_value + TMR1OF_count * 65536);

            if (Measure_time >= 2000) {
                Cx_value = Measure_time / (R_divide / 10000);

                if (1 <= Cx_value / 10 && Cx_value / 10 <= 1100) {
                    sprintf(buffer, "     = %lu.%lu nF", Cx_value / 10, Cx_value % 10);
                    Oled_ConstText(buffer, 2, 3);
                } else if (Cx_value / 10 > 1100) {
                    Oled_ConstText("      Over Range", 2, 2);
                    Oled_ConstText(" Select higher range", 2, 4);
                }
            }
            if (Measure_time < 2000) {
                Cx_value = (1000 * Measure_time / (R_divide / 10000)) - zero_offset * 10;
                if (Cx_value > 0xFFFFFFF / 10) {    /* This check is to guard against a negative calculation result
                                                     * if the zero offset value is slightly out */
                    Oled_Clear(0x00);               // Clear the OLED screen                   
                    Oled_ConstText("  Value of Cx too low", 2, 1);
                    Oled_ConstText("  or Cx not connected", 2, 3);
                } else {
                    sprintf(buffer, "     = %lu pF", Cx_value / 10);
                    Oled_ConstText(buffer, 2, 3);
                }
            }
        }

        Charge_Cx = 1;      // Stop charging Cx
        Discharge_Cx = 1;   // Start discharging Cx
        Delay_ms(2000);     // Allow sufficient discharge time to ensure Cx is discharged and safe  

        Oled_ConstText(" Press MEASURE button", 2, 6);
        Oled_ConstText("   to continue ....", 2, 7);

        while (Start_Btn == 1) {    // Wait until the MEASURE button is pressed to continue
        }
    }
}   // End of Main Program

//======================= Interrupt Service Routine ============================

void __interrupt() ISR(void) {
    if (TMR1IF) {               // Timer 1 Interrupt is pending
        TMR1OF_count++;         // Increment the Timer 1 overflow counter
        TMR1IF = 0;             // Reset Timer 1 interrupt flag
    }

    if (C1IF) {                 // Comparator C1 interrupt flag pending
        PEIE = 0;               // Disables all peripheral interrupts
        C1IE = 0;               // Disables the  Comparator 1 interrupt
        measure_done = 1;       // Sets the measurement done flag
    }
}

//======================  Function Definitions  ================================

void Battery_check(void) { // Checks voltage status of the supply battery
    uint32_t ADC_Reg_value;
    uint32_t ADC_volts;
    uint32_t V_Battery;

    ADON = 1;           // Turn on the ADC
    Delay_us(20);       // Allow acquisition time (ie charging of holding capacitor)
    GO_nDONE = 1;       // Start the conversion

    while (GO_nDONE) {  // Wait until the conversion completes
    }

    ADC_Reg_value = ADRESH * 256 + ADRESL;      // Store the 10 bit ADC value by combining the ADC high and low register values
    ADC_volts = 500 * ADC_Reg_value / 1023;     // ADC +Vref = Vdd = 5V (Note: Vdd is then multiplied by 100 to increase resolution for integer arithmetic)
    V_Battery = ADC_volts * (900 + 35) / 5000;  // This also allows approximately 0.35V for the voltage drop across schottky diode D1

    if (378 <= ADC_volts && ADC_volts <= 405) { // Check if battery voltage is between about 7 and 7.5 volts
        Oled_Clear(0x00);                       // Clear the OLED screen
        sprintf(buffer, "Battery Volts = %lu.%luV", V_Battery / 10, V_Battery % 10);
        Oled_ConstText(buffer, 2, 1);
        Oled_ConstText("Battery Volts LOW", 2, 4);
        Delay_ms (2000);
    } else if (ADC_volts < 378) {               /* Check if battery voltage is less than about 7 Volts (ie 500*7/9.27 = 378) and
                                                 * if so then display "Replace Battery" warning on the OLED */
        Oled_Clear(0x00);                       // Clear the OLED screen
        sprintf(buffer, "Battery Volts = %lu.%luV", V_Battery / 10, V_Battery % 10);
        Oled_ConstText(buffer, 2, 1);
        Oled_ConstText("Battery Volts too LOW", 2, 4);
        Oled_ConstText(" !!REPLACE BATTERY!!", 2, 7);
        while (1) {                             // loop continuously
        }
    }
}


