//for low power
#include <avr/sleep.h>

//datasheet says 115200, actually 19200
#define BAUDRATE 19200
#define MODEM Serial1
#define DEBUG Serial
#define PWRKEY 12
#define SHUTDOWN_PIN 7
#define BUFSIZE 100
#define TOKENSIZE 100
#define SHORTRESULT 30
#define URLLENGTH 300
#define MSGLENGTH 255
#define DATELENGTH 50
#define NUMBERLENGTH 50
char token[TOKENSIZE];
char shortToken[SHORTRESULT];
char url[URLLENGTH];
char msg[MSGLENGTH];
char date[DATELENGTH];
char phoneno[NUMBERLENGTH];
//message storage size
#define STORESIZE 30
#define SLEEP_TIME (60000L)

//set the following:
#define API_WRITE_KEY "API_KEY_HERE"
#define AUTH_NUMBER "AUTH_NUMBER_HERE"
#define OUTBOUND_NUMBER "OUTGOING_NUMBER_HERE"

//ADC ratios (V/ADC count)
#define VSUPRATIO 0.015272
#define VBATRATIO 0.015272

#include "cwrite.h"
cwrite c;
cwrite m;

//GPSBUF is length to store fields
#define GPSBUF 30

struct GPSdata {
  byte fix;           //0 or 1
  unsigned long time; //UNIX time
  float lat;          //in degrees
  float lon;          //in degrees
  float alt;          //in metres
  float speed;        //km/h (SIM7000 native)
  float course;       //in degrees
  int fixmode;        //0,1,2  
};

GPSdata gnss;

//BME280 interface
//with bit 0 pulled low (as on SIM7000 shield)
#define BME280_ADDRESS 0x76
#include <Wire.h>
#include "SparkFunBME280.h"
BME280 sensor;

void(* resetFunc) (void) = 0;//declare reset function at address 0

void setup() {
  c.set(url);
  m.set(msg);
  Wire.begin();
  Wire.setClock(100000);    //I2C scanner locks up at 400kHz?
  sensor.setI2CAddress(BME280_ADDRESS);
  sensor.beginI2C();
  //while((!Serial)&&(millis()<2000)){}   //wait for serial or timeout after 2s, helpful when debugging
  delay(100);  
  DEBUG.begin(BAUDRATE);
  MODEM.begin(BAUDRATE);
  digitalWrite(PWRKEY,LOW);
  pinMode(PWRKEY,OUTPUT);  
}

void loop() {
  unsigned long tt,uptime;
  int i;
  boolean gnssValid=false;
  uptime=millis();
  modemInit();
  tt=millis();  

  while((millis()-tt<10000L)&&(gnssValid==false)){   //wait time or til GPS gets a fix
    gnssValid=getGNSSdata(&gnss);
    delay(1000);
    DEBUG.println(F("Wait for GNSS"));
  }
  //check for alert conditions eg geofence/speed, valid GNSS data is present in gnss if gnssValid==true
  if(gnssValid&&(gnss.speed>100.0)){
    DEBUG.println(F("Sending message, speed limit exceeded."));
    if(sendSMS(OUTBOUND_NUMBER,"Speed limit exceeded.")){
      DEBUG.println(F("SMS Send OK"));              
    }else{
      DEBUG.println(F("SMS Send fail"));                            
    }    
  }
  
  sendToThingspeak();         //gets all info and sends it
  checkForMessages();         //do this as often as necessary
  
  if(getGNSSdata(&gnss)||(millis()>600000L)){       //if GNSS OK, shutdown, otherwise leave on to improve fix, if we've been on more than 10min, shutdown anyway to reboot
    DEBUG.println(F("Powering Down"));
    powerToggle();              //shut down modem
    delay(100);                 //wait for shutdown
    pinMode(SHUTDOWN_PIN,OUTPUT);
    digitalWrite(SHUTDOWN_PIN,HIGH);    //initiate hard shutdown of Arduino    
    delay(10);
    resetFunc();                        //if we're still getting power, reset so we're in a known state
  }
  
  //uptime typically lasts for ~30s
  DEBUG.println(F("Staying Powered On"));
  tt=millis();
  while(millis()-tt<SLEEP_TIME){              //go into deep sleep instead to save a little bit of power
    set_sleep_mode (SLEEP_MODE_IDLE); 
    sleep_enable();
    sleep_cpu ();   //wake up by interrupt
  }
}

void checkForMessages(){
  int i;
  i=checkSMS();
  if(i){
    DEBUG.print(i);
    DEBUG.println(F(" messages received."));
    for(i=0;i<STORESIZE;i++){
      if(readSMS(i)){
        DEBUG.print(i);
        DEBUG.print(":");
        DEBUG.print(date);
        DEBUG.print("|");
        DEBUG.print(phoneno);
        DEBUG.print("|");
        DEBUG.println(msg);
        DEBUG.println(SMStoUNIX(date));         //convert time in SMS to UTC
        if(strlen(AUTH_NUMBER)<=strlen(phoneno)){   //match AUTH_NUMBER to trailing digits of received number
          if(strmatch(AUTH_NUMBER,&phoneno[strlen(phoneno)-strlen(AUTH_NUMBER)])){
            DEBUG.println(F("Authorised number detected."));
            //do auth stuff here
            
            //and send confirmation of receipt
            if(sendSMS(OUTBOUND_NUMBER,"SMS received and acted upon.")){
              DEBUG.println(F("SMS Send OK"));              
            }else{
              DEBUG.println(F("SMS Send fail"));                            
            }
          }
        }
        
        if(deleteSMS(i)){
          DEBUG.println(F("Delete OK"));
        }
        if(checkSMS()==0){i=STORESIZE;}   //skip messages if we have deleted all               
      }
    }
  }else{
    DEBUG.println(F("No messages received.")); 
  }
}

int sendToThingspeak(){   //fields are (1-8)=(LAT,LON,SPD,HEADING,PRESSURE,TEMP,HUMIDITY,ALTITUDE)
  float t,h,p;
  float vsup,vbat,adc2,adc3;
  int i;
  //double read analog channels for stability
  analogRead(A1);  
  vsup=analogRead(A1)*VSUPRATIO;
  analogRead(A0);  
  vbat=analogRead(A0)*VBATRATIO;
  analogRead(A2);    
  adc2=analogRead(A2);
  analogRead(A3);  
  adc3=analogRead(A3);  
  c.clear();
  c.print(F("http://api.thingspeak.com/update?api_key="));
  c.print(API_WRITE_KEY);
  i=getGNSSdata(&gnss);
  if(!i){i=getGNSSdata(&gnss);}   //try twice
  if(i){
    DEBUG.println(F("GNSS OK"));
    DEBUG.println(gnss.fix);
    DEBUG.println(gnss.time);          
    DEBUG.println(gnss.lat,6);          
    DEBUG.println(gnss.lon,6);          
    DEBUG.println(gnss.alt);          
    DEBUG.println(gnss.speed);          
    DEBUG.println(gnss.course);          
    DEBUG.println(gnss.fixmode);          
    DEBUG.println();                
    c.print(F("&field8="));
    c.print(gnss.alt,2);
    c.print(F("&lat="));
    c.print(gnss.lat,6);
    c.print(F("&long="));
    c.print(gnss.lon,6);
    c.print(F("&elevation="));
    c.print((int)gnss.alt);    
  }else{
    DEBUG.println(F("GNSS fix fail"));      
  }
  t=sensor.readTempC();
  h=sensor.readFloatHumidity();
  p=sensor.readFloatPressure();
  DEBUG.print("T=");
  DEBUG.print(t,2);
  DEBUG.println("C");
  DEBUG.print("H=");
  DEBUG.print(h,2);
  DEBUG.println("%");
  DEBUG.print("P=");
  DEBUG.print(p,2);
  DEBUG.println("Pa");
  DEBUG.println();    
  c.print(F("&field1="));
  c.print(vsup,2);
  c.print(F("&field2="));
  c.print(vbat,2);
  c.print(F("&field3="));
  c.print(adc2,2);
  c.print(F("&field4="));
  c.print(adc3,2);
  c.print(F("&field5="));
  c.print(p,2);
  c.print(F("&field6="));
  c.print(t,2);
  c.print(F("&field7="));
  c.print(h,2);
  DEBUG.println(url);
  if(httpGet(url)){
    DEBUG.println(F("ThingSpeak Success"));
    return 1;
  }else{
    DEBUG.println(F("ThingSpeak Fail"));
    return 0;
  }
}

int getGNSSdata(GPSdata* g){
  static char buf[GPSBUF]="";
  int i;
  for(i=0;i<GPSBUF;i++){buf[i]=0;}
  purgeModem();  
  MODEM.println(F("AT+CGNSINF"));     //send command
  if(parseFor("+CGNSINF: ",200)){    
    scanUntil(',',buf,GPSBUF,100);
    //ignore 1st, run status
    for(i=0;i<GPSBUF;i++){buf[i]=0;}
    scanUntil(',',buf,GPSBUF,100);
    g->fix=atoi(buf);                 //fix
    for(i=0;i<GPSBUF;i++){buf[i]=0;}
    scanUntil(',',buf,GPSBUF,100);
    g->time=GNStoUNIX(buf);           //time
    for(i=0;i<GPSBUF;i++){buf[i]=0;}
    scanUntil(',',buf,GPSBUF,100);
    g->lat=atof(buf);                 //latitude
    for(i=0;i<GPSBUF;i++){buf[i]=0;}
    scanUntil(',',buf,GPSBUF,100);
    g->lon=atof(buf);                 //longitude
    for(i=0;i<GPSBUF;i++){buf[i]=0;}
    scanUntil(',',buf,GPSBUF,100);
    g->alt=atof(buf);                 //altitude
    for(i=0;i<GPSBUF;i++){buf[i]=0;}
    scanUntil(',',buf,GPSBUF,100);
    g->speed=atof(buf);                 //speed
    for(i=0;i<GPSBUF;i++){buf[i]=0;}
    scanUntil(',',buf,GPSBUF,100);
    g->course=atof(buf);                //course/heading
    for(i=0;i<GPSBUF;i++){buf[i]=0;}
    scanUntil(',',buf,GPSBUF,100);
    g->fixmode=atoi(buf);               //fix mode
    if((g->fixmode<1)||(g->fixmode>2)){return 0;}
    if(parseFor("OK",50)){return 1;}    //all data received
    //else{DEBUG.println("NO OK");}
  }
  return 0;  
}

unsigned long SMStoUNIX(char* s){   //format is "19/09/26,16:27:04+40" from text message
  int year,month,day,hour,minute,second,tz;
  year=2000;
  year+=(s[0]-'0')*10;
  year+=(s[1]-'0');
  month=(s[3]-'0')*10;
  month+=(s[4]-'0');
  day=(s[6]-'0')*10;
  day+=(s[7]-'0');
  hour=(s[9]-'0')*10;
  hour+=(s[10]-'0');
  minute=(s[12]-'0')*10;
  minute+=(s[13]-'0');
  second=(s[15]-'0')*10;
  second+=(s[16]-'0');
  tz=(s[18]-'0')*10;
  tz+=(s[19]-'0');
  if(s[17]=='-'){tz=-tz;}  
  return toUNIX(year,month,day,hour,minute-tz*15,second);   //timezones are counted in 15 minute increments
}

unsigned long GNStoUNIX(char* s){
  int year,month,day,hour,minute,second;
  year=(s[0]-'0')*1000;
  year+=(s[1]-'0')*100;
  year+=(s[2]-'0')*10;
  year+=(s[3]-'0');
  month=(s[4]-'0')*10;
  month+=(s[5]-'0');
  day=(s[6]-'0')*10;
  day+=(s[7]-'0');
  hour=(s[8]-'0')*10;
  hour+=(s[9]-'0');
  minute=(s[10]-'0')*10;
  minute+=(s[11]-'0');
  second=(s[12]-'0')*10;
  second+=(s[13]-'0');
  return toUNIX(year,month,day,hour,minute,second);  
}

unsigned long toUNIX(int year, int month, int day, int hour, int minute, int second){
  unsigned long retval=0;
  unsigned long daycount=0;
  year=year;
  if(year<1970){return 0;}    //implement unsigned time
  retval=(hour*60L+minute)*60L+second;
  daycount=day-1;             //roll back to first of month
  while(month>1){ //work back to January 1
    switch(month){
      case 12: daycount=daycount+30;month--;break;    //DEC 1 to NOV 1
      case 11: daycount=daycount+31;month--;break;    //NOV 1 to OCT 1
      case 10: daycount=daycount+30;month--;break;    //OCT 1 to SEP 1
      case 9: daycount=daycount+31;month--;break;     //SEP 1 to AUG 1
      case 8: daycount=daycount+31;month--;break;     //AUG 1 to JUL 1
      case 7: daycount=daycount+30;month--;break;     //JUL 1 to JUN 1
      case 6: daycount=daycount+31;month--;break;     //JUN 1 to MAY 1
      case 5: daycount=daycount+30;month--;break;     //MAY 1 to APR 1
      case 4: daycount=daycount+31;month--;break;     //APR 1 to MAR 1
      case 2: daycount=daycount+31;month--;break;     //FEB 1 to JAN 1
      case 3:
        month--;
        if(isLeapYear(year)){
          daycount=daycount+29;
        }else{
          daycount=daycount+28;
        }        
        break;
    }
  }
  while(year>1970){
    if(isLeapYear(year-1)){             //going to to last year's Jan 1 means going through last year's Feb
      daycount=daycount+366;
    }else{
      daycount=daycount+365;
    }
    year--;
  }
  return retval+86400L*daycount;
}

boolean isLeapYear(int y){
  if(y&3){return 0;}      //not a multiple of 4
  if(y%100){
    return 1;             //divisible by 4 but not 100
  }else{
    if(y%400){
      return 0;           //1900,2100 etc
    }else{
      return 1;           //2000 etc
    }
  }
}

int deleteSMS(int n){       //delete slot n
  purgeModem();  
  MODEM.print(F("AT+CMGD="));
  MODEM.print(n);  
  MODEM.println(F(",0"));
  purgeModem();  
  if(parseFor("OK",5000)){return 1;}
  return 0;
}

int readSMS(int n){       //read slot n, put into msg, phoneno and date
  int i;  
  for(i=0;i<MSGLENGTH;i++){msg[i]=0;}       //clear incoming buffers
  for(i=0;i<DATELENGTH;i++){date[i]=0;}
  for(i=0;i<NUMBERLENGTH;i++){phoneno[i]=0;}
  purgeModem();
  MODEM.print(F("AT+CMGR="));
  MODEM.println(n);
  if(parseFor("+CMGR: ",200)){  //valid response
    //can't use regular parse as we may have too much data for serial buffer
    while(MODEM.read()!=','){}    //find first field
    scanUntil(',',phoneno,NUMBERLENGTH,10);
    scanUntil(',',phoneno,0,10);             //ignore next entry- it should be phone book name
    scanUntil(',',date,DATELENGTH-1,10);
    date[strlen(date)]=',';                   //add back stripped out comma
    scanUntil(10,date,DATELENGTH,10);        //date includes comma ends with <CR><LF>
    scanUntil(0,msg,MSGLENGTH,300);           //this should timeout unless message fills up
    //DEBUG.println(phoneno);
    //DEBUG.println(date);
    //DEBUG.println(msg);  
    //DEBUG.println(strlen(msg));    
    //DEBUG.println();
    if(strlen(msg)>=8){
      if(strmatch(&msg[strlen(msg)-8],"\xD\xA\xD\xAOK\xD\xA")){
        msg[strlen(msg)-8]=0;     //remove trailing excess data
        stripQuotes(date);
        stripQuotes(phoneno);     //tidy up results        
        return 1;
      }
    }
  }
  //DEBUG.println("FAIL");        
  msg[0]=0;     //null
  phoneno[0]=0;
  date[0]=0;
  return 0;     //fail
}

void scanUntil(char c, char* dest, int s, int t){    //scan until c received, put into dest up to s, timeout t
  unsigned long tout;
  int d=-1;
  int destptr=strlen(dest);
  tout=millis();
  while((d!=c)&&(millis()-tout<t)){
    if(MODEM.available()){
      d=MODEM.read();
      if((d!=c)&&(destptr<s-2)){
        dest[destptr]=d;
        destptr++;
      }
    }
  }  
}

int checkSMS(){   //any incoming SMS?
  int v;
  purgeModem();
  MODEM.println(F("AT+CPMS?"));   //check message storage also reports usage
  if(parseFor("+CPMS: ",200)){  //valid response
    if(tokenise(7,100,shortToken,SHORTRESULT)){ //find tokenised data, number of used receive slots is in #7
      if(parseFor("OK",100)){       //check OK
        v=atoi(shortToken);
        //DEBUG.println(shortToken);
        return v;    
      }
    }
  }
  return 0;         //if we don't get a result, there's nothing to do anyway 
}

void modemInit(){
  MODEM.println();                      //end any unfinished commands
  delay(200);
  if(!commandResponse("AT","OK",100)){
    DEBUG.println(F("Starting"));
    powerToggle();
    passThrough(5000);    
  }else{
    DEBUG.println(F("Already on"));    
  }
  if(commandResponse("AT+CMGF=1","OK",100)){    //set text SMS format
    DEBUG.println(F("Format set"));
  }else{
    DEBUG.println(F("Fail"));
  }
  if(commandResponse("AT+CGNSPWR=1","OK",100)){    //power on GNSS module
    DEBUG.println(F("GNSS on"));
  }else{
    DEBUG.println(F("Fail"));
  }
  if(commandResponseTokenise("AT+COPS?","+COPS:",100,2,200,token,TOKENSIZE)){
    DEBUG.println(F("Success:"));
    stripQuotes(token);
    DEBUG.println(token);
  }else{
    DEBUG.println(F("Fail"));
  }
  deInitHttp();           //ensure HTTP API is in known state
}

int getCSQ(){     //check signal quality
  int v;
  purgeModem();
  MODEM.println(F("AT+CSQ"));  
  if(parseFor("+CSQ: ",200)){
    if(tokenise(0,100,shortToken,SHORTRESULT)){ //find tokenised data
      if(parseFor("OK",100)){       //check OK
        v=atoi(shortToken);
        //DEBUG.println(shortToken);
        //DEBUG.println(v);
        if((v>0)&&(v<32)){return v;} //99 means unknown, discard          
      }
    }
  }
  return 0;   //not known
}

int sendSMS(char* number, char* body){
  if(strlen(body)>160){return 0;}   //body too long
  purgeModem();
  MODEM.print(F("AT+CMGS=\""));
  MODEM.print(number);
  MODEM.println(F("\""));
  if(parseFor(">",200)){
    MODEM.print(body);
    purgeModem();         //purge to avoid false OK from body text
    MODEM.write(26);      //ctrl-z, message done
    if(parseFor("OK",5000)){return 1;}
  }
  MODEM.write(27);      //escape to cancel if we're stuck
  return 0;  
}

void passThrough(unsigned long t){
  unsigned long s;
  s=millis();
  while((millis()-s)<t){
    passThrough();
  }
}

void passThrough(){
  while(MODEM.available()){
    DEBUG.write(MODEM.read());
  }
  while(DEBUG.available()){
    MODEM.write(DEBUG.read());
  }
}

void powerToggle(){   //notes say 2s, but 5s works
  digitalWrite(PWRKEY,HIGH);
  delay(5000);
  digitalWrite(PWRKEY,LOW);    
}

int commandResponse(char* c, char* r, int t){    //send c, expect r within t ms
  MODEM.println(c);
  return parseFor(r,t);    
}

int strmatch(char *x, char* y){           //1 on match, 0 on mismatch
  int i;
  if(strlen(x)!=strlen(y)){return 0;}     //diff length => no match
  for(i=0;i<strlen(x);i++){
    if(x[i]!=y[i]){return 0;}
  }
  return 1;
}

int parseFor(char* r, int t){
  static char buf[BUFSIZE];  
  int i,rlen,d;
  unsigned long s;
  rlen=strlen(r);
  for(i=0;i<BUFSIZE;i++){buf[i]=0;}   //clear buffer
  i=0;
  s=millis();
  while(millis()-s<t){
    if(MODEM.available()){
      buf[i]=MODEM.read();
      if(i>rlen-2){
        d=strmatch(r,&buf[i-rlen+1]);
        //DEBUG.println(d);
        if(d){return 1;}
      }
      if(i<BUFSIZE-2){i++;}   //clamp on overflow
    }
  }
  //DEBUG.println(strlen(buf));
  //DEBUG.println(buf);
  return 0;
}

int commandResponseTokenise(char* c, char* r, int t, int n, int t2, char* k, int s){  //do command/response, parse csv response for nth entry, put in r of length s
  int tok=0;  
  boolean inquotes=0;           //ignore commas inside quotes
  int i,d,rptr;  
  unsigned long t2out;
  rptr=0;
  for(i=0;i<s;i++){k[i]=0;}     //clear result
  if(commandResponse(c,r,t)){   //command succeeded
    t2out=millis();             //times are compounded
    while(millis()-t2out<t2){
      if(MODEM.available()){
        d=MODEM.read();
        if(d=='"'){inquotes=!inquotes;}   //toggle quote state
        if((d==',')&&(!inquotes)){        //if comma outside quotes
          tok++;                          //next token
        }else{
          if(tok==n){                     //otherwise check if correct spot
            if(rptr<s-2){                 //and if it fits
              if(d>31){
                k[rptr]=d;                  //add to output
                rptr++;
              }
            }
          }          
        }
        if(d<32){                       //EOL
          if(tok>=n){
            return 1;                   //we found n commas
          }else{
            return 0;                   //didn't find that token
          }
        }
      }
    }    
  }
  return 0;   //fail
}

int tokenise(int n, int t2, char* k, int s){  //parse csv response for nth entry, put in r of length s
  int tok=0;  
  boolean inquotes=0;           //ignore commas inside quotes
  int i,d,rptr;  
  unsigned long t2out;
  rptr=0;
  for(i=0;i<s;i++){k[i]=0;}     //clear result
  t2out=millis();             //times are compounded
  while(millis()-t2out<t2){
    if(MODEM.available()){
      d=MODEM.read();
      if(d=='"'){inquotes=!inquotes;}   //toggle quote state
      if((d==',')&&(!inquotes)){        //if comma outside quotes
        tok++;                          //next token
      }else{
        if(tok==n){                     //otherwise check if correct spot
          if(rptr<s-2){                 //and if it fits
            if(d>31){
              k[rptr]=d;                  //add to output
              rptr++;
            }
          }
        }          
      }
      if(d<32){                       //EOL
        if(tok>=n){
          return 1;                   //we found n commas
        }else{
          return 0;                   //didn't find that token
        }
      }
    }
  }
  return 0;
}

void stripQuotes(char* s){        //remove trailing/leading quotes from s
  int n=strlen(s);
  int i;
  if(n<2){return;}
  if(s[0]='"'){
    if(s[n-1]='"'){
      for(i=1;i<n;i++){
        s[i-1]=s[i];
      }
      s[n-2]=0;
    }
  }
}

void purgeModem(){
  while(MODEM.available()){MODEM.read();}
}

int httpGet(char* urlToSend){        //wrapper for everything needed
  if(!commandResponseTokenise("AT+CGNAPN","+CGNAPN:",100,1,200,token,TOKENSIZE)){
    DEBUG.println(F("APN failed"));
    return 0;
  }
  //APN is in token
  if(strlen(token)<2){
    DEBUG.println(F("APN empty"));
    return 0;    
  }
  purgeModem();
  MODEM.print(F("AT+CSTT="));   //set APN in CSTT
  MODEM.print(token);
  MODEM.println(F(",\"\",\"\""));
  if(!parseFor("OK",200)){
    DEBUG.println(F("CSTT fail"));
    deInitHttp();
    return 0;    
  }
  purgeModem();
  MODEM.print(F("AT+SAPBR=3,1,\"APN\","));
  MODEM.println(token);
  if(!parseFor("OK",200)){
    DEBUG.println(F("Set bearer fail"));
    deInitHttp();
    return 0;    
  }
  if(!commandResponse("AT+SAPBR=1,1","OK",2000)){    //turn on bearer
    DEBUG.println(F("Activate bearer fail"));
    deInitHttp();
    return 0;    
  }
  if(!commandResponse("AT+HTTPINIT","OK",2000)){    //init HTTP
    DEBUG.println(F("HTTPINIT fail"));
    deInitHttp();
    return 0;    
  }
  MODEM.print("AT+HTTPPARA=\"URL\",\"");
  MODEM.print(urlToSend);
  delay(200);
  purgeModem();
  MODEM.println("\"");
  if(!parseFor("OK",200)){
    DEBUG.println(F("URL set fail"));
    deInitHttp();
    return 0;    
  }  
  if(!commandResponse("AT+HTTPACTION=0","OK",2000)){    //make GET request
    DEBUG.println(F("GET fail"));
    deInitHttp();
    return 0;    
  }
  if(!parseFor("+HTTPACTION:",20000)){
    DEBUG.println(F("no response to GET"));
    deInitHttp();
    return 0;        
  }
  tokenise(1,200,token,TOKENSIZE);
  DEBUG.println(token);
  deInitHttp();
  return 1;
}

void deInitHttp(){              //shut down so next try is set up correctly
  MODEM.println(F("AT+HTTPTERM"));
  delay(200);
  MODEM.println(F("AT+SAPBR=0,1"));  
  delay(200);
  MODEM.println(F("AT+CIPSHUT"));  
  delay(200);
  purgeModem();
}

