'''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
'
' Air Quality Sensor V1
' Geoff Graham, October 2019
'
' Requires MMBasic 5.2 or later and an ILI9341 based LCD panel with touch
' Plus a CCS811 VOC sensor with a HDC1800 temperature and humidity sensor
'
'''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''

Option Autorun on
Option Explicit
Const true = 1, false = 0

' define the colours used in the graph
Const CValue = RGB(cyan)
Const CGraph = RGB(white)
Const CGraphBorder = RGB(200, 128, 0)
Const CGraphBackground = RGB(0, 0, 64)
Const CGraphCapton = RGB(255, 104, 255)

Const hRdg = 200                      ' h posit of reading
Const vGraph = MM.VRes -  20          ' v posit of the graph baseline
Const GraphHeight = vGraph - 64
Const HScale = 300                    ' nbr of horiz points in graph
Const hGraph = MM.HRes - HScale - 1   ' h posit of the graph LH side

' various horizontal graph scaling values
' each value is the number of seconds for each pixel in the graph
Dim Integer hMaxValue(11) = (3, 6, 12, 24, 48, 96, 192, 288, 576, 1152, 2304, 4608)

' these variables are restored with VAR RESTORE
' the initialisations set the defaults for the first run of the firmware
Dim Integer VScale = 500, hMax = 0, InitMessage = false
Dim Integer LCDdimactive = false, LCDbright = 100, LCDdimtime = 2, LCDdimval = 10
Dim Integer Baseline(2)

' global variables for holding the current data from the sensors
Dim Float CO2, VOC, Temp, Humid, aVOC(hScale)

' general global variables
Dim Integer HeadVOC, TailVOC, ValidReading, IncrementGraph
Dim Integer i, x, y, Dimmed

' the device mode, 1 = default graph, 2 = config screen
Dim Integer mode = 1

' global variables used to manage and update on screen buttons
Const NbrBtn = 6
Dim Float bx(NbrBtn), by(NbrBtn), bw(NbrBtn), bh(NbrBtn)
Dim Integer bfc(NbrBtn), bbc(NbrBtn)
Dim String bs(NbrBtn)

' start running the program
CLS
Font 8, 1
VAR Restore
PWM 2, 250, LCDbright
InitSensors

' first time startup warnings
If InitMessage = false Then
  Text 160, 10, "NEW CCS811 SENSOR", C
  If CCS811GetVer() = "1.1.0" Then
    Text 0, 40, "- Upgrade the sensor"
    Text 0, 64, "  firmware"
    i = 54
  EndIf
  Text 0, i + 40, "- Run sensor for"
  Text 0, i + 64, "  minimum 48 hours"
  Text 0, i + 94, "- Then set Baseline"
  InitBtn 0, 90, 185, 140, 38, RGB(yellow),RGB(104,104,104), "OK"
  Do
    WatchDog 4000
    If CheckBtnDown(0) Then
      WaitBtnUp(0)
      Exit Do
    EndIf
  Loop
  InitMessage = true
  VAR SAVE InitMessage
Else
  Text 160, 46, "CCS811 VOC", CM
  Text 160, 73, "Sensor", CM
  Text 160, 115, "Warm Up Time", CM
  InitBtn 0, 90, 185, 140, 38, RGB(yellow),RGB(104,104,104), "Ignore"
  Do While Timer < 1200000
    WatchDog 4000
    i = (1200000 - Timer) \ 1000
    If CheckBtnDown(0) Then
      WaitBtnUp(0)
      Exit Do
    EndIf
    Text 160, 142, " " + Str$(i \ 60, 1, 0) + ":" + Str$(i Mod 60, 2, 0, "0") + " ", CM
  Loop
EndIf

' get the baseline from flash and write it to the CCS811
CCS811SetBaseline

CLS
DrawMainScreen true

' the main program loop
Do
  GetReadings
  Select Case mode
    Case 1                     ' maintain the graph
      If Touch(x) <> -1 Then
        If Dimmed Then
          PWM 2, 250, LCDbright
          Do While Touch(x) <> -1 : GetReadings : Loop
          SetTick LCDdimtime * 60000, AutoDim
          Dimmed = false
        Else
          mode = 2
          DrawSetupScreen true
          Do While Touch(x) <> -1 : WatchDog 4000: Loop
        EndIf
      Else
        DrawMainScreen false
      EndIf
    Case 2                     ' do the setup screen
      DrawSetupScreen false
      If CheckBtnDown(0) Then  ' user is changing the vert scale
        VScale = VScale * 2
        If VScale > 64000 Then VScale = 500
        bs(0) = Str$(VScale)
        DrawBtn bx(0), by(0), bw(0), bh(0), bbc(0), bfc(0), bs(0)
        VAR SAVE VScale
        WaitBtnUp(0)
      EndIf
      If CheckBtnDown(1) Then  ' user is changing the time scale
        hMax = hMax + 1
        If hMax > 11 Then hMax = 0
        bs(1) = GetHScaleString()
        DrawBtn bx(1), by(1), bw(1), bh(1), bbc(1), bfc(1), bs(1)
        For i = 0 To HScale : aVOC(i) = -1 : Next i
        VAR SAVE hMax
        WaitBtnUp(1)
      EndIf
      If CheckBtnDown(2) Then
        WaitBtnUp(2)
        SetBaseline
      EndIf
      If CheckBtnDown(3) Then
        WaitBtnUp(3)
        Firmware
      EndIf
      If CheckBtnDown(4) Then
        WaitBtnUp(4)
        Brightness
      EndIf
      If CheckBtnDown(5) Then
        WaitBtnUp(5)
        mode = 1
        IncrementGraph = true
        DrawMainScreen true
      EndIf
  End Select
Loop
End


''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
' one time initialisation of both the sensors and I2C channel
Sub InitSensors
  Local Integer i, j, d(4)

  On Error Skip
  I2C Close
  I2C Open 100, 100

  ' setup CCS811
  I2C Write &H5A, 0, 5, &HFF, &H11, &HE5, &H72, &H8A  ' reset the chip
  Pause 150
  I2C Write &H5A, 0, 1, &H20      ' address the hardware reg
  I2C Read &H5A, 0, 1, i          ' read the register
  If MM.I2C <> 0 Or i <> &H81 Then
    ErrorBox "ERROR", "CCS811 not", "responding"
    Exit Sub
  EndIf
  I2C Write &H5A, 0, 1, &Hf4      ' start app
  Pause 10
  I2C Write &H5A, 0, 2, 1, &H18   ' mode 1 with interrupt on data ready
  Pause 10
  I2C Write &H5A, 0, 1, 0         ' address the status reg
  I2C Read &H5A, 0, 1, i          ' read the status
  If (i And &B10010001) <> &H90 Then
    I2C Write &H5A, 0, 1, &HE0    ' address the error reg
    I2C Read &H5A, 0, 1, j        ' read the error
    ErrorBox "Faulty CCS811", "Status: " + Hex$(i, 2), "Error: " + Hex$(j, 2)
    Exit Sub
  EndIf

  ' setup HDC1800
  I2C Write &H40, 0, 1, &HFF      ' address the hardware reg
  I2C Read &H40, 0, 2, d()        ' read the register
  If MM.I2C <> 0 Or d(0) <> &H10 Or d(1) <> &H50 Then
    ErrorBox "ERROR", "HDC1080 not", "responding"
    Exit Sub
  End If
  I2C Write &H40, 0, 3, 2, &B00010000, 0   ' 14 bit temp & humidity

  ' clear history
  For i = 0 To HScale : aVOC(i) = -1 : Next i
End Sub


''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
' draw and maintain the main screen (ie, the graph)
' if DrawAll is true the whole screen will be drawn
' otherwise just the current reading and graph will be updated
Sub DrawMainScreen DrawAll As integer
  Local Integer x, y1, y2, y3, y, h1, i
  Local String vs1, vs2, hs
  If DrawAll Then
    If LCDdimactive Then
      SetTick LCDdimtime* 60000, AutoDim
      PWM 2, 250, LCDbright
    EndIf
    CLS
    Text hRdg + 4, 25, "ppb", LB, 8, 1, CValue
    Text hRdg + 4, 25, "VOC", LT, 8, 1, CValue
    Text 0, vGraph- GraphHeight - 28, "x", B, 1, 1, CGraphCapton
    Text 0, vGraph- GraphHeight - 15, "1000", B, 1, 1, CGraphCapton
    Box hGraph - 2, vGraph - GraphHeight - 1, HScale + 3, GraphHeight + 3, 1, CGraphBorder, CGraphBackground
    Text 8, vGraph, "0", CM, 1, 1, CGraphCapton
    If vScale = 500 Then
      vs1 = ".5" : vs2 = " "
    Else If vScale = 1000 Then
      vs1 = "1" : vs2 = ".5"
    Else
      vs1 = Str$(vScale/1000) : vs2 = Str$(vScale/2000)
    EndIf
    Text 8, vGraph - GraphHeight - 1, vs1, CM, 1, 1, CGraphCapton
    Text 8, vGraph - GraphHeight/2 - 1, vs2, CM, 1, 1, CGraphCapton

    Text hGraph, MM.VRes, GetHMarkString(1), CB, 1, 1, CGraphCapton
    Text hGraph + HScale*0.75, MM.VRes, GetHMarkString(0.25), CB, 1, 1, CGraphCapton
    Text hGraph + HScale*0.5,  MM.VRes, GetHMarkString(0.5), CB, 1, 1, CGraphCapton
    Text hGraph + HScale*0.25, MM.VRes, GetHMarkString(0.75), CB, 1, 1, CGraphCapton
    Text MM.HRes, MM.VRes, GetHMarkString(0), RB, 1, 1, CGraphCapton
  EndIf
  If ValidReading Then
    Text hRdg - 4, 0, Str$(VOC, 5, 0), R, 7, 1, CValue
    ValidReading = false
  EndIf
  If IncrementGraph Then
    IncrementGraph = false
    Line hGraph - 1, vGraph, hGraph - 1, vGraph - GraphHeight, 2, CGraphBackground
    i = NextVOC(HeadVOC) : x = hGraph
    y1 = -1
    y2 = Min(aVOC(PrevVOC(i))/vScale * GraphHeight, GraphHeight)
    Do While i <> HeadVOC
      If y1 >= 0 And y2 >= 0 Then
        Line x - 1, vGraph - y1, x, vGraph - y2, 1, CGraphBackground
      EndIf
      If y2 >= 0 And aVOC(i) >= 0 Then
        y3 = Min(aVOC(i)/vScale * GraphHeight, GraphHeight)
        Line x - 1, vGraph - y2, x, vGraph - y3, 1, CGraph
      EndIf
      y1 = y2 : y2 = y3
      i = NextVOC(i) : x = x + 1
    Loop
  EndIf
End Sub


' used by the DrawMainScreen sub to find the next value in the circular buffer
Function NextVOC(i As Integer) As Integer
  NextVOC = i + 1
  If NextVOC > HScale Then NextVOC = 0
End Function


' used by the DrawMainScreen sub to find the previous value in the circular buffer
Function PrevVOC(i As Integer) As Integer
  PrevVOC = i - 1
  If PrevVOC < 1 Then PrevVOC = HScale
End Function




''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
' draw and maintain the setup screen
' if DrawAll is true the whole screen will be drawn
' otherwise just the current readings will be updated
Sub DrawSetupScreen DrawAll As Integer
  If DrawAll Then
    SetTick 0, 0
    PWM 2, 250, LCDbright
    Dimmed = false

    CLS
    Text 6*16, 0, "VOC       CO", , 8, 1, CValue
    Text 5*16, 32, " `C        %RH", , 8, 1, CValue
    Text 18*16, 24, "2",LB , 1, 1, CValue
    Text 0, 75, "Vert Scale", , 8, 1, RGB(255, 104, 255)
    Text 0, 113, "Time Scale", , 8, 1, RGB(255, 104, 255)
    InitBtn 0, 176, 69, 140, 32, RGB(255, 104, 255), RGB(black), Str$(VScale)
    InitBtn 1, 176, 110, 140, 32, RGB(255, 104, 255) , RGB(black), GetHScaleString()

    InitBtn 2, 5, 153, 145, 38, RGB(yellow),RGB(104,104,104), "Baseline"
    InitBtn 3, 170, 153, 145, 38, RGB(yellow),RGB(104,104,104), "Firmware"
    InitBtn 4, 5, 203, 176, 38, RGB(yellow),RGB(104,104,104), "Brightness"
    InitBtn 5, 218, 203, 96, 38, RGB(white) , RGB(black), "Exit"
  EndIf
  If ValidReading Then
    Text 0, 0, Str$(VOC, 5, 0), , 8, 1, CValue
    Text 9*16, 0, Str$(CO2, 6, 0), , 8, 1, CValue
    Text 0, 32, Str$(Temp, 5, 0), , 8, 1, CValue
    Text 9*16, 32, Str$(Humid, 6, 0), , 8, 1, CValue
    ValidReading = false
  EndIf
End Sub



''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
' implements the BASELINE screen
Sub SetBaseline
  Local Integer i, timeout, donereset
  CLS
  Text 160, 0,   "SET CCS811 BASELINE", C, 8, 1, RGB(yellow)
  Text 160, 38,  "Place the sensor", C
  Text 160, 62,  "outside in clean air", C
  Text 160, 86,  "then press START", C
  Text 160, 110, "This can take 30 min", C
  InitBtn 0, 100, 150, 120, 38, RGB(green),RGB(black), "START"
  InitBtn 1, 100, 203, 120, 38, RGB(red) , RGB(black), "Abort"
  Do While Touch(x) <> -1 : WatchDog 4000 : Loop
  Do
    GetReadings
    If CheckBtnDown(0) Then
      WaitBtnUp(0)
      Box 0, 36, MM.HRes, 164, 1, 0, 0
      Text 160, 100, "Wait Time", CM
      If Timer < 20 * 60 * 1000 Then timeout = (20 * 60 * 1000) - Timer
      timeout = timeout + Timer + 10 * 60 * 1000
      Do While Timer < timeout
        WatchDog 4000
        i = (timeout - Timer) \ 1000
        If CheckBtnDown(1) Then
          WaitBtnUp(1)
          DrawSetupScreen true
          Exit Sub
        EndIf
        Text 160, 132, " " + Str$(i \ 60, 1, 0) + ":" + Str$(i Mod 60, 2, 0, "0") + " ", CM
        If i < 550 And Not donereset Then
          donereset = true
          InitSensors
        EndIf
      Loop
      CCS811GetBaseline  ' get the baseline and save it to our flash
      For i = 0 To HScale : aVOC(i) = -1 : Next i
      CLS
      mode = 1
      DrawMainScreen true
      Exit Sub
    EndIf
    If CheckBtnDown(1) Then
      WaitBtnUp(1)
      DrawSetupScreen true
      Exit Do
    EndIf
  Loop
End Sub


''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
' implements the FIRMWARE screen
Sub Firmware
  Local Integer d%(8)
  Local Integer t, cs, i, j, x, count, cnt
  Local String s, ver = CCS811GetVer()

  CLS
  Text 160, 0,   "CCS811 FIRMWARE", C, 8, 1, RGB(yellow)
  Text 160, 28,  "Current Ver " + ver, C, 8, 1, RGB(yellow)
  If ver <> "2.0.1" Then
    Text 160, 58,  "To upgrade to", C
  Else
    Text 160, 58,  "To reload", C
  EndIf
  Text 160, 82,  "Ver 2.0.1", C
  Text 160, 106,  "press UPGRADE", C
  InitBtn 0, 84, 150, 152, 38, RGB(green),RGB(black), "UPGRADE"
  InitBtn 1, 100, 203, 120, 38, RGB(white) , RGB(black), "Exit"
  Do While Touch(x) <> -1 : WatchDog 4000 : Loop
  Do
    GetReadings
    If CheckBtnDown(0) Then
      WaitBtnUp(0)
      Exit Do
    EndIf
    If CheckBtnDown(1) Then
      WaitBtnUp(1)
      DrawSetupScreen true
      Exit Sub
    EndIf
  Loop
  Box 0, 26, MM.HRes, 164, 1, 0, 0
  Text MM.HRes/2, MM.VRes/2, "0%", CM
  d(0) = &HF2
  For t = 1 To 2
    If t = 2 Then
      I2C write &H5A, 0, 5, &HFF, &H11, &HE5, &H72, &H8A  ' reset the chip
      Pause 150
      I2C write &H5A, 0, 5, &HF1, &HE7, &HA7, &HE6, &H09  ' erase
      Pause 300
    EndIf
    Restore
    Read count, cs
    cnt = 0
    Do
      WatchDog 4000
      Read s
      For i = 1 To Len(s) Step 16
        For j = 0 To 7
          d(1 + j) = Val("&H" + Mid$(s, i + j * 2, 2))
        Next j
        ' send data
        x = x + d(1)+ d(2) + d(3)+ d(4) + d(5)+ d(6) + d(7)+ d(8)
        If t = 2 Then
          I2C write &H5A, 0, 9, d()
          Text MM.HRes/2, MM.VRes/2, Str$(cnt/count * 100, 1, 0) + "%", CM
          Pause 10
        EndIf
        cnt = cnt + 8
      Next i
    Loop Until cnt >= count
    If t = 1 And x <> cs Then
      ErrorBox "Corrupt BASIC", "program", "cannot update"
      CPU Restart
    End If
  Next t

  Pause 100
  Baseline(2) = false
  VAR SAVE Baseline()
  InitSensors

  Text 160, 28,  "Current Ver " + CCS811GetVer(), C, 8, 1, RGB(yellow)
  Text MM.HRes/2, MM.VRes/2 - 24, "Firmware Upgraded", CM, 8, 1, RGB(green)
  Text MM.HRes/2, MM.VRes/2,      "Next do 48h burn in", CM, 8, 1, RGB(green)
  Text MM.HRes/2, MM.VRes/2 + 24, "Then set Baseline", CM, 8, 1, RGB(green)
  InitBtn 1, 100, 203, 120, 38, RGB(white) , RGB(black), "Exit"
  Do
    WatchDog 4000
    If CheckBtnDown(1) Then
      WaitBtnUp(1)
      mode = 2
      DrawSetupScreen true
      Exit Do
    EndIf
  Loop
End Sub



''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
' implements the BRIGHTNESS screen
Sub Brightness
  Local Integer br = LCDbright, dimactive = LCDdimactive
  Local Integer dimtime = LCDdimtime, dimbr = LCDdimval
  Local Integer x, y
  Const cbsize = 24, cx = 0, cy = 90

  CLS
  Text 160, 0,   "LCD BRIGHTNESS", C, 8, 1, RGB(yellow)
  Text 0, 50,  "Current Value", , 8, 1, RGB(white)
  InitBtn 0, MM.HRes - 101, 46, 100, 32, RGB(255, 104, 255), RGB(black), Str$(br) + "%"
  Box cx, cy, cbsize, cbsize, 1, RGB(white), RGB(black)
  Text cx + 10 + cbsize, cy,  "Auto Dimming", , 8, 1, RGB(white)
  If dimactive Then
    Line cx, cy, cx + cbsize, cy + cbsize
    Line cx + cbsize, cy, cx, cy + cbsize
    Text cx + 10 + cbsize, cy + 35,  "Timeout", , 8, 1, RGB(white)
    Text cx + 10 + cbsize, cy + 70,  "Dim Level", , 8, 1, RGB(white)
    InitBtn 1, MM.HRes - 101, cy + 31, 100, 32, RGB(255, 104, 255), RGB(black), Str$(dimtime) + " m"
    InitBtn 2, MM.HRes - 101, cy + 66, 100, 32, RGB(255, 104, 255), RGB(black), Str$(dimbr) + "%"
  EndIf
  InitBtn 3, 40, MM.VRes - 40, 110, 38, RGB(green) , RGB(black), "Save"
  InitBtn 4, 171, MM.VRes - 40, 110, 38, RGB(green) , RGB(black), "Cancel"
  Do
    GetReadings
    If CheckBtnDown(0) Then
      WaitBtnUp(0)
      br = br + 10
      If br > 100 Then br = 10
      InitBtn 0, MM.HRes - 101, 46, 100, 32, RGB(255, 104, 255), RGB(black), Str$(br) + "%"
      PWM 2, 250, br
    EndIf
    x = Touch(x) : y = Touch(y)
    If x >= cx And x <= cx + cbsize And y >= cy And y <= cy + cbsize Then
      dimactive = Not dimactive
      If dimactive Then
        Line cx, cy, cx + cbsize, cy + cbsize
        Line cx + cbsize, cy, cx, cy + cbsize
        Text cx + 10 + cbsize, cy + 35,  "Timeout", , 8, 1, RGB(white)
        Text cx + 10 + cbsize, cy + 70,  "Dim Level", , 8, 1, RGB(white)
        InitBtn 1, MM.HRes - 101, cy + 31, 100, 32, RGB(255, 104, 255), RGB(black), Str$(dimtime) + " m"
        InitBtn 2, MM.HRes - 101, cy + 66, 100, 32, RGB(255, 104, 255), RGB(black), Str$(dimbr) + "%"
      Else
        Box cx, cy, cbsize, cbsize, 1, RGB(white), RGB(black)
        Box 0, cy + 30, MM.HRes, 78, 1, RGB(black), RGB(black)
      EndIf
      Do While Touch(x) <> -1 : WatchDog 4000 : Loop
    EndIf
    If dimactive Then
      If CheckBtnDown(1) Then
        WaitBtnUp(1)
        dimtime = dimtime * 2
        If dimtime > 64 Then dimtime = 1
        InitBtn 1, MM.HRes - 101, cy + 31, 100, 32, RGB(255, 104, 255), RGB(black), Str$(dimtime) + " m"
      EndIf
      If CheckBtnDown(2) Then
        WaitBtnUp(2)
        dimbr = dimbr + 10
        If dimbr > br Then dimbr = 0
        InitBtn 2, MM.HRes - 101, cy + 66, 100, 32, RGB(255, 104, 255), RGB(black), Str$(dimbr) + "%"
      EndIf
    EndIf
    If CheckBtnDown(3) Then
      WaitBtnUp(3)
      If dimbr >= br Then dimbr = br : LCDdimactive = false
      LCDdimactive = dimactive : LCDbright = br : LCDdimtime = dimtime : LCDdimval = dimbr
      VAR Save LCDdimactive, LCDbright, LCDdimtime, LCDdimval
      Exit Do
    EndIf
    If CheckBtnDown(4) Then
      WaitBtnUp(4)
      Exit Do
    EndIf
  Loop
  mode = 2
  DrawSetupScreen true
End Sub


''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
' utility functions for the CCS811
''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''

Function CCS811GetVer() As String
  Local Integer d(2)
  I2C write &H5A, 0, 1, &H24      ' address the firmware reg
  I2C read &H5A, 0, 2, d()        ' read the register
  CCS811GetVer = Str$(d(0) >> 4) + "." + Str$(d(0) And &B1111) + "." + Str$(d(1))
End Function


Sub CCS811GetBaseline
  I2C write &H5A, 0, 1, &H11      ' address the baseline reg
  I2C read &H5A, 0, 2, Baseline() ' read the register
  Baseline(2) = true
  VAR Save Baseline()
End Sub


Sub CCS811SetBaseline
  If Baseline(2) = true Then
    I2C write &H5A, 0, 1, &H11       ' address the baseline reg
    I2C write &H5A, 0, 2, Baseline() ' write the data
  EndIf
End Sub


Sub GetReadings
  Static float ave, count
  Local Integer d(4), t
  Local integer i, j

  WatchDog 4000
  I2C Write &H5A, 0, 1, 0
  I2C Read &H5A, 0, 1, t
  If MM.I2C <> 0 Or Not (t And &H08) Then Exit Sub

  ' get CO2 and VOC
  I2C Write &H5A, 0, 1, &H02
  I2C Read &H5A, 0, 4, d()
  CO2 = (d(0) << 8) + d(1)
  VOC = (d(2) << 8) + d(3)

  ' now get the temp and humidity
  I2C Write &H40, 0, 1, 0
  Pause 20
  I2C Read &H40, 0, 4, d()
  Temp = (d(0) * 256) + d(1)
  Temp = ((Temp/(1<<16))*165)-40
  Humid = (d(2) *256) + d(3)
  Humid = (Humid/(1<<16))*100

  ' write temp and humidity to the CCS811
  d(0) = Int(Humid * 2) : d(1) = 0        ' load the humidity
  D(2) = Int((Temp + 25) * 2) : d(3) = 0  ' and the temperature
  I2C Write &H5A, 0, 1, &H05              ' address the environment register
  I2C Write &H5A, 0, 4, d()               ' write the data

  ' accumulate the VOC value
  ave = ave + VOC
  count = count + 1

  ' if we have the required number of readings average and store the value
  If count >= hMaxValue(hMax) Then
    aVOC(HeadVOC) = ave/count
    ave = 0 : count = 0
    HeadVOC = NextVOC(HeadVOC)
    IncrementGraph = true
  EndIf
  ValidReading = true
End Sub


''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
' general utility routines
''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''


' dim the display if auto dimming is set
Sub AutoDim
  SetTick 0, 0
  PWM 2, 250, LCDdimval
  Dimmed = true
End Sub


' returns the current horizontal scaling as a string
Function GetHScaleString() As String
  If hMax < 3 Then
    GetHScaleString = Str$(hMaxValue(hMax) * 5) + " min"
  Else If hMax < 7 Then
    GetHScaleString = Str$(hMaxValue(hMax) / 12) + " hours"
  Else If hMax = 7 Then
    GetHScaleString = Str$(hMaxValue(hMax) / 288) + " day"
  Else
    GetHScaleString = Str$(hMaxValue(hMax) / 288) + " days"
  EndIf
End Function


' get the horizontal scaling number as a string
Function GetHMarkString(mark As Float) As String
  If (mark = 0.25 Or mark = 0.75) And (hMax <= 1 Or hMax = 7) Then Exit Function
  If hMax < 3 Then
    GetHMarkString = Str$(hMaxValue(hMax) * 5 * mark) + "m"
  Else If hMax < 7 Then
    GetHMarkString = Str$(hMaxValue(hMax) / 12* mark) + "h"
  Else
    GetHMarkString = Str$(hMaxValue(hMax) / 288* mark) + "d"
  EndIf
End Function


Sub ErrorBox s1 As String, s2 As String, s3 As String
  RBox 20, 40, MM.HRes - 40, MM.VRes - 80,,RGB(magenta), RGB(black)
  Text MM.HRes/2, MM.VRes/2 - 48, s1, CM
  Text MM.HRes/2, MM.VRes/2 - 24, s2, CM
  Text MM.HRes/2, MM.VRes/2, s3, CM
  InitBtn 0, 40, MM.VRes - 92, 110, 38, RGB(red) , RGB(black), "Ignore"
  InitBtn 1, 171, MM.VRes - 92, 110, 38, RGB(green) , RGB(black), "Reboot"
  Do
    WatchDog 4000
    If CheckBtnDown(0) Then
      WaitBtnUp(0)
      CLS
      Exit Do
    EndIf
    If CheckBtnDown(1) Then
      WaitBtnUp(1)
      CPU Restart
    EndIf
  Loop
End Sub




''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
' functions dedicated to managing on screen buttons
' see "Getting Started with the Micromite" for a description
''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''

Sub DrawBtn x, y, w, h, fc As Integer, bc As Integer, s As String
  RBox x, y, w, h, , fc, bc
  Text x + w/2, y + h/2, s, CM, , , fc, bc
End Sub

Sub InitBtn n, x, y, w, h, fc As Integer, bc As Integer, s As String
  bx(n) = x : by(n) = y : bw(n) = w : bh(n) = h
  bfc(n) = fc : bbc(n) = bc : bs(n) = s
  DrawBtn x, y, w, h, fc, bc, s
End Sub

Function CheckBtnDown(n)
  Local x = Touch(x), y = Touch(y)
  If x > bx(n) And x < bx(n)+bw(n) And y > by(n) And y < by(n)+bh(n) Then
    DrawBtn bx(n), by(n), bw(n), bh(n), bbc(n), bfc(n), bs(n)
    CheckBtnDown = 1
  EndIf
End Function

Sub WaitBtnUp(n)
  Do While Touch(x) <> -1 : WatchDog 4000 : Loop
  DrawBtn bx(n), by(n), bw(n), bh(n), bfc(n), bbc(n), bs(n)
End Sub




''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
' Ver 2.0.1 firmware for the CCS811
''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
Data  5112, 655753   ' the number of bytes and the checksum
Data "704EF41E2DA78D4FFC88804902DC1710B00E5A648C35B88981D4ACB38FA3529B8A8FA0A47E19763EB249B952E5F61E693CB1445F91A2DCD22D36352E99D1441B"
Data "0547C006E4BC914121256FB11E5E811EF6FA591DCE74ED1517E7D02DDED46C7219DB890E05763D9EB6DAF90854B00D4A9F2CDAC7CEDA09DD3841F995CC91828F"
Data "99C3FB14CD77629680654CFBE269ACD0B30F3037B62EC7394408E186EAE45D2157A0E6C98B287F5341B8AE1E4B32D5ED2904835C16B5F83DF5E4A8E2F0C90890"
Data "E4AB0360CBF8CA2FF3505D9D2830323B596A70F10077B3A2D87BB3A7D8711CEC320CE5C7645AA97B72D51631E7D971D9BD425C9167C251BED85A3C842409DCC2"
Data "DBB98E63C38C94D4D0FEF6C2B22CECD2565265452424325199D2808702FF713B9B31A787AD771E17253B3A599E2FC5FFDC8E64A85AC0F840E609AD030BF4E6E3"
Data "73F9919DB03059D36B4EA472D65D9BD6838ACCF64545A070E10BB2AB015951CB917B940FFBAD74968306A1B28B8BD5E6987EECCB3813AF000DB72024A1F545DC"
Data "A2BB2FB0AD904AC6C893F810243DB487F5CD27DDAF3C35165E0D388EE3397CD6F428C4711CE630F9EF017DF4794C51937C78C12911CBB5C17F3DEAF66BDA02E5"
Data "78566721E2D8BE7C09FA1DC9E464D3955B0C9EE101BC28BD1FF43AEC15D9096202EEA1578B30D13D4056C099470668999A907DAC827524EBCC4EA2B5EB2CACCC"
Data "878A196F70A2C6DAEF7D2B18FDF24071BC5868797BD886B076593AF965FC3634307A795569D2B03F17740B0F28B6463535774F830EE3DD2DA1014C7283D2D360"
Data "297697999C03730A1DFBB9D0ED6C17154075D3677D28F4109A427AD60F87A5C2DAC293B9A523FC1D6263AE6165F6BCA513A29C0FBA28045DE3984F5306ED7030"
Data "651E1A72BCE3322D2139EE09BA3D0CD986018BF43340A4820B0029CA390FD36E5BFC22DF6D663235458F3B1DBAA8D68D43604C9CA683BC668F5198557A995476"
Data "C1929C3BE63A67609B80D5FC258DF2206FDFFBC612C1DAD548B440BE348186B4394F06A83E1768CE01CF2264FB8804D873DACB0FF0A7B09E1B543E40D3C8CC17"
Data "2E1954696BE94D2B4054BB75EC75FFECAE60586CE66ABCFF9532E0760E87C48301FF749158B719FD0B644FFD836C85D8846B676FFB409320619C885175F9EEDD"
Data "D224C0F40165A4B4B9B44BCD309377F4C2AAE23BA5B975C0EDD10BEA63B4FAE74D465683A36450D129E565EFB327146EE3CD0F19075588918B5FDA219CAC928B"
Data "B4FA1AB3EE1204D38E3A70D2A9862FAA8D22D39F82CFBE9B8835B09E3BAC16783AC84D690561CB8797C52DCA4F81CCA714A788F6D0F70EEBD39DF5D3A89E73B4"
Data "74A1FD8EEA72DC24201A70A1923B42E024BF57B11792DF48FB912F09969D2A9FC9F878D01EF614031C7E1E0EB374A43E43487AA89447D56F13C9D73A0D90E7A8"
Data "D4AC51AB923EF9EBF38D0A2575D9B9F5C7AE9AEE3EACA81FCE9E872E06660865419037110AB292562F6F8A11FB8B34B5FA7C1E57CF95394795EA08F41F5F5665"
Data "EC01E0407609029EB3B828EFEE56426568537272A1C57366A08CB931E918E468C1684C7FFBF0ED5939A273E45413453A5BA6C6C48750B7239E40191091CEBAF4"
Data "B4ED663E97BBB1AD1B6E1C1D1B885FF7AC6CCFE494E5B0699EC33910C5016BD60A3124FE755DD0F99F5EA0071037D834B93265F209317C4AC086130537C52D4C"
Data "FE32C8600F69578BD9E95DFC0FFCC6021FC2BA792C52C6D89FF13377EE48C385631ED8E78842B2EB3737891F3B0EF35641256ED06D1D1F9A3B9797B52F02AA3F"
Data "71A4BC0AA0C7AEB8E8087AED64EE68EF29C04A07B803AAE3E7296BD0406E19A4F57DCB75972F64C6C9782E69BEBFBB1022767CB1ECAA4CB398156CD9735B8B73"
Data "A404B167DB7F83AE77A830B54396CF4171CF3FDD3655CB8CD8DFB3CFF114EB87CC6E90FED0CF3A5A6B07E4C4B78CA330FCEC6632DB3BF667018873BB64FEEC2B"
Data "E33C6E9BA56D22F91C7F1C10EA16AFCF18684B8C2A4583AED8534F73B134CAE77955BDE338230688BB8F5BD430408F7CD312928575B0A5445BECE76D45BE4F81"
Data "25EE7825C6C7BB6E8C1B21A292A7C20D14726AFC44EAB90EA9201D7568CD0A43239AF57E316E5FC86DADA1F6687CB2A7C5AD7569D5D23401D6C16DBE8F69D529"
Data "50AE2FA42C689674A128B0F095A578BCC0A6A5399534BBE37E9F973DC0681B78021357CFFA68DBEE6B8D63F6200EBF1750F1936A657E33D3704F6ACE0EB8F0EE"
Data "19CAA2E99078421A50D486F1F513821AE439CAB479D62111DEFFCB52877A1DB81AC44B3FC9DAD8DAAD6C82D4870D876FB8525877B3A8F5E2DF3113125AAE3FAF"
Data "85D8E38ED2A35002D37C8A973958F4CA397D56AE60CAB5CE9181485E3C435EBB5186269E296F5E57C2BA3AEA0F2EA73A73A09F9696F459788A5AF03E8A2B4AD2"
Data "DA5580AFD9D5A6C3929DB540038D74BB355E6EA8D58444242B0D78503A354594F56F938544C5D225EBDCF311EEEC335BC4063AE1B4273A63818279C56D2249C8"
Data "1E2EEC569B6C73A7E6BABCC8E6716D39B46A3F6F1BA6CE319EA07B2273E971E5CBD6BF692F1B68262F57D8C97744D3FACFA57E39E0A7A0587F206CABD9BE9FAA"
Data "5556DCF7F0C5CDD8A9367EFBF8628311749AB0398A9871ACE3074217207EEFDCEA2F6431BA08727C8B8030B830A8521D98ACA1F0FED340476372E47934A5967C"
Data "7F345C741194828F7B7B02BA9CB30F24532BAC05F39FA75D84614511FF3A740A9421969E58BED1139209825F7F663B5D2B012463E2D545F1B9CDB3D15098E7FC"
Data "140C7CA4519CDBAEC5F1979CCFA7B67C0B593E2FF64C24C43A35EDC4A5C8CD4515213C1BC00B0122848B945682A9BC03C54F12B3E0A4E1B98D10663EEB61C921"
Data "79C1C3265EC5B05FA28776AE350A54D6099F0A2E98FFB598FF85D9006DF305268C63EF695F5C8278A275425F6F73835BC22F0272F4F67B2FC5C0B584BC4630FD"
Data "F7A110BF368C909466071F57B461B5B18A989E5C0AB653B56C01A8ECE1C35021BD6CE436C91C9D4A36FA216096596F2DB8758E45038CC5EE0C306082995B4B79"
Data "CD68B3C0E31A6B343D951106569529617B5E86A91A5F4D98689C8D10913CBF4C0DBC8BF78B850CF66E7E82685C7AF25787D9AFE50FF1B9F77EEFC474BD01307E"
Data "F49326086EE9D74E1431E99DB3BFEB2086102C0D94E9B31DA554952B360B906AC786BF09661200B33B2DF110009B94EB0685413C750F0075283BF8555C74AA5B"
Data "CF885466FB979F23BC7963577558C49B39E93B34331CCC751B3EB9B750A9BF90C5B9E3312352B9CDDA526FA41E6F1D01920A2B97830703870B71F823234A239F"
Data "F6E3B174D17C4158FA4BB258BF004CC19ABD17AF77EE27632AC0D681CEBA66000F69EAD225F6CCD85ADA5ED0371156ECB20499BD359FA6D92E4DA78180146EE6"
Data "B7C0FC1CFA4C5ABE332E81B7D216CC8457729A0CA61EC7C5641866A03EF79BC6D99DEB4FAA2B89F3D2027F7CD7F5002C92A6515BE44371A4AF22A10CEC51FF72"
Data "BE1F48A5D63B73852AA2E21ECAE008E3A397E3FE9BE18AA1B43F81D734596546A86D990773A418814942F383E08179E43FC69B5E63430CFA2762635A5A7ABCA4"
Data "622E099FF0F91B560907A518537E8A7BA22FC1F79BC28A8C1509B9FD6E5E4F479C30251D359A1771F277816B301026A681355DD096E17B7D9E8C5C283A9DD9E6"
Data "4AE6CE089FB63FAE0A80F990629D04F03854773E22C2E1F4A5B3884AC9D3DC75B8936F756F6B6E3C64D23A6E7B3C2339677894925D8738117C99975F2FFE4F4D"
Data "2E1CB6C2B22D3570DEC96CC5D935BBD5DBFA5AF8E0F082333D87F654B32536D85D5F3C962FD88A809A6963C95200367A6E25A41B8BB2F649CD939E8C50D52D5E"
Data "859569C75359733DD2FB40EABC74E02179859557CD5910D2D2DD059879E00C92F5CF0BA6E20CE355D7D2F312A7F2C81C70215B8B4D9CCD2801F16A0CB1B55598"
Data "F40BFA40D9E57DFF0DA34604BA28A6788AE1E3CFD97B623169A5B389DAE67BE35D6A8CB48508741A7FD349F631F05EF14654C20930A9ECABB4BFE95E5BD34545"
Data "657895C563BCC02D2A6ACCF1DB9EE8426683391138CF3FAB6C8286D76B734E4A5099F1F156B57550F4080B417D1D21C6A2DE524BC2A96B5B135BEEDF7ED1EFE6"
Data "6CB6FF1D54CBDF3FE348BBEF428F3A793709321FC7AD2B7497761A6D23612A93E3DB146D5E3956E7420D6D5E9DC9D4509F19FD7F6339DAB1D173B42B9494C3F1"
Data "63FB97D01505A1234537892A294AA055749F926ED54EC8F30E20EDD5F102FC89E0B3BE014FEE87B1B6160979502DFCF91E1B1CF4051FA8D1FBE8DB842DB3F696"
Data "69D895C52D52E29DB4C132450FECFBBC8CBAEC9052C69CCE06F7384836038595B0C80DD7EFD8D1622EF28E507502DF147C915A7DD215A52C87B8DF91E1750735"
Data "A4DBB0D347320593799D60CECDC3C20D3B422D1BFF3DCB86E34B5B359871E7FC32C909D2A45F5BDB6E90DEBF31BB1DFD0108E50C6D5286ACD1FE956E317F45A1"
Data "9318D955C2C6C009BCBFC2385123069C2A79C3A7F484A66B0952FC8B3CADCBFF58A0612BD3CB9697504E3A1F209F799B9213CD1CC9DF624D44927747AEA53FF1"
Data "685228F8D25883E3E1140C85FF0E3E54E5F1E0ADB73117E82F1FF13C829CF13C81F26C3615C6F0D3AF4DD6AA462F759D8F4EAC100D5C7D4F9E3587D8EF28F19D"
Data "D26AC0B52BA3BFA259C6AC7BE982465C78ED4497021D4E10F6FD2A81D287F71C4D6B93629215CF31C3566F44674711CE982F721D5D75A4D47C694D08ED6C8902"
Data "C90B3F2A56DD307CFD4BBB36B516A07FC570E58C1A443EFCD708441396AD525AA7B9D67D0154079F360C30E3E7AC4560840E355C90947C0C76B4E4652C3C6089"
Data "34F24F60FBBAFF22D52446FC809F14CB508E64C59BB2C07EA6EDADB1129BD7EF4F33663C6EC368BA9EBF7FD95DBB2EA68FBF5702C625881BA57C3C46E4C12E31"
Data "86B4C14FCB6F577C8DE2B32E9CDF6E828889FD62079FEE8520E5DD6959C7C7A887C974524DE4302E2D7264F8D26D1670506A01ED18F0F5F766EBF3F4BAC68FBE"
Data "AD3569F8135E1AD1358DEA1F53D886736C2ACB74A42DA26554E2296D24A54B6FFE12E784CFF75DC8B520362528E469118C79AB4653067E1D9386EA8F03D1514D"
Data "9C734059C8667BC6A02535F1031BC7C5B7F5288AF775107DE4E7E382BE356723B765BFE1895EEF1CF61737B195139C8A698C168566279ABCC38AF3F9A2D35EB5"
Data "8706C84B0F483213F1C3DD8124F857A1CEDFEB7246046F6B6B4EC32A84987D051F0B2A7ED63BC0A65CDFC13BF8EB7FE62A6BFBA735EEFB53C9773246E7ADC877"
Data "FB0686D4A4742FA477ACA89B710D022A547A867175BEB2B839E5C06487304AC1AF32A30293B30FAC53BBEE88F6CD05F9ACFEE0AB9A7059076E6B7EB076D42A4E"
Data "92E9E1DAF6BC91030D3FEE8AD07254B3492E40A21B61FACA47D26BA4E073E31C56949FD048870F7702F627C12112E8E8F032D85D850E2247A2DF794EA8A8CB4D"
Data "D64A8BA635D31BDE98D38A928168AE82F9729892840C7C75500B5031D468D4069361ACF3984CE321300E99138D8E03C9794F9337025A25701F063D912F7315BF"
Data "630F7A225B5BFA7ACD86DCC0B75CA8BF3CE4BF9913BC6CE2073A2E4FD0EEB36C8644C96F3A0EC55F84D42A8F97A5568364B72B2170711A63D40EF0D9830F7DE4"
Data "23DD59195158D28A6C774A87BD72CBB3B41542608416532A262C1D839B7867E64E49311498F595B39F3F3F772BBB267BB0E71A93E823FD3D870AB3034F9D077F"
Data "F694D9EB44A3A58A9D631C29E96F8E7A051FB560D71C88CA279ED6920BA31E3E4E07D40F1C2ADDB11A150869AE4EAED9D2E88FD978EE370BF343BAC7C203DEEE"
Data "A7BE87F6564B0BD6ACE08DC341A281D16C040460CC40ABAA29822152C4D5FB2BFF5F3EF27216D7F479CB1D2923F5F490EE559F5E4876C69D1CB878503B81F067"
Data "BE48D90A16A70E3364B9798204C60C294B7D1D40E65514AD4216C073A26CA015E6C3E859A028FE513480E3D30CD0828ED260E684E87FDF2AD26D00CEE4BE71F4"
Data "F48D331D99C412167414434D25E8D78AF1EC369714110090E87891BF07FF708D58A5C5F7339DF8430FC3763B8B0A78A4A0627DE88C4EFFCBA72C17F4F3B01528"
Data "58B998BAE51C16D642965E3B0ED0DE83D5D068F3D506087EECF7FCF4B9B931034C3B82A594A78E74B64696B52479DA703A6F2B8A05BBFFF87B7BBC1984A28F16"
Data "074C6D808DE5DF34D851825BC713126E2C99E39AF103AE19B97D8941E9CC78972CADCEC703E5A1930914F4FB04317272059EC1DE1F26B24B18CF1F4B903A0815"
Data "BFAD404C58287641DEBB75E95DA15AE8E8701D44E866F8201B00778201EBBCC3B38D36DC9D2A2A087180D86BB954EE58A00863EC55C6B3F59B890E1638AFFFF7"
Data "3E33C5560989E27ABFA251EE033049BE32A324E55B0E95819C588140AD8035BCA3E084269DC374970F521DDB4D7F26DE55B8401DC653C3774D8C00E1D4C1A5A2"
Data "DB069FC54C4B7555300E1E8D015656FD633BAA94CCE89EBDBADA39A8F2021AEB45B430EE458EEA29B9017D11CCD4E8360C44B847903E4275953C2C94C5DAF226"
Data "1B47026F989F1C4F069557C40839214A106BBFD97C63E5909DAB7E1E7F5D26B1559A720B6F430D7E3E1FBD9C2FC47DDAB40E98AA58DB26EFCAC802BE9DA69756"
Data "27F640AA232616E97EF08F5BAAF32C0D2ED8DED3CF8A9EF83C63AE41F3B85DE32F4A4FE3EC1F6A894968AEAB8D25A5BDEE4777387B0A00C4D447A3A3AD868B2F"
Data "61818D7531FDAA2CEB89CE46AF58FF5CE02083A8F8C6B24D0941480A4E7893DC119315D828A990F913191CE99BF7D53A649B15B224E6C4C15C1505E39920951A"
Data "7CD80A3D34A90F41F37B24BFEEE87D6062243123F8D3FEB6ECA5EBD3295C606A8245C4936797248EB7CFAA7A491DFC91FD4CDB1511E20D9A0084D56792DBE36F"
Data "7AA6CDF4EAEF8A59A558D5649A961A9E47F87AC22F283DBF7583CF91E7E0B4EA7BBAA09C93C262AA96AA1798CCEBB2D6EDC36265C590F55F964EC1377519EA96"
Data "3CF19F0CBB012FB0ECB75A5D81913837D2C32EC1DBA7DF49BB56EA21114CCAA28CA881F82ECB1EDD09719DA41A4C21DABCA670D4225119EA0167BB0126FD64CD"
Data "1D32B4FE3E73A426899A7758841ACD02FA7B3036A0717E13139B58FCAA7D33273C73A69308AAEAE480BB5A3D3A7C0F9C302FD402CB40EC91"



''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
' ArialNumFontPlus.bas
' Font type    : Special (SubSet)
' Font size    : 32x50 pixels
' Memory usage : 2204 bytes
DefineFont #7
  0B303220 00000000 00000000 00F00F00 00F81F00 00FE7F00 00FFFF00 00FFFF01
  801FF803 C00FF003 C007E007 E003C007 E003C00F F001800F F001800F F001801F
  F800001F F800001F F800001F F800001F FC00003F 7C00003E 7C00003E 7C00003E
  7C00003E 7C00003E 7C00003E 7C00003E 7C00003E 7C00003E 7C00003E 7C00003F
  F800001F F800001F F800001F F800001F F001800F F001800F F001800F E003C007
  E003C007 C007E003 C00FF003 801FF801 00FFFF00 00FEFF00 00FC7F00 00F81F00
  00E00700 00000000 00000000 00000000 00000000 00000000 00F00000 00F00100
  00F00300 00F00700 00F00F00 00F01F00 00F03F00 00F07F00 00F0FF01 00F0FD03
  00F0F10F 00F0E10F 00F0810F 00F0010F 00F0010C 00F00100 00F00100 00F00100
  00F00100 00F00100 00F00100 00F00100 00F00100 00F00100 00F00100 00F00100
  00F00100 00F00100 00F00100 00F00100 00F00100 00F00100 00F00100 00F00100
  00F00100 00F00100 00F00100 00F00100 00F00100 00F00100 00F00100 00F00100
  00F00100 00F00100 00F00100 00000000 00000000 00000000 00000000 00F00F00
  00FE7F00 00FFFF01 C0FFFF03 E0FFFF03 F00FE007 F003C00F F001800F F800001F
  F800001F F800003E 7C00003E 7C00003E 7C000000 7C000000 7C000000 78000000
  F8000000 F8010000 F0030000 F0070000 E00F0000 C01F0000 803F0000 007F0000
  00FE0000 00FC0100 00F80300 00F00700 00E00F00 00C01F00 00803F00 00007F00
  0000FE00 0000FC01 0000F803 0000F007 0000E007 0000C00F 0000800F 0000801F
  FCFFFF3F FCFFFF3F FCFFFF7F FCFFFF7F FCFFFF7F 00000000 00000000 00000000
  00000000 00F00F00 00FC3F00 00FEFF00 00FFFF01 80FFFF03 C01FE003 C00FC007
  E0078007 E003800F E003000F F003000F F001000F F0030000 E0030000 E0030000
  E0070000 C00F0000 C01F0000 803F0000 00FF0700 00FF0700 00FF0700 80FF0700
  C01F0000 E00F0000 E0030000 F0010000 F0000000 F8000000 F8000000 FC000000
  FC000000 FC000000 FC00001F FC00001F F800001F F801001F F801800F F003800F
  F007C00F E00FE007 C03FFC03 80FFFF01 00FFFF00 00FC7F00 00F00F00 00000000
  00000000 00000000 00000000 000F0000 001F0000 001F0000 003F0000 003F0000
  007F0000 007F0000 00FF0000 00FF0100 00FF0100 00FF0300 00DF0700 00DF0700
  009F0F00 009F0F00 001F1F00 001F3E00 001F7E00 001F7C00 001FF800 001FF800
  001FF001 001FE003 001FE003 001FC007 001FC007 001F800F 001F801F 001F001F
  001F003E FCFFFF3F FCFFFF3F FCFFFF3F FCFFFF3F FCFFFF3F 001F0000 001F0000
  001F0000 001F0000 001F0000 001F0000 001F0000 001F0000 001F0000 001F0000
  001F0000 00000000 00000000 00000000 00000000 F0FFFF01 F0FFFF01 F0FFFF01
  F0FFFF01 F0FFFF01 0000E003 0000C003 0000C003 0000C003 0000C007 0000C007
  0000C007 0000C007 00008007 0000800F 00F0800F 00FE8F0F 80FFBF0F C0FFFF0F
  E0FFFF0F F007FF1F F003E01F F801C01F F800001F F8000000 FC000000 7C000000
  7C000000 7C000000 7C000000 7C000000 7C000000 7C000000 F800003F F800003F
  F800003F F000001F F001001F E003801F E007C00F C01FE007 80FFFF03 00FFFF03
  00FEFF01 00F87F00 00E00F00 00000000 00000000 00000000 00000000 00F00F00
  00FE3F00 00FFFF00 80FFFF01 C0FFFF03 E007E007 F003C007 F001C00F F800800F
  F800801F F800801F 0000001F 0000001F 0000003F 0000003F 0000003F 00F0033F
  00FC1F3F 00FE7F3F 80FFFF3F C0FFFF3F E00FF83F F003E03F F003E03F F801C03F
  F800803F F800803F 7C00003F 7C00003F 7C00003F 7C00003F 7C00003F 7C00003F
  7C00001F 7800801F F800801F F800801F F800800F F001C00F F003F007 E00FF803
  C0FFFF01 80FFFF00 00FF7F00 00FC1F00 00F00300 00000000 00000000 00000000
  00000000 FCFFFF3F FCFFFF3F FCFFFF3F FCFFFF3F FCFFFF3F FC000000 FC000000
  F8010000 F0010000 E0030000 E0030000 C0070000 800F0000 001F0000 003E0000
  007C0000 00FC0000 00F80000 00F00000 00F00100 00F00100 00E00300 00E00300
  00E00300 00C00700 00C00700 00800F00 00800F00 00800F00 00001F00 00001F00
  00001F00 00001F00 00003E00 00003E00 00003E00 00003E00 00003E00 00007C00
  00007C00 00007C00 00007C00 00007C00 0000F800 0000F800 0000F800 00000000
  00000000 00000000 00000000 00F00F00 00FE7F00 80FFFF00 80FFFF01 C01FF803
  E007E007 E003C007 F003C00F F001800F F001800F F001800F F001800F F001800F
  F001800F E003C007 E003C007 C007E003 8007F001 003FFC00 00FF7F00 00FC3F00
  00FE7F00 00FFFF00 C0FFFF03 E00FF007 F003C00F F001800F F801801F F800001F
  F800001F 7C00003E 7C00003E 7C00003E 7C00003E 7C00003E 7C00003E F800001F
  F800001F F001800F F003800F E007C007 C00FF003 80FFFF01 00FFFF00 00FC3F00
  00F00F00 00000000 00000000 00000000 00000000 00F00F00 00FC3F00 00FFFF00
  80FFFF01 C0FFFF03 E01FF803 E007E007 F007E007 F003C00F F803C00F F801801F
  F801801F FC00001F FC00001F FC00001F FC00001F FC00001F FC00001F FC01801F
  FC01801F FC03C00F FC03C00F FC07E007 FC07E007 FC1FF803 FCFFFF01 FCFFFF00
  FCFEFF00 FCF87F00 FCE01F00 FC000000 FC000000 FC000000 FC000000 F8010000
  F801003F F001003F F003001F E007801F E00FC00F C03FE00F 80FFFF07 00FFFF03
  00FEFF01 00F8FF00 00E03F00 00000000 00000000 00000000 00000000 00000000
  00000000 00000000 00000000 00000000 00000000 00000000 00000000 00000000
  00000000 00E00000 00F00100 00F80300 00F80300 00F80300 00F00100 00E00000
  00000000 00000000 00000000 00000000 00000000 00000000 00000000 00000000
  00000000 00000000 00000000 00000000 00000000 00000000 00E00000 00F00100
  00F80300 00F80300 00F80300 00F00100 00E00000 00000000 00000000 00000000
  00000000 00000000 00000000 00000000 00000000 00000000 00000000
End DefineFont



''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
' Arial_round_16x24.bas
' Font type    : Full (95 characters)
' Font size    : 16x24 pixels
' Memory usage : 4564 bytes
DefineFont #8
  5F201810 00000000 00000000 00000000 00000000 00000000 00000000 00000000
  00000000 00000000 00000000 00000000 00000000 C0038001 C003C003 C003C003
  C003C003 8003C003 80018001 00008001 80010000 C003C003 00008001 00000000
  00000000 00000000 30063006 600C600C 700E700E 00003006 00000000 00000000
  00000000 00000000 00000000 00000000 00000000 00000000 18031803 38073807
  FC3FFC3F 700E7006 600E700E FC3FFC3F E01CE01C C018C01C 0000C018 00000000
  00000000 80018001 F01FC007 B83DF81F 9839B839 801F803D F803F00F 9C31BC01
  9C399C39 F81FB83D 8001E007 80018001 00000000 10780000 20CC30F8 40CC60CC
  80CCC0CC 3C018079 66026603 660C6606 3C18660C 00003C10 00000000 00000000
  00000000 00000000 00000000 C00F8007 E01CE01C C00FE01C 081F0007 D8719C3B
  F070F871 DE3FFC78 00008C0F 00000000 00000000 C000E000 8001C001 00000000
  00000000 00000000 00000000 00000000 00000000 00000000 00000000 00000000
  00000000 C0004000 8001C001 80038003 00070007 00070007 00070007 00070007
  80038003 C0018001 4000C000 00000000 00000000 00030002 80018003 C001C001
  E000E000 E000E000 E000E000 E000E000 C001C001 80038001 00020003 00000000
  00000000 00000000 00000000 00000000 C0000000 C000C000 F807D806 E001C000
  10033003 00000000 00000000 00000000 00000000 00000000 00000000 00000000
  80018001 80018001 F81FF81F 80018001 80018001 00000000 00000000 00000000
  00000000 00000000 00000000 00000000 00000000 00000000 00000000 00000000
  00000000 80030003 80018003 00038001 00000000 00000000 00000000 00000000
  00000000 00000000 F00FF00F 00000000 00000000 00000000 00000000 00000000
  00000000 00000000 00000000 00000000 00000000 00000000 00000000 00000000
  00000000 80038003 00008003 00000000 00000000 70003000 60007000 E000E000
  C000E000 C001C001 8001C001 80038003 00038003 00070007 00000006 00000000
  00000000 E0070000 F81FF00F 1C38381C 1C381C38 1C381C38 1C381C38 1C381C38
  381C1C38 F00FF81F 0000E007 00000000 00000000 60000000 E001E000 E007E003
  E01CE01F E000E018 E000E000 E000E000 E000E000 E000E000 0000E000 00000000
  00000000 E0070000 F81FF81F 1C383C3C 1C001C38 78003800 E003F000 000F8007
  003C001E FC3FF83F 0000FC3F 00000000 00000000 E0070000 F81FF00F 3838783C
  78003800 F001F001 3C00F800 1C381C30 783C3C38 F00FF81F 0000E007 00000000
  00000000 60000000 F000F000 F003F001 70077003 700C700E 7038701C FC3F7038
  7000FC3F 70007000 00007000 00000000 00000000 FC1F0000 F81FFC1F 001C001C
  E01F001C F81FF01F 1C003C1C 1C001C00 383C1C38 F01FF83F 0000E007 00000000
  00000000 E0030000 F81FF00F 183C381C 00380038 F03FE03B 3C3CF83F 1C381C38
  3C1C1C38 F00FF81F 0000E003 00000000 00000000 FC3F0000 FC1FFC3F 70003800
  E0007000 C001C001 80038003 80078003 00070007 00070007 00000006 00000000
  00000000 E0070000 F81FF00F 381C781E 781E381C F00FF00F 1C38381C 1C381C38
  3C3C1C38 F00FF81F 0000E007 00000000 00000000 C0070000 F81FF00F 1C38383C
  1C381C38 FC1F3C3C DC07FC0F 1C001C00 381C3C18 F00FF81F 0000C007 00000000
  00000000 00000000 00000000 80030000 80038003 00000000 00000000 00000000
  80038003 00008003 00000000 00000000 00000000 00000000 00000000 00000000
  80030000 80038003 00000000 00000000 00000000 80030003 80018003 00038001
  00000002 00000000 00000000 38000800 F803F800 001FC00F 001F001C F803C00F
  3800F800 00000800 00000000 00000000 00000000 00000000 00000000 00000000
  00000000 FC3FFC3F 0000FC3F FC3F0000 FC3FFC3F 00000000 00000000 00000000
  00000000 00000000 00100000 001F001C F003C01F 3800F800 F003F800 001FC01F
  0010001C 00000000 00000000 00000000 00000000 F00FC007 783CF01F 38383838
  70003830 E001F000 8003C003 00018003 80030000 80038003 00008003 00000000
  00000000 00000000 00000000 F80FE003 04101C1C F227B623 625C724C 62586258
  EC586458 7027F86F 0C180630 F003FC0F 00000000 00000000 C0038001 E007C003
  600EE007 700E700E 381C781C FC3F381C 1C38FC3F 0E700E70 00000E60 00000000
  00000000 00000000 F03FE03F 38387838 38383838 F03F3838 3838F03F 1C381C38
  1C381C38 F83F3C38 0000F03F 00000000 00000000 00000000 F00FE007 3C3C781E
  0C781C38 00700070 00700070 0C700C70 3C3C1C38 F00F781E 0000E007 00000000
  00000000 00000000 F03FC03F 7838F038 1C383838 1C381C38 1C381C38 1C381C38
  78383838 F03FF038 0000C03F 00000000 00000000 00000000 FC1FFC1F 001C001C
  001C001C F81F001C 001CF81F 001C001C 001C001C FC1F001C 0000FC1F 00000000
  00000000 00000000 FC0FFC0F 000E000E 000E000E F80F000E 000EF80F 000E000E
  000E000E 000E000E 0000000E 00000000 00000000 00000000 F80FE003 1C383C1C
  0C700E38 00700070 FE70FE70 0E700E70 0E380E38 FC0F3E1C 0000F003 00000000
  00000000 00000000 1C381C38 1C381C38 1C381C38 FC3F1C38 1C38FC3F 1C381C38
  1C381C38 1C381C38 00001C38 00000000 00000000 00000000 80038003 80038003
  80038003 80038003 80038003 80038003 80038003 80038003 00008003 00000000
  00000000 00000000 E000E000 E000E000 E000E000 E000E000 E000E000 E070E070
  E079E070 C03FC03F 0000000F 00000000 00000000 00000000 1C380C38 78383C38
  E039F038 C03FC03B F03EE03F 7838703C 3C383C38 0E381E38 00000E38 00000000
  00000000 00000000 001C001C 001C001C 001C001C 001C001C 001C001C 001C001C
  001C001C F81F001C 0000F81F 00000000 00000000 00000000 1FF81FF8 3FFC3FFC
  37EC3FFC 77EE77EE 67E677EE E7E7E7E7 C7E3E7E7 C7E3C7E3 000087E1 00000000
  00000000 00000000 1C3C1C18 1C3E1C3C 1C3F1C3E 9C3B1C3B DC399C39 FC38DC38
  7C387C38 3C383C38 00001C38 00000000 00000000 00000000 F00FE007 3C3CF81F
  1E781C38 0E700E70 0E700E70 1E780E70 3C3C1C38 F00FF81F 0000E007 00000000
  00000000 00000000 F83FF03F 1C383C38 1C381C38 38381C38 E03FF83F 00380038
  00380038 00380038 00000038 00000000 00000000 00000000 F00FE007 3C3CF81F
  0E781C38 0E700E70 0E700E70 9E790E70 7C3CFC38 FE0FF81F 0700EF07 00000000
  00000000 00000000 FC3FF83F 0E381E38 0E380E38 FC3F1E38 E038F03F 78387038
  3C383C38 0E381E38 00000E38 00000000 00000000 00000000 F01FE007 3838381C
  003C1838 E01F003F F801F80F 1C303C00 1C381C38 F81F383C 0000E007 00000000
  00000000 00000000 FC7FFC7F 80038003 80038003 80038003 80038003 80038003
  80038003 80038003 00008003 00000000 00000000 00000000 1C381C38 1C381C38
  1C381C38 1C381C38 1C381C38 1C381C38 383C1C38 F00FF81F 0000E007 00000000
  00000000 00000000 0E380638 1C1C0E38 1C1C1C1C 380E180E 300E380E 70077007
  E0036007 C003E003 0000C001 00000000 00000000 00000000 C3C383C1 C7E3C7E3
  C663C7E3 66666666 6E766E76 3C3C6C36 3C3C3C3C 381C3C3C 00001818 00000000
  00000000 00000000 1E380C30 3C1C1C3C F00F781E E007F007 E007C003 F81FF00F
  3C3C781E 1E781E78 00000E70 00000000 00000000 00000000 1C781830 383C3838
  F01E701C C007E00E 8003C007 80038003 80038003 80038003 00008003 00000000
  00000000 00000000 F83FF83F 78007800 F001F000 C003E001 80078007 001E000F
  003C001E FC7F0078 0000FC7F 00000000 00000000 E0010000 8001E001 80018001
  80018001 80018001 80018001 80018001 80018001 80018001 80018001 E001E001
  00000000 00000000 001E000C 000F000E 00070007 80038007 C0038003 C001C001
  E000E001 F000E000 00007000 00000000 00000000 800F0000 8001800F 80018001
  80018001 80018001 80018001 80018001 80018001 80018001 80018001 800F800F
  00000000 00000000 80078003 C0078007 E00CC00E 601CE01C 00007038 00000000
  00000000 00000000 00000000 00000000 00000000 00000000 00000000 00000000
  00000000 00000000 00000000 00000000 00000000 00000000 FFFF0000 00000000
  00000000 00000000 003E001C 00630063 003E0063 0000001C 00000000 00000000
  00000000 00000000 00000000 00000000 00000000 00000000 00000000 00000000
  C00F0000 7018E01F F0007010 701EF00F 70387038 F01FF038 0000380E 00000000
  00000000 00000000 001C001C 001C001C F01D001C 381EF81F 1C1C1C1E 1C1C1C1C
  1C1E1C1C F81F381E 0000E01D 00000000 00000000 00000000 00000000 00000000
  C0070000 701CF00F 1038383C 00380038 383C1038 F00F781C 0000C007 00000000
  00000000 00000000 38003800 38003800 B80F3800 781CF81F 38387838 38383838
  78383838 F81F781C 0000B807 00000000 00000000 00000000 00000000 00000000
  C0070000 701CE00F 38383838 F03FF83F 10380038 F00F381C 0000C007 00000000
  00000000 00000000 E007E003 00070007 C01F0007 0007C01F 00070007 00070007
  00070007 00070007 00000007 00000000 00000000 00000000 00000000 00000000
  B80F0000 781CF81F 38383838 38383838 38383838 F81F781C 3810B807 701C3838
  C007F01F 00000000 00380038 00380038 C03B0038 F03CE03F 70387038 70387038
  70387038 70387038 00007038 00000000 00000000 00000000 80038003 00008003
  80030000 80038003 80038003 80038003 80038003 80038003 00008003 00000000
  00000000 00000000 80038003 00008003 80030000 80038003 80038003 80038003
  80038003 80038003 80038003 80038003 001F801F 00000000 000E000E 000E000E
  180E000E 700E380E C00FE00E 700FE00F 380E700E 1C0E380E 00000C0E 00000000
  00000000 00000000 80038003 80038003 80038003 80038003 80038003 80038003
  80038003 80038003 00008003 00000000 00000000 00000000 00000000 00000000
  9EEF0000 E7F1FFFF C7E1C7E1 C7E1C7E1 C7E1C7E1 C7E1C7E1 0000C7E1 00000000
  00000000 00000000 00000000 00000000 E01D0000 781EF01F 381C381C 381C381C
  381C381C 381C381C 0000381C 00000000 00000000 00000000 00000000 00000000
  800F0000 E038C01F 70707070 70707070 70707070 C01FE038 0000800F 00000000
  00000000 00000000 00000000 00000000 C03B0000 703CF03F 3838383C 38383838
  383C3838 F03F703C 0038C03B 00380038 00380038 00000000 00000000 00000000
  B8070000 781CF81F 38387838 38383838 78383838 F81F781C 3800B807 38003800
  38003800 00000000 00000000 00000000 E00E0000 100FF00F 000E000E 000E000E
  000E000E 000E000E 0000000E 00000000 00000000 00000000 00000000 00000000
  C0070000 701CF00F 001F301C F003E00F 381C7800 F00F381E 0000E007 00000000
  00000000 00000000 80038003 80038003 E0078003 8003E00F 80038003 80038003
  80038003 F0038003 0000F001 00000000 00000000 00000000 00000000 00000000
  381C0000 381C381C 381C381C 381C381C 381C381C F80F781E 0000B807 00000000
  00000000 00000000 00000000 00000000 18180000 381C381C 700E300C 6006700E
  C003E007 C003C003 00008001 00000000 00000000 00000000 00000000 00000000
  83C10000 C7E3C7E3 E667C663 6C366E76 7C3E6C36 381C381C 0000381C 00000000
  00000000 00000000 00000000 00000000 38380000 F01E783C C007E00F E00FC007
  F01EE00E 3838783C 00003830 00000000 00000000 00000000 00000000 00000000
  18180000 381C381C 700E300C 6006700E E0076007 C003C003 8001C003 80038003
  001E001F 00000000 00000000 00000000 F83F0000 7800F83F E001F000 8007C003
  000E0007 F83F001E 0000F83F 00000000 00000000 F0007000 C001F001 C001C001
  C001C001 8003C001 80078007 C0018003 C001C001 C001C001 F001C001 7000F000
  00000000 80038003 80038003 80038003 80038003 80038003 80038003 80038003
  80038003 80038003 80038003 80038003 00000000 001E001C 0007001F 00070007
  00070007 80030007 C003C003 00078003 00070007 00070007 001F0007 001C001E
  00000000 00000000 081E0000 F83FF83F 0000F020 00000000 00000000 00000000
  00000000 00000000 00000000 00000000 00000000
End DefineFont                               