#include <WiFiS3.h>
#define SSID "****"
#define PASS "****"
//this in minutes
#define STD_TZ_OFFSET 600

//Some colours:
#define RED 255,0,0
#define GREEN 0,255,0
#define BLUE 0,0,255
#define CYAN 0,255,255
#define YELLOW 255,255,0
#define MAGENTA 255,0,255
#define WHITE 255,255,255
//set colour here, brightness is also a scale of 0-255
#define CLOCK_COLOUR CYAN
#define BRIGHTNESS 70

//these in minutes
#define NTP_RETRY_INTERVAL 15
#define NTP_UPDATE_INTERVAL 15
#define NTP_UPDATE_LIMIT 60

unsigned int localPort = 2390;      // local port to listen for UDP packets
const int NTP_PACKET_SIZE = 48; // NTP timestamp is in the first 48 bytes of the message
byte packetBuffer[NTP_PACKET_SIZE]; //buffer to hold incoming and outgoing packets
WiFiUDP Udp;
#define NTP_SERVER "pool.ntp.org"
//#define NTP_SERVER IPAddress(162, 159, 200, 123)

#include <Adafruit_NeoPixel.h>
#define LED_PIN A0
#define LED_COUNT 120
Adafruit_NeoPixel strip(LED_COUNT, LED_PIN, NEO_GRB + NEO_KHZ800);

struct seg7_t {
  unsigned int n; //number of pixels/segment
  int a;  //start pixel of segment a, <0 for absent
  int b;
  int c;
  int d;
  int e;
  int f;
  int g;
};

struct rgbCol_t{
  uint8_t r;
  uint8_t g;
  uint8_t b;
};

unsigned char segMap[]={
    0b00111111, // 0
    0b00000110, // 1
    0b01011011, // 2
    0b01001111, // 3
    0b01100110, // 4
    0b01101101, // 5
    0b01111101, // 6
    0b00000111, // 7
    0b01111111, // 8
    0b01101111, // 9
    0b01110111, // A
    0b01111100, // b
    0b00111001, // C
    0b01011110, // d
    0b01111001, // E
    0b01110001, // F
    0b01010100  //n, for 'on'
};
seg7_t digits[5]={    //mapping pixels to segments
  {5,-1,5,0,-1,-1,-1,-1},   //1/4 digit, b&c segments only
  {5, 35, 40, 15, 20, 25, 30, 10},
  {5,-1,-1,-1,-1,-1,-1,45}, //dash in g segment only
  {5, 75, 80, 55, 60, 65, 70, 50},
  {5,110,115, 90, 95,100,105, 85},
};

rgbCol_t f={CLOCK_COLOUR};
rgbCol_t b={0,0,0};

int wifiStatus=0;
int timeStatus=0; 

unsigned long ntpUpdate=-(60000UL*NTP_UPDATE_LIMIT);  //from millis(), time when update occurred
unsigned long lastNtpFetch=-(60000UL*NTP_RETRY_INTERVAL);  //from millis(), limit time between attempts
#include "RTC.h"

bool dstActive=0;
#define SWITCH_PIN A2

void setup() {
  int i;
  Serial.begin(115200);
  RTC.begin();
  pinMode(SWITCH_PIN,INPUT_PULLUP);
  WiFi.begin(SSID,PASS);
  Udp.begin(localPort);
  strip.begin(); //start LEDs
  strip.setBrightness(BRIGHTNESS); //scale is 255
  for(i=0;i<LED_COUNT;i++){strip.setPixelColor(i,CLOCK_COLOUR);}  //on, lamp test
  strip.show();
  delay(2000);    //time for lamp test, also allow serial to reconnect
  for(i=0;i<LED_COUNT;i++){strip.setPixelColor(i,0,0,0);}  //off, lamp test done
  strip.show();
}

void loop() {
  static unsigned int tLast=0;
  char d;
  int serialData;
  int newStatus;
  RTCTime now;
  newStatus=WiFi.status();
  if(newStatus!=wifiStatus){
    switch(newStatus){
      case WL_CONNECTED:
        Serial.println("Connected");
        Serial.print("IP Address: ");
        Serial.println(WiFi.localIP());
        Serial.println("Checking UTP on connection");
        checkNTP();
        break;
      case WL_NO_MODULE: Serial.println("No WiFi hardware detected"); break;
      case WL_CONNECT_FAILED: Serial.println("Connection failed"); break;
      case WL_CONNECTION_LOST: Serial.println("Connection lost"); break;
      case WL_DISCONNECTED: Serial.println("Disconnected"); break;
      case WL_IDLE_STATUS: Serial.println("Idle (no connection)"); break;
      default: Serial.println("Status not known"); break;
    }
    wifiStatus=newStatus;
  }  
  if((millis()-ntpUpdate)>(60000UL*NTP_UPDATE_LIMIT)){   //time is too stale to be accurate
    timeStatus=0;
  }
  if(((millis()-ntpUpdate)>(60000UL*NTP_UPDATE_INTERVAL))||(timeStatus==0)){   //refresh time if needed
    if((millis()-lastNtpFetch)>(60000UL*NTP_RETRY_INTERVAL)){ //retry if time not set
      Serial.println("Auto update: Checking UTP");
      RTC.getTime(now); 
      Serial.print("Time is ");
      Serial.println(now.toString()); 
      checkNTP();
    }
  }
  if((millis()%1000)>500){d=0;}else{d=64;}  //blinking dash
  setStripSeg(&strip,digits[2],d,f,b);
  if(timeStatus){   //time OK, show time
    RTC.getTime(now);    
    unsigned int h,m;
    m=now.getMinutes();
    h=((now.getHour()+11)%12)+1; //wrapping for 12 hour clock
    if(h>9){
      setStripSeg(&strip,digits[0],segMap[1],f,b);
    }else{
      setStripSeg(&strip,digits[0],0,f,b);    //leading zero blank
    }
    setStripSeg(&strip,digits[1],segMap[h%10],f,b);  
    setStripSeg(&strip,digits[3],segMap[(m/10)%10],f,b);
    setStripSeg(&strip,digits[4],segMap[(m/1)%10],f,b);
    if((h*60+m)!=tLast){ //show time as it changes
      String t=now.toString();
      Serial.print("Time is ");
      Serial.println(t);      
      tLast=(h*60+m); //should catch any change
    }
  }else{            //time not valid
    if(wifiStatus){ //wifi OK
      setStripSeg(&strip,digits[0],0,f,b);  //off
      setStripSeg(&strip,digits[1],0,f,b);
      setStripSeg(&strip,digits[3],segMap[0xE],f,b);
      setStripSeg(&strip,digits[4],segMap[0x1],f,b); //E1
    }else{          //wifi not OK
      setStripSeg(&strip,digits[0],0,f,b);  //off
      setStripSeg(&strip,digits[1],0,f,b);
      setStripSeg(&strip,digits[3],segMap[0xE],f,b);
      setStripSeg(&strip,digits[4],segMap[0x0],f,b); //E0
    }
  }
  strip.show();
  if(digitalRead(SWITCH_PIN)==LOW){
    setStripSeg(&strip,digits[0],0,f,b);  //off
    setStripSeg(&strip,digits[1],0,f,b);
    setStripSeg(&strip,digits[2],0,f,b);
    setStripSeg(&strip,digits[3],0,f,b);
    setStripSeg(&strip,digits[4],0,f,b);
    strip.show();
    delay(1000);
    while(digitalRead(SWITCH_PIN)==LOW) //wait for release
    delay(100);                         //crude debounce
    if(dstActive){  //turn off
      setStripSeg(&strip,digits[1],segMap[0x0],f,b);  //0
      setStripSeg(&strip,digits[3],segMap[0xF],f,b);  //F
      setStripSeg(&strip,digits[4],segMap[0xF],f,b);  //F
      strip.show();
      changeDST(0);
      delay(1000);
    }else{  //turn on
      setStripSeg(&strip,digits[3],segMap[0],f,b);  //0
      setStripSeg(&strip,digits[4],segMap[16],f,b); //n
      strip.show();
      changeDST(1);
      delay(1000);
    }
  }
  if(Serial.available()){
    serialData=Serial.read();
    if(serialData=='u'){
      Serial.println("Manual check NTP");
      checkNTP();
    }
    if(serialData=='r'){
      NVIC_SystemReset(); //software reboot
    }
    if(serialData=='t'){
      timeStatus=0; //invalidate time status
    }
    if(serialData=='0'){
      Serial.println("Daylight savings off");
      changeDST(0);
    }
    if(serialData=='1'){
      Serial.println("Daylight savings on");
      changeDST(1);
    }
  }
}

void setStripSeg(Adafruit_NeoPixel* s, seg7_t p, char bm,rgbCol_t on, rgbCol_t off){
  int i,j;
  if(bm&  1){setPixels(s,p.a,p.a+p.n,on);}else{setPixels(s,p.a,p.a+p.n,off);}
  if(bm&  2){setPixels(s,p.b,p.b+p.n,on);}else{setPixels(s,p.b,p.b+p.n,off);}
  if(bm&  4){setPixels(s,p.c,p.c+p.n,on);}else{setPixels(s,p.c,p.c+p.n,off);}
  if(bm&  8){setPixels(s,p.d,p.d+p.n,on);}else{setPixels(s,p.d,p.d+p.n,off);}
  if(bm& 16){setPixels(s,p.e,p.e+p.n,on);}else{setPixels(s,p.e,p.e+p.n,off);}
  if(bm& 32){setPixels(s,p.f,p.f+p.n,on);}else{setPixels(s,p.f,p.f+p.n,off);}
  if(bm& 64){setPixels(s,p.g,p.g+p.n,on);}else{setPixels(s,p.g,p.g+p.n,off);}
}

void setPixels(Adafruit_NeoPixel* s, unsigned int p0, unsigned int p1, rgbCol_t c){
  int i;
  if(p0>=0){
    for(i=p0;i<p1;i++){s->setPixelColor(i,c.r,c.g,c.b);}
  }  
}

void changeDST(bool dstOn){
  bool curDST;
  RTCTime now,newNow;
  RTC.getTime(now);
  curDST=dstActive;
  if((dstOn) && (!curDST)){ //turn on
    newNow.setUnixTime(now.getUnixTime()+3600); //go forward
    dstActive=1;
    RTC.setTime(newNow);
    Serial.println("Clock set forward");
  }else if((!dstOn) && (curDST)){ //turn off
    newNow.setUnixTime(now.getUnixTime()-3600); //go backward
    dstActive=0;
    RTC.setTime(newNow);
    Serial.println("Clock set back");
  }else{
    Serial.println("No change needed");
  }
}

void checkNTP() {
  unsigned long t0,t1;  
  lastNtpFetch=millis();
  memset(packetBuffer, 0, NTP_PACKET_SIZE);
  packetBuffer[0] = 0b11100011;   // LI, Version, Mode
  packetBuffer[1] = 0;     // Stratum, or type of clock
  packetBuffer[2] = 6;     // Polling Interval
  packetBuffer[3] = 0xEC;  // Peer Clock Precision
  packetBuffer[12]  = 49;
  packetBuffer[13]  = 0x4E;
  packetBuffer[14]  = 49;
  packetBuffer[15]  = 52;
  Udp.beginPacket(NTP_SERVER, 123); //NTP requests are to port 123
  Udp.write(packetBuffer, NTP_PACKET_SIZE);
  Serial.println("Starting NTP check");
  if(Udp.endPacket()){
    Serial.println("UDP packet sent");
  }else{
    Serial.println("UDP packet failed");
    return;
  }
  t0=millis();    //get time
  t1=t0-1;
  while((millis()-t0<2000)&&(t1<t0)){
    if(Udp.parsePacket()){
      t1=millis();    //flag OK
      Serial.print(t1-t0);
      Serial.println("ms round trip. Packet received.");
      Udp.read(packetBuffer, NTP_PACKET_SIZE); // read the packet into the buffer
      unsigned long highWord = word(packetBuffer[40], packetBuffer[41]);
      unsigned long lowWord = word(packetBuffer[42], packetBuffer[43]);
      unsigned long ntpNow = highWord << 16 | lowWord;  
      RTCTime old,now;
      RTC.getTime(old);
      now.setUnixTime(ntpNow-2208988800UL);//convert to Unix time
      Serial.println("Time OK");      
      String t=now.toString();
      Serial.print("UTC is ");
      Serial.println(t);
      now.setUnixTime(ntpNow+(60UL*STD_TZ_OFFSET)-2208988800UL);  //Unix time with TZ offset
      RTC.setTime(now);
      if(dstActive){
        dstActive=0;
        changeDST(1);
      }
      ntpUpdate=t1;
      timeStatus=1;
    }
  }
  if(t1<t0){
    Serial.println("UDP timeout");
  }
}