/*  
 * USB CDC-ACM Demo
 *
 * This file may be used by anyone for any purpose and may be used as a
 * starting point making your own application using M-Stack.
 *
 * It is worth noting that M-Stack itself is not under the same license as
 * this file.
 *
 * M-Stack is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  For details, see sections 7, 8, and 9
 * of the Apache License, version 2.0 which apply to this file.  If you have
 * purchased a commercial license for this software from Signal 11 Software,
 * your commerical license superceeds the information in this header.
 *
 * Alan Ott
 * Signal 11 Software
 * 2014-05-12
  
 * Modified Tim Blythman 2019
 * Serial output is sent to attached HD44780 LCD
 * Some basic control codes implemented
 * can output ascii codes plus pseudo graphics
 * a set of media related graphics is set up by default
 *
 * If USB not enumerated after 1s, falls back to panel meter function
 *  
 */
//#define _XTAL_FREQ 16000000
#include "usb.h"
#include <xc.h>
#include <stdint.h>
#include <string.h>
#include "usb_config.h"
#include "usb_ch9.h"
#include "usb_cdc.h"
#include "hardware.h"
#include "hd44780.h"
#include "io.h"
#include "thermistor.h"

#define VOLTS_DP 1
#define VOLTS_FS 60
#define AMPS_DP 2
#define AMPS_FS 9

#ifdef MULTI_CLASS_DEVICE
static uint8_t cdc_interfaces[] = { 0 };
#endif

#define BUFSIZE 128

char txbuf[BUFSIZE]="";
char rxbuf[BUFSIZE]="";

int txptr=0;
int rxptr=0;

char scrbuf[80];     //virtual screen buffer
int8_t scrptr=0;

static struct cdc_line_coding line_coding =
{
	115200,
	CDC_CHAR_FORMAT_1_STOP_BIT,
	CDC_PARITY_NONE,
	8,
};

void initGraphics(void);
void USBdisplay(void);
void panelMeter(void);
void printNum(long adc, int fs, int dp);    //prints adc result 

int main(void)
{
    uint8_t i;          //trusty counter
    unsigned char j;    //temp character
    char s=0;           //pointer for screen output
	OSCCONbits.IRCF = 0b1111; /* 0b1111 = 16MHz HFINTOSC postscalar */
    ioinit();           //set ANSEL/TRIS etc
    LCD_init();         //sets up IO and everything, do this first so that USB can't time out
    initGraphics();     //sets up pseudo graphics
    //splash screen
    LCD_cmd(0b10000000);         //home cursor
    LCD_print("Silicon Chip USB");
    LCD_cmd(0b10101000);         //start of line 2
    LCD_print("Display & Meter ");
    //USB 
    hardware_init();
#ifdef MULTI_CLASS_DEVICE
	cdc_set_interface_list(cdc_interfaces, sizeof(cdc_interfaces));
#endif
	usb_init();
    for(i=0;i<250;i++){        //can't do simple delay as it might interfere with USB
        if (usb_is_configured()){break;}        //jump out if USB configured
        __delay_ms(4);          //delay and loop for up to a second
    }
    //check for USB and call appropriate looping function
    if (usb_is_configured()){
        USBdisplay();
    }else{
        panelMeter();
    }
}

void panelMeter(void){
    int a;                    //ADC results
    //set up degrees symbol
    LCD_cmd(0b01000000);    //start of CGRAM
    LCD_data(0b00000010);   //  degrees symbol
    LCD_data(0b00000101);   //
    LCD_data(0b00000010);   //
    LCD_data(0b00000000);   //
    LCD_data(0b00000000);   //
    LCD_data(0b00000000);   //
    LCD_data(0b00000000);   //
    LCD_data(0b00000000);   //
    LCD_cmd(0b10000000);    //start of data ram (ready for output)
     while(1){
        LCD_cmd(0b10000000);         //home cursor
        a=getADC(ADC1);
        a=getADC(ADC1);           //twice is nice
        LCD_print("SET: ");
        printNum(a,VOLTS_FS,VOLTS_DP);
        a=getADC(ADC3);
        a=getADC(ADC3);           //twice is nice
        LCD_print("V ");
        printNum(a,AMPS_FS,AMPS_DP);
        LCD_print("A");
        LCD_cmd(0b10101000);         //start of line 2
        a=getADC(ADC5);
        a=getADC(ADC5);           //twice is nice
        //a=thermistor[a];          //convert to degrees
        a=thermistor[a/4];          //convert to degrees (use reduced table)
        if((a>-1)&&(a<100)){
            LCD_data((a/10)%10+'0');    //tens
            LCD_data(a%10+'0');   //units          
        }else{
            LCD_print("--");        //error of some sort
        }        
        LCD_data(0);                //degrees symbol at code point 0
        LCD_print("C ");
        a=getADC(ADC2);
        a=getADC(ADC2);           //twice is nice
        printNum(a,VOLTS_FS,VOLTS_DP);
        a=getADC(ADC4);
        a=getADC(ADC4);           //twice is nice
        LCD_print("V ");
        printNum(a,AMPS_FS,AMPS_DP);
        LCD_print("A");
        __delay_ms(300);
    }
}

void USBdisplay(void){
    int i;      //trusty counter
    unsigned char j;     //temp character
    char s=0;   //pointer for screen output
    for(i=0;i<80;i++){
        scrbuf[i]=' ';  //blank screen buffer to start
    }
    while (1) {
        //process serial buffers into virtual screen buffer
        for(i=0;i<rxptr;i++){
            j=rxbuf[i];
            if((j>31)||(j<8)){                      //allow ascii+pseudo graphics
                scrbuf[scrptr]=j;
                scrptr=scrptr+1;
            }
            if(j==13){                              //CR=start of current line
                if((scrptr>= 0)&&(scrptr<20)){scrptr= 0;}
                if((scrptr>=20)&&(scrptr<40)){scrptr=20;}
                if((scrptr>=40)&&(scrptr<60)){scrptr=40;}
                if((scrptr>=60)&&(scrptr<80)){scrptr=60;}
            }
            if(j==10){scrptr=scrptr+20;}            //LF=same position in next line
            if(j==9){scrptr=((scrptr/5)+1)*5;}      //TAB=move to next multiple of 5
            if(j==12){scrptr=0;}                    //FF=move home
            if(j==8){scrptr=scrptr-1;if(scrptr<0){scrptr=79;}}
            if(j==27){                              //ESC=clear screen (?)
                char t;
                for(t=0;t<80;t++){
                    scrbuf[t]=' ';
                }
            }
            if(scrptr>79){scrptr=scrptr-80;}                //handle wrap
        }
        rxptr=0;        //buffer has been processed
        //output one character per cycle to screen
        if(s== 0){LCD_cmd(128);} //start of 1st line
        if(s==20){LCD_cmd(192);} //start of 2nd line
        if(s==40){LCD_cmd(148);} //start of 3rd line
        if(s==60){LCD_cmd(212);} //start of 4th line
        LCD_data(scrbuf[s]);
        s=s+1;
        if(s>79){
            s=0;
        }
        
		/* Send data to the PC */
		// needs to be populated by user code
        
        // any data in txbuf[] of size txptr is sent here
        
        if (usb_is_configured() &&
		    !usb_in_endpoint_halted(2) &&
		    !usb_in_endpoint_busy(2)) {
			int i,k;
			unsigned char *buf = usb_get_in_buffer(2);
            k=txptr;
            if(k){
                if(k>32){k=32;}     //limit data sent            
                for (i = 0; i < k; i++) {buf[i] = txbuf[i];}
                for(i=0;i<txptr;i++){txbuf[i]=txbuf[i+k];}  //move buffer up
                txptr=txptr-k;
                usb_send_in_buffer(2, k);
            }
		}

		/* Handle data received from the host */
		
        //data is added to rxbuf[]
        
        if (usb_is_configured() &&
		    !usb_out_endpoint_halted(2) &&
		    usb_out_endpoint_has_data(2)) {
			const unsigned char *out_buf;
			size_t out_buf_len;
			int i,k;
			out_buf_len = usb_get_out_buffer(2, &out_buf);
            k=rxptr+out_buf_len;
            if(k>BUFSIZE-2){k=BUFSIZE-2;}
            for(i=rxptr;i<k;i++){rxbuf[i]=out_buf[i-rxptr];}
            rxptr=k;
            usb_arm_out_endpoint(2);
		}
    }
}

void printNum(long adc, int fs, int dp){   //print out scaled number
    int fsa=fs;                            //adjusted full scale for calculations
    int i=dp;
    char zf=0;                             //zero flag for suppress leading zeros
    char d;                                //to be displayed
    int n=1;                               //most sig fig to show
    while(i){                              //adjust full scale for decimal points
        fsa=fsa*10;
        i--;
    }
    if(fsa>9){n=10;}
    if(fsa>99){n=100;}
    if(fsa>999){n=1000;}
    if(fsa>9999){n=10000;}
    adc=(fsa*adc)/1024;                    //now adjusted (10bit ADC)
    while(fs>0){                           //left of dp
        d=adc/n;
        adc=adc%n;
        if(d){zf=1;}
        if(fs<10){zf=1;}
        d=d+'0';
        if(zf==0){d=' ';}
        LCD_data(d);
        fs=fs/10;
        n=n/10;
    }
    LCD_data('.');                          //dp
    while(n>0){                             //right of dp
        d=adc/n;
        adc=adc%n;
        if(d){zf=1;}
        d=d+'0';
        if(zf==0){d=' ';}
        LCD_data(d);
        n=n/10;        
    }
}


/* Callbacks. These function names are set in usb_config.h. */
void app_set_configuration_callback(uint8_t configuration)
{

}

uint16_t app_get_device_status_callback()
{
	return 0x0000;
}

void app_endpoint_halt_callback(uint8_t endpoint, bool halted)
{

}

int8_t app_set_interface_callback(uint8_t interface, uint8_t alt_setting)
{
	return 0;
}

int8_t app_get_interface_callback(uint8_t interface)
{
	return 0;
}

void app_out_transaction_callback(uint8_t endpoint)
{

}

void app_in_transaction_complete_callback(uint8_t endpoint)
{

}

int8_t app_unknown_setup_request_callback(const struct setup_packet *setup)
{
	/* To use the CDC device class, have a handler for unknown setup
	 * requests and call process_cdc_setup_request() (as shown here),
	 * which will check if the setup request is CDC-related, and will
	 * call the CDC application callbacks defined in usb_cdc.h. For
	 * composite devices containing other device classes, make sure
	 * MULTI_CLASS_DEVICE is defined in usb_config.h and call all
	 * appropriate device class setup request functions here.
	 */
	return process_cdc_setup_request(setup);
}

int16_t app_unknown_get_descriptor_callback(const struct setup_packet *pkt, const void **descriptor)
{
	return -1;
}

void app_start_of_frame_callback(void)
{

}

void app_usb_reset_callback(void)
{

}

/* CDC Callbacks. See usb_cdc.h for documentation. */

int8_t app_send_encapsulated_command(uint8_t interface, uint16_t length)
{
	return -1;
}

int16_t app_get_encapsulated_response(uint8_t interface,
                                      uint16_t length, const void **report,
                                      usb_ep0_data_stage_callback *callback,
                                      void **context)
{
	return -1;
}

int8_t app_set_comm_feature_callback(uint8_t interface,
                                     bool idle_setting,
                                     bool data_multiplexed_state)
{
	return -1;
}

int8_t app_clear_comm_feature_callback(uint8_t interface,
                                       bool idle_setting,
                                       bool data_multiplexed_state)
{
	return -1;
}

int8_t app_get_comm_feature_callback(uint8_t interface,
                                     bool *idle_setting,
                                     bool *data_multiplexed_state)
{
	return -1;
}

int8_t app_set_line_coding_callback(uint8_t interface,
                                    const struct cdc_line_coding *coding)
{
	line_coding = *coding;
	return 0;
}

int8_t app_get_line_coding_callback(uint8_t interface,
                                    struct cdc_line_coding *coding)
{
	/* This is where baud rate, data, stop, and parity bits are set. */
	*coding = line_coding;
	return 0;
}

int8_t app_set_control_line_state_callback(uint8_t interface,
                                           bool dtr, bool dts)
{
	return 0;
}

int8_t app_send_break_callback(uint8_t interface, uint16_t duration)
{
	return 0;
}


#ifdef _PIC14E
void interrupt isr()
{
	usb_service();
}
#elif _PIC18

#ifdef __XC8
void interrupt high_priority isr()
{
	usb_service();
}
#elif _PICC18
#error need to make ISR
#endif

#endif

void initGraphics(void){
    LCD_cmd(0b01000000);    //start of CGRAM
    //only 5 lsb are valid
    //character 0
    LCD_data(0b00011111);   //      empty progress bar
    LCD_data(0b00000000);   //
    LCD_data(0b00000000);   //
    LCD_data(0b00000000);   //
    LCD_data(0b00000000);   //
    LCD_data(0b00000000);   //
    LCD_data(0b00000000);   //
    LCD_data(0b00011111);   //
    //character 1
    LCD_data(0b00000011);   //      start progress bar
    LCD_data(0b00000010);   //
    LCD_data(0b00000010);   //
    LCD_data(0b00000010);   //
    LCD_data(0b00000010);   //
    LCD_data(0b00000010);   //
    LCD_data(0b00000010);   //
    LCD_data(0b00000011);   //
    //character 2
    LCD_data(0b00011000);   //      end progress bar
    LCD_data(0b00001000);   //
    LCD_data(0b00001000);   //
    LCD_data(0b00001000);   //
    LCD_data(0b00001000);   //
    LCD_data(0b00001000);   //
    LCD_data(0b00001000);   //
    LCD_data(0b00011000);   //
    //character 3
    LCD_data(0b00011111);   //      half progress bar
    LCD_data(0b00000000);   //
    LCD_data(0b00011000);   //
    LCD_data(0b00011000);   //
    LCD_data(0b00011000);   //
    LCD_data(0b00011000);   //
    LCD_data(0b00000000);   //
    LCD_data(0b00011111);   //
    //character 4
    LCD_data(0b00011111);   //      full progress bar
    LCD_data(0b00000000);   //
    LCD_data(0b00011011);   //
    LCD_data(0b00011011);   //
    LCD_data(0b00011011);   //
    LCD_data(0b00011011);   //
    LCD_data(0b00000000);   //
    LCD_data(0b00011111);   //
    //character 5
    LCD_data(0b00000000);   //      media stop
    LCD_data(0b00011111);   //
    LCD_data(0b00011111);   //
    LCD_data(0b00011111);   //
    LCD_data(0b00011111);   //
    LCD_data(0b00011111);   //
    LCD_data(0b00000000);   //
    LCD_data(0b00000000);   //
    //character 6
    LCD_data(0b00000000);   //      media pause
    LCD_data(0b00011011);   //
    LCD_data(0b00011011);   //
    LCD_data(0b00011011);   //
    LCD_data(0b00011011);   //
    LCD_data(0b00011011);   //
    LCD_data(0b00000000);   //
    LCD_data(0b00000000);   //
    //character 7
    LCD_data(0b00000000);   //      media play
    LCD_data(0b00010000);   //
    LCD_data(0b00011100);   //
    LCD_data(0b00011111);   //
    LCD_data(0b00011100);   //
    LCD_data(0b00010000);   //
    LCD_data(0b00000000);   //
    LCD_data(0b00000000);   //
    
    LCD_cmd(0b10000000);    //start of data ram (ready for output)
    
}