/* AudiomVmeterMk2_sketch.ino -- a sketch for the 
   Audio MillivoltMeter Mk2, using an Arduino Nano
   with an AD8307 logarithmic amplifier and a 24-bit
   LTC2400 ADC.
   Written by Jim Rowe for Silicon Chip
   Version 1.0, revised 22/07/2019 at 7:30 am
   Note: the following SPI interface connections
   are assumed by the Arduino SPI library:
   LTC2400  <---------> Arduino Nano
   pin 5 (CSbar) <-  from   DigIO pin 10 (SS-bar)
   pin 6 (SDO)   ->  to     DigIO pin 12 (MISO)
   pin 7 (SCK)   <-  from   DigIO pin 13 (SCK)
   Note also that the I2C interface library used to drive
   the LCD display uses the A4 pin of the Arduino for the
   SDA line, and the A5 pin for the SCK line.
   Also note that if the LCD serial 'backpack' uses a PCF8574T
   device, its address will be 0x27 -- not the address of 0x3F
   shown below, which is the address for a PCF8574AT device.
 */

#include <SPI.h>
#include <Wire.h> 
#include <LiquidCrystal_I2C.h>

// first set the LCD address to 0x3F, with 2 x 16 char lines
LiquidCrystal_I2C lcd(0x3F,16,2); 

long inSample;  // the incoming 32-bit words from the ADC
float AvgMeas;  // the average of 10 measurements
float MaxdBV;   // the dBV figure corresp to Vref (i.e., FSD)
// value approx -2.2dBV for low range, +37.8dBV for high range
float CfgVal = 25.50039; // config value for dBV correction
// (corresponds to a Vref of 2.5000)
String LCDLine1;  // used for 1st line of LCD display
String LCDLine2;  // used for the 2nd line of LCD display
String RngSymbl;  // used to indicate current range
String InpStr;    // icon to show selected input

const float Vref = 2.5008; // Vref = ADC reference voltage
   // (change for calibration if LT1019 output is measured)
const byte SSPin = 10;    // digIO pin 10 for CSbar/SS-bar
const byte MISOPin = 12;  // digIO pin 12 for MISO
const byte SCKPin = 13;   // & digIO pin 13 for SCK

const int SLED = 9;      // label D9 SLED (sampling LED)
const int InpSel = 3;    // label D3 InpSel (input select)
const int RngSel = 2;    // & label D2 RngSel (range select)

void setup()
{
  // start serial comms to PC at 115,200 baud, 8N1
  Serial.begin(115200);
  // then send back start message
  Serial.println("Silicon Chip Digital Millivoltmeter");
  Serial.println(" ");
   
  lcd.init();                   // initialise LCD
  lcd.backlight();              // & enable backlight
  lcd.setBacklight(HIGH);       // then turn backlight on
  LCDLine1 = "  Silicon Chip";  // define line 1 message
  LCDLine2 = "Digital mV Meter";  // then line 2 message
  DisplayonLCD();               // then send them both to LCD
  
  pinMode(SSPin, OUTPUT);       // make D10 an output (CS-bar)
  digitalWrite(SSPin, HIGH);    // and initialise to HIGH
  pinMode(MISOPin, INPUT);      // make D12 an input (MISO)
  pinMode(SCKPin, OUTPUT);      // make D13 an output (SCK)
  digitalWrite(SCKPin,LOW);     // and set low (for Ext SCK)
  
  pinMode(SLED, OUTPUT);   // make D9 an output (sampling LED)
  pinMode(RngSel, OUTPUT); // & D2 an output to drive relay
  pinMode(InpSel, INPUT);  // but make D3 an input (from S1b)
  
  digitalWrite(SLED, LOW); // make sure sampling LED off
  digitalWrite(RngSel,LOW); // set RngSel low for high range
               // i.e., RLY1 off, for 40dB input attenuation
  RngSymbl = "(H)"; // set RngSymbl for high range
  // then work out the actual MaxdBV reading, since
  // 2.500V corresponds to -2.22dBV (or +37.78dBV, on H range)
  // Note: Arduino log function is really natural log fn
  MaxdBV = CfgVal + 20.0 * (log(Vref) - log(10.0));
  MaxdBV = MaxdBV + 40.0;  // but on high range, so add 40dB

  // next initialise SPI port for max speed, data MSB first,
  // data mode 0 -- with SCK idle low (CPOL = 0),
  //                MISO read on rising edge (CPHI = 0)
  SPI.beginTransaction(SPISettings(2000000, MSBFIRST, SPI_MODE0));
  delay(2000);  // delay 2s to allow init screen to be seen
  WipeCleanLCD(); // then wipe the LCD to prepare for readings
} // end of setup function

 // ==========================================================
 // main loop() function begins here
 // ==========================================================
void loop()
{
  if( digitalRead(InpSel) == LOW)
  {
    InpStr = "(BAL)";
  }
  else
  {
    InpStr = "(U/B)";
  }
  // now since we are on the high range, start by
  // taking the average of 10 samples to check if we
  // need to switch down to the low range
  AvgMeas = Get10MeasAvg();
  // now if Avg reading is less than 57.82% of FSD
  if(AvgMeas < 9700000.0)
  {
    SwtoLowRange(); // switch to the low range
  }
  // We should now be on the most appropriate range, so
  // we can take the average of 10 new samples
  AvgMeas = Get10MeasAvg();
  // then process & display result on LCDs
  // (also send to IDE's Serial Monitor for logging)
  CalcnShow();
  // Now if we are on low range, switch back to high range
  // to prevent overload if a high input is applied
  if(RngSymbl == "(L)")
  {
    SwtoHighRange();
  }
}      // end of main loop

// ===========================================================
// CalcnShow() function: to process the average of 10 readings
// from the LTC2400, calculate the equivalent dBV & V/mV figs,
// display them on the LCD display & also send back to the PC.
// ===========================================================
long CalcnShow()
{
  float ADCIvolts;// used for the measured ADC input volts
  float dBVmeas;  // used for the measured dBV figure
  float Expmeas; // used for the Volts exponent
  float Volts;    // used for the corresponding Volts reading
  float milliV;   // used for the corresponding mV reading
  
  // first calculate the ADC input voltage using the equation
  // ADCIvolts = Vref * sample / 16777215 (FFFFFF = 16777215)
  ADCIvolts = Vref * (AvgMeas / 16777215.0);
  // next work out the dBVmeas reading (40 = number of 1dB
  // steps in 1V, since the AD8307 slope = 25mV per dB)
  dBVmeas = MaxdBV - 40 * (Vref - ADCIvolts);
  LCDLine2 = "= " + String(dBVmeas, 2) + " dBV " + RngSymbl;
  Expmeas = dBVmeas/20.0;  // find Volts exponent
  // then work out the volts or millivolts
  if (Expmeas < 0)
    {
      milliV = 1000.0 * pow(10.0, Expmeas); // get measured mV
      LCDLine1 = String(milliV, 3) + " mV " + InpStr;
    }
  else
    {
      Volts = pow(10.0, Expmeas); // get measured volts
      LCDLine1 = String(Volts, 3) + " V " + InpStr;
    }
  // now clear the LCD screen
  WipeCleanLCD();  
  // and display lines on LCD
  DisplayonLCD();
  // then also send them back to the PC (IDE serial Monitor)
  Serial.println(LCDLine1 + " " + LCDLine2); 
} // and return
 
// ========================================================
// Get10MeasAvg() function -- take 10 measurements and get
// their average value
// ========================================================
float Get10MeasAvg()
  {
  int x;          // counter
  long NxtSample; // next sample from ADC
  
  AvgMeas = 0;   // first zero the average measurement      
  for(x = 0; x < 10; x++)
    {
       NxtSample = TakeMeas();  // take a sample measurement
       AvgMeas = AvgMeas + NxtSample; // then add to AvgMeas
    }
  AvgMeas = AvgMeas / 10.000;  // work out 10-sample average
  return(AvgMeas);    // & return with the average    
  }

// ========================================================
// TakeMeas() function: to take a 32-bit measurement from
// the LTC2400, convert to 24 bits and return
// ========================================================
long TakeMeas()
{
   inSample = SpiRead(); // get a 32-bit sample
   inSample = 0;         // then discard (previous sample)
   inSample = SpiRead(); // now get a new 32-bit sample
   inSample >>= 4;       // shift out the 4 'sub-LSB' bits    
   inSample &= 0xFFFFFF; // now mask off the status bits,
                         // leaving just the 24 data bits
   return(inSample);     // & return with the 24-bit sample
}
// ========================================================
// SpiRead() function: to fetch a four-byte sample from the
// LTC2400 via the SPI interface
// ========================================================
long SpiRead()
{
  long result = 0; // result declared a 32-bit long
  long b = 0;      // likewise for b
  digitalWrite(SLED, HIGH); // turn on sampling LED to begin  
  digitalWrite(SCKPin,LOW); // ensure LTC2400 set for Ext SCK
  digitalWrite(SSPin,LOW);  // pull SS low to start transfer

  // wait until a sample is available (when MISO pin goes low)
  while( digitalRead(MISOPin) )
    ;

  b = SPI.transfer(0xff); // get B3, most significant byte
  result = b<<8;   // shift left by 8 bits, pass to result 
  b = SPI.transfer(0xff); // get B2, next significant byte
  result |= b;            // bitwise OR with result
  result = result<<8;     // shift left by 8 bits again
  b = SPI.transfer(0xff); // get B1, next significant byte
  result |= b;            // bitwise OR with result
  result = result<<8;     // shift left by 8 bits once more
  b = SPI.transfer(0xff); // get B0, least significant byte
  result |= b;            // bitwise OR with result, which
                       // should give result = [B3,B2,B1,B0]
  digitalWrite(SSPin, HIGH); // pull SS high again to end txfr
  digitalWrite(SLED,LOW);  // then turn off the sampling LED
  return(result);          // and return with 32-bit result
}  // end of SpiRead function

// =========================================================
// WipeCleanLCD() -- function to clear the LCD screen
// =========================================================
void WipeCleanLCD()
  {
    lcd.setCursor(0,0);
    lcd.print("                ");
    lcd.setCursor(0,1);
    lcd.print("                ");
  } // then return
  
 // =========================================================
 // DisplayonLCD() -- function to display 2 lines of 16 chars
 // on LCD (LCDLine1, LCDLine2)
 // =========================================================
 void DisplayonLCD()
 {
  lcd.setCursor(0,0);   // set LCD cursor to start of line 1
  lcd.print(LCDLine1);  // display line 1 message
  lcd.setCursor(0,1);   // then set cursor to start of line 2
  lcd.print(LCDLine2);  // then display line 2 message
 }  // and return
 
 // =========================================================
 // SwtoLowRange() -- function to switch to the 
 // LOW measuring range
 // =========================================================
 void SwtoLowRange()
 {
   digitalWrite(RngSel, HIGH); // turn on RLY1
   delay(2);  // pause so relay can stabilise
   RngSymbl = "(L)";   //  set RngSymbl for L range
   MaxdBV = MaxdBV - 40.0; // lower MaxdBV by 40dB 
 }

 // =========================================================
 // SwtoHighRange() -- function to switch to the HIGH
 // measuring range
 // =========================================================
 void SwtoHighRange()
 {
      digitalWrite(RngSel, LOW); // turn off RLY1
      delay(2);  // pause so relay can stabilise      
      RngSymbl = "(H)";  // reset RngSymbl
      MaxdBV = MaxdBV + 40.0; // & reset MaxdBV
 }

 // =========================================================
//    end of code
