/*********************************************************************************************************************************
    GPS Analog Clock Driver V1.2
  	Copyright (C) 2023 Geoff Graham (projects@geoffg.net)
	All rights reserved.

    V1.0 Original
    V1.1 Small change to improve accuracy
    V1.2 Added configuration of sweep pulse width

	This file and the program created from it are FREE FOR COMMERCIAL AND
	NON-COMMERCIAL USE as long as the following conditions are followed.

	Copyright remains Geoff Graham's, and as such any Copyright notices in the
	code are not to be removed.  If this code is used in a product, Geoff Graham
	should be given attribution as the author of the parts used.  This can be in
	the form of a textual message at program startup or in documentation (online
	or textual) provided with the program or product.

	Redistribution and use in source and binary forms, with or without modification,
	are permitted provided that the following conditions  are met:
	1. Redistributions of source code must retain the copyright notice, this list
	   of conditions and the following disclaimer.
	2. Redistributions in binary form must reproduce the above copyright notice, this
	   list of conditions and the following disclaimer in the documentation and/or
	   other materials provided with the distribution.
	3. All advertising materials mentioning features or use of this software must
	   display the following acknowledgement:
	   This product includes software developed by Geoff Graham (projects@geoffg.net)

	THIS SOFTWARE IS PROVIDED BY GEOFF GRAHAM ``AS IS'' AND  ANY EXPRESS OR IMPLIED
	WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
	MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT
	SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
	SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
	PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
	BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
	CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
	IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
	SUCH DAMAGE.

	The licence and distribution terms for any publicly available version or
	derivative of this code cannot be changed.  i.e. this code cannot simply be copied
	and put under another distribution licence (including the GNU Public Licence).

 ********************************************************************************************************************************

    USB code licence
    ================
    M-Stack is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  For details, see sections 7, 8, and 9
    of the Apache License, version 2.0 which apply to this file.  If you have
    purchased a commercial license for this software from Signal 11 Software,
    your commercial license supersedes the information in this header.

    Alan Ott
    Signal 11 Software
    2014-05-12
 ********************************************************************************************************************************/

/*****************************************************************
 * This requires Microchip XC8 Pro compiler V1.44
 * NO OTHER VERSION WILL WORK
 *
 * In Config set the XC8 linker memory model as follows:
 * ROM Ranges:  default,-1fe0-1fff
 *
 *****************************************************************/

#include "usb.h"
#include <xc.h>
#include <string.h>
#include "usb_config.h"
#include "usb_ch9.h"
#include "usb_cdc.h"
#include "hardware.h"
#include <stdint.h>
#include <stdio.h>


/* Configuration settings
CONFIG1
    FOSC -- Oscillator Selection Bits (bitmask:0x0003)
        FOSC = INTOSC	0x3FFC	INTOSC oscillator: I/O function on CLKIN pin.
        FOSC = ECL	0x3FFD	ECL, External Clock, Low Power Mode (0-0.5 MHz): device clock supplied to CLKIN pins.
        FOSC = ECM	0x3FFE	ECM, External Clock, Medium Power Mode (0.5-4 MHz): device clock supplied to CLKIN pins.
        FOSC = ECH	0x3FFF	ECH, External Clock, High Power Mode (4-20 MHz): device clock supplied to CLKIN pins.
    WDTE -- Watchdog Timer Enable (bitmask:0x0018)
        WDTE = OFF	0x3FE7	WDT disabled. SWDTEN bit is ignored.
        WDTE = SWDTEN	0x3FEF	WDT controlled by the SWDTEN bit in the WDTCON register.
        WDTE = NSLEEP	0x3FF7	WDT enabled while running and disabled in Sleep. SWDTEN bit is ignored.
        WDTE = ON	0x3FFF	WDT enabled. SWDTEN bit is ignored.
    PWRTE -- Power-up Timer Enable (bitmask:0x0020)
        PWRTE = ON	0x3FDF	PWRT enabled.
        PWRTE = OFF	0x3FFF	PWRT disabled.
    MCLRE -- MCLR Pin Function Select (bitmask:0x0040)
        MCLRE = OFF	0x3FBF	MCLR/VPP pin function is digital input.
        MCLRE = ON	0x3FFF	MCLR/VPP pin function is MCLR.
    CP -- Flash Program Memory Code Protection (bitmask:0x0080)
        CP = ON	0x3F7F	Program memory code protection is enabled.
        CP = OFF	0x3FFF	Program memory code protection is disabled.
    BOREN -- Brown-out Reset Enable (bitmask:0x0600)
        BOREN = OFF	0x39FF	Brown-out Reset disabled. SBOREN bit is ignored.
        BOREN = SBODEN	0x3BFF	Brown-out Reset controlled by the SBOREN bit in the BORCON register.
        BOREN = NSLEEP	0x3DFF	Brown-out Reset enabled while running and disabled in Sleep. SBOREN bit is ignored.
        BOREN = ON	0x3FFF	Brown-out Reset enabled.
    CLKOUTEN -- Clock Out Enable (bitmask:0x0800)
        CLKOUTEN = ON	0x37FF	CLKOUT function is enabled on the CLKOUT pin.
        CLKOUTEN = OFF	0x3FFF	CLKOUT function is disabled. I/O or oscillator function on the CLKOUT pin.
CONFIG2
    WRT -- Flash Memory Self-Write Protection (bitmask:0x0003)
        WRT = ALL	0x3FFC	000h to FFFh write protected, no addresses may be modified by PMCON control.
        WRT = HALF	0x3FFD	000h to 7FFh write protected, 800h to FFFh may be modified by PMCON control.
        WRT = BOOT	0x3FFE	000h to 1FFh write protected, 200h to FFFh may be modified by PMCON control.
        WRT = OFF	0x3FFF	Write protection off.
    STVREN -- Stack Overflow/Underflow Reset Enable (bitmask:0x0200)
        STVREN = OFF	0x3DFF	Stack Overflow or Underflow will not cause a Reset.
        STVREN = ON	0x3FFF	Stack Overflow or Underflow will cause a Reset.
    BORV -- Brown-out Reset Voltage Selection (bitmask:0x0400)
        BORV = HI	0x3BFF	Brown-out Reset Voltage (Vbor), 2.7V trip point selected.
        BORV = LO	0x3FFF	Brown-out Reset Voltage (Vbor), 1.9V trip point selected.
    LPBOR -- Low-Power Brown Out Reset (bitmask:0x0800)
        LPBOR = ON	0x37FF	Low-Power BOR is enabled.
        LPBOR = OFF	0x3FFF	Low-Power BOR is disabled.
    DEBUG -- Debugger enable bit (bitmask:0x1000)
        DEBUG = ON	0x2FFF	Background debugger enabled.
        DEBUG = OFF	0x3FFF	Background debugger disabled.
    LVP -- Low-Voltage Programming Enable (bitmask:0x2000)
        LVP = OFF	0x1FFF	High-voltage on MCLR/VPP must be used for programming.
        LVP = ON	0x3FFF	Low-voltage programming enabled.
*/

// CONFIG1
#pragma config FCMEN = 0            // Fail-Safe Clock Monitor Enable - disabled
#pragma config IESO = 0             // Internal External Switchover - disabled
#pragma config CLKOUTEN = 1         // Clock Out Enable - disabled
#pragma config BOREN = 0            // Brown-out Reset - disabled
#pragma config CP = 1               // Code Protection - disabled
#pragma config MCLRE = 0            // MCLR/VPP pin function is digital input
#pragma config PWRTE = 1            // Power-Up Timer - disabled
#pragma config WDTE = NSLEEP        // WDT enabled while running and disabled in Sleep
#pragma config FOSC = INTOSC        // Oscillator Selection - INTOSC oscillator: I/O function on OSC1 pin

// CONFIG2
#pragma config LVP = 0              // High-voltage on MCLR must be used for programming
#pragma config LPBOR = 1            // Low-Power Brown-out Reset is disabled
#pragma config BORV = LO            // Brown-out Reset Voltage Selection
#pragma config STVREN = ON          // Stack Overflow or Underflow will cause a Reset
#pragma config PLLEN = 1            // PLL is enabled
#pragma config PLLMULT = 3x         // PLL Multiplier Selection
#pragma config USBLSCLK = 48MHz     // USB Clock divide-by 8 (48 MHz system input clock expected)
#pragma config CPUDIV = NOCLKDIV    // No CPU system clock divide
#pragma config WRT = OFF            // Flash Memory Self-Write Protection


// debug settings
// pulsedebug = pulse on pin 6 (LED) using dbhi or dblo
// printfdebug = use printf() at 9600 baud on pin 6 (LED))
//#define pulsedebug
//#define printfdebug  // dbug

#if !defined(printfdebug) && !defined(pulsedebug)
    #define nodebug
#endif

#if defined(pulsedebug)
    #define dbhi {PORTCbits.RC4 = 1;}
    #define dblo {PORTCbits.RC4 = 0;}
    #define dbpulse {byte i = 75; PORTCbits.RC4 = 1; while(i--);PORTCbits.RC4 = 0;}
#else
    #define dbhi Error
    #define dbhi Error
#endif

#if defined(printfdebug)
    void pause(unsigned int i);
    void putch(char data) {
        while( ! TXIF) // check buffer
            continue;  // wait till ready
        TXREG = data;  // send data
        pause(2);
}
#else
#define printf() Error
#define putch()  Error
#endif

// basic defines
#define SECONDS_IN_HOUR (60L * 60L)
#define SECONDS_IN_DAY (24L * 60L * 60L)
typedef signed char byte;

// function declarations
void SyncTime(int startup);
char GetGPSData(char startup);
char CheckGpsOK(void);
void GpsSkipComma(unsigned char nbr);
int GetGpsDataByte(void);
void CheckGpsChar(unsigned char c);
unsigned char GetGpsChar(void);
void putUSB(unsigned char a);
void checkConsole(unsigned char time, unsigned char number);
void pause(unsigned int i);
long GPSDateToSeconds(void);
long CalcDST(long tempClkTime, int day_of_week, int dstmonth, int dstweek);
void flashLED(byte n);
char FlashRead(void *address);
void LoadConfigStruct(void);
char SaveConfigStruct(void);
void prt(const char *s);
void prt2(const char *s1, const char *s2);
void erase(byte *i);
byte GetNbr(char *p);
byte GetTZ(char *p);
void ManualStep(void);


//Global variables and buffers
char in_buf[56];
unsigned char in_count=0;
volatile unsigned char h_millis = 0;        // counter that increments every 1/2 millisecond
volatile uint16_t resetcount = 0;
byte ButtonDown = false;
byte sweep;
volatile byte InManualStep = false;


volatile long ClockTime;                    // the current time as shown by the clock's hands
volatile int ClkError;                      // the number of seconds that ClockTime is in error
long LastSyncTime;
byte SyncInterval;                          // * 12 hours
long GpsSync;
byte GpsError, SyncFailed = 0;
volatile byte BatteryLow = 0;               // true if in battery saving mode

// variables for correcting any crystal error
int XtalClockCorrect;
long XtalCorrectionTime, XtalCorrectCountdown;
int XtalCorrectionsMade;

// data returned from the GPS
int hrs, min, sec, day, mth, year;

// calculated DST data
int totaldays;
long DstStart, DstEnd;

// Global static variables used in the timer 1 interrupt
volatile byte StepPhase, NewSecond;
volatile byte ExtendStage, ErrorCorrect, ErrorPhase;
volatile byte StartOfNewSecond;
volatile byte TmrHiByteIdle, TmrLoByteIdle, TmrHiBytePulse, TmrLoBytePulse;

// this structure holds the configuration settings and is saved and loaded to/from flash
struct config_struct {
    long TZ;
    byte ClockType;    // 0 = stepping, 1 = sweep
    byte PulseWidth;   // 0 = sweep normal, 1 = sweep wide pulse, 2 = sweep square wave, 3 = stepping
    byte UseDST;
    // DOW Sunday = 1, Mth Jan = 1, Week 1 = first, 4 = last in the month
    byte StartMth, StartDOW, StartWeek, EndMth, EndDOW, EndWeek;
} Config;

#define CONFIG_MAGIC_NBR 0x43

// configuration menu text
const char ptitle1[] = "\x1b[2J\x1b[H Precision Clock Driver V1.2";
const char pcw[] =     " (c) Geoff Graham 2023\r\n";
const char psweep[] =  " Sweep Clock";
const char ppw1[] =    "\r\n Pulse Width (1 = Normal (recommended)";
const char ppw2[] =    "   2 = Wide   3 = Extra Wide) ? ";
const char pdst[] =    " Use Daylight Saving";
const char pstart[] =  " Start of Daylight Saving ";
const char pend[] =    " End of Daylight Saving ";
const char pmth[] =    "Month (1-12)";
const char pdoy[] =    "Day (1=Sunday)";
const char pdim[] =    "Day in Month (1 to 4=Last)";
const char ptz[] =     "\n Time Zone (-12.5 to +12.5)";
const char pyn[] =     " (Y/N)";
const char psaved[] =  "\r\n\n Configuration Saved\r\n Unplug USB ";


// timing parameters for a stepping clock
#define STEPER_PULSE         0xFA9F      // 42ms
#define STEPER_NORMAL_IDLE   0x855F      // 1 second - 42ms
#define STEPER_FAST_IDLE     0xC55F      // (1 second - 42ms - 42ms)/2

#define STEPER_SLOW          0x8000      // 1 second

// timing parameters for a sweep clock
#define SWEEP_NORMAL         0xFC00      // 31.25ms or 32 pulse widths per second

#define SWEEP_FAST           0xFC72      // approx 35.7ms or 7 pulse cycles per second
#define SWEEP_FAST_ADD_NBR   8           // to get the exact nbr of cycles in a second this many cycles must be 1 timer tick more
#define FAST_CYCLESINSECOND  36          // the number of pulse widths in a second
#define FAST_SECONDADDSECOND 8           // the number of seconds to add one additional second to the hands

#define SWEEP_SLOW           0xFB6E      // approx 27.7ms or 9 pulse cycles per second
#define SWEEP_SLOW_ADD_NBR   8           // to get the exact nbr of cycles in a second this many cycles must be 1 timer tick more
#define SLOW_CYCLESINSECOND  28          // the number of pulse widths in a second
#define SLOW_SECONDADDSECOND 8           // the number of seconds to subtract one additional second from the hands

// disassemble a timing word
#define HIBYTE(a)  (a >> 8)
#define LOBYTE(a)  (a & 0xFF)

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// calculate the timing for the various sweep pulse widths
// index:  0 = sweep normal, 1 = sweep wide, 2 = sweep square wave
// sweep normal is 50% pulse 50% idle,  sweep wide is 75% pulse 25% idle and square wave is 100% pulse and no idle
/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// normal speed timing
byte PulseNormalH[3] = { HIBYTE(SWEEP_NORMAL), HIBYTE(SWEEP_NORMAL - (0x10000 - SWEEP_NORMAL)/2), HIBYTE(SWEEP_NORMAL - (0x10000 - SWEEP_NORMAL) + 10) };
byte PulseNormalL[3] = { LOBYTE(SWEEP_NORMAL), LOBYTE(SWEEP_NORMAL - (0x10000 - SWEEP_NORMAL)/2), LOBYTE(SWEEP_NORMAL - (0x10000 - SWEEP_NORMAL) + 10) };
byte IdleNormalH[3] =  { HIBYTE(SWEEP_NORMAL), HIBYTE(SWEEP_NORMAL + (0x10000 - SWEEP_NORMAL)/2), HIBYTE(0x10000 - 10) };
byte IdleNormalL[3] =  { LOBYTE(SWEEP_NORMAL), LOBYTE(SWEEP_NORMAL + (0x10000 - SWEEP_NORMAL)/2), LOBYTE(0x10000 - 10) };
// slow speed timing
byte PulseSlowH[3] =   { HIBYTE(SWEEP_SLOW), HIBYTE(SWEEP_SLOW - (0x10000 - SWEEP_SLOW)/2), HIBYTE(SWEEP_SLOW - (0x10000 - SWEEP_SLOW) + 10) };
byte PulseSlowL[3] =   { LOBYTE(SWEEP_SLOW), LOBYTE(SWEEP_SLOW - (0x10000 - SWEEP_SLOW)/2), LOBYTE(SWEEP_SLOW - (0x10000 - SWEEP_SLOW) + 10) };
byte IdleSlowH[3] =    { HIBYTE(SWEEP_SLOW), HIBYTE(SWEEP_SLOW + (0x10000 - SWEEP_SLOW)/2), HIBYTE(0x10000 - 10) };
byte IdleSlowL[3] =    { LOBYTE(SWEEP_SLOW), LOBYTE(SWEEP_SLOW + (0x10000 - SWEEP_SLOW)/2), LOBYTE(0x10000 - 10) };
// fast speed timing
byte PulseFastH[3] = { HIBYTE(SWEEP_FAST), HIBYTE(SWEEP_FAST - (0x10000 - SWEEP_FAST)/2), HIBYTE(SWEEP_FAST - (0x10000 - SWEEP_FAST) + 10) };
byte PulseFastL[3] = { LOBYTE(SWEEP_FAST), LOBYTE(SWEEP_FAST - (0x10000 - SWEEP_FAST)/2), LOBYTE(SWEEP_FAST - (0x10000 - SWEEP_FAST) + 10) };
byte IdleFastH[3] =  { HIBYTE(SWEEP_FAST), HIBYTE(SWEEP_FAST + (0x10000 - SWEEP_FAST)/2), HIBYTE(0x10000 - 10) };
byte IdleFastL[3] =  { LOBYTE(SWEEP_FAST), LOBYTE(SWEEP_FAST + (0x10000 - SWEEP_FAST)/2), LOBYTE(0x10000 - 10) };


// program starts here
int main(void) {
    unsigned char ch;
    const unsigned char *out_buf;
    size_t out_buf_len;
    byte state = 0;
    char inbuf[6];
    byte incount = 0;
    byte nbr;

   	OSCCONbits.IRCF = 0b1101;               // 0b1101 = 4MHz clock
    OPTION_REGbits.PS = 0b110;              // watchdog set to 1 second

    TRISA =  0b11111111;                    // set inputs/outputs
    ANSELA = 0;                             // all pins digital
    ANSELC = 0b00000001;
    TRISC =  0b11100111;

    PORTC = 0;                              // all outputs should start low
    VREGCON = 0b11;                         // use the low power regulator in sleep

    // if the USB is connected we will start up in configuration mode with USB enabled
    // when the USB is removes a reboot will be forced and this code will be skipped
    // and execution will fall through to the clock driver code below
    if(PORTCbits.RC1) {

#if defined(nodebug)
        TMR1ON = 0;                         // turn off timer 1 (can be left on after a watchdog reset)

        flashLED(3);
    	OSCCONbits.IRCF = 0b1111;           // 0b1111 = 16MHz HFINTOSC postscalar
        ACTCONbits.ACTSRC = 1;              // Enable Active clock-tuning from the USB, 1=USB
        ACTCONbits.ACTEN = 1;
		INTCONbits.PEIE = 1;                // Configure interrupts
		INTCONbits.GIE = 1;

        // set up 0.5 millisecond interrupts
        // Timer2 Registers Prescaler= 4 - TMR2 PostScaler = 12 - PR2 = 125 - Freq = 2000.00 Hz - Period = 0.000500 seconds
        T2CON |= 88;                        // bits 6-3 Post scaler 1:1 thru 1:16
        T2CONbits.T2CKPS1 = 0;              // bits 1-0  Prescaler Rate Select bits
        T2CONbits.T2CKPS0 = 1;
        PR2 = 125;                          // PR2 (Timer2 Match value)
        PIE1bits.TMR2IE = 1;                // timer 2 interrupt enabled
        T2CONbits.TMR2ON = 1;               // turn timer 2 on

        usb_init();

        while (1) {
            CLRWDT();

            if(!usb_out_endpoint_halted(2) && usb_out_endpoint_has_data(2)){

                /* Check for an empty transaction. */
                out_buf_len = usb_get_out_buffer(2, &out_buf);
                if (out_buf_len <= 0)
                    goto empty;
                else {
                    flashLED(1);

                    // get the character from the USB
                    ch = out_buf[0];
                    if(ch > 0x60) ch -= 0x20;

                    // if it is printable, add to the buffer and echo
                    if(ch > 0x2A && incount < 5) {
                        inbuf[incount++] = ch;
                        putUSB(ch);
                    }

                    // handle backspace
                    if(ch == '\b' && incount != 0) {
                        prt("\b \b");
                        incount--;
                    }

                    // if it is a carriage return we must process the buffer
                    if(ch == '\r' || ch == '\n') {
                        // try decoding one or two numeric digits
                        // the result will be in nbe
                        nbr = 0;
                        if(incount == 1 || incount == 2) {
                            if(inbuf[0] >= '0' && inbuf[0] <= '9') {
                                nbr = inbuf[0] - '0';
                                if(incount == 2) {
                                    if(inbuf[1] >= '0' && inbuf[1] <= '9')
                                        nbr = (nbr * 10) + (inbuf[1] - '0');
                                    else
                                        nbr = 0;
                                }
                            }
                        }
                        // if nbr is invalid make it easier for the following code to detect
                        if(nbr == 0) nbr = 99;

                        switch(state) {
                            case 0:
                                // get answer to Sweep Clock?
                                if(incount != 1 || !(*inbuf == 'Y' || *inbuf == 'N')) {
                                    incount = 0;
                                    // on return or error redisplay the title and prompt
                                    prt(ptitle1);
                                    while(in_count) {checkConsole(40,60);}
                                    prt(pcw);
                                    prt2(psweep, pyn);
                                } else {
                                    Config.ClockType = (*inbuf == 'Y');
                                    incount = 0;
                                    state++;
                                    if(!Config.ClockType) {
                                        state++;                // skip the next question if stepping clock
                                        prt2(pdst, pyn);
                                    } else {
                                        prt(ppw1);
                                        while(in_count) {checkConsole(40,60);}
                                        prt(ppw2);
                                    }
                                }
                                break;
                            case 1:
                                // get answer to Pulse Width?
                                if(incount != 1 || !(*inbuf == '1' || *inbuf == '2' || *inbuf == '3'))  {
                                    erase(&incount);
                                } else {
                                    Config.PulseWidth = *inbuf - '1';
                                    incount = 0;
                                    state++;
                                    putUSB('\n'); prt2(pdst, pyn);
                                }
                                break;
                            case 2:
                                // get answer to DST?
                                if(incount != 1 || !(*inbuf == 'Y' || *inbuf == 'N'))  {
                                    erase(&incount);
                                } else {
                                    if(Config.UseDST = (*inbuf == 'Y')) {
                                        incount = 0;
                                        state++;
                                        putUSB('\n'); prt2(pstart, pmth);
                                    } else {
                                        goto SaveConfig;
                                    }
                                }
                                break;
                            case 3:
                                // get answer to Start DST Mth?
                                if(nbr > 12) {
                                    erase(&incount);
                                } else {
                                    Config.StartMth = nbr;
                                    incount = 0;
                                    state++;
                                    prt2(pstart, pdoy);
                                }
                                break;
                            case 4:
                                // get answer to Start DST day
                                if(nbr > 7) {
                                    erase(&incount);
                                } else {
                                    Config.StartDOW = nbr;
                                    incount = 0;
                                    state++;
                                    prt2(pstart, pdim);
                                }
                                break;
                            case 5:
                                // get answer to Start DST day in mth?
                                if(nbr > 4) {
                                    erase(&incount);
                                } else {
                                    Config.StartWeek = nbr;
                                    incount = 0;
                                    state++;
                                    putUSB('\n'); prt2(pend, pmth);
                                }
                                break;
                            case 6:
                                // get answer to end DST Mth?
                                if(nbr > 12) {
                                    erase(&incount);
                                } else {
                                    Config.EndMth = nbr;
                                    incount = 0;
                                    state++;
                                    prt2(pend, pdoy);
                                }
                                break;
                            case 7:
                                // get answer to end DST day
                                if(nbr > 7) {
                                    erase(&incount);
                                } else {
                                    Config.EndDOW = nbr;
                                    incount = 0;
                                    state++;
                                    prt2(pend, pdim);
                                }
                                break;
                            case 8:
                                // get answer to end DST day in mth?
                                if(nbr > 4) {
                                    erase(&incount);
                                } else {
                                    Config.EndWeek = nbr;
                                    incount = 0;
                                    state++;
                                    prt2(ptz, "");
                                }
                                break;
                            case 9:
                                // get TZ!
                                inbuf[incount] = 0;
                                nbr = GetTZ(inbuf);
                                if(nbr > 125 || nbr < -125)
                                    erase(&incount);
                                else {
                                    Config.TZ = (long)nbr * 360L;
                    SaveConfig:
                                    SaveConfigStruct();
                                    prt(psaved);
                                    state = 10;
                                }
                                break;
                        }
                    }

                    /* Send a zero-length packet if the transaction
                    * length was the same as the endpoint
                    * length. This is for demo purposes. In real
                    * life, you only need to do this if the data
                    * you're transferring ends on a multiple of
                    * the endpoint length. */
//                    if (out_buf_len == EP_2_LEN) {
//                        /* Wait until the IN endpoint can accept it */
//                        while (usb_in_endpoint_busy(2))	;
//                        usb_send_in_buffer(2, 0);
//                    }
                }
empty:
                 usb_arm_out_endpoint(2);
            }
            checkConsole(40,60);
            while(!PORTCbits.RC1);                                  // force a watchdog timeout if USB is removed
        }
#endif
    }

    // we get here if the USB is not connected
    // this code drives the clock
    // if USB is connected the watchdog timer will be used to force a reboot so that the configuration code (above) can run.
#if defined(nodebug)
    flashLED(2);
#endif
    pause(500);

#if defined(printfdebug)
    SPBRG = 25;                                 // baud rate selector, 25 = 9600 baud
    BRGH = 1;                                   // hi speed range
    BRG16 = 0;                                  // 8-bit baud rate generator
    CREN = 0;
    ADDEN = 0;                                  // no addressing
    SYNC = 0;                                   // async
    RCIE = 0;                                   // no interrupts
    TXEN = 1;                                   // we want to transmit
    SPEN = 1;                                   // enable serial port
    CREN = 1;                                   // enable receive
#endif

    LoadConfigStruct();
    if(Config.ClockType & 0b11111110) {         // means not configured
#if defined(nodebug)
        Config.ClockType = 0;                   // default to a stepping clock
        Config.UseDST = 0;                      // default is no DST
#else
        Config.ClockType = 0;                   // default to a stepping clock
        Config.PulseWidth = 0;                  // default to 5o%
        Config.UseDST = 1;                      // default is no DST
        Config.StartMth = 8;
        Config.StartDOW = 2;
        Config.StartWeek = 1;
        Config.EndMth = 11;
        Config.EndDOW = 1;
        Config.EndWeek = 2;
        Config.TZ = 8L * 3600L; //36000;  // NOTE: Must be in seconds
#endif
    }

    // setup timer 1 which we use to generate the clock pulses
    // this is done first so that the oscillator has time to settle
    T1OSCEN = 1;                                // start the 32KHx xtal oscillator
    T1CONbits.TMR1CS = 2;                       // select it as the clock for timer 1
    T1CONbits.T1CKPS = 0;                       // no pre scaler
    nT1SYNC = 1;                                // async clock
    TMR1IE = 1;                                 // enable interrupts
    PEIE = 1;

    SyncTime(true);                             // get the starting time from the GPS

    // pre load the timer and timer interrupt variables to simulate a stepping
    // clock that is too fast.  This will not generate any clock driving signals
    // but will count the seconds until the clock start time (next half hour).
    sweep = false;
    TMR1H = TmrHiByteIdle = 0x80;                   // this is one second
    TMR1L = TmrLoByteIdle = 0x00;
    NewSecond = 2;
    StepPhase = 1;
    ErrorCorrect = -1;
    ClkError = 1800 - (int)(ClockTime % 1800L);
    ClockTime += (long)ClkError;
    TMR1ON = 1;                                 // turn on timer 1 which will now be used to count the seconds until start
    GIE = 1;                                    // enable interrupts

    // now wait for the next half hour to arrive
    do {
        while(PORTCbits.RC1);                               // force a watchdog timeout if USB is connected
        CLRWDT();
        // if the button is pressed then manually step the clock
        InManualStep = false;
        if(PORTAbits.RA3 == 0) {
            InManualStep = true;
            ManualStep();
        } else {
            PORTCbits.RC4 = 1; pause(600); PORTCbits.RC4 = 0;   // long flash of the LED
        }
        SLEEP();                                            // wait for the timer 1 interrupt
    } while(ClkError);

    // we have reached the next exact hour or half hour
    // start running the clock
    // if we are running a stepper clock nothing is needed but a sweep clock needs some reconfiguration
    if(Config.ClockType) {
        StepPhase = 0;
        TmrHiByteIdle = IdleNormalH[2];
        TmrLoByteIdle = IdleNormalL[2];
        TmrHiBytePulse = PulseNormalH[2];
        TmrLoBytePulse = PulseNormalL[2];
        sweep = true;
        NewSecond = 32;
        ExtendStage = 99;
        ErrorCorrect = 0;
        ErrorPhase = 0;
    }

    // now the clock is running
    while(1) {
        while(PORTCbits.RC1);                           // force a watchdog timeout if USB is connected
        CLRWDT();
        SLEEP();                                        // the timer 1 interrupt will wake us up
        CLRWDT();

        if(StartOfNewSecond) {
            // do activities that must occur on each second
            // we have about 28ms before the next timer 1 interrupt

            // Check for DST correction
            // When DST starts we must advance the hours by one hour
            // This is indicated by setting ClockError negative by one hour.
            // We also change ClockTime by a similar amount to keep it at at UTC time after the correction
            if(DstStart <= ClockTime) {
                ClockTime -= SECONDS_IN_HOUR;           // keep on UTC time
                ClkError -= (int)SECONDS_IN_HOUR;       // indicate one hour slow
                DstStart = 0x7fffffff;                  // this will never be reached
            }

            // When DST ends we do the opposite
            if(DstEnd <= ClockTime) {
                ClockTime += SECONDS_IN_HOUR;           // keep on UTC time
                ClkError += (int)SECONDS_IN_HOUR;       // indicate one hour fast
                DstEnd = 0x7fffffff;                    // this will never be reached
            }

            // correct for any inaccuracy in the crystal's frequency
            if(--XtalCorrectCountdown < 0L) {
                XtalCorrectCountdown = XtalCorrectionTime;
                ClkError += XtalClockCorrect;
                XtalCorrectionsMade += XtalClockCorrect;
            }

            // flash the LED if we have not been able to sync
            if(SyncFailed) flashLED(1);

            // check if the button has been pressed and add or subtract one second
            if(!ButtonDown) {                           // if the button was NOT pressed on the previous pass
                if(PORTAbits.RA3 == 0) {                // and if it is pressed now
                    ButtonDown = true;                  // record that
                    PORTCbits.RC4 = 1;                  // and turn the LED on
                }
            } else {                                    // else the button WAS pressed on the previous pass
                if(PORTAbits.RA3 == 0) {                // if it is still pressed
                    ClockTime++;                        // signal that we are fast by one second
                    ClkError++;
                } else {                                // else not pressed
                    ClockTime--;                        // signal that we are too slow
                    ClkError--;
                }
                PORTCbits.RC4 = 0;                      // regardless, turn the LED off
                ButtonDown = false;                     // and pretend that the button has been released
            }

            // if we need to sync then get the correct time from the GPS and update the timing variables
            if(GpsSync < ClockTime && !BatteryLow && ClkError == 0) SyncTime(false);
        }
    }

// we should never reach here
}


///////////////////////////////////////////////////////////////////////////////////////////////////////////////
//


void interrupt isr() {

    // timer 1 interrupt, this drives the clocks motor
    // at the most this code takes 80uS to complete
    if(TMR1IF) {

        if(StepPhase & 1) {
            TMR1L = TmrLoByteIdle;                  // this must be done early before the xtal osc increments timer 1
            TMR1H = TmrHiByteIdle;
        } else {
            TMR1L = TmrLoBytePulse;                 // this must be done early before the xtal osc increments timer 1
            TMR1H = TmrHiBytePulse;
        }

        // set the output pin for the clock driver op amp
        if(!InManualStep) {                             // don't do this if we are manually driving the second hand around the dial
            PORTCbits.RC2 = (StepPhase >> 1) & 1;       // first set the pin in case it becomes an output
            TRISCbits.TRISC2 = StepPhase & 1;           // then set it as an output or high impedance (1 = hi impedance)
        }
        StepPhase++;
        StartOfNewSecond = false;

        if(sweep) {
            byte idx = Config.PulseWidth;
            if(idx < 2) idx += BatteryLow;

            // when going faster or slower we need to add one count to the timing value for the last nbr of phases
            // this ensures that we will have an exact number of clock steps per second
            if(StepPhase == ExtendStage) {
                TmrLoByteIdle -= 1;
                TmrLoBytePulse -= 1;
            }

            if(StepPhase == NewSecond) {
                // we have just finished stepping through one second
                // normally we have 32 transitions per second or 36 when going faster or 28 going slower
                StartOfNewSecond = true;
                StepPhase = 0;
                ClockTime++;
                if(ErrorPhase) {
                    if(--ErrorPhase == 0) {
                        ClockTime += (long)ErrorCorrect;                        // if running fast or slow adjust the clock time
                        ClkError += (int)ErrorCorrect;
                    }
                }

                // set the values for the next second
                if(ClkError == 0) {
                    // there is no error to correct
                    TmrHiByteIdle = IdleNormalH[idx];
                    TmrLoByteIdle = IdleNormalL[idx];
                    TmrHiBytePulse = PulseNormalH[idx];
                    TmrLoBytePulse = PulseNormalL[idx];
                    NewSecond = 32;
                    ExtendStage = 99;
                    ErrorCorrect = 0;
                    ErrorPhase = 0;
                } else if(ClkError > 0) {
                    // we are running fast
                    TmrHiByteIdle = IdleSlowH[idx];
                    TmrLoByteIdle = IdleSlowL[idx];
                    TmrHiBytePulse = PulseSlowH[idx];
                    TmrLoBytePulse = PulseSlowL[idx];
                    NewSecond = SLOW_CYCLESINSECOND;
                    ExtendStage = NewSecond - SWEEP_SLOW_ADD_NBR;
                    ErrorCorrect = -1;
                    if(!ErrorPhase) ErrorPhase = SLOW_SECONDADDSECOND;
                } else {
                    // we are running slow
                    TmrHiByteIdle = IdleFastH[idx];
                    TmrLoByteIdle = IdleFastL[idx];
                    TmrHiBytePulse = PulseFastH[idx];
                    TmrLoBytePulse = PulseFastL[idx];
                    NewSecond = FAST_CYCLESINSECOND;
                    ExtendStage = NewSecond - SWEEP_FAST_ADD_NBR;
                    ErrorCorrect = +1;
                    if(!ErrorPhase) ErrorPhase = FAST_SECONDADDSECOND;
                }
            }
        } else {
            // stepping clock movement
            if(StepPhase == NewSecond) {
                // we have just finished stepping through one second
                StartOfNewSecond = true;
                ClockTime += (long)(ErrorCorrect + 1);           // adjust the clock time to reflect double or skipped steps
                ClkError += (int)ErrorCorrect;

                // set the values for the next second
                // ClockError positive means that the clock is running too fast and must slow down
                // ClockError negative means that the clock is running too slow and must speed up
                if(ClkError == 0) {
                    // there is no error to correct
                    NewSecond = StepPhase + 2;
                    ErrorCorrect = 0;
                } else if(ClkError > 0) {
                    // clock is fast so slow it down by skipping a second
                    // when we do this the timer is simply set to one second
                    NewSecond = StepPhase;                      // and this means wait for just one cycle of timer 1
                    StepPhase--;                                // this ensures that the output is not driven during the second
                    ErrorCorrect = -1;                          // record the fact that we are skipping a second
                } else {
                    // clock is slow so add a second
                    NewSecond = StepPhase + 4;
                    ErrorCorrect = +1;                           // record that we will be double stepping
                }
            }

            // this executes on each transition of the clock pulse.  Set the values for the next timer 1 cycle
            // unlike the sweep clock we have different values for the pulse width and the idle time between pulses
            if(ErrorCorrect < 0) {
                // slow speed - generate no pulses so skip a step
                TmrHiByteIdle = HIBYTE(STEPER_SLOW);            // this is one second
                TmrLoByteIdle = LOBYTE(STEPER_SLOW);
            } else {
                if(StepPhase & 1) {
                    // idle time
                    if(ErrorCorrect == 0) {
                        // normal speed
                        TmrHiByteIdle = HIBYTE(STEPER_NORMAL_IDLE);
                        TmrLoByteIdle = LOBYTE(STEPER_NORMAL_IDLE);
                    } else {
                        // fast speed
                        TmrHiByteIdle = HIBYTE(STEPER_FAST_IDLE);
                        TmrLoByteIdle = LOBYTE(STEPER_FAST_IDLE);
                    }
                } else {
                    // pulse time (same for both normal and fast stepping)
                    TmrHiBytePulse = HIBYTE(STEPER_PULSE);
                    TmrLoBytePulse = LOBYTE(STEPER_PULSE);
                }
            }
        }

        TMR1IF = 0;
        return;
    }   // end of timer 1 interrupt


    ///////////////////////////////////////////////////////////////////
    // these interrupts are only used by the USB in configuration mode
    ///////////////////////////////////////////////////////////////////

    // timer 2 interrupt at 2000 Hz used in USB only
    if (PIR1bits.TMR2IF) {
        h_millis++;
        resetcount++;

        PIR1bits.TMR2IF = 0;    // reset flag
        return;
    }

    //////////////////////////////////////////////////
    // used in USB only
    usb_service();
}




///////////////////////////////////////////////////////////////////////////////////////////////////////////////
// Perform all the work involved in getting the time from the GPS, correcting for any errors
// and setting up for the clock driver
//
void SyncTime(int startup) {
    long GpsTime, tempClkTime;
    int Err;

    if(!GetGPSData(startup)) {
        // no GPS signal - don't do anything and try again after one day
        SyncInterval = 1;
        GpsSync += SECONDS_IN_DAY;
        SyncFailed = true;
        return;
    }

    // successful GPS sync
    SyncFailed = false;
    GpsTime = GPSDateToSeconds();

    if(startup) {
        tempClkTime = ClockTime = GpsTime;
        XtalCorrectionTime = 0x7FFFFFFFL;                                       // don't correct
        ClkError = 0;
        SyncInterval = 1;                                                       // next sync in 12 hours
    } else {
        GIE = 0;                                                                // no interrupts
        tempClkTime = ClockTime;
        Err = (int)(ClockTime - GpsTime);
        ClkError += Err;// + XtalClockCorrect;
        GIE = 1;                                                                // interrupts back on

        Err += XtalCorrectionsMade;                                             // this is the total number of seconds that the xtal is in error
        if(Err == 0) {
            XtalCorrectionTime = 0x7fffffffL;                                   // never correct
        } else {
            XtalCorrectionTime = ((GpsTime - LastSyncTime))/(long)Err;          // this is the time between each single second correction
            if(Err > 0) {
                // xtal and therefor the clock is running fast
                XtalClockCorrect = +1;
            } else {
                // xtal and therefor the clock is running slow
                XtalCorrectionTime = -XtalCorrectionTime;
                XtalClockCorrect = -1;
            }
        }
        // on every sync increase the sync interval by one day until it reaches 5 days
        if(SyncInterval < 10) SyncInterval++;
    }
    GpsSync = GpsTime + ((long)SyncInterval * SECONDS_IN_DAY/2);

    // get the DST start/end
    DstStart = CalcDST(tempClkTime, Config.StartDOW, Config.StartMth, Config.StartWeek);
    if(labs(DstStart - GpsSync) < SECONDS_IN_HOUR*2L) GpsSync = DstStart + SECONDS_IN_HOUR*2L;
    DstEnd = CalcDST(tempClkTime, Config.EndDOW, Config.EndMth, Config.EndWeek);
    if(labs(DstEnd - GpsSync) < SECONDS_IN_HOUR*2L) GpsSync = DstEnd + SECONDS_IN_HOUR*2L;

    XtalCorrectionsMade = 0;                                // reset the count of corrections made during the next period between syncs
    LastSyncTime = GpsTime;
    XtalCorrectCountdown = XtalCorrectionTime/2L;           // set the countdown to the first correction
}



/************************************************************************************************
Get data from the GPS module

Returns true if a valid RMC msg was received.
Returns false if it timed out (1/2 hour) without a valid RMC msg

Format of the NEMA RMC message.  We only use the time, date and valid flag
    $GPRMC,043356.000,A,3158.7599,S,11552.8689,E,0.24,54.42,101008,,*20
    ====== ======     =                                     ======   ==
    fixed   time   valid                                    date    checksum
    header hhmmss   data                                    ddmmyy
************************************************************************************************/

char GetGPSData(char startup) {
    int timeout;
    char BaudRates[3] = {52, 25, 12};                       // baud rate selector, 25 = 9600 baud
    static char idx = 0;

    PORTCbits.RC3 = 1;                                      // power up the GPS

    // find the baudrate.  Searches for a $GP header at 4800, 9600 and 19200 baud.
    // this will search forever
    if(startup) {
#if !defined(printfdebug)
        // setup the UART
        SPBRG = BaudRates[idx];
        BRGH = 1;                                           // hi speed range
        BRG16 = 0;                                          // 8-bit baud rate generator
        CREN = 0;
        ADDEN = 0;                                          // no addressing
        SYNC = 0;                                           // async
        RCIE = 0;                                           // no interrupts
        TXEN = 0;                                           // we don't want to transmit
        SPEN = 1;                                           // enable serial port
        CREN = 1;                                           // enable receive
        while(1) {
            timeout = 0;
            if(idx++ == 3) idx = 0;
            SPBRG = BaudRates[idx];
            do {
                if(PORTAbits.RA3 == 0) ManualStep();        // manually move the hands if the switch is held down
                GpsError = false;
                CheckGpsChar('$');
            } while(GpsError && ++timeout < 120) ;
            CheckGpsChar('G');
            CheckGpsChar('P');
            if(!GpsError) break;                            // only stop if we have a valid header
        }
#else
        NOP();
#endif
    } else {
        // check the battery voltage and set BatteryLow accordingly
        FVRCON = 0b10000001;                                // turn on the fixed voltage ref & select 1.024V
        ADCON0 = 0b01111101;                                // turn on the ADC and select the voltage ref as input
        ADCON1 = 0b00010000;                                // Fosc/4 and Vref is Vdd
        pause(100);                                         // let the boost regulator get running
        ADCON0 = 0b01111111;                                // start the conversion
        while(ADCON0 & 2);                                  // wait for it to complete
        ADCON0 = 0;                                         // turn off the ADC
        FVRCON = 0;                                         // turn off the voltage ref
        BatteryLow = (ADRESH > 116);                        // 119 = 1.1V, 113 = 1.15V, 107 = 1.2V (all are per cell)
    }

    SPEN = 1;                                               // enable serial port

    // sit in a loop waiting for a valid RMC message
    // if startup=true then flash the led every second and keep searching forever
    // if startup=false (ie, the clock is running) then don't flash the LED but timeout after 1/2 hour
    // will always flash the LED if the previous sync failed
    // returns true is valid record found or false if a timeout occurred.
    timeout = 3600 / 2;                                     // timeout 1/2 hour - timeout is decremented every RMC record = 1 second
    while(1) {
        do {
            if(PORTAbits.RA3 == 0) ManualStep();            // manually move the hands if the switch is held down
            GpsError = false;
            CheckGpsChar('$');
        } while(GpsError) ;
        CheckGpsChar('G');
        CheckGpsChar('P');
        CheckGpsChar('R');
        CheckGpsChar('M');
        CheckGpsChar('C');
        CheckGpsChar(',');
        if(!GpsError) {
            if(startup)
#if defined(nodebug)
                PORTCbits.RC4 = 1;                          // flash LED on
#else
            NOP();
#endif
            else
                timeout--;
        }
        hrs = GetGpsDataByte();                             // get the hours
        min = GetGpsDataByte();                             // get the minutes
        sec = GetGpsDataByte();                             // and seconds
        GpsSkipComma(1);
        CheckGpsChar('A');
        GpsSkipComma(7);
        day = GetGpsDataByte();                             // get the day
        mth = GetGpsDataByte();                             // get the month
        year = GetGpsDataByte();                            // and year
#if defined(nodebug)
//        pause(7);                                           // lengthen the LED on time
        PORTCbits.RC4 = 0;                                  // LED off
#endif
        if(!GpsError || timeout < 0) {                      // valid message or timeout
#if !defined(printfdebug)
            SPEN = 0;                                       // turn off the UART
#endif
            PORTCbits.RC3 = 0;                              // power down the GPS
            return !GpsError;
        }
    }
    return 0;                                               // should never reach here but keep the compiler happy
}


///////////////////////////////////////////////////////////////////////////////////////////////////////////////
// Get from the GPS input stream a number of commas.  Used for skipping over unwanted data
//
void GpsSkipComma(unsigned char nbr) {
	while(nbr && !GpsError)
		if(GetGpsChar() == ',') nbr--;
}



///////////////////////////////////////////////////////////////////////////////////////////////////////////////
// Get from the GPS input stream a data byte consisting of two ascii chars
//
int GetGpsDataByte(void) {
	int  b;

    b = (GetGpsChar() - 48) * 10;
    b += GetGpsChar() - 48;
	return b;
}



///////////////////////////////////////////////////////////////////////////////////////////////////////////////
// Get from the GPS input stream a character and test it against the argument.  Set GpsError if mismatch.
//
void CheckGpsChar(unsigned char c) {
	if(GetGpsChar() != c) GpsError = true;
}



///////////////////////////////////////////////////////////////////////////////////////////////////////////////
// Return a character from the GPS serial interface
//	- Converts the char to uppercase
//  - returns with zero on timeout of >10ms
//  - will exit immediately if GpsError == true
//
unsigned char GetGpsChar(void) {
    unsigned int timeout;
	unsigned char ch;

    while(PORTCbits.RC1) {};                                           // force a watchdog timeout if USB is connected
    CLRWDT();                                                          // reset the watchdog
    if(!GpsError) {
        for(timeout = 0; timeout < 300; timeout++) {                   // wait for the char, with 10ms timeout
            if(OERR) { CREN = 0; CREN = 1; }
            if(FERR) { ch = RCREG; ch = RCREG; ch = RCREG; }
            if(RCIF) {
                ch = ((RCREG) & 0x7f);
                if(ch >= 'a') ch -= 'a' - 'A';						    // convert to uppercase
                return ch;
            }
        }
    }
    GpsError = true;
    return 0;                                                           // return zero on timeout
}



///////////////////////////////////////////////////////////////////////////////////////////////////////////////
// Convert the GPS date/time (stored in GPSdata) to number of seconds since 1/1/2000
// Also fills totaldays with the nbr of days since 1/1/2000
//
long GPSDateToSeconds(void) {
	long t;
    static int DaysToMonth[13] = {
        0,
        0,
        31,
        31 + 28,
        31 + 28 + 31,
        31 + 28 + 31 + 30,
        31 + 28 + 31 + 30 + 31,
        31 + 28 + 31 + 30 + 31 + 30,
        31 + 28 + 31 + 30 + 31 + 30 + 31,
        31 + 28 + 31 + 30 + 31 + 30 + 31 + 31,
        31 + 28 + 31 + 30 + 31 + 30 + 31 + 31 + 30,
        31 + 28 + 31 + 30 + 31 + 30 + 31 + 31 + 30 + 31,
        31 + 28 + 31 + 30 + 31 + 30 + 31 + 31 + 30 + 31 + 30
	};

	totaldays = DaysToMonth[mth] + year * 365 + ((year + 3)/4) + day - 1;
	if(mth > 2 && (year % 4) == 0) totaldays++;		// leap year. Ignore centuries as the next is in 70+ years

    t = (long)sec + (long)min * 60L + (long)hrs * SECONDS_IN_HOUR + (long)totaldays * SECONDS_IN_DAY;
    return t;
}



///////////////////////////////////////////////////////////////////////////////////////////////////////////////
// Calculate the seconds since 1/1/2000 when daylight saving starts or ends in the current year
// Note:  This relies on year being pre filled with the year (getting data from the GPS does that)
//
long CalcDST(long CurrentTime, int day_of_week, int dstmonth, int dstweek) {
	long t, tt;
	int j, ty;

	sec = min = 0;                                          // set the date for the DST event
    if(Config.TZ == 0L)
        hrs = 1;                                            // The UK is different, DST starts at 1AM
    else
        hrs = 2;                                            // DST starts at 2AM for most countries
	day = 1; mth = dstmonth;                                // the year was filled in by the GPS
	t = GPSDateToSeconds();									// convert to seconds since 1/1/2000
	j = totaldays % 7;                                      // get the day of the week with sun = 0 (because 1/1/2000 is a Sunday)
	j = 7 - j + day_of_week; if(j >= 7) j -= 7;			    // number of days to the first day_of_week (Sunday = 0)
	j += (dstweek - 1) * 7;									// number of days to when DST starts
	t += (long)j * SECONDS_IN_DAY;                          // this is the start time
	if(dstweek == 4) {										// this is a special case.  4 can also mean the last in the month
		tt = t + (7L * SECONDS_IN_DAY);                     // add another week - will this still be in the month?
		ty = year;                                          // save the year in case we change it below
        day = 1;                                            // move to the next month
		if(++mth > 12) {
			mth = 1; year++;                                // adjust if it pushed us into a new year
		}
		if(tt < GPSDateToSeconds()) t = tt;					// retain the new value if it is still in the month
		year = ty;                                          // replace the year in case we changed it
	}

    t -= Config.TZ;                                         // return the time in UTC (ie, GPS time)
    if(!Config.UseDST || t <= CurrentTime) return 0x7fffffff;
	return t;
}


////////////////////////////////////////////////////////////////////////////////
// functions used in getting the configuration data

// print a string to the USB
void prt(const char *s) {
    while(*s) putUSB(*s++);
}

// print two strings to the USB
void prt2(const char *s1, const char *s2) {
    prt("\r\n"); prt(s1); prt(s2); prt(" ? ");
}

// erase a char on the USB terminal emulator
void erase(byte *i) {
    while(*i) { prt("\b \b"); (*i)--; }
}

// simple get a digit with error return
byte GetNbr(char *p) {
    if(*p >= '0' && *p <= '9')
        return *p - '0';
    else
        return 127;
}

// process the time zone entered by the user and return a value in hours * 10
// a return of 127 is an error
// this is heavily optimised to reduce the amount of flash that it uses (which is a lot)
byte GetTZ(char *p) {
    byte r, tz;
    byte negative = 0;

    if(*p == '+') {
        p++;
    } else if(*p == '-') {
        negative = 1;
        p++;
    }

    r = GetNbr(p);
    if(r == 127) return 127;
    tz = r * 10;
    p++;
    if(*p != 0) {
        if(*p == '.') {
            p++;
            r = GetNbr(p);
            if(r == 127) return 127;
            tz += r;
        } else {
            r = GetNbr(p);
            if(r == 127) return 127;
            tz += r;
            if(tz > 12) return 127;
            tz *= 10;
            p++;
            if(*p != 0) {
                if(*p != '.') return 127;
                p++;
                r = GetNbr(p);
                if(r == 127) return 127;
                tz += r;
            }
        }
    }

    if(negative) tz = -tz;
    return tz;
}


////////////////////////////////////////////////////////////////////////////////////
// routines to save/load the configuration data from the high endurance flash
//

// unlock Flash Sequence
void FlashUnlock(void)
{
    #asm
        BANKSEL PMCON2
        MOVLW 0x55
        MOVWF PMCON2 & 0x7F
        MOVLW 0xAA
        MOVWF PMCON2 & 0x7F
        BSF PMCON1 & 0x7F,1 ; set WR bit
        NOP
        NOP
    #endasm
}


// write data to the write buffer or trigger a write of the buffer to the high endurance flash
// note: this assumes that interrupts are enabled
void FlashWrite(unsigned address, unsigned data, char latch)
{
    INTCONbits.GIE = 0;
    PMADR = address;
    PMDAT = data;
    PMCON1bits.LWLO = latch;        // 1 = latch, 0 = write row
    PMCON1bits.CFGS = 0;            // select the Flash address space
    PMCON1bits.FREE = 0;            // next operation will be a write
    PMCON1bits.WREN = 1;            // enable Flash memory write/erase
    FlashUnlock();
    INTCONbits.GIE = 1;
}


// erase one row of the high endurance flash
// note: this assumes that interrupts are enabled
void FlashErase(unsigned address)
{
    INTCONbits.GIE = 0;
    PMADR = address;
    PMCON1bits.CFGS = 0; // select the Flash address space
    PMCON1bits.FREE = 1; // next operation will be an erase
    PMCON1bits.WREN = 1; // enable Flash memory write/erase
    FlashUnlock();
    PMCON1bits.WREN = 0; // disable Flash memory write/erase
    INTCONbits.GIE = 1;
}


// load the configuration structure from high endurance flash
void LoadConfigStruct(void) {
    unsigned flash_addr = 0x1FE0;
    char *config_addr = (char *)&Config;
    byte cnt = sizeof(struct config_struct);
    while(cnt--) {
        PMADR = flash_addr++;
        PMCON1bits.CFGS = 0;    //select the Flash address space
        PMCON1bits.RD = 1;      //next operation will be a read
        NOP();
        NOP();
        *config_addr++ = PMDAT;
    }
}


// save the configuration structure to the high endurance flash
char SaveConfigStruct(void) {
    unsigned flash_addr = 0x1FE0;
    char *config_addr = (char *)&Config;
    byte cnt = sizeof(struct config_struct);
    FlashErase(flash_addr);
    cnt--;
    while (cnt--) {
        //load data in latches without writing
        FlashWrite(flash_addr++, (unsigned) *config_addr++, 1);
    }
    // write the last byte signalling that this and previous bytes should be written to flash
    FlashWrite(flash_addr, (unsigned) *config_addr, 0);
    return PMCON1bits.WRERR;            //0 success, 1 = write error
}


////////////////////////////////////////////////////////////////////////////////
// General functions


// pause execution for a number of milliseconds
// maximum is 780ms
// this assumes a clock speed of 4MHz
void pause(unsigned int i) {
    i = i * 83 - 8;
    while(i--) { CLRWDT(); }
}

void flashLED(byte n) {
    while(n--) {
        PORTCbits.RC4 = 1;
        pause(75);
        PORTCbits.RC4 = 0;
        if(n) pause(250);
    }
}

// manually drive the clock - used to set the seconds hand
void ManualStep(void) {
    byte i = 0;
    PORTCbits.RC4 = 1;                                                  // turn the LED full on
    do {                                                                // while the switch is held down
        if(Config.ClockType == 0) {
            // stepping clock
            TRISCbits.TRISC2 = i & 1;
            PORTCbits.RC2 = (i >> 1) & 1;
            i++;
            CLRWDT();
            pause(40);
            TRISCbits.TRISC2 = i & 1;
            PORTCbits.RC2 = (i >> 1) & 1;
            i++;
            pause(460);
        } else {
            // sweep clock (just use a square wave)
            for(i = 0; i < 16; i++) {
                if(PORTAbits.RA3 != 0) break;
                TRISCbits.TRISC2 = 0;
                PORTCbits.RC2 = 0;
                pause(58);
                PORTCbits.RC2 = 1;
                pause(58);
                CLRWDT();
            }
        }
    } while(PORTAbits.RA3 == 0);
    PORTCbits.RC4 = 0;                                                  // turn off the LED
}


////////////////////////////////////////////////////////////////////////////////
// USB related functions

void putUSB(unsigned char a){
    in_buf[in_count]=a;
    in_count++;
}

void checkConsole(unsigned char time, unsigned char number){
    if(h_millis>=time || in_count>=number){ //output every time/2 msecs or if number characters buffered
        if(in_count){
            while (usb_in_endpoint_busy(2))	;
            memcpy(usb_get_in_buffer(2),in_buf,in_count);
            usb_send_in_buffer(2,in_count);
            in_count=0;
        }
        h_millis=0;
        TMR2=0; //make sure the timer starts  at zero
    }
}


// USB Callbacks.
// nothing useful here
// These function names are set in usb_config.h. */

void app_set_configuration_callback(uint8_t configuration)
{

}

uint16_t app_get_device_status_callback()
{
	return 0x0000;
}

void app_endpoint_halt_callback(uint8_t endpoint, bool halted)
{

}

int8_t app_set_interface_callback(uint8_t interface, uint8_t alt_setting)
{
	return 0;
}

int8_t app_get_interface_callback(uint8_t interface)
{
	return 0;
}

void app_out_transaction_callback(uint8_t endpoint)
{

}

void app_in_transaction_complete_callback(uint8_t endpoint)
{

}

int8_t app_unknown_setup_request_callback(const struct setup_packet *setup)
{
	/* To use the CDC device class, have a handler for unknown setup
	 * requests and call process_cdc_setup_request() (as shown here),
	 * which will check if the setup request is CDC-related, and will
	 * call the CDC application callbacks defined in usb_cdc.h. For
	 * composite devices containing other device classes, make sure
	 * MULTI_CLASS_DEVICE is defined in usb_config.h and call all
	 * appropriate device class setup request functions here.
	 */
	return process_cdc_setup_request(setup);
}

int16_t app_unknown_get_descriptor_callback(const struct setup_packet *pkt, const void **descriptor)
{
	return 0;
}

void app_start_of_frame_callback(void)
{

}

void app_usb_reset_callback(void)
{

}

/* CDC Callbacks. See usb_cdc.h for documentation. */

int8_t app_send_encapsulated_command(uint8_t interface, uint16_t length)
{

	return -1;
}

int16_t app_get_encapsulated_response(uint8_t interface,
                                      uint16_t length, const void **report,
                                      usb_ep0_data_stage_callback *callback,
                                      void **context)
{
	return -1;
}

int8_t app_set_comm_feature_callback(uint8_t interface,
                                     bool idle_setting,
                                     bool data_multiplexed_state)
{
	return -1;
}

int8_t app_clear_comm_feature_callback(uint8_t interface,
                                       bool idle_setting,
                                       bool data_multiplexed_state)
{
	return -1;
}

int8_t app_get_comm_feature_callback(uint8_t interface,
                                     bool *idle_setting,
                                     bool *data_multiplexed_state)
{
	return -1;
}

int8_t app_set_line_coding_callback(uint8_t interface,
                                    const struct cdc_line_coding *coding)
{
    return -1;
}

int8_t app_get_line_coding_callback(uint8_t interface,
                                    struct cdc_line_coding *coding)
{
	return 0;
}

int8_t app_set_control_line_state_callback(uint8_t interface,
                                           bool dtr, bool dts)
{
	return -1;
}

int8_t app_send_break_callback(uint8_t interface, uint16_t duration)
{
    return -1;
}
