;**********************************************************************
;                                                                     *
;    Filename:	    HID_Rev5_A.asm                        *
;    Date:                      20-May 2013                           *
;    File Version:              A                                     *
;                                                                     *
;    Author:                    Pete Griffiths                        *
;    Company:                   picprojects.org.uk                    *
;                                                                     * 
;                                                                     *
;**********************************************************************
;                                                                     *
;    Files Required: P10F202.INC                                      *
;                                                                     *
;**********************************************************************
;                                                                     *
;    Notes:  Code written for HID Technologies                        *
;					*
;    Instructions not present in 12 bit instruction set 	*
;	RETURN  use RETLW			*
;	ADDLW				*
;	SUBLW				*
;					*
;   The stack is only 2 calls deep 			*
;   The CALL instruction can only hold an 8 bit address, 9th bit is   *
;   always 0 so you can only CALL into page 0, but the stack pops	*
;   all 9 bits so you can return from anywhere		*
;					*
;                                                                     *
;**********************************************************************
;
; Rev5. Version A - 20-May-2013
;
; *** Operation *** 
; 
; From off state
;
; Short press - power on and start continuous modes
;   Then short press cycles through modes 1-2-3
;   mode 1 = PWM duty 20%
;   mode 2 = PWM duty 60%  with over temp-detection.  On over temp' drop to 40% duty for 2 minutes then 20% until over temp' clears 
;   mode 3 = PWM duty 100% with over temp-detection. On over temp'  drop to 60% duty for 2 minutes then 20% until over temp' clears  
;
; Long press - power on and start flashing modes 
;   Then short press cycles through modes 4-5-6
;   mode 4 = PWM duty 20%
;   mode 5 = PWM duty 60% 
;   mode 6 = PWM duty 100% 
;
; From any on state (mode 1-6)
;   long press - switch to off state
;
;**********************************************************************
; Output active state.
; Output will normally be active high.
; To assemble code with output active low, uncomment the line below. 
#define	invertOutput


;*******************************************************************************************************
;For HID Tech PCB leave line below commented out.  For Picprojects code development PCB, uncomment line
;#define	devcode


	list      p=10F202            ; list directive to define processor
	#include <p10F202.inc>        ; processor specific variable definitions

	; for development and testing, assemble code with 'Code Protect' disabled. _CP_OFF
	ifdef devcode
	__CONFIG   _MCLRE_OFF & _CP_OFF & _WDT_ON
	endif

	; for production version, assemble code with 'Code Protect' enabled. _CP_ON
	ifndef devcode
	__CONFIG   _MCLRE_OFF & _CP_ON & _WDT_ON
	endif
; '__CONFIG' directive is used to embed configuration word within .asm file.
; The lables following the directive are located in the respective .inc file. 
; See respective data sheet for additional information on configuration word.


;***** VARIABLE DEFINITIONS
	; range is 0x10 to 0x1F on 10F200  (16 bytes)
	; range is 0x08 to 0x1F on 10F202  (24 bytes)
	; so we'll use 0x10 so code runs on either
	cblock 0x10
	mode	;1
	output	;2
	period	;3
	duty	;4
	dutyreload	;5
	flag	;6
	timeLo	;7
	timeHi	;8
	timeUp	;9
	seconds	;10
	flashcounter	;11
	switchDelay	;12
	switch3sec	;13
	endc
	
;***** Flag bits 
Finhibit	equ 0	; When set output will be forced off, regardless of PWM value
Fovertemp	equ 1	; Set when overtemp condition exists
FswitchDn	equ 2	; When switch input is debounced and low, set this flag
FswitchLong	equ 3	; If switch held down for > 3 seconds set this flag


;***** Constants
	constant	Ctime	= .1000000 / .4  ; Seconds counter prescaler
	constant	CovrTmpDly	= .120	; Overtemp delay before shutdown (Seconds)
	constant	CswitchDelay  = .24	; Switch debounce delay (1.024mS x CswitchDelay)

	; If period constant changed, duty cycle constants will need changing also
	constant	Cperiod	= .5	; Duty cycle period (1.024mS x Cperiod)
	constant	Cduty0	= .1	; 0%
	constant	Cduty20	= .2	; 20%
	constant	Cduty40	= .3	; 40%
	constant	Cduty60	= .4	; 60%
	constant	Cduty80	= .5	; 80%
	constant	Cduty100	= .6	; 100%

	

	ifdef devcode
;***** Port bit names Picprojects test PCB
pb.LED	equ     2
pb.switch	equ     0
pb.tsense     equ     1
	endif

	ifndef devcode
;;***** Port bit names HID Tech PCB
pb.LED 	equ 1
pb.switch 	equ 0
pb.tsense   	equ 2
	endif

; Note: no weak pull-up on GP2

; Pin 1 / GP0 -> Switch (since it's normally open it won't affect programming)
; Pin 3 / GP1 -> Output to AMC7140 (this will also be driven by the ICSP clock signal during in-circuit programming)
; Pin 4 / GP2 -> Input from TC6502 
; Pin 6 / GP3 -> leave as dedicated MCLR input. 

	ORG     0x1FF             ; processor reset vector

; Internal RC calibration value is placed at location 0x1FF by Microchip
; as a movlw k, where the k is a literal value.  Make sure it is present
; and correct as timings will be incorrect if isn't.

;-----------------------------------------------------------
;
; Main code section
	; This code intialise the PIC FSR registers and the program variable register
	; It runs once after Power-on-reset, or after a Watch Dog Time-out


	ORG     0x000             ; coding begins here
	goto    start.main

	dt " HID Technologies (C) 2009-2013 "
	dt "Rev 5, Version A"

start.main	movwf   OSCCAL            ; update register with factory cal value 

	clrwdt	
	movlw	~(1<<NOT_GPPU | 1<<T0CS | 1<<PSA | 1<<PS2 | 1<<PS1)
	option		; bits above are set to 0 in option reg.
			; Weak pull-ups enabled GP0,1,3
			; TMR0 Clock Source internal 4Mhz
			; TMR0 Prescaler 1:4

	
	ifdef invertOutput	; Output is active low, set LED port bit high at start-up
	movlw	1<<pb.LED
	movwf	GPIO
	endif

	movlw	~(1<<pb.LED)	; Set LED GPIO port bit to output
	tris 	GPIO
	

	; clear all GPRs to 0x00
	movlw	0x10
	movwf	FSR
clear.gpr	clrf	INDF
	incf	FSR,F
	btfsc	FSR,4	; test for bit 4 set (0x1F -> 0x20)
	goto	clear.gpr
	
	; initialise program variables that need non-zero values at startup
	call	sectimer.load
	call	switch.up	
	
;-----------------------------------------------------------
;
; Main code section

	; 
	
	; Wait in a loop until timer0 reaches 0
	; There are no interrupts on the 10Fxxx PICs so we have to poll the timer.
	; Timer Prescaler 1:4 | Timer is 8 bits (/256) | Clock input is 4Mhz/4 = 1Mhz (1e6)
	; Timer period is ( 4 x 256 ) / 1e6 = 1.024mS
	; All other timings are based on this timer period so if TMR0 prescaler
	; is modified, so must all other timeout constants.

	; With a prescaler of 4, TMR0 will have a value of zero for 4 instruction
	; cycles before incrementing.  The polling loop takes exactly 4 instructions
	; so we will always see TMR0 count through zero. If the polling loop is
	; increased or the prescaler reduced, then this polling loop will fail to
	; detect TMR0 at zero on every cycle.

poll.timer	movfw	TMR0	; Load TMR0 into W (will modify status Z flag)	
	skpz		; If TMR0 reached 0, run control code
	goto	poll.timer	; otherwise continue to poll timer
	
	clrwdt		; clear watchdog timer	

	call	pwm	; Generate the output PWM 
	call	write.port	; update output based on PWM and Flash inhibit setting
	call	sectimer	; update seconds timer
	call	switch	; Poll switch input
	
	;---------------------
	; Run mode code for the current mode setting
	; use computed Goto to select mode code to run
state.select	movfw	mode	; put current mode value into W reg
	addwf	PCL,F	; then add W to PCL to 'select' mode to run
	goto	state.mode0	; 0ff
	goto	state.mode1	;  20%, continuous
	goto	state.mode2	;  60%, continuous
	goto	state.mode3	; 100%, continuous
	goto	state.mode4	;  20%, flash
	goto	state.mode5	;  60%, flash
	goto	state.mode6	; 100%, flash 
	;---------------------

	;Mode 0 - Output off
state.mode0	movlw	Cduty0
	movwf	dutyreload
	goto	poll.timer
	
	; Mode 1 - Output 20% duty cycle
state.mode1	movlw	Cduty20
	movwf	dutyreload
	bcf	flag, Finhibit
	goto	poll.timer
	

	; Mode 2 - Output 60% duty cycle with overtemp monitoring
state.mode2	bcf	flag, Finhibit	; clear output inhibit flag
	call	tsense.in		; check temperature sensor, this code will modify dutyreload if overtemp 
	movlw	Cduty60		; load W with 60% duty
	btfss	flag, Fovertemp	; if overtemp condition exists skip next and leave dutyreload unchanged
	movwf	dutyreload		; load dutyreload variable with W reg
	goto	poll.timer


	; Mode 3 - 100% duty cycle with overtemp monitoring
state.mode3	bcf	flag, Finhibit	; clear output inhibit flag
	call	tsense.in		; check temperature sensor, this code will modify dutyreload if overtemp 
	movlw	Cduty100		; load W with 100% duty value
	btfss	flag, Fovertemp	; if overtemp condition exists skip next and leave dutyreload unchanged
	movwf	dutyreload		; load dutyreload variable with W reg
	goto	poll.timer

	; Mode 4 - 20% duty cycle, 2Hz modulation (flashing)
state.mode4	movlw	Cduty20
	movwf	dutyreload
	call	flashoutput
	goto	poll.timer

	; Mode 5 - 60% duty cycle, 2Hz modulation (flashing)
state.mode5	movlw	Cduty60
	movwf	dutyreload
	call	flashoutput
	goto	poll.timer

	; Mode 6 - 100% duty cycle, 2Hz modulation (flashing)
state.mode6	movlw	Cduty100
	movwf	dutyreload
	call	flashoutput
	goto	poll.timer


	; End of main code loop section
;-------------------------------------------------------------------------------------------------------------------------------
;	
;	Functions
;
;-------------------------------------------------------------------------------------------------------
	; Update GPIO port with output variable

write.port	clrw			; set W reg to 0
	btfss	flag, Finhibit	; see if inhibit flag is set
	movfw	output		; if not, load W with value in output
	
	ifdef invertOutput	; Output is active low, invert LED port bit data
	xorlw	1<<pb.LED
	endif
	
	movwf	GPIO		; write W to GPIO port
	retlw	0x00

;-------------------------------------------------------------------------------------------------------
	; Generate PWM and set bit in output variable
	; write to physical port is done by the write.port function which is called
	; from the main loop
	
pwm	decfsz	period,F
	goto	pwm.run
	
pwm.reload	movlw	Cperiod
	movwf	period
	movfw	dutyreload
	movwf	duty
	bsf	output, pb.LED ; set output port bit high
	
pwm.run	decfsz	duty,F	; decrement duty value and skip next if it has reached 0
	retlw	0x00
	bcf	output, pb.LED ; reset output port bit if duty reached 0
	retlw	0x00


;-------------------------------------------------------------------------------------------------------
	; flash output 
	; works by toggling the inhibit flag which tells the write.port code
	; to override the PWM output setting and force the I/O port pin low.

flashoutput	decfsz	flashcounter,F ; decrement timer until it reaches zero
	retlw	0x00
	movlw	.122	; adjust for 4Hz flash rate (250mS / 1.024)
	movwf	flashcounter	; reload counter
	movlw	1<<Finhibit	
	xorwf	flag,F	; toggle inhibit flag bit
	retlw	0x00
		

;-------------------------------------------------------------------------------------------------------
sectimer	
	; Seconds countdown timer
	; Timer prescaler 1:4
	; Clock 1Mhz
	; 1Mhz / 4 = 250,000  = 0x3D090
	; 
	; | Up  |  Hi  | Lo |
	; 65536 4096 256 16 1
	;     3    D   0  9 0
	; Timer0 is 8 bit counter so it divides 250,000 / 256 which in hex is 
	;  0x3D090 / 0x100 = 0x3D0  
	; 
	;  3D090 -> decrement Up (03) & Hi (D0) bytes until we get 0 00 90
	;
	;  3D090
	;  00090 +  add the remainder
	;  3D120 count down until Up and Hi == 0 so we get 0 00 20
	;     
	;  3D090
	;  00020 + add the remainder
	;  3D0B0 count down until Up and Hi == 0 so we get 0 00 B0
	;
	;  3D090
	;  000B0 + add the remainder
	;  3D140 count down until Up and Hi == 0 sowe get 0 00 40
	;
	; For any given 1 second interval there is an error of +448uS or -576uS, but over 16 seconds
	; adding the remainder to the count each time cancels the error out giving an accurate 
	; interval timer
	
	movlw	-.1
	addwf	timeHi,F	; subtract 1 using 2's compliment
	skpc		; if carry clear there was a borrow
	addwf	timeUp,F	; so subtract 1 
	skpnc		; if carry clear count reached -1
	retlw	0x00
	

sectimer.load	movlw	low Ctime
	addwf	timeLo,F

	movlw	high (Ctime & 0xFFFF)
	skpnc
	movlw	1 + high (Ctime & 0xFFFF)

	addwf	timeHi,F
	skpnc
	incf	timeUp,F

	movlw	upper Ctime
	addwf	timeUp,F

	decf	seconds,F
	decf	switch3sec,F

	retlw	0x00



	
;-------------------------------------------------------------------------------------------------------
	; Mode Switch 
	; Check switch, debounce and increment through modes
	
switch	; This code checks the state of the switch 
	; if the switch is down we run a debounce timer
	; if it's still down after the debounce time out we test if it has been down for longer than 3 seconds
	; if it has we turn it off or enter mode 3 depending on current mode.
	; if switch is released in less than 3 seconds we either 
	; turn on to mode 1 if was off or toggle between modes 1 and 2 on each short press.	


	movfw	GPIO	
	andlw	1<<pb.switch
	skpz
	goto	switch.up


switch.dn	decfsz	switchDelay,F	; count down while switch input is low
	retlw	0x00	; 
	
	incf	switchDelay,F	; force it back to it comes back here while switch is down
	btfsc	flag, FswitchLong	; have we already done 3 seconds press?
	retlw	0x00		; if yes, we've done for this press.
	bsf	flag, FswitchDn	; set switch down flag
	
	movf	switch3sec,F	; test 3 second timeout
	skpz
	retlw	0x00

	movlw	.4		; preload W with mode 4
	movf	mode,F		; test if mode is 0 (off)
	skpz			; if mode is zero skip next
	movlw	.0		; mode wasn't zero so load W with 0 (off)
	movwf	mode		; write W to mode variable
	bsf	flag, FswitchLong	; set flag to indicate switch down > 3 seconds
	retlw	0x00


	
switch.up	btfss	flag, FswitchDn	; Has switch just been down?
	goto	switch.done		; No, goto switch.done
	btfsc	flag, FswitchLong	; If yes, was it down < 3 seconds? 
	goto	switch.done		; No, goto switch.done

	movlw	.4		; Compare current mode
	subwf	mode,W		; 
	skpnc			; If mode between 1-3 cycle through continuous modes
	goto	switch.fl		; ese mode is between 4-6 so cycle through flashing modes

	; Cycle through continuous modes
	bcf	flag, Fovertemp	; to ensure when we change modes, over temp duty is reset
	movf	mode,W		; load mode into W
	xorlw	.3		; compare to highest continuous mode 
	skpnz			; skip next unless already at highest mode
	clrf	mode		; in which case reset mode to 0
	incf	mode,F		; then increment mode so it cycles 1-2-3-1-2-3...
	goto	switch.done		; all done

	; Cycle through flashing modes
switch.fl	movfw	mode
	xorlw	.6
	movlw	.3
	skpnz
	movwf	mode
	incf	mode,F


switch.done	bcf	flag, FswitchDn	; Clear switch flags
	bcf	flag, FswitchLong	; 
	movlw	CswitchDelay		; reload debounce counter
	movwf	switchDelay
	movlw	.3		; force 3 second counter while switch up
	movwf	switch3sec
	retlw	0x00

;-------------------------------------------------------------------------------------------------------
	; Overtemp sensor PWM control

tsense.in	btfss	GPIO, pb.tsense		; is overtemp input true
	goto	tsense.no		; no

	; yes - overtemp sense input is true
	btfss	flag, Fovertemp	; is overtemp flag set
	goto	tsense.yes.flag.no	; no

	; Yes, overtemp is active AND the overtemp flag is set
	; so the overtemp delay timer is now running.
	; so test if its been overtemp for a time > overtempDelay
	movf	seconds,W
	skpz
	goto	tsense.exit	; if overtempDelay not expired then exit

	; If overtemp condition still present after overtemp delay expired
	; then reduce duty to 20%
	movlw	Cduty20
	movwf	dutyreload
	goto	tsense.exit

	
	; If overtemp only just occured, 
	; set overtemp flag
	; start overtemp shutdown delay timer
	; reduce PWM
tsense.yes.flag.no

	bsf 	flag, Fovertemp
	movlw	CovrTmpDly
	movwf	seconds

	movfw	mode
	call	tsense.duty
	movwf	dutyreload
	goto	tsense.exit

	; If no overtemp condition is present then always clear overtemp flag
tsense.no	bcf	flag, Fovertemp


tsense.exit	retlw	0x00


tsense.duty	addwf	PCL,F
	nop		; mode off (can never get here from mode 0)
	retlw	Cduty20	; mode 1
	retlw	Cduty40	; mode 2
	retlw	Cduty60	; mode 3
	retlw	Cduty20	; mode 4
	retlw	Cduty40	; mode 5
	retlw	Cduty60	; mode 6


	; end of functions
;-------------------------------------------------------------------------------------------------------

	

	END                       ; directive 'end of program'

