#define inbuffersize 20
int inbuffer[inbuffersize];
int inbufferend = 0;

#define outbuffersize 20
double outbuffer[outbuffersize];
int outbufferend = 0;

double taps[outbuffersize];

boolean fastnow = false; // tracks whether the ADC is at a higher speed or not

#define Fs 500.0 // sampling frequency
double Fu = 250.0; // upper frequency bound
double Fl = 70.0; // lower frequency bound
unsigned long lastReadTime = 0;
unsigned long Ps = (1000000.0 / Fs) + 0.5; // sampling period, in microseconds - 0.5 is to round it rather than truncate

double talktoolong; // seconds of speech without a pause that's too long
#define pause 3 // seconds of silence that constitute a pause

#define blocklength 0.3 // seconds per measurement block
#define memorylength 65 // seconds stored for evaluate() - should be more than pause+ max talktoolong
const int blockbuffersize = (memorylength / blocklength) + 0.5; // number of blocks that need to be stored
bool blockbuffer[blockbuffersize]; // whether there is speech in each block
int blockbufferend = 0;

const int fasttomedium = inbuffersize; // how often round the fast loop does the medium loop trigger?
const int mediumtoslow = (Fs / (1 / blocklength)) / fasttomedium; // how often round the medium loop does the slow loop trigger?
int fasttomediumcounter = 0;
int mediumtoslowcounter = 0;

int DCoffset = 512;
double threshhold = 1000; // threshhold of audio for speech trigger

bool setupmode = false; // true if the talktoolong control is set all the way to the left for manual level control adjustment

#define LIGHTPIN 3
#define TALKTIMECHANNEL 2
#define AUDIOCHANNEL 0

void setup()
{
  pinMode(LIGHTPIN, OUTPUT); // external LED

  Serial.begin(9600);

  initialise();
}

void initialise()
{
  Serial.println("Restarted");
  setTaps();
  lastReadTime = micros(); // initialise the loop timer
  adc_init();
    
}

void loop()
{
  unsigned long delay, rms;
  int i, j;
  boolean noise;
  long totalvalue=0;

  
  // fast part of the loop - operates at sampling frequency ************************************

  // calculate delay since last sample
  if (micros() <= lastReadTime) { delay = 0; }
  else { delay = micros() - lastReadTime; }

  // delay until next sample
  if (Ps < delay) 
  { Serial.print(Ps); Serial.print(" "); Serial.print(delay); Serial.println(" Sample rate too high"); 
     }
  else { delayMicroseconds(Ps - delay); }

  // note the time
  lastReadTime = micros();

  // read a sample and put it into the buffer
  inbuffer[inbufferend] = analogMeasure(0, true) - DCoffset;
  totalvalue += inbuffer[inbufferend]; // used to adjust DCoffset to keep totalvalue zero
  
  // calculate the filter and put the result into outbuffer
  outbuffer[outbufferend] = 0;
  for (i = 0; i < inbuffersize; i++)
  {
    j = i + inbufferend;
    if (j >= inbuffersize) j -= inbuffersize; // it's a circular buffer
    outbuffer[outbufferend] += (float)inbuffer[j] * taps[i];
  }
  outbufferend++;
  if (outbufferend == outbuffersize) outbufferend = 0; // circular buffer

  // now increment inbufferend and deal with the overflow
  inbufferend++;
  if (inbufferend == inbuffersize)  inbufferend = 0; 
  
  fasttomediumcounter++;
  if (fasttomediumcounter >= fasttomedium)
  {
    fasttomediumcounter = 0;

    // medium part of the loop - roughly once per full outbuffer *******************************

    // calculate the rms value of the signal in outbuffer
    rms = 0;
    for (i = 0; i < outbuffersize; i++)
    {
      rms += (outbuffer[i] * outbuffer[i]) / outbuffersize;
    }

    // compare to threshhold and accumulate
    noise |= (rms > threshhold);
        
    readtalktime(); // read the value of talktime, and set setupmode if necessary

    // if we're in setup mode, use this result to drive the LED directly for manual level adjustment
    if (setupmode)
    {
      lighton(noise,false);
      noise = false;
    }
    else
    {
      mediumtoslowcounter++;
      if (mediumtoslowcounter >= mediumtoslow)
      {
        mediumtoslowcounter = 0;

        // slow part of the loop - once per block ******************************

        if (totalvalue > 0) { DCoffset++; } else { DCoffset--; }; // fix any long-term DC bias
        
        // load the value of noise into the block buffer
        blockbuffer[blockbufferend] = noise;
        blockbufferend++;
        if (blockbufferend == blockbuffersize) { blockbufferend = 0; } // circular buffer
        noise = false; // reset it

        evaluate(); // figure out if they've talked too long, and turn the LED on if they have
      }
    }
  }

}

void lighton(bool on, bool flash) {
  if (on ) 
  { 
    if (flash) 
    {
      if (flashon()) { digitalWrite(LIGHTPIN, HIGH); }
      else { digitalWrite(LIGHTPIN, LOW); }
    }
    else
    {
      digitalWrite(LIGHTPIN, HIGH);
    }
  }
  else { digitalWrite(LIGHTPIN, LOW); }
}

boolean flashon() // true for 1/3 second, then false for 1/3 second
{
  double i;
  i = int(millis() / 300);
  if (int(i / 2) == (i / 2)) { return true; }
  else { return false; }
}

void readtalktime()
{
  bool oldsetupmode = setupmode;
  talktoolong = analogMeasure(TALKTIMECHANNEL, false);
  
  
  setupmode = (talktoolong < 10);
  talktoolong = talktoolong / 1023 ; // scale it to 1
  talktoolong = talktoolong * talktoolong * 60;
  
}

void evaluate()
{
  int b_pause = pause / blocklength; // convert pause to blocks
  int b_talktoolong = talktoolong / blocklength; // convert talktoolong to blocks
  int thisblock = 0; // pointer to circular buffer
  int silences = 0; // number of blocks of continuous silence
  int pauseend = -2; // block at which last pause ended

  for (int t = 0; t < blockbuffersize; t++) // t is how many blocks to go back from current
  {
    thisblock = blockbufferend - t;
    if (thisblock < 0) thisblock += blockbuffersize;

    if (blockbuffer[thisblock])
    {
      silences = 0;
    }
    else
    {
      silences++;
    }

    if (silences >= b_pause)
    {
      pauseend = t - b_pause; // how long ago the pause ended
      t = blockbuffersize; // end of loop
      if (pauseend < b_talktoolong)
      {
        lighton(false,false);
        return;
      } // turn the light off
    }
  }
  
  if (pauseend == -2) { lighton(true,true); } // no pause at all - turn the light on and flash it
  if (pauseend > b_talktoolong) { lighton(true,false); } // paused too long ago - turn the light on
  if (pauseend > (b_talktoolong*1.5)) { lighton(true, true); } // paused too long ago - turn the light on and flash it

}

// set the taps for the filter
void setTaps()
{
  double mm;

  double m_lambda = M_PI * Fl / (Fs / 2.0);
  double m_phi = M_PI * Fu / (Fs / 2.0);
  for (int i = 0; i < outbuffersize; i++)
  {
    mm = (double)i - ((double)outbuffersize - 1.0) / 2.0;
    if (mm == 0.0)
    {
      taps[i] = (m_phi - m_lambda) / M_PI;
    }
    else { taps[i] = (sin(mm * m_phi) - sin(mm * m_lambda)) / (mm * M_PI); }

  }

  double scale = 0;
  for (int i = 0; i < outbuffersize; i++) { scale += taps[i]; }
  for (int i = 0; i < outbuffersize; i++) { taps[i] += taps[i] / scale; }
}

int measure()
{
  int v;
  v = analogMeasure(AUDIOCHANNEL, true);
  return v - 512; // scale to +/- 512
}

int analogMeasure(uint8_t ch, boolean fast)
{
  
  if (fast) 
  {
    if (!fastnow) 
    {
      ADCSRA = (1 << ADEN) | (1 << ADPS2) /* | (1 << ADPS1) | (1 << ADPS0) */ ; // set prescaler to 16
      fastnow = fast; 
    }
  }
  else 
  {
    if (fastnow) 
    {
      ADCSRA = (1 << ADEN) | (1 << ADPS2) | (1 << ADPS1) | (1 << ADPS0); // set prescaler to 128
      fastnow = fast; 
    }
  }

  // select the corresponding channel 0~7
  // ANDing with ’7′ will always keep the value
  // of ‘ch’ between 0 and 7
  ch &= 0b00000111;  // AND operation with 7
  ADMUX = (ADMUX & 0xF8) | ch; // clears the bottom 3 bits before ORing

  // start single convertion
  // write ’1′ to ADSC
  ADCSRA |= (1 << ADSC);

  // wait for conversion to complete
  // ADSC becomes ’0′ again
  // till then, run loop continuously
  while (ADCSRA & (1 << ADSC));

  byte m = ADCL;
  byte j = ADCH;
  int k = ((int)j << 8) | m;

  return k;
}

void adc_init()
{
  // AREF = AVcc
  ADMUX = (1 << REFS0);

  // ADC Enable and prescaler of 128
  // 16000000/128 = 125000
  ADCSRA = (1 << ADEN) | (1 << ADPS2) | (1 << ADPS1) | (1 << ADPS0);
}

