//SD shield has CS on D8
//DHT shield uses D4
//set WiFi credentials here:
#define STASSID "xxxxxxxx"
#define STAPSK "xxxxxxxx"

//simple timekeeping, not worrying about DST
#define NORMAL_OFFSET (600)

#define SD_CS_PIN D8
#define DHT_PIN D4

#include <dhtnew.h>
DHTNEW dhtSensor(DHT_PIN);
float humid,temp;   //globals

#include "SD.h"
File file,root;
char csvFileName[32]="";

#include <ESP8266WiFi.h>
#include <ESP8266HTTPClient.h>
#include <WiFiUdp.h>
unsigned int localPort = 2390;  // local port to listen for UDP packets
const int NTP_PACKET_SIZE = 48;  // NTP time stamp is in the first 48 bytes of the message
byte packetBuffer[NTP_PACKET_SIZE];  // buffer to hold incoming and outgoing packets
WiFiUDP udp;
#define NTP_SERVER "pool.ntp.org"
unsigned long long ntpOffset=0;
unsigned long long epoch=0;
//date/time handling
int date,mon,yr;              //calculated by daysn()
int h,m,s;                    //hour, minute, second
int lastHour,lastDate;        //to check for rollover
bool flag=false;              //does NTP neet update?
char timeBuffer[128];
unsigned long oldMillis=0;    //to help detect rollover
#define CONNECT_TIMEOUT 30000
//HTTP server
#include "ESP8266WebServer.h"
ESP8266WebServer server(80);
#define HOSTNAME "weatherlogger"

char status=0;
#define CARD_OK 2
#define TIME_OK 1

void setup() {
  Serial.begin(115200);
  delay(2000);
  Serial.println("Starting");
  //connect to WiFi first
  WiFi.begin(STASSID, STAPSK);
  WiFi.setHostname(HOSTNAME);
  Serial.printf("Connecting to %s.\r\n", WiFi.SSID().c_str());
  while (WiFi.status() != WL_CONNECTED) {
    delay(500);
    Serial.print(".");
    if(millis()>CONNECT_TIMEOUT){
      Serial.println("\r\nCould not connect, rebooting");
      delay(100);
      ESP.restart();
    }
  }
  Serial.println("");
  Serial.print("Connected! IP address: ");
  Serial.println(WiFi.localIP());
  udp.begin(localPort);
  doTime();     //try once at boot
  updateTime();
  if((status&TIME_OK)==0){
    Serial.println("Retrying time");
    delay(5000);
    doTime();
  }
  lastHour=h;
  lastDate=date;
  //SD card
  if (SD.begin(SD_CS_PIN)){
    Serial.println("SD card initialised");
    root=SD.open("/");
    if(root){
      Serial.println("Root directory found");
      status=status|CARD_OK;
    }else{
      Serial.println("File system error");
    }
  }else{
    Serial.println("SD card error");
  }
  Serial.print(showStatus());
  server.on("/", handleRoot);
  server.onNotFound(handleOther);
  server.begin();
}

void loop(){  
  int d,n;  
  updateTime();
  if(date!=lastDate){ //check and refresh daily
    doTime(); 
    lastDate=date;
  }
  if(status==(TIME_OK|CARD_OK)){
    dhtSensor.read();
    humid=dhtSensor.getHumidity();
    temp=dhtSensor.getTemperature();
  }  
  if(h!=lastHour){  //hour rollover, time to log if time OK
    if((status&TIME_OK)==0){
      Serial.println("Retrying time");
      delayCheckServer(5000);   //avoid rapid checks
      doTime();
    }else{
      dhtSensor.read();
      humid=dhtSensor.getHumidity();
      temp=dhtSensor.getTemperature();
      file=SD.open(csvFileName,FILE_WRITE);
      if(file){
        if(file.size()<5){  //add headers
          file.println("DSN,Date_Time,T,%RH");
          Serial.println("Header added");  
        }
        file.printf("%11.4f,%04d%02d%02d_%02d%02d%02d,%5.1f,%5.1f\r\n",epoch/86400.0+2,yr,mon,date,h,m,s,temp,humid);  //CSV line: DSN, yyyymmdd_hhmmss,t,h, +2 is due to Excel date epoch
        file.close();
        Serial.println("Entry logged");
      }else{
        Serial.println("File error, log failed");
      }
    }
    lastHour=h; //ready for next
  }
  if(Serial.available()){
    d=Serial.read();
    if(d=='s'){ //get status
      Serial.print(showStatus());
    }
    if(d=='~'){         //listing
      Serial.println("Card listing:");
      root.rewindDirectory();
      n=0;
      file=root.openNextFile();
      while(file){
        if(file.isDirectory()){
          Serial.printf("%3d %s  \t[FOLDER]\r\n",n,file.name());
        }else{
          Serial.printf("%3d %s  \t%d bytes\r\n",n,file.name(),file.size());
        }        
        file.close();
        n++;
        file=root.openNextFile();
      }      
      file.close();
    }
    if((d>='0')&&(d<='9')){
      d=d-'0';
      root.rewindDirectory();      
      root.openNextFile();
      while(d--){
        file=root.openNextFile();
      }
      if(file){
        Serial.printf("%s  \t%d bytes\r\n",file.name(),file.size());
        while(file.available()){
          Serial.write(file.read());
        }
      }else{
        Serial.println("File not found");
      }
      file.close();
    }
  }
  delayCheckServer(500);
  server.handleClient();
}

void updateTime(void){  //this happens whenever possible and just before using time
  unsigned long mTemp;
  mTemp=millis();
  if(mTemp<oldMillis){    //rollover has happened
    ntpOffset=ntpOffset+4294967UL   ;//adjust
    Serial.print("millis() has rolled over, offset adjusted.");
    //ESP.restart();    //might be an option to properly reset the time
    delay(100);
  }
  oldMillis=mTemp;
  epoch=ntpOffset+mTemp/1000+(60*NORMAL_OFFSET);   //seconds, non-DST time
  dodsn(epoch/86400UL);   
  h=(epoch/3600)%24;
  m=(epoch/60)%60;
  s=epoch%60;
  sprintf(timeBuffer,"Time is %02d:%02d:%02d on %02d/%02d/%04d (local time)\r\n",h,m,s,date,mon,yr); //hh:mm:ss, dd/mm/yyyy
  getFileName();
  //Serial.println(csvFileName);
}

void doTime(void){
  int cb;
  unsigned long t=0;
  Serial.println("Getting time from NTP");
  sendNTPpacket();
  delay(1000);
  cb=udp.parsePacket();
  if(cb){
    Serial.printf("%d bytes received.\r\n",cb);
    udp.read(packetBuffer, NTP_PACKET_SIZE);  // read the packet into the buffer
    t=packetBuffer[40];
    t=(t<<8)|packetBuffer[41];
    t=(t<<8)|packetBuffer[42];
    t=(t<<8)|packetBuffer[43];
    Serial.printf("NTP is 0x%x\r\n",t);
    oldMillis=millis();
    ntpOffset=t-(oldMillis/1000);
    updateTime();
    Serial.print(timeBuffer);
    status=status|TIME_OK;
  }else{
    Serial.println("No response.");
  }
}

// send an NTP request to the time server at the given address
// modified to use address
void sendNTPpacket(void) {
  Serial.println("sending NTP packet...");
  // set all bytes in the buffer to 0
  memset(packetBuffer, 0, NTP_PACKET_SIZE);
  // Initialize values needed to form NTP request
  packetBuffer[0] = 0b11100011;  // LI, Version, Mode
  packetBuffer[1] = 0;           // Stratum, or type of clock
  packetBuffer[2] = 6;           // Polling Interval
  packetBuffer[3] = 0xEC;        // Peer Clock Precision
  // 8 bytes of zero for Root Delay & Root Dispersion
  packetBuffer[12] = 49;
  packetBuffer[13] = 0x4E;
  packetBuffer[14] = 49;
  packetBuffer[15] = 52;
  // all NTP fields have been given values, now
  // you can send a packet requesting a timestamp:
  udp.beginPacket(NTP_SERVER, 123);  // NTP requests are to port 123
  udp.write(packetBuffer, NTP_PACKET_SIZE);
  udp.endPacket();
}

void dodsn(unsigned long dsn){
  byte dim[]={0,31,28,31,30,31,30,31,31,30,31,30,31};   //days in month
  yr=1900;
  mon=1;
  date=dsn+1;                       //1/1/1900 has timestamp 0, ie date of 0 is 1
  while(date>365){                  //count years             
    if(((yr%4)==0)&&(yr!=1900)){
      date--;
    }
    yr++;
    date=date-365;
  }
  if(((yr%4)==0)&&(yr!=1900)){dim[2]++;}  //if a leap year, Feb has 29
  while(date>dim[mon]){                   //count months
    date=date-dim[mon];
    mon++;
  }
  if((date==0)&&(mon==1)){                //glitch on 31/12 of leap years
    yr--;
    date=31;
    mon=12;
  }
}

void getFileName(void){ //put "yyyymm.csv" csvFileName
  sprintf(csvFileName,"/%04d%02d.csv",yr,mon);
}

const char* showStatus(void){
  static char statusStr[128]="";
  sprintf(statusStr,"STATUS: IP=%s, %s, %s\r\n",ipStr(),(status&CARD_OK)?"Card OK":"Card error",(status&TIME_OK)?"NTP OK":"NTP not updated");
  //Serial.print(statusStr);
  return statusStr;
}

const char* ipStr(void){
  static String ip = WiFi.localIP().toString();
  return ip.c_str();
}

void delayCheckServer(unsigned long n){ //delay for n millis, but check server
  unsigned long t0=millis();
  while((millis()-t0)<n){
    delay(10);
    server.handleClient();
  }
}

void handleRoot(void){  
  char s[128]="";   //local for use
  updateTime();
  Serial.println("Server hit: /");
  server.setContentLength(CONTENT_LENGTH_UNKNOWN);
  server.send(200, "text/html", "");
  server.sendContent("<head><title>WiFi Weather Logger</title></head><body><h2>WiFi Weather Logger</h2>");
  sprintf(s,"Temp: %5.1f&deg;C, Humidity: %5.1f&percnt;RH<p>",temp,humid);
  server.sendContent(s);
  file=SD.open(csvFileName,FILE_READ);  //check existence of current file
  if(file){
    sprintf(s,"Current file:<a href=\"%s\">%s</a><p>",file.name(),file.name());
    file.close();
    server.sendContent(s);
  }  
  server.sendContent(showStatus());
  server.sendContent("<p>Card file listing:<br>");
  root.rewindDirectory();
  file=root.openNextFile();
  while(file){
    if(!file.isDirectory()){
      sprintf(s,"<a href=\"%s\">%s</a>      %d bytes<br>",file.name(),file.name(),file.size());
      server.sendContent(s);
    }        
    file.close();    
    file=root.openNextFile();    
  }      
  file.close();
  server.sendContent("<p>");
  server.sendContent(timeBuffer);
  server.sendContent("</body>");
  server.client().stop();
}

void handleOther(void){ //files and 404's
  char s[130]="";
  int n=1;
  String uri = ESP8266WebServer::urlDecode(server.uri());
  Serial.print("Server hit:");
  Serial.println(uri);
  file=SD.open(uri.c_str(),FILE_READ);  //read so we can go from start
  if(file){
    server.setContentLength(CONTENT_LENGTH_UNKNOWN);
    server.send(200, "text/csv", "");
    while(n){
      n=file.readBytes(s,128);
      s[n]=0;   //null term
      //Serial.printf("Sending %d bytes\r\n",n);
      server.sendContent(s);
    }
    file.close();
    server.client().stop();
  }else{
    server.setContentLength(CONTENT_LENGTH_UNKNOWN);
    server.send(404, "text/html", "");
    server.sendContent("<head><title>WiFi Weather Logger</title></head><body><h2>WiFi Weather Logger</h2>");
    server.sendContent(uri);
    server.sendContent(" not found<p>");
    server.sendContent("<a href=\"/\">Click here to go back</a>");
    server.client().stop();    
  }
}