// Zener Diode Tester
// Arduino Mega + Custom PCB + 2.8" SPI LCD
// I/V test does a sweep and graphs the result
// LED test, current and voltage limited mode, displays stats

//for EEPROM check and save
#include <EEPROM.h>
char eepromid[]="SC04112181";   //DIODE TESTER PCB
char checkid[12]="";

//SYSTEM COLOURS
#define BACKGROUND_COLOUR BLACK
#define FOREGROUND_COLOUR WHITE
#define TEXT_COLOUR WHITE
#define AXIS_COLOUR ((RED_1+GREEN_1+BLUE_1)*20)
#define GRID_COLOUR ((RED_1+GREEN_1+BLUE_1)*10)
#define HIGHLIGHT_COLOUR CYAN
#define ERROR_COLOUR RED
#define OVERSAMPLE 32

//a utility to allowing printing to an internal buffer to assist formatting
#include "awriter.h"
char nbuf[21]="";
awriter n;

//backpack library
#include "backpack.h"
backpack bp;

//fonts for display
#include "arial.h"
#include "SmallFont.h"

//logo
#include "sclogo.h"

//gui interface
#include "guiobjects.h"
#define GUI_OBJECT_COUNT 10
guiObject gui[GUI_OBJECT_COUNT];
#define GUI_TEST 0
#define GUI_SETTINGS 1
#define GUI_LED 2
#define GUI_REVERSE 3
#define GUI_BACK 4
//only used on settings screen
#define GUI_NEXT 5
#define GUI_PREV 6
#define GUI_UP 7
#define GUI_DOWN 8
#define GUI_CALIBRATE 9

guiObject guiCup,guiCdown,guiVup,guiVdown;

// for backpack on Diode Tester
#define LCD_RS (5)
#define LCD_CS (7)
#define TOUCH_CS (2)
#define LCD_RESET (6)

// for Diode tester
//OPTOCTL should be PWM for fine control
#define OPTOCTL (10)
#define RLYCTL (4)
#define BOOSTCTL (3)
#define R29CON (A5)
#define R30CON (A4)
#define R32CON (9)
#define R33CON (8)
#define VTEST1B (A3)
#define VTEST2B (A0)
#define ITEST (A2)
#define VHHA (A1)

//Calibration constants
//float VHHconst=0.1106445;                          // V/adc count
//float VTEST1const[4]={0,0.1106445,0.0182,0.00449}; //3 gain settings V/adc count
//float VTEST2const[4]={0,0.1106445,0.0182,0.00449}; //3 gain settings V/adc count
//float ITESTconst=0.0446;                       // mA/adc count

float VHHconst=0.11064453125;     //V/ADC
float VTEST1const[4]={0,0.11064453125,0.0127771326013514,0.00299039273648649};    //V/ADC
float VTEST2const[4]={0,0.11064453125,0.0127771326013514,0.00299039273648649};    //V/ADC
float ITESTconst=0.0322265625;     //mA/ADC

char gainMode1=1;
char gainMode2=1;
float targetPower=10;                           //mW = V x mA

#define SAMPLECOUNT 100
float vRes[SAMPLECOUNT],iRes[SAMPLECOUNT];
float graphVscale=5,graphIscale=20;
float vCursor=0,iCursor=0;

#define OPTION_COUNT 7
const char PROGMEM optionText[OPTION_COUNT][20]={
  "  Standard Device  ",
  "   Maximum Power   ",
  "   Target Power    ",
  "   Current Limit   ",
  "   Voltage Limit   ",  
  "Current Graph Scale",
  "Voltage Graph Scale"
};

enum deviceTypes{
  MANUAL=1,
  _400mW=2,
  _1000mW=3
};

int optionValues[OPTION_COUNT][4]={     // value in use/min/max/step
  {_400mW,1,3,1,},    //use enum for values
  {400,50,1000,50},    //as per 400mW device
  {100,50,1000,50},    //as per 400mW device
  {30,1,30,1},        //max device Current
  {100,5,100,1},      //max device Voltage
  {30,5,50,5},        //current scale
  {5,5,100,5}         //voltage scale
};

const char PROGMEM optionUnits[OPTION_COUNT][3]={   //matching optionValues
  "  ",   //not used
  "mW",
  "mW",
  "mA",
  "V ",
  "mA",
  "V "
};

const char PROGMEM deviceText[4][20]={
  "                   ",
  "     MANUAL SET    ",
  "400mW (100mWtarget)",
  " 1W (250mW target) ",  
};

const char PROGMEM calText[8][20]={
  "   VHH constant   ",
  "V1 Gain 1 constant",
  "V1 Gain 2 constant",
  "V1 Gain 3 constant",
  "V2 Gain 1 constant",
  "V2 Gain 2 constant",
  "V2 Gain 3 constant",
  " Current constant "  
};


void setup() {
  analogReference(EXTERNAL);       // 3.3V, before we do anything else
  Serial.begin(115200);
  loadFromEEPROM();
  graphVscale=optionValues[6][0];
  graphIscale=optionValues[5][0];
  n.begin(nbuf,21);
  pinMode(LCD_RESET,OUTPUT);
  digitalWrite(LCD_RESET,LOW);
  delay(10);
  digitalWrite(LCD_RESET,HIGH);
  bp.init(LCD_RS,LCD_CS,TOUCH_CS);
  bp.rotate(4);
  bp.clear(BACKGROUND_COLOUR);
  guiInit();                     //set up locations etc
  pinMode(BOOSTCTL,OUTPUT);
  digitalWrite(BOOSTCTL,LOW);    //turn off boost
  pinMode(RLYCTL,OUTPUT);
  digitalWrite(RLYCTL,LOW);      //turn off relay
  analogWrite(OPTOCTL,0);         //opto off
  TCCR2B = TCCR2B & 0b11111000 | 0x1;   //set PWM freq of Timer2 to 31.250kHz, affects D10, D11 on Mega
  //TCCR3B = TCCR3B & 0b11111000 | 0x1;   //set PWM freq of Timer3 to 31.250kHz, affects D2, D3, D5 on Mega (check it works)
  setGain(1,gainMode1);
  setGain(2,gainMode2);
  drawMainPage();
}

void loop() {             //aka main menu
  int x,y,i;
  float f,fp;
  if(guiButton(gui[GUI_REVERSE])){
    reverseGraph();
  }
  if(guiButton(gui[GUI_BACK])){
    bp.clear(BACKGROUND_COLOUR);
    drawMainPage();
    }
  if(guiButton(gui[GUI_LED])){
    LEDTest();
  }
  if(guiButton(gui[GUI_CALIBRATE])){
    doCalibrate();
  }
  if(guiButton(gui[GUI_SETTINGS])){
    doSettings();
    graphVscale=optionValues[6][0];   //load new scale
    graphIscale=optionValues[5][0];
  }
  if(guiButton(gui[GUI_TEST])){
    graphTest();
  }
  if(gui[GUI_REVERSE].visible){     //only visible on graph page
    x=bp.touchx();
    y=bp.touchy();  
    if(x>160){
      if(y<120){    //top right of screen- change I cursor
        iCursor=graphIscale*(120-y)/120;    //iCursor is in mA
        vCursor=0;
        f=findTargetCurrentIndexFloat(iCursor);
        if(f>-0.5){
          i=f;        //integer part
          fp=f-i;      //fractional part
          vCursor=vRes[i]*(1-fp)+vRes[i+1]*fp;
        }
        drawAuxBox(f,20,150);
        drawIVCursors(graphVscale,graphIscale);
        delay(100);     //help stop flickering on repeated redraw          
      }else{        //bottom right, change V cursor
        vCursor=graphVscale*(x-160)/160;    //iCursor is in mA
        iCursor=0;
        f=findTargetVoltageIndexFloat(vCursor);
        if(f>-0.5){
          i=f;        //integer part
          fp=f-i;      //fractional part
          iCursor=iRes[i]*(1-fp)+iRes[i+1]*fp;
        }        
        drawAuxBox(f,20,150);
        drawIVCursors(graphVscale,graphIscale);          
        delay(100);     //help stop flickering on repeated redraw          
      }
    }
  }
}

void doCalibrate(){
  int sOption=0;        //which item are we editing  
  gui[GUI_BACK].y=10;   //move back button to top
  gui[GUI_CALIBRATE].visible=false;
  gui[GUI_SETTINGS].visible=false;
  gui[GUI_LED].visible=false;
  gui[GUI_TEST].visible=false;
  gui[GUI_REVERSE].visible=false;
  gui[GUI_BACK].visible=true;
  gui[GUI_NEXT].visible=true;
  gui[GUI_PREV].visible=true;
  bp.clear(BACKGROUND_COLOUR);
  guiDrawAll();
  calDraw(sOption);
  bp.chararrayfont(64,110,"ADC:",HIGHLIGHT_COLOUR,BACKGROUND_COLOUR,Arial_round_16x24); 
  bp.chararrayfont(32,140,"Value:",HIGHLIGHT_COLOUR,BACKGROUND_COLOUR,Arial_round_16x24); 
  while(!guiButton(gui[GUI_BACK])){
    if(guiButton(gui[GUI_NEXT])){
      sOption++;
      if(sOption>7){sOption=7;}
      calDraw(sOption);
    }
    if(guiButton(gui[GUI_PREV])){
      sOption--;
      if(sOption<0){sOption=0;}
      calDraw(sOption);
    }
    calOutputs(sOption);
    delay(50);            //reduce screen update
  }
  digitalWrite(BOOSTCTL,LOW);    //turn off boost
  analogWrite(OPTOCTL,0);        //current off
  gui[GUI_NEXT].visible=false;
  gui[GUI_PREV].visible=false;
  gui[GUI_BACK].y=80;
  bp.clear(BACKGROUND_COLOUR);
  drawMainPage();
}

void calOutputs(int opt){      //setup state for cals, return current ADC value as appropriate
  int i,port;
  float retval;
  switch(opt){
    case 0:     //VHH
      analogWrite(BOOSTCTL,127);      //turn on boost, halfway
      analogWrite(OPTOCTL,0);         //current off
      digitalWrite(RLYCTL,LOW);       //turn off relay
      bpFloat(128,140,getVHH(),2,"V",HIGHLIGHT_COLOUR,BACKGROUND_COLOUR,Arial_round_16x24);
      port=VHHA;      
      break;
    case 1:     //V1 Gain #1
      analogWrite(BOOSTCTL,127);      //turn on boost
      analogWrite(OPTOCTL,255);       //turn on to power TEST+
      digitalWrite(RLYCTL,LOW);       //turn off relay
      bpFloat(128,140,getV1(),2,"V",HIGHLIGHT_COLOUR,BACKGROUND_COLOUR,Arial_round_16x24);
      port=VTEST1B;
      gainMode1=1;
      setGain(1,gainMode1);
      break;
    case 2:     //V1 Gain #2
      analogWrite(BOOSTCTL,0);        //turn off boost
      analogWrite(OPTOCTL,255);       //turn on to power TEST+
      digitalWrite(RLYCTL,LOW);       //turn off relay
      bpFloat(128,140,getV1(),2,"V",HIGHLIGHT_COLOUR,BACKGROUND_COLOUR,Arial_round_16x24);
      port=VTEST1B;
      gainMode1=2;
      setGain(1,gainMode1);
      break;
    case 3:     //V1 Gain #3
      analogWrite(BOOSTCTL,0);        //turn off boost
      analogWrite(OPTOCTL,160);       //turn on to power TEST+
      digitalWrite(RLYCTL,HIGH);       //turn on relay
      bpFloat(128,140,getV1(),2,"V",HIGHLIGHT_COLOUR,BACKGROUND_COLOUR,Arial_round_16x24);
      port=VTEST1B;
      gainMode1=3;
      setGain(1,gainMode1);
      break;
    case 4:     //V2 Gain #1
      analogWrite(BOOSTCTL,127);      //turn on boost
      analogWrite(OPTOCTL,255);       //turn on to power TEST-
      digitalWrite(RLYCTL,HIGH);       //turn on relay
      bpFloat(128,140,getV2(),2,"V",HIGHLIGHT_COLOUR,BACKGROUND_COLOUR,Arial_round_16x24);
      port=VTEST2B;
      gainMode2=1;
      setGain(2,gainMode2);
      break;
    case 5:     //V2 Gain #2
      analogWrite(BOOSTCTL,0);        //turn off boost
      analogWrite(OPTOCTL,255);       //turn on to power TEST-
      digitalWrite(RLYCTL,HIGH);       //turn on relay
      bpFloat(128,140,getV2(),2,"V",HIGHLIGHT_COLOUR,BACKGROUND_COLOUR,Arial_round_16x24);
      port=VTEST2B;
      gainMode2=2;
      setGain(2,gainMode2);
      break;
    case 6:     //V2 Gain #3
      analogWrite(BOOSTCTL,0);        //turn off boost
      analogWrite(OPTOCTL,160);       //turn on to power TEST-
      digitalWrite(RLYCTL,LOW);       //turn off relay
      bpFloat(128,140,getV2(),2,"V",HIGHLIGHT_COLOUR,BACKGROUND_COLOUR,Arial_round_16x24);
      gainMode2=3;
      setGain(2,gainMode2);
      port=VTEST2B;
      break;
    case 7:     //current
      analogWrite(BOOSTCTL,0);        //turn off boost
      analogWrite(OPTOCTL,255);       //turn on to pass current
      digitalWrite(RLYCTL,LOW);       //turn off relay
      bpFloat(128,140,getI(),1,"mA",HIGHLIGHT_COLOUR,BACKGROUND_COLOUR,Arial_round_16x24);
      port=ITEST;
      break;
    default:    //use any other value to shut down everything
      analogWrite(BOOSTCTL,0);        //turn off boost
      analogWrite(OPTOCTL,0);        //current off
      digitalWrite(RLYCTL,LOW);       //turn off relay
      break;
  }
  retval=0;
  for(i=0;i<OVERSAMPLE;i++){
    retval=retval+analogRead(port);
  }
  retval=retval/OVERSAMPLE;
  if(retval>9999){retval=9999;}
  if(retval<0){retval=0;}
  n.clear();
  n.print(retval,1);
  n.padLeft(' ',6);
  bp.chararrayfont(144,110,nbuf,HIGHLIGHT_COLOUR,BACKGROUND_COLOUR,Arial_round_16x24); 
}

void calDraw(int sOption){
  int i;
  n.clear();
  for(i=0;i<strlen_P(calText[sOption]); i++){
    n.write(pgm_read_byte_near(&calText[sOption][i]));
  }
  bp.chararrayfont(8,50,nbuf,TEXT_COLOUR,BACKGROUND_COLOUR,Arial_round_16x24);
  if((sOption==3)||(sOption==6)||(sOption==7)){
    bp.chararrayfont(8,80,"Terminals shorted",TEXT_COLOUR,BACKGROUND_COLOUR,Arial_round_16x24);    
  }else{
    bp.chararrayfont(8,80,"  Terminals open ",TEXT_COLOUR,BACKGROUND_COLOUR,Arial_round_16x24);        
  }
  switch(sOption){
    case 0:
      bp.chararrayfont(168,20," TP1 TP3 ",TEXT_COLOUR,BACKGROUND_COLOUR,Arial_round_16x24);    
      break;
    case 1:
    case 2:
    case 3:
      bp.chararrayfont(168,20,"Term+ TP3",TEXT_COLOUR,BACKGROUND_COLOUR,Arial_round_16x24);    
      break;    
    case 4:
    case 5:
    case 6:
      bp.chararrayfont(168,20,"Term- TP3",TEXT_COLOUR,BACKGROUND_COLOUR,Arial_round_16x24);    
      break;    
    case 7:
      bp.chararrayfont(168,20,"Terms + -",TEXT_COLOUR,BACKGROUND_COLOUR,Arial_round_16x24);    
      break;      
  }
}

void doSettings(){
  int sOption=0;        //which item are we editing  
  gui[GUI_BACK].y=10;   //move back button to top
  gui[GUI_CALIBRATE].visible=false;
  gui[GUI_SETTINGS].visible=false;
  gui[GUI_LED].visible=false;
  gui[GUI_TEST].visible=false;
  gui[GUI_REVERSE].visible=false;
  gui[GUI_BACK].visible=true;
  gui[GUI_NEXT].visible=true;
  gui[GUI_PREV].visible=true;
  gui[GUI_UP].visible=true;
  gui[GUI_DOWN].visible=true;  
  bp.clear(BACKGROUND_COLOUR);
  guiDrawAll();
  optionDraw(sOption);
  while(!guiButton(gui[GUI_BACK])){
    if(guiButton(gui[GUI_NEXT])){
      sOption++;
      if(sOption>OPTION_COUNT-1){sOption=OPTION_COUNT-1;}
      optionDraw(sOption);    
    }
    if(guiButton(gui[GUI_PREV])){
      sOption--;
      if(sOption<0){sOption=0;}
      optionDraw(sOption);    
    }
    if(guiButton(gui[GUI_UP])){
      optionValues[sOption][0]=optionValues[sOption][0]+optionValues[sOption][3];
      if(optionValues[sOption][0]>optionValues[sOption][2]){optionValues[sOption][0]=optionValues[sOption][2];}
      checkOptionState();
      optionDraw(sOption);    
    }
    if(guiButton(gui[GUI_DOWN])){
      optionValues[sOption][0]=optionValues[sOption][0]-optionValues[sOption][3];
      if(optionValues[sOption][0]<optionValues[sOption][1]){optionValues[sOption][0]=optionValues[sOption][1];}      
      checkOptionState();
      optionDraw(sOption);    
    }
  }
  saveToEEPROM();         //save on leaving settings page
  gui[GUI_NEXT].visible=false;
  gui[GUI_PREV].visible=false;
  gui[GUI_UP].visible=false;
  gui[GUI_DOWN].visible=false;
  gui[GUI_BACK].y=80;
  bp.clear(BACKGROUND_COLOUR);
  drawMainPage();
}

void checkOptionState(){    //set power parameters based on standard device
  if(optionValues[0][0]==_400mW){   //400mW device
    optionValues[1][0]=400;
    optionValues[2][0]=100;
  }
  if(optionValues[0][0]==_1000mW){   //1W device
    optionValues[1][0]=1000;
    optionValues[2][0]=250;
  }
}

void optionDraw(int sOption){
  int i;
  n.clear();
  for(i=0;i<strlen_P(optionText[sOption]); i++){
    n.write(pgm_read_byte_near(&optionText[sOption][i]));
  }
  bp.chararrayfont(8,50,nbuf,TEXT_COLOUR,BACKGROUND_COLOUR,Arial_round_16x24);
  if(sOption==0){   //use string descriptor
    n.clear();
    for(i=0;i<strlen_P(deviceText[optionValues[0][0]]); i++){
      n.write(pgm_read_byte_near(&deviceText[optionValues[0][0]][i]));
    }
    bp.chararrayfont(8,80,nbuf,HIGHLIGHT_COLOUR,BACKGROUND_COLOUR,Arial_round_16x24);
  }else{            //otherwise numeric
    n.clear();
    n.print(optionValues[sOption][0]);
    n.write(pgm_read_byte_near(&optionUnits[sOption][0]));  //we know these are 2 chars long
    n.write(pgm_read_byte_near(&optionUnits[sOption][1]));
    n.padLeft(' ',13);
    n.padRight(' ',19); //roughly centre
    bp.chararrayfont(8,80,nbuf,HIGHLIGHT_COLOUR,BACKGROUND_COLOUR,Arial_round_16x24);
  }
}

void LEDTest(){
  bool cupState=0,cdownState=0,vupState=0,vdownState=0,bState,iClip=0,vClip=0;
  int iLim,vLim;      //limits
  int n=0;              //counter to limit screen updates
  float i,v;
  float VHHtemp,v1,v2;
  iLim=10;            //load from settings eventually
  vLim=5;        
  int opto=0;
  int boost=0;
  guiLEDTest();
  bpFloat(80,135,vLim,2,"V",TEXT_COLOUR,BACKGROUND_COLOUR,Arial_round_16x24);
  bpFloat(80,175,iLim,2,"mA",TEXT_COLOUR,BACKGROUND_COLOUR,Arial_round_16x24);
  while(!guiButton(gui[GUI_BACK])){
    bState=checkPress(guiCup);
    if(bState&&(!cupState)){
      iLim=iLim+1;
      if(iLim>30){iLim=30;}    
      bpFloat(80,175,iLim,2,"mA",TEXT_COLOUR,BACKGROUND_COLOUR,Arial_round_16x24);
    }
    cupState=bState;
    bState=checkPress(guiCdown);
    if(bState&&(!cdownState)){
      iLim=iLim-1;
      if(iLim<0){iLim=0;}    
      bpFloat(80,175,iLim,2,"mA",TEXT_COLOUR,BACKGROUND_COLOUR,Arial_round_16x24);
    }
    cdownState=bState;
    bState=checkPress(guiVup);
    if(bState&&(!vupState)){
      vLim=vLim+1;
      if(vLim>100){vLim=100;}    
      bpFloat(80,135,vLim,2,"V",TEXT_COLOUR,BACKGROUND_COLOUR,Arial_round_16x24);
    }
    vupState=bState;
    bState=checkPress(guiVdown);
    if(bState&&(!vdownState)){
      vLim=vLim-1;
      if(vLim<5){vLim=5;}    
      bpFloat(80,135,vLim,2,"V",TEXT_COLOUR,BACKGROUND_COLOUR,Arial_round_16x24);
    }
    vdownState=bState;
    gainMode1=3;   //reset gain
    gainMode2=3;
    setGain(1,gainMode1);
    setGain(2,gainMode2);
    i=getI();
    v1=getV1();
    v2=getV2();
    v=abs(v1-v2);
    VHHtemp=getVHH();
    while((VHHtemp<vLim+min(v1,v2))&&(boost<255)&&(n<10)){   //set voltage, plus headroom for shunt
      boost++;
      analogWrite(BOOSTCTL,boost);
      delay(1);
      v1=getV1();
      v2=getV2();
      VHHtemp=getVHH();
      v=abs(v1-v2);
      n++;
    }
    while((VHHtemp>vLim+min(v1,v2))&&(boost>0)&&(n<25)){
      boost--;
      analogWrite(BOOSTCTL,boost);
      delay(1);
      v1=getV1();
      v2=getV2();
      VHHtemp=getVHH();
      v=abs(v1-v2);
      n++;
    }
    while((opto<255)&&(i<iLim)&&(n<35)){   //set current
      opto++;
      analogWrite(OPTOCTL,opto);
      delay(1);
      i=getI();
      v1=getV1();
      v2=getV2();
      v=abs(v1-v2);
      n++;
    }
    while((opto>0)&&(i>iLim)&&(n<50)){   //set current
      opto--;
      analogWrite(OPTOCTL,opto);
      delay(1);
      i=getI();
      v1=getV1();
      v2=getV2();
      v=abs(v1-v2);
      n++;
    }
    delay(5);      // a little time to settle   
    gainMode1=3;   //reset gain
    gainMode2=3;
    setGain(1,gainMode1);
    setGain(2,gainMode2);
    i=getI();
    v1=getV1();
    v2=getV2();
    v=abs(v1-v2);
    VHHtemp=getVHH();
    if(i>iLim+1){iClip=1;}    //margin of error
    if(v>vLim+1){vClip=1;}      //margin of error
    n++;
    if(n>20){                          //only update display intermittently for legibility
      n=0;
      if(iClip){
        bp.chararrayfont(290,35,"*",ERROR_COLOUR,BACKGROUND_COLOUR,Arial_round_16x24);
        iClip=0;
      }else{
        bp.chararrayfont(290,35," ",ERROR_COLOUR,BACKGROUND_COLOUR,Arial_round_16x24);         
      }
      if(vClip){
        bp.chararrayfont(290,5,"*",ERROR_COLOUR,BACKGROUND_COLOUR,Arial_round_16x24);
        vClip=0;
      }else{
        bp.chararrayfont(290,5," ",ERROR_COLOUR,BACKGROUND_COLOUR,Arial_round_16x24);         
      }
      bpFloat(160,5,v,2,"V",TEXT_COLOUR,BACKGROUND_COLOUR,Arial_round_16x24);
      bpFloat(160,35,i,2,"mA",TEXT_COLOUR,BACKGROUND_COLOUR,Arial_round_16x24);
      bpFloat(160,65,i*v,0,"mW",TEXT_COLOUR,BACKGROUND_COLOUR,Arial_round_16x24);
      if(i>0.001){
        if(v/i<9){
          bpFloat(160,95,v*1000/i,1,"~",TEXT_COLOUR,BACKGROUND_COLOUR,Arial_round_16x24);
        }else if(v/i<9000){
          bpFloat(160,95,v/i,1,"k~",TEXT_COLOUR,BACKGROUND_COLOUR,Arial_round_16x24);        
        }else{      
          bpFloat(160,95,v/i/1000,1,"M~",TEXT_COLOUR,BACKGROUND_COLOUR,Arial_round_16x24);        
        }      
      }else{
        bp.chararrayfont(160,95,"    ---~",TEXT_COLOUR,BACKGROUND_COLOUR,Arial_round_16x24);      
      }
      bpFloat(108,210,getVHH(),2,"V",TEXT_COLOUR,BACKGROUND_COLOUR,Arial_round_16x24);
    }
  }
  analogWrite(BOOSTCTL,0);    //turn off boost
  analogWrite(OPTOCTL,0);             //turn off
  gui[GUI_BACK].y=80;
  bp.clear(BACKGROUND_COLOUR);
  drawMainPage();
}

void reverseGraph(){
  float vResT[SAMPLECOUNT],iResT[SAMPLECOUNT];
  int x,y,i;
  float c,v;
  for(i=0;i<SAMPLECOUNT;i++){   //copy
    iResT[i]=iRes[i];
    vResT[i]=vRes[i];
  }
  for(i=0;i<SAMPLECOUNT;i++){   //reverse sign and order
    iRes[i]=-iResT[SAMPLECOUNT-1-i];
    vRes[i]=-vResT[SAMPLECOUNT-1-i];
  }  
  dumpData();
  drawGraph(graphVscale,graphIscale);           //
  drawPowerPoint(graphVscale,graphIscale);      //use same scale       
  gui[GUI_CALIBRATE].visible=false;
  gui[GUI_SETTINGS].visible=false;
  gui[GUI_LED].visible=false;
  gui[GUI_TEST].visible=true;
  gui[GUI_REVERSE].visible=true;
  gui[GUI_BACK].visible=true;
  guiDrawAll();  
}


void graphTest(){
  int x,y,i;
  float c,v;
  //digitalWrite(BOOSTCTL,HIGH);    //turn on boost
  gui[GUI_TEST].text="RUNNING";
  gui[GUI_CALIBRATE].visible=false;
  gui[GUI_SETTINGS].visible=false;
  gui[GUI_LED].visible=false;
  guiDrawAll();
  gui[GUI_TEST].text="I/V Test";
  targetPower=optionValues[2][0];
  for(i=0;i<SAMPLECOUNT/2;i++){
    doSample(-i,SAMPLECOUNT/2-i-1);    //start at 0, ramp up on negative side
  }
  digitalWrite(RLYCTL,LOW);     //relay off
  analogWrite(BOOSTCTL,0);    //turn off boost
  analogWrite(OPTOCTL,0);   //shutdown
  for(i=0;i<SAMPLECOUNT/2;i++){
    doSample(i,SAMPLECOUNT/2+i);    //start at 0, ramp up on positive side
  }
  digitalWrite(RLYCTL,LOW);     //relay off
  analogWrite(BOOSTCTL,0);    //turn off boost
  analogWrite(OPTOCTL,0);   //shutdown
  dumpData();
  drawGraph(graphVscale,graphIscale);           //draw from -5 to 5V, -20 to 20mA
  drawPowerPoint(graphVscale,graphIscale);      //use same scale       
  gui[GUI_TEST].visible=true;
  gui[GUI_REVERSE].visible=true;
  gui[GUI_BACK].visible=true;
  guiDrawAll();
}

void dumpData(){        //puts data to serial port
  int i;
  Serial.println("V(V)\tI(mA)\tP(mW)");
  for(i=0;i<SAMPLECOUNT;i++){
    Serial.print(vRes[i],4);
    Serial.write('\t');
    Serial.print(iRes[i],4);    
    Serial.write('\t');
    Serial.println(iRes[i]*vRes[i],4);    
  }  
}

void drawIVCursors(float vScale,float iScale){
  static int oldx=159,oldy=121;
  int x,y;
  int i,j,k;
  int x1,x2,y1,y2;  
  //erase previous
  if(oldy<121){
    bp.line(158,oldy,153,oldy+5,BACKGROUND_COLOUR);
    bp.line(158,oldy,153,oldy-5,BACKGROUND_COLOUR);
  }
  if(oldx>159){
    bp.line(oldx,122,oldx+5,127,BACKGROUND_COLOUR);
    bp.line(oldx,122,oldx-5,127,BACKGROUND_COLOUR);
  }
  //redraw scale and axes that might have been erased
  bp.vline(160,0,239,AXIS_COLOUR);
  bp.hline(0,120,319,AXIS_COLOUR);  
  j=1;
  if(vScale>10){j=5;}
  if(vScale>50){j=10;}
  for(i=j;i<vScale;i=i+j){
    k=(160*i)/vScale;
    bp.vline(160+k,122,127,GRID_COLOUR);
    bpNumber(161+k,122,i,"",AXIS_COLOUR,BACKGROUND_COLOUR,SmallFont);
  }
  j=1;
  if(iScale>10){j=5;}
  for(i=j;i<iScale;i=i+j){
    k=(120*i)/iScale;
    bp.hline(153,120-k,158,GRID_COLOUR);     
  }
  //draw plots that might have been erased
  x1=160+(160*vRes[0])/vScale;
  y1=120-(120*iRes[0])/iScale;    //y is inverted compared to cartesian
  for(i=0;i<SAMPLECOUNT-1;i++){
    x2=x1;
    y2=y1;
    x1=160+(160*vRes[i+1])/vScale;
    y1=120-(120*iRes[i+1])/iScale;    //y is inverted compared to cartesian
    if((vRes[i]>-vScale)&&(vRes[i]<vScale)&&(iRes[i]>-iScale)&&(iRes[i]<iScale)){   //1st in bounds
      if((vRes[i+1]>-vScale)&&(vRes[i+1]<vScale)&&(iRes[i+1]>-iScale)&&(iRes[i+1]<iScale)){   //2nd in bounds
        bp.line(x1,y1,x2,y2,FOREGROUND_COLOUR);
      }
    }
  }
  drawAuxBoxLine(findTargetPowerIndexFloat(),graphVscale,graphIscale);
  //calculate new
  x=160+(vCursor*160)/vScale;
  y=120-(iCursor*120)/iScale;
  if(x>319){x=319;}
  if(y<0){y=0;}
  //draw if ok
  if(y<120){
    bp.line(158,y,153,y+5,HIGHLIGHT_COLOUR);
    bp.line(158,y,153,y-5,HIGHLIGHT_COLOUR);
  }
  if(x>160){
    bp.line(x,122,x+5,127,HIGHLIGHT_COLOUR);
    bp.line(x,122,x-5,127,HIGHLIGHT_COLOUR);
  }
  oldx=x;
  oldy=y;
}

void drawGraph(float vScale,float iScale){
  int i,j,x1,x2,y1,y2,k;
  if(vScale<0.0001){return;}          //dodge /0, also negative scale
  if(iScale<0.0001){return;}          //dodge /0, also negative scale
  //todo add axes etc  
  bp.clear(BACKGROUND_COLOUR);
  j=1;
  if(vScale>10){j=5;}
  if(vScale>50){j=10;}
  for(i=j;i<vScale;i=i+j){
    k=(160*i)/vScale;
    bp.vline(160+k,0,239,GRID_COLOUR);
    bp.vline(160-k,0,239,GRID_COLOUR);      
    bpNumber(161+k,122,i,"",AXIS_COLOUR,BACKGROUND_COLOUR,SmallFont);
    bpNumber(161-k,122,-i,"",AXIS_COLOUR,BACKGROUND_COLOUR,SmallFont);
  }
  j=1;
  if(iScale>10){j=5;}
  for(i=j;i<iScale;i=i+j){
    k=(120*i)/iScale;
    bp.hline(0,120-k,319,GRID_COLOUR);     
    bp.hline(0,120+k,319,GRID_COLOUR);     
    bpNumber(162,121-k,i,"",AXIS_COLOUR,BACKGROUND_COLOUR,SmallFont);
    bpNumber(162,121+k,-i,"",AXIS_COLOUR,BACKGROUND_COLOUR,SmallFont);
  }
  bp.chararrayfont(161,0,"I(mA)",AXIS_COLOUR,BACKGROUND_COLOUR,SmallFont);  
  bp.chararrayfont(288,134,"V(V)",AXIS_COLOUR,BACKGROUND_COLOUR,SmallFont);    
  bp.vline(160,0,239,AXIS_COLOUR);
  bp.hline(0,120,319,AXIS_COLOUR);  
  
  x1=160+(160*vRes[0])/vScale;
  y1=120-(120*iRes[0])/iScale;    //y is inverted compared to cartesian
  for(i=0;i<SAMPLECOUNT-1;i++){
    x2=x1;
    y2=y1;
    x1=160+(160*vRes[i+1])/vScale;
    y1=120-(120*iRes[i+1])/iScale;    //y is inverted compared to cartesian
    if((vRes[i]>-vScale)&&(vRes[i]<vScale)&&(iRes[i]>-iScale)&&(iRes[i]<iScale)){   //1st in bounds
      if((vRes[i+1]>-vScale)&&(vRes[i+1]<vScale)&&(iRes[i+1]>-iScale)&&(iRes[i+1]<iScale)){   //2nd in bounds
        bp.line(x1,y1,x2,y2,FOREGROUND_COLOUR);
      }
    }
  }
}

void drawPowerPoint(float vScale,float iScale){
  float f,v,c,p;
  int i;
  //interpolating float version
  f=findTargetPowerIndexFloat();
  //f=findTargetVoltageIndexFloat(2);
  //f=findTargetCurrentIndexFloat(3);
  drawAuxBox(f,210,150);
  drawAuxBoxLine(f,graphVscale,graphIscale);
}

void drawAuxBoxLine(float f,float vScale,float iScale){
  int i;
  float v,c;
  if(f>-0.5){
    i=f;        //integer part
    f=f-i;      //fractional part
    v=vRes[i]*(1-f)+vRes[i+1]*f;
    c=iRes[i]*(1-f)+iRes[i+1]*f;
    if((v<vScale)&&(c<iScale)){       //line draw fails outside screen
      bp.line(160+(160*v)/vScale,120-(120*c)/iScale,300,150,HIGHLIGHT_COLOUR);
    }
  }  
}

void drawAuxBox(float f,int x,int y){
  float v,c,p;
  int i;
  bp.box(x,y,x+90,y+70,BACKGROUND_COLOUR);
  bp.hline(x,y,x+90,HIGHLIGHT_COLOUR);
  bp.hline(x,y+70,x+90,HIGHLIGHT_COLOUR);
  bp.vline(x,y,y+70,HIGHLIGHT_COLOUR);
  bp.vline(x+90,y,y+70,HIGHLIGHT_COLOUR);
  if(f>-0.5){
    i=f;        //integer part
    f=f-i;      //fractional part
    v=vRes[i]*(1-f)+vRes[i+1]*f;
    c=iRes[i]*(1-f)+iRes[i+1]*f;
    bpFloat(x+10,y+5,v,2,"V",HIGHLIGHT_COLOUR,BACKGROUND_COLOUR,SmallFont);
    bpFloat(x+10,y+20,c,2,"mA",HIGHLIGHT_COLOUR,BACKGROUND_COLOUR,SmallFont);
    bpFloat(x+10,y+35,c*v,0,"mW",HIGHLIGHT_COLOUR,BACKGROUND_COLOUR,SmallFont);
    if(iRes[i+1]-iRes[i]>0.001){
      p=(vRes[i+1]-vRes[i])/(iRes[i+1]-iRes[i]);
      if(p<9.0){
        bpFloat(x+10,y+50,p*1000.0,1,"~",HIGHLIGHT_COLOUR,BACKGROUND_COLOUR,SmallFont);        // tilde => ohms
      }else if(p<9000.0){
        bpFloat(x+10,y+50,p,1,"k~",HIGHLIGHT_COLOUR,BACKGROUND_COLOUR,SmallFont);        // tilde => ohms        
      }else{
        bpFloat(x+10,y+50,p/1000.0,1,"M~",HIGHLIGHT_COLOUR,BACKGROUND_COLOUR,SmallFont);        // tilde => ohms
      }
    }else{
      bp.chararrayfont(x+10,y+50,"---",HIGHLIGHT_COLOUR,BACKGROUND_COLOUR,SmallFont);      
    }
  }    
}

float findTargetPowerIndexFloat(){   //find index V/I pairs near power point with interpolation
  int i,retval;
  float f,fp,fstep,t;
  retval=-1;
  for(i=0;i<SAMPLECOUNT-1;i++){
    if((vRes[i]>0)&&(iRes[i]>0)){       //1st quadrant only
      if((vRes[i]*iRes[i]<=targetPower)&&(vRes[i+1]*iRes[i+1]>targetPower)){
        retval=i;
        break;
      }
    }
  }
  if(retval<0){return -1;}
  f=0.5;
  fstep=0.25;
  for(i=0;i<15;i++){      //iterations - 10=>2^10~1000
    fp=1-f;
    t=(vRes[retval]*fp+vRes[retval+1]*f)*(iRes[retval]*fp+iRes[retval+1]*f);
    if(t>targetPower){
      f=f-fstep;
    }else{
      f=f+fstep;      
    }
    fstep=fstep/2;
  }
  return retval+f;
}

float findTargetVoltageIndexFloat(float v){   //find index matching V with interpolation
  int i,retval;
  retval=-1;
  for(i=0;i<SAMPLECOUNT-1;i++){
    if(vRes[i+1]>0){       //1st quadrant only
      if((vRes[i]<=v)&&(vRes[i+1]>v)){
        retval=i;
        break;
      }
    }
  }
  if(retval<0){return -1;}
  return retval+(v-vRes[i])/(vRes[i+1]-vRes[i]);
}

float findTargetCurrentIndexFloat(float c){   //find index matching I with interpolation
  int i,retval;
  retval=-1;
  for(i=0;i<SAMPLECOUNT-1;i++){
    if(iRes[i+1]>0){       //1st quadrant only
      if((iRes[i]<=c)&&(iRes[i+1]>c)){
        retval=i;
        break;
      }
    }
  }
  if(retval<0){return -1;}
  return retval+(c-iRes[i])/(iRes[i+1]-iRes[i]);
}

void doSample(int c, int i){     //do sample near target current
  static int opto =0;
  static int boost=0;
  float iTemp,vTemp,VHHtemp,v1,v2;
  iTemp=getI();
  v1=getV1();
  v2=getV2();
  vTemp=abs(v1-v2);
  VHHtemp=getVHH();
  if(c==0){     //reset on sample start
    opto=0;
    boost=0;
    analogWrite(OPTOCTL,opto);
    analogWrite(BOOSTCTL,boost);    //turn off boost
    gainMode1=3;   //reset gain
    gainMode2=3;
    setGain(1,gainMode1);
    setGain(2,gainMode2);
  }else{
    while((VHHtemp<optionValues[4][0]+min(v1,v2))&&(boost<255)){   //set voltage to max, plus headroom for shunt
      boost++;
      analogWrite(BOOSTCTL,boost);
      delay(1);
      v1=getV1();
      v2=getV2();
      VHHtemp=getVHH();
    }
    while((VHHtemp>optionValues[4][0]+min(v1,v2))&&(boost>0)){    //if it needs to reduce for any reason
      boost--;
      analogWrite(BOOSTCTL,boost);
      delay(1);
      v1=getV1();
      v2=getV2();
      VHHtemp=getVHH();
    }
    while((iTemp<(abs(c)*float(optionValues[3][0]*2)/SAMPLECOUNT))&&(opto<255)&&(iTemp<optionValues[3][0])&&(vTemp<optionValues[4][0])&&(vTemp*iTemp<optionValues[1][0])){   //ramp up current, stay <256, <iMax,<vMax,<pMax
      opto++;
      analogWrite(OPTOCTL,opto);
      delay(1);
      iTemp=getI();
      v1=getV1();
      v2=getV2();
      vTemp=abs(v1-v2);
    }
  }
  if(c<0){                  
    if(digitalRead(RLYCTL)==LOW){   //extra time to settle on relay change
      digitalWrite(RLYCTL,HIGH);
      delay(500);
    }
  }else{
    if(digitalRead(RLYCTL)==HIGH){
      digitalWrite(RLYCTL,LOW);
      delay(500);
    }
  }
  delay(10);            //settle
  vRes[i]=getV1()-getV2();
  if(digitalRead(RLYCTL)){
    iRes[i]=-getI();
  }else{
    iRes[i]=getI();    
  }
}

float getV1(){        //read V1, taking into account all factors
  int a=0;
  int i;
  float retval;
  for(i=0;i<OVERSAMPLE;i++){
    a=a+analogRead(VTEST1B);
  }
  retval=a*VTEST1const[gainMode1]/OVERSAMPLE;
  //auto adjust gain if necessary
  if((gainMode1>1)&&(a/OVERSAMPLE>900)){
    gainMode1=gainMode1-1;
    setGain(1,gainMode1);
    delay(1);
    retval=getV1();
  }
  return retval;
}

float getV2(){        //read V2, taking into account all factors
  int a=0;
  int i;
  float retval;
  for(i=0;i<OVERSAMPLE;i++){
    a=a+analogRead(VTEST2B);
  }
  retval=a*VTEST2const[gainMode2]/OVERSAMPLE;
  //auto adjust gain if necessary
  if((gainMode2>1)&&(a/OVERSAMPLE>900)){
    gainMode2=gainMode2-1;
    setGain(2,gainMode2);
    delay(1);
    retval=getV2();
  }
  return retval;
}

float getI(){
  int i,a;
  a=0;
  for(i=0;i<OVERSAMPLE;i++){
    a=a+analogRead(ITEST);
  }    
  return a*ITESTconst/OVERSAMPLE;
}

float getVHH(){
  int i,a;
  a=0;
  for(i=0;i<OVERSAMPLE;i++){
    a=a+analogRead(VHHA);
  }    
  return a*VHHconst/OVERSAMPLE;  
}

void setGain(char n,char s){      //set gain on opamp n to level s
  if(n==1){
    switch(s){
      case 1:     //level 1 - unity
        pinMode(R29CON,INPUT);
        pinMode(R30CON,INPUT);
        break;
      case 2:     //level 2  ~x8
        pinMode(R29CON,OUTPUT);
        digitalWrite(R29CON,LOW);
        pinMode(R30CON,INPUT);
        break;
      case 3:     //level 3 ~x37
        pinMode(R29CON,INPUT);
        pinMode(R30CON,OUTPUT);
        digitalWrite(R30CON,LOW);
        break;
    }
  }else if(n==2){
    switch(s){
      case 1:     //level 1 - unity
        pinMode(R32CON,INPUT);
        pinMode(R33CON,INPUT);
        break;
      case 2:     //level 2  ~x8
        pinMode(R32CON,OUTPUT);
        digitalWrite(R32CON,LOW);
        pinMode(R33CON,INPUT);
        break;
      case 3:     //level 3 ~x37
        pinMode(R32CON,INPUT);
        pinMode(R33CON,OUTPUT);
        digitalWrite(R33CON,LOW);
        break;
    }  
  }
}

void bpNumber(int x,int y,int i,char u[],unsigned int c,unsigned int b,const uint8_t* font){
  n.clear();
  n.print(i);
  n.print(u);
  bp.chararrayfont(x,y,nbuf,c,b,font);  
}

void bpFloat(int x,int y,float i,int dp,char u[],unsigned int c,unsigned int b,const uint8_t* font){
  n.clear();
  n.print(i,dp);
  n.print(u);
  n.padLeft(' ',8);
  bp.chararrayfont(x,y,nbuf,c,b,font);  
}

void guiInit(){
  int i;
  gui[GUI_TEST].text="I/V Test";
  gui[GUI_TEST].y=10;
  gui[GUI_SETTINGS].text="Settings";
  gui[GUI_SETTINGS].y=45;
  gui[GUI_LED].text="LED Test";
  gui[GUI_LED].y=10;
  gui[GUI_REVERSE].text="Reverse";
  gui[GUI_REVERSE].y=45;
  gui[GUI_BACK].text="Back";
  gui[GUI_BACK].y=80;
  gui[GUI_NEXT].text="Next";
  gui[GUI_NEXT].y=180;
  gui[GUI_PREV].text="Previous";
  gui[GUI_PREV].y=180;
  gui[GUI_UP].text="Up";
  gui[GUI_UP].y=140;
  gui[GUI_DOWN].text="Down";
  gui[GUI_DOWN].y=140;
  gui[GUI_CALIBRATE].text="Calib.";
  gui[GUI_CALIBRATE].y=45;
  for(i=0;i<GUI_OBJECT_COUNT;i++){
    gui[i].x=10;
    gui[i].w=140;
    gui[i].h=30;
    gui[i].value=0;
    gui[i].pressed=false;
    gui[i].visible=false;
    gui[i].enabled=true;
    gui[i].type=BUTTON;
    //guiDraw(gui[i]);    //no point if not visible
  }
  gui[GUI_NEXT].x=170;
  gui[GUI_UP].x=170;
  gui[GUI_LED].x=170;
  gui[GUI_CALIBRATE].x=170;
}

void guiDrawAll(){
  int i;
  for(i=0;i<GUI_OBJECT_COUNT;i++){  
    guiDraw(gui[i]);
  }
}

bool guiButton(guiObject g){
  bool t;
  if(g.visible==false){return false;}
  if(g.enabled==false){return false;}
  t=checkPress(g);
  if(t){
    g.pressed=true;
    guiDraw(g);
    delay(10);
    while(checkPress(g)){}      //wait for release
    g.pressed=false;
    guiDraw(g);
    return true;                            //press and release occurred
  }
  return false;
}

void guiLEDTest(){      //setup GUI for LED test
  gui[GUI_BACK].y=10;
  gui[GUI_SETTINGS].visible=false;
  gui[GUI_CALIBRATE].visible=false;
  gui[GUI_LED].visible=false;
  gui[GUI_TEST].visible=false;
  gui[GUI_REVERSE].visible=false;
  gui[GUI_BACK].visible=true;
  bp.clear(BACKGROUND_COLOUR);
  guiDrawAll();
  guiCup.x=250;
  guiCup.y=170;
  guiCup.w=30;
  guiCup.h=30;
  guiCup.pressed=false;
  guiCup.visible=true;
  guiCup.enabled=true;
  guiCup.type=BUTTON;
  guiCup.text=">";
  guiCdown.x=40;
  guiCdown.y=170;
  guiCdown.w=30;
  guiCdown.h=30;
  guiCdown.pressed=false;
  guiCdown.visible=true;
  guiCdown.enabled=true;
  guiCdown.type=BUTTON;
  guiCdown.text="<";
  guiVup.x=250;
  guiVup.y=130;
  guiVup.w=30;
  guiVup.h=30;
  guiVup.pressed=false;
  guiVup.visible=true;
  guiVup.enabled=true;
  guiVup.type=BUTTON;
  guiVup.text=">";
  guiVdown.x=40;
  guiVdown.y=130;
  guiVdown.w=30;
  guiVdown.h=30;
  guiVdown.pressed=false;
  guiVdown.visible=true;
  guiVdown.enabled=true;
  guiVdown.type=BUTTON;
  guiVdown.text="<";
  guiDraw(guiCup);
  guiDraw(guiCdown);
  guiDraw(guiVup);
  guiDraw(guiVdown);
  bp.chararrayfont(60,210,"V =",TEXT_COLOUR,BACKGROUND_COLOUR,Arial_round_16x24);        
  bp.chararrayfont(76,225,"HH",TEXT_COLOUR,BACKGROUND_COLOUR,SmallFont);          
}

void drawMainPage(){
  gui[GUI_TEST].visible=true;         //main page buttons
  gui[GUI_CALIBRATE].visible=true;
  gui[GUI_SETTINGS].visible=true;
  gui[GUI_LED].visible=true;
  gui[GUI_REVERSE].visible=false;
  gui[GUI_BACK].visible=false;
  guiDrawAll();
  //bp.imaget(75,100,sclogo,0);
  bp.image(75,100,sclogo);
  //bp.chararrayfont(64,130,"Silicon Chip",YELLOW,BACKGROUND_COLOUR,Arial_round_16x24);        
  bp.chararrayfont(72,170,"Zener Diode",YELLOW,BACKGROUND_COLOUR,Arial_round_16x24);        
  bp.chararrayfont(112,200,"Tester",YELLOW,BACKGROUND_COLOUR,Arial_round_16x24);        
}

void writeEEPROMid(char* id){                       //write id to end of eeprom
  int i;
  unsigned int s=EEPROM.length();
  unsigned int n=strlen(id);
  if(n>s-2){return;}                //fail if it won't fit
  for(i=0;i<n+1;i++){
    EEPROM.write(s-n-1+i,id[i]);    
  }  
}

void readEEPROMid(char* id,unsigned int idsize){      //read idsize+null bytes into id buffer
  unsigned int i=0;
  byte b=1;                 //to start in while loop
  unsigned int s=EEPROM.length();
  unsigned int p=s-idsize-1;
  if(p<0){return;}
  while(b&&(p<s)){
    b=EEPROM.read(p);
    id[i]=b;
    i++;
    p++;
  }  
}

bool strmatch(char* x, char* y){   //true if strings match
  unsigned int n;
  n=strlen(x);
  if(n!=strlen(y)){return 0;}       //can't match if different lengths
  for(int i=0;i<n;i++){
    if(x[i]!=y[i]){return 0;}      //different at some point
  }
  return 1;                        //all match
}

void loadFromEEPROM(){
  int i;
  bool eepromOK=1;              //flag if there is a problem  
  readEEPROMid(checkid,strlen(eepromid));       //read id flag
  if(!strmatch(checkid,eepromid)){
    Serial.println(F("EEPROM signature not found, loading and saving defaults"));
    saveToEEPROM();                   //save current values
    writeEEPROMid(eepromid);          //set id flag in EEPROM
  }else{
    Serial.println(F("EEPROM signature found"));
  }
  for(i=0;i<OPTION_COUNT;i++){
    EEPROM.get(1+i*2,optionValues[i][0]);   //ints 2 bytes, start at 1 in case of errant writes
  }
  for(i=0;i<OPTION_COUNT;i++){
    if(optionValues[i][0]<optionValues[i][1]){eepromOK=0;Serial.print(F("Option minimum fail:"));Serial.println(i);}//out of bound
    if(optionValues[i][0]>optionValues[i][2]){eepromOK=0;Serial.print(F("Option maximum fail:"));Serial.println(i);}
  }
  if(!eepromOK){
    Serial.println(F("EEPROM Error, defaults loaded but not saved"));
    setDefaults();
  }else{
    Serial.println(F("EEPROM loaded OK"));
  }
}

void saveToEEPROM(){
  int i;
  for(i=0;i<OPTION_COUNT;i++){
    EEPROM.put(1+i*2,optionValues[i][0]);   //ints 2 bytes, start at 1 in case of errant writes
  }
}

void setDefaults(){
  optionValues[0][0]=_400mW;
  optionValues[1][0]=400;
  optionValues[2][0]=100;
  optionValues[3][0]=30;
  optionValues[4][0]=100;
}

