#include "Arduino.h"
#include "backpack.h"
#include <SPI.h>

backpack::backpack(){}    //nothing to do

void backpack::init(char dc,char cs,char touchcs){
  _dcpin=dc;
  _cspin=cs;
  _touchcspin=touchcs;
  SPI.begin();
  SPI.setClockDivider(SPI_CLOCK_DIV2);  //8MHz
  pinMode(_dcpin,OUTPUT);
  digitalWrite(_dcpin,LOW);
  pinMode(_cspin,OUTPUT);
  digitalWrite(_cspin,HIGH);
  pinMode(_touchcspin,OUTPUT);
  digitalWrite(_touchcspin,HIGH);
  delay(40);
  digitalWrite(_cspin,LOW);  
  cmd(0xC0);           //power control
  data(0x23);
  cmd(0xC1);
  data(0x10);
  cmd(0xC5);           //VCOM control
  data(0x3e);
  data(0x28); 
  cmd(0xC7);
  data(0x86);  
  cmd(0x36);           //Memory Access Control 
  data(0x48);              //normal orientation=1
  cmd(0x3A);           //Pixel format 0x55=16bpp, 0x66=18bpp
  data(0x55);              
  cmd(0xB1);           //Frame Control 
  data(0x00);  
  data(0x18); 
  cmd(0xB6);           //Display Function Control 
  data(0x08); 
  data(0x82);
  data(0x27);  
  cmd(0x11);           //Sleep out
  delay(120); 
  cmd(0x29);           //Display on
  //cmd(0x2c);         //Drawing mode
  digitalWrite(_cspin,HIGH);
  rotate(1);
  areaset(0,0,_width-1,_height-1);
}

int backpack::width(){
  return _width;
}

int backpack::height(){
  return _height;
}

void backpack::data(unsigned char d){
  SPI.transfer(d);  
}

void backpack::cmd(unsigned char d){
  digitalWrite(_dcpin,LOW);
  SPI.transfer(d);  
  digitalWrite(_dcpin,HIGH);  
}

void backpack::rotate(char a){
  digitalWrite(_cspin,LOW);  
  switch(a){
    case 1:
    cmd(0x36);           //Memory Access Control 
    data(0x48);              //1=top is 12 o'clock
    _width=240;
    _height=320;
    _rotation=1;
    break;
    case 2:
    cmd(0x36);           //Memory Access Control 
    data(0x28);              //2=top is 3 o'clock
    _width=320;
    _height=240;
    _rotation=2;
    break;
    case 3:
    cmd(0x36);           //Memory Access Control 
    data(0x88);              //3=top is 6 o'clock
    _width=240;
    _height=320;
    _rotation=3;
    break;
    case 4:
    cmd(0x36);           //Memory Access Control 
    data(0xE8);              //4=top is 9 o'clock
    _width=320;
    _height=240;
    _rotation=4;
    break;
  }
  digitalWrite(_cspin,HIGH);
}

void backpack::areaset(int x1,int y1,int x2,int y2){
  if(x2<x1){int i=x1;x1=x2;x2=i;}   //sort x
  if(y2<y1){int i=y1;y1=y2;y2=i;}   //sort y
  digitalWrite(_cspin,LOW);  
  cmd(42);               //set x bounds  
  data(x1>>8);
  data(x1);
  data(x2>>8);
  data(x2);
  cmd(43);               //set y bounds
  data(y1>>8);
  data(y1);
  data(y2>>8);
  data(y2);
  cmd(44);               //drawing data to follow
  digitalWrite(_cspin,HIGH);
}

void backpack::box(int x1,int y1,int x2,int y2,unsigned int c){
  if(x2<x1){int i=x1;x1=x2;x2=i;}
  if(y2<y1){int i=y1;y1=y2;y2=i;}
  areaset(x1,y1,x2,y2);
  x2++;
  y2++;
  digitalWrite(_cspin,LOW);  
  for(int x=x1;x<x2;x++){
    for(int y=y1;y<y2;y++){
      data(c>>8);
      data(c);  
    }
  }
  digitalWrite(_cspin,HIGH);
}

void backpack::clear(unsigned int c){
  box(0,0,_width-1,_height-1,c);
}

void backpack::point(int x,int y, unsigned int c){
  areaset(x,y,x,y);
  digitalWrite(_cspin,LOW);  
  data(c>>8);
  data(c);  
  digitalWrite(_cspin,HIGH);  
}

void backpack::line(int x1,int y1,int x2,int y2, unsigned int c){
  int steps,stepsx,stepsy,xinc,yinc,x,y,d;
  stepsx=abs(x1-x2);
  stepsy=abs(y1-y2);
  steps=max(stepsx,stepsy)+1;   //if start and end are the same, there's still 1 point
  xinc=constrain(x2-x1,-1,1);
  yinc=constrain(y2-y1,-1,1);
  x=x1;
  y=y1;  
  if(stepsx>stepsy){
    d=stepsx/2;
    for(int i=0;i<steps;i++){
      point(x,y,c);
      x=x+xinc;
      d=d+stepsy;
      if(d>stepsx){d=d-stepsx;y=y+yinc;}
    }
  }else{
    d=stepsy/2;
    for(int i=0;i<steps;i++){
      point(x,y,c);
      y=y+yinc;
      d=d+stepsx;
      if(d>stepsy){d=d-stepsy;x=x+xinc;}
    } 
  }  
}

void backpack::character(int x,int y,char c, unsigned int f, unsigned int b){
  c=c-32;
  if(c<0){c=0;}                     //valid chars only
  if(c>96){c=0;}
  areaset(x,y,x+11,y+15);    //set area
  digitalWrite(_cspin,LOW);  
  for(byte v=0;v<16;v++){
    for(byte u=0;u<12;u++){
      unsigned int d=pgm_read_word(&font1216[c][u]);
      if((1<<v)&d){data(f>>8);data(f);}
      else{data(b>>8);data(b);}
    }
  }
  digitalWrite(_cspin,HIGH);  
}

int backpack::charfont(int x0, int y0, char c, unsigned int f, unsigned int b, const uint8_t* font){     //x0,y0 character c, f on b colour, font is point in progmem
  unsigned int w,h,c0,c1,p;
  const uint8_t* p0;
  char d;
  w=pgm_read_byte(font);
  h=pgm_read_byte(font+1);
  c0=pgm_read_byte(font+2);
  c1=pgm_read_byte(font+3)+c0;
  if(c<c0){return 0;}   //out of range
  if(c>c1){return 0;}   //out of range
  p0=font+4+(w*h*(c-c0))/8;
  //Serial.println(w);
  //Serial.println(h);
  //Serial.println(c0);
  //Serial.println(c1);
  areaset(x0,y0,x0+w-1,y0+h-1);    //set area
  digitalWrite(_cspin,LOW);  
  for(p=0;p<w*h;p++){
    if((p&7)==0){
      d=pgm_read_byte(p0+(p/8));
    }
    if(d&128){
      data(f>>8);data(f);
    }else{
        data(b>>8);data(b);
    }
    d=d<<1;
  }
  digitalWrite(_cspin,HIGH);
  return w;             //allows arrays to be printed in a row
}


void backpack::chararray(int x,int y,char *c, unsigned int f, unsigned int b){
  while(*c){
    character(x,y,*c++,f,b);
    x=x+12;
    if(x>_width-12){x=0;y=y+16;}      //wrap around (will probably look ugly)
  }  
}

void backpack::chararrayfont(int x0, int y0, char *c, unsigned int f, unsigned int b, const uint8_t* font){     //x0,y0 character c, f on b colour, font is point in progmem
  while(*c){
    x0=x0+charfont(x0,y0,*c++,f,b,font);
    if(x0>_width){x0=0;}      //wrap around (will probably look ugly)
  }  
}

void backpack::hline(int x1,int y1,int x2,unsigned int c){
  if(x2<x1){int i=x1;x1=x2;x2=i;}
  areaset(x1,y1,x2,y1);
  x2++;
  digitalWrite(_cspin,LOW);  
  for(int x=x1;x<x2;x++){
    data(c>>8);
    data(c);  
  }
  digitalWrite(_cspin,HIGH);  
}

void backpack::vline(int x1,int y1,int y2,unsigned int c){
  if(y2<y1){int i=y1;y1=y2;y2=i;}
  areaset(x1,y1,x1,y2);
  y2++;
  digitalWrite(_cspin,LOW);  
  for(int y=y1;y<y2;y++){
    data(c>>8);
    data(c);  
  }
  digitalWrite(_cspin,HIGH);  
}

int backpack::touchxraw(){ 
  long n=0;
  int k;
  digitalWrite(_touchcspin,LOW);  
  SPI.transfer(0x91);
  n=SPI.transfer16(0x91);   //ignore first
  for(k=0;k<TOUCH_OVERSAMPLE-1;k++){
    n=n+SPI.transfer16(0x91);
  }
  n=n+SPI.transfer16(0x90);   //and one more
  digitalWrite(_touchcspin,HIGH);  
  return (n/TOUCH_OVERSAMPLE)>>4;      //only 12 bits resolution
}

int backpack::touchyraw(){
  long n=0;
  int k;
  digitalWrite(_touchcspin,LOW);  
  SPI.transfer(0xD1);
  n=SPI.transfer16(0xD1);   //ignore first
  for(k=0;k<TOUCH_OVERSAMPLE-1;k++){
    n=n+SPI.transfer16(0xD1);
  }
  n=n+SPI.transfer16(0xD0);   //and one more
  digitalWrite(_touchcspin,HIGH);  
  return (n/TOUCH_OVERSAMPLE)>>4;      //only 12 bits resolution
}

int backpack::touchzraw(){    //needed to validate press
  int n;
  digitalWrite(_touchcspin,LOW);  
  SPI.transfer(0xB1);
  n=SPI.transfer16(0xB1);   //ignore first
  n=SPI.transfer16(0xB1);
  n=SPI.transfer16(0xB0);
  digitalWrite(_touchcspin,HIGH);  
  return n>>4;      //only 12 bits resolution
}

int backpack::touchx(){
  int n,p,z;
  z=touchzraw();
  if(z<Z_TOUCH_THRESHOLD){return -1;} //no touch
  switch(_rotation){
    case 1:
      n=touchyraw();
      p=map(n,TOUCH_Y1,TOUCH_Y0,0,width()-1);
      if(p<0){return -1;}
      if(p>width()-1){return -1;}
      return p;
      break;
    case 2:
      n=touchxraw();
      p=map(n,TOUCH_X1,TOUCH_X0,0,width()-1);
      if(p<0){return -1;}
      if(p>width()-1){return -1;}
      return p;
      break;
    case 3:
      n=touchyraw();
      p=map(n,TOUCH_Y0,TOUCH_Y1,0,width()-1);
      if(p<0){return -1;}
      if(p>width()-1){return -1;}
      return p;
      break;
    case 4:
      n=touchxraw();
      p=map(n,TOUCH_X0,TOUCH_X1,0,width()-1);
      if(p<0){return -1;}
      if(p>width()-1){return -1;}
      return p;
      break;
    default:
      return -1;
  }  
}

int backpack::touchy(){
  int n,p,z;
  z=touchzraw();
  if(z<Z_TOUCH_THRESHOLD){return -1;} //no touch
  switch(_rotation){
    case 1:
      n=touchxraw();
      p=map(n,TOUCH_X1,TOUCH_X0,0,height()-1);
      if(p<0){return -1;}
      if(p>height()-1){return -1;}
      return p;
      break;
    case 2:
      n=touchyraw();
      p=map(n,TOUCH_Y0,TOUCH_Y1,0,height()-1);
      if(p<0){return -1;}
      if(p>height()-1){return -1;}
      return p;
      break;
    case 3:
      n=touchxraw();
      p=map(n,TOUCH_X0,TOUCH_X1,0,height()-1);
      if(p<0){return -1;}
      if(p>height()-1){return -1;}
      return p;
      break;
    case 4:
      n=touchyraw();
      p=map(n,TOUCH_Y1,TOUCH_Y0,0,height()-1);
      if(p<0){return -1;}
      if(p>height()-1){return -1;}
      return p;
      break;
    default:
      return -1;
  }  
}

void backpack::imaget(int x,int y,const byte img[],unsigned int t){        //draws with colour t transparent, optimized to reduce areasets- heaps faster, and handles clipping at RHS better
  int w,h,u,v;
  byte aset=1;                   //do we need to set new area?
  unsigned int c,p;                 //colour, pointer to data
  w=pgm_read_byte(&img[0]);         //assume no image is bigger than 255 pixels
  h=pgm_read_byte(&img[1]);         //assume no image is bigger than 255 pixels
  p=0;
  digitalWrite(_cspin,LOW); 
  for(int v=0;v<h;v++){
    aset=1;                         //assume set area at start of row
    for(int u=0;u<w;u++){
      p=p+2;
      c=pgm_read_byte(&img[p])<<8;
      c=c|pgm_read_byte(&img[p+1]);
      if(c!=t){
        if(aset){areaset(x+u,y+v,x+w-1,y+v);digitalWrite(_cspin,LOW);aset=0;}    //areaset from current point to end of row
        data(c>>8);
        data(c);
      }else{
        aset=1;
      }
    }
  }
  digitalWrite(_cspin,HIGH); 
}

byte backpack::imagewidth(const byte img[]){     //get image width
  return pgm_read_byte(&img[0]);
}

byte backpack::imageheight(const byte img[]){     //get image height
  return pgm_read_byte(&img[1]);
}
