'Alternate firmware for the Silicon Chip DAB+ radio (version 3.0)
'(Original articles published in Jan-March 2019 editions of Silicon Chip Magazine)
'
'It incorporates some elements of the original SC firmware, with thanks and gratitude to
'the original authors Duraid Madina and Nicholas Vinen.
'
'Copyright (C) 2020 Stefan Keller-Tuberg (skt at keller-tuberg.homeip.net)
'The original firmware published on the SC web site does not contain a copyright notice.
'It is presumed to be (C) of the original authors Duraid Madina and Nicholas Vinen under
'the GPL license version 3
'
'Google 'Silicon Chip DAB+ radio firmware' for a copy of the original firmware
'
'This alternate firmware for the Silicon Chip DAB+ radio is free software: you can
'redistribute it and/or modify it under the terms of the GNU General Public License
'as published by the Free Software Foundation, either version 3 of the License, or
'(at your option) any later version.
'
'The alternate firmware for the Silicon Chip DAB+ radio is distributed in the hope
'that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty
'of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public
'License for more details.
'
'You should have received a copy of the GNU General Public License along with
'the this file.  If not, see <http://www.gnu.org/licenses/>.

'-----------------------------------------------------
'This file is too large to load into the limited flash on the micromite plus board.
'You will need to CRUNCH it to reduce its size.
'
'If you normally use MMBASIC on a Windows computer for crunching and transferring
'programmes to your micromite, you will discover that the MMBASIC CRUNCH function
'is less than awesome, and that this alternative radio firmware IS TOO LARGE to
'transfer to the micromite EVEN WHEN MMBASIC CRUNCHES IT.
'
'I have provided a bash/linux/OSX script for crunching MMBASIC programmes. My script
'will crunch the output to about 15% smaller than MMBASIC's internal crunch function
'will achieve. I have also supplied a pre-crunched version in the zip archive. You
'can load the pre-CRUNCHED version into MMBASIC and transfer to the micromite, even
'if MMBASIC can't crunch this full-size file.
'-----------------------------------------------------
'option lcdpanel SSD1963_5, landscape, 48
'gui test lcdpanel
'option touch 1, 40, 39
'gui calibrate
'gui test touch
'option sdcard 47 (for LCD SDMMC slot, or 52 for micromite SDMMC slot)
'option rtc 67, 66 (optional, if you have fitted a real time clock module)
'option keyboard us
'option display 65, 130 (recommended but adjust to suit)
'option baudrate 230400 (recommended but adjust to suit)
'option controls 110 (a different number may be required if you change MAX_RADIO)

option explicit 'help avoid bugs of auto-defined / misspelled variables
option base 0 'arrays start from 0
option default none 'don't assume a variable type
option autorun on 'autorun this upon powerup rather than going to prompt
'-----------------------------------------------------
'READ THIS SECTION FIRST
'Getting this programme running on your hardware:
'
'This programme doesn't contain code to load the 32 Mbit flash on the radio board.
'Use the original SC programme to perform this task. Once the 32 Mbit flash is
'programmed and working with the SC programme, this code will work also.
'
'The SC code sets the SPI bus to run at 2 MHz. This programme sets it to 10 MHz. My HW
'wouldn't work at either rate with the SC software until I added pull up resistors onto
'the radio card SPI bus. If you experience issues with this software not detecting the
'radio chip or TOS framer chip, try reducing the SPI rate and consider adding pullups.
'
'I've set the CPU speed to 120 MHz. This works fine in my setup. If you experience
'issues, it may be that you need to reduce CPU clock rate. The SC software configures
'the CPU to 80 MHz. This code will work OK at 80 MHz - but some things like the sorting
'of the presets file will take noticeably longer.
'
'When running MMBASIC version 5.05.03, both of my micromites would throw bizarre errors
'from time to time - every hour or so. This didn't happen with either version 5.05.01
'or 5.05.02. The errors looked real (eg 'bus error' or a 'font error etc) except there
'was no error! On a hunch from experience with other PIC32 devices, I added a 47uF
'tantalum cap in parallel with the 10uF cap adjacent to the PIC. This improved the
'stability but has not fixed it completely. I have observed severe spikes using a CRO
'in the vicinity of the LCD panel edge connector and around the CPU and have added
'additional tantalum capacitors on the 3V3 line near those locations. Still, I
'experienced occasional infrequent bizarre micromite crashes and I added four 100uF
'tantalum capacitors on the underside of the micromite board immediately under the
'PIC and across the 3V3 and ground connections of four of the 0.1uF bypass caps -
'one tantalum cap positioned on each of the four sides of the PIC. This again reduced
'crashes and might also have improved radio chip crashes (fatal errors). I have a
'feeling that if you reduce the PIC clock frequency, you can get away with less
'capacitance. This might be a root cause for PIC issues that many SC readers seem to
'have experienced over the years.
'
'Despite the extra capacitance, MMBASIC 5.05.03 has caused other non-sensical errors
'at various times - to the point I have given up on 5.05.03. I would recommend not
'using this version of MMBASIC as it clearly gives more problems than the other
'versions. I cannot explain this definitively as a software problem - it may be
'voltage / noise related and therefore a micromite plus PCB / design issue.
'
'This programme assumes there's an SDMMC flash card plugged permanently into the
'Micromite or LCD card slot. Any old small SDMMC card will do. A couple of Mbyte will
'be more than enough. The flash card is used to keep a running list of station presets
'(in a preset file) and some settings from the last session (eg last radio station,
'analogue/digital output), and if there's a radio chip comms issue, a log will be
'appended to an error file recording some critical status bytes from the Si4689.
'-----------------------------------------------------
'The user interface and some of the functions:
'
'The user interface has two modes configured using the 'Show Fav Buttons' switch on
'the setup screen. If you prefer an uncluttered and simple look and feel, then
'switch the Fav buttons off (the volume control and band indication will also be
'suppressed from the status screen when the favourite buttons are turned off).
'In either mode, touch the status screen to reveal the main menu.
'
'On many screens, a close box is provided in the upper left corner to return to the
'previous screen. Also, a timer will automatically close most menu screens and return
'to the previous screen if there is no input for a certain amount of time.
'
'When running this programme for the first time, you may want to run the setup option
'and select 'Station Scan'. This will scan the Australian AM, FM and then DAB bands
'to look for 'stronger' signals, populating the Preset0.csv file. Depending on the
'vagueries of signal levels and noise, some non-stations may be recorded, and some real
'stations may be missed. Refer to discussion below on AntCap settings for one reason for
'these 'mistakes' when scanning. You can edit these 'mistakes' later.
'
'The configuration and presets are saved as CSV files on the SDMMC flash card so they
'can be edited offline in a spreadsheet, or copied to a different micromite board. The
'preset settings can also be edited on the radio itself using the 'Edit Presets' menu
'option. In the presets editor, you will find the option to 'hide' or 'show' each
'preset, and another button to identify the preset as a 'favourite'.
'
'Hiding/Showing relates to whether the station will be visible in the 'Tune to Preset'
'menu function. If you choose to hide a particular station, its information will still
'remain in the presets list, and (for example) its station name will be displayed if
'you tune to its frequency, and its AntCap setting will be applied. You can always
'unhide an entry later.
'
'Making a station a 'favourite' relates to the favourite buttons on the status screen
'(when the 'Show Fav Buttons' mode is 'yes'). You can also choose a favourite with a
'single button IR remote control press regardless of the display mode.
'
'If a station with an AM frequency is selected in the presets menu, you will see a
'spinbox control setting the analogue bandwidth. The digital radio supports up to
'10kHz AM bandwidth, which is much higher than most AM radios. Although mono, it the
'sound is more comparable with FM than a traditional AM radio. The original SC
'software didn't set the analogue bandwidth, and the radio chip defaults to an
'analogue bandwidth of 3.5 kHz - which will produce the typical muffled AM sound
'most of us are familar with for most AM radios. When setting a higher bandwidth
'such as 10kHz, the antenna placement becomes critical. If you cannot elimiante noise,
'consider building yourself a more sophisticated AM antenna such as a tuned loop like
'that described in SC Oct 2007 or SC Jan 2009 or countless examples on the internet.
'Separating your AM antenna from the radio by a couple of metres using a highly
'twisted (cat-5-like) pair - the micromite and LCD spew heaps of noise in the AM
'band and moving the antenna away really helps.
'
'If a station with either an FM or DAB+ frequency is selected in the presets menu,
'the spinbox will change to an antcap control. An antcap value of 0 is 'default' and
'means that the radio will take a guess - its not a great guess, but if your signal
'is decent, it will be fine. Otherwise, you can specify an optimised value as
'described elsewhere.
'
'When editing presets, you can clear (delete) an entry. The resulting blank/hole in
'the list will be removed when you exit the editor.
'
'From the setup menu, you may also want to set the backlight level, and choose the
'output mode. There are three output modes: digital, RCA and speakers. The headphones
'can be used with any of these output mode settings.
'
'In digital and RCA output mode, the volume control will be hidden unless the
'headphones are plugged in. Volume is set at maximum level in each of these modes
'unless the headphones are plugged in. For the speaker output mode, the volume
'control will always be visible. (The RCA outputs will also work in RCA mode but
'levels cannot be adjusted).

'After making a configuration change and after a short delay, the settings
'are saved to a file called Config0.csv on the SDMMC card, along with the station
'to which you are currently tuned. When you change stations, the Config0.csv file is
'rewritten to remember the last station.
'
'The IR remote control codes in this programme are not the same as in the orignal SC
'programme. I had a different remote control unit on hand and could not replicate the
'codes supported by the altronics and jaycar devices. This is not such a big deal as the
'codes are trivial to change. Edit the file called InfraRed.csv which needs to be
'located in the root directory of the SDMMC card, and configure the codes for your
'remote control. Note: The file format is very strict. Don't change row orders or
'any of the comments.
'
'The CSV files mentioned above can be edited in a spreadsheet programme such as
'Microsoft Excel. They can also be edited in a text editor. The order of the rows of
'data are important and should not be changed, and so are their relative position. You
'can insert comments above the commencement of real data, but don't change the ordering
'of the data itself!
'-----------------------------------------------------
'Loading a new version of this software onto the radio:
'
'There's a built in semi-automated way of loading a new version of this BASIC programme.
'
'Upon power-on, this programme will look for a file called 'Radio0.bas' on the SDMMC
'flash card. If a file with that name is found, it will be renamed to Radio1.bas (and
'all earlier files called RadioN.bas will be renamed one number up). The new basic code
'will be loaded into the micromite and run without further intervention. In this way,
'new BASIC software can be loaded via SDMMC without a serial interface being attached.
'
'So once this programme is copied onto the micromite for the first time in the
'traditional way, you can edit the code on your computer and transfer the new code to
'the radio by copying the new version to a file called 'Radio0.bas' on the SDMMC card.
'Then power cycle the radio and it will load the new version of code as it boots.
'-----------------------------------------------------
'Bugs:
'
'I have tested this software on my own DAB+ radio card and a friend (Ingo Evers) has
'tested the software on his DAB+ radio. Both of our radios include all optional chips.
'The software exercises almost all the hardware capabilities of the DAB+ radio board,
'but does not exercise the volume functions of the PAM8804 chip. Analogue volume is
'managed exclusively by the volume function in the radio chip, for this SW.
'
'The programme works OK on my hardware and Ingo's hardware. From time to time, my radio
'crashes (for example, when there's a big RF surge due to the gas stove ignitor). The
'issues seem less common when the incoming radio signal is strong and has improved a
'lot since adding extra power supply bypass capacitance. I am reasonably confident
'the noise is entering the radio via the antenna, not the power supply, especially
'because of the kind of error that the radio chip returns in its status messages.
'When this programme recognises such a condition, it will reboot the radio. This seems
'to be a bug with the radio, not with this BASIC programme. You will find a time series
'record of the crashes / reboots in the ErrorN.csv file.
'
'Ingo and I have also noticed that if you fiddle with the AM or VHF antenna connection
'when tuned to that band, (such as unplugging or replugging the antenna in), the radio
'chip will crash instantly. The errors indicated by the radio chip suggest its internal
'DSP has locked up. This behaviour seems to be a bug with the radio chip software
'rather than this BASIC programme so if new versions of the Si4689 software are ever
'released, it might be interesting to try them out to see if the radio stability
'improves.
'
'Ingo has also noticed that his radio is very prone to throwing SPI bus errors when
'changing from one band to another. My radio behaves rock solidly in exactly the same
'circumstances. I am guessing that because my radio has had a tremendous amount of
'additional power supply capacitance added to both the micromite plus board and the
'radio board, and this is essentially the main difference between our implementations,
'that this may be the reason for the behaviour. Keeping in mind that this software
'is essentially stable and working well on my radio in the same circumstances, if you
'notice strange behaviour on your radio, there is a chance that you will see
'improvement by adding power supply capacitance liberally, as I did.
'
'Another difference between our radios is that for *AM*, Ingo notices a significant
'increase in noise if the *FM/DAB* (VHF) antenna is removed and left floating. The
'additional AM noise disappears when the FM/DAB roof-top antenna is reconnected. This
'does not happen with my radio which is quiet provided the AM antenna loop is rotated
'to minimise noise. For the record, both our radios have fully isolated the antenna,
'RCA audio, SPDIF coax and power supply from the chassis / mains earth. That is, both
'our radios are floating. Ingo found that this additional AM noise can be completely
'quenched by paralleling a 0.1uF ceramic capacitor and a 10uF bipolar electrolytic
'between the shield of his FM/DAB input socket, and the mains earth of the metal case
'in which he has built his radio. He speculates that one of his other in-home VHF
'appliances also grounds the shield, and that when the antenna input is removed from
'the DAB radio, the earth is removed as well. That is, while his DAB radio
'implementation is fully isolated from mains earth in and of itself, when an earthed
'FM/DAB antenna is connected to the radio, the radio is earthed via the antenna, and
'by virtue of its earthing, plays AM quietly without noise. While this sounds vaguely
'plausible, I am uncertain because I don't see the same behaviour with my radio.
'
'Ingo and I have extensively tested this software on our own radios. To the best of
'our ability, we believe the BASIC programme is essentially free of critical bugs. Any
'querky behaviour can be explained as a 'feature', for example the delays you might
'notice when changing bands or output mode or plugging in and removing headphones are a
'consequence of the software - both the radio software and this BASIC programme.
'There is of course a chance there are other bugs that we've not found. I apologise
'for those too. Like you, I am human and the only thing I can guarantee you is that my
'code contains bugs I have not yet found. ;-) If in doubt that this code is working for
'you, substitute the Silicon Chip code and if that works, its clear you've found a bug
'in mine. If you find how to fix a problem, I'd love to hear about it.
'
'A comment about SNR and signal strength. Basically, I don't trust the values that the
'radio chip reports. They're OK in a relative sense, but if you have a decent signal
'strength meter, you will see what I mean when you compare with the values the radio
'reports.
'-----------------------------------------------------
'Frustrations writing this version of the software:
'
'It seems that those that start working on Si4689 software are doomed to frustration,
'eventually. The original authors comments in the magazine article hinted at this. You
'can take it from this comment that my own experience aligns with theirs. But perhaps
'for different reasons. All things considered, the Si4689 and its documentation
'aren't so bad! (Although I have never been able to find decent electrical specs...)
'
'I was motivated to write this programme independently from the SC code in order to test
'the theory that the Si4689 doesn't support digital output for the DAB+ band, as
'the original authors had concluded. The Si4689 programming guide doesn't mention this,
'so I was skeptical, and hopeful for a fix. I started trying to modify the SC code, but
'as I was trying to closely follow the flow charts in the prog guide, it was easier to
'write the state machine you will find below in 'CheckSpi' as completely new code.
'
'To my joy, I discovered that the DAB mode will indeed output digital data, and this
'makes me very happy because my 2013 SC 'crystal dac' sounds really awesome. I am unsure
'what the issue with the original SC BASIC programme is, and I have not been able to fix
'it in the original firmware. I think the root cause may be with SPI message sequencing.
'Turn on D=3 debugging in this code and trace the Si4689 message sequence. It is
'somewhat different to the sequence followed in the original SC code.
'
'My frustrations weren't with the DAB digital output, but included the AntCap
'(antenna capacitance) values which I comment on below.
'-----------------------------------------------------
'Improving received signal strength and SNR/CNR:
'
'Whether you need to use this feature or not will depend on your antenna and where you
'live. If your signal strength is high (eg you live in the central core of a major
'capital city), then your signal strength will probably be great regardless of whether
'you choose to connect a rooftop antenna or a telescopic antenna. I live in southern
'Canberra where the surrounding hills do an excellent job of shielding FM and DAB, and
'the telescopic antenna picks up literally nothing anywhere in the house, and the
'rooftop antenna cannot reliably pick up any of the FM or DAB frequencies transmitted
'from Black Mountain tower. We have an FM/DAB retransmitter down south, but not all FM
'stations and only one of the two DAB frequencies are carried. So I had a need to
'experiment with and develop this AntCap feature.
'
'I found that I could get >3 dBuV of additional signal strength and better SNR in the FM
'and DAB bands by applying optimised AntCap values. This programme includes an AntCap
'optimising algorithm based on that described in the app note. Some further explanation
'is required.
'
'After many 'goes', I found that the 'optimal' AntCap settings critically vary with
'small changes in the antenna setup. This ultimately was one of my sources of
'frustration, because it took me quite a while to work out that the variation was due to
'the changes in antenna cable lengths. The Si4689's 'optimal' settings change as the
'antenna feed characteristics change (ie the capacitance and inductance of the
'combination of the antenna and the feed cable). This means that you need your radio in
'its final (normal usage) location connected to the home's VHF antenna feed to properly
'set the AntCap configuration values.
'
'This suggests the hardware design would benefit from a wideband unity gain amp between
'the external antenna input and the radio chip, so that the radio chip always sees the
'same characteristics and the AntCap values can be set once-and-for-all for this design,
'i.e. so every implementation will behave the same as every other implementation. This
'would make it easier to terminate the external antenna into a nominal 75 Ohm resistive
'load to eliminate reflections back into the home wiring (which cause nulls through the
'spectrum). Anyway, too late for that for this particular design.... Maybe Mk II?
'
'It is overwhelmingly probable that YOUR implementation will work best with different
'AntCap values than MINE (and probably different values than by default).
'
'I suggest you experiment with this feature to see if you can get a benefit. If not,
'then don't worry about it or use it. If, you get a signal boost, then some one-off home
'work will be required. You will need to run the AntCap algorithm for each frequency you
'wish to keep in the presets file. At the conclusion of the algorithm, the optimal
'AntCap value that was measured will be written to the presets file for that frequency.
'As it takes several minutes per frequency, you'll need to be persistent to measure
'optimal AntCap values for all stations. There is however a shortcut you can take.
'
'Start by measuring an AntCap value for an FM station near the bottom of the FM band.
'Then work your way up the FM band in a few steps. You can edit the presets file offline
'in (eg microsoft excel) and interpolate the optimal settings for other stations between
'the ones you actually measured. This way, you'll get close to optimal without having to
'do every station. (You can just run the algorithm for every station too.)
'
'The Si4689 kind-of does a kludge like this too. The app note says to measure AntCap
'values for a number of frequencies across the band, and then plot the points. When you
'plot the points, you will find the line is kind-of a straight line (not quite!) but if
'you assume a straight line and calculate the slope and Y intercept, you can then
'configure the Si6849 with a number representing 1000x the slope, and another number
'representing the Y intercept, and the chip will guess the value of AntCap to use based
'on the straight line you told it about. I chose to implement explicit values for each
'station - no approximations are required - you get the precise optimal AntCap setting
'for each frequency.
'
'The second frustration with AntCap settings is that the radio chip seems to read-back
'different AntCap values than you programme, even though it is clear that your
'programmed values are improving the performance. Sigh. You won't notice the differences
'unless you enable D=2 debug mode and watch the status reports come back at you
'reporting the AntCap value the radio says it has applied. I am not sure if this is
'an Si4689 bug (reporting different values than has been programmed) or a 'feature'.
'
'Yet a third frustration with the radio is that the signal strength readings that are
'reported seem to 'saturate' or even 'reduce' at a particular level, and then the SNR
'or CNR readings start to misbehave. This may be because there are unequal level
'signals in the spectrum, and the strongest are saturating or confusing the front
'end of the radio? I noticed this when adding an antenna amplifier in-line with
'the rooftop antenna feed just prior to the radio. While expecting to see a decent
'signal boost, the radio reported a signal *drop*, but a substantially better CNR.
'I also noticed that some stations that were previously good, could not be received
'any more. It seems that the amplifier was distorting them, or the radio was not
'coping with dramatically unequal signal levels for different stations.
'
'My AntCap experiments seem to suggest that the values for L1 and L2 and L3 and
'associated capacitors in the SC design may not be optimal for this PCB layout. I say
'this because the optimal AntCap values hit their absolute minimum configurable value
'mid-band rather than at the very top of the FM and DAB bands. According to the
'app note, components should be chosen so that the AntCap function hits a minimum at
'the top of the FM (and DAB) bands. I have not tried to experiment with alternate
'component values because of the difficulty working in this section of the board.
'However it seems likely that gains might be found by experimenting with these component
'values. Maybe not worthwhile unless there's a wideband buffer amp designed into the
'circuit first.
'
'Additional suggestion that L1, L2 and L3 are suboptimal comes from my experience with
'FM RDS and AntCap. I have mentioned I live in a marginal reception area. Also, all
'local FM broadcasts are actually rebroadcast from a local repeater - we cannot receive
'the original broadcast from Black Mountain Tower due to the mountainous terrain.
'It seems that for most stations, the rebroadcast FM generally excludes RDS, so there
'are few signals with which I can test the RDS. We have one signal for which I am
'certain there is an RDS subcarrier - 107.1 MHz - the very top of the FM band.
'Because the optimised AntCap setting reaches absolute minimum well below 107MHz,
'the signal strength at 107 is well down, lets say >10dB down but potentially much
'more, than the 'optimal' receive values lower down the spectrum. Whilst I previously
'mentioned that neither the SC software nor this software receive RDS reliably, I will
'now add that when running this software, the radio reports around an additional 3dBuV
'compared with the SC software, and the number of RDS interrupts at 107.1MHz is
'noticeably greater than the number of interrupts received by the SC software. This
'suggests to me further corroboration that the analogue front end of this design
'would benefit from optimisation, including a unity-gain wideband amp to decouple
'the external antenna characteristics from the internal design, and also benefit from
'carefully chosen optimal component values for L1, L2 and L3 different than the current
'component values.

'The following constants are up for customisation... these values are the values that
'worked best for my radio / antenna setup. I would recommend tuning to a frequency
'in the middle of the FM band and running the AntCap optimisation routine with the
'switch set to 0, then try again at the same frequency with the switch set to 1. Then
'set the constant to the value that gives the greatest optimised signal strength.
'You will then of course need to run the AntCap optimisation algorithm for FM stations
'at either end of the band, and if the range of optimised settings will work for you,
'run the AntCap algorithm for a number of stations all across the FM band to optimise.
'You'll find fewer DAB frequencies to experiment with. Choose the DAB switch setting
'that gives the best overall performance on all of the DAB frequencies that are
'accessible at your location.

const FmSwitch=0, DabSwitch=1
'-----------------------------------------------------
'Some example AntCap measurements:
'
'The Silicon Labs reference design AntCap defaults are as follows: (from AN851 page 24)
'FM: Slope=0xEDB5, Intercept=0x01E3, Switch=Open
'    (Note: App note says closed for FM, but the results are better with it open!)
'DAB: Slope=0xF8A9, Intercept=0x01C6, Switch=Closed
'Note that the slopes are signed 16 bit numbers. You need to use twos complement
'decoding to work out what the absolute value of the negative number is.
'
'The chip applies an AntCap value calculated using the following equation
'  AntCap = ((slope/1000) * (freq expressed as MHz)) + intercept
'  FM and DAB: Capacitance = (AntCap-1) * 250 fF (fF=femto farad - 1000th of a pF)
'  AM:         Capacitance = (AntCap-1) * 142 fF [from AN649 p269]
'
'In one of my test configurations, the AntCap algorithm returned the following
''optimal' results:
'87.6MHz: 20.75pF
'88.7MHz: 17.5pF
'90.3MHz: 11pF
'91.9MHz: 6.75pF
'94.3MHz: 1.5pF
'For the results above, the line of best fit is of the form Cap = (-11.6 * Freq) + 1096
'which equates to a slope setting of 0xD2B0 and a Y intercept setting of 0x0448
'
'Changing only the length of the coax feed between the folded dipole and the radio, the
'measurements change as follows:
'88.7MHz: 26.75pF
'89.1MHz: 25.75pF
'93.7MHz: 16.0pF
'96.0MHz: 12.0pF
'99.1MHz: 8.0pF
'100.7MHz: 6.0pF
'103.9MHz: 4.5pF
'And for these points, the line of best fit is of the form Cap = (-6.6 * Freq) + 684
'This equates to a slope setting of 0xE638 and a Y intercept of 0x02AC
'
'(Of course, every time I changed the antenna arrangement, I got totally different
'numbers. I could repeatedly measure values for the same arrangement and get the same
'results, but changing the arrangement immediately led to different 'optimal' results,
'which themselves were repeatable. So only test when you finish your radio and it has
'its long term antenna connected as you want to leave it)
'-----------------------------------------------------
'Another frustration with the Si4689 included its digital output mode in all three
'bands: AM, FM and DAB+ alike. I found the Si4689 has issues with outputting unwanted
'analogue noise. The radio chip outputs rubbish too often and easily, and I think the
'developers might have been able to do a better job avoiding this. There are some
'software work arounds in this code that try and manage it. I have used the WM8804 as
'a kind of brute force firewall, forcing that device into reset mode at the times I
'found the radio chip most likely to output uncontrolled non-audio. Essentially, the
'WM8804 acts as a hardware mute for the digital output. Its ugly, but it improves
'things.
'-----------------------------------------------------
'Physical construction of my particular radio:
'
'I have built my system into a 3RU enclosure to use as a 'hifi' tuner for my sound
'system. My intent was always to use only the digital interface for all bands. This is
'how I am presently using the radio - no analogue connections whatsoever.

'I use a 75ohm RG-6 coax feed from the home television distribution amplifier and a
'rooftop DAB+ folded dipole antenna. I made my own AM loop with about 10 turns of
'0.4mm hookup wire around a rigid non-conductive non-magnetic circular form of
'around 40cm diameter. I included approximately 3m of lead-wire between the radio and
'the circular loop, and I put the ends of the pair of antenna wires into a hand drill
'and added a high twist similar to cat5 cable twist in the wire between the antenna
'loop and the radio. The twist is really important for eliminating noise
'pickup and reducing losses between the radio loop and the terminals on the DAB+
'radio board. A loop antenna is somewhat directional and is extremely good at
'avoiding noise from nearby appliances, including noise from the radio itself.
'The AM signal levels I receive this way far exceed the FM and DAB signal levels
'from the rooftop antenna, easily achieving more than 75dBuV with the 40cm loop.
'
'The SC design sandwitches the LCD, micromite and the radio board onto one assembly.
'I found noise from the LCD display breaks into weaker AM and FM stations. I used
'several strategies to reduce the noise, including extra power supply filtering, clamp
'on ferrites, and I separated the DAB radio board away from the micromite and LCD with a
'short length of 40 conductor ribbon and used a clamp-on ferrite on this ribbon.
'
'Other than reduced noise, separating the radio from the micromite with a short ribbon
'made no difference to the behaviour of the radio. Those problems I've described I
'experienced with my firmware were also present with the Silicon Chip firmware when
'the radio was sandwitched together as per the original published design.
'-----------------------------------------------------
'Hardware modifications to the original design:
'
'I have made some hardware modifications to my DAB+ card in order to reduce audible
'noise in the analgoue (and / or digital) outputs. Particularly:
'(i)   I have hard wired the -5V inverter so that it is always enabled by cutting the
'      track connecting CON8 pin 35 and REG4 pin 1, and then grounding REG4 pin 1. The
'      article warns of latch up. I have never experienced any issues with latch up with
'      the DAB radio and this modification.
'
'(ii)  I have changed the value of the the electrolytic capacitors connected to FB1 and
'      FB2 to 4.7uF to reduce the 'cracks' in the analogue outputs that occur when
'      changing bands (reloading firmware onto the Si4689). I've extended the length of
'      the software mute also. I believe these cracks are caused because the radio chip
'      outputs a non-zero voltage on its analogue audio outputs while firmware is
'      loading/booting. This voltage charges the capacitors (which were 100uF in the
'      original circuit). The discharge path for those capacitors has too long a time
'      constant to dissipate the unwanted voltage before the 4052 analogue switch
'      enables audio. Hence CRACK.
'
'(iii) I have added 150uF tantalum capacitors at the 40 pin edge connector between pins
'      1+3 and 1+5. (i.e. across +5 and ground, and +3.3 and ground). I added two of
'      these to the DAB radio board, and I added another two to the micromite - that is,
'      I added four 150uF tantalum capacitors in total - two on either side of the 40
'      wire ribbon. You may not need these in your implementation.
'
'(iv)  I constructed a DC input voltage monitor to look for loss of voltage that
'      indicates power has been turned off. This drives an interrupt line to the
'      micromite which uses it to mute the outputs. I found that the Si4689 is really
'      prone to locking itself up and outputting random noise. This seems to happen
'      about one in 30 power-downs?? These are flagged by the chip itself as 'fatal
'      errors' and according to the AN649 ap note, this is DSP related. These fatal
'      errors don't appear to be caused by this programme, but by the Si4689 firmware
'      itself, triggered by susceptibility to strong RF noise or PSU noise??? If you
'      find a fix, let me know! Note: The code that supervises the voltage monitor is
'      enabled by setting the constant Pv to non-zero (below). If you don't implement
'      this hardware, set Pv to 0.
'
'(v)   I added 200 ohm at 10MHz clamp on ferrite sleeves on the 5V micromite supply and
'      on the short ribbon cable I used to connect the micromite to the DAB+ radio board
'
'The following changes are not radio noise related:
'(vi)  I added 3k3 pull up resistors between the 3.3V pin on CON3 and the DAB+ radio
'      side of the three 47R resistors. Without these, the SPI interface would not
'      communicate with the Si4689 nor the WM8804 reliably. This happened both when the
'      three cards were sandwitched as per original design, and with the short ribbon
'      cable I used to separate the micromite and the radio board for noise improvement.
'      I tried with two different micromite boards, but only the one DAB radio card. Now
'      the pull-ups are in place, I've not seen a single SPI failure since and the SPI
'      bus is now configured for 10MHz operation, a little faster than originally.
'
'(vii) I added many tantalum capacitors onto the 3V3 power rail on the micromite board
'      as described elsewhere in this long comment.
'
'Changes (i), (ii) and (vi) have been implemented by Ingo Evers and were verified
'effective in addressing similar noise and SPI issues they were experiencing with his
'setup. Ingo has also made modifications to the class AB headphone amplifier design as
'described below. I have not implemented these on my radio as I do not have a headphone
'output built into my unit.
'
'On a single occasion out of perhaps five hundred or more power-on/power-off cycles,
'my output transistors Q1/Q2 in the headphone amp section smoked. Ingo Evers who has
'also constructed the DAB radio has experienced this a few times and we think our
'experiences are related. If the D1 and D2 components you happen to have purchased have
'a HIGHER forward biased voltage drop from pin 1 to pin 2 of the BAV99 device than the
'sum of the Vbe voltage drops of Q1+Q2 or Q3+Q4, then there is a good chance that your
'transistors will smoke in your DAB radio at some stage. The larger the pin1-2 voltage
'drop of D1 or D2 above Q1+Q2 or Q3+Q4, the greater the chance.
'
'An earlier version of this alternate software set the logic pin that controls the -5V
'regulator to 1 rather than 0. This will also increase the chance of smoking one of the
'headphone amp transistors if the suggested hardware mod of cutting the control track
'and grounding pin 1 of the regulator has not been performed. This is corrected in this
'version.
'
'If you require the headphone output, then either (a) carefully select the D1, D2, Q1,
'Q2, Q3, and Q4 components so that the forward biased voltage drop across D1 and D2 does
'not exceed the sum of forward biased Vbe of Q1+Q2 and Q3+Q4 - ie test and choose
'matched diodes and transistors before constructing this part of the circuit, or
'(b) modify the circuit to insert four 2R2 resistors between the emitters of Q1, Q2, Q3
'and Q4 and the points in the original design where the emitters join. Inserting
'resistors will be tricky because of the limited space. You could stand surface mount
'resistors on end and substitute BC549/BC559 transistors for the surface mount devices
'in the original design, or rebuild the headphone amplifier on veroboard and run hookup
'wire from the DAB radio PCB to the veroboard.
'
'Kudos for diagnosing the root cause of class AB headphone amp problem goes to Ingo
'Evers and apologies for the earlier error in this software.
'-----------------------------------------------------
CPU 120 'I was worried about noise, but running CPU quickly makes no apparent difference

const SPIrate = 10000000 'Clock rate (in Hz) for the SPI bus
'-----------------------------------------------------
'The following constant is a debug flag.
'Set it to 0 to turn off debug messages. 1=INFO only, 2=MIDDLE, 3=ALL
const D=0 'Recommend keeping as 0 unless you are specifically debugging/modifying code

'The following enables Stand Alone mode. 0=DAB Radio HW connected, 1=Micromite only
'Use State Alone mode when debugging this software on a micromite without radio HW.
const SA=0 'set to 1 to enable stand alone mode

'The following constant enables a voltage monitor interrupt. Unless you have
'specifically added a hardware mod as described in comments below, set this to 0
const Pv=0

'The following enables pin 35 as output to flash an LED when IR command received
'1=enabled, 0=disabled
const IrLed=0

'The following sounds the buzzer when IR command received
'1=enabled, 0=disabled
const IrBuzz=1

'The following enables a plug-in DS1307 RTC which will ensure file and log times are
'correctly captured in all cases. The RTC is automatically set according to time/date
'obtained via DAB or FM RDS
'1=enabled, 0=disabled
const RtcInstalled=0

'The following is the default frequency chosen when Config.csv has bad value
'Note: will only work as desired if an AM or FM frequency is chosen!
const DefaultChannel=666 'kHz (ABC in Canberra)

'The following define the LCD colour scheme. There are some alternate colour
'suggestions. Play around to suit your own taste!
'After several spirited discussions about colour with my friend Ingo, I think I've
'worked out that the perceived colour will depend strongly on the viewing angle of
'the LCD. This will of course depend on how you have built your radio, and where
'you mount it. Mine is at head-height on a shelf, and we look straight at the screen.
'I believe this causes a different perception of colour than when the LCD is
'at table height and angled.

'const AmColour = rgb(cyan) 'colour representing AM band
'const FmColour = rgb(green) 'colour representing FM band
'const DabColour = rgb(magenta) 'colour representing DAB band
'const NonBandColour = rgb(yellow) 'colour of 'next' 'prev' and 'cancel' options

const AmColour = rgb(0,170,170) 'colour representing AM band
const FmColour = rgb(0,170,50) 'colour representing FM band
const DabColour = rgb(200,170,0) 'colour representing DAB band
const NonBandColour = rgb(red) 'colour of 'next' 'prev' and 'cancel' options

'const ButtonColour = rgb(blue) 'Background colour of GUI buttons
const ButtonColour = rgb(0,0,144) 'Background colour of GUI buttons
const ButtonTextColour = rgb(white) 'Text colour of GUI buttons
const BMuteBackgrounddColour = rgb(128,64,0) 'Special treatment for the mute button

const DigitalColour = rgb(blue) 'colour representing the Digital outputs
const RCAColour = rgb(brown) 'colour representing the RCA connector output
const SpeakerColour = rgb(red) 'colour representing the in-built speakers

const OtherGadgetColour = rgb(brown) 'colour for volume control, and other spin boxes

const ConsoleOutlineColour = rgb(green) 'colour of box around outside of boot console
const ConsoleTextColour = rgb(white) 'plain text on black background colour

const CancelTextColour = rgb(80,80,80)
const CancelBackgroundColour = rgb(48,48,48)
'-----------------------------------------------------
'Set up hardware

'Si4689 is the radio chip
'Flash is the 32Mbit flash associated with the Si4689 radio chip
'WM8804 is the TOSLINK formatter
'MUX is the 4052
'PAM8407 is the on-board amplifier
'REG is the LM2663 Voltage Regulator used for analogue audio output
'POWER_ALARM is an optional modification I incorporated as described in comments below

const P_SI4689_SMODE=44, P_SI4689_RES=74, P_SI4689_CS=80
const P_Flash_SPI_MISO=88, P_Flash_SPI_MOSI=90, P_Flash_SPI_CLK=91, P_Flash_SPI_CS=92
const P_WM8804_CSB=59, P_WM8804_IFM=60, P_WM8804_RST=61, P_WM8804_SPI_MOSI=72
const P_MUX_0=25, P_MUX_1=24, P_LED_YELLOW=38, P_LED_RED=58
const P_PAM8407_ENAB=68, P_PAM8407_UP=67, P_PAM8407_DN=66
const P_REG=21, P_HEADPHONE=14, P_IR_ACK_LED=35
const P_POWER_ALARM=34 'Hardware mod: low for power OK. Open circuit for power drop

'Enumerated states of the analogue audio MUX
const MODE_AM=0, MODE_FM=1, MODE_DAB=2
const AUDIO_DIG=0, AUDIO_RCA=1, AUDIO_SPKR=2

'These constants define the limits of the radio bands, and the channel spacing
'AmMIN could be 520, but rounded to the nearest 9kHz to align with Aus channel spacing
'Note for FM and DAB. The minstren constant (minimum signal strength) is used by the
'scan algorithm. Because the scan is undertaken BEFORE the AntCap value has been
'optimised, it is important to select a relatively low signal strength - this has the
'unintended consequence that more dodgy / invalid frequencies will be flagged as
'valid stations.
'
'Note: The minimum to maximum Australian DAB frequencies are taken from ACMA document
'https://www.acma.gov.au/sites/default/files/2019-12/General%20Information_0.pdf
const AmMIN=522, AmMAX=1710, AmDELTA=9, AmSNR=10, AmMinStren=45, FmMIN=76000
const FmMAX=108000, AFmMIN=87500, FmDELTA=100, FmSNR=6, FmMinStren=25, DabMIN=174928
const DabMAX=239200, ADabMIN=195936, ADabMAX=208064, DabCNR=4, DabMinStren=40
const BWmin=1500, BWmax=10000, BWdef=5000

const Runtuned=0, Rtuned=1, Rselected=2, Rplaying=3

'This is a delay used to wait for a newly tuned channel to settle before reading stats
const SnrSettle=40 'ms (default is 40ms)

'The default silicon labs AntCap numbers, the same as those used by the SC programme
const FmSlope=&HEDB5, FmIntercept=&H01E3
const DabSlope=&HF8A9, DabIntercept=&H01C6

'These constants relate to the presets that are stored and displayed
const MAX_DAB_FREQS=41, MAX_DAB_SERVICES=32
const MAX_PRESETS=90 'This is the max number of radio stations in the presets list
const MAX_FAVS=6 'More Favourites can be remembered than can be assigned buttons

'You can define the number of preset stations that will appear on each preset screen.
'The default I've gone for is 30 (three columns of 10) but practically, you can fit
'up to 39 (three columns of 13) but you will need sharp fingers to press the buttons!
'If you have fatter fingers than me, you can reduce the number too, but it must
'always be multiple of 3! (Because there are 3 columns of radio buttons on the LCD)
'NOTE: you may need to twiddle with the 'option controls' setting if you increase
'the number of radio buttons. You'll know if you need to do this because you'll get
'run time errors when the radio buttons are created with too few option controls.
const MAX_RADIO=30
const NUM_PRESET_ROWS = MAX_RADIO/3
const RADIO_RADIUS=16 'this is actually half the size of the circle used for button

'The following MUST correspond with MMBASIC 'option controls' setting. The constant
'111 is one more than the highest index specified by 'option controls nnn'
'The preset selection radio buttons occupy the last MAX_RADIO coltrol values
const Rad1=111 - MAX_RADIO

const MUTE_RELEASE=150 '150x5ms ticks

'The following enumerations represent the LCD state machine
'These enumerations are the different screens/windows
const Xconsole=1, Xstatus=2, Xmain=3, Xedit=4, Xselect=5, Xsetup=6, Xscan=7, XIRTest=8
const Xcap=9, Xdisp=10, XstatusE=11, XmainE=12, XscanE=13, XcapE=14, XdispE=15

'This holds the max number of bytes that can be received from radio chip via SPI message
const READ_REPLY_BUFSIZE = 1028

'These constants define a moving average range for stats polled from the radio chip
'The stats move about rapidly, so a moving average is a good way of making better sense
'multiplier of 0.975, step change hits 50% in 28 samples
'multiplier of 0.95, step change hits 50% in 14 samples
'multiplier of 0.9, step change hits 50% in 7 samples
'multiplier of 0.85, step change hits 50% in 4 samples
'multiplier of 0.8, step change hits 50% in 3 samples
const MA_MULT! = 0.99
const MA_INV_MULT! = 1 - MA_MULT!

dim string S1, S2, S3
'----------------------------------------------------------
'Set initial output values for the output pins
'Do the WM8804 pins first - give them more time in this state
'We want the WM8804 to power up in software mode, dependent on SDIN, SCLK and SDOUT

pin(P_WM8804_SPI_MOSI) = 1 'This is for configuring the WM8804 device
pin(P_WM8804_RST) = 0
pin(P_WM8804_IFM) = 1
pin(P_WM8804_CSB) = 1
setpin P_WM8804_SPI_MOSI, dout
setpin P_WM8804_RST, dout
setpin P_WM8804_IFM, dout
setpin P_WM8804_CSB, dout

pin(P_SI4689_SMODE) = 0 'should always be low indicating SPI mode
pin(P_SI4689_RES) = 0
pin(P_SI4689_CS) = 1

pin(P_PAM8407_ENAB) = 0 'default to 0 to shut down amp
pin(P_PAM8407_UP) = 1
pin(P_PAM8407_DN) = 1

pin(P_REG) = 0 'default to 0 to turn regulator on (ShutDown pin off)

setpin P_SI4689_SMODE, dout
setpin P_SI4689_RES, dout
setpin P_SI4689_CS, dout

setpin P_Flash_SPI_MISO, din
setpin P_Flash_SPI_MOSI, din
setpin P_Flash_SPI_CLK, din
setpin P_Flash_SPI_CS, din

setpin P_PAM8407_ENAB, dout
setpin P_PAM8407_UP, dout
setpin P_PAM8407_DN, dout

setpin P_REG, dout

setpin P_HEADPHONE, din

setpin P_MUX_0, dout
setpin P_MUX_1, dout

if (IrLed > 0) then
  pin(P_IR_ACK_LED) = 1 'IR Ack LED off - HW addition LED flashes with IR activity
  setpin P_IR_ACK_LED, dout
endif

pin(P_LED_YELLOW) = 1
pin(P_LED_RED) = 1
setpin P_LED_YELLOW, dout
setpin P_LED_RED, dout
'-----------------------------------------------------
'Reset the WM8804 Toslink framer. This requires DISABLING the SPI for a moment

dim integer WM8804_State=1

sub ResetWM8804

  'If the WM8804 is already reset, don't need to reset again
  if (WM8804_State = 0) then exit sub
  WM8804_State = 0

  on error skip 'the spi close will throw an error if it has not already been opened
  spi close

  'The operating mode of the WM8804 is dependent upon the state of SDIN, SCLK, SDOUT, CSB
  'and GPO0 when the device is powered up or a hardware reset occurs. Have a go at taking
  'WM8804 device out of reset now before we proceed further. The device will configure
  'itself into software controlled mode when it comes out of reset with MOSI/SDIN held
  'high.

  pin(P_WM8804_RST) = 0 'Put WM8804 into reset mode
  pin(P_WM8804_SPI_MOSI) = 1 'Configure pins so that WM8804 enters SW mode when it
  pin(P_WM8804_IFM) = 1      'comes out of reset
  pin(P_WM8804_CSB) = 1
  pin(P_WM8804_RST) = 1  'Take the WM8804 out of reset (and wait at least 26us)

  'The SPI port we just used above to configure the WM8804 will now be reconfigured as an
  'SPI port. I have increased the frequency to 10 MHz and it works great. Original FW
  'used 2 MHz.
  '
  'My DAB+ radio board would not work with any SPI frequency until I added three 3k3 pull
  'ups to 3.3V onto the DAB+ board for MISO MOSI and SCK, near Con8. With the pull ups in
  'place, both my programme and the original SC firmware work fine. Without pull ups,
  'neither work at all. I tried this both with and without the ribbon cable I used to
  'separate the micromite and DAB+ card and two different micromites.
  '
  'I could detect no radio noise difference running 10 MHz vs 2 MHz or anything else
  spi open SPIrate, 0, 8 'SC FW used 2 MHz (BASIC will not run faster than 26us)
  if (D > 0) then print "Reset WM8804 Toslink framer and SPI bus"
end sub

ResetWM8804
'-----------------------------------------------------
'WM8804 access routine - writing to the TOSLINK framer

sub WriteWM8804 (reg as integer, v as integer)

  pin(P_WM8804_CSB) = 0
  spi write 2, reg, v
  pin(P_WM8804_CSB) = 1
  if (D > 2) then print "*WM8804: " hex$(reg,2) "=" hex$(v,2)

end sub

'- - - - - - - - - - - - - - - - - - - - - - - - - - -
'INITIALISE WM8804
'Note the WM8804 is a set-and-forget device.
'Once it is initialised, we can ignore it and it will just do its thing.

sub InitWM8804
  'If the WM8804 is already running, don't need to reinitialise
  if (WM8804_State <> 0) then exit sub
  WM8804_State = 1

  'write integer part of PLL frequency ratio - refer tables 21-23 of WM8804 data sheet
  WriteWM8804 (&H06, &H08)

  'write fractional part of PLL frequency ratio
  WriteWM8804 (&H03, &HBA)
  WriteWM8804 (&H04, &H49)
  WriteWM8804 (&H05, &H0C)

  'Tell WM8804 to substitute zeros if the valid bit is not set
  WriteWM8804 (&H08, &H38)

  'All functions of the device are powered down by default and must be powered up
  'individually by writing to the relevant bits of the PWRDN register (0x1E)
  '0x3F=: PLL disabled, Spdif receiver disabled, spdif transmitter disabled,
  '       oscillator off, digital audio interface off, outputs tristated
  'From boot: manually shut down all components
  WriteWM8804 (&H1E, &H3F)

  'enable the WM8804 functions
  '0x02=: PLL enabled, Spdif receiver disabled, spdif transmitter enabled
  '           oscillator on, digital audio interface on, outputs enabled
  WriteWM8804 (&H1E, &H02)

  'write reg 0x1B (AIFTX): default except AIFTX_WL set to 2b10 (24b word length)
  WriteWM8804 (&H1B, &H0A)

  '0xCA: SYNC=1 Lclk+Bclk continue to output when S/pdif source removed
  '      AIF_MS=1=master mode MCLK, AIFRX_LRP=0 lrclk not inverted
  '      AIFRX_BCP=0 bclk not inverted, AIFRX_WL=10 24 bits, AIFRX_FMT=10 I2S mode
  WriteWM8804 (&H1C, &HCA)

  if (D > 0) then print "Sent WM8804 initialisation"
end sub

'-----------------------------------------------------
'There's a 4052 used as an analogue audio mux. The main functions are a mute (ground)
'and pass through (norm)
'
'There are two ways mute can be applied....
' i) the state machine can ask for mute (because its doing something that requires mute)
' ii) the user can ask for mute (using IR remote, or LCD panel command)
'
'Need to keep track of why the system is muted by ORing these states
'
'Muted: Bit 1=auto mute, Bit 2=manual mute
'WM8804 reset (muted) at boot
dim integer MuteDowncount=0, Muted=1, AudioOutMode=-1, WM8804=-1

sub SetMux (auto_mute as integer, man_mute as integer)
  local integer nextmute

  nextmute = Muted and 3 'keep only the bottom two bits

  'create a bitmap of the new mute state - an OR of automatic and manual mutes
  if (auto_mute > 0) then
    nextmute = nextmute or 1 'turn on bit 1 for mute

  elseif (auto_mute = 0) then
    nextmute = nextmute and 2 'turn off bit 1 for unmute

  endif

  if (man_mute > 0) then
    nextmute = nextmute or 2 'turn on bit 2 for mute

  elseif (man_mute = 0) then
    nextmute = nextmute and 1 'turn off bit 2 for unmute

  endif

  'nextmute now holds the new muted state
  if ((Muted = nextmute) and (AudioOutMode <> AUDIO_DIG)) then exit sub

  if ((D > 0) and (Muted <> nextmute)) then
    print "AudioMux HWmute=" auto_mute ", SWmute=" man_mute ", WM8804=" WM8804 ", Prev mute state=" Muted ", Next state=" nextmute
  endif

  'nextmute zero means not muted
  if (nextmute = 0) then

    'Decide whether to swap or maintain standard channel order
    if (CtrlVal(Sswap) = 0) then
      pin(P_MUX_1) = 0
      pin(P_MUX_0) = 1
      if (D > 1) then print "AudioMux set to Si4689 stereo normal"
    else
      pin(P_MUX_1) = 1
      pin(P_MUX_0) = 0
      if (D > 1) then print "AudioMux set to Si4689 stereo reverse"
    endif

    'If we are outputting digital, take the toslink framer out of mute mode
    if ((AudioOutMode = AUDIO_DIG) and (WM8804_State = 0)) then InitWM8804

  else
    pin(P_MUX_1) = 0
    pin(P_MUX_0) = 0
    MuteDowncount = 0
    if (D > 1) then print "AudioMux set to mute"

  endif

  Muted = nextmute

end sub

'-----------------------------------------------------
'Optional power sag interrupt
'pin 34 is configured as an open collector input with pull up. The hardware pulls
'the input low (to Digital Ground) when power is OK and goes open when power
'is lost. Reading 0 means power OK. Reading 1 means voltage sag. The power supply
'is a full wave rectified 18VAC nominal input, filtered by 40,000uF, which sits
'at about 23 volt a few seconds after power up. A 7805 pin-compatible switchmode
'regulator efficiently converts the smoothed supply to 5V without a heatsink.
'The alarm circuit has an 18V zenner from +23V, via two 10k resistors in series to
'ground. The junction of the two 10k resistors goes to base of BC549. Emitter goes
'to ground. Collector goes to both pin 34 and also a 1nF cap to ground.

dim integer DoPowerAlarm=0

'Power (voltage sag) Alarm pin requires hardware addition
if (Pv > 0) then setpin P_POWER_ALARM, INTB, PowerAlarmInt, PULLUP

sub PowerAlarmInt
  'Remember this is an interrupt. Exit the interrupt ASAP!
  'Power has either been lost (apply mute) or restored (unmute after a delay)
  if (D > 0) then print "Power Alarm Int:" pin(P_POWER_ALARM)

  if (pin(P_POWER_ALARM) = 0) then
    'If this ever happens, it must mean end of a brownout. This should be rare
    MuteDowncount = MUTE_RELEASE<<1

  else
    'Voltage is starting to sag. Immediately mute, to manage rare case when radio
    'output goes into wild oscillation on power spike, emitting piercing noise.
    SetMux (1, -1) 'HW mute the 4052
    DoPowerAlarm = 1
  endif
end sub

'-----------------------------------------------------
'Tick interrupt occurs every 5ms
'The tick interrupt will trip several other flags periodically to cause those sub-systems
'(state machines) to run at the next opportunity

'How long a pulse (in units of 5ms) to drive the audio amplifier up/down function
const HP_PULSE_WIDTH=16

dim integer SpiWait=0, PollWait=0, CapWait=0, DABwait=0, IRwait=-1, IRnum=-1, Dwait=0
dim integer Pwait=0, Reboot=0, GuiCounter=0, UpTime=0, HP_removed=0

SetTick 5, TickInterrupt '200 Hz interrupt

sub TickInterrupt
  'UpTime counts how long the micromite has been running (in units of 5ms)
  UpTime = UpTime + 1

  'GuiCounter is used for timing default closure of various screen pages
  GuiCounter = GuiCounter + 1

  'This is just an up-counter. Will trigger the SPI state machine every 5ms
  SpiWait = SpiWait + 1

  'PollWait is used to time requests to the radio chip for status updates
  PollWait = PollWait + 1

  'CapWait is a timer used in the AntCap search process
  CapWait = CapWait + 1

  'Dwait is a timer for auto-saving defaults. Non zero means its running
  if (Dwait > 0) then Dwait = Dwait + 1

  'DABwait used during the band-scan function to wait for DAB programme info to come in
  if (DABwait > 0) then DABwait = DABwait + 1

  'Pwait is a timer to hold off rewriting the presets file. Timer is stopped if <= 0
  if (Pwait > 0) then Pwait = Pwait + 1

  'IRwait used for accumulating digits / buttons in the IR remote decoding routine
  if (IRwait >= 0) then
    IRwait = IRwait + 1

    'max 10 seconds with no button presses
    if (IRwait > 2000) then
      IRnum = -1
      IRwait = -1
      CtrlVal(RLN8) = "" 'blank the line where button presses are accumulated
      CtrlVal(RLN9) = "" 'blank the 'press OK to ...' message
    endif
  endif

  'Reboot is a special reboot counter - it counts down and then does a reboot
  if (Reboot > 0) then
    Reboot = Reboot - 1

    'this is where the programme does a reboot!!
    if (Reboot = 0) then
      if (Pwait > 0) then WritePresetFile 'write presets (if pending) before reboot
      if (Dwait > 0) then WriteDefaultFile 'write config (if pending) before reboot
      cpu restart
    endif

  endif
  '- - - - - - - - - - - - - - - - - - - - - - - - - -
  'MuteDowncount is used to hold off enabling audio output after configuration changes
  'The original design outputs severe thumps/clicks under certain conditions
  if (MuteDowncount > 0) then
    MuteDowncount = MuteDowncount - 1

    if ((CapSearch <= 0) and (MuteDowncount = 0)) then
      SetMux (0, -1) 'will HW 'unmute' (i.e. initialise) the TosLink framer too
      if ((Smode = Xconsole) or (Smode = XstatusE) or (Smode = XmainE) or (Smode = XScanE)) then Change_Smode = Xstatus
    endif
  endif

  '- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  'Some care required below. This is an interrupt routine, and we don't want
  'to start writing radio messages or trouncing anything in non-interrupt land.
  'When headphone plugged in, HP_removed will be 1, and pin(P_HEADPHONE)=0
  if (pin(P_HEADPHONE) = 0) then

    if (HP_removed <> 0) then
      'Freshly detected headphone removal
      'HW mute the 4052 - this mutes HP when plugged in to stop loud volume
      'Note: call to setmux would be dangerouts from this interrupt
      'routine if it were called to UNMUTE. But as written, setmux will be OK
      'when calling to mute.
      if (Muted = 0) then SetMux (1, -1)
      HP_removed = 0
    endif

  'when else is taken, pin(P_HEADPHONE) <> 0
  elseif (HP_removed = 0) then

    HP_removed = 1 'Freshly detected that headphones plugged in

  endif

end sub

'-----------------------------------------------------
'The infra-red remote is defined here. Don't add code to process the IR messages here
'because this is an interrupt service routine! Do it from the main DO loop

dim integer IrDev=0, IrButton=0, IrTest=0

ir IrDev, IrButton, IrInt 'start the IR decoder (hard wired to pin 78 of MPU)

sub IrInt
  local S as string length 50 'This is an interrupt, so can't use S1 or S2!!

  if (D > 0) or (IrTest <> 0) then S = "IR Device Code=" + hex$(IrDev) + " Button=" + hex$(IrButton)
  if (D > 0) then print S

  if (IrTest <> 0) then
    WriteMsg (S)
    IrDev = 0 'Don't process this keypress normally! We're in a test mode!
  endif

  'Otherwise, just exit without doing anything here in the interrupt routine
  'IrDev and IrButton will be non-zero if something got pressed, and those variables will
  'be scanned in the main DO loop, outside the interrupt, where they are permitted to
  'take longer to process
end sub

'-----------------------------------------------------
'Note: Buried in the micromite plus manual is a section that describes what happens when
'gui elements overlap. The lower numbered element will take precedence. So we should make
'sure that the AREA buttons occur with the HIGHEST NUMBERS
const BOOT_TITLE=1, LN1=2, LN2=3, LN3=4, LN4=5, LN5=6, LN6=7, LN7=8, LN8=9, LN9=10
const LN10=11, LN11=12, LN12=13, LN13=14, RLN1=15, RLN2=16, RLN3=17, RLN4=18, RLN5=19
const RLN6=20, RLN7=21, RLN8=22, RLN9=23, RLN10=24, ERR1=25, ERR2=26, Ename=27, F1=28
const F2=29, F3=30, F4=31, F5=32, F6=33, Efreq=34, Eclr=35, Eband=36, EkHz=37, Edata=38
const Eframe=39, Eindex=40, Eactive=41, Efav=42, Eantcap=43, Ebandwidth=44, BCancel=45
const BSetup=46, BPreset=47, BEdit=48, BVolume=49, BScan=50, BCap=51, BRemote=52
const BLight=53, Bdown=54, Bup=55, BSeekDn=56, BSeekUp=57, BRotate=58, BDisp=59, Ram=60
const Rfm=61, Rdab=62, Rdigital=63, Rrca=64, Rspeaker=65, Sswap=66, Vvolume=67, VLight=68
const Vaudio=69, Vswap=70, Vantcap=71, Vbandwidth=72, Vfav=73, Vactive=74, Vgui=75
const Sgui=76, BMute=77, AR1=78, AR2=79

'NameLen is the maximum string length for a radio station name
const NameLen=16

'MAX_SET is used for concatenating FM station names for stations like ABC Classic FM
const MAX_SET=4

'must start with Smode 0, so the change to console is registered
dim integer Smode=0, Change_Smode=Xconsole, RadioMode=-1, DoScan=0, CapSearch=0
dim integer Frequency=-1, NewFrequency=DefaultChannel, ServiceID=-1, CompID=-1
dim integer NewServiceID=-1, NewCompID=-1, Seeking=0, SeekStart=0, NumDabFreqs=-1
dim integer DabFreqIndex=-1, NumDabServices=0, DabID=-1, DabFreqs(MAX_DAB_FREQS-1)
dim integer NumPresets=0, NumActivePresets=-1, Pfirst=-1, Plast=-1, Pprev, Pnext
dim integer PbPrev=-1, PbNext=-1, PbCan=-1, NumFavs=0, ActiveProgData=-1, CLine=LN1
dim integer Pfreq(MAX_PRESETS-1), Pid(MAX_PRESETS-1), Pcomp(MAX_PRESETS-1)
dim integer Pbw(MAX_PRESETS-1), Pflags(MAX_PRESETS-1), Pcap(MAX_PRESETS-1)
dim integer Pfav(MAX_FAVS-1)

dim string Pname(MAX_PRESETS-1) length NameLen, ProgName(3) length 2
dim string ProgData(31) length 4, RdsNameSet(MAX_SET) length 8
dim integer RdsNames=0

dim integer i, j

'-----------------------------------------------------
'PAGE 1 - used for text console
'PAGE 2 - status and information
'PAGE 3 - Most buttons for status and information (used with 2)
'PAGE 4 - << >> scan buttons (used with 3)
'PAGE 5 - Volume Control box (used with 2/3)
'PAGE 6 - AM/FM/DAB band buttons (used with 2/3)
'PAGE 7 - Favourites buttons (used with 2)
'PAGE 8 - Edit Preset screen
'PAGE 9 - Select Preset radio buttons
'PAGE 10 - Setup / config screen
'PAGE 11 - holds error message (use with 2)
'PAGE 31 - only contains touch area (used with 2)
'PAGE 32 - dim cancel button at top left corner used with several pages

cls
gui hide all 'just in case there is any crap left over from a previous run
gui delete all
colour ConsoleTextColour, rgb(black)

'- - - - - - - - - - - - - - - - - - - - - - - -
'Text console
gui setup 1
font 3

'Customise the console message by modifying below!
gui frame BOOT_TITLE, "Stefan" + chr$(39) + "s Covid-19 Radio Project", 0, MM.FontHeight>>1, MM.HRes, MM.VRes-(MM.FontHeight>>1),ConsoleOutlineColour

i = 0

for j = LN1 to LN13
  i = i + (MM.FontHeight * 1.4) 'used to set line spacing
  gui caption j, "", 9, i, "lt" 'each of these lines has 49 usable characters
next

gui area AR1, 0, 0, mm.hres, mm.vres 'full screen
'-----------------------------------------------------
sub WriteMsg (msg as string)
  local integer i

  'If we need to scroll, do that now
  if (CLine > LN13) then
    for i = LN1 to LN12
      CtrlVal(i) = CtrlVal(i+1)
    next

    CLine = LN13
  endif

  'We can now write the message to the last console line
  CtrlVal(CLine) = msg
  CLine = CLine + 1
  print msg
end sub

'-----------------------------------------------------
sub ClearConsole
  local integer i

  'Blank the console info on the screen
  for i = LN1 to LN13
    CtrlVal(i) = ""
  next

  CLine = LN1
end sub

'-----------------------------------------------------
'The BASIC software, preset and config files (and up to 9 backups of these files) are
'stored on an SD card. The files are small, so you can use your ancient 8 Mbyte SD cards
'and still heaps of flash to spare

function RotateAndOpen (base as string, ext as string, do_open as integer) as integer
  local integer i

  'Remove the oldesst backup file (if it exists)
  on error ignore
  kill base + "9" + ext

  'Rename any backups if they exist
  for i = 8 to 0 step -1
    S1 = base + str$(i) + ext
    S2 = base + str$(i+1) + ext
    name S1 as S2
  next

  if (do_open <> 0) then
    'Open a new version 0 file
    Open base + "0" + ext for output As 1
  endif

  on error abort

  if (MM.Errno <> 0) then
    WriteErr "SD Card or write err"
    name S2 as base + "0" + ext 'rename the last version back to vers 0
    RotateAndOpen = 0 'return 0 if error

  else
    RotateAndOpen = 1 'return 1 if no error
  endif
end function

'- - - - - - - - - - - - - - - - - - - - - - - -
'is there a new version of the BASIC programme to load from the SDMMC flash card?
'This allows us to update software by writing a new file called "radio0.bas" onto the
'SD card so we avoid having to set up a serial interface to the micromite and have
'computer on hand.

S1 = ""
on error skip
S1 = dir$("Radio0.bas", FILE)

if (S1 <> "") then
  if (RotateAndOpen ("Radio", ".bas", 0) = 1) then 'returns 0 if error
    'now load and run the new version (if possible)
    WriteMsg ("LOADING NEW SOFTWARE")
    pause 100 'can't avoid this or the print doesn't get out in time
    on error skip
    load "Radio1.bas",r
  endif

  'and if we fall through, the load didn't work, but by then, the original
  'will have been blown away and we are now stuffed
endif

'-----------------------------------------------------
sub ClearStatus
  local integer i

  'Blank the status info on the screen
  for i = RLN1 to RLN10
    CtrlVal(i) = ""
  next
end sub

'-----------------------------------------------------
'DefineTouch sets up the size of AR2 - which depends on whether we are going
'to be showing favourite buttons, volume control and band selection (or not)

sub DefineTouch (mode as integer)

  gui setup 31

  'reconfigure the status screen touch area based on the new setting
  on error skip
  gui delete AR2

  if (mode = 0) then
    gui area AR2, 0, 0, mm.hres, mm.vres 'minimalist screen format

  else
    gui area AR2, 0, 0, mm.hres, 334 'show it all - reduced touch area

  endif

end sub

'-----------------------------------------------------
'Status and information
gui setup 2

font 5,1.4
gui caption RLN1, "", mm.hres>>1, 1, "ct"

font 3 'this is the regular font size
gui caption RLN2, "", mm.hres>>1, 42, "ct"

font 2
gui caption RLN3, "", mm.hres>>1, 75, "ct"

'The spacing of the following lines varies in the different screen modes
font 3
i = 108

for j = RLN4 to RLN10
  gui caption j, "", mm.hres>>1, i, "ct"
  i = i + 32 'font height is nominally 32 pixels
next

'The size of Area 2 depends on the mode we're in.
DefineTouch (0) 'at the moment, we don't know what's been selected - assume 0
'- - - - - - - - - - - - - - - - - - - - - - - -
'these are buttons and other gadgets to be used with [3]
gui setup 3

font 4
gui button BPreset, "Tune to Preset", 0, 410, 245, 70, ButtonTextColour, ButtonColour
gui button BEdit, "Edit Presets", 250, 410, 245, 70
gui button BSetup, "Setup", 500, 410, 190, 70

font 3
gui button Bdown, "<", 195, 28, 60, 50, ConsoleTextColour, rgb(black)
gui button Bup, ">", 540, 28, 60, 50

'- - - - - - - - - - - - - - - - - - - - - - - -
'these are buttons and other gadgets to be used with [3]
gui setup 4

gui button BSeekDn, "<<", 125, 28, 60, 50
gui button BSeekUp, ">>", 610, 28, 60, 50

'- - - - - - - - - - - - - - - - - - - - - - - -
gui setup 5 'This is the volume control box, used with [3] when analogue audio is enabled

font 2
gui frame Vvolume, "Analogue Volume", 135, 335, 210, 68, OtherGadgetColour
gui spinbox BVolume, 135, 350, 205, 45, OtherGadgetColour, rgb(black), 1, 0, 63

CtrlVal(BVolume) = 20 'sensible default for the time being

'- - - - - - - - - - - - - - - - - - - - - - - -
gui setup 6 'This is the Band (AM/FM/DAB) radio buttons and Mute, used with [3]

font 3
gui radio Ram, "AM", 720, 360, 12, AmColour
gui radio Rfm, "FM", 720, 408, 12, FmColour
gui radio Rdab, "DAB", 720, 456, 12, DabColour

font 4
'start at x=1 to prevent the green console border being erased
gui button BMute, "Mute", 1, 335, 130, 70, ButtonTextColour, BMuteBackgrounddColour

'- - - - - - - - - - - - - - - - - - - - - - - -
sub FetchPresetDetails
  local integer i, j

  CtrlVal(Edata) = ""
  i = CtrlVal(Eindex) - 1 'index counts upwards from 1

  if (i >= NumPresets) then
    'important to keep the default short enough to fit in the box
    CtrlVal(Ename) = "##type station identifier"
    CtrlVal(Efreq) = "##freq"
    CtrlVal(Eband) = "--"
    CtrlVal(Eantcap) = 0
    CtrlVal(Eactive) = 0
    CtrlVal(Efav) = 0
    'note: don't change AM bandwidth - keep current value
    exit sub
  endif

  'at this point, the index corresponds with a valid entry
  if (Pname(i) = "") then
    CtrlVal(Ename) = "##type station identifier"
  else
    CtrlVal(Ename) = Pname(i)
  endif

  j = Pfreq(i)
  CtrlVal(Efreq) = j

  if ((j >= AmMIN) and (j <= AmMAX)) then
    CtrlVal(Eband) = "AM"
    gui hide Vantcap, Eantcap
    gui show Vbandwidth, Ebandwidth

  elseif ((j >= FmMIN) and (j <= FmMAX)) then
    CtrlVal(Eband) = "FM"
    gui hide Vbandwidth, Ebandwidth
    gui show Vantcap, Eantcap

  elseif ((j >= DabMIN) and (j <= DabMAX)) then
    CtrlVal(Eband) = "DAB"
    CtrlVal(Edata) = "Service ID:" + hex$(Pid(i)) + " Component ID:" + hex$(Pcomp(i))
    gui hide Vbandwidth, Ebandwidth
    gui show Vantcap, Eantcap

  else
    CtrlVal(Efreq) = "##freq"
    CtrlVal(Eband) = "--"
  endif

  CtrlVal(Eantcap) = Pcap(i) '0=auto, 1..128=configured
  CtrlVal(Ebandwidth) = Pbw(i)
  CtrlVal(Eactive) = Pflags(i) and 1 'Bit2=Favourite|Bit1=Manual Name|Bit0=Preset Active
  if ((Pflags(i) and 4) <> 0) then CtrlVal(Efav) = 1 else CtrlVal(Efav) = 0

end sub

'- - - - - - - - - - - - - - - - - - - - - - - -
'Favourites screen - just a dummy for now
gui setup 7

gui button F1, "", 0, 400, 4, 4, rgb(black), rgb(black)

'- - - - - - - - - - - - - - - - - - - - - - - -
'Edit Presets screen
gui setup 8

font 2
gui frame Eframe, "Preset Index", 265, 300, 250, 70, OtherGadgetColour
gui frame Vantcap, "AntCap Setting", 0, 300, 250, 70
gui frame Vbandwidth, "AM audio bandwidth", 0, 300, 250, 70
gui frame Vfav, "Favourites", 530, 300, 240, 70
gui frame Vactive, "Preset list", 530, 410, 240, 65
gui caption Edata, "", MM.HRes*.5, 160, "cm"

font 4
gui textbox Ename, mm.hres*.2, 26, mm.hres*.6, 38, OtherGadgetColour, rgb(black)
gui numberbox Efreq, mm.hres*.42, 96, mm.hres*.15, 38
gui caption Eband, "AM", mm.hres*.38, 104, "ct"
gui caption EkHz, "kHz", mm.hres*.58, 104, "lt"
gui button Eclr, "Clear", 10, 410, 180, 65, ButtonTextColour, ButtonColour

gui spinbox Eindex, 275, 313, 230, 50, OtherGadgetColour, rgb(black), 1, 1, MAX_PRESETS
gui spinbox Ebandwidth, 10, 313, 230, 50, OtherGadgetColour, rgb(black), 500, BWmin, BWmax
gui spinbox Eantcap, 10, 313, 230, 50, OtherGadgetColour, rgb(black), 1, 0, 128

font 2
gui switch Efav, "Normal|Favourite", 535, 315, 230, 45, rgb(black), OtherGadgetColour
gui switch Eactive, "Hide|Show", 535, 425, 230, 40

CtrlVal(Eantcap) = 0 'a sensible default for the time being
CtrlVal(Ebandwidth) = BWdef 'a sensible default for the time being

gui hide Vantcap, Vbandwidth, Eantcap, Ebandwidth

'Brace yourself for *another* MMBASIC bug workaround!
'The gui hide command above corrupts part of the frame around the BOOT_TITLE
'page that's already been drawn (even though the four buttons are NOT on the
'page that's currently displayed. The work around is to redraw the BOOT_TITLE
gui redraw BOOT_TITLE
'- - - - - - - - - - - - - - - - - - - - - - - -
'Select Preset screen
'The selecct preset screen is dynamic - it changes as new stations are discovered,
'and existing presets are edited. The screen needs to be built and knocked down
'dynamically because the station preset count and preset names can change dynamically

'NOTE: Seems there could be a micromite issue that causes ClearGuiPresets to only
'behave if called while page 6 is displayed. If you delete button elements while the
'buttons are hidden on a page that is not currently displayed, they are still
'erased from the page that is currently being displayed, messing up the display!

sub ClearGuiPresets
  local integer n, numbuttons

  'Pfirst is the index of the first preset station displayed on screen
  'Plast is the index of the last preset station displayed on screen
  'Pnext > 0 if there is a next button
  'Pprev >= 0 if there is a prev button

  if ((Pfirst < 0) or (Plast < 0)) then exit sub

  'Allow one extra button for cancel (which is always present)
  numbuttons = Plast - Pfirst + 2

  'Pnext > 0 if there is to be a next button. Next is last button last col
  'Pprev >= 0 if there is to be a prev button. Prev is first button in first col
  if (Pnext > 0) then numbuttons = numbuttons + 1
  if (Pprev >= 0) then numbuttons = numbuttons + 1

  'We will clear only numbuttons button
  for n = 0 to (numbuttons-1)
    gui delete Rad1 + n
  next

  Plast = -1
end sub

'- - - - - - - - - - - - - - - - - - - - - - - -
'ChooseColour is used to choose the on-screen colour of a particular frequency

function ChooseColour (f as integer) as integer
  if ((f >= AmMIN) and (f <= AmMAX)) then
    ChooseColour = AmColour 'AM band

  elseif ((f >= FmMIN) and (f <= FmMAX)) then
    ChooseColour = FmColour 'FM band

  elseif ((f >= DabMIN) and (f <= DabMAX)) then
    ChooseColour = DabColour 'DAB band

  else
    ChooseColour = NonBandColour 'Not a radio band

  endif
end function

'- - - - - - - - - - - - - - - - - - - - - - - -
'i is the number of the button - and counts from 0 up to MAX_RADIO-1
'x and y are the coordinates at which to draw the radio button
'f is frequency of the station
'n is the name of the station (or prev or next)

sub Define_Radio_Button (i as integer, x as integer, y as integer, f as integer, n as string)
  local integer col, j

  S2 = n
  if (S2 = "") then S2 = str$(f) + " kHz"
  col = ChooseColour (f)

  'Try to use a larger font, but the number of characters cannot exceed 12
  if (len(S2) > 12) then
    font 2
  else
    font 3
  endif

  colour ConsoleTextColour, rgb(black)
  gui radio Rad1+i, S2, x, y, RADIO_RADIUS, col

  'If the radio is currently tuned to this station, colour the circle
  j = Pfirst + i

  if ((Frequency = f) and ((Frequency < DabMIN) or ((ServiceID = Pid(j)) and (CompID = Pcomp(j))))) then
    CtrlVal(Rad1+i) = 1
  else
    CtrlVal(Rad1+i) = 0
  endif
end sub

'- - - - - - - - - - - - - - - - - - - - - - - -
'SetupGuiPresets is called when you want to display the station presets menu
'Because the station preset names can be learned in real time, or edited by
'the user, we will set up and tear down the station presets menu each time
'it is required using this subroutine
'
'When the number of presets exceeds MAX_RADIO, we need to handle next/prev

sub SetupGuiPresets
  local integer n, x, y, z, spacing

  'Pfirst is the index of the first preset station to display on screen
  'Plast is the index of the last preset station to display on screen
  'Pnext > 0 if there is to be a next button
  'Pprev >= 0 if there is to be a prev button

  Pprev = -1
  Pnext = -1
  PbPrev = -1
  PbNext = -1

  'Determine the range of presets that are next to be displayed on screen
  'Firstly, check whether there is anything to display at all
  if (NumActivePresets <= 0) then
    Pfirst = -1
    exit sub
  endif

  'There is something to display.
  'There are MAX_RADIO radio buttons on the screen. But always leave space for
  'a cancel button, and we might need to leave space for a next or prev or
  'both of these. So we may need ONE, TWO or THREE extra buttons.
  if (NumActivePresets < MAX_RADIO) then
    Pfirst = 0
    Plast = NumActivePresets - 1

  else 'There must be more than one screen-load of preset stations
    'The range to be displayed on next screen depends on the value in Pfirst
    z = 1
    if (Pfirst < 0) then Pfirst = 0

    'Will we need to display a prev button? Yes if Pfirst > 0
    if (Pfirst > 0) then
      'there must be a prev button. Reduce index of last to be displayed
      'to make way for a prev button
      Pprev = Pfirst - (MAX_RADIO - 1) 'If you are clever, you will see the 'bug'
      if (Pprev < 0) then Pprev = 0
      z = 2
    endif

    'how many outstanding presets are left to display?
    n = NumActivePresets - Pfirst 'calc valid if this screen not full

    'Will we need to display a next choice?
    if (n > (MAX_RADIO - z)) then
      'there must be a next button
      z = z + 1
      n = MAX_RADIO - z
    endif

    Plast = Pfirst + n - 1 'If you are clever, you will see the 'bug'
    if (NumActivePresets > (Plast + 1)) then Pnext = Plast + 1

  endif

  gui setup 9

  'Work out row spacing based on there being a minimum row height (of the characters)
  'of RADIO_RADIUS*2 pixels, and the maximum display height of MM.VRes
  spacing = int((MM.VRes - (RADIO_RADIUS << 1)) / (NUM_PRESET_ROWS - 1))

  for n=0 to (MAX_RADIO - 1)

    'determine x and y of the next button
    if (n < NUM_PRESET_ROWS) then
      x = RADIO_RADIUS
      y = RADIO_RADIUS + (spacing * n)

    elseif (n < (NUM_PRESET_ROWS<<1)) then
      x = 275
      y = RADIO_RADIUS + (spacing * (n-NUM_PRESET_ROWS))

    else
      x = 535
      y = RADIO_RADIUS + (spacing * (n-(NUM_PRESET_ROWS<<1)))
    endif

    'add the next and prev markers if required
    if ((Pfirst + n) > Plast) then

      if ((Pprev >= 0) and (PbPrev < 0)) then
        Define_Radio_Button (n, x, y, 0, "Previous Page")
        PbPrev = n
        continue for
      endif

      if ((Pnext > 0) and (PbNext < -0)) then
        Define_Radio_Button (n, x, y, 0, "Next Page")
        PbNext = n
	continue for
      endif

      'Always define a cancel button
      Define_Radio_Button (n, x, y, 0, "Cancel")
      PbCan = n
      exit for
    endif

    'its a station preset - add that
    Define_Radio_Button (n, x, y, Pfreq(Pfirst+n), Pname(Pfirst+n))
  next

end sub

'- - - - - - - - - - - - - - - - - - - - - - - -
'Config screen
gui setup 10

font 4
gui button BRemote, "IR Remote Test", 0, 300, 255, 65, ButtonTextColour, ButtonColour
gui button BCap, "AntCap Scan", 265, 300, 255, 65
gui button BScan, "Station Scan", 530, 300, 255, 65

gui button BRotate, "New Error Log", 0, 410, 255, 65
gui button BDisp, "Disp Error Log", 530, 410, 255, 65

font 2
gui frame Vlight, "Backlight", 0, 180, 200, 75, OtherGadgetColour

font 3
gui spinbox BLight, 0, 195, 200, 50, OtherGadgetColour, rgb(black), 1, 10, 100
CtrlVal(BLight) = 50 'a sensible default for the time being

font 2
gui frame Vaudio, "Audio Output Port Selection", 120, 15, 620, 140, OtherGadgetColour

gui radio Rdigital, "Optical or coax digital output [or headphones]", 145, 45, 15, DigitalColour
gui radio Rrca, "Analogue output via RCA [or headphones]", 145, 90, 15, RCAColour
gui radio Rspeaker, "Analogue output via speakers [or headphones]", 145, 135, 15, SpeakerColour

gui frame Vswap, "Analogue Channels", 265, 180, 255, 75, OtherGadgetColour
gui switch Sswap, "Normal|Swapped", 280, 195, 230, 50, rgb(black), OtherGadgetColour

gui frame Vgui, "Show Fav Buttons", 530, 180, 255, 75, OtherGadgetColour
gui switch Sgui, "No|Yes", 543, 195, 230, 50, rgb(black), OtherGadgetColour

CtrlVal(Sswap) = 0
CtrlVal(Sgui) = 0

'- - - - - - - - - - - - - - - - - - - - - - - -
gui setup 11 'This is the error screen - to be overlaid onto other pages

font 5,1.5 'scale size larger - enough for 17 characters
i = mm.fontheight
gui caption ERR1, "Muting due to", mm.hres>>1, 275, "ct", rgb(red)

font 5 'normal scale but still large font - enough for 33 characters
gui caption ERR2, "", mm.hres>>1, 275+i, "ct", rgb(red)

'- - - - - - - - - - - - - - - - - - - - - - - -
'The last page is the cancel button at the top left - it is used with several pages
gui setup 32

font 5
gui button BCancel, "X", 0, 0, 60, 60, CancelTextColour, CancelBackgroundColour

'- - - - - - - - - - - - - - - - - - - - - - - -
sub WriteErr (msg as string)
  CtrlVal(ERR2) = msg 'write the error on LCD error page in red
  WriteMsg (msg)

  select case Smode
    case Xstatus
      Change_Smode = XstatusE

    case Xmain, Xedit, Xselect, Xsetup
      Change_Smode = XmainE

    case Xscan
      Change_Smode = XscanE

    case Xcap
      Change_Smode = XcapE

    case Xdisp
      Change_Smode = XdispE

    case 0
      Change_Smode = Xconsole

  end select
end sub

'- - - - - - - - - - - - - - - - - - - - - - - -
'SetRadioModeColour sets the radio buttons on the default screen to show the band
'that is currently selected

sub SetRadioModeColour
  'Determine the current band and set the state of the radio button for the band
  if (RadioMode = MODE_AM) then CtrlVal(Ram) = 1 else CtrlVal(Ram) = 0
  if (RadioMode = MODE_FM) then CtrlVal(Rfm) = 1 else CtrlVal(Rfm) = 0
  if (RadioMode = MODE_DAB) then CtrlVal(Rdab) = 1 else CtrlVal(Rdab) = 0
end sub

'- - - - - - - - - - - - - - - - - - - - - - - -
'setup page 7 from scratch to grab latest (that's the favourite buttons)
'Because I would like to satisfy Ingo's desire that AM stations occupy
'the topmost favourite positions, but my desire not to have 'hanging buttons'
'when fewer than five favourites have been specified, there will be some
'jiggery pokery with the button positions below

sub SetupPageSeven
  local integer i, f, x, y, z, col
  local string s, t

  gui setup 7

  on error skip
  gui delete F1, F2, F3, F4, F5, F6

  if (NumFavs > 0) then
    'Now recreate the buttons - and the position of the first button
    'depends on whether there are 1-4 or 5-6 buttons to display
    if (NumFavs < 5) then
      x = 0 'start on the bottom row if 1-4 buttons
      y = 410
    else
      x = 350 'start on the top row if 5-6 buttons
      y = 335
    endif

    for i = 0 to (NumFavs - 1)
      f = Pfreq(Pfav(i))
      col = ChooseColour (f)

      s = Pname(Pfav(i))
      if (s = "") then s = str$(f) + " kHz"

      'Try to display the station name in the largest possible font
      'This means creating a line break at a space (if required) and
      'making sure each of the cut lines will fit into the button.
      'The max number of chars that will fit in a line using font 4
      'is 10. Using font 2, you can fit 13 characters on a line.
      font 4
      z = len(s)

      if (z > 10) then
        'determine position of last space in string s
        for f=z to 2 step -1
          if (mid$(s, f, 1) = " ") then exit for
        next

        'By splitting string s, can we end up with two lines that are
        'each no more than 10 characters? If so, font 4 will be OK over
	'two lines of button text. Or if we split into two lines of no
        'more than 13 characters, we can use font .
        if ((f < 14) and ((z-f) < 14)) then
          'add a line break character '~'
          t = left$(s, f-1) + "~" + mid$(s, f+1)
          s = t

          if ((f > 10) or ((z-f) > 10)) then font 2

        else
          font 1

        endif
      endif

      gui button F1+i, s, x, y, 170, 70, rgb(black), col
      x = x + 175

      'When x reaches the right hand side, change row
      'This will only happen if we are doing 5-6 buttons and
      'we started on the top row. (because if we are doing 1-4
      'buttons, we started on the bottom row and there are no more
      'buttons to draw when x>=700)
      if (x >= 700) then
        x = 0 'move to the bottom row if 5-6 buttons
        y = 410
      endif
    next
  endif
end sub
'- - - - - - - - - - - - - - - - - - - - - - - -
'ProcessScreenChange does the swapping from one screen to another. It mops up the mess
'from the previous screen before changing to the next

dim integer ExitTimer=0

sub ProcessScreenChange

  if (Smode = Change_Smode) then exit sub 'just a precaution

  if (Smode = Xedit) then
    gui textbox cancel
    gui numberbox cancel

  elseif (Smode = Xselect) then
    ClearGuiPresets 'Note mmbasic issue! Must only be called when presets displayed!

  elseif ((Smode = XIRTest) or (Smode = Xcap)) then
    ClearConsole

  else
    CtrlVal(RLN8) = ""
    CtrlVal(RLN9) = ""
    CtrlVal(RLN10) = ""
  endif

  'Now change to the new screen
  select case Change_Smode
    case Xconsole, XIRTest, Xcap, Xdisp
      page 1
      ExitTimer = 0

    case Xstatus
      DefineTouch (CtrlVal(Sgui)) 'in case the favourites mode changed

      'There's a choice about what is displayed with page 2 depending on Sgui setting
      if (CtrlVal(Sgui) = 0) then
        page 2, 31

      else
        SetupPageSeven

        'Don't display the volume control unless in AUDIO_SPKR mode
        if (AudioOutMode <> AUDIO_SPKR) then
          page 2, 6, 7, 31
        else
          page 2, 5, 6, 7, 31
        endif

      endif

      ExitTimer = 0

    case Xmain
      SetRadioModeColour
      DefineTouch (CtrlVal(Sgui)) 'in case the headphones status changed

      'In DAB mode, we don't display page 4 (the seek buttons)
      'Don't display the volume control unless in AUDIO_SPKR mode
      if (RadioMode <> MODE_DAB) then
        if (AudioOutMode <> AUDIO_SPKR) then
          page 2, 3, 4, 6, 32
        else
          page 2, 3, 4, 5, 6, 32
        endif

      else 'we are in DAB mode
        if (AudioOutMode <> AUDIO_SPKR) then
          page 2, 3, 6, 32
        else
          page 2, 3, 5, 6, 32
        endif
      endif

      ExitTimer = 2000 '10 seconds

    case Xedit 'edit presets
      page 8, 32
      ExitTimer = 12000 '60 seconds

    case Xselect 'select a preset
      SetupGuiPresets
      page 9
      ExitTimer = 3000 '15 seconds

    case Xsetup 'setup screen
      page 10, 32
      ExitTimer = 2000 '10 seconds

    case Xscan 'scan (setup the preset file)
      page 2
      ExitTimer = 0

    case XstatusE, XscanE, XcapE, XdispE 'error in status, scan, AntCap, Disp screens
      page 2, 12
      ExitTimer = 2000 '10 seconds

    case XmainE 'error in main screen
      DefineTouch (CtrlVal(Sgui)) 'in case the headphones status changed

      'In DAB mode, we don't display page 4 (the seek buttons)
      'Don't display the volume control unless in AUDIO_SPKR mode
      if (RadioMode <> MODE_DAB) then
        if (AudioOutMode <> AUDIO_SPKR) then
          page 2, 3, 4, 6, 11, 32
        else
          page 2, 3, 4, 5, 6, 11, 32
        endif

      else 'we are in DAB mode
        if (AudioOutMode <> AUDIO_SPKR) then
          page 2, 3, 6, 11, 32
        else
          page 2, 3, 5, 6, 11, 32
        endif
      endif

      ExitTimer = 1000 '5 seconds

    case else
      S1 = "Unknown screen mode " + str$(Change_Smode) 'We have a bug
      error S1
  end select

  if (D > 1) then print "Changing Screen Mode from " Smode " to " Change_Smode " ExitTimer=" ExitTimer
  Smode = Change_Smode
  Change_Smode = 0
  GuiCounter = 0

  'Try and sync file writes with a screen change, to minimise perceived delay
  if (Pwait > 0) then
    WritePresetFile 'write presets (if pending, including compacting)
    ReadPresetFile 'reload because something might have changed
    SetFavs
  endif

  if (Dwait > 0) then WriteDefaultFile 'write config (if pending)
end sub

'----------------------------------------------------------
'Touch interrupts handled here - note they need to be SHORT!

dim integer ButtonDn=0, ButtonUp=0
gui Interrupt TouchDown, TouchUp

'TouchDown called when touch first detected
sub TouchDown
  ButtonDn = Touch(REF)
  if (D > 1) then print "Touch down:" ButtonDn " Smode=" Smode
  GuiCounter = 0 'This will reset the exit timer back to the start

  select case ButtonDn
    case AR1 'transparent area behind page 1
      if (CapSearch = 2) then 'touch moves from state 2 to state 3
        CapSearch = 3

      elseif ((Smode = XIRTest) or (CapSearch = 10)) then
        Change_Smode = Xsetup 'this is the escape back to setup screen
        CapSearch = 0
        IrTest = 0

      elseif (DispErrors = 1) then
        DispErrors = 2

      endif

    case AR2 'transparent area behind page 31
      Change_Smode = Xmain

  end select
end sub

'- - - - - - - - - - - - - - - - - - - - - - - -
'TouchUp called when touch released
sub TouchUp
  ButtonUp = Touch(LASTREF)
  if (D > 1) then print "Touch up:" ButtonUp
  GuiCounter=0
end sub

'----------------------------------------------------------
'Routine that searches a desired DAB tune frequency and returns an index into DAB
'frequency table

function DabFreqToIndex (freq as integer) as integer
  for DabFreqToIndex = 0 to (NumDabFreqs-1)
    if (DabFreqs(DabFreqToIndex) = freq) then exit for
  next

  if (DabFreqToIndex >= NumDabFreqs) then
    if (D > 0) then Print "DAB frequency not in DabFreqs table:" freq
    DabFreqToIndex=0
  endif
end function

'----------------------------------------------------------
'The exit timer is used to automatically change screens if there has been no input for a
'while

sub ProcessExitTimer
  select case Smode
    case Xmain
      Change_Smode = Xstatus

    case Xedit, Xsetup, Xselect
      Change_Smode = Xmain

    case XstatusE
      Change_Smode = Xstatus

    case XmainE
      Change_Smode = Xmain

    case XscanE
      Change_Smode = Xscan

    case XcapE
      Change_Smode = Xcap

    case Xdisp
      on error skip
      close 2
      DispErrors = 0
      Change_Smode = Xsetup

    case XdispE
      Change_Smode = Xdisp

  end select

  ExitTimer = 0
end sub

'----------------------------------------------------------
'The button pressing routines happen below. Note that Infra Red commands are sometimes
'made to appear like GUI button presses.
'NOTE: THIS ROUTINE SHOULD NOT BE CALLED UNLESS STATE=0
'(because we are going to change state or write to the radio SPI directly from here!)
'
'The MMBasic interrupt routine seems to have bugs or is not consistent. Sometimes a
'button press returns BOTH a button down and a button up interrupt. Other times it seems
'only to return a button up interrupt. Its as if the buttons on the LCD are polled rather
'than interrupt driven within MMBASIC and if you happen to press briefly, then it might
'miss something? This doesn't seem like a good explanation though. It feels like the
'cause is different. The following variable is a kind of kludge to try and get the button
'up interrupt. I noticed this problem with the SC firmware as well as my own, so assume
'another MMBASIC issue.

dim integer BDvalue=0, DispErrors=0
dim integer DirtyPresets=0 'The 'dirty' flag is set when a sort is required
const DEFAULT_WAIT_TIME=6000 'equals 30 seconds

sub ProcessButtonDown
  local integer i, j=-1, k

  if (D > 0) then print "Button Down:" ButtonDn " State=" State ", Smode=" Smode
  GuiCounter = 0
  BDvalue = ButtonDn

  select case ButtonDn
    case BEdit
      Change_Smode = Xedit
      if (Pwait > 0) then Pwait = 1 'Restart the preset file rewrite timer if running
      FetchPresetDetails

    case BPreset
      if (NumActivePresets <= 0) then
        CtrlVal(RLN9) = "Presets not yet defined"
      else
        Change_Smode = Xselect
      endif

    case BSetup
      Change_Smode = Xsetup

    case BLight
      backlight CtrlVal(BLight)
      Dwait = DEFAULT_WAIT_TIME * 0.9 'Start the default counter ticking

    case Ram 'manually selected AM band
      for i = 0 to (NumActivePresets - 1)
        if ((Pfreq(i) >= AmMIN) and (Pfreq(i) <= AmMAX)) then
          if (j < 0) then j = i
          k = i
        endif
      next

      'Set a default new frequency in the middle of the AM band - Remember 9 kHz spacing
      NewFrequency = ((AmMIN + AmMAX) >> 1) / AmDELTA
      NewFrequency = NewFrequency * AmDELTA
      goto JUMP_MIDDLE

    case Rfm 'manually selected FM band
      'Scan all active presets. j will be first FM index, k is last FM index
      for i = 0 to (NumActivePresets - 1)
        if ((Pfreq(i) >= FmMIN) and (Pfreq(i) <= FmMAX)) then
          if (j < 0) then j = i
          k = i
        endif
      next

      'Set a default new frequency in the middle of the FM band - 100 kHz spacing
      NewFrequency = ((AFmMIN + FmMAX) >> 1) / FmDELTA
      NewFrequency = NewFrequency * FmDELTA
      goto JUMP_MIDDLE

    case Rdab 'manually selected DAB band
      for i = 0 to (NumActivePresets - 1)
        if ((Pfreq(i) >= DabMIN) and (Pfreq(i) <= DabMAX)) then
          if (j < 0) then j = i
          k = i
        endif
      next

      NewFrequency = 206352 'in the middle of the DAB band

    JUMP_MIDDLE:
      if (j >= 0) then NewFrequency = Pfreq((j + k) >> 1)
      goto RETUNE

    case Rdigital 'Trying to select digital output
      if (WM8804 = 0) then 'don't permit digital output if no hardware for it!
        ReadDefaultFile

      else
        Dwait = DEFAULT_WAIT_TIME * 0.9 'Start the default counter ticking
      endif

    case Rrca, Rspeaker, BVolume, Sswap
      Dwait = DEFAULT_WAIT_TIME * 0.9 'Start the default counter ticking

    case Bdown
      if ((Frequency > AmMIN) and (Frequency <= AmMAX)) then NewFrequency = Frequency - AmDELTA
      if ((Frequency > FmMIN) and (Frequency <= FmMAX)) then NewFrequency = Frequency - FmDELTA

      if ((Frequency > DabMIN) and (Frequency <= DabMAX)) then 'Dab needs to use the list
        DabID = DabFreqToIndex(Frequency)

        if (DabID > 0) then
          DabID = DabID - 1
          NewFrequency = DabFreqs(DabID)
        endif
      endif

      goto RETUNE

    case Bup
      if ((Frequency >= AmMIN) and (Frequency < AmMAX)) then NewFrequency = Frequency + AmDELTA
      if ((Frequency >= FmMIN) and (Frequency < FmMAX)) then NewFrequency = Frequency + FmDELTA

      if ((Frequency >= DabMIN) and (Frequency < DabMAX)) then 'Dab needs to use the list
        DabID = DabFreqToIndex(Frequency)

        if (DabID < (NumDabFreqs-1)) then
          DabID = DabID + 1
          NewFrequency = DabFreqs(DabID)
        endif
      endif

      goto RETUNE

    case BSeekDn
      'SEEK the next station up or down
      'Seeking:
      'Bit0: 0=seek down, 1=seek up
      'Bit1: 0=single seek, 1=full scan
      'Bit6: 1=Seek Completed
      'Bit7: 1=Seek Requested

      'The radio will throw command errors if you try and scan past the end of the band
      if (((RadioMode = MODE_AM) and (Frequency > AmMIN)) or ((RadioMode = MODE_FM) and (Frequency > FmMIN))) then
        Seeking = &H80 'Seek requested + seek down
        goto SEEK_SET
      endif

    case BSeekUp
      if (((RadioMode = MODE_AM) and (Frequency < AmMAX)) or ((RadioMode = MODE_FM) and (Frequency < FmMAX))) then
        Seeking = &H81 'Seek requested + seek up

      SEEK_SET:
        'Are we in the AM band, or FM band? Choose correct seek state
        if (RadioMode = MODE_AM) then State = 1000
        if (RadioMode = MODE_FM) then State = 1100
      RETUNE:
        ServiceID = -1
        CompID = -1
        NewServiceID = -1
        NewCompID = -1
        LastDabServList = -1
        Dwait = 1  'Start the default counter ticking
	AntCap = 0
        ClearStatus
      endif

    case BMute
      if ((Muted and 2) = 0) then
        SetMux (-1, 1) 'mute the 4052 (manual mute)
        SetRadioMute 'apply Si4689 mute to both channels
        gui bcolour rgb(red), BVolume, RLN1, RLN2

      else
        SetMux (-1, 0) 'unmute (manual unmute)
        SetRadioMute 'apply Si4689 unmute to both channels
        gui bcolour rgb(black), BVolume, RLN1, RLN2
        State = 452 'set current volume
      endif

    case BScan
      DoScan = 1

    case BCap
      CapSearch = 1

    case BRotate
      'Only perform an error log rotate if the file Error0.csv exists
      S1 = ""
      on error skip
      S1 = dir$("Error0.csv", FILE)
      i = 0 '0 means error

      'If the file exists, the string will be non-null and we can rotate
      'otherwise, play a long beep indicating rotate failed
      if (S1 <> "") then i = RotateAndOpen ("Error", ".csv", 1) 'returns 0 if error

      if (i = 0) then
        gui beep 700 'play a long tone

      else
        'file is open - initialise by writing a heading
        on error ignore
        print #1, DATE$ "," TIME$ ",Created new error file"
        print #1, "Date Time UpTime Type State Status Freq Serv Comp"
        Close 1
        on error abort
      endif

    case BDisp
      'Display the contents of the latest error file
      if (DispErrors = 0) then
        'Try opening Error0.csv
        on error skip
        Open "Error0.csv" for input As 2

        if (MM.Errno <> 0) then
          gui beep 700

        else
          DispErrors = 2 'and file opened as 2
          Change_Smode = Xdisp

        endif
      endif

    case BCancel 'Same response as if the exit timer expired
      ProcessExitTimer

    case BRemote
      Change_Smode = XIRTest
      ClearConsole
      WriteMsg ("Infrared remote test mode")
      WriteMsg ("** Touch screen to exit this mode **")
      IrTest = 1

    case F1
      if (NumFavs > 0) then
        i = 0
        goto B_FAV
      endif

    case F2
      if (NumFavs > 1) then
        i = 1
        goto B_FAV
      endif

    case F3
      if (NumFavs > 2) then
        i = 2
        goto B_FAV
      endif

    case F4
      if (NumFavs > 3) then
        i = 3
        goto B_FAV
      endif

    case F5
      if (NumFavs > 4) then
        i = 4
        goto B_FAV
      endif

    case F6
      if (NumFavs > 5) then
        i = 5

       B_FAV:
        if (D > 0) then print "Choosing favourite: " i+1 ", Name=" Pname(Pfav(i)) " Freq=" Pfreq(Pfav(i))

        'Set short gui timeout after favourite button pressed
        ExitTimer = 300 '1.5 seconds
        NewFrequency = Pfreq(Pfav(i))
        NewServiceID = Pid(Pfav(i))
        NewCompID = Pcomp(Pfav(i))
        AntCap = Pcap(Pfav(i))
	CtrlVal(Ebandwidth) = Pbw(Pfav(i))
        Dwait = 1 'Start the default counter ticking
      endif

    case Sgui
      Dwait = 1 'Start the default counter ticking

    case Ename, Efreq
      'Button Down interrupt means the keyboard has been brought up
      'For some reason, MMBASIC ignores the current CtrlVal() setting, and
      'draws a blank keyboard. You need to write back the same characters that are
      'already there! That is: CtrlVal(Ename)=CtrlVal(Ename) will work!!!
      'This seems like another MMBASIC bug to me. But we will solve a little
      'more elegantly because the simple method breaks if there's no default.
      i = CtrlVal(Eindex)
      CtrlVal(Ename) = Pname(i-1)
      CtrlVal(Efreq) = Pfreq(i-1)

    case Eclr
      'Clear (erase) this preset from the preset list
      i = CtrlVal(Eindex)
      if ((Pname(i-1) <> "") or (Pfreq(i-1) >= AmMIN)) then
        Pwait = 1
        DirtyPresets = 1 ' set the dirty flag
      endif

      Pname(i-1) = ""
      Pfreq(i-1) = 0
      Pid(i-1) = -1
      Pcomp(i-1) = -1
      Pcap(i-1) = -1
      Pbw(i-1) = BWdef
      Pflags(i-1) = 0 'Bit2=Favourite|Bit1=Manual Name|Bit0=Preset Active
      FetchPresetDetails

    case Eindex 'Process button down as well as up to catch auto-repeat
      FetchPresetDetails

    case Eantcap 'AntCap setting
      i = CtrlVal(Eindex)
      Pcap(i-1) = CtrlVal(Eantcap)
      Pwait = 1

    case Ebandwidth 'AM bandwidth setting (only visible when AM selected)
      i = CtrlVal(Eindex)
      Pbw(i-1) = CtrlVal(Ebandwidth)

      'If the currently tuned station is this one, change the bandwidth in real time
      if (Frequency = Pfreq(i-1)) then
        State = 50 'change the bandwidth, then retune
      endif

      'After the new bandwidth is written to the radio, we must retune to same
      'frequency in order to force the new setting to take effect
      NewFrequency = Pfreq(i-1)
      AntCap = -1
      Pwait = 1

    case Efav 'Swap between favourite and not favourite
      'Bit2=Favourite|Bit1=Manual Name|Bit0=Preset Active
      i = CtrlVal(Eindex)
      Pflags(i-1) = Pflags(i-1) and &HFB
      if (CtrlVal(Efav) <> 0) then Pflags(i-1) = Pflags(i-1) or 4
      Pwait = 1

    case Eactive 'Swap between active and inactive preset record
      'Bit2=Favourite|Bit1=Manual Name|Bit0=Preset Active
      i = CtrlVal(Eindex)
      Pflags(i-1) = (Pflags(i-1) and &HFE) or (CtrlVal(Eactive) and 1)
      Pwait = 1

    case is >= Rad1 'Preset selected
      i = ButtonDn - Rad1

      if (PbPrev = i) then
        if (D > 0) then print "Previous preset screen"
	ClearGuiPresets
        Pfirst = Pprev
        SetupGuiPresets
        ExitTimer = 2000 '10 seconds

      elseif (PbNext = i) then
        if (D > 0) then print "Next preset screen"
	ClearGuiPresets
        Pfirst = Pnext
        SetupGuiPresets
        ExitTimer = 2000 '10 seconds

      elseif (PbCan = i) then
        if (D > 0) then print "Cancel preset screen"
        ProcessExitTimer 'initiate a screen change

      else
        i = i + Pfirst 'adjust for second and subsequent preset screen
        if (D > 0) then print "Choosing preset: Name=" Pname(i) " Freq=" Pfreq(i)
        NewFrequency = Pfreq(i)
        NewServiceID = Pid(i)
        NewCompID = Pcomp(i)
        AntCap = Pcap(i)
	CtrlVal(Ebandwidth) = Pbw(i)
        Dwait = 1 'Start the default counter ticking

        'Set short gui timeout after radio button pressed
        ExitTimer = 300 '1.5 seconds
      endif
  end select

  ButtonDn = 0
end sub

'- - - - - - - - - - - - - - - - - - - - - - - -
'refer to button down regarding BDvalue kludge and gui issues

sub ProcessButtonUp
  local integer i

  if (D > 0) then print "Button Up:" ButtonUp
  GuiCounter=0

  select case ButtonUp
    case Ename 'changed the station name
      i = CtrlVal(Eindex)
      if (i > NumPresets) then NumPresets = i

      if (Pname(i-1) <> CtrlVal(Ename)) then
        Pname(i-1) = left$(CtrlVal(Ename), NameLen)
        Pwait = 1
      endif

      if (CtrlVal(Ename) <> "") then
        'Bit2=Favourite|Bit1=Manual Name|Bit0=Preset Active
        Pflags(i-1) = Pflags(i-1) or 2
      else
        Pflags(i-1) = Pflags(i-1) and 5
      endif

      if (D > 1) then print "Name change:" CtrlVal(Ename) " index:" i
      DirtyPresets = 1 ' set the dirty flag

    case Efreq 'change the station frequency
      i = CtrlVal(Eindex)
      if (i > NumPresets) then NumPresets = i

      'Update the frequency in the database if it has changed
      if (Pfreq(i-1) <> CtrlVal(Efreq)) then
        Pfreq(i-1) = CtrlVal(Efreq)
        if ((Pflags(i-1) and 1) = 0) then Pflags(i-1) = Pflags(i-1) or 1
        FetchPresetDetails
        if (CtrlVal(Eband) <> "--") then Pwait = 1 'restart the preset write timer
      endif

      if (D > 1) then print "Freq change:" CtrlVal(Efreq) " index:" i
      DirtyPresets = 1 ' set the dirty flag

    case Eindex 'Selected a new index
      FetchPresetDetails 'this will display the new data

    case else 'here comes the gui kludge
      if (ButtonUp <> BDvalue) then 'button up that doesn't have a matching button down
        ButtonDn = ButtonUp
        if (D > 0) then print "Button down interrupt missed? " ButtonUp
        ProcessButtonDown
      endif
  end select

  ButtonUp = 0
  BDvalue = 0
end sub

'-----------------------------------------------------
'Is the frequency valid? Does it fall between the two ends of a known band?

function ValidFreq (f as integer) as integer

  if (((f >= AmMIN) and (f <= AmMAX)) or ((f >= FmMIN) and (f <= FmMAX)) or ((f >= DabMIN) and (f <= DabMAX))) then
    ValidFreq = 1 'accept as valid
  else
    ValidFreq = 0 'frequency is not within a supported band
  endif

end function
'-----------------------------------------------------
'Function to find a station's name from preset list. The function also updates the
'presets list with a station name, if no station name already registered.
'The function is used to find a station's name if just casually tuning to a
'frequency, and to store the staion name when we learn it via DAB or RDS data.
'set defname to null if doing a name lookup only
'set defname to non-null if wanting to update the station name in presets file

function MatchName (freq as integer, defname as string, sid as integer, cid as integer, SetAntcap as integer) as string
  local integer i, kludge

  for i = 0 to (NumPresets - 1)
    if ((sid >= 0) and (sid <> Pid(i))) then continue for
    if ((cid >= 0) and (cid <> Pcomp(i))) then continue for
    if (Pfreq(i) = freq) then exit for
  next

  'The following is a kludge because the BASIC language keeps evaluating all the
  'expressions along a long IF statement, even if the earlier ones are false
  'and this can cause array index issues when the index variable is out of bounds
  'which causes a run time error without this kludge
  kludge = 1

  if (i < NumPresets) then
    kludge = (sid <> Pid(i)) or (cid <> Pcomp(i))
  endif

  'Choose a sensible default if we cannot match an entry in the preset table
  if ((defname = "") or ((freq >= DabMIN) and (kludge > 0))) then
    MatchName = str$(freq) + " kHz"
  else
    MatchName = defname
  endif

  if (i < NumPresets) then
    'we found an entry in the preset table - Should we update preset?
    'Bit 1 of Pflags() means we set the station name manually

    'Is there an opportunity to update the name in the presets file?
    if (((Pflags(i) and 2) = 0) and (len(defname) > len(Pname(i))) and (right$(defname,3) <> "kHz")) then
      Pname(i) = defname
      Pwait = 1
    endif

    if (((Pflags(i) and 2) = 0) and (right$(Pname(i),3) = "kHz") and (defname <> "") and (right$(defname,3) <> "kHz")) then
      Pname(i) = defname
      Pwait = 1
    endif

    if ((Pname(i) <> "") and ((freq <= FmMAX) or ((sid = Pid(i)) and (cid = Pcomp(i))))) then MatchName = Pname(i)

    'Shall we retune this station because the AntCap setting has changed?
    if ((SetAntcap <> 0) and (Pcap(i) <> AntCap) and (AntCap = 0)) then
      NewFrequency = Pfreq(i)
      NewServiceID = Pid(i)
      NewCompID = Pcomp(i)
      AntCap = Pcap(i)
      CtrlVal(Ebandwidth) = Pbw(i)
      if (D > 1) then print "Retuning because antcap has changed"
    endif

  elseif ((ValidFreq(freq) <> 0) and (defname <> "") and (right$(defname,3) <> "kHz") and (NumPresets < MAX_PRESETS)) then

    'We didn't find the entry in the preset table, but we want to save
    'if its a real station name
    Pfreq(NumPresets) = freq
    Pname(NumPresets) = defname
    Pid(NumPresets) = sid
    Pcomp(NumPresets) = cid
    Pcap(NumPresets) = 0
    Pbw(NumPresets) = -1
    Pflags(NumPresets) = 1  'Bit2=Favourite|Bit1=Manual Name|Bit0=Preset Active
    NumPresets = NumPresets + 1
    Pwait = 1 'starts the timer to write the presets file back out

  endif

end function

'-----------------------------------------------------
'DispIRnum used by IR remote control routines to display a preset or favourite

sub DispIRnum
  local integer f=0, p=0
  local string s=""

  if ((IRnum > 0) and (IRnum <= NumFavs)) then
    'Display the candidate favourite
    f = Pfreq(Pfav(IRnum-1))
    s = "Fav " + str$(IRnum) + "="

    if (Pname(Pfav(IRnum-1)) <> "") then
      s = s + chr$(39) + Pname(Pfav(IRnum-1)) + chr$(39)
    else
      s = s + str$(f) + " kHz"
      if ((f >= DabMIN) and (f <= DabMAX)) then s = s + "(" + str$(Pid(Pfav(IRnum-1))) + "/" + str$(Pcomp(Pfav(IRnum-1))) +")"
    endif
  endif

  if ((IRnum > 0) and (IRnum <= NumPresets)) then
    'Display the candidate preset
    p = Pfreq(IRnum)

    if (f > 0) then s = s + ", "
    s = s + "Preset " + str$(IRnum) + "="

    if (Pname(IRnum) <> "") then
      s = s + chr$(39) + Pname(IRnum-1) + chr$(39)
    else
      s = s + str$(p) + " kHz"
      if ((p >= DabMIN) and (p <= DabMAX)) then s = s + "(" + str$(Pid(IRnum-1)) + "/" + str$(Pcomp(IRnum-1)) +")"
    endif
  endif

  if ((f <= 0) and (p <= 0)) then s = MatchName (IRnum, "", -1, -1, 0)
  CtrlVal(RLN8) = s

  if (f > 0) then
    CtrlVal(RLN9) = "Press Ok to select preset or F for favourite"

  elseif (p > 0) then
    CtrlVal(RLN9) = "Press Ok to select preset"

  else
    CtrlVal(RLN9) = "Press Ok or F to select frequency"
  endif

end sub

'-----------------------------------------------------
'ChooseNext used by IR remote control routine to choose the next or previous preset
'in the list.

sub ChooseNext (x as integer)
  local integer i

  'Check if we are already accumulating buttons. If not, choose starting value
  if ((IRnum <= 0) or (IRnum > NumPresets)) then
    'Not yet accmulating numbers - try and match the currently selected channel
    IRnum = NumPresets >> 1 'Default: Start scanning from the middle index in the presets list

    'See if we can find the station to which the radio is currently tuned
    if (NumPresets > 0) then
      for i = 0 to NumPresets-1
        if ((Frequency = Pfreq(i)) and ((Frequency < DabMIN) or ((ServiceID = Pid(i)) and (CompID = Pcomp(i))))) then exit for
      next

      if (i < NumPresets) then IRnum = i+1
    endif
  endif

  'IRnum holds the current candidate preset. Choose next (or prev) by adding
  '(or subtracting) and loop around if we go past the beginning or end
  IRnum = IRnum + x
  if (IRnum > NumPresets) then IRnum = 1
  if (IRnum <= 0) then IRnum = NumPresets

  DispIRnum
end sub
'- - - - - - - - - - - - - - - - - - - - - - - - - - -
'Read the infra red remote control settings from InfraRed.csv
'
'You can specify different device codes for each command, although they will commonly
'be the same value. Some IRs use different device codes when configured for (eg)
'DVD + VCR or DVD + TV controllers

const Ivolup=0, Ivoldn=1, Imute=2, Ichanup=3, Ichandn=4, Iscanup=5, Iscandn=6
const Iam=7, Ifm=8, Idab=9, I0=10, I1=11, I2=12, I3=13, I4=14, I5=15, I6=16, I7=17
const I8=18, I9=19, Iok=20, IF0=21, IF1=22, IF2=23, IF3=24, IF4=25, IF5=26, Ifavourite=27
const Icancel=28, Ialtcancel=29, Ibackspace=30, I_NUM=31

dim integer Idev(I_NUM-1), Icmd(I_NUM-1)

sub ReadIRCodesFile
  local integer i

  'Now try and open the file containing the IR codes and read it
  S2 = "InfraRed.csv"
  on error ignore
  Open S2 for input As 1

  if (MM.Errno <> 0) then
    WriteErr "SD card or open IR err"

  else 'the file is open, we can now read the codes
    WriteMsg ("Reading " + S2)
    S1 = ""

    'skip all lines up until a line that starts with the characters "'Label"
    'used mid$ below because simple crunching on MAC tricked by single quotes
    Do While ((Not Eof(1)) and (mid$(S1, 2, 6) <> "Label"))
      Input #1, S1, S2, S3

      if (MM.Errno <> 0) then
        WriteErr ("Read error prior to IR data")
        exit do
      endif

    loop

    'the next line should contain the first value (volume up)
    if (MM.Errno = 0) then
      for i = 0 to (I_NUM - 1)
        'read next line, and decode as hex
        Input #1, S1, S2, S3

        if (MM.Errno <> 0) then
          WriteErr ("IR read error at button " + str$(i+1))
          exit for
        endif

        'Convert the number from string to decimal (base 10 or preceded by &H or &O)
        Idev(i) = val (S2)
        Icmd(i) = val (S3)

        if (D > 0) then print str$(i+1) ": Device Code=" S2 ", Cmd=" S3
      next
    endif

  endif

  Close 1
  on error abort

end sub

'- - - - - - - - - - - - - - - - - - - - - - - - - - -
'IR Codes are configured by YOU in the file called "InfraRed.csv"
'
'Use the 'IR Remote Test' function in the setup menu to learn your IR remote
'characteristics and then edit the InfraRed.csv file being careful NOT to change
'the order of rows, or to add new rows!!
'
'On entry to ProcessIR, State=0

sub ProcessIR
  local integer n=0

  if (CapSearch > 0) then 'Ignore the IR during an antcap operation
    n = -2

  '---------------------
  'Volume up and down

  elseif ((IrDev = Idev(Ivolup)) and (IrButton = Icmd(Ivolup))) then
    'Volume UP only works when Headphones plugged in, or speaker output
    'Note: If headphones plugged in, the mode goes to AUDIO_SPKR automatically
    if (AudioOutMode = AUDIO_SPKR) then
      if (CtrlVal(BVolume) < 63) then
        CtrlVal(BVolume) = CtrlVal(BVolume) + 1
        State = 452
      endif

    else
      n = -2
    endif

  elseif ((IrDev = Idev(Ivoldn)) and (IrButton = Icmd(Ivoldn))) then
    'Volume DOWN only works when Headphones plugged in, or speaker output
    'Note: If headphones plugged in, the mode goes to AUDIO_SPKR automatically
    if (AudioOutMode = AUDIO_SPKR) then
      if (CtrlVal(BVolume) > 0) then
        CtrlVal(BVolume) = CtrlVal(BVolume) - 1
        State = 452
      endif

    else
      n = -2
    endif

  '---------------------
  elseif ((IrDev = Idev(Imute)) and (IrButton = Icmd(Imute))) then
    ButtonDn = BMute 'make it look like we touched the LCD

  '---------------------
  'Channel Up and Down

  elseif ((IrDev = Idev(Ichanup)) and (IrButton = Icmd(Ichanup))) then
    ChooseNext (1) 'chooses from the list of presets (then press OK)

  elseif ((IrDev = Idev(Ichandn)) and (IrButton = Icmd(Ichandn))) then
    ChooseNext (-1)

  '---------------------
  'Frequency scan up and down

  elseif ((IrDev = Idev(Iscanup)) and (IrButton = Icmd(Iscanup))) then
    ButtonDn = BSeekUp 'seek up to next station (as if we touched LCD)

  elseif ((IrDev = Idev(Iscandn)) and (IrButton = Icmd(Iscandn))) then
    ButtonDn = BSeekDn 'seek down to next station

  '---------------------
  'Band selection AM/FM/DAB

  elseif ((IrDev = Idev(Iam)) and (IrButton = Icmd(Iam))) then
    ButtonDn = Ram 'make it look like we touched the LCD

  elseif ((IrDev = Idev(Ifm)) and (IrButton = Icmd(Ifm))) then
    ButtonDn = Rfm 'make it look like we touched the LCD

  elseif ((IrDev = Idev(Idab)) and (IrButton = Icmd(Idab))) then
    ButtonDn = Rdab 'make it look like we touched the LCD

  '---------------------
  'The following statements process the number keys and the 'OK' button
  'Used to select a favourite, or a preset, or a frequency

  elseif ((IrDev = Idev(I0)) and (IrButton = Icmd(I0))) then
    n = 0
    goto NEXT_IR_NUM

  elseif ((IrDev = Idev(I1)) and (IrButton = Icmd(I1))) then
    n = 1
    goto NEXT_IR_NUM

  elseif ((IrDev = Idev(I2)) and (IrButton = Icmd(I2))) then
    n = 2
    goto NEXT_IR_NUM

  elseif ((IrDev = Idev(I3)) and (IrButton = Icmd(I3))) then
    n = 3
    goto NEXT_IR_NUM

  elseif ((IrDev = Idev(I4)) and (IrButton = Icmd(I4))) then
    n = 4
    goto NEXT_IR_NUM

  elseif ((IrDev = Idev(I5)) and (IrButton = Icmd(I5))) then
    n = 5
    goto NEXT_IR_NUM

  elseif ((IrDev = Idev(I6)) and (IrButton = Icmd(I6))) then
    n = 6
    goto NEXT_IR_NUM

  elseif ((IrDev = Idev(I7)) and (IrButton = Icmd(I7))) then
    n = 7
    goto NEXT_IR_NUM

  elseif ((IrDev = Idev(I8)) and (IrButton = Icmd(I8))) then
    n = 8
    goto NEXT_IR_NUM

  elseif ((IrDev = Idev(I9)) and (IrButton = Icmd(I9))) then
    n = 9

   NEXT_IR_NUM:
    if (IRnum < 0) then IRnum = 0
    IRnum = (IRnum * 10) + n 'accumulate the most recent digit
    DispIRnum
    n = 1 'short beep

  elseif ((IrDev = Idev(Ibackspace)) and (IrButton = Icmd(Ibackspace)) and (IRnum >= 0)) then
    IRnum = int(IRnum / 10) 'Divide away the previous digit
    DispIRnum
    n = 1 'short beep

  '---------------------
  elseif ((IrDev = Idev(IF0)) and (IrButton = Icmd(IF0)) and (NumFavs > 0)) then
    n = 0
    goto IR_FAV

  elseif ((IrDev = Idev(IF1)) and (IrButton = Icmd(IF1)) and (NumFavs > 1)) then
    n = 1
    goto IR_FAV

  elseif ((IrDev = Idev(IF2)) and (IrButton = Icmd(IF2)) and (NumFavs > 2)) then
    n = 2
    goto IR_FAV

  elseif ((IrDev = Idev(IF3)) and (IrButton = Icmd(IF3)) and (NumFavs > 3)) then
    n = 3
    goto IR_FAV

  elseif ((IrDev = Idev(IF4)) and (IrButton = Icmd(IF4)) and (NumFavs > 4)) then
    n = 4
    goto IR_FAV

  elseif ((IrDev = Idev(IF5)) and (IrButton = Icmd(IF5)) and (NumFavs > 5)) then
    n = 5

   IR_FAV:
    if (D > 0) then print "IR favourite selected: " n+1
    NewFrequency = Pfreq(Pfav(n))
    NewServiceID = Pid(Pfav(n))
    NewCompID = Pcomp(Pfav(n))
    AntCap = Pcap(Pfav(n))
    CtrlVal(Ebandwidth) = Pbw(Pfav(n))
    n = 0
    goto IR_CANCEL

  '---------------------
  'F button - jump to a favourite or a frequency

  elseif ((IrDev = Idev(Ifavourite)) and (IrButton = Icmd(Ifavourite))) then
    if ((IRnum <= 0) or (IRnum > NumFavs)) then goto IR_FREQUENCY
    if (D > 0) then print "IR favourite selected: " IRnum ", " Pname(Pfav(IRnum-1))
    NewFrequency = Pfreq(Pfav(IRnum-1))
    NewServiceID = Pid(Pfav(IRnum-1))
    NewCompID = Pcomp(Pfav(IRnum-1))
    AntCap = Pcap(Pfav(IRnum-1))
    CtrlVal(Ebandwidth) = Pbw(Pfav(IRnum-1))
    n = 0
    goto IR_CANCEL

  '---------------------
  'OK button - jump to a preset or a frequency

  elseif ((IrDev = Idev(Iok)) and (IrButton = Icmd(Iok))) then

   IR_FREQUENCY:
    if ((IRnum > 0) and (IRnum <= NumPresets)) then
      if (D > 0) then print "IR preset selected: " IRnum ", " Pname(IRnum-1)
      NewFrequency = Pfreq(IRnum-1)
      NewServiceID = Pid(IRnum-1)
      NewCompID = Pcomp(IRnum-1)
      AntCap = Pcap(IRnum-1)
      CtrlVal(Ebandwidth) = Pbw(IRnum-1)
      n = 0

    elseif (ValidFreq(IRnum) <> 0) then
      NewFrequency = IRnum

    else
      n = -1 'play a long beep indicating number not recognised

    endif

    goto IR_CANCEL

  '---------------------
  'Cancel button - clear the accumulated digits

  elseif (((IrDev = Idev(Icancel)) and (IrButton = Icmd(Icancel))) or ((IrDev = Idev(Ialtcancel)) and (IrButton = Icmd(Ialtcancel)))) then
    n = 0

   IR_CANCEL:
    CtrlVal(RLN8) = ""
    CtrlVal(RLN9) = ""
    IRnum = -1 'Return the number accumulator to off condition
    IRwait = -1

  '---------------------
  else
    n = -2 'we don't recognise this button or previous error - short beep

  endif

  'All that's left is to provide provide feedback. Buzzer + optional LED + on-board LED
  if (IrBuzz > 0) then
    if (n >= 0) then
      gui beep 50

    elseif (n = -1) then
      gui beep 700

    elseif (n = -2) then
      gui beep 150

    endif
  endif

  if (IrLed > 0) then pulse P_IR_ACK_LED, 20 'optional LED
  pulse P_LED_YELLOW, 20 'on-board LED 'this LED is not optional

  IrDev = 0
  IrButton = 0
  IRwait = 0 'reset the IR wait timer for another period

end sub

'----------------------------------------------------------
dim integer Dvolume=20, Dbacklight=50, Daudio=AUDIO_DIG, Dswap=0, Dgui=0

sub SetDefaultValues

  CtrlVal(BVolume) = Dvolume
  CtrlVal(BLight) = Dbacklight
  Backlight Dbacklight
  CtrlVal(Sswap) = Dswap
  CtrlVal(Sgui) = Dgui

  if (Daudio = AUDIO_DIG) then CtrlVal(Rdigital) = 1 else CtrlVal(Rdigital) = 0
  if (Daudio = AUDIO_RCA) then CtrlVal(Rrca) = 1 else CtrlVal(Rrca) = 0
  if (Daudio = AUDIO_SPKR) then CtrlVal(Rspeaker) = 1 else CtrlVal(Rspeaker) = 0

  if (AudioOutMode <> Daudio) then AudioOutMode = -1 'HW change applied in main loop

  DefineTouch (CtrlVal(Sgui)) 'in case the favourites mode changed

  WriteMsg ("Vol=" + str$(Dvolume) + ", BackLight=" + str$(Dbacklight) + ", Audio Output Mode=" + str$(Daudio))
  WriteMsg ("Swap=" + str$(Dswap) + ", Fav On Status Screen=" + str$(Dgui))
  WriteMsg ("Frequency=" + str$(NewFrequency) + ", Service=" + str$(NewServiceID) + ", Component=" + str$(NewCompID))

end sub

'- - - - - - - - - - - - - - - - - - - - - - - -
'Load the default configuration file (if it can be found)

sub ReadDefaultFile
  local integer i

  'Try loading Config0.csv, and if that doesn't work, try Config1.csv etc
  on error ignore
  S1 = dir$("Config?.csv", FILE)

  if (S1 <> "") then
    for i = 0 to 9
      S2 = "Config" + str$(i) + ".csv"

      Open S2 for input As 1

      if (MM.Errno <> 0) then
        WriteErr ("File open error " + S2)
        continue for
      endif

      'The file exists - we can try and read it as a CSV
      WriteMsg ("Reading " + S2)

      'skip all lines up until a line that starts with the characters "Don't change the order"
      S1 = ""

      Do While ((Not Eof(1)) and (left$(S1, 5) <> "Don" + chr$(39) + "t"))
        Input #1, S1

        if (MM.Errno <> 0) then
          WriteErr ("Read error prior to first record")
          exit do
        endif

      loop

      if (MM.Errno <> 0) then
        Close 1

        WriteErr ("Read error " + S2)
        continue for

      else
        'the sequence of rows is fixed and aligns with the sequence in WriteDefaultFile
	'The first field is a descriptive name for the value (which we ignore here
	'because the order of the rows is fixed)
	'The second field is the value of that variable
        Input #1, S1, Dvolume
        if ((Dvolume < 0) or (Dvolume > 63)) then Dvolume = 20

        Input #1, S1, Dbacklight
        if ((Dbacklight < 10) or (Dbacklight > 100)) then Dbacklight = 50

        Input #1, S1, Daudio
        if ((Daudio < AUDIO_DIG) or (Daudio > AUDIO_SPKR)) then Daudio = AUDIO_DIG
        if ((WM8804 = 0) and (Daudio = AUDIO_DIG)) then Daudio = AUDIO_SPKR

        Input #1, S1, NewFrequency
	if (ValidFreq(NewFrequency)) = 0) then NewFrequency = DefaultChannel

        Input #1, S1, NewServiceID
        Input #1, S1, NewCompID

        if (NewFrequency < DabMIN) then
          NewServiceID = -1
          NewCompID = -1
        endif

        Input #1, S1, Dswap
        if ((Dswap < 0) or (Dswap > 1)) then Dswap = 0

        Input #1, S1, Dgui
        if ((Dgui < 0) or (Dgui > 1)) then Dgui = 0

        if (MM.Errno <> 0) then
          Close 1

          WriteErr ("Unintelligible defaults " + S2)
          continue for
        endif

        Close 1
        SetDefaultValues

        'i holds the version of config file that was loaded
        if (i <> 0) then
          Frequency = NewFrequency
          ServiceID = NewServiceID
          CompID = NewCompID
          WriteDefaultFile
          Frequency = Frequency + 1 'to make it different so state machine gets tripped
        endif

        on error abort
        exit sub
      endif

      Close 1
    next
  endif

  'to reach here, we have not successfully read a configuration file. Set some defaults
  on error abort
  SetDefaultValues
  AudioOutMode = AUDIO_SPKR
  NewFrequency = DefaultChannel

  'Set the flag that will cause a config file to be written. This will at least leave
  'a template that will give some help if the file is to be edited externally
  'Note: accelerated write timeout will trip in a few seconds from now
  Dwait = DEFAULT_WAIT_TIME * 0.9
end sub

'- - - - - - - - - - - - - - - - - - - - - - - -
sub WriteDefaultFile
  local integer audio_out, sid, cid

  audio_out = AUDIO_RCA
  if (CtrlVal(Rdigital) <> 0) then audio_out = AUDIO_DIG
  if (CtrlVal(Rspeaker) <> 0) then audio_out = AUDIO_SPKR

  'if we are in the middle of a DAB tune, then ServiceID and CompID may be -1
  'and the real target will be in NewServiceID and NewCompID
  sid = ServiceID
  cid = CompID

  if ((sid < 0) or (cid < 0)) then
    sid = NewServiceID
    cid = NewCompID
  endif

  if (RotateAndOpen ("Config", ".csv", 1) = 1) then 'returns 0 if error
    'file open, ready to write
    on error ignore
    print #1, chr$(34) "Don" chr$(39) "t change the order of the following rows" chr$(34)
    print #1, chr$(34) "Volume" chr$(34) "," CtrlVal(BVolume)
    print #1, chr$(34) "BackLight" chr$(34) "," CtrlVal(BLight)
    print #1, chr$(34) "0=Dig 1=RCA 2=Spkr" chr$(34) "," audio_out
    print #1, chr$(34) "Frequency (kHz)" chr$(34) "," Frequency
    print #1, chr$(34) "DAB Service ID" chr$(34) "," sid
    print #1, chr$(34) "DAB Component ID" chr$(34) "," cid
    print #1, chr$(34) "Swap Left-Right" chr$(34) "," CtrlVal(Sswap)
    print #1, chr$(34) "Show Fav Buttons" chr$(34) "," CtrlVal(Sgui)
    Close 1
    on error abort

    if (D > 0) then
      print "Wrote config file:"
      print "  Volume=" CtrlVal(BVolume)
      print "  Backlight=" CtrlVal(BLight)
      print "  Audio output mode=" audio_out
      print "  Frequency=" Frequency
      print "  Service ID=" sid
      print "  Component ID=" cid
      print "  Swap analogue outputs=" CtrlVal(Sswap)
      print "  Display Favourite Buttons=" CtrlVal(Sgui)
    endif
  endif

  Dwait = 0 'turn off the timer
end sub

'-----------------------------------------------------
'The quicksort implementation below is hard-coded to use the
'freq, sid, cid and active arrays, and will co-sort them together.
'The values in freq, sid, cid and active are used TOGETHER to determine order
'Look at a wiki on quicksort to see how it works.
'
'I originally coded a bubble sort, and even with 50 elements, was too slow!

dim integer pf, px, pc, pa, plen

'this compare routine uses (a) the record at the given index and (b) a 'pivot' record
'stored in the variables defined above. This is the function that decides whether one
'record comes 'before' another. In this sense, record is the collection of data stored
'in several different arrays. The result of the comparison is as follows:
'  -1 if record before pivot
'  0 if record same as or 'equivalent to' the pivot
'  +1 if record after pivot

function compare (a as integer) as integer
  local integer r=-1 'default result is record BEFORE pivot

  'record invalid and pivot valid?
  if ((Pfreq(a) < AmMIN) and (pf >= AmMIN)) then
    compare = 1
    exit function
  endif

  'pivot invalid and record valid?
  if ((pf < AmMIN) and (Pfreq(a) >= AmMIN)) then
    compare = -1
    exit function
  endif

  'both invalid?
  if ((pf < AmMIN) and (Pfreq(a) < AmMIN)) then
    compare = 0
    exit function
  endif

  'at this point, record and pivot frequencies are valid. Need to formally compare
  if (Pfreq(a) < pf) then goto rec_before_piv
  if (Pfreq(a) > pf) then goto rec_after_piv

  'at this point the frequencies are both the same and both valid
  'Are we doing DAB or not?
  if ((pf >= DabMIN) and (pf <= DabMAX)) then

    if (Pid(a) < px) then goto rec_before_piv
    if (Pid(a) > px) then goto rec_after_piv

    'at this point, both frequencies and station ID are valid
    if (Pcomp(a) < pc) then goto rec_before_piv
    if (Pcomp(a) > pc) then goto rec_after_piv
  endif

  'at this point, the frequencies, station ID and component IDs are identical
  'is one name shorter than the other?
  if (len(Pname(a)) < plen) then goto rec_before_piv
  if (len(Pname(a)) > plen) then goto rec_after_piv

  r = 0 'treat the two records as identical
  goto rec_before_piv

rec_after_piv:
  r = 1

rec_before_piv:
  'The result of comparing frequency, station ID and component ID is in r.
  'Now compare 'active' flags. The dormant goes to the back of the sorted list
  'Bit2=Favourite|Bit1=Manual Name|Bit0=Preset Active

  if ((Pflags(a) and 1) > pa)) then
    r = -1
  elseif ((Pflags(a) and 1) < pa)) then
    r = 1
  endif

  'the result of the comparison is in r
  compare = r

end function
'- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
'And here comes the sorting algorithm itself. The first call specifies the lowest
'index (which is always 0) and the highest index (which is NumPresets-1)

sub quicksort (lower as integer, upper as integer)
  local integer i, j, k, pivot
  local string s

  'i, j and pivot are indices into the arrays, not values themselves!
  i = lower
  j = upper

  'grab and remember the pivot value, as it may change as we shuffle below
  k = (lower + upper) >> 1 'set pivot in the middle of lower and upper
  pf = Pfreq(k)
  px = Pid(k)
  pc = Pcomp(k)
  pa = Pflags(k) and 1 'Bit2=Favourite|Bit1=Manual Name|Bit0=Preset Active
  plen = len (Pname(k))

  do while (i <= j)
    do while (compare(i) < 0) 'scan up from bottom until finding element not in sequence
      i = i + 1
    loop

    do while (compare(j) > 0) 'scan down from top until finding element not in sequence
      j = j - 1
    loop

    'remember that the i and j values are on either side of the pivot.
    'If not the same point, we know they are in reverse sequence and need to be swapped
    if (i < j) then
      k = Pfreq(i)
      Pfreq(i) = Pfreq(j)
      Pfreq(j) = k

      k = Pid(i)
      Pid(i) = Pid(j)
      Pid(j) = k

      k = Pcomp(i)
      Pcomp(i) = Pcomp(j)
      Pcomp(j) = k

      k = Pflags(i)
      Pflags(i) = Pflags(j)
      Pflags(j) = k

      k = Pcap(i)
      Pcap(i) = Pcap(j)
      Pcap(j) = k

      k = Pbw(i)
      Pbw(i) = Pbw(j)
      Pbw(j) = k

      s = Pname(i)
      Pname(i) = Pname(j)
      Pname(j) = s
    endif

    if (i <= j) then
      i = i + 1
      j = j - 1
    endif
  loop

  'recursively sort the two reduced segments on either side of pivot
  if (lower < j) then quicksort (lower, j)
  if (i < upper) then quicksort (i, upper)
end sub

'-----------------------------------------------------
sub PrintPreset (n as integer)

  print n+1 ": Freq=" Pfreq(n) " Name=" chr$(39) Pname(n) chr$(39) " ID=" Pid(n) " Comp=" Pcomp(n);
  print " AntCap=" Pcap(n) " Flags=" Pflags(n) " BW=" Pbw(n)

end sub
'- - - - - - - - - - - - - - - - - - - - - - - -
'Consolidate preset data currently in memory (eliminating duplicates)
'and sort it alphabetically. This should be done prior to writing
'the presets file, so that the data is ALWAYS in order and compact

sub CompactPresets
  local integer i, j, diff

  'If 0 or 1 preset entry, there is nothing to sort. But check if its a favourite
  if ((NumPresets < 2) or (DirtyPresets = 0)) then exit sub

  if (D > 0) then print "Sorting Presets"
  quicksort (0, NumPresets-1)
  DirtyPresets = 0 ' clear the dirty flag

  if (D > 0) then
    print "Sorted..."

    for j = 0 to (NumPresets - 1)
      PrintPreset (j)
    next
  endif

  'null duplicates (actually, overwrite them. They will be adjacent to each other
  'Accumulate the count of actives in i
  NumActivePresets = 0
  i = 0 'references the preset we are comparing against (looking for duplicates)

  for j = 1 to (NumPresets - 1)
    'Ignore entries with invalid frequency
    if ((Pfreq(j) < AmMIN) or (Pfreq(j) > DabMAX)) then continue for

    'For AM and FM, only compare frequency. For DAB, also compare id and component
    diff = 0
    if (Pfreq(i) <> Pfreq(j)) then diff = 1
    if ((diff = 0) and (Pfreq(i) >= DabMIN) and ((Pid(i) <> Pid(j)) or (Pcomp(i) <> Pcomp(j)))) then diff = 1

    if (diff <> 0) then
      'The record at j is different than the record at i.
      'We want to keep j, so inc running count, then copy over any gap that formed
      i = i + 1

      'if we have already skipped ('nulled') a record, copy the record at j
      if (i < j) then
        Pfreq(i) = Pfreq(j)
        Pid(i) = Pid(j)
        Pcomp(i) = Pcomp(j)
        Pcap(i) = Pcap(j)
	Pbw(i) = Pbw(j)
        Pflags(i) = Pflags(j)
        Pname(i) = Pname(j)
      endif

      if ((Pflags(i) and 1) <> 0) then NumActivePresets = i + 1

    else
      'the record at j is the same as the record at i.
      'We want to remove j - this will create a gap
      if (D > 0) then
        print "Record " j " is the same as record at " i ". Deleting " j
        print " Keeping " Pfreq(i) ", " Pname(i) ", " Pflags(i)
        print " Discarding " Pfreq(j) ", " Pname(j) ", " Pflags(j)
      endif

      'If the second record is active but first is not, copy attributes
      'Pflags: Bit2=Favourite|Bit1=Manual Name|Bit0=Preset Active
      if ((Pflags(i) and 1) = 0) then Pflags(i) = Pflags(i) or (Pflags(j) and 1)
      if ((Pflags(i) and 4) = 0) then Pflags(i) = Pflags(i) or (Pflags(j) and 4)

      'If first name not set, then copy the name from the second
      if (Pname(i) = "") then
        Pname(i) = Pname(j)
	Pflags(i) = Pflags(i) or (Pflags(j) and 2)

      elseif (((Pflags(i) and 2) = 0) and ((Pflags(j) and 2) <> 0)) then
        Pname(i) = Pname(j)
	Pflags(i) = Pflags(i) or 2 'set manual station name bit because it is set in (j)

      endif

      Pwait = DEFAULT_WAIT_TIME * .8 'accelerated write time
    endif
  next

  'at this point, i points to the last valid entry in presets list
  NumPresets = i + 1

  'blank the remainder
  for i = NumPresets to MAX_PRESETS-1
    Pname(i) = ""
    Pfreq(i) = 0
    Pid(i) = -1
    Pcomp(i) = -1
    Pcap(i) = 0
    Pbw(i) = -1
    Pflags(i) = 0 'Bit2=Favourite|Bit1=Manual Name|Bit0=Preset Active
  next

  if (D > 1) then print "Compacted. #Presets=" NumPresets ", #Active Presets=" NumActivePresets : print
end sub

'-----------------------------------------------------
'WritePresetFile - saves the presets currently in memory out to a new preset file

sub WritePresetFile
  local integer i, bw, ac, id, comp

  'Check if we need to compact the preset data before writing out
  CompactPresets

  if (RotateAndOpen ("Preset", ".csv", 1) = 1) then 'returns 0 if error
    'file open, ready to write
    on error ignore
    print #1, "Band,Frequency (kHz),Station Name,DAB Service ID,DAB Component ID,AntCap,Bit2=Favourite|Bit1=Manually Specified Name|Bit0=Preset Active,AM Audio Bandwidth (Hz)"

    for i = 0 to NumPresets-1
      bw = 0
      ac = 0
      id = -1
      comp = -1

      if ((Pfreq(i) >= AmMIN) and (Pfreq(i) <= AmMAX)) then
        S1 = "AM,"
        bw = Pbw(i) 'Only accept the bandwidth argument for AM frequencies

      elseif ((Pfreq(i) >= FmMIN) and (Pfreq(i) <= FmMAX)) then
        S1 = "FM,"
        ac = Pcap(i) 'only accept antcap argument for FM/DAB

      elseif ((Pfreq(i) >= DabMIN) and (Pfreq(i) <= DabMAX)) then
        S1 = "DAB,"
        ac = Pcap(i) 'only accept antcap argument for FM/DAB
        id = Pid(i) 'only accept id and comp for DAB
        comp = Pcomp(i)

      else
        continue for 'don't write entries with an invalid frequency

      endif

      S1 = S1 + str$(Pfreq(i)) + "," + chr$(34) + Pname(i) + chr$(34)
      S1 = S1 + "," + str$(id) + "," + str$(comp) + "," + str$(ac)
      S1 = S1 + "," + str$(Pflags(i)) + "," + str$(bw)

      print #1, S1
      if (D > 0) then PrintPreset (i)
    next

    Close 1
    on error abort
    WriteMsg ("Wrote Preset0.csv")
  endif

  Pwait = 0 'Stop the preset file write timer
end sub

'-----------------------------------------------------
sub SetFavs
  local integer i

  NumFavs = 0 'Set favourites to zero, it will be recalculated as we load

  'Determine which of the presets are favourites
  for i = 0 to (NumPresets - 1)
    if ((NumFavs < MAX_FAVS) and ((Pflags(i) and 4) <> 0)) then
      Pfav(NumFavs) = i
      NumFavs = NumFavs + 1
    endif
  next

end sub

'-----------------------------------------------------
'Load the most recent preset file (if it can be found)
'and then determine which of the presets are favourites

sub ReadPresetFile
  local integer i

  'write new preset file before reading, or we will lose our changes!
  if (Pwait > 0) then WritePresetFile 'write presets (if pending) before reboot

  NumPresets = 0
  on error ignore

  'Try loading Preset0.csv, and if that doesn't work, try Preset1.csv etc
  S1 = dir$("Preset?.csv", FILE)

  if (S1 <> "") then
    for i = 0 to 9
      NumPresets = 0
      S2 = "Preset" + str$(i) + ".csv"

      Open S2 for input As 1

      if (MM.Errno <> 0) then
        WriteErr ("Open error " + S2)
        continue for
      endif

      'The file exists - we can try and read it as a CSV
      WriteMsg ("Reading " + S2)

      'Discard the first line of the file - this only contains column headings
      Input #1, S1

      if (MM.Errno <> 0) then
        Close 1

        WriteErr ("Read error " + S2)
        continue for
      endif

      'read each line of the CSV file until end of file
      Do While ((Not Eof(1)) and (NumPresets < MAX_PRESETS))
        Input #1, S1, Pfreq(NumPresets), S3, Pid(NumPresets), Pcomp(NumPresets), Pcap(NumPresets), Pflags(NumPresets), Pbw(NumPresets)

        'MMBASIC bug? - works ok unix format text files, but not DOS! (reads extra line!)
        if (S1 = "") then exit do

	if (ValidFreq(Pfreq(NumPresets)) = 0) then
          WriteMsg (S2 + " entry " + str$(NumPresets+1) + " " + S3 + " BadFreq=" + str$(Pfreq(NumPresets)) )
          continue do
        endif

        if (Pfreq(NumPresets) > AmMAX) then
          Pbw(NumPresets) = -1

        elseif (Pbw(NumPresets) < BWmin) or (Pbw(NumPresets) > BWmax)) then
          Pbw(NumPresets) = BWdef

        endif

	Pname(NumPresets) = left$(S3, NameLen) 'in case user has added long name externally
	Pflags(NumPresets) = Pflags(NumPresets) and 7 'keep only bottom three bits
        if (D > 0) then PrintPreset (NumPresets)
        NumPresets = NumPresets + 1
      Loop

      Close 1
      WriteMsg ("(Read " + str$(NumPresets) + " presets)")

      'set the dirty flag - unknown state of presets
      'commented out - this speeds things up, and only useful when end user
      'edits outside of radio software. Will be caught at next rewrite anyway
      'DirtyPresets = 1

      if (NumPresets > 0) then
        'Write back what we just read as Preset0.csv - because we didn't find Preset0.csv
        if (i > 0) then WritePresetFile
        exit for
      endif
    next
  endif

  on error abort
end sub

'-----------------------------------------------------
'SI4689 radio chip access routines - these are rather specific to the hardware
'Will not run properly without working hardware connected to the micromite plus.
'But if you don't have a working DAB+ radio board connected, you can set the
'constant SA (near top of the file) to be non-zero, and the code will kind-of
'work OK for the purpose of GUI testing or checking for run-time errors etc.
'The state machine below follows the data-sheet flowchart reasonably closely.

dim integer SpiBuf(READ_REPLY_BUFSIZE) 'used as a SPI read buffer

'Read the SI4689 radio chip SPI reply
sub Read_Radio_Reply (len as integer)
  if (len > READ_REPLY_BUFSIZE) then len = READ_REPLY_BUFSIZE

  'CMD=0x00 (RD_REPLY)
  pin(P_SI4689_CS) = 0
  spi write 1,0
  Pause .1 'wait 100us - this should be the ONLY pause statement in programme!
  spi read len, SpiBuf()
  pin(P_SI4689_CS) = 1

end sub

'- - - - - - - - - - - - -
'Put radio into firmware loading mode

sub Issue_LOAD_INIT

  'CMD=0x06 (LOAD_INIT)
  'ARG1=0 (Fixed - not configurable)
  pin(P_SI4689_CS) = 0
  spi write 2,&H06,&H00
  pin(P_SI4689_CS) = 1
  if (D > 2) then print "*06,00 Load Init"

end sub

'- - - - - - - - - - - - -
'Set a radio property
'The values are passed as LSB/MSB pairs to avoid having to recalculate in real time
sub SetProperty(mp AS integer, lp AS integer, mv AS integer, lv AS integer)

  'CMD=0x13 (SET_PROPERTY)
  'ARG1=00 (Fixed - not configurable)
  'ARG2-3 (Little Endian LSB first. property)
  'ARG4-5 (Little Endian LSB first. value)
  pin(P_SI4689_CS) = 0
  spi write 6,&H13,&H00,lp,mp,lv,mv
  pin(P_SI4689_CS) = 1

  if (D > 2) then print "*13,00 Property " hex$(mp,2) hex$(lp,2) "=" hex$(mv,2) hex$(lv,2)
  CTSwait = 9
  SpiWait = 1 'Don't need to wait for next tick to continue - property commands can chain

end sub

'- - - - - - - - - - - - -
'Tell the built-in DAC on the radio chip to enter or exit mute depending on
'current state of HW + SW mutes

sub SetRadioMute
  local integer m=0

  '0=unmute, 3=mute both channels (left and right)
  if (Muted <> 0) then m = 3

  'property=0x0301 (AUDIO_MUTE)
  SetProperty(&h03, &h01, 0, m)

end sub

'-----------------------------------------------------
'Convert non printable ascii values to a space and return as a one character string

function CleanChr (c as integer) as string
  if ((c < 32) or (c > 127)) then
    CleanChr = " "
  else
    CleanChr = chr$(c)
  endif
end function

'-----------------------------------------------------
'Remove leading and trailing spaces from string

function TruncStr (s as string) as string
  local integer i, n, f, l

  TruncStr = s
  n = len(s)
  if (n = 0) then exit function

  'find the first and last non-whitespace character
  f = 0
  l = 0

  for i = 1 to n
    if (mid$(s, i, 1) <> " ") then
      if (f = 0) then f = i 'remember the first non-white character
      l = i 'remember the last non-white character
    endif
  next

  'If the entire string is space, then f will be zero
  if (f = 0) then
    TruncStr = ""
    exit function
  endif

  'truncate string between our two markers
  TruncStr = mid$(s, f, l-f+1)

end function

'-----------------------------------------------------
'Take a (potentially) long string, and display on requested lines
'(note - calling string s is modified by this routine!)

function DispString (s as string, first as integer, last as integer, lx as integer, blank as integer) as integer
  local integer i, j, l=first
  local s3 as string

  'The LCD display is exactly 50 characters wide
  'If the string is longer, will need to break over several lines
  i = len(s)

  if ((D > 0) and (i > 0)) then print "Displaying string:" s

  do while ((i > 0) and (l <= last))
    if (i <= lx) then
      s3 = ""

    else 'i > lx
      'Need to cut at a word boundary
      for j = lx to 1 step -1
        if (mid$(s, j, 1) = " ") then exit for
      next

      if (j > 1) then
        'We found whitespace
        s3 = right$(s, i-j)
        s = left$(s, j-1)
      else
        'Rare case of no whitespace - just cut ungracefully
        s3 = right$(s, i-lx)
        s = left$(s, lx)
      endif
    endif

    if (CtrlVal(l) <> s) then CtrlVal(l) = s
    s = s3
    i = len(s)
    l = l + 1
  loop

  DispString = l

  if (blank <> 0) then
    'blank remaining lines
    do while (l <= last)
      CtrlVal(l) = ""
      l = l + 1
    loop
  endif

end function

'-----------------------------------------------------
'If there is a complete RDS text string, display the string
'The parameter will be:
' 0 = don't blank anything
' 1 = blank section 1
' 17= blank section 2

sub TryToDispRDS (blank as integer)
  local integer n

  S1 = ""
  S2 = ""

  'Accumulate the two RDS strings in S1 and S2
  for n = 0 to 15
    S1 = S1 + ProgData(n)
    S2 = S2 + ProgData(16+n)

    if (blank <> 0) then ProgData(blank-1+n) = ""
  next

  if (len(S1) = 64) then S1 = TruncStr (S1) else S1 = ""
  if (len(S2) = 64) then S2 = TruncStr (S2) else S2 = ""

  if ((S1 = "") and (S2 = "")) then exit sub 'nothing to display

  'at least one of the strings has something to display
  if (S1 = "") then
    n = DispString (S2, RLN4, RLN7, 50, 1) 'only S2 is non-null
    exit sub
  endif

  'S1 is not null - are its contents the same as s2?
  if ((S2 <> "") and (S1 <> S2)) then S1 = S1 + " / " + S2

  'Finally, the complete text to display is in S1
  n = DispString (S1, RLN4, RLN7, 50, 1)

end sub

'-----------------------------------------------------
'AppendErrFile appends another error to the error log.
'The error log contains the following columns
' 1: Log date
' 2: Log time
' 3: MMBASIC Uptime (in seconds)
' 4: Error type (the subroutine call parameter)
' 5: State machine State
' 6: Radio poll status bytes [0][1][3]
' 7: Current Frequency
' 8: Current Service ID
' 9: Current Component ID
'
'The string parameter in subroutine call contains columns
'Error Type,

sub AppendErrFile (err as string)

  S2 = err + "," + str$(State) + "," + hex$(SpiBuf(0),2) + hex$(SpiBuf(1),2) + hex$(SpiBuf(3),2)
  WriteErr (S2)

  on error ignore
  open "Error0.csv" for append as #1
  print #1, DATE$ "," TIME$ "," str$(UpTime/200,0,1) "," S2 "," str$(Frequency) "," hex$(ServiceID) "," hex$(CompID)

  close #1

  on error abort
end sub

'-----------------------------------------------------
'convert a signed byte or 16 bit word into a signed integer

function Conv8bitSignedInt (x as integer) as integer
  if (x > 127) then
    Conv8bitSignedInt = x - 256
  else
    Conv8bitSignedInt = x
  endif
end function

'-----------------------------------------------------
'ChooseAudioOutMode determines which interface the sound is going to
'come out of. The parameter is a dummy parameter. MMBASIC has trouble
'understanding that you might want to define a function with NO parameters
'(ie MMBASIC bug)

function ChooseAudioOutMode (x as integer) as integer

  'pin(P_HEADPHONE)=0 means headphones are plugged in
  if ((pin(P_HEADPHONE) = 0) or (CtrlVal(Rspeaker) <> 0)) then
    ChooseAudioOutMode = AUDIO_SPKR

  elseif ((WM8804 > 0) and (CtrlVal(Rdigital) <> 0)) then
    ChooseAudioOutMode = AUDIO_DIG

  else
    ChooseAudioOutMode = AUDIO_RCA

  endif
end function

'-----------------------------------------------------
'State is the radio SPI state machine value. When State=0, the state machine is idle
'Rstate is the tuning status of the radio chip.

dim integer State=1, Rstate=Runtuned, Address, Reply_Size=4, StatusBits=0, AntCap=0
dim integer ReadAntCap, Radio_Running=-1, NewStatus=0, ExpectTune=0, NumProp
dim integer PropID, IntMask=0, CTSwait=0, LastDabServList=-1, AudioOut=0
dim integer DabServiceID(MAX_DAB_SERVICES-1), DabComponentID(MAX_DAB_SERVICES-1)
dim integer ReadFreq, SNR, CNR, M, FibErr, Quality, SigStrength, FreqOffset, Stereo
dim integer NumServices, Min_SNR, Min_SigStren, DispRefresh=0

dim string Status length 11, Band length 3, DabServiceName(MAX_DAB_SERVICES-1) length NameLen

dim float SigLevel, maSNR, maCNR, maSig, maQual

Band=""

'- - - - - - - - - - - - - - - - - - - - - - - -
'Create a new preset entry, merging with existing entry if there's already a similar
'entry. Returns non-zero if a change was made to the preset table

function AddPreset (freq as integer, defname as string, sid as integer, cid as integer, ac as integer) as integer
  local integer i, j

  AddPreset = 0

  'Scan the existing preset entries to see if there is already a matching entry
  'The two checks for Pid and Pcomp not equalling -1 is so that if we get
  'a new DAB channel that has a real Pid and Pcomp (as opposed to the placeholder
  'values of -1), then the new DAB channel will overwrite the placeholders
  for i = 0 to (NumPresets - 1)
    if ((sid <> Pid(i)) and (Pid(i) >= 0)) then continue for
    if ((cid <> Pcomp(i)) and (Pcomp(i) >= 0)) then continue for
    if (Pfreq(i) = freq) then exit for
  next

  'Did we find a match on the basis of frequency / SID / CID?
  if (i < NumPresets) then 'yes

    'If no antcap value specified, try and find another entry in the preset table
    'for the same frequency that has an antcap value
    if (ac = 0) then
      for j=0 to (NumPresets-1)
        if ((Pfreq(j) = freq) and (Pcap(j) > 0)) then
          ac = Pcap(j)
	  exit for
	endif
      next
    endif

    'Set the Pid and Pcomp values - they may be the same as what's there anyway
    'this will overwrite -1, if that's what we happened to find
    if ((Pid(i) <> sid) or (Pcomp(i) <> cid)) then
      Pid(i) = sid
      Pcomp(i) = cid
      Pname(i) = ""
      AddPreset = 1
    endif

    if ((Pname(i) = "") and (defname <> "")) then
      Pname(i) = defname
      AddPreset = 1
    endif

    'If the current AntCap value is default (auto), then set to the supplied value
    if (Pcap(i) <> ac) then
      Pcap(i) = ac
      AddPreset = 1
    endif

    if (AddPreset > 0) then
      Pwait = 1
      Pflags(i) = 1 'Bit2=Favourite|Bit1=Manual Name|Bit0=Preset Active
      DirtyPresets = 1 ' set the dirty flag
    endif

    exit function
  endif

  'at this point, we didn't find a matching entry for the preset
  if (NumPresets >= MAX_PRESETS) then
    WriteErr ("Preset Table Full")
    exit function 'no room for more
  endif

  'No match. Create a new entry in the table
  Pname(i) = defname
  Pfreq(i) = freq
  Pid(i) = sid
  Pcomp(i) = cid
  Pcap(i) = ac
  Pbw(i) = BWdef
  Pflags(i) = 1 'Bit2=Favourite|Bit1=Manual Name|Bit0=Preset Active
  NumPresets = NumPresets + 1
  AddPreset = 1
end function

'- - - - - - - - - - - - - - - - - - - - - - - -
Sub CheckSpi
  local integer next_state, i, a, b, c, e, n, id, comp
  static integer IntState=0

  'Reset SpiWait to zero means we will wait until the next tick before returning
  SpiWait=0
  '- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  'Check for pending frequency-change request (NewFrequency nonzero)
  if (ValidFreq(NewFrequency)) then
    i = RadioMode

    'We are changing frequency, but do we need to change band (load new FW) as well?
    if (NewFrequency <= AmMAX) then
      i = MODE_AM
      Min_SNR = AmSNR
      Min_SigStren = AmMinStren
      ServiceID = -1
      CompID = -1
      NewServiceID = -1
      NewCompID = -1

    elseif ((NewFrequency >= FmMIN) and (NewFrequency <= FmMAX)) then
      i = MODE_FM
      Min_SNR = FmSNR
      Min_SigStren = FmMinStren
      ServiceID = -1
      CompID = -1
      NewServiceID = -1
      NewCompID = -1

    elseif ((NewFrequency >= DabMIN) and (NewFrequency <= DabMAX)) then
      i = MODE_DAB
      Min_SNR = DabCNR
      Min_SigStren = DabMinStren

    else 'SW bug
      AppendErrFile ("Reboot:Can" + chr$(39) + "t scan to freq " + str$(NewFrequency))

      if (D = 0) then
        Reboot = 1 'almost instant reboot
      else
        Reboot = 2000 '10 seconds and then reboot
      endif

      exit sub
    endif

    'If a band change required, cold-reset the radio and then load the firmware for
    'the requested band
    if ((i >= 0) and (i <> RadioMode)) then 'band change required
      State = 1 'load the new band's firmware immediately
      RadioMode = i
      SetRadioModeColour
      goto BLANKIT

    elseif (State = 0) then 'Band is OK. But only do frequency change from state 0
      State = 300 'Set a new frequency in the current operating band
      ServiceID = -1 'NewServiceID and NewCompID, will be used when tuned
      CompID = -1
      LastDabServList = -1

    BLANKIT:
      ActiveProgData = -1
      IntMask = 0
      Frequency = NewFrequency
      NewFrequency = -1
      Rstate = Runtuned
      maSNR = -99
      maCNR = -99
      maSig = -1
      maQual = -1
      AudioOut = 0
      RdsNames=0

      'Clear the FM RDS strings
      for n = 0 to 3
        ProgName(n) = ""
      next

      for n = 0 to 31
        ProgData(n) = ""
      next

      for n = 0 to (MAX_SET-1)
        RdsNameSet(n) = ""
      next

      'Check if there is an AntCap value in the presets file for the new frequency
      'If not, set AntCap to 0, which tells the Si4689 to derive AntCap from programmed
      'straight line formula.
      AntCap = 0

      for n = 0 to (NumPresets-1)
        if (Frequency = Pfreq(n)) then
          AntCap = Pcap(n)
          exit for
        endif
      next

      ClearStatus

      'The following is a debugging hack for 'stand alone mode'
      if ((NewServiceID >= 0) and (SA <> 0)) then
        ServiceID = NewServiceID
        CompID = NewCompID
        NewServiceID = -1
        NewCompID = -1
      endif
    endif
  endif

  '- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  'Read the radio status every timer tick, and check for interrupts
  if ((Radio_Running > 0) and (SA = 0)) then

    if (Reply_Size < 4) THEN Reply_Size=4
    Read_Radio_Reply(Reply_Size)

    'Response:
    'byte 0: 0x80=CTS, 0x40=Cmd Error
    '        0x10=Digital Service (DAB) Interrupt
    '        0x04=RDS (FM) Interrupt
    '        0x01=Seek/Tune Complete Interrupt
    'byte 1: 0x20=Digital Service Event Interrupt
    'byte 2: don't care (ignore this byte)
    'byte 3: bits 7,6=system state, bits 5-0 are various error indications that if
    '        non-zero are fatal
    '        Will read as &HFF when there is nothing connected

    'If no radio hardware detected, then SPI read will return 255
    if ((State < 20) and (SpiBuf(0) = 255)) then
      S1 = "SPI com err"
      goto DO_REBOOT
    endif

    'Check for cmd error - basically when you send a command that the Si4689 doesn't
    'recognise in the current mode, or you send a command with a parameter that is
    'beyond the allowable limits - ie might be a SW bug ;-)
    'But I have a strong feeling that not all cmd errors are bugs, or serious bugs.
    'eg some cmd errors appear to happen due to timing. For example, if the state
    'machine is in the process of changing frequency, and a message such as an FM RDS
    'message comes in related to the *previous* frequency (eg a timing thing between
    'the moment of change, and clearing out the stuff that the radio chip had queued
    'up), this can cause the state machine to get a little ahead of itself. I guess
    'this is a bug. I am not sure how to avoid this - because few radio responses tell
    'you what frequency or channel the response relates to - you can't tell if they're
    'stale.
    if ((SpiBuf(0) and 64) <> 0) then
      if ((Reboot <= 0) and (D > 0)) then
        S1 = "Non fatal:Cmd err"

        if (Reply_Size < 6) then Read_Radio_Reply(6) 'Get the error code
        S1 = S1 + "[4]=" + hex$(SpiBuf(4),2) + " [5]=" + hex$(SpiBuf(5),2)
        AppendErrFile (S1)
      endif

      'In order to clear the CMD error, a command needs to be successfully sent
      'Send a benign command (GET_SYS_STATE) to achieve this
      'CMD=0x09 (GET_SYS_STATE)
      'ARG1=00 (Fixed - not configurable)
      pin(P_SI4689_CS) = 0
      spi write 2,&H09,&H00
      pin(P_SI4689_CS) = 1
      if (D > 2) then print "*09,00 Get Sys State - clear error"
      CTSwait = 9
      SpiWait = 1
      exit sub
    endif

    'Check for Si4689 'fatal' error. (Fatal errors require reboot of radio chip)
    if ((SpiBuf(3) and 63) <> 0) then
      'Si4689 fatal errors are described in AN649
      '0x01 = Non Recoverable Error. The Si4689 internal keep alive (watchdog) expired
      '0x02 = Si4689 arbiter error - its internal to radio chip, whatever this is
      '0x04 = Control interface dropped data during write. SPI clock rate too high.
      '0x08 = Control interface dropped data during read. SPI clock rate too high.
      '0x10 = Si4689 DSP frame overrun error
      '0x20 = RF Front end of Si4689 is in an unexpected state. Internal to the chip.
      S1 = "fatal err"
      goto DO_REBOOT
    endif

    'Check for Interrupts we are interested in. At present, only FM and DAB bands
    'configured for interrupts
    if (RadioMode = MODE_DAB) then
      if ((SpiBuf(0) and 16) <> 0) then IntMask=IntMask or 1 'Digital Service (DAB) Int
      if ((SpiBuf(1) and 32) <> 0) then IntMask=IntMask or 4 'Digital Event (DAB) Int

    elseif (RadioMode = MODE_FM) then
      if ((SpiBuf(0) and 4) <> 0) then IntMask=IntMask or 2 'RDS (FM) interrupt

    endif

    if (IntMask <> 0) then
      if (D > 0) then
        print "Int:" State "," hex$(SpiBuf(0),2) hex$(SpiBuf(1),2) hex$(SpiBuf(3),2)
      endif

      IntState = 60000 'this is the interrupt service entry state
    endif
  endif

  '- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  'If we are waiting on a CTS, has the radio signalled clear to send yet?
  'Only check if the CTS downcounter is running
  '
  'The CTS errors are sometimes caused by not giving the radio enough time to complete
  'a command (like boot, scan, and things that take a while). In this case, the status
  'bits will be non-zero and they will tell you something about what the problem might
  'be. In these cases, if the radio was already tuned to a station and playing, it will
  'continue to play even though the error has been thrown. These situations are probably
  'due to a bug in this BASIC programme.
  '
  'I have also encountered CTS errors that return all zeros below, and at the same time
  'the radio chip is locked up and outputting a high pitched tone, or noise, or something
  'else that strongly suggests the radio chip firmware has crashed for some reason. These
  'don't appear to be caused by this programme, but by something else environmental.
  'I have noticed this kind of problem when turning the radio off (the instant that the
  'mains power switch breaks the mains voltage) and when my gas heater igniter or gas
  'stove igniter or washing machine turns on (all obviously wideband RF noise or wideband
  'mains voltage noise). The radio chip seems to be prone to lock up with noise.
  '
  'To reduce the chance that mains noise was conducted via the DC power supply, I built
  'a rectified transformer fed DC supply. A bridge rectifier is filtered by 2000uF, which
  'in turn feeds an additional 44,000uF of filtering via 2R7, all caps low ESR. I have
  'liberally clamped several high inductance ferrites on the mains feed+rectified DC feed
  'I don't expect much of the above noise is being conduted in via the DC feed. I have
  'gone to some degree of trouble with the design and layout of the DC supply. I should
  'never say never regarding having built a perfect power supply, however I am a
  'professional engineer with almost 40 years experience, and I was happy with this PSU.
  'My suspicion is that the noise is getting into the radio via the antenna, not PSU.
  'It may be because I live in a marginal reception area, there is some susceptibility
  'to noise with low signal or low SNR/CNR levels?

  if ((CTSwait > 0) and (SA = 0)) then
    'Keep waiting until the CTS bit has been set, and the status bits are set (if the
    'status bits are specified)
    if (((SpiBuf(0) and 128) = 128) and ((SpiBuf(0) and StatusBits) = StatusBits)) then
      CTSwait = 0
      StatusBits = 0

    else 'conditions for continuing are not yet satisfied
      CTSwait = CTSwait - 1

      if (CTSwait = 0) then
        S1 = "CTS timeout"

        DO_REBOOT:
        if (Reboot <= 0) then
          SetMux (1, -1) 'mute the 4052
          ResetWM8804
          AppendErrFile ("Reboot:" + S1)

          if (D = 0) then
            Reboot = 1 'almost instant reboot
          else
            Reboot = 2000 '10 seconds and then reboot
          endif
        endif
      endif

      exit sub 'don't proceed with state machine because we are aborting, or waiting
    endif
  endif

  '- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  'We can process interrupts when state returns to 0, otherwise continue with
  'current flow
  if (State = 0) then
    'Return if State is 0 and there are no interrupts to process
    if (IntState = 0) then exit sub
    State = IntState
  endif

  '- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  'To get here, there is something in the state machine that needs attention
  'Preemptively guess the next state so we don't have to repeat every case statement
  next_state = State + 1

  if (D > 1) then print "State:" State " IntState:" IntState " IntMask:" hex$(IntMask,2) " Freq: " Frequency " SID: " ServiceID " NewSID: " NewServiceID
  Reply_Size = 4 'Set back to the default value after reading the reply above

  '- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  'When State=0, the state machine is idle
  select case State
    case 1
      'THIS IS THE COLD/WARM BOOT ENTRY
      'RESET THE RADIO CHIP and move into "POWER UP" mode
      SetMux (1, -1) 'apply HW mute the 4052, and maintain current SW mute
      ResetWM8804
      AudioOutMode = -1 'The correct mode set after WM8804 checked at State 11

      'Refer to SI4689 application note 649: eg s5.1 p381
      pin(P_SI4689_CS) = 1
      pin(P_SI4689_RES) = 1

    case 2
      pin(P_SI4689_RES) = 0

    case 3
      pin(P_SI4689_RES) = 1

    case 4
      'CMD=0x01 (Power_UP)
      'ARG1=00 (No interrupt on CTS)
      'ARG2=17 (OSC/Buffer=Xtal, OSX size=7)
      'ARG3=20 (Xtal startup bias=320uA)
      'ARG4-7=00 F8 24 01 (Little Endian LSB first. Xtal Freq=19200000Hz)
      'ARG8=10 (CTUN?)
      'ARG9-12=10 00 00 00 (Fixed - not configurable)
      'ARG13=10 (Xtal run bias=160uA)
      'ARG14-15=00 00 (Fixed - not configurable)
      pin(P_SI4689_CS) = 0
      spi write 16,&H01,&H00,&H17,&H20,&H00,&HF8,&H24,&H01,&H10,&H10,&H00,&H00,&H00,&H10,&H00,&H00
      pin(P_SI4689_CS) = 1
      if (D > 2) then print "*01,00,17,20,00,f8,24,01,10,10,00,00,00,10,00,00 Power Up"
      Radio_Running=1
      CTSwait = 9

    '- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
    'LOAD BOOTLOADER INTO RADIO CHIP
    'Refer to SI4689 application note 649: eg s5.1 p381
    case 5
      Issue_LOAD_INIT
      WriteMsg ("Loading bootloader")
      CTSwait = 9

    case 6,7
      'Copy the bootloader code using two SPI transfers
      Address = PEEK(CFUNADDR LOADERBIN)

      if (State=6) then
        a=0 : b=4092
      else
        a=4096 : b=5792
      endif

      'CMD=0x04 (HOST_LOAD)
      'ARG1-3=00 00 00 (Fixed - not configurable)
      'ARG4 .. 4095 (image bytes - sequential, fill command out to 4096 bytes)
      pin(P_SI4689_CS) = 0
      spi write 4,&H04,&H00,&H00,&H00
      if (D > 2) then print "*04,00,00,00 Host Load"

      FOR i = a TO b step 4
        spi write 4,PEEK(BYTE i+Address),PEEK(BYTE i+Address+1),PEEK(BYTE i+Address+2),PEEK(BYTE i+Address+3)
      next i

      pin(P_SI4689_CS) = 1
      CTSwait = 9

    case 8
      'Application note says wait another 4ms - so wait one more tick
      CTSwait = 0

    '- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
    'LOAD BAND SPECIFIC FIRMWARE INTO RADIO CHIP
    'Refer to SI4689 application note 649: eg s5.1 p381
    case 9
      Issue_LOAD_INIT

      'CMD=0x05 (FLASH_LOAD)
      'ARG1-3=00 00 00 (Fixed - not configurable)
      'ARG4-7=start address in flash (Little Endian LSB first)
      'ARG8-11=00 00 00 00 (Fixed - not configurable)
      pin(P_SI4689_CS) = 0
      spi write 4,&H05,&H00,&H00,&H00

      'Load firmware for selected radio band
      if (RadioMode = MODE_AM) then
     	spi write 4,&H00,&HE0,&H11,&H00
        Band="AM"
      elseif (RadioMode = MODE_FM) then
        spi write 4,&H00,&H60,&H00,&H00
        Band="FM"
      else
        spi write 4,&H00,&H20,&H09,&H00
        Band="DAB"
      endif

      'FLASH_LOAD trailing four zero arguments
      spi write 4,&H00,&H00,&H00,&H00

      'Now just wait for the CTS as the radio chip loads image from its FLASH rom
      pin(P_SI4689_CS) = 1
      if (D > 2) then print "*05,00,00,00, ... Flash Load " Band
      WriteMsg ("Loading " + Band + " firmware")
      CTSwait = 100

    '- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
    'BOOT THE BAND SPECIFIC FIRMWARE ON RADIO CHIP
    'Refer to SI4689 application note 649: S5.1 p381
    case 10
      'CMD=0x07 (BOOT)
      'ARG1=00 (Fixed - not configurable)
      pin(P_SI4689_CS) = 0
      spi write 2,&H07,&H00
      pin(P_SI4689_CS) = 1
      if (D > 2) then print "*07,00 Boot"

      WriteMsg ("Booting " + Band + " firmware")
      CTSwait = 60

    case 11
      InitWM8804

      'still don't know if hardware is fitted. Try to read WM8804 ID to see if chip there
      'The 16 bit ID is stored in two registers
      pin(P_WM8804_CSB) = 0
      spi write 1, &H80
      spi read 1, SpiBuf()
      pin(P_WM8804_CSB) = 1
      a = SpiBuf(0) << 8

      pin(P_WM8804_CSB) = 0
      spi write 1, &H81
      spi read 1, SpiBuf()
      pin(P_WM8804_CSB) = 1
      a = a + SpiBuf(0)

      if ((a = 1416) or (SA <> 0)) then '1416=0x0588
        WM8804 = 1
        WriteMsg ("Found WM8804 device")
      else
        WM8804 = 0
        WriteMsg ("Failed to find WM8804 device")
        if (D > 0) then print "Read WM8804 DeviceID:" hex$(a,4)

        'don't show digital output option if no hardware!
	if (SA = 0) then gui hide Rdigital
      endif

      SetRadioMute 'apply Si4689 mute to both channels
      AudioOutMode = ChooseAudioOutMode (0)

      'state machine branches to band-specific commands
      if (RadioMode = MODE_AM) then
        next_state = 50

      elseif (RadioMode = MODE_FM) then
        next_state = 100

      else 'it must be DAB
        next_state = 200

      endif

    '- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
    'This section is AM only
    case 50
      'property=0x4200 (AM_VALID_MAX_TUNE_ERROR)
      SetProperty(&h42, &h00, 0, 75) 'Default setting - 75

    case 51
      'property=0x4201 (AM_VALID_RSSI_TIME)
      SetProperty(&h42, &h01, 0, 8) 'Default 8ms

    case 52
      'property=0x4202 (AM_VALID_RSSI_THRESHOLD)
      SetProperty(&h42, &h02, 0, 35) 'Default 35dBuV

    case 53
      'property=0x4203 (AM_VALID_SNR_TIME)
      SetProperty(&h42, &h03, 0, SnrSettle) 'settling time before evaluating SNR

    case 54
      'property=0x4204 (AM_VALID_SNR_THRESHOLD)
      SetProperty(&h42, &h04, 0, AmSNR) 'if SNR exceeds this, consider it a valid channel

    case 55
      'property=0x4100 (AM_SEEK_BAND_BOTTOM)
      SetProperty(&h41, &h00, AmMIN >> 8, AmMIN and 255) 'AmMIN must be multiple of 9kHz!

    case 56
      'property=0x4101 (AM_SEEK_BAND_TOP)
      SetProperty(&h41, &h01, AmMAX >> 8, AmMAX and 255) '1710 kHz

    case 57
      'property=0x4102 (AM_SEEK_FREQUENCY_SPACING)
      SetProperty(&h41, &h02, 0, AmDELTA and 255) 'Seek in steps of AmDELTA kHz

    case 58
      'property=0x2200 (AM_CHBW_SQ_LIMITS)
      'Datasheet default is 30, 15 (30dB SNR chooses maximum audio BW, 15 chooses min)
      SetProperty(&h22, &h00, 29, 15) 'Sets SNR level for max,min audio BW (in dB)

    case 59
      'property=0x2201 (AM_CHBW_SQ_CHBW)
      'Datasheet default is 35, 20 (3500Hz / 2000Hz)
      a = CtrlVal(Ebandwidth) / 100 'Radio chip takes audio BW in units of 100Hz
      b = a >> 1 'set minimum to be half the full bandwidth
      if (b < (BWmin/100)) then b = BWmin/100 'The chip can go down to 1500 Hz
      SetProperty(&h22, &h01, a, b) 'Sets max, min audio BW

    case 60
      'property=0x2202 (AM_CHBW_SQ_WIDENING_TIME)
      'Time for audio filter to go from min to max audio BW
      'Datasheet default is 8, 0 (8x256+0=2048) meaning 2048 ms
      SetProperty(&h22, &h02, 8, 0)

    case 61
      'property=0x2203 (AM_CHBW_SQ_NARROWING_TIME)
      'Time for audio filter to go from max to min audio BW
      'Datasheet default is 0, 16 (0x256+16=16) meaning 16ms (the min is 16)
      SetProperty(&h22, &h03, 0, 16)

      next_state = 300

    '- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
    'This section is FM only
    case 100
      'Set SNR threshold for decoding stereo
      'property=0x3704 (FM_Blend_SNR_LIMITS)
      'Normal value is 0x180f
      SetProperty(&h37, &h04, &h18, &h0f)

    case 101
      'Set RSSI limits
      'property=0x3700 (FM_Blend_RSSI_LIMITS)
      'normal value is 0x2010
      SetProperty(&h37, &h00, &h20, &h10)

    case 102
      'property=0x1710 (FM_TUNE_FE_VARM)
      SetProperty(&H17, &H10, FmSlope >> 8, FmSlope and 255)

    case 103
      'property=0x1711 (FM_TUNE_FE_VARB)
      SetProperty(&H17, &H11, FmIntercept >> 8, FmIntercept and 255)

    case 104
      'property=0x1712 (FM_TUNE_FE_CFG)
      'ARG=0000 (Close VHF Switch)
      '00 means switch open, 01 means switch closed
      SetProperty(&H17, &H12, &H00, FmSwitch)

    case 105
      'property=0x3100 (FM_SEEK_BAND_BOTTOM)
      i = FmMIN / 10
      SetProperty(&h31, &h00, i >> 8, i and 255) '87500 kHz - in units of 10kHz

    case 106
      'property=0x3101 (FM_SEEK_BAND_TOP)
      i = FmMAX / 10
      SetProperty(&h31, &h01, i >> 8, i and 255) '107900 kHz - in units of 10kHz

    case 107
      'property=0x3102 (FM_SEEK_FREQUENCY_SPACING)
      SetProperty(&h31, &h02, 0, (FmDELTA \ 10) and 255) 'Seek (in units of 10kHz)

    case 108
      'property=0x3200 (FM_VALID_MAX_TUNE_ERROR)
      'Default 114 bppm (max frequency error before setting AFC rail indicator)
      SetProperty(&h32, &h01, 0, 114) 'Default 114

    case 109
      'property=0x3201 (FM_VALID_RSSI_TIME)
      'Default 15ms, but can set longer
      SetProperty(&h32, &h01, 0, 25) 'Default 15ms

    case 110
      'property=0x3202 (FM_VALID_RSSI_THRESHOLD)
      'Default is 17, but can set higher
      SetProperty(&h32, &h02, 0, 17) 'Default 17dBuV

    case 111
      'property=0x3203 (FM_VALID_SNR_TIME)
      'default is 40ms
      SetProperty(&h32, &h03, 0, SnrSettle) 'settling time before evaluating SNR

    case 112
      'property=0x3204 (FM_VALID_SNR_THRESHOLD)
      'default is 10dB, but can set lower
      SetProperty(&h32, &h04, 0, FmSNR) 'if SNR exceeds this, consider it a valid channel

    case 113
      if (CapSearch > 0) then goto RETRY_TUNE

      ' - - - - - RDS only below here
      'property=0x3C00 (FM_RDS_INTERRUPT_SOURCE)
      'Interrupt enable=0x0018
      '(0x10=RDSTPPTY, 0x08=RDSPI, 0x02=Sync change, 0x01=FIFO has data)
      SetProperty(&h3c, &h00, &h00, &h01)

    case 114
      'property=0x3C01 (FM_RDS_INTERRUPT_FIFO_COUNT)
      'Defines a minimum number of messages to be received before interrupting
      SetProperty(&h3c, &h01, &h00, 8) 'must be in the range 0..25

    case 115
      'Set RDS settings and set RDS block error thresholds
      'property=0x3C02 (FM_RDS_CONFIG)
      'Block error threshold for block in bits 7,6 and 5,4
      '  00 = accept no errors or corrections
      '  01 = accept 1-2 bits corrected
      '  10 = accept 3-5 bits corrected
      '  11 = accept uncorrectable errors
      'RDS enabled=0x0001 (enabled)
      ' Try setting to &H51?
      '(&H51) to accept 1-2 corrected errors, (&HA1)=3-5
      SetProperty(&h3c, &h02, &h00, &h01)
      next_state = 300
    '- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
    'This section is DAB only
    case 200
      'property=0x1710 (DAB_TUNE_FE_VARM)
      SetProperty(&H17, &H10, DabSlope >> 8, DabSlope and 255)

    case 201
      'property=0x1711 (DAB_TUNE_FE_VARB)
      SetProperty(&H17, &H11, DabIntercept >> 8, DabIntercept and 255)

    case 202
      'property=0x1712 (DAB_TUNE_FE_CFG)
      'ARG=0001 (Open VHF Switch)
      '00 means switch open, 01 means switch closed
      SetProperty(&H17, &H12, &H00, DabSwitch)

    case 203
      'property=0xB400 (DAB_XPAD_ENABLE)
      'ARG=0005 (Enable DLS packets, MOT slideshow)
      'Note: needs time for data ensemble to become available, check in RSQ loop
      SetProperty(&HB4, &H00, &H00, &H05)

    case 204
      'property=0xB200 (DAB_VALID_RSSI_TIME)
      SetProperty(&hB2, &h00, 0, 200) '200ms

    case 205
      'property=0xB201 (DAB_VALID_RSSI_THRESHOLD)
      SetProperty(&hB2, &h01, 0, 30) '30dBuV

    case 206
      'property=0xB202 (DAB_VALID_ACQ_TIME)
      SetProperty(&hB2, &h02, 7, 255) '2047ms

    case 207
      'property=0xB203 (DAB_VALID_SYNC_TIME)
      SetProperty(&hB2, &h03, 5, 0) '1280ms

    case 208
      'property=0xB204 (DAB_VALID_DETECT_TIME)
      SetProperty(&hB2, &h04, 0, 70) '70ms

    case 209
      if (CapSearch > 0) then goto PARSE_DAB_TABLE

      'property=0xB300 (DAB_EVENT_INTERRUPT_SOURCE)
      SetProperty(&hB3, &h00, 0, &H01) 'SRV_LIST_INT_Enable

    case 210
      'property=0x8100 (DIGITAL_SERVICE_INT_SOURCE)
      SetProperty(&h81, &h00, 0, 1) 'DSRVPCKTINT

    case 211
      'property=0x8101 (DIGITAL_SERVICE_RESTART_DELAY)
      SetProperty(&h81, &h01, 8, 0) '2048ms (default was 8s)

    case 212
      'property=0xB301 (DAB_EVENT_MIN_SVRLIST_PERIOD)
      SetProperty(&hB3, &h01, 0, 15) '1.5s minimum between service list updates

    case 213
      'property=0xB302 (DAB_EVENT_MIN_SVRLIST_PERIOD_RECONFIG)
      SetProperty(&hB3, &h02, 0, 5) '500ms minimum between service list updates

    case 214
      'property=0xB501 (DAB_ACF_MUTE_SIGLOSS_THRESHOLD)
      'Default is 6dBuV - way too low. 30 seems more like it. I get a working DAB
      'signal at 31dBuV, but its pretty marginal at that level.
      SetProperty(&hB5, &h01, 0, 30)

    case 215
      'property=0xB507 (DAB_ACF_SOFTMUTE_BER_LIMITS)
      SetProperty(&hB5, &h07, &he2, &hc4) 'Max at -30 (E2), min at -60 (C4)

    case 216
      'property=0xB508 (DAB_ACF_CMFTNOISE_LEVEL)
      i = &h100
      SetProperty(&hB5, &h08, i >> 8, i and 255) 'lower level than default level

    case 217
    PARSE_DAB_TABLE:
      'Read the DAB frequency table
      'CMD=0xB9 (DAB_GET_FREQ_LIST)
      'ARG1=00 (Fixed - not configurable)
      pin(P_SI4689_CS) = 0
      spi write 2,&HB9,&H00
      pin(P_SI4689_CS) = 1
      if (D > 2) then print "*B9,00 DAB get freq list"
      Reply_Size=5
      CTSwait = 9
      SpiWait = 1

    case 218
      'Parse the DAB frequency table we just loaded to remember the frequencies
      '
      'Firstly, receive the status message initiated above. The response bytes are in
      'SpiBuf() and the reply is a variable length. We asked for 5 bytes above.
      'The number of entries in the table will be in the 5th byte [ie (4)]
      NumDabFreqs = SpiBuf(4)
      if (NumDabFreqs > MAX_DAB_FREQS) then
        if (D > 0) then print "Limiting NumDabFreqs from:" NumDabFreqs
        NumDabFreqs=MAX_DAB_FREQS
      endif

      'Now we know # entries, re-read the entire message including all the frequencies
      Read_Radio_Reply(8 + (NumDabFreqs << 2))

      'Pull out the frequencies and put into DabFreqs() array
      for i = 0 to (NumDabFreqs - 1)
        a = 8 + (i << 2)
        DabFreqs(i) = (SpiBuf(a+2) << 16) + (SpiBuf(a+1) << 8) + SpiBuf(a)
        if (D > 1) then print "DAB Frequency[" i "]=" DabFreqs(i)
      next

      if (D > 1) then print "Num DAB frequencies:" NumDabFreqs
      next_state = 300
    '- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
    'select FREQUENCY
    'Refer to SI4689 application note 649: eg s5.1 p382

    case 300
      RETRY_TUNE: next_state = 301

      'Set property to select the requested output (digital/analogue)
      'Note: the radio chip supports either analogue output or digital output
      'but not both at the same time. The radio only changes output state at retune

      'Property=0x0800, 0x8001=DAC (analogue), 0x8002=I2S (digital). Can't do both
      i = 1 '1 = analogue
      if (AudioOutMode = AUDIO_DIG) then i = 2 '2 = digital
      if (i = AudioOut) then goto DOTUNE
      AudioOut = i
      SetProperty(&h08, &h00, &h80, i) 'set the chosen output

    case 301
      'Tune to a frequency
     DOTUNE:
      SetMux (1, -1) 'HW mute the 4052
      SetRadioMute 'apply Si4689 mute to both channels
      next_state = 302

    case 302
      pin(P_SI4689_CS) = 0

      if (RadioMode = MODE_AM) then
        'CMD=0x40 (AM_TUNE_FREQ)
        'ARG1=00 (Tune Mode=normal ASAP, Injection=Auto)
        'ARG2-3 (Frequency in kHz, LSB then MSB)
        'ARG4-5 (Antenna tuning capacitor if non-zero, 0=automatic)
        spi write 6,&H40,&H00, Frequency and 255, Frequency >> 8, AntCap and 255, AntCap >> 8
        if (D > 2) then print "*40,00," Frequency "," AntCap " AM tune"

      elseif (RadioMode = MODE_FM) then
        'CMD=0x30 (FM_TUNE_FREQ)
        'ARG1=00 (DirTune=MPS, Tune Mode=normal ASAP, Injection=Auto)
        'ARG2-3 (Frequency in kHz, LSB then MSB)
        'ARG4-5 (Antenna tuning capacitor if non-zero, 0=automatic)
        i = Frequency / 10
        spi write 6,&H30,&H00, i and 255, i >> 8, AntCap and 255, AntCap >> 8
        if (D > 2) then print "*30,00," Frequency "," AntCap " FM tune"

      else 'It must be DAB
        'Need to discover which index in DabFreqs corresponds with the selected channel
        i = DabFreqToIndex(Frequency)

        'Select the DAB index we just discovered
        'CMD=0xB0 (DAB_TUNE_FREQ)
        'ARG1=00 (Injection = auto)
        'ARG2=Frequency Index
        'ARG3=00 (Fixed - not configurable)
        'ARG4-5= (Antenna tuning capacitor if non-zero, 0=automatic)
        spi write 6,&HB0,&H00,i,0, AntCap and 255, AntCap >> 8
        if (D > 2) then print "*B0,00," Frequency ",0," AntCap " DAB tune"
        NumDabServices = 0
        LastDabServList = -1
      endif

      pin(P_SI4689_CS) = 1
      CTSwait = 500
      StatusBits=1 '1=Seek/Tune Complete bit
      if (D > 0) then print "Tuning to " str$(Frequency) " kHz (" str$(NewServiceID) "," str$(NewCompID) ")"
      ExpectTune = 1
      if (RadioMode <> MODE_DAB) then next_state = 400

      'skip over next state if we are doing an AntCap search
      'don't need any DLS packets while searching
      if (CapSearch > 0) then next_state = 400

    case 303
      'property=0xB400 (DAB_XPAD_ENABLE)
      'ARG=0005 (Enable DLS packets, MOT slideshow)
      'Note: needs time for data ensemble to become available, check in RSQ loop
      SetProperty(&HB4, &H00, &H00, &H05)
      next_state = 400

    '- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
    'READ RADIO SIGNAL QUALITY STATUS
    'This state may be entered from an earlier state (cold boot or change frequency).
    'It may also be entered from state 0 - just state poll during normal radio playing
    'Refer to SI4689 application note 649: eg s5.1 p381

    case 400
      PollWait=0 'counter pacing out auto-calling this state (to fetch signal status)

      ''ExpectTune' is set from either a tune command (state 300) or AM/FM seek command
      'We must first check if the tune worked. Retuning can happen by manually selecting
      'new channel, or during a scan/seek operation
      if (ExpectTune <> 0) then
        ExpectTune = 0

        'Check bit 1 of last status read - Set to 1 when seek/tune is complete
        'If zero, the seek is still taking place
        if ((SpiBuf(0) and 1) = 1) then
          'Bit0: 0=seek down, 1=seek up
          'Bit1: 0=single seek, 1=full scan
          'Bit6: 1=Seek Completed
          'Bit7: 1=Seek Requested
          if ((Seeking and 128) <> 0) then Seeking=(Seeking and 3) or 64

        else
          'We have arrived via either states 1008 or 1107. Need to RESTART the seek
          if ((Seeking and 128) <> 0) then
            if (RadioMode = MODE_AM) then goto AM_SEEK
            if (RadioMode = MODE_FM) then goto FM_SEEK
	    'fall through if seeking DAB - not done in HW for DAB
          endif

          'DAB seeking is by jumping to next DAB frequency by a SW config
          'At this point, seek/tune is not complete and we are not in a seek state
          if (D > 0) then print "Failed seek/tune. Retrying"
          goto RETRY_TUNE
        endif

        'At this point, we have a seek/tune complete flag which needs to be cleared
        '(below) and we can prepare to release the mute. The mute release below is a
        'further protection against thumps / cracks in the analogue output.
        if ((MuteDowncount <= 0) and (RadioMode <> MODE_DAB)) then MuteDowncount = MUTE_RELEASE
      endif

      'Request the radio status (and clear any seek/tune interrupts)
      'Different status read messages required during normal times,
      'and during the antenna capacitance search
      if (CapSearch = 0) then
        pin(P_SI4689_CS) = 0

        if (RadioMode = MODE_AM) then
          'CMD=0x42 (AM_RSQ_STATUS)
          'ARG1=03 (Cancel the seek + send Seek/Tune Ack)
          spi write 2,&H42,&H03
          if (D > 2) then print "*42,03 AM Status"
          Reply_Size = 17
          next_state = 410

        elseif (RadioMode = MODE_FM) then
          'CMD=0x32 (FM_RSQ_STATUS)
          'ARG1=03 (Cancel the seek + send Seek/Tune Ack)
          spi write 2,&H32,&H03
          if (D > 2) then print "*32,03 FM Status"
          Reply_Size = 22
          next_state = 430

        else 'It must be DAB
          'CMD=0xB2 (DAB_DIGRAD_STATUS)
          'ARG1=01 (Send Seek/Tune Ack)
          spi write 2,&HB2,&H01
          if (D > 2) then print "*B2,01 DAB status"
          Reply_Size = 23
          next_state = 450
        endif

        pin(P_SI4689_CS) = 1
        CTSwait = 30
        SpiWait = 1
      else
        'We are in the state machine searching for optimal AntCap
        next_state = IntState
      endif

    case 410 'AM status from CMD=0x42
      'Receive the status messages initiated above. The response bytes are in SpiBuf()
      '[0] bit 7 = CLEAR TO Send next command
      '[0] bit 6 = Command error (ie a bug in this software)
      '[0] bit 5 = Digital radio link change interrupt (digital radio ensemble)
      '[0] bit 4 = An enabled data component of a digital services needs attention
      '[0] bit 3 = Received signal quality outside defined limits
      '[0] bit 1 = Auto controlled features has crossed a limit
      '[0] bit 0 = Seek / Tune complete
      '[1] bit 5 = New event related to digital radio
      '[1] bit 0 = New interrupt related to HD radio
      '[4] bit 3 = FIC decoder encountered unrecoverable errors
      '[4] bit 2 = Change in ensemble acquisition state
      '[4] bit 1 = RSSI below DAB low threshold
      '[4] bit 0 = RSSI above DAB high threshold
      ReadFreq = (SpiBuf(7) << 8) + SpiBuf(6)
      FreqOffset = 2 * Conv8bitSignedInt(SpiBuf(8))
      SigStrength = Conv8bitSignedInt(SpiBuf(9))
      SNR = Conv8bitSignedInt(SpiBuf(10))
      M = Conv8bitSignedInt(SpiBuf(11)) 'AM=Modulation Index
      ReadAntCap = (SpiBuf(13) << 8) + SpiBuf(12)

      NewStatus = 1
      Stereo = 0
      Quality = -1
      FibErr = -1
      CNR = -1

      if ((SigStrength > AmMinStren) and (SNR > AmSNR)) then
        Rstate = Rplaying
      else
        Rstate = Rtuned
      endif

      goto SET_VOLUME

    case 430 'FM status from CMD=0x32
      ReadFreq = (SpiBuf(7) * 2560) + (SpiBuf(6)*10)
      FreqOffset = 2 * Conv8bitSignedInt(SpiBuf(8))
      SigStrength = Conv8bitSignedInt(SpiBuf(9))
      SNR = Conv8bitSignedInt(SpiBuf(10))
      M = Conv8bitSignedInt(SpiBuf(11)) 'FM=Multipath indication
      ReadAntCap = (SpiBuf(13) << 8) + SpiBuf(12)

      if ((SigStrength > FmMinStren) and (SNR > FmSNR)) then
        Rstate = Rplaying
      else
        Rstate = Rtuned
      endif

      'Send next status request message to understand stereo status
      'CMD=0x33 (FM_ACF_STATUS) automatically controlled features
      'ARG1=01 (Clear any ACF Ack)
      pin(P_SI4689_CS) = 0
      spi write 2,&H33,&H01
      pin(P_SI4689_CS) = 1
      if (D > 2) then print "*33,01 FM ACF status"
      Reply_Size = 11
      CTSwait = 9
      SpiWait = 1

    case 431
      'Receive the additional FM status message initiated above
      '[8] bit 7 set when FM stereo pilot received,
      Stereo = 0
      if ((SpiBuf(8) and 128) <> 0) then Stereo=1
      NewStatus = 1
      Quality = -1
      FibErr = -1
      CNR = -1
      goto SET_VOLUME

    case 450 'DAB status from CMD=0xB2
      SigStrength = Conv8bitSignedInt(SpiBuf(6))
      SNR = Conv8bitSignedInt(SpiBuf(7))
      Quality = Conv8bitSignedInt(SpiBuf(8))
      CNR = Conv8bitSignedInt(SpiBuf(9))
      FibErr = (SpiBuf(11) << 8) + SpiBuf(10)
      ReadFreq = (SpiBuf(14) << 16) + (SpiBuf(13) << 8) + SpiBuf(12)
      FreqOffset = Conv8bitSignedInt(SpiBuf(17))
      M = Conv8bitSignedInt(SpiBuf(22)) 'DAB=Fast Detect
      ReadAntCap = (SpiBuf(19) << 8) + SpiBuf(18)
      NewStatus = 1

      'Check for DAB valid bit and acceptable quality
      if ((SpiBuf(5) and 1) = 0) then
        Status="No signal"
        Stereo = 0
        if ((Seeking and 128) <> 0) then Seeking=(Seeking and 3) or 64
        Rstate = Runtuned
        goto Exit_State_Machine 'next_state = IntState
      endif

      Stereo = 1
      if (Rstate = Runtuned) then Rstate = Rtuned

      if (Quality < 25) then
        if (D > 1) then print "Low DAB signal quality:" Quality
        Status="Poor Signal"

      elseif ((SpiBuf(5) and 13) = 13) then
        Status="Errors"

      elseif ((SpiBuf(5) and 13) = 5) then
        Status="Signal OK"

      endif

      if ((ServiceID < 0) or (Rstate <> Rplaying)) then 'don't read time if no service!
        next_state = IntState 'Return to idle

      else
        'Note: the value of SECONDS which is returned is always incorrect (30) ignore it
        'CMD=0xBC (DAB_GET_TIME)
        'ARG1=00 (0 = local time)
        pin(P_SI4689_CS) = 0
        spi write 2,&HBC,0
        pin(P_SI4689_CS) = 1
        if (D > 2) then print "*BC,00 DAB get time"
        Reply_Size = 11
        CTSwait = 9
        SpiWait = 1
      endif

    case 451
      a = (SpiBuf(5) << 8) + SpiBuf(4) 'year
      b = SpiBuf(6) 'month
      c = SpiBuf(7) 'date
      e = SpiBuf(8) 'hour
      n = SpiBuf(9) 'minute
      'id = SpiBuf(10) 'ignore seconds - it is always wrong

      if ((c > 0) and (c < 32) and (b > 0) and (b < 13) and (a > 2019) and (a < 2100)) then
        S2 = str$(c,2,0,"0") + "/" + str$(b,2,0,"0") + "/" + str$(a)
      else
        S2 = ""
      endif

      'be careful of bit errors causing maths issues:
      if ((e >= 0) and (e < 24) and (n >= 0) and (n < 60)) then
        S3 = str$(e,2,0,"0") + ":" + str$(n,2,0,"0")
	S1 = S3
        if (S2 <> "") then S1 = S2 + " " + S3
      else
        S3 = ""
        S1 = S2
      endif

      if (D > 2) then print S1

      if (CtrlVal(RLN10) <> S1) then
        CtrlVal(RLN10) = S1 'display the time

        'Set the internal clock every now and then
        if (S2 <> "") then DATE$ = S2
        if (S3 <> "") then TIME$ = S3

        'if a real time clock is available, set that time also
        if ((RtcInstalled <> 0) and (S2 <> "") and (S3 <> "")) then rtc settime a, b, c, e, n, 0
      endif

      goto SET_VOLUME

    case 452 'requires a unique state because it is referred to by LCD button/IR remote
     SET_VOLUME:
      'If we are still in digital output mode, don't want to play with analogue mute
      if (AudioOutMode = AUDIO_DIG) then goto DO_UNMUTE

      i = CtrlVal(BVolume)

      'if headphones plugged in, set volume to half nominal level - to avoid deafness
      if (HP_removed = 0) then
        i = i >> 1

      elseif (AudioOutMode = AUDIO_RCA)) then
        i = 63 'if RCA audio output, just max the level as the preamp will control volume

      endif

      'Set analogue output (DAC) volume
      'property=0x0300 (AUDIO_ANALOG_VOLUME)
      '0x00=mute, 0x01=62dB attenuation, ... 0x3f=no attenuation - Volume in 1dB steps
      SetProperty(&h03, &h00, 0, i)

      if (D > 1) then print "Analogue volume set to " i
      next_state = 453

    case 453
     DO_UNMUTE:
      SetMux (0, -1) 'will HW 'unmute' (i.e. initialise) the TosLink framer too
      SetRadioMute 'set mute based on status of HW + SW mutes
      next_state = IntState 'Return to idle

    '- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
    'SEEK the next station up or down
    'Seeking:
    'Bit0: 0=seek down, 1=seek up
    'Bit1: 0=single seek, 1=full scan
    'Bit6: 1=Seek Completed
    'Bit7: 1=Seek Requested
    '
    'The Si4689 seek function appears to be 'sometimes' unreliable in that it will
    'seek but return with the same frequency. So when starting a seek, firstly nudge
    'the frequency one step, and then issue the seek

    case 1000
      if (D > 0) then print "Initiating AM seek"

      'Can't seek below bottom or top of band
      if (((Seeking and 1) = 0) and (Frequency <= AmMIN)) then goto NO_SEEK
      if (((Seeking and 1) = 1) and (Frequency >= AmMAX)) then goto NO_SEEK

      AM_SEEK:
      SetMux (1, -1) 'mute the 4052
      SetRadioMute 'apply Si4689 mute to both channels

      'CMD=0x41 (AM_SEEK_START)
      'ARG1=00 (start normal ASAP)
      'ARG2=08 (0=seek down, 1=seek up)
      'ARG3=00 (Fixed - not configurable)
      'ARG4,5=0000 (auto detect antenna capacitance)
      pin(P_SI4689_CS) = 0
      spi write 6,&H41,&H00,(Seeking and 1) << 1,0,0,0
      pin(P_SI4689_CS) = 1
      if (D > 2) then print "*41,00," hex$((Seeking and 1) << 1,2) ",0,0,0 AM seek start"

      CTSwait = 2000 'Allow potentially long time for seek to complete
      StatusBits = 1 '1=Seek/Tune Complete bit
      ExpectTune = 1
      SeekStart = Frequency
      next_state = 400

    '- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
    case 1100
      if (D > 0) then print "Initiating FM seek"

      'Can't seek below bottom or top of band
      if (((Seeking and 1) = 0) and (Frequency <= FmMIN)) then goto NO_SEEK

      if (((Seeking and 1) = 1) and (Frequency >= FmMAX)) then
        NO_SEEK:
	Seeking = 64
        goto Exit_State_Machine 'next_state = IntState
      endif

      FM_SEEK:
      SetMux (1, -1) 'mute the 4052
      SetRadioMute 'apply Si4689 mute to both channels

      'CMD=0x31 (FM_SEEK_START)
      'ARG1=00 (start normal ASAP)
      'ARG2=0x (0=seek down, 1=seek up)
      'ARG3=00 (Fixed - not configurable)
      'ARG4,5 (Antenna tuning capacitor if non-zero, 0=automatic)
      pin(P_SI4689_CS) = 0
      spi write 6,&H31,&H00,(Seeking and 1) << 1,0,0,0
      pin(P_SI4689_CS) = 1
      if (D > 2) then print "*31,00," hex$((Seeking and 1) << 1,2) ",0 FM seek start"

      CTSwait = 2000 'Allow potentially long time for seek to complete
      StatusBits = 1 '1=Seek/Tune Complete bit
      ExpectTune = 1
      SeekStart = Frequency
      next_state = 400

    '- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
    'Used for the varactor tuning algorithm described in AN851 page 17
    'Used prior to tuning the radio to known frequency of local FM/DAB station
    'VhfSw=0 means switch open (FM), VhfSw=1 means switch closed (DAB)

    case 10000
      if (RadioMode = MODE_FM) then
        i = FmSwitch
      else
        i = DabSwitch
      endif

      'property=0x1712 (DAB/FM_TUNE_FE_CFG)
      'ARG (Open or Close VHF Switch)
      SetProperty(&H17, &H12, &H00, i) '00 means switch open, 01 means switch closed

    case 10001
      'apply HW mute so we don't hear anything as we deliberately screw up received
      'signal strength
      SetMux (1, -1) 'mute the 4052
      SetRadioMute 'apply Si4689 mute to both channels
      next_state = IntState

    '- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
    'This is used in the AntCap search

    case 11000
      'CMD=0xE5 (TEST_GET_RSSI)
      'ARG1=00 (Fixed - not configurable)
      pin(P_SI4689_CS) = 0
      spi write 2,&HE5,&H00
      pin(P_SI4689_CS) = 1
      if (D > 2) then print "*E5,00 Test Get RSSI"
      Reply_Size = 6
      CTSwait = 9
      SpiWait = 1

    case 11001
      'Read the signal level requested above - it is in a fixed-point binary format
      SigLevel = Conv8bitSignedInt(SpiBuf(5)) + (SpiBuf(4) / 256.0)
      if (D > 1) then print "Signal Level:" SigLevel "dBuV at " Frequency "kHz"
      next_state = IntState

    '- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
    'Read one or more consecutive property values back from the radio
    'The values will appear as 16 bit words starting at SpiBuf(4)+SpiBuf(5)

    case 20000
      'CMD=0x14 (GET_PROPERTY)
      'ARG1=count (number of properties to read)
      'ARG2,3=property id (which property to retrieve)
      pin(P_SI4689_CS) = 0
      spi write 4,&H14, NumProp, PropID and 255, PropID >> 8
      pin(P_SI4689_CS) = 1
      if (D > 2) then print "*14," hex$(NumProp,2) "," hex$(PropID,4) " Get Property"
      Reply_Size = 4 + (NumProp << 1)
      CTSwait = 9
      SpiWait = 1

    case 20001
      'Read out the results from above
      if (D > 1) then
        for i = 0 to (NumProp-1)
          print "Property " hex$(PropID+i,4) "==" hex$(SpiBuf(5+(i<<1)),2) hex$(SpiBuf(4+(i<<1)),2)
        next
      endif

      Exit_State_Machine: next_state = IntState

    '- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
    'Interrupt handler for DAB
    case 60000
      IntState=0

      if ((IntMask and 1) <> 0) then 'Digital Service Interrupt
        'Get digital service data
        'CMD=0x84 (DAB_GET_DIGITAL_SERVICE_DATA)
        'ARG1=0x01 (Fetch the status only (non destructive read), and clear the interrupt)
        pin(P_SI4689_CS) = 0
        spi write 2,&H84,1
        pin(P_SI4689_CS) = 1
        if (D > 2) then print "*84,01 DAB get digital service data"

        'Read only 24 bytes - tells the size of the full string, which must be read again
        Reply_Size = 24
        CTSwait = 9
        SpiWait = 1

        IntMask = IntMask and &HFE 'turn off the int bit so we don't process it again
      else
        goto RdsInt
      endif

    case 60001 'Digital Service Interrupt
      'Receive the status message initiated above. The response bytes are in SpiBuf()
      'The response we just received may have additional data associated with it
      i = (SpiBuf(19) << 8) + SpiBuf(18)
      a = (SpiBuf(21) << 8) + SpiBuf(20)
      b = (SpiBuf(23) << 8) + SpiBuf(22)
      c = (SpiBuf(11) << 24) + (SpiBuf(10) << 16) + (SpiBuf(9) << 8) + SpiBuf(8)
      e = (SpiBuf(15) << 24) + (SpiBuf(14) << 16) + (SpiBuf(13) << 8) + SpiBuf(12)

      if (D > 1) then
        print "Digital Service Interrupt"
        print " Service ID:" c "/" hex$(c,8)
        print " Component ID:" e "/" hex$(e,8)
        print " Buf Count:" SpiBuf(5)

        '0=playing normally, 1=stopped, 2=Overflow, 3=new object, 4=errors
        print " Service State:" hex$(SpiBuf(6),2)

        '0x0x=std data, 0x4x=non DLS pad, 0x8x=PDS pad, 0xCx=reserved
        print " Byte 7:" hex$(SpiBuf(7),2)
        print " UA type:" hex$(SpiBuf(17),2) hex$(SpiBuf(16),2)
        print " Byte Count:" i
        print " Seg Num:" a
        print " Num Segs:" b
      endif

      'Now we know size, re-read the entire message
      S1 = ""
      Read_Radio_Reply(24+i)
      Rstate = Rplaying
      a = 0

      'Refer to table 20/21 in AN649 p433
      if ((SpiBuf(7) and 192) = 128) then
        if (D > 1) then print " Selected via byte 7"
        a = 1

      elseif (((SpiBuf(24) = 0) or (SpiBuf(24) = 128)) and (SpiBuf(25) = 0)) then
        if (D > 1) then print " Selected via initial bytes"
        a = 1

      endif

      if (a <> 0) then
        if (i > 256) then i=256 'Maximum string size is 256
        S1 = ""

        for c = 24 to 23+i
          S1 = S1 + CleanChr(SpiBuf(c))
        next

        S1 = TruncStr (S1)
        if (D > 1) then print "Dab data: " S1
        c = DispString (S1, RLN4, RLN9, 50, 1)
      endif
      goto RdsInt

    case 60100 'RDS Interrupt
    RdsInt:
      if ((IntMask and 2) <> 0) then 'RDS Interrupt
      DECODE_MORE:
        'CMD=0x34 (FM_RDS_STATUS)
        'ARG1=0x01 = Clear RDS interrupt
	'     0x02 = clear anything stored in the fifo
	'     0x04 = Only grab status and return last valid data, don't decrement fifo
        pin(P_SI4689_CS) = 0
        spi write 2,&H34, 1
        pin(P_SI4689_CS) = 1

        if (D > 2) then print "*34,01 FM RDS Status"
        Reply_Size = 20
        CTSwait = 9
        SpiWait = 1
        IntMask = IntMask and &HFD 'turn off the int bit so we don't process it again
        next_state = 60101

      else
        goto DigEvInt
      endif

    case 60101
      'There are four separate pairs of bytes the chip data calls groups A-D. A and B
      'contain status info. C and D contain text. But there are several different formats
      'transmitted in an interleaved way. We need to identify what the current set
      'contains each message. There is also a forward error correction mechanism
      'implemented that allows some errors to be repaired (if only a few bits wrong) and
      'the probable number of errors for each of the four groups is indicated in byte 11
      'of the message response. If only a few errors are allegedly corrected, we will
      'accept and display that information. Ignore if more. Each message can contain 2 or
      '4 bytes of ascii, so multiple consecutive messages need to be integrated to make
      'a complete message. This gives rise to a state machine necessary to reassemble
      'the full messages from two or four byte chunks. Explanation provided in AN243
      '(which doesn't apply to the Si4689, but is still helpful)
      '
      'Refer also to U.S. RBDS Standard 1998 Specification of the radio broadcast data
      '(RDS) System
      if (D > 0) then
        print "--------------------------------"
        print "FM RDS interrupt"
        print "0, 1, 3    -> " hex$(SpiBuf(0),2) " " hex$(SpiBuf(1),2) " " hex$(SpiBuf(3),2)
        print "4, 5, 6, 10-> " hex$(SpiBuf(4),2) " " hex$(SpiBuf(5),2) " " hex$(SpiBuf(6),2) " " hex$(SpiBuf(10),2)
      endif

      if (SpiBuf(10) > 0) then 'This is the count of the number of waiting messages
        'including this one. i.e. 0 means empty. If non-zero, Blocks A-D contain the
        'oldest entry in the FIFO and [10] will decrement by one on the next poll,
        'assuming no new data has been recevied in the interim. The error byte is
        'formated as four groups of 2 bits. The two bit binary pattern 00 means no
        'errors in that block. The two bit binary pattern 01 means one or two errors
        'that were corrected. The other two binary patterns mean more corrected or
        'uncorrected errors.

        'GT = Group Type Code (stored in id)
        'address/index (stored in e)
        id = SpiBuf(15) >> 4
        e = SpiBuf(14) and 31 '0..31

        if (D > 0) then
          if ((SpiBuf(5) and 1) <> 0) then print " RDS FIFO overrun"
          if ((SpiBuf(5) and 2) <> 0) then print " RDS in sync"
          print " Rx Error flags=" hex$(SpiBuf(11), 2)

          if ((SpiBuf(5) and 16) <> 0) then 'TP and PTY valid
            print " TP=" (SpiBuf(6) >> 5) and 1
            print " PTY=" hex$(SpiBuf(6) and 31, 2)
          endif

          if ((SpiBuf(5) and 8) <> 0) then 'PI valid
            print " PI=" hex$((SpiBuf(9) << 8) + SpiBuf(8), 4)
          endif

          print " Group=" hex$(id, 2)
          print " Addr=" hex$(e, 2)
          print " Block A: " hex$(SpiBuf(13),2) hex$(SpiBuf(12),2)
          print " Block B: " hex$(SpiBuf(15),2) hex$(SpiBuf(14),2)
          print " Block C: " hex$(SpiBuf(17),2) hex$(SpiBuf(16),2) " " chr$(39) CleanChr(SpiBuf(17)) CleanChr(SpiBuf(16)) chr$(39)
          print " Block D: " hex$(SpiBuf(19),2) hex$(SpiBuf(18),2) " " chr$(39) CleanChr(SpiBuf(19)) CleanChr(SpiBuf(18)) chr$(39)
        endif

        'group (id) 0 contains 'basic information' - should remain constant for the
        'radio station. Only accept station identifier string data if there were
        'apparently zero errors in that data. The data is in groups B and D
        if ((id = 0) and ((SpiBuf(11) and &H33) = 0)) then
          S1 = CleanChr(SpiBuf(19)) + CleanChr(SpiBuf(18)) 'String comes in 4x2 chunks

          'Same as what we had previously? This checks for changes in string
          'For example, ABC Classic FM sends three strings in succession - 'ABC', 'Classic', 'FM'
          if ((ProgName(e and 3) <> "") and (ProgName(e and 3) <> S1)) then
            if (D > 0) then print " RDS group 0 snippet " e and 3 " change"

            for i = 0 to 3
              ProgName(i) = ""
            next
          endif

          ProgName(e and 3) = S1
          if (D > 0) then print " RDS group 0 snippet " e and 3 "=" chr$(39) S1 chr$(39)

          'Reassemble station name and display on line 1
          S1 = ProgName(0) + ProgName(1) + ProgName(2) + ProgName(3)

          'only proceed if we have received error-free snippets for ALL four segments
          if (len(S1) = 8) then
            S1 = TruncStr (S1) 'remove leading and trailing spaces
            if (D > 0) then print " RDS station name:" S1

            'does this string already exist in the RdsNameSet?
            for i = 0 to (MAX_SET-1)
              if (RdsNameSet(i) = S1) then exit for
            next

            if (i >= MAX_SET) then 'We don't already have this string
              if (RdsNames < MAX_SET) then
                RdsNameSet(RdsNames) = S1
                RdsNames = RdsNames + 1
              endif
            endif

           'Determine the concatenation of the strings. Concatenate in order UNLESS
           'one of them ends in the letters "FM" or "fm"
            for i = 0 to (MAX_SET-1)
              if ((len(RdsNameSet(i)) >= 2) and (lcase$(right$(RdsNameSet(i),2)) = "fm")) then exit for
            next

            S1 = ""
            j = 0

            'If none of the strings end in the letters 'FM', present strings in order
            if (i < MAX_SET) then j = i+1

            for i = 0 to (MAX_SET-1)
              if (RdsNameSet((i+j) mod MAX_SET) <> "") then
                if (S1 <> "") then S1 = S1 + " "
                S1 = S1 + RdsNameSet((i+j) mod MAX_SET)
              endif
            next

            'The maximum length of the combined string cannot exceed NameLen characters
            S1 = left$(S1, NameLen)

            'Do we already have a station name in the preset file?
            S2 = MatchName (Frequency, S1, -1, -1, 0)
            if (CtrlVal(RLN1) <> S2) then CtrlVal(RLN1) = S2

            'If the RDS string is different than the configured station name, display
            if ((S1 <> S2) and (CtrlVal(RLN9) <> S1)) then
              CtrlVal(RLN9) = S1
            endif
          endif

        elseif ((id = 2) and ((SpiBuf(11) and &H15) = 0)) then
          'group 2 contains 'radiotext' as two groups of 64 chars - only one is
          'active at a time. Tolerate a few corrected errors for radiotext.
          i = e and 16 'will be 0 if new section is a, or 16 if new section is b

          'Has the active group changed? start assembling the next string
	  ' i will be either 0 or 16
          if (i <> ActiveProgData) then

            TryToDispRDS (i+1) 'non-zero param forces string blanking
            ActiveProgData = i
          endif

          S1 = CleanChr(SpiBuf(17)) + CleanChr(SpiBuf(16)) + CleanChr(SpiBuf(19)) + CleanChr(SpiBuf(18))
          ProgData(e) = S1
          if (D > 0) then print " RDS group 2 snippet " e "=" chr$(39) S1 chr$(39)

        elseif ((id = 4) and ((e and 16) = 0) and ((SpiBuf(11) and &H3f) = 0)) then
          'group 4A contains TIME. Don't accept any errors in groups B, C, or D!
          'The algorithm is a mess and a kludge. It is described in the RBDS reference
          'section 3.1.5.6 and Annex G.

          'Extract the time
          'j=hours, n=minutes, c=timezone offset
          j = ((SpiBuf(16) and 1) << 4) + ((SpiBuf(19) and &hf0) >> 4)
          n = ((SpiBuf(19) and 15) << 2) + ((SpiBuf(18) and &Hc0) >> 6)
	  c = SpiBuf(18) and 31

          'The sign of the offset is in bit5 of SpiBuf(18)
          if ((SpiBuf(18) and 32) <> 0) then c = -c

          'Add the time zone offset to the specified time
	  'The offset is specified in chunks of half an hour
	  'I found that the ABC sets timezone to 0, but other stations set real TZ
	  id = (j * 60) + n + (c * 30)

          'We need to do some carrying and borrowing when the adjusted local time
          'crosses midnight
          if (id < 0) then
            id = id + 1440 'add one day (in minutes)
            c = -1

          elseif (id >= 1440) then
            id = id - 1440 'remove one day
            c = 1

          else
            c = 0

          endif

          'recalculate the hours and minutes after we adjusted to the local TZ
          e = id / 60 'hours
          n = id mod 60 'minutes

          'be careful of bit errors causing maths issues:
	  if ((e >= 0) and (e < 24) and (n >= 0) and (n < 60)) then
            S3 = str$(e,2,0,"0") + ":" + str$(n,2,0,"0")
          else
	    S3 = ""
          endif

          'j will be Modified Julian Day, a will beYear, b will be Month, c will be Date
          'c is the carry resulting from timezone offset adjustment.
          j = ((SpiBuf(14) and 3) << 15) + (SpiBuf(17) << 7) + (SpiBuf(16) >> 1) + c

          b = int((j - 15078.2) / 365.25) 'What an awesome piece of computer science
          comp = int((j - 14956.1 - int(b * 365.25)) / 30.6001) 'like the kludge?
          c = j - 14956 - int(b * 365.25) - int(comp * 30.6001) 'c=date. No - I didn't write this formula.

          'the algorithm defines something called k, which I store in id
          id = 0 'I mean, 'id' is about as meaningful as 'k' in this awesome kludge
	  if ((comp = 14) or (comp = 15)) then id = 1

	  a = b + id + 1900 'year
	  b = comp - 1 - (id * 12) 'month
          S1 = ""

          'Convert the date/month/year to a string, avoiding data errors (if any)
          if ((c > 0) and (c < 32) and (b > 0) and (b < 13) and (a > 2019) and (a < 2100)) then
            S2 = str$(c,2,0,"0") + "/" + str$(b,2,0,"0") + "/" + str$(a)
	    S1 = S2

            if (S3 <> "") then S1 = S2 + " " + S3
          endif

          if (D > 0) then print " RDS group 4A. Julian Day=" j ", Time=" S1

          if (CtrlVal(RLN10) <> S1) then
            CtrlVal(RLN10) = S1 'display the time

            'Set the internal clock every now and then
            if (S2 <> "") then DATE$ = S2
            if (S3 <> "") then TIME$ = S3

            'if a real time clock is available, set that time also
            if ((RtcInstalled <> 0) and (S2 <> "") and (S3 <> "")) then rtc settime a, b, c, e, n, 0
          endif
        endif

        'Is there more RDS data queued up to decode? Loop back if more.
        if (SpiBuf(10) > 1) then goto DECODE_MORE
        TryToDispRDS (0) 'after all messages loaded, try and display what we got
      endif

      goto DigEvInt

    case 60200 'Digital Event Interrupt
    DigEvInt:
      if ((IntMask and 4) <> 0) then 'Digital Event Interrupt
        'Get the DAB event status when interrupt tells us it is ready
        'CMD=0xB3 (DAB_GET_EVENT_STATUS)
        'ARG1=01 (Clear EVENT int)
        pin(P_SI4689_CS) = 0
        spi write 2,&HB3,&H01
        pin(P_SI4689_CS) = 1
        if (D > 2) then print "*B3,01 DAB get event status"
        if (D > 1) then print "Waiting for DAB service list"
        Reply_Size = 8
        CTSwait = 9
        SpiWait = 1

        IntMask = IntMask and &HFB 'turn off the int bit so we don't process it again
        next_state = 60201
      else
        next_state = IntState 'no other kinds of interrupt to process here
      endif

    case 60201 'get the service list identifier, and only get list if ID is new
      i = (SpiBuf(7) << 8) + SpiBuf(6)

      'Check that we have a new service list
      if (((SpiBuf(5) and 1) = 0) or (LastDabServList = i)) then
        if (D > 1) then print "No new service list"
        goto Exit_State_Machine 'next_state = IntState
      endif

      if (D > 1) then print "Received DabServList=" i " (Last was " LastDabServList ")"
      LastDabServList = i

      'CMD=0x80 (GET_DIGITAL_SERVICE_LIST)
      'ARG1=00 (Get Audio service list)
      pin(P_SI4689_CS) = 0
      spi write 2,&H80,&H00
      pin(P_SI4689_CS) = 1
      if (D > 2) then print "*80,00 Get audio service list"

      'Read only a short 6 bytes - tells us the actual size of the full string
      Reply_Size = 6
      CTSwait = 9
      SpiWait = 1

    case 60202
      i = 6 + (SpiBuf(5) << 8) + SpiBuf(4)

      if (i >= READ_REPLY_BUFSIZE) then
        if (D > 1) then print "Digital service list string size:" i
        cpu restart 'This is a major panic and should never happen
      endif

      'Now we know # entries, re-read the entire message
      Read_Radio_Reply(i)

      'parse and decode the dab service list
      n = SpiBuf(8) 'this is the number of potential services in this list
      if (D > 1) then print "Num Dab Services in list:" n

      'Table format:
      '[0-3]= service ID
      '[4]= Service Info 1 (ignored)
      '[5]= Service Info 2
      '[6]= Service Info 3
      '[7]= padding (ignored)
      '[8-23]= Service Label / name of this service
      '[24-25]= Component ID
      '[26]= Component info
      '[27]= Valid flags
      'Refer to AN649 table 14 + EN 300 401 standard to understand format
      'Table components can arrive in a different order every time the radio is booted!
      a = 12 'this will be an offset into a variable length table
      NumDabServices = 0

      for i = 0 to (n - 1)
        id = SpiBuf(a) + (SpiBuf(a+1)<<8) + (SpiBuf(a+2)<<16) + (SpiBuf(a+3)<<24)
        e = SpiBuf(a+4) 'Service info #1 field, bit0=1 means digital service (like EPG)
        b = SpiBuf(a+5) and 15 'Number of components
        S3 = ""

        for c = 8 to 23 '16 characters
          'the MMbasic file read command has trouble with unprintable characters
          if ((SpiBuf(a+c) < 32) or (SpiBuf(a+c) > 127)) then exit for
          S3 = S3 + CHR$(SpiBuf(a+c))
        next

        S3 = TruncStr (S3) 'remove leading and trailing spaces

        if (D > 1) then
          print i ": Service ID:" id "/0x" hex$(id) "=" chr$(34) S3 chr$(34)
          print "    Conditional Access:" SpiBuf(a+5)>>4
          print "    # Components:" b
          print "    Service Info 1:0x" hex$(e,2)
          print "    Service Info 3:0x" hex$(SpiBuf(a+6),2)
        endif

        a = a + 24 'point to the component field(s) - this is a variable length

        for c = 1 to b
          comp = SpiBuf(a) + (SpiBuf(a+1)<<8)

          if (D > 1) then
            print "    Component ID[" c "]:" comp "/0x" hex$(comp)
            print "    Component info[" c "]:0x" hex$(SpiBuf(a+2),2) hex$(SpiBuf(a+3),2)
          endif

          a = a + 4 'ignore the second two bytes

          'If we want to keep this record, store it in the array
          'if its a data service like an EPG - we're not interested
          if (((e and 1) = 0) and (S3 <> "") and (NumDabServices < MAX_DAB_SERVICES)) then
            S2 = S3

            if (b > 1) then
              S2 = left$(S3,NameLen-1) + str$(c)
            endif

            DabServiceName(NumDabServices) = S2
            DabServiceID(NumDabServices) = id
            DabComponentID(NumDabServices) = comp
            NumDabServices = NumDabServices + 1

            'Do we already have a station name in the preset file? MatchName will add it
            S1 = MatchName (Frequency, S2, id, comp, 0)
          endif
        next
      next

      if (D > 1) then print "Keeping " NumDabServices " service descriptions"

      'If we are ready to start a service, can we find a match? If so, start the service
      if ((NewServiceID >= 0) and (NewCompID >= 0)) then
        for i = 0 to (NumDabServices - 1)
          if ((NewServiceID = DabServiceID(i)) and (NewCompID = DabComponentID(i))) then exit for
        next

        if (i < NumDabServices) then
          DabID=i
          if (D > 0) then print "Starting digital service ID=" NewServiceID " component=" NewCompID
          ServiceID = NewServiceID
          CompID = NewCompID
          NewServiceID = -1
          NewCompID = -1

          'prepare the separated hex values
          id = ServiceID and 255
          a = (ServiceID >> 8) and 255
          b = (ServiceID >> 16) and 255
          c = (ServiceID >> 24) and 255

          comp = CompID and 255
          e = (CompID >> 8) and 255
          n = (CompID >> 16) and 255
          i = (CompID >> 24) and 255

          'CMD=0x81 (DAB_START_DIGITAL_SERVICE)
          'ARG1-3=0 (Start audio service)
          'ARG4-7= Service ID
          'ARG8-11=Component ID
          pin(P_SI4689_CS) = 0
          spi write 12,&H81,0,0,0,id,a,b,c,comp,e,n,i
          pin(P_SI4689_CS) = 1
          if (D > 2) then print "*81,0,0,0," hex$(ServiceID,8) "," hex$(CompID,8) " DAB start digital service"
          CTSwait = 9
          SpiWait = 1

          Rstate = Rselected
          MuteDowncount = MUTE_RELEASE
	  State = 452
	  exit sub
        endif
      endif

      next_state = IntState
  end select

  'Jump to the next state
  State = next_state
end sub

'-----------------------------------------------------
'Manage scanning for the preset file, and seeking up and down
'
'The 'Seeking' variable is used as follows:
'Bit0: 0=seek down, 1=seek up
'Bit1: 0=single seek, 1=full scan
'Bit6: 1=Seek Completed
'Bit7: 1=Seek Requested

sub ProcessScan
  local integer i, x

  if ((D > 0) and (DABwait = 0)) then
    print "Seek ack Freq:" ReadFreq " SNR:" SNR " SigStrength:" SigStrength " min:" Min_SigStren " CNR:" CNR " Qual:" Quality
  endif

  Frequency = ReadFreq

  'bit1=0 means this scan was a one-off - ie a one off seek up/down. We can now exit
  if ((Seeking and 2) = 0) then
    Seeking=0
    exit sub
  endif

  'To reach here, we are in the middle of a full scan
  'State=0 to reach here
  'Did we arrive at a frequency where signal strength is OK and we want to remember it?
  if ((SigStrength >= Min_SigStren) and ((RadioMode = MODE_DAB) or (SNR > Min_SNR))) then
    if (RadioMode = MODE_DAB) then
      'DAB takes longer to sync and read service names
      if (DABwait <= 0) then
        DABwait=1
        exit sub

      elseif (DABwait < 2000) then 'have we have waited long enough for names to come in?
        exit sub
      endif

      'we have now waited ten seconds - if NumDabServices is still zero, give up
      DABwait = 0

      if (NumDabServices > 0) then
        for i = 0 to (NumDabServices - 1)
          x = AddPreset (Frequency, DabServiceName(i), DabServiceID(i), DabComponentID(i), 0)
        next
      endif

    else 'AM or FM
      x = AddPreset (Frequency, "", -1, -1, 0)
    endif
  endif

  'Prepare in case none of the if statements below are taken
  Seeking = &H83 'Seek requested + full scan + seek up
  AntCap = 0 'set the default (auto) AntCap

  if (RadioMode = MODE_AM) then State = 1000 'AM seek
  if (RadioMode = MODE_FM) then State = 1100 'FM seek

  'Reached end of AM or FM bands yet?
  if (Frequency = AmMAX) then 'reached top of AM band
    NewFrequency = AFmMIN
    CapWait = 0

  elseif (Frequency = FmMAX) then 'reached top of FM band
    NewFrequency = ADabMIN 'Dummy frequency to force DAB mode
    CapWait = 0
    State = 300 'For DAB, the seeks are manual frequency selections

  elseif ((NumDabFreqs > 0) and (Frequency >= DabFreqs(0))) then 'doing DAB
    'DAB is different than AM and FM - the chip doesn't scan for us
    'we need to select freq manually

    DabFreqIndex = DabFreqToIndex (Frequency) + 1 'choose the next index
    if (DabFreqIndex >= NumDabFreqs) then goto FINSCAN
    Frequency = DabFreqs(DabFreqIndex)

    if (Frequency > ADabMAX) then
      'reached the end of the DAB band - we can compact and rewrite the preset file
      FINSCAN:
      WritePresetFile
      ReadPresetFile
      SetFavs

      if (D > 0) then print "Scan complete. " NumServices " services in preset table"
      Seeking = 0 'stop the complete scan by turing off bit 2
      CapWait = 0
      Change_Smode = Xstatus

      'What to do now? lets just load the first radio station
      NewFrequency = Pfreq(0)
      NewServiceID = Pid(0)
      NewCompID = Pcomp(0)
      AntCap = Pcap(0)
      CtrlVal(Ebandwidth) = Pbw(0)
      State = 0

    else 'choose the next DAB channel frequency in the list of potentials
      NewFrequency = DabFreqs(DabFreqIndex)
      DABwait = 0 'we are waiting on a DAB channel
      CapWait = 0
      State = 300 'For DAB, the seeks are manual frequency selections
    endif

  else
    MuteDowncount = MUTE_RELEASE << 1 'remute
  endif
end sub

'-----------------------------------------------------
'Manage search for optimal antenna capacitance value
'Tuning algorithm described in AN851 page 17
'The application note describes a process using an expensive piece of test gear. As I
'can't afford that, I have implemented a poor-man's substitute.... I am using radio
'stations themselves as the test sources. The purpose of the AntCap algorithm described
'in the application note, and implemented below, is to try and get the radio chip to
'configure an antenna capacitance value that maximises signal strength. The radio has an
'in-built algorithm to calculate the antcap value, but you, - the developer - need to
'tell (config) the radio chip with the Y intercept and slope of the straight-line
'approximation of the function that represents the AntCap value as a function of tuned
'frequency. So this algorithm below will try and find the AntCap value that maximises
'RxStrength for the currently tuned frequency. You - the designer - need to take as many
'readings at different frequencies as you can, then plot them on a graph (eg using
'microsoft excel) and then calculate the Y intercept and slope of the line of best fit
'to programme into the radio chip. I have found that I could achieve a boost of around
'3.5dB signal strength by optimising the AntCap settings compared with the defaults used
'in the origianl SC software. This happened to help *a lot* for me because I live in a
'fringe signal area, and the extra power made all the difference in the world to a
'usable and an unreliable DAB signal.
'
'BTW: be patient while this algorithm runs - it is somewhat slow and takes minutes each
'frequency. It is taking many readings, and then averaging both across the target
'frequency, and also dragging in the values from surrounding frequencies. The documented
'algorithm takes way fewer readings and doesn't drag in the surrounding frequencies, but
'then again, my poor-mans implementation uses radio stations in a fringe reception area,
'and I found I needed way more averaging to reduce the noise in the results. Doing this
'seemed to work very well for me and improved performance. Your mileage may vary.
'
'Capsearch is another state machine variable
'CapSearch=1 - instruction to commence search at current Frequency
'CapSearch=2 - dummy state used to wait for touch
'CapSearch=3 - Set AntCap value in hardware and initialise
'CapSearch=4 - Set inductance switch
'CapSearch=5 - Set the new capacitance value
'CapSearch=6 - Delay a little to let hardware settle
'CapSearch=7 - Average RSSI using TEST_GET_RSSI
'CapSearch=10 - dummy state used to wait for touch

'Refer AN649 p26 for AntCap to capacitance formula:
'FM and DAB: C=(AntCap-1)*250fF (an fF is a femto farad - 1000 times smaller than a pF)
'AM: C=(AntCap-1)*142fF [from AN649 p269]

const MAX_SAMPLES=20 'Take 20 samples of our real radio station and average them

dim integer highest_antcap

function What_Cap (x as integer) as float
  if (RadioMode = MODE_AM) then
    What_Cap = (x - 1) * 142.0/1000.0
  else
    What_Cap = (x - 1) * 250.0/1000.0
  endif
end function

sub ProcessCap
  static integer i, bestcap, of, oi, oc
  static float bin, maxsig, f1, f2, f3, f4

  PollWait=0

  select case CapSearch
    case 1
      Change_Smode = Xcap
      ClearConsole

      'display welcome message to appear on the console screen
      CtrlVal(LN1) = "Antenna Capacitance Search: " + str$(Frequency) + " kHz"
      CtrlVal(LN2) = "Refer to Silicon Labs App Note AN851 Appendix A"
      CtrlVal(LN3) = "for more details about the algorithm."

      CtrlVal(LN5) = "Run this function for a range of stations"
      CtrlVal(LN6) = "across the FM and DAB bands and note results."
      CtrlVal(LN7) = "The optimal AntCap values will be automatically"
      CtrlVal(LN8) = "saved to the presets. Edit the presets and"
      CtrlVal(LN9) = "interpolate any missing AntCap values."

      CtrlVal(LN11) = "Scanning will take several minutes to complete."

      CtrlVal(LN13) = "Touch screen to start."
      CapSearch = 2

    'CASE 2 'Wait for touch - not coded into in this select statement!
    'The state is set to 3 by AR1 touch in touchdown interrupt routine

    case 3 'set the frequency, and wait for new firmware to load (if required)
      ClearConsole
      WriteMsg ("Commencing AntCap search...")

      if (RadioMode = MODE_AM) then
        highest_antcap = 4096 'Not much value in trying to do AM, it works OK as is
      else
        highest_antcap = 128
      endif

      maxsig = -1
      bestcap = -1
      of = Frequency 'the tune will default to the current AntCap setting
      oi = ServiceID
      oc = CompID
      f1 = 0
      f2 = 0
      f3 = 0
      f4 = 0

      'Tune to the current frequency, and if DAB, don't select a service ID
      NewFrequency = of
      NewServiceID = -1
      NewCompID = -1
      RadioMode = -1 'to force a firmware reload
      CapSearch = 4

    case 4 'set inductance switch
      State = 10000 'Set VHF inductance and mute
      AntCap = 1 'we want to scan from AntCap=1
      CapSearch = 5

    case 5 'set cap
    do_setcap:
      bin = 0.0
      i = 0 'our sample counter
      CapWait = -200 'Wait 1s for first sample
      CapSearch = 6
      State = 300 'Issue set frequency command (to set antcap) and read the RSSI status

    case 6 'waiting to settle
    do_wait:
      if (CapWait > 30) then
        State = 11000 'Request a signal level reading
        CapSearch = 7
      endif

    case 7
      bin = bin + SigLevel
      i = i + 1

      if (i < MAX_SAMPLES) then
        CapWait = 0 'Wait and then read another sample
        CapSearch = 6
        goto do_wait
      endif

      bin = bin / MAX_SAMPLES
      S1 = "AntCap=" + str$(AntCap) + " (" + str$(What_Cap(AntCap),0,2) + "pF), Av SigStren=" + str$(bin,0,1) + "dBuV"
      WriteMsg (S1)
      f1 = (bin + f1 + f2 + f3 + f4) / 5 'take average of last five frequencies

      if (f1 > maxsig) then
        maxsig = f1
        bestcap = AntCap - 3 'We are averaging across five caps, middle is three back
      endif

      if (AntCap < highest_antcap) then
        f1 = f2
	f2 = f3
	f3 = f4
	f4 = bin
        AntCap = AntCap + 1
        goto do_setcap
      endif

      AntCap = bestcap + 1

      'Can we apply this new AntCap setting to any existing preset?
      for i = 0 to (NumPresets - 1)
        if (Pfreq(i) = of) then
          Pcap(i) = AntCap
          Pwait = 1 'Set timer to rewrite the presets table
        endif
      next

      'create a new Preset record, if we didn't find one already
      'This new record will hold the measured AntCap value for this frequency
      if ((Pwait < 1) and (NumPresets < MAX_PRESETS)) then
        Pwait = AddPreset (of, "", oi, oc, AntCap)
        DirtyPresets = 1 ' set the dirty flag
      endif

      'We now have the maximum capacitance and signal level at that capacitance
      CapSearch = 10 'a dummy value for the AR1 gui change
      ClearConsole
      WriteMsg ("Antenna Capacitance Search: " + str$(of) + " kHz")
      WriteMsg ("")
      WriteMsg ("Optimal AntCap value=" + str$(AntCap))
      WriteMsg ("Optimal Capacitance=" + str$(What_Cap(bestcap),0,2) + " pF")
      WriteMsg ("Optimal Signal Strength=" + str$(maxsig,0,1) + " dBuV")
      CtrlVal(LN13) = "Touch screen to return to setup."

      'Retune the station, selecting the optimal AntCap value just calculated
      'This will also unmute the radio
      NewFrequency = of
      NewServiceID = oi
      NewCompID = oc
      RadioMode = -1 'to force a firmware reload
  end select
end sub

'-----------------------------------------------------
' Display the current error file, page by page
' Works a little bit like the 'more' programme under unix
'
' On entry, the file pointer is at the next line to read / display

sub DispErrFile
  local integer l=LN1
  local string s3 length 32, s4 length 32, s5 length 32, s6 length 32, s7 length 32, s8 length 32, s9 length 32
  ClearConsole

  'read each line of the CSV file until end of file
  on error ignore

  Do While ((Not Eof(2)) and (l <= LN12))
    Input #2, S1, S2, s3, s4, s5, s6, s7, s8, s9
    S1 = TruncStr (S1 + " " + S2 + " " + s3 + " " + s4 + " " + s5 + " " + s6 + " " + s7 + " " + s8 + " " + s9)
    l = DispString (S1, l, LN13, 49, 0)
  Loop

  if (Eof(2) and (l = LN1)) then
    Close 2
    DispErrors = 0
    Change_Smode = Xsetup
  else
    DispErrors = 1
  endif

  on error abort
end sub

'-----------------------------------------------------
'THIS IS THE MAIN PROGRAMME
if (RtcInstalled <> 0) then
  rtc gettime
  S1 = "Initialising at " + DATE$ + " " + TIME$
  WriteMsg (S1)
endif

ReadDefaultFile
ReadPresetFile
ReadIRCodesFile

'Have we restarted because of a watchdog timeout?
'I have found MMBASIC somewhat unreliable. It crashes from time to time with strange
'errors - as if it makes up a syntax error in the basic programme that's not there,
'or makes up a font error in the BASIC programme that's not there and similar problems.
'These crashes are particularly bad with MMBASIC version 5.05.03 (which crashes like
'this every few seconds from power-on and autorun) but it happens infrequently (once per
'12 hours?) with MMBASIC version 5.05.02 and 5.05.01.
'
'When MMBASIC crashes, it will error-out to a prompt (which is undesirable for a radio
'because you don't want to have a terminal permanently connected. i.e. The micromite
'is being used as an embedded controller). When MMBASIC crashes, you generally don't
'notice right away because the radio chip will keep doing what it was last doing, and
'outputting audio. But when MMBASIC crashes out to a prompt, the radio GUI no longer
'works and the only recovery is a power cycle.
'
'I have two micromites that behave similarly in this respect so it doesn't appear to be
'a one-off with my hardware. I suspect that the SC micromite design doesn't have enough
'capacitance or the caps supplied in the micromite kits are not up to low ESR spec or
'the PCB is not optimally laid out for power supply noise or ??? It might also be that
'because the 3.3V regulators that were supplied in the SC kits output just below 3.3V,
'the hardware is a little more susceptible to voltage droop due to below spec capacitor
'ESR etc. Whatever, all the components individually appear to be in spec, but as a
'constructed micromite kit, they don't work well all the time as an ensemble. It has the
'feeling of a design issue, especially because my two micromites, constructed at
'different times, behave similarly.

if (MM.WATCHDOG <> 0) then
  AppendErrFile ("Reboot:Watchdog restart")
endif

'Here is the main loop that runs in the foreground and just spins around all day. The
'timing for the activities within the do-loop is generated from the tick interrupt but
'the work is all done in the foreground from this loop. The GUI mechanics run from
'interrupts, but navigating the menu structure is done by a state machine run from the
'do-loop below.

DO
  watchdog 10000 'reset the watchdog for another ten seconds

  if (DoPowerAlarm > 0) then
    if (Pwait > 0) then WritePresetFile 'emergency write out the presets! ASAP!
    if (Dwait > 0) then WriteDefaultFile 'emergency write out the config! ASAP!

    'display alarm
    WriteErr ("Power Drop")
    DoPowerAlarm = 0 'note: the gui functions get restored when MuteDowncount expires
  endif

  '- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  if (Change_Smode > 0) then ProcessScreenChange

  '- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  'The exit timer is a function to automatically fall back from one of the special
  'function screens (like setup, or choose a preset, etc). These screens also display
  'a dim 'X' gadget at the top left. Pressing the gadget does the same thing
  'The GuiCounter starts at zero and counts up. It resets to zero each new touch.
  if ((ExitTimer > 0) and (GuiCounter > ExitTimer)) then ProcessExitTimer

  '- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  'The following is a debugging hack for 'stand alone mode'
  if ((SA <> 0) and ((GuiCounter and 255) = 255)) then
    NewStatus = 1
    ReadFreq = Frequency
  endif

  '- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  'If a radio status update has been received, display that here

  if (NewStatus <> 0) then
    'These funny RLNx string arrangements below reduce flickering on the LCD
    S3 = CtrlVal(RLN1)
    if (right$(S3, 4) = " kHz") then S3 = ""
    S1 = MatchName (ReadFreq, S3, ServiceID, CompID, 1)
    if (CtrlVal(RLN1) <> S1) then CtrlVal(RLN1) = S1

    if (ValidFreq(ReadFreq) <> 0) then
      S1 = str$(ReadFreq) + " kHz"
      if (CtrlVal(RLN2) <> S1) then CtrlVal(RLN2) = S1
    endif

    if (maSNR < -90) then maSNR = SNR else maSNR = (MA_MULT! * maSNR) + (MA_INV_MULT! * SNR)
    if (maCNR < -90) then maCNR = CNR else maCNR = (MA_MULT! * maCNR) + (MA_INV_MULT! * CNR)
    if (maSig < 0) then maSig = SigStrength else maSig = (MA_MULT! * maSig) + (MA_INV_MULT! * SigStrength)
    if (maQual < 0) then maQual = Quality else maQual = (MA_MULT! * maQual) + (MA_INV_MULT! * Quality)

    S1 = "Signal:" + str$(maSig, -1, 1)

    if (RadioMode = MODE_DAB) then
      S1 = S1 + "dBuV    CNR:" + str$(maCNR, -1, 1) + "dB    Quality:" + str$(maQual,0,0)
    else
      S1 = S1 + "dBuV    SNR:" + str$(maSNR, -1, 1) + "dB"
    endif

    'display the stats - but at a lower rate to reduce screen flicker
    if (((Rstate <> Rplaying) or (DispRefresh > 8)) and (CtrlVal(RLN3) <> S1)) then
      CtrlVal(RLN3) = S1
      DispRefresh = 0
    else
      DispRefresh = DispRefresh + 1
    endif

    if (D > 1) then
      print
      print "Freq:" ReadFreq ", SNR:" SNR ", SigStrength:" SigStrength ", CNR:" CNR ", Qual:" Quality ", Foffset:" FreqOffset;
      print ", DAB Status:" Status ", DAB Err:" FibErr ", M:" M ", Stereo:" Stereo ", AntCap:" ReadAntCap ", Seeking:" Seeking
    endif

    NewStatus = 0

    'If a Toslink framer was detected, read status and see what it says
    'if (WM8804 > 0) then
    '  pin(P_WM8804_CSB) = 0
    '  spi write 1, &H8C 'Register 12. Bit7 set means read command
    '  spi read 1, SpiBuf()
    '  pin(P_WM8804_CSB) = 1
    '
    '  'The normal value that reads back is 0x40 (UNLOCK??)
    '  'bit 0 = Audio_N (1=invalid PCM samples)
    '  'bit 1 = PCM_N (1 = sync code detected, not PCM)
    '  'bit 5/4= PLL freqency 10=48 kHz
    '  'bit 6= UNLOCK (1 = not locked to incoming stream)
    'endif

    if (Smode = Xconsole) then Change_Smode = Xstatus 'Flip to GUI if currently console
  endif
  '- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  'The config is stored in Config0.csv
  if (Dwait > DEFAULT_WAIT_TIME) then WriteDefaultFile 'saved 30 seconds after last delta

  '- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  'Save an update of the presets file 30 seconds after the last change
  if (Pwait > DEFAULT_WAIT_TIME) then '200 ticks per second. 6k=30 seconds
    WritePresetFile
    ReadPresetFile
    SetFavs
  endif

  '- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  'Run the radio state machine every timer tick
  if (SpiWait > 0) then CheckSpi

  '- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  'If we are not in an idle state, then don't process user input (below)
  if (State <> 0) then continue do

  '- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  'Process Infrared remote control messages here - state=0 so we can change volume etc
  if (IrDev <> 0) then ProcessIR

  '- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  'ButtonDn and ButtonUp are set by the associated interrupt service routines. Don't want
  'to process these events in the interrupt, because that would take too long and the
  'GUI would judder under load. So we process here instead, outside of the ISR
  if (ButtonDn <> 0) then ProcessButtonDown
  if (ButtonUp <> 0) then ProcessButtonUp

  '- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  'If we are in a SCAN (Seek) process, check if a new station has been discovered
  'Bit 6 is set when a station has been discovered (or not discovered)
  if ((Seeking and 64) = 64) then ProcessScan

  '- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  'If we are in a Antenna Capacitance search process, process now
  if (CapSearch <> 0) then
    ProcessCap

  elseif (PollWait > 50) then
    'If nothing else is in progress, then re-fetch stats for current station every 250ms
    State=400 'If idle, grab fresh radio stats

  endif

  '- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  'Trigger a scan here, which will be further processed by ProcessScan above

  if ((Seeking = 0) and (DoScan > 0)) then
    if (D > 0) then print "Scan commencing"
    DoScan = 0
    NumServices = 0
    NewFrequency = AmMIN
    Seeking = &H83
    ServiceID = -1
    CompID = -1
    Change_Smode = Xscan
    DABwait = 0
    AntCap = 0
    ClearStatus
  endif

  '- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  'Manage the digital mode changes (the Si4689 supports either digital output mode
  'or analogue output mode, but not both modes together).
  'Need to determine what audio output mode we are in...
  'The appearance of the volume control will depend on whether the headphone is
  'plugged in or we are in AUDIO_SPKR mode

  i = ChooseAudioOutMode (0)

  if (AudioOutMode <> i) then
    if (D > 0) then print "AudioOutMode was " AudioOutMode ", has changed to " i

    'If we are on the status or main screens and the audio output status changes
    'then we need to redraw because the volume control might need to appear or
    'disappear
    if ((Smode = Xstatus) or (Smode = Xmain)) then
      Change_Smode = Smode
      Smode = 0
    endif

    'Changes between analogue and digital are not immediately recognised by the
    'radio chip. The radio won't do anything until a new frequency selected
    'but we can force this if we select the same frequency as already selected.
    'If either the previous or the next state is digital, we need to perform
    'this frequency selection kludge
    if (((AudioOutMode = AUDIO_DIG) or (i = AUDIO_DIG)) and (NewFrequency < AmMIN)) then
      NewFrequency = Frequency
      if (NewServiceID < 0) then NewServiceID = ServiceID
      if (NewCompID < 0) then NewCompID = CompID
      if (D > 1) then print "Retuning to force an audio output mode change"
    endif

    AudioOutMode = i

    if (D > 0) then
      print "Changed audio output mode to " AudioOutMode "=";

      if (AudioOutMode = AUDIO_DIG) then
        print "Digital"
      elseif (AudioOutMode = AUDIO_RCA) then
        print "RCA"
      elseif (AudioOutMode = AUDIO_SPKR) then
        print "Speaker"
      else
        print "Unknown"
      endif
    endif

  endif

  '- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  'Enable PAM8407 speaker-amp when headphones removed and mode set to AUDIO_SPKR
  if ((AudioOutMode = AUDIO_SPKR) and (HP_removed <> 0)) then
    pin(P_PAM8407_ENAB) = 1  'set to 1 to turn on the amp

  else
    'Turn off PAM8407 speaker-amp when headphones plugged in
    'NOTE: the amp has a volume setting. But we will ignore this and use the volume
    'setting of the radio. All we need to do is turn the amp on and off. It will
    'power up at its default volume which is 2/3 max which we will just use as-is
    pin(P_PAM8407_ENAB) = 0 'set to 0 to turn off the amp

  endif

  '- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  ' This is only used at boot time after the radio has come up
  ' I put this code in the main loop rather than above, to speed up booting
  if (NumActivePresets < 0) then
    DirtyPresets=1

    'Calling compactpresets will sort the active from inactive presets
    CompactPresets
    SetFavs
    Change_Smode = Smode
    Smode = 0
  endif
  '- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  'The DispErrors variable is a positive number when displaying the error file
  'but it can either have the value '1' (waiting for user to touch the screen)
  'or '2' (user has touched the screen - display the next screenfull of data)
  if (DispErrors > 1) then DispErrFile
LOOP
end
'-----------------------------------------------------
'Si4689 loader.bin file

CFunction LOADERBIN      'not an actual Cfunction, but an array to read data direct from flash
  00000000               'offset in 32bit words, set to zero to start below
  'data starts here
  FF000010 00000000 00000000 12345678 00000000 00000000 6CCB84FD 00000001
  DEADBEEF 00001648 00000004 00000004 00000000 50200544 00000000 00000000
  00000000 00000000 00000000 00000000 00000000 076FCF88 8100006F 5000D800
  5000E048 5000E150 50002A00 502867AC 5028656C A0002E00 50002A80 502896A0
  FFF0FF0C A0001E00 4FFFFF80 502895E8 50000040 502892F8 50000050 50000060
  FFFF0000 00001003 502008A8 502008C8 50200874 502007E0 502009F0 50200660
  5000FE00 90200403 502008FC 50286778 50002A50 50002200 501FFFFF 50260000
  A0000E00 502897D4 50008E00 A003F300 FFFFEFFF A002BE00 50002534 500027E0
  5000E134 5000E130 502860A8 50285198 000FFFFF FFF00000 50286594 5000E050
  0000ED0C 0000DEC0 0000C0DE 50286658 5000E000 00000BB7 0000C350 A000BE00
  5000E0B0 5028A570 03000000 9FFFFE00 00001000 A00020C0 000F00F3 A00020D0
  A00020E0 A00020F0 A0001004 00000800 A000C080 5000E0A8 A000C090 0000FFFF
  A000C0F0 A000C010 A000C070 50201544 A000C100 A00010F0 A0001044 A0003100
  A0001060 A0001080 A00011F0 00C00000 00400000 00004E20 5028A6A4 C3500000
  00008808 0001FFE3 A0001008 EFFC37FF EFFC3FFF A000C0E0 00060000 00040000
  A000C110 A000C120 A000C060 00020000 5028A580 02000000 5000E020 50284C50
  5000E030 5000E140 A003DE00 000103F7 000103E7 00001017 00001037 81000011
  5000E000 030361A8 004B000B 00042002 000040C7 500E0000 00000000 00000000
  00000000 4649423C 5244485F 4345535F 0A3E4E54 49422F3C 44485F46 45535F52
  3E4E5443 0000000A 21000042 5000E048 8000050A 50200400 21004136 20C03202
  15229200 820020C0 98971422 0020C00E C01522B2 22A20020 F01AB714 C6000065
  0000FFF6 21004136 20C03202 14223200 C0A03320 23520020 0005521A 420020C0
  441B1A23 0C0020C0 321B0C4A 03811E23 30930B32 08E09339 2B452600 26465526
  75265065 6085265A 1587980C 6A952664 1597B90C CFA0A26E 811D15A7 1A0C3204
  0C0008E0 0002C61A 228203BD E004AD23 0A0C0008 1D004E25 8203BDF0 04AD2B22
  060008E0 03BDFFF7 AD242282 0008E004 BDFFF3C6 25228203 08E004AD FFF08600
  E0262282 EE460008 272282FF 060008E0 2282FFEC 0008E028 82FFE9C6 08E02922
  FFE78600 E02A2282 E5460008 000000FF A2004136 052180A0 0020C032 A0C42292
  20C02099 32068100 82C46292 88AC0808 20C01A0C CC229200 C02099A0 62920020
  320781CC 08E0DA0C C0EC7C00 22B20020 10BBC0CC B20020C0 F01DCC62 0C004136
  3203814A 08E00B0C 41170C00 09313208 32065132 92F2A062 05210805 C0F99C32
  22B20020 20BB70CC B20020C0 20C0CC62 9023A200 6010AA40 20C020AA 9063A200
  C10111E5 0CC2320A 35EC3789 B250A0A2 0B8104A0 320CC132 A10008E0 0D81320C
  320EB132 A20008E0 A0B260A0 320B8104 E0320CC1 0CA10008 320D8132 E0320FB1
  0A4C0008 0B814B0C 320CC132 C10008E0 11A13210 3202B132 920020C0 99C08422
  2099A010 A10020C0 16C13217 3215D132 F13214E1 62923213 32128184 F2286B82
  6BE22B6B 236BD225 A2246BC2 0592266B C0F99C08 22E20020 20EE70CC E20020C0
  20C0CC62 9023D200 6010DD40 20C020DD 9063D200 C1321AB1 18A1321C 32199132
  92321B81 4A0C5C6A 0C0008E0 3203814A 08E01B0C 00F01D00 81004136 20C03205
  D8288200 66348080 0FA50238 00006500 0000F01D 0C006136 3205B115 820020C0
  1D31D82B 34808032 0C2E3826 3203814A 08E00B0C 32048100 08E01A0C 3202A100
  520020C0 090C166A 0C0020C0 186A921B 0C320381 0008E04A 020CF01D 42D22392
  23C2CD23 321EF1D1 0C094CF6 90620CED 0106832D 904E0C00 BF47932E 321F8105
  5002B847 AEE22022 3220910F D20020C0 DDE0D82B 0020C010 C2D86BD2 20C0F0A1
  D82BA200 C020AAC0 6BA20020 0020C0D8 C09E6952 69520020 0020C096 65966952
  0AF10009 890FF232 8105EF57 08E03221 814A0C00 0B0C3203 C80008E0 A201B811
  2291D023 9D098D32 F2043D0A FF8002A5 86A0E211 FF40FFEA EAA1E211 0FA0FFEA
  00F01D00 81004136 0882320A 471A0C89 2D0C3D68 C03220B1 2BC20020 20CCD098
  C20020C0 20C0986B 982B9200 C02099A0 07810020 986B9232 820008E0 23F100A1
  0020C032 80201FE2 20C020EE 205FE200 0000F01D 21004136 2481320A 89022232
  27322541 20C00EE2 8C243200 C0103380 64320020 00F01D8C 7C004136 32054105
  320533B6 0086FDC3 00A03200 820020C0 8080D824 33286634 C220D330 26A103C2
  3227B132 1600DC25 20C004BA D824B200 C010BB50 64B20020 C03A0CD8 24920020
  2099A0D8 920020C0 F01DD864 B20020C0 BB50D824 0020C010 0CD864B2 0020C07A
  A0D82492 20C02099 D8649200 0C320481 0008E01A 1DC1F01D F32CC232 81FF4C16
  1A0C3204 1D0008E0 000000F0 B6004136 02821433 FFA0A200 920B98A7 02B20102
  0299A702 81151BA7 28A1322A 3229B132 0B99090C 03BD0A99 08E002AD 00F01D00
  31004136 05213202 0020C032 20D82222 22803420 0020C011 1D306322 000000F0
  0C008136 269387F8 0C60A0A2 320B814B 08E001CD 8101AD00 C1B2320D 0008E010
  90904198 8169CCF4 02AD322B 1D0008E0 000000F0 A8004136 201D0C12 6B98A0BA
  923205C1 20C00009 982C8200 89F48080 E8BA1BAB 660A0C52 42D90479 26000106
  32D90149 AD01BBE7 322CE10B 98A09A20 322DB169 C01099E0 2C820020 1088B080
  C0208890 6C820020 A942F880 B8AFDC12 A732C802 190C069B 004622D9 26229800
  1C260819 322E8105 0C0008E0 3203814A 08E01B0C 00F01D00 0C004136 3203814A
  08E00B0C 31040C00 20C03202 1423B200 20C0BB1B 1463B200 A20020C0 20C01923
  14239200 C00539A7 63420020 0020C014 9C1723C2 2212663C C0FFB365 63420020
  0020C017 06166342 20C00004 1623D200 B1E57D8C 0020C0FF 0C166342 3203814A
  08E01B0C 00F01D00 B6004136 2FA17033 00686532 410002A2 3A163202 023AE606
  82371AE6 7816F0CA EFCA9216 B2169916 2BA7F3A0 F0A0C205 D255AAC7 DAD0F4A0
  16CD16C0 E0FEA0E2 4E16C0EA FFA0F20D 16C0FAF0 020C113F 0C000506 0DB837E8
  C3C2B23B 0016A5FD A20020C0 120C3064 56007BE5 04810082 01A0A232 1D0008E0
  37A90CF0 B23BE7B9 65FDC3C2 F7060014 02B3F6FF C1002146 02B23230 0102D202
  D011BB80 9BC720BB 0FC2B273 92090282 02F20402 F1C3D205 E20E02C2 CC800D02
  20CCE011 800C02E2 CCE011CC 0B02E220 E011CC80 CCD020CC 0602E263 D010AFD2
  EE8080DA 20EEF011 F211EE80 EE900A02 03029220 8011FF80 FF8011EE 08028220
  9011FF80 FF8020EE 07028220 8011FF80 AFF020FF 00296520 D4860A2D 46120CFF
  73B6FFD3 3231E13F F20202D2 DD800102 20DDF011 A22D9DE7 02B20602 11AA8005
  B220AAB0 AA800402 20AAB011 800302B2 AAB011AA 003B2520 ABA01B0C 460A2D93
  0000FFC3 46120C00 32D1FFC1 0202C232 800102E2 CCE011CC 0F9CD720 0C003365
  93ADA01D B9860A2D 000000FF B786120C E5A23BFF B5060017 0202A2FF 800102B2
  AAB011AA 00102520 A20020C0 AF063064 FDC3C2FF 0E0C0D0C B20202A2 AA800102
  20AAB011 AA40B23B 001E2511 00FFA786 BD006136 7C04CD03 0C32CCFA 0003464D
  C208A0D2 C3B2FCC4 0061A204 2F510368 AD01A932 00706505 7A2505AD 41030C00
  20C0321D B6D5C800 2146023C 08235600 56F324D2 05AD07CD 16008425 0578FE3A
  01ADD28C 81A0C750 5CB83233 08E09CC8 5622F600 503226A1 27B1A0D7 052DC232
  65092DD2 3AA0009A 0CAAAC20 C0E5B91B 05F10020 D82F9232 99A00A7C 0020C010
  C0D86F92 2FE20020 80380CD8 20C020EE D86FE200 C20004C6 DC16F324 01A0A200
  8101A092 E5993204 AD0008E0 00822505 A8FFDB86 A7B28C01 04810916 E01A0C32
  01A80008 F01D0A2D BC004136 32345102 26341226 B34C3622 42361237 124701A1
  02A18235 92341287 129701A2 02A2A233 B23212A7 12B703A2 04A2C231 0C0312C7
  22F01D02 F01D0C05 1D001522 020522F0 1522F01D 22F01D03 F01D0305 1D040522
  080522F0 0522F01D 22F01D09 F01D0515 0C004136 81BA4C17 A1B23234 3235D101
  E201A2C2 A2F202A2 52B90C03 45160012 01126207 1516224B 551526FF 97442526
  15A73B15 2615B72F E71D15C7 15F71415 04A2320B 62439537 F3460C48 055862FF
  62FFF1C6 F0460948 084862FF 62FFEEC6 ED460348 643060FF 06035832 4862FFEB
  FFE98604 620546F6 E7460248 06070CFF BD67FFE6 32364105 0C04B467 FFE28607
  06005862 072DFFE1 0000F01D 0C008136 810B0CDA 61293203 032D5169 043D6178
  4DF8C262 E0025D05 31490008 41292169 06BD14AC 01AD2169 F97CC38B 99323381
  0008E001 6A1731A8 B851C80B 041BC701 2C46020C 32376100 0C068316 FFA08204
  52000522 128701C5 01A0A250 C1001665 0CC23234 0020C008 C2322F91 B0705849
  0020C0F5 705949B2 20C041A8 5A49A200 720020C0 20C05B49 5C492200 E505A0A2
  20C00044 CC26D200 C0086D07 26E20020 F6EE07CC 070014E5 771BF9EA 1BFFA082
  9B934744 68073188 BD61A83B E541C803 20C0001B CC269200 C0086907 26A20020
  F6EA07CC C38B21B8 F97CA14B 99323381 0008E011 11C81A0C 020C01B8 B0C0BBC0
  0046832A 0C120C00 320381DA 08E01B0C 00F01D00 E5004136 1A0C000E 0C000B65
  0013252A A0000DE5 288C7480 F01D020C 0A251A0C 32349100 92322FA1 20C00C09
  584A9200 3A651A0C 32379100 B20020C0 6B07CC29 0020C008 07CC29C2 0A65F6EC
  F9EA0700 F01D120C A5004136 1A0C0009 0C000625 000DE52A A00008A5 288C7480
  F01D020C 04E51A0C 3234C100 C2322F91 20C0090C 5849C200 C0F5B020 49B20020
  41A82059 A20020C0 20C05A49 5B492200 33A54A0C 32379100 D20020C0 6D07CC29
  0020C008 07CC29E2 03A5F6EE F9EA0700 F01D120C 0C004136 91480C6A 8A20322F
  0020C093 49821A0C 00302558 C0323791 29B20020 086B07CC C20020C0 EC07CC29
  00F01DF6 21004136 2CB13237 32383132 B0322DC1 20C010B3 8022A200 B010AAC0
  20C020AA 8062A200 0CA0C392 0020C058 1B0C1A0C 65584982 20C0002E CC22C200
  C0086C07 22D20020 F6ED07CC 220020C0 F01D0003 0C004136 322F8119 920020C0
  20C05848 222A0C00 28255948 32379100 B20020C0 6B07CC29 0020C008 07CC29C2
  F825F6EC F9EA07FF 0000F01D A1004136 03BD3234 C1040A92 C9CC322F C0030A82
  4C820020 C64A0C58 20C00003 584C9200 20C00A0C 5C4CA200 80205A0C 3237D1F5
  820020C0 F820594C 0020C041 C05A4CF2 4C220020 322CE15B E0322DF1 20C010E4
  802DC200 E010CCF0 20C020CC 806DC200 1D0022A5 000000F0 0C004136 318C5C0B
  3981322F E003AD32 3AB10008 323BC132 910020C0 6CB2323C B13A0CA2 A0C2323E
  9932A9F2 323DA142 A1004F25 3EB1323F F2A0C232 A1004E65 3EB13240 F2A0C232
  A1004DA5 3EB13241 F1A0C232 A5004CE5 42A10006 3243B132 A1004E25 2CB13244
  3245C132 A1004B65 47B13246 E57C0C32 48A1004A 0C3B0C32 004A251C B23248A1
  0C0C70A0 A1004965 42C83249 0B3247B1 0048A5CC B2324AA1 1C0C73A0 CD0047E5
  81DA0C03 4BB1321B 0008E032 1C324CA1 A50C0CCB F01D0046 A1004136 10B1324D
  A50C0C32 4EA10045 3243B132 310046E5 13823223 32342120 A1376887 A4B2324F
  00A4C200 81004365 A3A23207 0008E0E8 B13250A1 43C13243 00422532 3C324FA1
  A50C1C0B 51A10041 0CCB0C32 0040E50C A1001586 53B13252 3254C132 B2003FE5
  55C10012 B71A0C32 AAF00B3C 11BBF011 46F6BCB7 1A0C0000 0B325681 0E42929A
  E03257A1 DAB20008 41BCB008 B211BBF0 A2C20D42 74B0B000 C020BBC0 53B20020
  3258A190 A20020C0 4CA18053 3259B132 D20202C2 E1C00E02 11DD2004 F004C0C0
  DDE011EE 20CCD020 1D003925 000000F0 A1004136 3EB1323D 258C0C32 40A10038
  323EB132 37658C0C 3241A100 0C323EB1 0036A58C B1323FA1 8C0C323E A10035E5
  43B1325A 00376532 0381DA0C E00B0C32 5BB10008 3220A132 B20020C0 5C91886A
  0020C032 1D886A92 000000F0 A1004136 47B1325D 650C0C32 4CA10032 325EB132
  A5325FC1 02CD0031 7C3260A1 0030E5FB 0C3261A1 651C0C1B F01D0030 A1004136
  47B1325D 650C0C32 62A1002F 3247B132 2EA50C0C 324CA100 C1325EB1 2DE53263
  3260A100 C280FB7C 20C3C001 A1002CE5 1B0C3261 2C651C0C 00F01D00 61004136
  A0823238 0D384780 06AD03BD CD326481 0008E004 0E0C063D 0ECD02DD 390020C0
  E9E2E9F2 0020C0D2 C0106242 62E20020 0020C012 C01362E2 62E20020 1562E214
  0C3202A1 1AA97649 B20020C0 AA4B1A2A F8071367 C0FF50F2 1B051BF7 4B5DB9CC
  C9F03DDD C102E932 42C9323C C03237A1 42B80020 BB0B12E9 B03210C1 20C0F4B0
  842A9200 B01099C0 20C02099 921B0C00 0381846A E0DA0C32 F01D0008 C0004136
  D2880020 6856E298 0A39560A A20020C0 2A561222 0020C004 881022B2 378BB6F2
  20C00888 11628200 1FF8F2F8 E80020C0 1262F2F2 F2E9EE8B D20020C0 CDD21022
  0020C0F8 C01062D2 22C20020 917CCC12 20C03265 12629200 A20020C0 1C0C1222
  C004EA16 22820020 0020C012 12C84238 20633380 5CC8A0CC A20020C0 03BD1122
  C0FFC0A5 22B20020 C0BB3A11 62B20020 0020C011 301222A2 20C0C0AA 1262A200
  920020C0 991B1322 920020C0 F01D1362 20C0F01D 1DD2C900 000000F0 C0004136
  D2880020 0C0338B6 0CF01D02 320381DA 08E00B0C C0130C00 12A80020 99A70298
  0020C009 0B0C22C8 0C833BC0 320381DA 08E01B0C C063CC00 D2980020 C0CB29B6
  D2A80020 3AB6120C 1D020C03 00F01DF0 0C004136 920B0CDA 03811522 92991B32
  08E01562 A8030C00 1B32B802 B702A9AA 0239013A B80020C0 A19B8C22 EBE5322F
  0020C0FF DA0C2239 0C320381 0008E01B 20C03A0C B8D29800 2429F6E2 20C01BAC
  1322D200 C20020C0 BCD71422 0020C00E C01322F2 22E20020 F03EF714 A90020C0
  0020C0D2 1866D288 0020C015 C01322B2 22920020 B72C0C14 20C00439 C0D2C900
  D2D80020 C0102D66 22F20020 1522E214 C0043EF7 D2A90020 0000F01D D1004136
  3BE1323A 0020C032 C0A26ED2 12B80020 203237C1 20C0A0DB 982CC200 C0C00A0C
  889DC9F4 87BB1B32 0BAD01BB A90020C0 0020C012 02F81288 9F87190C 0020C007
  01462299 322FA100 C0FFDF25 22A20020 C0AA1B14 62A20020 00F01D14 40004136
  F77C1063 C0307370 02580020 60105570 20C02055 1D025900 000000F0 C0004136
  02390020 0000F01D 29006136 AC013911 326631F5 E2322971 C7C210C3 0C079810
  26E9AC06 29667F19 A805DD15 8101B811 04CD3267 E10008E0 69C13268 56050C32
  11A8FDA5 2AA2190C A2020C27 29A0F8CA D8F01D83 3A0F0C17 E2B6572D 0BFDFF56
  B29DCA55 02820004 1B441B00 B2DD1B22 17D90049 A81F1B87 8101B811 69C13267
  0008E032 C13268E1 290C3269 1F0C0799 17D90D0C 26000086 661B04BD 0CFFEE46
  C607A91A 0F0CFFF9 060C17D8 B6572DEA F8AF568D 661B550B 820002B2 441B0004
  2D081B87 D90D0C0E 00040617 191C221B 17D9DD1B 0C059D97 A92A0C1F FFF34607
  41004136 0482320A 04848089 C216D816 052C0FAE 6A31023C 3205A132 B20020C0
  BBC0D82A 0020C010 C0D86AB2 2A920020 209920D8 920020C0 20C0D86A 88635200
  1C320781 0008E0EA 220020C0 07818863 E0EA1C32 A2920008 0020C030 81886392
  EA1C3207 920008E0 20C030A3 88639200 1C320781 0008E0EA C0326B91 63920020
  32078184 08E0EA1C 326C9100 920020C0 07818463 E0EA1C32 A1920008 0020C030
  81886392 EA1C3207 910008E0 2321326D 0020C032 81385292 EA1C3207 910008E0
  20C0326E 38529200 1C320781 0008E0EA C03FA392 52920020 32078130 08E0EA1C
  1CA19200 920020C0 07812052 E0EA1C32 A1920008 0020C010 81886392 EA1C3207
  C00008E0 52520020 32078120 08E0EA1C 08A19200 920020C0 07816852 E0EA1C32
  991C0008 920020C0 07815852 2CA1A232 920008E0 A1B28A04 A2A9CC00 20C01BA1
  7052A200 B0000206 20C020B9 7052B200 1C320781 0008E0EA 20A1DD7C 0020C032
  D0982AC2 20C010CC 986AC200 20C0EB7C 982A9200 C01099B0 6A920020 00F01D98
  00000000
End CFunction
