'Alternate firmware for the Silicon Chip DAB+ radio (version 1.0)
'(Original articles published in Jan-March 2019 editions of Silicon Chip Magazine)
'
'It incorporates some elements of the original SC firmware, with thanks and gratitude to
'the original authors Duraid Madina and Nicholas Vinen.
'
'Copyright (C) 2020 Stefan Keller-Tuberg (skt at keller-tuberg.homeip.net)
'The original firmware published on the SC web site does not contain a copyright notice.
'It is presumed to be (C) of the original authors Duraid Madina and Nicholas Vinen under
'the GPL license version 3
'
'Google 'Silicon Chip DAB+ radio firmware' for a copy of the original firmware
'
'This alternate firmware for the Silicon Chip DAB+ radio is free software: you can
'redistribute it and/or modify it under the terms of the GNU General Public License
'as published by the Free Software Foundation, either version 3 of the License, or
'(at your option) any later version.
'
'The alternate firmware for the Silicon Chip DAB+ radio is distributed in the hope
'that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty
'of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public
'License for more details.
'
'You should have received a copy of the GNU General Public License along with
'the this file.  If not, see <http://www.gnu.org/licenses/>.

'-----------------------------------------------------
'This file is too large to load into the limited flash on the micromite plus board.
'You will need to CRUNCH it to reduce its size.
'
'If you don't have an windows computer, (i.e. you have a Mac or a Linux computer) then
'here is a bash script to crunch this program without having to run MMEDIT in a virtual
'windows machine.
'
'#!/bin/bash -noprofile
'
'if [[ -z "${1}" ]] || [[ -z "${2}" ]] || [[ "${1}" == "${2}" ]] ; then
'  echo "Usage: `basename ${0}` <source> <destination>"
'  echo "CRUNCH source file and write back as destination file"
'  exit 1
'fi
'
'cat "${1}" | \
'    sed -e "s/'.*$//g" | \
'    sed -e '/^[[:space:]]*$/d' | \
'    sed -e 's/^[[:space:]]*//g' | \
'    sed -e 's/[[:space:]]*$//g' | \
'    sed -e 's/[[:space:]]*\([<=][=>]\)[[:space:]]*/\1/g' | \
'    sed -e 's/)[[:space:]]*and[[:space:]]*(/)and(/g' | \
'    sed -e 's/)[[:space:]]*or[[:space:]]*(/)or(/g' | \
'    sed -e 's/if[[:space:]]*(/if(/g' | \
'    sed -e 's/)[[:space:]]*then/)then/g' | \
'    sed -e 's/[[:space:]]*\([=><+*\/()-]\)[[:space:]]*/\1/g' | \
'    sed -e 's/is\([><]\)/is \1/g' | \
'    unix2dos > "${2}"
'
'-----------------------------------------------------
'READ THIS SECTION FIRST
'Getting this programme running on your hardware:
'
'This programme doesn't contain code to load the 32 Mbit flash on the radio board.
'Use the original SC programme to perform this task. Once the 32 Mbit flash is
'programmed and working with the SC programme, this code will work also.
'
'The SC code sets the SPI bus to run at 2 MHz. This programme sets it to 10 MHz. My HW
'wouldn't work at either rate with the SC software until I added pull up resistors
'onto the radio card. If you experience issues with this software not detecting the
'radio chip or TOS framer chip, try reducing the SPI rate and consider adding pullups.
'
'I've set the CPU speed to 120 MHz. This works fine in my setup. If you experience
'issues, it may be that you need to reduce CPU clock rate. The SC software configures
'the CPU to 80 MHz. This code will work OK at 80 MHz - but some things like the sorting
'of the presets file will take noticeably longer.
'
'When running MMBASIC version 5.5.03, both of my micromites would throw bizarre errors
'from time to time - every hour or so. This didn't happen with either version 5.05.01
'or 5.05.02. The errors looked real (eg 'bus error' or a 'font error etc) except there
'was no error! On a hunch from experience with other PIC32 devices, I added a 47uF
'tantalum cap in parallel with the 10uF cap adjacent to the PIC. This fixed the
'problems. Either the 10uF cap specified in the data sheets is too small, or the
'10uF capacitors supplied in SC kits are not up to the task, or something else....
'but you can fix these problems by adding a 47u cap in parallel as mentioned.
'
'This programme assumes there's an SDMMC flash card plugged permanently into the
'Micromite or LCD card slot. Any old small SDMMC card will do. A couple of Mbyte will
'be more than enough. The flash card is used to keep a running list of station presets
'(in a preset file) and some settings from the last session (eg last radio station,
'analogue/digital output), and if there's a radio chip comms issue, a log will be
'appended to an error file recording some critical status bytes from the Si4689.
'-----------------------------------------------------
'The user interace and some of the functions:
'
'The user interface has been designed so that controls / menu are hidden under normal
'operation (to unclutter the screen). Touch the screen to reveal the main menu. A
'close box is provided in the upper left corner to return to the previous screen. A
'timer will automatically close most menu screens and return to the previous screen
'if there is no input for a certain amount of time.
'
'When running this programme for the first time, you may want to run the setup option
'and select 'Station Scan'. This will scan the Australian AM, FM and then DAB bands
'to look for 'stronger' signals, populating the Preset0.csv file. Depending on the
'vagueries of signal levels and noise, some non-stations may be recorded, and some real
'stations may be missed. Refer to discussion below on AntCap settings for one reason for
'these 'mistakes' when scanning. You can edit these 'mistakes' later.
'
'The configuration and presets are saved as CSV files on the SDMMC flash card so they
'can be edited offline in a spreadsheet, or copied to a different micromite board. The
'preset settings can also be edited on the radio itself using the 'Edit Presets' menu
'option. In the presets editor, you will find the option to 'hide' or 'show' each
'preset. This relates to whether the setting will be visible in the 'Tune to Preset'
'menu function. If you choose to hide a particular station, its information will still
'remain in the presets list, and (for example) its station name will be displayed if
'you tune to its frequency, and its AntCap setting will be applied. You can always
'unhide an entry later. When editing presets, you can clear (delete) an entry. The
'resulting blank/hole in the list will be removed when you exit the editor.
'
'From the setup menu, you may also want to set the backlight level, and choose whether
'to activate the analogue or digital outputs. If the analogue output is selected, a
'volume setting appears on the main menu window. After a short delay, these settings
'are saved to a file called Config0.csv on the SDMMC card, along with the station
'to which you are currently tuned. When you change stations, the Config0.csv file is
'rewritten to remember the last station.
'
'The IR remote control codes in this programme are not the same as in the orignal SC
'programme. I had a different remote control unit on hand and could not replicate the
'codes supported by the altronics and jaycar devices. This is not such a big deal as the
'codes are trivial to change. There is a function under the setup screen to help discover
'the device ID and key codes for any remote control supported by MMBASIC. When MMBASIC
'receives a valid code, it will be displayed on the LCD and, if you take notes, it will
'be straightforward to edit the codes in 'ProcessIR' below.
'-----------------------------------------------------
'Loading a new version of this software onto the radio:
'
'There's a built in semi-automated way of loading a new version of this BASIC programme.
'
'Upon power-on, this programme will look for a file called 'Radio0.bas' on the SDMMC
'flash card. If a file with that name is found, it will be renamed to Radio1.bas (and
'all earlier files called RadioN.bas will be renamed one number up). The new basic code
'will be loaded into the micromite and run without further intervention. In this way,
'new BASIC software can be loaded via SDMMC without a serial interface being attached.
'
'So once this programme is copied onto the micromite for the first time in the
'traditional way, you can edit the code on your computer and transfer the new code to
'the radio by copying the new version to a file called 'Radio0.bas' on the SDMMC card.
'Then power cycle the radio and it will load the new version of code as it boots.
'-----------------------------------------------------
'Bugs:
'
'I have only tested this software on my own DAB+ radio card. My card includes all
'optional chips but I have not tried to use every capability - eg on-board amp is not
'exercised fully.
'
'The programme works OK on my hardware - touch wood. From time to time, the radio crashes
'(for example, when there's a big RF surge due to the gas stove ignitor). The issues
'seem less common when the incoming radio signal is strong. I am reasonably confident
'the noise is entering the radio via the antenna, not the power supply, especially
'because of the kind of error that the radio returns in its status messages. When this
'programme recognises such a condition, it will reboot the radio.
'This seems to be a bug with the radio, not with this basic programme.
'
'There is of course a chance there are other bugs that I've not found. I apologise
'for those too. Like you, I am human and the only thing I can guarantee you is that my
'code contains bugs I have not yet found. ;-) If in doubt that this code is working for
'you, substitute the Silicon Chip code and if that works, its clear you've found a bug in
'mine.
'-----------------------------------------------------
'Frustrations writing this version of the software:
'
'It seems that those that start working on Si4689 software are doomed to frustration,
'eventually. The original authors comments in the magazine article hinted at this. You
'can take it from this comment that my own experience aligns with theirs. But perhaps
'for different reasons. All things considered, the Si4689 and its documentation
'aren't so bad! (Although I have never been able to find decent electrical specs...)
'
'I was motivated to write this programme independently from the SC code in order to test
'the theory that the Si4689 doesn't support digital output for the DAB+ band, as
'the original authors had concluded. The Si4689 programming guide doesn't mention this,
'so I was a skeptical and hopeful for a fix. I started trying to modify the SC code, but
'as I was trying to closely follow the flow charts in the prog guide, it was easier to
'write the state machine you will find below in 'CheckSpi' as completely new code.
'
'To my joy, I discovered that the DAB mode will indeed output digital data, and this
'makes me very happy because my 2013 SC 'crystal dac' sounds really awesome. I am unsure
'what the issue with the origianl SC basic programme is, and I have not been able to fix
'it in the original firmware. I think the root cause may be with SPI message sequencing.
'Turn on D=3 debugging in this code and trace the Si4689 message sequence. It is somewhat
'different to the sequence followed in the original SC code.
'
'My frustrations weren't with the DAB digital output, but included the AntCap
'(antenna capacitance) values which I comment on below.
'-----------------------------------------------------
'Improving received signal strength and SNR/CNR:
'
'I found that I could get >3 dBuV of additional signal strength and better SNR in the FM
'and DAB bands by applying optimised AntCap values. This programme includes an AntCap
'optimising algorithm based on that described in the app note. Some further explanation
'is required.
'
'After many 'goes', I found that the 'optimal' AntCap settings critically vary with small
'changes in the antenna setup. This ultimately was one of my sources of frustration,
'because it took me quite a while to work out that the variation was due to the changes
'in antenna cable lengths. The Si4689's 'optimal' settings change as the antenna feed
'characteristics change (ie the capacitance and inductance of the combination of the
'antenna and the feed cable). This means that you need your radio in its final (normal
'usage) location connected to the home's VHF antenna feed to properly set the AntCap
'configuration values.
'
'This suggests the hardware design would benefit from a wideband unity gain amp between
'the external antenna input and the radio chip, so that the radio chip always sees the
'same characteristics and the AntCap values can be set once-and-for-all for this design,
'i.e. so every implementation will behave the same as every other implementation. This
'would make it easier to terminate the external antenna into a nominal 75 Ohm resistive
'load to eliminate reflections back into the home wiring (which cause nulls through the
'spectrum). Anyway, too late for that for this particular design.... Maybe Mk II?
'
'It is overwhelmingly probable that YOUR implementation will work best with different
'AntCap values than MINE (and probably different values than by default).
'
'I suggest you experiment with this feature to see if you can get a benefit. If not, then
'don't worry about it or use it. If, you get a signal boost, then some one-off homework
'will be required. You will need to run the AntCap algorithm for each frequency you
'wish to keep in the presets file. At the conclusion of the algorithm, the optimal
'AntCap value that was measured will be written to the presets file for that frequency.
'As it takes several minutes per frequency, you'll need to be persistent to measure
'optimal AntCap values for all stations. There is however a shortcut you can take.
'
'Start by measuring an AntCap value for an FM station near the bottom of the FM band.
'Then work your way up the FM band in a few steps. You can edit the presets file offline
'in (eg microsoft excel) and interpolate the optimal settings for other stations between
'the ones you actually measured. This way, you'll get close to optimal without having to
'do every station. (You can just run the algorithm for every station too.)
'
'The Si4689 kind-of does a kludge like this too. The app note says to measure AntCap
'values for a number of frequencies across the band, and then plot the points. When you
'plot the points, you will find the line is kind-of a straight line (not quite!) but if
'you assume a straight line and calculate the slope and Y intercept, you can then
'configure the Si6849 with a number representing 1000x the slope, and another number
'representing the Y intercept, and the chip will guess the value of AntCap to use based
'on the straight line you told it about. I chose to implement explicit values for each
'station - no approximations are required - you get the precise optimal AntCap setting
'for each frequency.
'
'The second frustration with AntCap settings is that the radio chip seems to read-back
'different AntCap values than you programme, even though it is clear that your
'programmed values are improving the performance. Sigh. You won't notice the differences
'unless you enable D=2 debug mode and watch the status reports come back at you
'reporting the AntCap value the radio says it has applied. I am not sure if this is
'an Si4689 bug (reporting different values than has been programmed) or a 'feature'.
'
'My AntCap experiments seem to suggest that the values for L1 and L2 and L3 and
'associated capacitors in the SC design may not be optimal for this PCB layout. I say
'this because the optimal AntCap values hit their absolute minimum configurable value
'mid-band rather than at the very top of the FM and DAB bands. According to the
'app note, components should be chosen so that the AntCap function hits a minimum at
'the top of the FM (and DAB) bands. I have not tried to experiment with alternate
'component values because of the difficulty working in this section of the board. However
'it seems likely that gains might be found by experimenting with these component values.
'Maybe not worthwhile unless there's a wideband buffer amp designed into the circuit
'first.
'
'Additional suggestion that L1, L2 and L3 are suboptimal comes from my experience with
'FM RDS and AntCap. I have mentioned I live in a marginal reception area. Also, all
'local FM broadcasts are actually rebroadcast from a local repeater - we cannot receive
'the original broadcast from Black Mountain Tower due to the mountainous terrain.
'It seems that for most stations, the rebroadcast FM generally excludes RDS, so there
'are few signals with which I can test the RDS. We have one signal for which I am
'certain there is an RDS subcarrier - 1t 107.1 MHz - the very top of the FM band.
'Because the optimised AntCap setting reaches absolute minimum well below 107MHz,
'the signal strength at 107 is well down, lets say >10dB down but potentially much
'more, than the 'optimal' receive values lower down the spectrum. Whilst I previously
'mentioned that neither the SC software nor this software receive RDS reliably, I will
'now add that when running this software, the radio reports around an additional 3dBuV
'compared with the SC software, and the number of RDS interrupts at 107.1MHz is
'noticeably greater than the number of interrupts received by the SC software. This
'suggests to me further corroboration that the analogue front end of this design
'would benefit from optimisation, including a unity-gain wideband amp to decouple
'the external antenna characteristics from the internal design, and also benefit from
'carefully chosen optimal component values for L1, L2 and L3 different than the current
'component values.

'The following constants are up for customisation... these values are the values that
'worked best for my radio / antenna setup. I would recommend tuning to a frequency
'in the middle of the FM band and running the AntCap optimisation routine with the
'switch set to 0, then try again at the same frequency with the switch set to 1. Then
'set the constant to the value that gives the greatest optimised signal strength.
'You will then of course need to run the AntCap optimisation algorithm for FM stations
'at either end of the band, and if the range of optimised settings will work for you,
'run the AntCap algorithm for a number of stations all across the FM band to optimise.
'You'll find fewer DAB frequencies to experiment with. Choose the DAB switch setting
'that gives the best overall performance on all of the DAB frequencies that are
'accessible at your location.

const FmSwitch=0, DabSwitch=1
'-----------------------------------------------------
'Some example AntCap measurements:
'
'The Silicon Labs reference design AntCap defaults are as follows: (from AN851 page 24)
'FM: Slope=0xEDB5, Intercept=0x01E3, Switch=Open
'    (Note: App note says closed for FM, but the results are better with it open!)
'DAB: Slope=0xF8A9, Intercept=0x01C6, Switch=Closed
'
'The chip applies an AntCap value calculated using the following equation
'  AntCap = ((slope/1000) * (freq expressed as MHz)) + intercept
'  FM and DAB: Capacitance = (AntCap-1) * 250 fF (fF=femto farad - 1000th of a pF)
'  AM:         Capacitance = (AntCap-1) * 142 fF [from AN649 p269]
'
'In one of my test configurations, the AntCap algorithm returned the following
''optimal' results:
'87.6MHz: 20.75pF
'88.7MHz: 17.5pF
'90.3MHz: 11pF
'91.9MHz: 6.75pF
'94.3MHz: 1.5pF
'For the results above, the line of best fit is of the form Cap = (-11.6 * Freq) + 1096
'which equates to a slope setting of 0xD2B0 and a Y intercept setting of 0x0448
'
'Changing only the length of the coax feed between the folded dipole and the radio, the
'measurements change as follows:
'88.7MHz: 26.75pF
'89.1MHz: 25.75pF
'93.7MHz: 16.0pF
'96.0MHz: 12.0pF
'99.1MHz: 8.0pF
'100.7MHz: 6.0pF
'103.9MHz: 4.5pF
'And for these points, the line of best fit is of the form Cap = (-6.6 * Freq) + 684
'This equates to a slope setting of 0xE638 and a Y intercept of 0x02AC
'
'(Of course, every time I changed the antenna arrangement, I got totally different
'numbers. I could repeatedly measure values for the same arrangement and get the same
'results, but changing the arrangement immediately led to different 'optimal' results,
'which themselves were repeatable. So only test when you finish your radio and it has
'its long term antenna connected as you want to leave it)
'-----------------------------------------------------
'Another frustration with the Si4689 included its digital output mode in all three bands:
'AM, FM and DAB+ alike. I found the Si4689 has issues with sending unwanted noise. The
'radio chip outputs rubbish too often and easily, and I think the developers might have
'been able to do a better job avoiding this. There are some software work arounds in this
'code that try and manage it. I have used the WM8804 as a kind of brute force firewall,
'forcing that device into reset mode at the times I found the radio chip most likely to
'output uncontrolled non-audio. Essentially, the WM8804 acts as a hardware mute for the
'digital output. Its ugly, but it improves things.
'-----------------------------------------------------
'Physical construction of my particular radio:
'
'I have built my system into a 3RU enclosure to use as a 'hifi' tuner for my sound
'system. My intent was always to use only the digital interface for all bands. This is
'how I am presently using the radio - no analogue connections whatsoever.

'I use a 75ohm RG-6 coax feed from the home television distribution amplifier and a
'rooftop DAB+ folded dipole antenna. I made my own AM loop with about 10 turns of
'hookup wire around a tin of milo, with 8 cable ties around the circle to hold it all
'together. Then remove from the milo tin and connect to the AM input of the radio.
'The AM signal levels far exceed the FM and DAB signal levels from the rooftop antenna.
'
'The SC design sandwitches the LCD, micromite and the radio board onto one assembly.
'I found noise from the LCD display breaks into weaker AM and FM stations. I used several
'strategies to reduce the noise, including extra power supply filtering, clamp on
'ferrites, and I separated the DAB radio board away from the micromite and LCD with a
'short length of 40 conductor ribbon and used a clamp-on ferrite on this ribbon.
'
'Other than reduced noise, separating the radio from the micromite with a short ribbon
'made no difference to the behaviour of the radio. Those problems I've described I
'experienced with my firmware were also present with the Silicon Chip firmware when
'the radio was sandwitched together as per the original published design.
'-----------------------------------------------------
'Hardware modifications to the original design:
'
'I have made some hardware modifications to my DAB+ card in order to reduce audible noise
'in the analgoue (and / or digital) outputs. Particularly:
'(i)   I have hard wired the -5V inverter so that it is always enabled by cutting the
'      track connecting CON8 pin 35 and REG4 pin 1, and then grounding REG4 pin 1. The
'      article warns of latch up. I have never experienced any issues with latch up with
'      the DAB radio and this modification.
'
'(ii)  I have changed the value of the the electrolytic capacitors connected to FB1 and
'      FB2 to 4.7uF to reduce the 'cracks' that occur when changing bands (reloading
'      firmware onto the Si4689). I've extended the length of the software mute also.
'
'(iii) I have added 150uF tantalum capacitors at the 40 pin edge connector between pins
'      1+3 and 1+5. (i.e. across +5 and ground, and +3.3 and ground). I added two of
'      these to the DAB radio board, and I added another two to the micromite - that is,
'      I added four 150uF tantalum capacitors in total - two on either side of the 40
'      wire ribbon.
'
'(iv)  I constructed a DC input voltage monitor to look for loss of voltage that
'      indicates power has been turned off. This drives an interrupt line to the
'      micromite which uses it to mute the outputs. I found that the Si4689 is really
'      prone to locking itself up and outputting random noise. This seems to happen
'      about one in 30 power-downs?? These are flagged by the chip itself as 'fatal
'      errors' and according to the AN649 ap note, this is DSP related. These fatal
'      errors don't appear to be caused by this programme, but by the Si4689 firmware
'      itself, triggered by susceptibility to strong RF noise or PSU noise??? If you
'      find a fix, let me know! Note: The code that supervises the voltage monitor is
'      enabled by setting the constant Pv to non-zero (below). If you don't implement
'      this hardware, set Pv to 0.
'
'(v)   I added 200 ohm at 10MHz clamp on ferrite sleeves on the 5V micromite supply and
'      on the short ribbon cable I used to connect the micromite to the DAB+ radio board
'
'The following change is not noise related:
'(vi)  I added 3k3 pull up resistors between the 3.3V pin on CON3 and the DAB+ radio side
'      of the three 47R resistors. Without these, the SPI interface would not communicate
'      with the Si4689 nor the WM8804 reliably. This happened both when the three cards
'      were sandwitched as per original design, and with the short ribbon cable I used
'      to separate the micromite and the radio board for noise improvement. I tried with
'      two different micromite boards, but only the one DAB radio card. Now the pull-ups
'      are in place, I've not seen a single SPI failure since.
'-----------------------------------------------------
CPU 120 'I was worried about noise, but running CPU quickly makes no apparent difference
'-----------------------------------------------------
option autorun on
option explicit 'help avoid bugs of auto-defined / misspelled variables
option base 0 'arrays start from 0
option default none 'don't assume a variable type
option autorun on 'autorun this upon powerup rather than going to prompt

'Recommend setting baud rate to 230400 - i.e. option baudrate 230400
'Set display size to your screen size - eg option display 65,130
'-----------------------------------------------------
'The following constant is a debug flag.
'Set it to 0 to turn off debug messages. 1=INFO only, 2=MIDDLE, 3=ALL
const D=0 'Recommend keeping as 0 unless you are specifically debugging/modifying code

'The following constant enables a voltage monitor interrupt. Unless you have specifically
'added a hardware mod as described in comments below, set this to 0
const Pv=0
'-----------------------------------------------------
'Set up hardware

'Si4689 is the radio chip
'Flash is the 32Mbit flash associated with the Si4689 radio chip
'WM8804 is the TOSLINK formatter
'MUX is the 4052
'PAM8407 is the on-board amplifier
'REG is the LM2663 Voltage Regulator used for analogue audio output
'POWER_ALARM is an optional modification I incorporated as described in comments below

const P_SI4689_SMODE=44, P_SI4689_RES=74, P_SI4689_CS=80
const P_Flash_SPI_MISO=88, P_Flash_SPI_MOSI=90, P_Flash_SPI_CLK=91, P_Flash_SPI_CS=92
const P_WM8804_CSB=59, P_WM8804_IFM=60, P_WM8804_RST=61, P_WM8804_SPI_MOSI=72
const P_MUX_0=25, P_MUX_1=24
const P_PAM8407_ENAB=68, P_PAM8407_UP=67, P_PAM8407_DN=66
const P_REG=21, P_HEADPHONE=14
const P_POWER_ALARM=34 'Hardware mod: low for power OK. Open circuit for power drop

'Enumerated states of the analogue audio MUX
const MUX_MUTE=0, MUX_NORM=1, MUX_REV=2, MUX_EXT=3
const MODE_AM=0, MODE_FM=1, MODE_DAB=2

'These constants define the limits of the radio bands, and the channel spacing
'AmMIN could be 520, but rounded to the nearest 9kHz to align with Aus channel spacing
'Note for FM and DAB. The minstren constant (minimum signal strength) is used by the
'scan algorithm. Because the scan is undertaken BEFORE the AntCap value has been
'optimised, it is important to select a relatively low signal strength - this has the
'unintended consequence that more dodgy / invalid frequencies will be flagged as
'valid stations.
const AmMIN=522, AmMAX=1710, AmDELTA=9, AmSNR=6, AmMinStren=50, FmMIN=76000, FmMAX=108000
const AFmMIN=87500, FmDELTA=100, FmSNR=6, FmMinStren=25, DabMIN=174928, DabMAX=239200
const ADabMIN=199360, ADabMAX=211648, DabCNR=4, DabMinStren=35
const Runtuned=0, Rtuned=1, Rselected=2, Rplaying=3

'This is a delay used to wait for a newly tuned channel to settle before reading stats
const SnrSettle=40 'ms (default is 40ms)

'The default silicon labs AntCap numbers, the same as those used by the SC programme
const FmSlope=&HEDB5, FmIntercept=&H01E3
const DabSlope=&HF8A9, DabIntercept=&H01C6

'These constants relate to the presets that are stored and displayed
const MAX_DAB_FREQS=41, MAX_DAB_SERVICES=32
const MAX_PRESETS=90 'More presets can be remembered than can be displayed
const MAX_RADIO=30 'must be multiple of 3 - 3 columns of radio buttons on the LCD
const RADIO_RADIUS=16

const MUTE_RELEASE=170 '170x5ms ticks
const HEADPHONE_COUNT=40 'number of times we need to detect headphones (debounce)

'The following enumerations represent the LCD state machine
'These enumerations are the different screens/windows
const Xconsole=1, Xstatus=2, Xmain=3, Xedit=4, Xselect=5, Xsetup=6, Xscan=7, XIRTest=8
const Xcap=9, XstatusE=10, XmainE=11, XscanE=12, XcapE=13

'This holds the max number of bytes that can be received from radio chip via SPI message
const READ_REPLY_BUFSIZE = 1028

'These constants define a moving average range for stats polled from the radio chip
'The stats move about rapidly, so a moving average is a good way of making better sense
'multiplier of 0.975, step change hits 50% in 28 samples
'multiplier of 0.95, step change hits 50% in 14 samples
'multiplier of 0.9, step change hits 50% in 7 samples
'multiplier of 0.85, step change hits 50% in 4 samples
'multiplier of 0.8, step change hits 50% in 3 samples
const MA_MULT! = 0.99
const MA_INV_MULT! = 1 - MA_MULT!

dim string String1, String2
'----------------------------------------------------------
'Set initial output values for the output pins
'Do the WM8804 pins first - give them more time in this state
'We want the WM8804 to power up in software mode, dependent on SDIN, SCLK and SDOUT

pin(P_WM8804_SPI_MOSI) = 1 'This is for configuring the WM8804 device
pin(P_WM8804_RST) = 0
pin(P_WM8804_IFM) = 1
pin(P_WM8804_CSB) = 1
setpin P_WM8804_SPI_MOSI, dout
setpin P_WM8804_RST, dout
setpin P_WM8804_IFM, dout
setpin P_WM8804_CSB, dout

pin(P_SI4689_SMODE) = 0 'should always be low indicating SPI mode
pin(P_SI4689_RES) = 0
pin(P_SI4689_CS) = 1

pin(P_PAM8407_ENAB) = 0 'default to 0 to shut down amp
pin(P_PAM8407_UP) = 1
pin(P_PAM8407_DN) = 1

pin(P_REG) = 1 'default the regulator on

setpin P_SI4689_SMODE, dout
setpin P_SI4689_RES, dout
setpin P_SI4689_CS, dout

setpin P_Flash_SPI_MISO, din
setpin P_Flash_SPI_MOSI, din
setpin P_Flash_SPI_CLK, din
setpin P_Flash_SPI_CS, din

setpin P_PAM8407_ENAB, dout
setpin P_PAM8407_UP, dout
setpin P_PAM8407_DN, dout

setpin P_REG, dout

setpin P_HEADPHONE, din

setpin P_MUX_0, dout
setpin P_MUX_1, dout

'-----------------------------------------------------
'Reset the WM8804 Toslink framer. This requires DISABLING the SPI for a moment

sub ResetWM8804
  on error skip 'the spi close will throw an error if it has not already been opened
  spi close

  'The operating mode of the WM8804 is dependent upon the state of SDIN, SCLK, SDOUT, CSB
  'and GPO0 when the device is powered up or a hardware reset occurs. Have a go at taking
  'WM8804 device out of reset now before we proceed further. The device will configure
  'itself into software controlled mode when it comes out of reset with MOSI/SDIN held
  'high.

  pin(P_WM8804_RST) = 0 'Put WM8804 into reset mode
  pin(P_WM8804_SPI_MOSI) = 1 'Configure pins so that WM8804 enters SW mode when it
  pin(P_WM8804_IFM) = 1      'comes out of reset
  pin(P_WM8804_CSB) = 1
  pin(P_WM8804_RST) = 1  'Take the WM8804 out of reset (and wait at least 26us)

  'The SPI port we just used above to configure the WM8804 will now be reconfigured as an
  'SPI port. I have increased the frequency to 10 MHz and it works great. Original FW
  'used 2 MHz.
  '
  'My DAB+ radio board would not work with any SPI frequency until I added three 3k3 pull
  'ups to 3.3V onto the DAB+ board for MISO MOSI and SCK, near Con8. With the pull ups in
  'place, both my programme and the original SC firmware work fine. Without pull ups,
  'neither work at all. I tried this both with and without the ribbon cable I used to
  'separate the micromite and DAB+ card and two different micromites.
  '
  'I could detect no radio noise difference running 10 MHz vs 2 MHz or anything else
  spi open 10000000, 0, 8 'SC FW used 2 MHz (BASIC will not run faster than 26us)
end sub

ResetWM8804
'-----------------------------------------------------
'WM8804 access routine - writing to the TOSLINK framer

sub WriteWM8804 (reg as integer, v as integer)

  pin(P_WM8804_CSB) = 0
  spi write 2, reg, v
  pin(P_WM8804_CSB) = 1
  if (D > 2) then print "*WM8804: " hex$(reg,2) "=" hex$(v,2)

end sub

'- - - - - - - - - - - - - - - - - - - - - - - - - - -
'INITIALISE WM8804
'Note the WM8804 is a set-and-forget device.
'Once it is initialised, we can ignore it and it will just do its thing.

sub InitWM8804
  'write integer part of PLL frequency ratio - refer tables 21-23 of WM8804 data sheet
  WriteWM8804 (&H06, &H08)

  'write fractional part of PLL frequency ratio
  WriteWM8804 (&H03, &HBA)
  WriteWM8804 (&H04, &H49)
  WriteWM8804 (&H05, &H0C)

  'Tell WM8804 to substitute zeros if the valid bit is not set
  WriteWM8804 (&H08, &H38)

  'All functions of the device are powered down by default and must be powered up
  'individually by writing to the relevant bits of the PWRDN register (0x1E)
  '0x3F=: PLL disabled, Spdif receiver disabled, spdif transmitter disabled,
  '       oscillator off, digital audio interface off, outputs tristated
  'From boot: manually shut down all components
  WriteWM8804 (&H1E, &H3F)

  'enable the WM8804 functions
  '0x02=: PLL enabled, Spdif receiver disabled, spdif transmitter enabled
  '           oscillator on, digital audio interface on, outputs enabled
  WriteWM8804 (&H1E, &H02)

  'write reg 0x1B (AIFTX): default except AIFTX_WL set to 2b10 (24b word length)
  WriteWM8804 (&H1B, &H0A)

  '0xCA: SYNC=1 Lclk+Bclk continue to output when S/pdif source removed
  '      AIF_MS=1=master mode MCLK, AIFRX_LRP=0 lrclk not inverted
  '      AIFRX_BCP=0 bclk not inverted, AIFRX_WL=10 24 bits, AIFRX_FMT=10 I2S mode
  WriteWM8804 (&H1C, &HCA)
end sub

'-----------------------------------------------------
'There's a 4052 used as an analogue audio mux. The main functions are a mute (ground)
'and pass through (norm)

dim integer MuteDowncount=0, Muted=1 'The WM8804 is reset (i.e. muted) at boot

sub SetMux (n as integer)
  local integer i

  i = Muted

  select case n
    case MUX_MUTE
      pin(P_MUX_1) = 0
      pin(P_MUX_0) = 0
      if ((WM8804 <> 0) and (Muted = 0)) then ResetWM8804
      MuteDowncount = 0
      Muted = 1
      if (D > 1) then print "Audio set to mute"

    case MUX_NORM
      pin(P_MUX_1) = 0
      pin(P_MUX_0) = 1
      Muted = 0
      if (D > 1) then print "Audio set to Si4689 stereo normal"

    case MUX_REV
      pin(P_MUX_1) = 1
      pin(P_MUX_0) = 0
      Muted = 0
      if (D > 1) then print "Audio set to Si4689 stereo reverse"

    case MUX_EXT
      pin(P_MUX_1) = 1
      pin(P_MUX_0) = 1
      Muted = 0
      if (D > 1) then print "Audio set to AUX"
  end select

  'Do we need to take the toslink framer out of mute mode?
  'Only do this if previous state was muted, and new state is not muted
  if ((WM8804 <> 0) and (i = 1) and (Muted = 0)) then InitWM8804
end sub

'-----------------------------------------------------
'Optional power sag interrupt
'pin 34 is configured as an open collector input with pull up. The hardware pulls
'the input low (to Digital Ground) when power is OK and goes open when power
'is lost. Reading 0 means power OK. Reading 1 means voltage sag. The power supply
'is a full wave rectified 18VAC nominal input, filtered by 40,000uF, which sits
'at about 23 volt a few seconds after power up. A 7805 pin-compatible switchmode
'regulator efficiently converts the smoothed supply to 5V without a heatsink.
'The alarm circuit has an 18V zenner from +23V, via two 10k resistors in series to
'ground. The junction of the two 10k resistors goes to base of BC549. Emitter goes
'to ground. Collector goes to both pin 34 and also a 1nF cap to ground.

dim integer DoPowerAlarm=0

'Power (voltage sag) Alarm pin requires hardware addition
if (Pv > 0) then setpin P_POWER_ALARM, INTB, PowerAlarmInt, PULLUP

sub PowerAlarmInt
  'Remember this is an interrupt. Exit the interrupt ASAP!
  'Power has either been lost (apply mute) or restored (unmute after a delay)
  if (D > 0) then print "Power Alarm Int:" pin(P_POWER_ALARM)

  if (pin(P_POWER_ALARM) = 0) then
    'If this ever happens, it must mean end of a brownout. This should be rare
    MuteDowncount = MUTE_RELEASE<<1

  else
    'Voltage is starting to sag. Immediately mute, to manage rare case when radio
    'output goes into wild oscillation on power spike, emitting piercing noise.
    SetMux (MUX_MUTE)
    DoPowerAlarm = 1
  endif
end sub

'-----------------------------------------------------
'Tick interrupt occurs every 5ms
'The tick interrupt will trip several other flags periodically to cause those sub-systems
'(state machines) to run at the next opportunity

'How long a pulse (in units of 5ms) to drive the audio amplifier up/down function
const HP_PULSE_WIDTH=16

dim integer SpiWait=0, PollWait=0, CapWait=0, DABwait=0, IRwait=0, IRnum=-1, Dwait=0
dim integer Pwait=0, Reboot=0, GuiCounter=0

SetTick 5, TickInterrupt '200 Hz interrupt

sub TickInterrupt
  'GuiCounter is used for timing default closure of various screen pages
  GuiCounter = GuiCounter + 1

  'This is just an up-counter. Will trigger the SPI state machine every 5ms
  SpiWait = SpiWait + 1

  'PollWait is used to time requests to the radio chip for status updates
  PollWait = PollWait + 1

  'Dwait is a timer for auto-saving defaults. Non zero means its running
  if (Dwait > 0) then Dwait = Dwait + 1

  'DABwait used during the band-scan function to wait for DAB programme info to come in
  if (DABwait > 0) then DABwait = DABwait + 1

  'Pwait is a timer to hold off rewriting the presets file
  if (Pwait > 0) then Pwait = Pwait + 1

  'CapWait is a timer used in the AntCap search process
  CapWait = CapWait + 1

  'IRwait used for accumulating digits / buttons in the IR remote decoding routine
  if (IRwait > 0) then
    IRwait = IRwait + 1

    'max 5 seconds with no button presses
    if (IRwait > 1000) then
      IRnum = -1
      CtrlVal(RLN8) = "" 'blank the line where button presses are accumulated
    endif
  endif

  'Reboot is a special reboot counter - it counts down and then does a reboot
  if (Reboot > 0) then
    Reboot = Reboot - 1

    if (Reboot = 0) then
      clear
      run 'this is where the programme does a reboot!!
    endif
  endif
  '- - - - - - - - - - - - - - - - - - - - - - - - - -
  'MuteDowncount is used to hold off enabling audio output after configuration changes
  'The original design outputs severe thumps/clicks under certain conditions
  if (MuteDowncount > 0) then
    MuteDowncount = MuteDowncount - 1

    if ((CapSearch <= 0) and (MuteDowncount = 0)) then
      SetMux (MUX_NORM) 'will 'unmute' (i.e. initialise) the TosLink framer too
      if ((Smode = Xconsole) or (Smode = XstatusE) or (Smode = XmainE) or (Smode = XScanE)) then Change_Smode = Xstatus
    endif
  endif

end sub

'-----------------------------------------------------
'The infra-red remote is defined here. Don't add code to process the IR messages here
'because this is an interrupt service routine! Do it from the main DO loop

dim integer IrDev=0, IrButton=0, IrTest=0

ir IrDev, IrButton, IrInt 'start the IR decoder (hard wired to pin 78 of MPU)

sub IrInt
  local S as string length 50 'This is an interrupt, so can't use string1 or string2!!

  if (D > 0) or (IrTest <> 0) then S = "IR Device Code=" + hex$(IrDev) + " Button=" + hex$(IrButton)
  if (D > 0) then print S

  if (IrTest <> 0) then
    WriteMsg (S)
    IrDev = 0 'Don't process this keypress normally! We're in a test mode!
  endif

  'Otherwise, just exit without doing anything here in the interrupt routine
  'IrDev and IrButton will be non-zero if something got pressed, and those variables will
  'be scanned in the main DO loop, outside the interrupt, where they are permitted to
  'take longer to process
end sub

'-----------------------------------------------------
const BOOT_TITLE=1, LN1=2, LN2=3, LN3=4, LN4=5, LN5=6, LN6=7, LN7=8, LN8=9, LN9=10
const LN10=11, LN11=12, LN12=13, LN13=14, RLN1=15, RLN2=16, RLN3=17, RLN4=18, RLN5=19
const RLN6=20, RLN7=21, RLN8=22, RLN9=23, ERR1=24, ERR2=25, AR1=26, AR2=27, Ename=28
const Efreq=29, Eclr=30, Eband=31, EkHz=32, Edata=33, Eframe=34, Eindex=35, Eactive=36
const Eantcap=37, BCancel=38, BSetup=39, BPreset=40, BEdit=41, BVolume=42, BScan=43
const BCap=44, BRemote=45, BLight=46, Bdown=47, Bup=48, BSeekDn=49, BSeekUp=50, Ram=51
const Rfm=52, Rdab=53, Sdigital=54, Vvolume=55, VLight=56, Vaudio=57, Vantcap=58
const Vactive=59, NameLen=16

'max gui index is 100, so define the radio buttons from the top backwards
const Rad1=101-MAX_RADIO

'must start with Smode 0, so the change to console is registered
dim integer Smode=0, Change_Smode=Xconsole, RadioMode=-1, DoScan=0, CapSearch=0
dim integer Frequency=-1, NewFrequency=666, ServiceID=-1, CompID=-1, NewServiceID=-1
dim integer NewCompID=-1, Seeking=0, SeekStart=0, NumDabFreqs=-1, DabFreqIndex=-1
dim integer NumDabServices=0, DabID=-1, DabFreqs(MAX_DAB_FREQS-1), NumPresets=0
dim integer NumActivePresets=0, ActiveProgData=-1, DigitalMode=-1, CLine=LN1
dim integer Pfreq(MAX_PRESETS-1), Pid(MAX_PRESETS-1), Pcomp(MAX_PRESETS-1)
dim integer Pactive(MAX_PRESETS-1), Pcap(MAX_PRESETS-1)

dim string Pname(MAX_PRESETS-1) length NameLen, ProgName(3) length 2, ProgData(31) length 4
const MAX_SET=4
dim string RdsNameSet(MAX_SET) length 8
dim integer RdsNames=0

'-----------------------------------------------------
'PAGE 1 - used for text console
'PAGE 2 - status and information
'PAGE 3 - Most buttons for display over the status and information
'PAGE 4 - Scan buttons for display over the status and information
'PAGE 5 - Edit Preset screen
'PAGE 6 - Select Preset radio buttons
'PAGE 7 - Setup
'PAGE 9 - Volume Control box (used with 3)
'PAGE 10 - holds error message (use with 2, 4)
'PAGE 32 - dim cancel button at top left corner used with several pages

dim integer i, j

cls
gui hide all 'just in case there is any crap left over from a previous run
gui delete all
colour rgb(white), rgb(black)

'- - - - - - - - - - - - - - - - - - - - - - - -
'Text console
gui setup 1

font 3
i=MM.FontHeight * 1.4 'used to set line spacing

'Customise the console message by modifying below!
gui frame BOOT_TITLE, "Stefan" + chr$(39) + "s Covid-19 Radio Project", 0, MM.FontHeight>>1, MM.HRes, MM.VRes-(MM.FontHeight>>1),RGB(green)

gui caption LN1, "", 20, i, "lt"
gui caption LN2, "", 20, i*2, "lt"
gui caption LN3, "", 20, i*3, "lt"
gui caption LN4, "", 20, i*4, "lt"
gui caption LN5, "", 20, i*5, "lt"
gui caption LN6, "", 20, i*6, "lt"
gui caption LN7, "", 20, i*7, "lt"
gui caption LN8, "", 20, i*8, "lt"
gui caption LN9, "", 20, i*9, "lt"
gui caption LN10, "", 20, i*10, "lt"
gui caption LN11, "", 20, i*11, "lt"
gui caption LN12, "", 20, i*12, "lt"
gui caption LN13, "", 20, i*13, "lt"

gui area AR1, 0, 0, mm.hres, mm.vres
'-----------------------------------------------------
sub WriteMsg (msg as string)
  local integer i

  'If we need to scroll, do that now
  if (CLine > LN13) then
    for i = LN1 to LN12
      CtrlVal(i) = CtrlVal(i+1)
    next

    CLine = LN13
  endif

  'We can now write the message to the last console line
  CtrlVal(CLine) = msg
  CLine = CLine + 1
  print msg
end sub

'-----------------------------------------------------
sub ClearConsole
  local integer i

  'Blank the console info on the screen
  for i = LN1 to LN13
    CtrlVal(i) = ""
  next

  CLine = LN1
end sub

'-----------------------------------------------------
'The basic software, preset and config files (and up to 9 backups of these files) are
'stored on an SD card. The files are small, so you can use your ancient 8 Mbyte SD cards
'and still heaps of flash to spare

function RotateAndOpen (base as string, ext as string, do_open as integer) as integer
  local integer i

  'Remove the oldesst backup file (if it exists)
  on error ignore
  kill base + "9" + ext

  'Rename any backups if they exist
  for i = 8 to 0 step -1
    String1 = base + str$(i) + ext
    String2 = base + str$(i+1) + ext
    name String1 as String2
  next

  if (do_open <> 0) then
    'Open a new version 0 file
    Open base + "0" + ext for output As 1
  endif

  on error abort

  if (MM.Errno <> 0) then
    WriteErr "SD Card or write err"
    name String2 as base + "0" + ext 'rename the last version back to vers 0
    RotateAndOpen = 0

  else
    RotateAndOpen = 1
  endif
end function

'- - - - - - - - - - - - - - - - - - - - - - - -
'is there a new version of the basic programme to load from the SDMMC flash card?
'This allows us to update software by writing a new file called "radio0.bas" onto the
'SD card so we avoid having to set up a serial interface to the micromite and have
'computer on hand.

String1 = ""
on error skip
String1 = dir$("Radio0.bas", FILE)

if (String1 <> "") then
  if (RotateAndOpen ("Radio", ".bas", 0) = 1) then
    'now load and run the new version (if possible)
    WriteMsg ("LOADING NEW SOFTWARE")
    pause 100 'can't avoid this or the print doesn't get out in time
    on error skip
    load "Radio1.bas",r
  endif

  'and if we fall through, the load didn't work, but by then, the original
  'will have been blown away and we are now stuffed
endif

'-----------------------------------------------------
sub ClearStatus
  local integer i

  'Blank the status info on the screen
  for i = RLN1 to RLN9
    CtrlVal(i) = ""
  next
end sub

'-----------------------------------------------------
'Status and information
gui setup 2

font 5,1.4
i = 32

gui caption RLN1, "", mm.hres>>1, i>>1, "ct"

font 2
gui caption RLN3, "", mm.hres>>1, i*3.5, "ct"

font 3 'this is the regular font size
gui caption RLN2, "", mm.hres>>1, i*2.1, "ct"
gui caption RLN4, "", mm.hres>>1, i*5, "ct"
gui caption RLN5, "", mm.hres>>1, i*6, "ct"
gui caption RLN6, "", mm.hres>>1, i*7, "ct"
gui caption RLN7, "", mm.hres>>1, i*8, "ct"
gui caption RLN8, "", mm.hres>>1, i*9, "ct"
gui caption RLN9, "", mm.hres>>1, i*11, "ct" 'move a little lower on the display

gui area AR2, 0, 0, mm.hres, mm.vres

'- - - - - - - - - - - - - - - - - - - - - - - -
'these are buttons and other gadgets to be used with page 3
gui setup 3

font 4
gui button BPreset, "Tune to Preset", 5, 410, 255, 65, rgb(white), rgb(blue)
gui button BEdit, "Edit Presets", 270, 410, 255, 65
gui button BSetup, "Setup", 535, 410, 255, 65

font 3
gui radio Ram, "AM", 690, i*9.5, 12, rgb(cyan)
gui radio Rfm, "FM", 690, i*10.7, 12, rgb(green)
gui radio Rdab, "DAB", 690, i*11.9, 12, rgb(magenta)

gui button Bdown, "<", 195, i*1.7, 60, 50, rgb(white), rgb(black)
gui button Bup, ">", 540, i*1.7, 60, 50

'- - - - - - - - - - - - - - - - - - - - - - - -
'these are buttons and other gadgets to be used with page 3
gui setup 4

gui button BSeekDn, "<<", 125, i*1.7, 60, 50
gui button BSeekUp, ">>", 610, i*1.7, 60, 50

'- - - - - - - - - - - - - - - - - - - - - - - -
sub FetchPresetDetails
  local integer i, j

  CtrlVal(Edata) = ""
  i = CtrlVal(Eindex) - 1 'index counts upwards from 1

  if (i >= NumPresets) then
    'important to keep the default short enough to fit in the box
    CtrlVal(Ename) = "##type station identifier"
    CtrlVal(Efreq) = "##freq"
    CtrlVal(Eband) = "--"
    CtrlVal(Eantcap) = 0
    CtrlVal(Eactive) = 0
    exit sub
  endif

  'at this point, the index corresponds with a valid entry
  if (Pname(i) = "") then
    CtrlVal(Ename) = "##type station identifier"
  else
    CtrlVal(Ename) = Pname(i)
  endif

  j = Pfreq(i)
  CtrlVal(Efreq) = j

  if ((j >= AmMIN) and (j <= AmMAX)) then
    CtrlVal(Eband) = "AM"

  elseif ((j >= FmMIN) and (j <= FmMAX)) then
    CtrlVal(Eband) = "FM"

  elseif ((j >= DabMIN) and (j <= DabMAX)) then
    CtrlVal(Eband) = "DAB"
    CtrlVal(Edata) = "Service ID:" + hex$(Pid(i)) + " Component ID:" + hex$(Pcomp(i))

  else
    CtrlVal(Efreq) = "##freq"
    CtrlVal(Eband) = "--"
  endif

  CtrlVal(Eantcap) = Pcap(i) '0=auto, 1..128=configured
  CtrlVal(Eactive) = Pactive(i) and 1 '0=disabled, 1=active
end sub

'- - - - - - - - - - - - - - - - - - - - - - - -
'Edit Presets screen
gui setup 5

font 2
gui frame Eframe, "Preset Index", 270, 300, 250, 70, RGB(brown)
gui frame Vantcap, "AntCap Setting", 5, 303, 230, 65, RGB(brown)
gui frame Vactive, "Preset list", 545, 410, 230, 65, RGB(brown)
gui caption Edata, "", MM.HRes*.5, i*5, "cm"

font 4
gui textbox Ename, mm.hres*.2, i-6, mm.hres*.6, i+6, rgb(brown), rgb(black)
gui numberbox Efreq, mm.hres*.42, i*3, mm.hres*.15, i+6
gui caption Eband, "AM", mm.hres*.38, i*3.25, "ct"
gui caption EkHz, "kHz", mm.hres*.58, i*3.25, "lt"
gui button Eclr, "Clear", 10, 410, 180, 65, rgb(white), rgb(blue)

gui spinbox Eindex, 280, 313, 230, 50, RGB(brown), RGB(black), 1, 1, MAX_PRESETS
gui spinbox Eantcap, 5, 313, 220, 50, RGB(brown), RGB(black), 1, 0, 128

font 2
gui switch Eactive, "Hide|Show", 550, 425, 220, 40, rgb(black), rgb(brown)

CtrlVal(Eantcap) = 0 'a sensible default for the time being

'- - - - - - - - - - - - - - - - - - - - - - - -
'Select Preset screen
'The selecct preset screen is dynamic - it changes as new stations are discovered,
'and existing presets are edited. The screen needs to be built and knocked down
'dynamically because the station preset count and preset names can change dynamically

const NUM_PRESET_COLS = MAX_RADIO/3
dim integer NumGuiPresets=0

'NOTE: Seems there could be a micromite issue that causes ClearGuiPresets to only
'behave if called while page 6 is displayed. If you delete button elements while the
'buttons are hidden on a page that is not currently displayed, they are still
'erased from the page that is currently being displayed, messing up the display!

sub ClearGuiPresets
  local integer n

  if (NumGuiPresets > 0) then
    for n = 1 to NumGuiPresets
      gui delete Rad1+n-1
    next

    NumGuiPresets = 0
  endif

end sub

'- - - - - - - - - - - - - - - - - - - - - - - -
sub Define_Radio_Button (n as integer, x as integer, y as integer)
  local integer f, col

  f = Pfreq(n-1)
  String2 = Pname(n-1)
  if (String2 = "") then String2 = str$(f) + " kHz"

  if ((f >= AmMIN) and (f <= AmMAX)) then
    col = RGB(cyan)

  elseif ((f >= FmMIN) and (f <= FmMAX)) then
    col = RGB(green)

  else
    col = RGB(magenta)

  endif

  'Try to use a larger font, but the number of characters cannot exceed 12
  if (len(String2) > 12) then
    font 2
  else
    font 3
  endif

  colour rgb(white), rgb(black)
  gui radio Rad1+n-1, String2, x, y, RADIO_RADIUS, col

  if ((Frequency = Pfreq(n-1)) and ((Frequency < DabMIN) or ((ServiceID = Pid(n-1)) and (CompID = Pcomp(n-1))))) then
    CtrlVal(Rad1+n-1) = 1
  else
    CtrlVal(Rad1+n-1) = 0
  endif

  NumGuiPresets = n
end sub

'- - - - - - - - - - - - - - - - - - - - - - - -
Sub SetupGuiPresets
  local integer i, j, n

  ClearGuiPresets
  gui setup 6
  j=49 'this is the line spacing

  'first column
  if (NumActivePresets > 0) then
    n = NUM_PRESET_COLS
    if (NumActivePresets < n) then n = NumActivePresets

    for i = 0 to (n-1)
      Define_Radio_Button (i+1, RADIO_RADIUS, RADIO_RADIUS+(j*i))
    next
  endif

  'second column
  if (NumActivePresets > NUM_PRESET_COLS) then
    n = NUM_PRESET_COLS << 1
    if (NumActivePresets < n) then n = NumActivePresets

    for i = NUM_PRESET_COLS to (n-1)
      Define_Radio_Button (i+1, 275, RADIO_RADIUS+(j*(i-NUM_PRESET_COLS)))
    next
  endif

  'third column
  if (NumActivePresets > (NUM_PRESET_COLS<<1)) then
    n = NUM_PRESET_COLS * 3
    if (NumActivePresets < n) then n = NumActivePresets

    for i = NUM_PRESET_COLS<<1 to (n-1)
      Define_Radio_Button (i+1, 535, RADIO_RADIUS+(j*(i-(NUM_PRESET_COLS<<1))))
    next
  endif
end sub

'- - - - - - - - - - - - - - - - - - - - - - - -
'Config screen
gui setup 7

font 4
gui button BRemote, "IR Remote Test", 5, 410, 255, 65, rgb(white), rgb(blue)
gui button BCap, "AntCap Scan", 270, 410, 255, 65
gui button BScan, "Station Scan", 535, 410, 255, 65

font 3
gui frame Vlight, "Backlight", 5, 320, 200, 75, RGB(brown)
gui spinbox BLight, 5, 335, 200, 50, RGB(brown), RGB(black), 1, 10, 100
GUI frame Vaudio, "Audio Output", 535, 335, 250, 65, RGB(brown)

CtrlVal(BLight) = 50 'a sensible default for the time being

font 2
gui switch Sdigital, "Analogue|Digital", 545, 350, 230, 40, rgb(black), rgb(brown)
CtrlVal(Sdigital) = 0

'- - - - - - - - - - - - - - - - - - - - - - - -
gui setup 9 'This is the volume control box, used with [3] when analogue audio is enabled

font 2
gui frame Vvolume, "Analogue Volume", 5, 335, 210, 65, RGB(brown)
gui spinbox BVolume, 5, 350, 200, 45, RGB(brown), RGB(black), 1, 0, 63

CtrlVal(BVolume) = 20 'a sensible default for the time being

'- - - - - - - - - - - - - - - - - - - - - - - -
gui setup 10 'This is the error screen - to be overlaid onto other pages

font 5,1.5 'scale size larger - enough for 17 characters
i = mm.fontheight
gui caption ERR1, "Muting due to", mm.hres>>1, 275, "ct", rgb(red)

font 5 'normal scale but still large font - enough for 33 characters
gui caption ERR2, "", mm.hres>>1, 275+i, "ct", rgb(red)

'- - - - - - - - - - - - - - - - - - - - - - - -
'The last page is the cancel button at the top left - it is used with several pages
gui setup 32

font 5
gui button BCancel, "X", 0, 0, 60, 60, rgb(48,48,48), rgb(24,24,24)

'- - - - - - - - - - - - - - - - - - - - - - - -
sub WriteErr (msg as string)
  CtrlVal(ERR2) = msg 'write the error on LCD error page in red
  WriteMsg (msg)

  select case Smode
    case Xstatus
      Change_Smode = XstatusE

    case Xmain, Xedit, Xselect, Xsetup
      Change_Smode = XmainE

    case Xscan
      Change_Smode = XscanE

    case Xcap
      Change_Smode = XcapE

    case 0
      Change_Smode = Xconsole

  end select
end sub

'- - - - - - - - - - - - - - - - - - - - - - - -
'ProcessScreenChange does the swapping from one screen to another. It mops up the mess
'from the previous screen before changing to the next

dim integer ExitTimer=0

sub ProcessScreenChange
  local integer volbox=0

  if (Smode = Change_Smode) then exit sub 'just a precaution

  if (Smode = Xedit) then
    gui textbox cancel
    gui numberbox cancel

  elseif (Smode = Xselect) then
    ClearGuiPresets 'Note mmbasic issue! Must only be called when presets displayed!

  else if ((Smode = XIRTest) or (Smode = Xcap)) then
    ClearConsole
  endif

  'Determine if we need or do not need a volume box on the main screen [3]
  if ((CtrlVal(Sdigital) = 0) or (pin(P_HEADPHONE) = 0) or (WM8804 = 0)) then volbox = 1

  'Now change to the new screen
  select case Change_Smode
    case Xconsole, XIRTest, Xcap
      gui hide AR2
      gui show AR1
      page 1
      ExitTimer = 0

    case Xstatus
      gui hide AR1
      gui show AR2
      page 2
      ExitTimer = 0

    case Xmain
      gui hide AR2

      'Determine the current band and set the state of the radio button for the band
      if (RadioMode = MODE_AM) then CtrlVal(Ram) = 1 else CtrlVal(Ram) = 0
      if (RadioMode = MODE_FM) then CtrlVal(Rfm) = 1 else CtrlVal(Rfm) = 0
      if (RadioMode = MODE_DAB) then CtrlVal(Rdab) = 1 else CtrlVal(Rdab) = 0

      'In DAB mode, we don't have fast scan buttons. Also the volume control changes
      if (RadioMode <> MODE_DAB) then
        if (volbox <> 0) then
          page 2, 3, 4, 9, 32
        else
          page 2, 3, 4, 32
        endif
      else
        if (volbox <> 0) then
          page 2, 3, 9, 32
        else
          page 2, 3, 32
        endif
      endif

      ExitTimer = 2000 '10 seconds

    case Xedit 'edit presets
      page 5, 32
      ExitTimer = 8000 '40 seconds

    case Xselect 'select a preset
      SetupGuiPresets
      page 6
      ExitTimer = 3000 '15 seconds

    case Xsetup 'setup screen
      gui hide AR1
      gui hide AR2
      page 7, 32
      ExitTimer = 3000 '15 seconds

    case Xscan 'scan (setup the preset file)
      gui hide AR1
      gui hide AR2
      page 2
      ExitTimer = 0

    case XstatusE, XscanE, XcapE 'error in status, scan, AntCap screens
      page 2, 10
      ExitTimer = 3000 '15 seconds

    case XmainE 'error in main screen
      if (RadioMode <> MODE_DAB) then
        if (volbox <> 0) then
          page 2, 3, 4, 9, 10, 32
        else
          page 2, 3, 4, 10, 32
        endif
      else
        if (volbox <> 0) then
          page 2, 3, 9, 10, 32
        else
          page 2, 3, 10, 32
        endif
      endif

      ExitTimer = 1000 '5 seconds

    case else
      String1 = "Unknown screen mode " + str$(Change_Smode) 'We have a bug
      error String1
  end select

  if (D > 1) then print "Changing Screen Mode from " Smode " to " Change_Smode " ExitTimer=" ExitTimer
  Smode = Change_Smode
  Change_Smode = 0
  GuiCounter = 0
end sub

'----------------------------------------------------------
'Touch interrupts handled here - note they need to be SHORT!

dim integer ButtonDn=0, ButtonUp=0
gui Interrupt TouchDown, TouchUp

'TouchDown called when touch first detected
sub TouchDown
  ButtonDn = Touch(REF)
  if (D > 1) then print "Touch down:" ButtonDn " Smode=" Smode
  GuiCounter=0

  select case ButtonDn
    case AR1 'transparent area behind page 1
      if (CapSearch = 2) then 'touch moves from state 2 to state 3
        CapSearch = 3

      elseif ((Smode = XIRTest) or (CapSearch = 10)) then
        Change_Smode = Xsetup 'this is the escape back to setup screen
        CapSearch = 0
        IrTest = 0

      endif

    case AR2 'transparent area behind page 2
      Change_Smode = Xmain

  end select
end sub

'- - - - - - - - - - - - - - - - - - - - - - - -
'TouchUp called when touch released
sub TouchUp
  ButtonUp = Touch(LASTREF)
  if (D > 1) then print "Touch up:" ButtonUp
  GuiCounter=0
end sub

'----------------------------------------------------------
'Routine that searches a desired DAB tune frequency and returns an index into DAB
'frequency table

function DabFreqToIndex (freq as integer) as integer
  for DabFreqToIndex = 0 to (NumDabFreqs-1)
    if (DabFreqs(DabFreqToIndex) = freq) then exit for
  next

  if (DabFreqToIndex >= NumDabFreqs) then
    if (D > 0) then Print "DAB frequency not in DabFreqs table:" freq
    DabFreqToIndex=0
  endif
end function

'----------------------------------------------------------
'The exit timer is used to automatically change screens if there has been no input for a
'while

sub ProcessExitTimer
  select case Smode
    case Xmain
      Change_Smode = Xstatus

    case Xedit, Xsetup, Xselect
      Change_Smode = Xmain

    case XstatusE
      Change_Smode = Xstatus

    case XmainE
      Change_Smode = Xmain

    case XscanE
      Change_Smode = Xscan

    case XcapE
      Change_Smode = Xcap

  end select

  ExitTimer = 0
end sub

'----------------------------------------------------------
'The button pressing routines happen below. Note that Infra Red commands are sometimes
'made to appear like GUI button presses.
'NOTE: THIS ROUTINE SHOULD NOT BE CALLED UNLESS STATE=0
'
'The MMBasic interrupt routine seems to have bugs or is not consistent. Sometimes a
'button press returns BOTH a button down and a button up interrupt. Other times it seems
'only to return a button up interrupt. Its as if the buttons on the LCD are polled rather
'than interrupt driven within MMBASIC and if you happen to press briefly, then it might
'miss something? This doesn't seem like a good explanation though. It feels like the
'cause is different. The following variable is a kind of kludge to try and get the button
'up interrupt. I noticed this problem with the SC firmware as well as my own, so assume
'another MMBASIC issue.

dim integer BDvalue=0

sub ProcessButtonDown
  local integer i, j=-1, k

  if (D > 1) then print "Button Down:" ButtonDn " State=" State ", Smode=" Smode
  GuiCounter=0
  BDvalue = ButtonDn

  select case ButtonDn
    case BEdit
      Change_Smode = Xedit
      Pwait = 0
      FetchPresetDetails

    case BPreset
      if (NumActivePresets = 0) then
        CtrlVal(RLN8) = "Presets not yet defined"
      else
        Change_Smode = Xselect
      endif

    case BSetup
      Change_Smode = Xsetup

    case BLight
      backlight CtrlVal(BLight)
      Dwait = 1 'Start the default counter ticking

    case BVolume
      Dwait = 1

    case Ram 'manually selected AM band
      for i = 0 to NumActivePresets-1
        if ((Pfreq(i) >= AmMIN) and (Pfreq(i) <= AmMAX)) then
          if (j < 0) then j = i
          k = i
        endif
      next

      'Set a default new frequency in the middle of the AM band - Remember 9 kHz spacing
      NewFrequency = ((AmMIN + AmMAX) >> 1) / AmDELTA
      NewFrequency = NewFrequency * AmDELTA
      goto JUMP_MIDDLE

    case Rfm 'manually selected FM band
      'Scan all active presets. j will be first FM index, k is last FM index
      for i = 0 to NumActivePresets-1
        if ((Pfreq(i) >= FmMIN) and (Pfreq(i) <= FmMAX)) then
          if (j < 0) then j = i
          k = i
        endif
      next

      'Set a default new frequency in the middle of the FM band - 100 kHz spacing
      NewFrequency = ((AFmMIN + FmMAX) >> 1) / FmDELTA
      NewFrequency = NewFrequency * FmDELTA
      goto JUMP_MIDDLE

    case Rdab 'manually selected DAB band
      for i = 0 to NumActivePresets-1
        if ((Pfreq(i) >= DabMIN) and (Pfreq(i) <= DabMAX)) then
          if (j < 0) then j = i
          k = i
        endif
      next

      NewFrequency = 206352 'in the middle of the DAB band

    JUMP_MIDDLE:
      if (j >= 0) then NewFrequency = Pfreq((j + k) >> 1)
      goto RETUNE

    case Bdown
      if ((Frequency > AmMIN) and (Frequency <= AmMAX)) then NewFrequency = Frequency - AmDELTA
      if ((Frequency > FmMIN) and (Frequency <= FmMAX)) then NewFrequency = Frequency - FmDELTA

      if ((Frequency > DabMIN) and (Frequency <= DabMAX)) then 'Dab needs to use the list
        DabID = DabFreqToIndex(Frequency)

        if (DabID > 0) then
          DabID = DabID - 1
          NewFrequency = DabFreqs(DabID)
        endif
      endif

      goto RETUNE

    case Bup
      if ((Frequency >= AmMIN) and (Frequency < AmMAX)) then NewFrequency = Frequency + AmDELTA
      if ((Frequency >= FmMIN) and (Frequency < FmMAX)) then NewFrequency = Frequency + FmDELTA

      if ((Frequency >= DabMIN) and (Frequency < DabMAX)) then 'Dab needs to use the list
        DabID = DabFreqToIndex(Frequency)

        if (DabID < (NumDabFreqs-1)) then
          DabID = DabID + 1
          NewFrequency = DabFreqs(DabID)
        endif
      endif

      goto RETUNE

    case BSeekDn
      'SEEK the next station up or down
      'Seeking:
      'Bit0: 0=seek down, 1=seek up
      'Bit1: 0=single seek, 1=full scan
      'Bit6: 1=Seek Completed
      'Bit7: 1=Seek Requested

      'The radio will throw command errors if you try and scan past the end of the band
      if (((RadioMode = MODE_AM) and (Frequency > AmMIN)) or ((RadioMode = MODE_FM) and (Frequency > FmMIN))) then
        Seeking = &H80 'Seek requested + seek down
        goto SEEK_SET
      endif

    case BSeekUp
      if (((RadioMode = MODE_AM) and (Frequency < AmMAX)) or ((RadioMode = MODE_FM) and (Frequency < FmMAX))) then
        Seeking = &H81 'Seek requested + seek up

      SEEK_SET:
        'Are we in the AM band, or FM band? Choose correct seek state
        if (RadioMode = MODE_AM) then State = 1000
        if (RadioMode = MODE_FM) then State = 1100
      RETUNE:
        ServiceID = -1
        CompID = -1
        NewServiceID = -1
        NewCompID = -1
        LastDabServList = -1
        Dwait = 1
	AntCap = 0
      endif

    case BScan
      DoScan = 1

    case BCap
      CapSearch = 1

    case BCancel 'Same response as if the exit timer expired
      ProcessExitTimer

    case BRemote
      Change_Smode = XIRTest
      ClearConsole
      WriteMsg ("Infrared remote test mode")
      WriteMsg ("** Touch screen to exit this mode **")
      IrTest = 1

    case Eclr
      i = CtrlVal(Eindex)
      if ((Pname(i-1) <> "") or (Pfreq(i-1) >= AmMIN)) then Pwait = 1
      Pname(i-1) = ""
      Pfreq(i-1) = 0
      Pid(i-1) = -1
      Pcomp(i-1) = -1
      Pcap(i-1) = -1
      Pactive(i-1) = 0 '0=disabled, 1=active
      FetchPresetDetails

    case Eindex 'Process button down as well as up to catch auto-repeat
      FetchPresetDetails

    case Eantcap 'AntCap setting
      i = CtrlVal(Eindex)
      Pcap(i-1) = CtrlVal(Eantcap)

    case Eactive 'Swap between active and inactive preset record
      i = CtrlVal(Eindex)
      Pactive(i-1) = (Pactive(i-1) and 254) or (CtrlVal(Eactive) and 1)

    case is >= Rad1 'Preset selected
      i = ButtonDn - Rad1
      if (D > 0) then print "Choosing preset: Name=" Pname(i) " Freq=" Pfreq(i)
      NewFrequency = Pfreq(i)
      NewServiceID = Pid(i)
      NewCompID = Pcomp(i)
      AntCap = Pcap(i)
      Dwait = 1

      'Set short gui timeout after radio button pressed
      ExitTimer = 300 '1.5 seconds

  end select

  ButtonDn = 0
end sub

'- - - - - - - - - - - - - - - - - - - - - - - -
'refer to button down regarding BDvalue kludge and gui issues

sub ProcessButtonUp
  local integer i

  if (D > 1) then print "Button Up:" ButtonUp
  GuiCounter=0

  select case ButtonUp
    case Ename 'changed the station name
      i = CtrlVal(Eindex)
      if (i > NumPresets) then NumPresets = i

      if (Pname(i-1) <> CtrlVal(Ename)) then
        Pname(i-1) = CtrlVal(Ename)
        Pwait = 1
      endif

      if (CtrlVal(Ename) <> "") then
        Pactive(i-1) = (Pactive(i-1) and 1) or 2 'bit 1 means we set the station name manually
      else
        Pactive(i-1) = Pactive(i-1) and 1
      endif

    case Efreq 'change the station frequency
      i = CtrlVal(Eindex)
      if (i > NumPresets) then NumPresets = i

      if (Pfreq(i-1) <> CtrlVal(Efreq)) then
        Pfreq(i-1) = CtrlVal(Efreq)
        FetchPresetDetails
        if (CtrlVal(Eband) <> "--") then Pwait = 1
      endif

      if (D > 1) then print "Freq change:" CtrlVal(Efreq) " NumPresets=" NumPresets

    case Eindex 'Selected a new index
      FetchPresetDetails 'this will display the new data

    case else 'here comes the gui kludge
      if (ButtonUp <> BDvalue) then 'button up that doesn't have a matching button down
        ButtonDn = ButtonUp
        if (D > 0) then print "Button down interrupt missed? " ButtonUp
        ProcessButtonDown
      endif
  end select

  ButtonUp = 0
  BDvalue = 0
end sub

'-----------------------------------------------------
'ChooseNext used by IR remote control routine to choose the next or previous preset
'in the list.

sub ChooseNext (x as integer)
  local integer i
  local string s

  'Check if we are already accumulating buttons. If not, choose starting value
  if ((IRnum < 0) or (IRnum >= NumPresets)) then
    'Not yet accmulating numbers - try and match the currently selected channel
    IRnum = NumPresets >> 1

    if (NumPresets > 0) then
      for i = 0 to NumPresets-1
        if ((Frequency = Pfreq(i)) and ((Frequency < DabMIN) or ((ServiceID = Pid(i)) and (CompID = Pcomp(i))))) then exit for
      next

      if (i < NumPresets) then IRnum = i
    endif
  endif

  'IRnum holds the current candidate preset. Choose next (or prev) by adding
  '(or subtracting) and loop around if we go past the beginning or end
  IRnum = IRnum + x
  if (IRnum >= NumPresets) then IRnum = 0
  if (IRnum < 0) then IRnum = NumPresets-1

  'Display the candidate preset
  s = str$(Pfreq(IRnum)) + " kHz"
  if (Pname(IRnum) <> "") then s = Pname(IRnum) + "    " + s
  CtrlVal(RLN8) = str$(Pfreq(IRnum)) + " kHz"
  IRwait = 1 'reset the IR wait timer for another period
end sub
'- - - - - - - - - - - - - - - - - - - - - - - - - - -
'The processing of IR messages can be customised here according to your preferred IR
'control. Use the 'IR Remote Test' function in the setup menu to learn your IR remote
'characteristics and then modify the code below to suit your settings. Map the IR
'buttons to GUI menu buttons. On entry, State=0

sub ProcessIR
  local integer n=0

  if (IrDev = &H409F) then 'Modify with the device ID of your remote control unit
    'modify each case statement with the button values for your remote control unit

    select case IrButton
      case &H40 'Volume UP (only works with analogue output)
	if ((CapSearch <= 0) and (CtrlVal(BVolume) < 62)) then
          CtrlVal(BVolume) = CtrlVal(BVolume) + 2
          State = 452
	endif

      case &HC0 'Volume DOWN (only works with analogue output)
	if ((CapSearch <= 0) and (CtrlVal(BVolume) > 1)) then
          CtrlVal(BVolume) = CtrlVal(BVolume) - 2
          State = 452
	endif

      case &H30 'MUTE
        if (CapSearch <= 0) then
          if (Muted = 0) then
            SetMux (MUX_MUTE) 'mute the 4052 and reset the WM8804
            State = 10001 'apply DAC hardware mute

          else
            Set_Mux (MUX_NORM)
            State = 452 'set current volume
          endif
	endif
      '---------------------
      case &H00 'Channel UP
        ChooseNext (1) 'chooses from the list of presets (then press OK)

      case &H80 'Channel DOWN
        ChooseNext (-1)
      '---------------------
      case &H10 'Right Arrow - make it look like we touched the LCD
        ButtonDn = Bup 'seek up to next station

      case &H20 'Left Arrow - make it look like we touched the LCD
        ButtonDn = Bdown 'seek down to next station
      '---------------------
      case &HD2 'Jump to AM band
        ButtonDn = Ram 'make it look like we touched the LCD

      case &H52 'Jump to FM band
        ButtonDn = Rfm 'make it look like we touched the LCD

      case &H92 'Jump to DAB band
        ButtonDn = Rdab 'make it look like we touched the LCD
      '---------------------
      'The following statements process the number keys and the 'OK' button
      'used to choose a frequency

      case &H08 '0
        n = 0
        goto NEXT_IR_NUM

      case &H88 '1
        n = 1
        goto NEXT_IR_NUM

      case &H48 '2
        n = 2
        goto NEXT_IR_NUM

      case &HC8 '3
        n = 3
        goto NEXT_IR_NUM

      case &H28 '4
        n = 4
        goto NEXT_IR_NUM

      case &HA8 '5
        n = 5
        goto NEXT_IR_NUM

      case &H68 '6
        n = 6
        goto NEXT_IR_NUM

      case &HE8 '7
        n = 7
        goto NEXT_IR_NUM

      case &H18 '8
        n = 8
        goto NEXT_IR_NUM

      case &H98 '9
        n = 9

      NEXT_IR_NUM:
        IRwait = 1 'reset the IR wait timer for another period
	if (IRnum < 0) then IRnum = 0
        IRnum = (IRnum * 10) + n 'accumulate the most recent digit
        String1 = str$(IRnum)
        CtrlVal(RLN8) = String1 'display the accumulated button presses on line 8

      case &HF8 'OK button - jump to frequency that has just been entered
        if (((IRnum >= AmMIN) and (IRnum <= AmMAX)) or ((IRnum >= FmMIN) and (IRnum <= FmMAX)) or ((IRnum >= DabMIN) and (IRnum <= DabMAX))) then
          NewFrequency = IRnum

        elseif ((IRnum >= 0) and (IRnum < NumPresets)) then
          'choose a preset
          NewFrequency = Pfreq(IRnum)
          NewServiceID = Pid(IRnum)
          NewCompID = Pcomp(IRnum)
	  n = 0

        else
          n = -1 'play a long beep indicating number not recognised
        endif

        goto CLEAR_IR_NUM

      case &H12 'similar to 'OK' for frequency, but jumps to entered preset number
        if (IRnum < NumPresets) then
          ButtonDn = Rad1 + IRnum
        else
          n = -1 'play a long beep indicating number not recognised
        endif

      CLEAR_IR_NUM:
        CtrlVal(RLN8) = ""
        IRnum = -1 'Return the number accumulator to off condition
        IRwait = 0
      '---------------------
      case else
        n = -10 'we don't recognise this button - remain silent

    end select

    if (n >= 0) then
      gui beep 50

    elseif (n = -1) then
      gui beep 700

    endif
  endif

  IrDev = 0
  IrButton = 0

end sub

'----------------------------------------------------------
'Load the default configuration file (if it can be found)
'It holds volume, backlight, digital or analogue output, and last station

sub ReadDefaultFile
  local integer i, volume, backlight, digital

  'Try loading Config0.csv, and if that doesn't work, try Config1.csv etc
  on error ignore
  String1 = dir$("Config?.csv", FILE)

  if (String1 <> "") then
    for i = 0 to 9
      String2 = "Config" + str$(i) + ".csv"

      Open String2 for input As 1

      if (MM.Errno <> 0) then
        WriteErr ("File open error " + String2)
        continue for
      endif

      'The file exists - we can try and read it as a CSV
      WriteMsg ("Reading " + String2)

      'Discard the first line of the file - this only contains column headings
      Input #1, String1

      if (MM.Errno <> 0) then
        Close 1

        WriteErr ("Read error " + String2)
        continue for
      endif

      'read a single line - which contains the defaults for a number of variables
      if (Not Eof(1)) then
        Input #1, volume, backlight, digital, NewFrequency, NewServiceID, NewCompID

        if (MM.Errno <> 0) then
          Close 1

          WriteErr ("Unintelligible defaults " + String2)
          continue for
        endif

        if ((volume < 0) or (volume > 63)) then volume = 20
        if ((backlight < 10) or (backlight > 100)) then backlight = 50
        CtrlVal(BVolume) = volume
        CtrlVal(BLight) = backlight
        Backlight backlight
        CtrlVal(Sdigital) = digital
	if (DigitalMode < 0) then DigitalMode = digital
        Close 1

        String1 = "Vol=" + str$(volume) + ", BackLight=" + str$(backlight)
	String1 = String1 + ", Digital=" + str$(digital)
	String2 = "Frequency=" + str$(NewFrequency) + ", Service=" + str$(NewServiceID)
	String2 = String2 + ", Component=" + str$(NewCompID)
	WriteMsg (String1)
	WriteMsg (String2)

	if (i <> 0) then
          Frequency = NewFrequency
	  ServiceID = NewServiceID
	  CompID = NewCompID
          WriteDefaultFile
	  Frequency = Frequency + 1 'to make it different so state machine gets tripped
        endif

        on error abort
	exit sub
      endif

      Close 1
    next
  endif

  'to reach here, we have not successfully read a configuration file. Set some defaults
  on error abort
  CtrlVal(BVolume) = 20
  CtrlVal(BLight) = 50
  Backlight 50
  CtrlVal(Sdigital) = 0
  DigitalMode = 0
  NewFrequency = 666 'the devil's number, and ABC in canberra

  'Set the flag that will cause a config file to be written. This will at least leave
  'a template that will give some help if the file is to be edited externally
  Dwait = 5000 'accelerated write timeout will trip in a few seconds from now
end sub

'- - - - - - - - - - - - - - - - - - - - - - - -
sub WriteDefaultFile

  if (RotateAndOpen ("Config", ".csv", 1) = 1) then
    'file open, ready to write
    on error ignore
    print #1, "Volume,BackLight,Digital Output,Frequency (kHz),DAB Service ID,DAB Component ID"
    print #1, CtrlVal(BVolume) "," CtrlVal(BLight) "," CtrlVal(Sdigital) "," Frequency "," ServiceID "," CompID
    Close 1
    on error abort

    if (D > 0) then print "Wrote config file"
  endif

  Dwait = 0 'turn off the timer
end sub

'- - - - - - - - - - - - - - - - - - - - - - - -
sub WritePresetFile
  local integer i

  if (RotateAndOpen ("Preset", ".csv", 1) = 1) then
    'file open, ready to write
    on error ignore
    print #1, "Band,Frequency (kHz),Station Name,DAB Service ID,DAB Component ID,AntCap,Bit1=Manual Station Name|Bit0=Preset Active"

    for i = 0 to NumPresets-1
      if ((Pfreq(i) >= AmMIN) and (Pfreq(i) <= AmMAX)) then
        String1 = "AM,"

      elseif ((Pfreq(i) >= FmMIN) and (Pfreq(i) <= FmMAX)) then
        String1 = "FM,"

      elseif ((Pfreq(i) >= DabMIN) and (Pfreq(i) <= DabMAX)) then
        String1 = "DAB,"

      else
        continue for
      endif

      String1 = String1 + str$(Pfreq(i)) + "," + chr$(34) + Pname(i) + chr$(34)
      String1 = String1 + "," + str$(Pid(i)) + "," + str$(Pcomp(i)) + "," + str$(Pcap(i))
      String1 = String1 + "," + str$(Pactive(i))

      if (String1 <> "") then
        print #1, String1
        if (D > 1) then print i ":" String1
      endif
    next

    Close 1
    on error abort
    if (D > 0) then print "Wrote preset file"
    WriteMsg ("Wrote Preset0.csv")
  endif

  Pwait = 0 'turn off the timer
end sub

'-----------------------------------------------------
'The quicksort implementation below is hard-coded to use the
'freq, sid, cid and active arrays, and will co-sort them together.
'The values in freq, sid, cid and active are used TOGETHER to determine order
'Look at a wiki on quicksort to see how it works.
'
'I originally coded a bubble sort, and even with 50 elements, was too slow!

dim integer pf, px, pc, pa

'this compare routine uses (a) the record at the given index and (b) a 'pivot' record
'stored in the variables defined above. This is the function that decides whether one
'record comes 'before' another. In this sense, record is the collection of data stored
'in several different arrays. The result of the comparison is as follows:
'  -1 if record before pivot
'  0 if record same as or 'equivalent to' the pivot
'  +1 if record after pivot

function compare (a as integer) as integer
  local integer r=-1

  'record invalid and pivot valid?
  if ((Pfreq(a) < AmMIN) and (pf >= AmMIN)) then
    compare = 1
    exit function
  endif

  'pivot invalid and record valid?
  if ((pf < AmMIN) and (Pfreq(a) >= AmMIN)) then
    compare = -1
    exit function
  endif

  'both invalid?
  if ((pf < AmMIN) and (Pfreq(a) < AmMIN)) then
    compare = 0
    exit function
  endif

  'at this point, record and pivot frequencies are valid. Need to formally compare
  if (Pfreq(a) < pf) then goto final_comp
  if (Pfreq(a) > pf) then goto exit_b

  'at this point the frequencies are both the same and both valid
  if (Pid(a) < px) then goto final_comp
  if (Pid(a) > px) then goto exit_b

  'at this point, both frequencies and station ID are valid
  if (Pcomp(a) < pc) then goto final_comp
  if (Pcomp(a) > pc) then goto exit_b

  'at this point, the frequencies, station ID and component IDs are identical
  r = 0
  goto final_comp

exit_b:
  r = 1

final_comp:
  'The result of comparing frequency, station ID and component ID is in r.
  'Now compare 'active' flags. The dormant goes to the back of the sorted list

  if ((Pactive(a) and 1) > pa)) then
    r = -1
  elseif ((Pactive(a) and 1) < pa)) then
    r = 1
  endif

  'the result of the comparison is in r
  compare = r

end function
'- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
'And here comes the sorting algorithm itself. The first call specifies the lowest
'index (which is always 0) and the highest index (which is NumPresets-1)

sub quicksort (lower as integer, upper as integer)
  local integer i, j, k, pivot
  local string s

  'i, j and pivot are indices into the arrays, not values themselves!
  i = lower
  j = upper

  'grab and remember the pivot value, as it may change as we shuffle below
  k = (lower + upper) >> 1 'set pivot in the middle of lower and upper
  pf = Pfreq(k)
  px = Pid(k)
  pc = Pcomp(k)
  pa = Pactive(k) and 1

  do while (i <= j)
    do while (compare(i) < 0) 'scan up from bottom until finding element not in sequence
      i = i + 1
    loop

    do while (compare(j) > 0) 'scan down from top until finding element not in sequence
      j = j - 1
    loop

    'remember that the i and j values are on either side of the pivot.
    'If not the same point, we know they are in reverse sequence and need to be swapped
    if (i < j) then
      k = Pfreq(i)
      Pfreq(i) = Pfreq(j)
      Pfreq(j) = k

      k = Pid(i)
      Pid(i) = Pid(j)
      Pid(j) = k

      k = Pcomp(i)
      Pcomp(i) = Pcomp(j)
      Pcomp(j) = k

      k = Pactive(i)
      Pactive(i) = Pactive(j)
      Pactive(j) = k

      k = Pcap(i)
      Pcap(i) = Pcap(j)
      Pcap(j) = k

      s = Pname(i)
      Pname(i) = Pname(j)
      Pname(j) = s
    endif

    if (i <= j) then
      i = i + 1
      j = j - 1
    endif
  loop

  'recursively sort the two reduced segments on either side of pivot
  if (lower < j) then quicksort (lower, j)
  if (i < upper) then quicksort (i, upper)
end sub

'- - - - - - - - - - - - - - - - - - - - - - - -
'Consolidate preset data currently in memory

sub CompactPresets
  local integer i, j

  if (NumPresets < 2) then exit sub
  quicksort (0, NumPresets-1)

  'null duplicates (actually, overwrite them. They will be adjacent to each other
  'Also count the actives
  NumActivePresets = 0
  i = 0

  for j = 1 to NumPresets - 1
    if ((Pfreq(i) <> Pfreq(j)) or (Pid(i) <> Pid(j)) or (Pcomp(i) <> Pcomp(j))) then
      'The record at j is different than the record at i. We want to keep j
      i = i + 1

      'if we have already skipped ('nulled') a record, copy the record at j
      if (i < j) then
        Pfreq(i) = Pfreq(j)
        Pid(i) = Pid(j)
        Pcomp(i) = Pcomp(j)
        Pcap(i) = Pcap(j)
        Pactive(i) = Pactive(j)
        Pname(i) = Pname(j)
      endif

    else
      'the record at j is the same as the record at i. We want to remove j
      if ((Pactive(i) and 1) = 0) then Pactive(i) = (Pactive(i) and 254) or (Pactive(j) and 1)

      if (Pname(i) = "") then
        Pname(i) = Pname(j)
	Pactive(i) = (Pactive(i) and 1) or (Pactive(j) and 2)

      elseif (((Pactive(i) and 2) = 0) and ((Pactive(j) and 2) <> 0)) then
        Pname(i) = Pname(j)
	Pactive(i) = (Pactive(i) and 1) or 2
      endif
    endif

    if ((Pfreq(i) >= AmMIN) and ((Pactive(i) and 1) = 1)) then NumActivePresets = i + 1
  next

  'at this point, i points to the last valid data
  NumPresets = i + 1

  'blank the remainder
  for i = NumPresets to MAX_PRESETS-1
    Pname(i) = ""
    Pfreq(i) = 0
    Pid(i) = -1
    Pcomp(i) = -1
    Pcap(i) = 0
    Pactive(i) = 0
  next

  if (NumActivePresets > MAX_RADIO) then NumActivePresets = MAX_RADIO 'LCD is limited!
  if (D > 1) then print "Compacted. Num Presets=" NumPresets ", Num Active Presets=" NumActivePresets : print
end sub

'- - - - - - - - - - - - - - - - - - - - - - - -
'Load the most recent preset file (if it can be found)

sub ReadPresetFile
  local integer i, r

  NumPresets = 0
  NumActivePresets = 0
  on error ignore

  'Try loading Preset0.csv, and if that doesn't work, try Preset1.csv etc
  String1 = dir$("Preset?.csv", FILE)

  if (String1 <> "") then
    for i = 0 to 9
      NumPresets = 0
      NumActivePresets = 0
      String2 = "Preset" + str$(i) + ".csv"

      Open String2 for input As 1

      if (MM.Errno <> 0) then
        WriteErr ("Open error " + String2)
        continue for
      endif

      'The file exists - we can try and read it as a CSV
      WriteMsg ("Reading " + String2)

      'Discard the first line of the file - this only contains column headings
      Input #1, String1

      if (MM.Errno <> 0) then
        Close 1

        WriteErr ("Read error " + String2)
        continue for
      endif

      'read each line of the CSV file until end of file
      Do While ((Not Eof(1)) and (NumPresets < MAX_PRESETS))
        Input #1, String1, Pfreq(NumPresets), Pname(NumPresets), Pid(NumPresets), Pcomp(NumPresets), Pcap(NumPresets), Pactive(NumPresets)

        if (String1 = "") then exit do 'MMBASIC bug? - works ok unix format text files, but not DOS! (reads extra line!)

        if (D > 1) then
          print NumPresets+1 ": Freq=" Pfreq(NumPresets) " Name=" Pname(NumPresets);
          print " ID=" Pid(NumPresets) " Comp=" Pcomp(NumPresets);
	  print  " AntCap=" Pcap(NumPresets) " Active=" Pactive(NumPresets)
        endif

        NumPresets = NumPresets + 1
      Loop

      Close 1
      CompactPresets

      if (NumPresets > 0) then
        if (D > 0) then print "Loaded " NumPresets " presets" : print
	if (i > 0) then WritePresetFile 'Save what we just read as Preset0.csv - because we didn't read Preset0.csv
        exit for
      endif
    next
  endif

  on error abort
end sub

'- - - - - - - - - - - - - - - - - - - - - - - -
'function to find a station's name from presets already in memory
'this lets us asynchronously edit names and to prevent the edits being overwritten by
'DAB data

function MatchName (freq as integer, defname as string, sid as integer, cid as integer, ac as integer) as string
  local integer i

  for i = 0 to (NumPresets - 1)
    if ((sid >= 0) and (sid <> Pid(i))) then continue for
    if ((cid >= 0) and (cid <> Pcomp(i))) then continue for
    if (Pfreq(i) = freq) then exit for
  next

  if (defname = "") then
    MatchName = str$(freq) + " kHz"
  else
    MatchName = defname
  endif

  if (i < NumPresets) then
    'we found an entry in the preset table - Should we update preset?
    'Bit 1 of Pactive() means we set the station name manually

    'Is there an opportunity to update the name in the presets file?
    if (((Pactive(i) and 2) = 0) and (len(defname) > len(Pname(i))) and (right$(defname,4) <> " kHz")) then
      Pname(i) = defname
      Pwait = 1
    endif

    if (((Pactive(i) and 2) = 0) and (right$(Pname(i),4) = " kHz") and (defname <> "") and (right$(defname,4) <> " kHz")) then
      Pname(i) = defname
      Pwait = 1
    endif

    if (Pname(i) <> "") then MatchName = Pname(i)

    'Shall we retune this station because the AntCap setting has changed?
    if ((ac <> 0) and (Pcap(i) <> AntCap) and (AntCap = 0)) then
      NewFrequency = Pfreq(i)
      NewServiceID = Pid(i)
      NewCompID = Pcomp(i)
      AntCap = Pcap(i)
    endif

  else if ((defname <> "") and (NumPresets < MAX_PRESETS)) then

    'We didn't find the entry in the preset table, but we want to save this entry
    Pfreq(NumPresets) = freq
    Pname(NumPresets) = defname
    Pid(NumPresets) = sid
    Pcomp(NumPresets) = cid
    Pcap(NumPresets) = 0
    Pactive(NumPresets) = 0 'the default for new entries is to make inactive and learned from RDS/DAB
    NumPresets = NumPresets + 1
    Pwait = 1
  endif

end function

'-----------------------------------------------------
'SI4689 radio chip access routines - these are rather specific to the hardware
'Will not run without working hardware connected to the micromite plus
'The state machine below follows the data-sheet flowchart reasonably closely.

dim integer SpiBuf(READ_REPLY_BUFSIZE) 'used as a SPI read buffer

'Read the SI4689 radio chip SPI reply
sub Read_Radio_Reply (len as integer)
  if (len > READ_REPLY_BUFSIZE) then len = READ_REPLY_BUFSIZE

  'CMD=0x00 (RD_REPLY)
  pin(P_SI4689_CS) = 0
  spi write 1,0
  Pause .1 'wait 100us
  spi read len, SpiBuf()
  pin(P_SI4689_CS) = 1

end sub

'- - - - - - - - - - - - -
'Put radio into firmware loading mode

sub Issue_LOAD_INIT

  'CMD=0x06 (LOAD_INIT)
  'ARG1=0 (Fixed - not configurable)
  pin(P_SI4689_CS) = 0
  spi write 2,&H06,&H00
  pin(P_SI4689_CS) = 1
  if (D > 2) then print "*06,00 Load Init"

end sub

'- - - - - - - - - - - - -
'Set a radio property
'The values are passed as LSB/MSB pairs to avoid having to recalculate in real time
sub SetProperty(mp AS integer, lp AS integer, mv AS integer, lv AS integer)

  'CMD=0x13 (SET_PROPERTY)
  'ARG1=00 (Fixed - not configurable)
  'ARG2-3 (Little Endian LSB first. property)
  'ARG4-5 (Little Endian LSB first. value)
  pin(P_SI4689_CS) = 0
  spi write 6,&H13,&H00,lp,mp,lv,mv
  pin(P_SI4689_CS) = 1

  if (D > 2) then print "*13,00 Property " hex$(mp,2) hex$(lp,2) "=" hex$(mv,2) hex$(lv,2)
  CTSwait = 9
  SpiWait = 1 'Don't need to wait for next tick to continue - property commands can chain

end sub

'- - - - - - - - - - - - -
'Set volume of the built-in DAC on the radio chip

sub SetVolume (v as integer)

  'Set analogue output (DAC) volume
  'property=0x0300 (AUDIO_ANALOG_VOLUME)
  '0x00=mute, 0x01=62dB attenuation, ... 0x3f=no attenuation - Volume in 1dB steps
  SetProperty(&h03, &h00, 0, v)

end sub

'- - - - - - - - - - - - -
'Tell the built-in DAC on the radio chip to enter or exit mute
'0=turn mute off, 3=mute both channels (left and right)

sub SetMute (m as integer)

  'property=0x0301 (AUDIO_MUTE)
  SetProperty(&h03, &h01, 0, m)

end sub

'-----------------------------------------------------
'Convert non printable ascii values to a space and return as a one character string

function clean_chr (c as integer) as string
  if ((c < 32) or (c > 127)) then
    clean_chr = " "
  else
    clean_chr = chr$(c)
  endif
end function

'-----------------------------------------------------
'Remove leading and trailing spaces from String1

function truncate_string (s as string) as string
  local integer i, n, f, l

  truncate_string = s
  n = len(s)
  if (n = 0) then exit function

  'find the first and last non-whitespace character
  f = 0
  l = 0

  for i = 1 to n
    if (mid$(s, i, 1) <> " ") then
      if (f = 0) then f = i 'remember the first non-white character
      l = i 'remember the last non-white character
    endif
  next

  'If the entire string is space, then f will be zero
  if (f = 0) then
    truncate_string = ""
    exit function
  endif

  'truncate string between our two markers
  truncate_string = mid$(s, f, l-f+1)

end function

'-----------------------------------------------------
'Take a (potentially) long string, and display across lines RLN4-RLN8
'(note - calling string s is modified by this routine!)

sub display_string (s as string)
  local integer i, j, l=RLN4
  local s2 as string

  'The LCD display is exactly 50 characters wide
  'If the string is longer, will need to break over several lines
  i = len(s)

  if ((D > 0) and (i > 0)) then print "Displaying string:" s

  do while ((i > 0) and (l <= RLN8))
    if (i <= 50) then
      s2 = ""

    else 'i > 50
      'Need to cut at a word boundary
      for j = 50 to 1 step -1
        if (mid$(s, j, 1) = " ") then exit for
      next

      if (j > 1) then
        'We found whitespace
        s2 = right$(s, i-j)
        s = left$(s, j-1)
      else
        'Rare case of no whitespace - just cut ungracefully
        s2 = right$(s, i-50)
        s = left$(s, 50)
      endif
    endif

    if (CtrlVal(l) <> s) then CtrlVal(l) = s
    s = s2
    i = len(s)
    l = l + 1
  loop

  'blank remaining lines
  do while (l <= RLN8)
    CtrlVal(l) = ""
    l = l + 1
  loop

end sub

'-----------------------------------------------------
'If there is a complete RDS text string, display the string
'The parameter will be:
' 0 = don't blank anything
' 1 = blank section 1
' 17= blank section 2

sub TryToDispRDS (blank as integer)
  local integer n

  String1 = ""
  String2 = ""

  'Accumulate the two RDS strings in String1 and String2
  for n = 0 to 15
    String1 = String1 + ProgData(n)
    String2 = String2 + ProgData(16+n)

    if (blank <> 0) then ProgData(blank-1+n) = ""
  next

  if (len(String1) = 64) then String1 = truncate_string (String1) else String1 = ""
  if (len(String2) = 64) then String2 = truncate_string (String2) else String2 = ""

  if ((String1 = "") and (String2 = "")) then exit sub 'nothing to display

  'at least one of the strings has something to display
  if (String1 = "") then
    display_string (String2) 'only String2 is non-null
    exit sub
  endif

  'String1 is not null - is its contents the same as string2?
  if ((String2 <> "") and (String1 <> String2)) then String1 = String1 + " / " + String2

  'Finally, the complete text to display is in String1
  display_string (String1)

end sub

'-----------------------------------------------------
sub Append_Err_File (err as string)
  on error ignore
  open "Error.csv" for append as #1

  String2 = str$(Frequency) + "," + str$(ServiceID) + "," + str$(CompID)
  print #1, DATE$ " " TIME$ "," err "," str$(State) "," String2

  close #1

  on error abort
end sub

'-----------------------------------------------------
'convert a signed byte or 16 bit word into a signed integer

function Conv8bitSignedInt (x as integer) as integer
  if (x > 127) then
    Conv8bitSignedInt = x - 256
  else
    Conv8bitSignedInt = x
  endif
end function

'-----------------------------------------------------
'State is the radio SPI state machine value. When State=0, the state machine is idle
'Rstate is the tuning status of the radio chip.

dim integer State=1, Rstate=Runtuned, Address, Reply_Size=4, StatusBits=0, AntCap=0
dim integer ReadAntCap, WM8804=0, Radio_Running=-1, NewStatus=0, ExpectTune=0, NumProp
dim integer PropID, IntMask=0, CTSwait=0, LastDabServList=-1, AudioOut=0
dim integer DabServiceID(MAX_DAB_SERVICES-1), DabComponentID(MAX_DAB_SERVICES-1)
dim integer ReadFreq, SNR, CNR, M, FibErr, Quality, SigStrength, FreqOffset, Stereo
dim integer NumServices, Min_SNR, Min_SigStren, HP_out=0, DispRefresh=0

dim STRING Status length 11, Band length 3, DabServiceName(MAX_DAB_SERVICES-1) length NameLen

dim float SigLevel, maSNR, maCNR, maSig, maQual

Band=""

'- - - - - - - - - - - - - - - - - - - - - - - -
'Create a new preset entry, merging with existing entry if there's already a similar
'entry. Returns non-zero if a change was made to the preset table

function AddPreset (freq as integer, defname as string, sid as integer, cid as integer, ac as integer) as integer
  local integer i, j

  AddPreset = 0

  'Scan the existing preset entries to see if there is already a matching entry
  'The two checks for Pid and Pcomp not equalling -1 is so that if we get
  'a new DAB channel that has a real Pid and Pcomp (as opposed to the placeholder
  'values of -1), then the new DAB channel will overwrite the placeholders
  for i = 0 to (NumPresets - 1)
    if ((sid <> Pid(i)) and (Pid(i) >= 0)) then continue for
    if ((cid <> Pcomp(i)) and (Pcomp(i) >= 0)) then continue for
    if (Pfreq(i) = freq) then exit for
  next

  'Did we find a match on the basis of frequency / SID / CID?
  if (i < NumPresets) then 'yes

    'If no antcap value specified, try and find another entry in the preset table
    'for the same frequency that has an antcap value
    if (ac = 0) then
      for j=0 to (NumPresets-1)
        if ((Pfreq(j) = freq) and (Pcap(j) > 0)) then
          ac = Pcap(j)
	  exit for
	endif
      next
    endif

    'Set the Pid and Pcomp values - they may be the same as what's there anyway
    'this will overwrite -1, if that's what we happened to find
    if ((Pid(i) <> sid) or (Pcomp(i) <> cid)) then
      Pid(i) = sid
      Pcomp(i) = cid
      Pname(i) = ""
      AddPreset = 1
    endif

    if ((Pname(i) = "") and (defname <> "")) then
      Pname(i) = defname
      AddPreset = 1
    endif

    'If the current AntCap value is default (auto), then set to the supplied value
    if (Pcap(i) <> ac) then
      Pcap(i) = ac
      AddPreset = 1
    endif

    if (AddPreset > 0) then
      Pwait = 1
      Pactive(i) = 0
    endif

    exit function
  endif

  'at this point, we didn't find a matching entry for the preset
  if (NumPresets >= MAX_PRESETS) then
    WriteErr ("Preset Table Full")
    exit function 'no room for more
  endif

  'No match. Create a new entry in the table
  Pname(i) = defname
  Pfreq(i) = freq
  Pid(i) = sid
  Pcomp(i) = cid
  Pcap(i) = ac
  Pactive(i) = 0 'defaults to being inactive
  NumPresets = NumPresets + 1
  AddPreset = 1
end function

'- - - - - - - - - - - - - - - - - - - - - - - -
Sub CheckSpi
  local integer next_state, i, a, b, c, e, n, id, comp
  static integer IntState=0
  local string SL length NameLen

  'Reset SpiWait to zero means we will wait until the next tick before returning
  SpiWait=0
  '- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  'Check for pending frequency-change request (NewFrequency nonzero)
  if (NewFrequency >= AmMIN) then
    i = RadioMode

    'We are changing frequency, but do we need to change band (load new FW) as well?
    if ((NewFrequency >= AmMIN) and (NewFrequency <= AmMAX)) then
      i = MODE_AM
      Min_SNR = AmSNR
      Min_SigStren = AmMinStren

    elseif ((NewFrequency >= FmMIN) and (NewFrequency <= FmMAX)) then
      i = MODE_FM
      Min_SNR = FmSNR
      Min_SigStren = FmMinStren

    elseif ((NewFrequency >= DabMIN) and (NewFrequency <= DabMAX)) then
      i = MODE_DAB
      Min_SNR = DabCNR
      Min_SigStren = DabMinStren

    else 'SW bug
      String1 = "Can" + chr$(39) + "t scan to freq " + str$(NewFrequency)
      WriteErr (String1)
      Append_Err_File (String1)

      if (D = 0) then
        Reboot = 1 'almost instant reboot
      else
        Reboot = 2000 '10 seconds and then reboot
      endif

      exit sub
    endif

    'If a band change required, cold-reset the radio and then load the firmware for
    'the requested band
    if ((i >= 0) and (i <> RadioMode)) then 'band change required
      if (Radio_Running > 0) then
        State = -2 'Mute before loading the new band's firmware
      else
        State = 1 'load the new band's firmware immediately
      endif

      RadioMode = i
      goto BLANKIT

    elseif (State = 0) then 'Band is OK. But only do frequency change from state 0
      State = 300 'Set a new frequency in the current operating band
      ServiceID = -1 'NewServiceID and NewCompID, will be used when tuned
      CompID = -1
      LastDabServList = -1

    BLANKIT:
      ActiveProgData = -1
      IntMask = 0
      Frequency = NewFrequency
      NewFrequency = -1
      Rstate = Runtuned
      maSNR = -99
      maCNR = -99
      maSig = -1
      maQual = -1
      AudioOut = 0
      RdsNames=0

      'Clear the FM RDS strings
      for n = 0 to 3
        ProgName(n) = ""
      next

      for n = 0 to 31
        ProgData(n) = ""
      next

      for n = 0 to (MAX_SET-1)
        RdsNameSet(n) = ""
      next

      'Check if there is an AntCap value in the presets file for the new frequency
      'If not, set AntCap to 0, which tells the Si4689 to derive AntCap from programmed
      'straight line formula.
      AntCap = 0

      for n = 0 to (NumPresets-1)
        if (Frequency = Pfreq(n)) then
          AntCap = Pcap(n)
          exit for
        endif
      next

      ClearStatus
    endif
  endif

  '- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  'Read the radio status every timer tick, and check for interrupts
  if (Radio_Running > 0) then

    if (Reply_Size < 4) THEN Reply_Size=4
    Read_Radio_Reply(Reply_Size)

    'Response:
    'byte 0: 0x80=CTS, 0x40=Cmd Error
    '        0x10=Digital Service (DAB) Interrupt
    '        0x04=RDS (FM) Interrupt
    '        0x01=Seek/Tune Complete Interrupt
    'byte 1: 0x20=Digital Service Event Interrupt
    'byte 2: don't care (ignore this byte)
    'byte 3: bits 7,6=system state, bits 5-0 are various error indications that if
    '        non-zero are fatal
    '        Will read as &HFF when there is nothing connected

    'If no radio hardware detected, then SPI read will return 255
    if ((State < 20) and (SpiBuf(0) = 255)) then
      String1 = "comms failure"
      goto DO_REBOOT
    endif

    'Check for cmd error - basically when you send a command that the Si4689 doesn't
    'recognise in the current mode, or you send a command with a parameter that is
    'beyond the allowable limits - ie might be a SW bug ;-)
    'But I have a strong feeling that not all cmd errors are bugs, or serious bugs.
    'eg some cmd errors appear to happen due to timing. For example, if the state
    'machine is in the process of changing frequency, and a message such as an FM RDS
    'message comes in related to the *previous* frequency (eg a timing thing between
    'the moment of change, and clearing out the stuff that the radio chip had queued
    'up), this can cause the state machine to get a little ahead of itself. I guess
    'this is a bug. I am not sure how to avoid this - because few radio responses tell
    'you what frequency or channel the response relates to - you can't tell if they're
    'stale.
    if ((SpiBuf(0) and 64) <> 0) then
      if ((Reboot <= 0) and (D > 0)) then
        String1 = "Si4689 CMD error:" + str$(State)
        String1 = String1 + "," + hex$(SpiBuf(0),2) + hex$(SpiBuf(1),2) + hex$(SpiBuf(3),2)

        if (Reply_Size < 6) then Read_Radio_Reply(6) 'Get the error code
        String1 = String1 + "," + hex$(SpiBuf(4),2) + hex$(SpiBuf(5),2)

        WriteErr (String1)
        Append_Err_File (String1)
      endif

      'In order to clear the CMD error, a command needs to be successfully sent
      'Send a benign command (GET_SYS_STATE) to achieve this
      'CMD=0x09 (GET_SYS_STATE)
      'ARG1=00 (Fixed - not configurable)
      pin(P_SI4689_CS) = 0
      spi write 2,&H09,&H00
      pin(P_SI4689_CS) = 1
      if (D > 2) then print "*09,00 Get Sys State - clear error"
      CTSwait = 9
      SpiWait = 1
      exit sub
    endif

    'Check for Si4689 'fatal' error. (Fatal errors require reboot of radio chip)
    if ((SpiBuf(3) and 63) <> 0) then
      'Si4689 fatal errors are described in AN649
      '0x01 = Non Recoverable Error. The Si4689 internal keep alive (watchdog) expired
      '0x02 = Si4689 arbiter error - its internal to radio chip, whatever this is
      '0x04 = Control interface dropped data during write. SPI clock rate too high.
      '0x08 = Control interface dropped data during read. SPI clock rate too high.
      '0x10 = Si4689 DSP frame overrun error
      '0x20 = RF Front end of Si4689 is in an unexpected state. Internal to the chip.
      String1 = "fatal"
      goto DO_REBOOT
    endif

    'Check for Interrupts we are interested in. At present, only FM and DAB bands
    'configured for interrupts
    if (RadioMode = MODE_DAB) then
      if ((SpiBuf(0) and 16) <> 0) then IntMask=IntMask or 1 'Digital Service (DAB) Int
      if ((SpiBuf(1) and 32) <> 0) then IntMask=IntMask or 4 'Digital Event (DAB) Int

    elseif (RadioMode = MODE_FM) then
      if ((SpiBuf(0) and 4) <> 0) then IntMask=IntMask or 2 'RDS (FM) interrupt

    endif

    if (IntMask <> 0) then
      if (D > 0) then
        print "Int:" State "," hex$(SpiBuf(0),2) hex$(SpiBuf(1),2) hex$(SpiBuf(3),2)
      endif

      IntState = 60000 'this is the interrupt service entry state
    endif
  endif

  '- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  'If we are waiting on a CTS, has the radio signalled clear to send yet?
  'Only check if the CTS downcounter is running
  '
  'The CTS errors are sometimes caused by not giving the radio enough time to complete
  'a command (like boot, scan, and things that take a while). In this case, the status
  'bits will be non-zero and they will tell you something about what the problem might
  'be. In these cases, if the radio was already tuned to a station and playing, it will
  'continue to play even though the error has been thrown. These situations are probably
  'due to a bug in this basic programme.
  '
  'I have also encountered CTS errors that return all zeros below, and at the same time
  'the radio chip is locked up and outputting a high pitched tone, or noise, or something
  'else that strongly suggests the radio chip firmware has crashed for some reason. These
  'don't appear to be caused by this programme, but by something else environmental.
  'I have noticed this kind of problem when turning the radio off (the instant that the
  'mains power switch breaks the mains voltage) and when my gas heater igniter or gas
  'stove igniter or washing machine turns on (all obviously wideband RF noise or wideband
  'mains voltage noise). The radio chip seems to be prone to lock up with noise.
  '
  'To reduce the chance that mains noise was conducted via the DC power supply, I built
  'a rectified transformer fed DC supply. A bridge rectifier is filtered by 2000uF, which
  'in turn feeds an additional 44,000uF of filtering via 2R7, all caps low ESR. I have
  'liberally clamped several high inductance ferrites on the mains feed+rectified DC feed.
  'I don't expect much of the above noise is being conduted in via the DC feed. I have
  'gone to some degree of trouble with the design and layout of the DC supply. I should
  'never say never regarding having built a perfect power supply, however I am a
  'professional engineer with almost 40 years experience, and I was happy with this PSU.
  'My suspicion is that the noise is getting into the radio via the antenna, not PSU.
  'It may be because I live in a marginal reception area, there is some susceptibility
  'to noise with low signal or low SNR/CNR levels?

  if (CTSwait > 0) then
    'Keep waiting until the CTS bit has been set, and the status bits are set (if the
    'status bits are specified)
    if (((SpiBuf(0) and 128) = 128) and ((SpiBuf(0) and StatusBits) = StatusBits)) then
      CTSwait = 0
      StatusBits = 0

    else 'conditions for continuing are not yet satisfied
      CTSwait = CTSwait - 1

      if (CTSwait = 0) then
        String1 = "CTS timeout"

        DO_REBOOT:
        if (Reboot <= 0) then
          SetMux (MUX_MUTE)
          String1 = "Si4689 " + String1 + " error:" + str$(State)
          String1 = String1 + "," + hex$(SpiBuf(0),2) + hex$(SpiBuf(1),2) + hex$(SpiBuf(3),2)
          WriteErr (String1)
          Append_Err_File (String1)

          if (D = 0) then
            Reboot = 1 'almost instant reboot
          else
            Reboot = 2000 '10 seconds and then reboot
          endif
        endif
      endif

      exit sub 'don't proceed with state machine because we are aborting, or waiting
    endif
  endif

  '- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  'We can process interrupts when state returns to 0, otherwise continue with current flow
  if (State = 0) then
    'Return if State is 0 and there are no interrupts to process
    if (IntState = 0) then exit sub
    State = IntState
  endif

  '- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  'To get here, there is something in the state machine that needs attention
  'Preemptively guess the next state so we don't have to repeat every case statement
  next_state = State + 1

  if (D > 1) then print "State:" State " IntState:" IntState " IntMask:" hex$(IntMask,2)
  Reply_Size = 4 'Set back to the default value after reading the reply above

  '- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  'When State=0, the state machine is idle
  select case State
    case -2
      'If switching from one band to another, mute before switching
      SetMux (MUX_MUTE)
      SetMute (3) 'mute the DAC output

    case -1
      SetVolume (0)
      next_state = 1 'state 0 is idle, remember

    case 1
      'THIS IS THE COLD/WARM BOOT ENTRY
      'RESET THE RADIO CHIP and move into "POWER UP" mode
      SetMux (MUX_MUTE)

      'Refer to SI4689 application note 649: eg s5.1 p381
      pin(P_SI4689_CS) = 1
      pin(P_SI4689_RES) = 1

    case 2
      pin(P_SI4689_RES) = 0

    case 3
      pin(P_SI4689_RES) = 1

    case 4
      'CMD=0x01 (Power_UP)
      'ARG1=00 (No interrupt on CTS)
      'ARG2=17 (OSC/Buffer=Xtal, OSX size=7)
      'ARG3=20 (Xtal startup bias=320uA)
      'ARG4-7=00 F8 24 01 (Little Endian LSB first. Xtal Freq=19200000Hz)
      'ARG8=10 (CTUN?)
      'ARG9-12=10 00 00 00 (Fixed - not configurable)
      'ARG13=10 (Xtal run bias=160uA)
      'ARG14-15=00 00 (Fixed - not configurable)
      pin(P_SI4689_CS) = 0
      spi write 16,&H01,&H00,&H17,&H20,&H00,&HF8,&H24,&H01,&H10,&H10,&H00,&H00,&H00,&H10,&H00,&H00
      pin(P_SI4689_CS) = 1
      if (D > 2) then print "*01,00,17,20,00,f8,24,01,10,10,00,00,00,10,00,00 Power Up"
      Radio_Running=1
      CTSwait = 9

    '- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
    'LOAD BOOTLOADER INTO RADIO CHIP
    'Refer to SI4689 application note 649: eg s5.1 p381
    case 5
      Issue_LOAD_INIT
      WriteMsg ("Loading bootloader")
      CTSwait = 9

    case 6,7
      'Copy the bootloader code using two SPI transfers
      Address = PEEK(CFUNADDR LOADERBIN)

      if (State=6) then
        a=0 : b=4092
      else
        a=4096 : b=5792
      endif

      'CMD=0x04 (HOST_LOAD)
      'ARG1-3=00 00 00 (Fixed - not configurable)
      'ARG4 .. 4095 (image bytes - sequential, fill command out to 4096 bytes)
      pin(P_SI4689_CS) = 0
      spi write 4,&H04,&H00,&H00,&H00
      if (D > 2) then print "*04,00,00,00 Host Load"

      FOR i = a TO b step 4
        spi write 4,PEEK(BYTE i+Address),PEEK(BYTE i+Address+1),PEEK(BYTE i+Address+2),PEEK(BYTE i+Address+3)
      next i

      pin(P_SI4689_CS) = 1
      CTSwait = 9

    case 8
      'Application note says wait another 4ms - so wait one more tick
      CTSwait = 0

    '- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
    'LOAD BAND SPECIFIC FIRMWARE INTO RADIO CHIP
    'Refer to SI4689 application note 649: eg s5.1 p381
    case 9
      Issue_LOAD_INIT

      'CMD=0x05 (FLASH_LOAD)
      'ARG1-3=00 00 00 (Fixed - not configurable)
      'ARG4-7=start address in flash (Little Endian LSB first)
      'ARG8-11=00 00 00 00 (Fixed - not configurable)
      pin(P_SI4689_CS) = 0
      spi write 4,&H05,&H00,&H00,&H00

      'Load firmware for selected radio band
      if (RadioMode = MODE_AM) then
     	spi write 4,&H00,&HE0,&H11,&H00
        Band="AM"
      elseif (RadioMode = MODE_FM) then
        spi write 4,&H00,&H60,&H00,&H00
        Band="FM"
      else
        spi write 4,&H00,&H20,&H09,&H00
        Band="DAB"
      endif

      'FLASH_LOAD trailing four zero arguments
      spi write 4,&H00,&H00,&H00,&H00

      'Now just wait for the CTS as the radio chip loads image from its FLASH rom
      pin(P_SI4689_CS) = 1
      if (D > 2) then print "*05,00,00,00, ... Flash Load " Band
      if (State = 16) then WriteMsg ("Loading " + Band + " firmware")
      CTSwait = 100

    '- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
    'BOOT THE BAND SPECIFIC FIRMWARE ON RADIO CHIP
    'Refer to SI4689 application note 649: S5.1 p381
    case 10
      'CMD=0x07 (BOOT)
      'ARG1=00 (Fixed - not configurable)
      pin(P_SI4689_CS) = 0
      spi write 2,&H07,&H00
      pin(P_SI4689_CS) = 1
      if (D > 2) then print "*07,00 Boot"

      WriteMsg ("Booting " + Band + " firmware")
      CTSwait = 60

    case 11
      InitWM8804

      'still don't know if hardware is fitted. Try to read WM8804 ID to see if chip there
      'The 16 bit ID is stored in two registers
      pin(P_WM8804_CSB) = 0
      spi write 1, &H80
      spi read 1, SpiBuf()
      pin(P_WM8804_CSB) = 1
      a = SpiBuf(0) << 8

      pin(P_WM8804_CSB) = 0
      spi write 1, &H81
      spi read 1, SpiBuf()
      pin(P_WM8804_CSB) = 1
      a = a + SpiBuf(0)

      if (a = 1416) then '1416=0x0588
        WM8804 = 1
        SetMux (MUX_MUTE) 'reset the WM8804 again, and put it back into mute mode
        WriteMsg ("Found WM8804 device")
      else
        WriteMsg ("Failed to find WM8804 device")
        if (D > 0) then print "WM8804 DeviceID:" hex$(a,4)
	gui hide Sdigital, Vaudio 'don't show digital output option if no hardware!
      endif

      SetMute (3) 'mute
      HP_out = pin(P_HEADPHONE) 'Checked and set at boot, and then status updates

      'state machine branches to band-specific commands
      if (RadioMode = MODE_AM) then
        next_state = 50

      elseif (RadioMode = MODE_FM) then
        next_state = 100

      else 'it must be DAB
        next_state = 200

      endif

    '- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
    'This section is AM only
    case 50
      'property=0x4200 (AM_VALID_MAX_TUNE_ERROR)
      SetProperty(&h42, &h00, 0, 75) 'Default setting - 75

    case 51
      'property=0x4201 (AM_VALID_RSSI_TIME)
      SetProperty(&h42, &h01, 0, 8) 'Default 8ms

    case 52
      'property=0x4202 (AM_VALID_RSSI_THRESHOLD)
      SetProperty(&h42, &h02, 0, 35) 'Default 35dBuV

    case 53
      'property=0x4203 (AM_VALID_SNR_TIME)
      SetProperty(&h42, &h03, 0, SnrSettle) 'settling time before evaluating SNR

    case 54
      'property=0x4204 (AM_VALID_SNR_THRESHOLD)
      SetProperty(&h42, &h04, 0, AmSNR) 'if SNR exceeds this, consider it a valid channel

    case 55
      'property=0x4100 (AM_SEEK_BAND_BOTTOM)
      SetProperty(&h41, &h00, AmMIN >> 8, AmMIN and 255) 'AmMIN must be multiple of 9kHz!

    case 56
      'property=0x4101 (AM_SEEK_BAND_TOP)
      SetProperty(&h41, &h01, AmMAX >> 8, AmMAX and 255) '1710 kHz

    case 57
      'property=0x4102 (AM_SEEK_FREQUENCY_SPACING)
      SetProperty(&h41, &h02, 0, AmDELTA and 255) 'Seek in steps of AmDELTA kHz
      next_state = 300

    '- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
    'This section is FM only
    case 100
      'Set SNR threshold for decoding stereo
      'property=0x3704 (FM_Blend_SNR_LIMITS)
      'Normal value is 0x180f
      SetProperty(&h37, &h04, &h18, &h0f)

    case 101
      'Set RSSI limits
      'property=0x3700 (FM_Blend_RSSI_LIMITS)
      'normal value is 0x2010
      SetProperty(&h37, &h00, &h20, &h10)

    case 102
      'property=0x1710 (FM_TUNE_FE_VARM)
      SetProperty(&H17, &H10, FmSlope >> 8, FmSlope and 255)

    case 103
      'property=0x1711 (FM_TUNE_FE_VARB)
      SetProperty(&H17, &H11, FmIntercept >> 8, FmIntercept and 255)

    case 104
      'property=0x1712 (FM_TUNE_FE_CFG)
      'ARG=0000 (Close VHF Switch)
      SetProperty(&H17, &H12, &H00, FmSwitch) '00 means switch open, 01 means switch closed

    case 105
      'property=0x3100 (FM_SEEK_BAND_BOTTOM)
      i = FmMIN / 10
      SetProperty(&h31, &h00, i >> 8, i and 255) '87500 kHz - in units of 10kHz

    case 106
      'property=0x3101 (FM_SEEK_BAND_TOP)
      i = FmMAX / 10
      SetProperty(&h31, &h01, i >> 8, i and 255) '107900 kHz - in units of 10kHz

    case 107
      'property=0x3102 (FM_SEEK_FREQUENCY_SPACING)
      SetProperty(&h31, &h02, 0, (FmDELTA \ 10) and 255) 'Seek (in units of 10kHz)

    case 108
      'property=0x3200 (FM_VALID_MAX_TUNE_ERROR)
      'Default 114 bppm (max frequency error before setting AFC rail indicator)
      SetProperty(&h32, &h01, 0, 114) 'Default 114

    case 109
      'property=0x3201 (FM_VALID_RSSI_TIME)
      'Default 15ms, but can set longer
      SetProperty(&h32, &h01, 0, 25) 'Default 15ms

    case 110
      'property=0x3202 (FM_VALID_RSSI_THRESHOLD)
      'Default is 17, but can set higher
      SetProperty(&h32, &h02, 0, 17) 'Default 17dBuV

    case 111
      'property=0x3203 (FM_VALID_SNR_TIME)
      'default is 40ms
      SetProperty(&h32, &h03, 0, SnrSettle) 'settling time before evaluating SNR

    case 112
      'property=0x3204 (FM_VALID_SNR_THRESHOLD)
      'default is 10dB, but can set lower
      SetProperty(&h32, &h04, 0, FmSNR) 'if SNR exceeds this, consider it a valid channel

    case 113
      if (CapSearch > 0) then goto RETRY_TUNE

      ' - - - - - RDS only below here
      'property=0x3C00 (FM_RDS_INTERRUPT_SOURCE)
      'Interrupt enable=0x0018 (0x10=RDSTPPTY, 0x08=RDSPI, 0x02=Sync change, 0x01=FIFO has data
      SetProperty(&h3c, &h00, &h00, &h01)

    case 114
      'property=0x3C01 (FM_RDS_INTERRUPT_FIFO_COUNT)
      'Defines a minimum number of messages to be received before interrupting
      SetProperty(&h3c, &h01, &h00, 8) 'must be in the range 0..25

    case 115
      'Set RDS settings and set RDS block error thresholds
      'property=0x3C02 (FM_RDS_CONFIG)
      'Block error threshold for block in bits 7,6 and 5,4
      '  00 = accept no errors or corrections
      '  01 = accept 1-2 bits corrected
      '  10 = accept 3-5 bits corrected
      '  11 = accept uncorrectable errors
      'RDS enabled=0x0001 (enabled)
      ' Try setting to &H51?
      SetProperty(&h3c, &h02, &h00, &h01) '(&H51) to accept 1-2 corrected errors, (&HA1)=3-5
      next_state = 300
    '- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
    'This section is DAB only
    case 200
      'property=0x1710 (DAB_TUNE_FE_VARM)
      SetProperty(&H17, &H10, DabSlope >> 8, DabSlope and 255)

    case 201
      'property=0x1711 (DAB_TUNE_FE_VARB)
      SetProperty(&H17, &H11, DabIntercept >> 8, DabIntercept and 255)

    case 202
      'property=0x1712 (DAB_TUNE_FE_CFG)
      'ARG=0001 (Open VHF Switch)
      SetProperty(&H17, &H12, &H00, DabSwitch) '00 means switch open, 01 means switch closed

    case 203
      'property=0xB400 (DAB_XPAD_ENABLE)
      'ARG=0005 (Enable DLS packets, MOT slideshow)
      'Note: needs time for data ensemble to become available, check in RSQ loop
      SetProperty(&HB4, &H00, &H00, &H05)

    case 204
      'property=0xB200 (DAB_VALID_RSSI_TIME)
      SetProperty(&hB2, &h00, 0, 200) '200ms

    case 205
      'property=0xB201 (DAB_VALID_RSSI_THRESHOLD)
      SetProperty(&hB2, &h01, 0, 30) '30dBuV

    case 206
      'property=0xB202 (DAB_VALID_ACQ_TIME)
      SetProperty(&hB2, &h02, 7, 255) '2047ms

    case 207
      'property=0xB203 (DAB_VALID_SYNC_TIME)
      SetProperty(&hB2, &h03, 5, 0) '1280ms

    case 208
      'property=0xB204 (DAB_VALID_DETECT_TIME)
      SetProperty(&hB2, &h04, 0, 70) '70ms

    case 209
      if (CapSearch > 0) then goto PARSE_DAB_TABLE

      'property=0xB300 (DAB_EVENT_INTERRUPT_SOURCE)
      SetProperty(&hB3, &h00, 0, &H01) 'SRV_LIST_INT_Enable

    case 210
      'property=0x8100 (DIGITAL_SERVICE_INT_SOURCE)
      SetProperty(&h81, &h00, 0, 1) 'DSRVPCKTINT

    case 211
      'property=0x8101 (DIGITAL_SERVICE_RESTART_DELAY)
      SetProperty(&h81, &h01, 8, 0) '2048ms (default was 8s)

    case 212
      'property=0xB301 (DAB_EVENT_MIN_SVRLIST_PERIOD)
      SetProperty(&hB3, &h01, 0, 15) '1.5s minimum between service list updates

    case 213
      'property=0xB302 (DAB_EVENT_MIN_SVRLIST_PERIOD_RECONFIG)
      SetProperty(&hB3, &h02, 0, 5) '500ms minimum between service list updates

    case 214
      'property=0xB501 (DAB_ACF_MUTE_SIGLOSS_THRESHOLD)
      'Default is 6dBuV - way too low. 30 seems more like it. I get a working DAB
      'signal at 31dBuV, but its pretty marginal at that level.
      SetProperty(&hB5, &h01, 0, 30)

    case 215
      'property=0xB507 (DAB_ACF_SOFTMUTE_BER_LIMITS)
      SetProperty(&hB5, &h07, &he2, &hc4) 'Max at -30 (E2), min at -60 (C4)

    case 216
      'property=0xB508 (DAB_ACF_CMFTNOISE_LEVEL)
      i = &h100
      SetProperty(&hB5, &h08, i >> 8, i and 255) 'lower level than default level

    case 217
      'Read the DAB frequency table
      'CMD=0xB9 (DAB_GET_FREQ_LIST)
      'ARG1=00 (Fixed - not configurable)
      pin(P_SI4689_CS) = 0
      spi write 2,&HB9,&H00
      pin(P_SI4689_CS) = 1
      if (D > 2) then print "*B9,00 DAB get freq list"
      Reply_Size=5
      CTSwait = 9
      SpiWait = 1

    case 218
    PARSE_DAB_TABLE:
      'Parse the DAB frequency table we just loaded to remember the frequencies
      '
      'Firstly, receive the status message initiated above. The response bytes are in
      'SpiBuf() and the reply is a variable length. We asked for 5 bytes above.
      'The number of entries in the table will be in the 5th byte [ie (4)]
      NumDabFreqs = SpiBuf(4)
      if (NumDabFreqs > MAX_DAB_FREQS) then
        if (D > 0) then print "Limiting NumDabFreqs from:" NumDabFreqs
        NumDabFreqs=MAX_DAB_FREQS
      endif

      'Now we know # entries, re-read the entire message including all the frequencies
      Read_Radio_Reply(8 + (NumDabFreqs << 2))

      'Pull out the frequencies and put into DabFreqs() array
      for i = 0 to (NumDabFreqs - 1)
        a = 8 + (i << 2)
        DabFreqs(i) = (SpiBuf(a+2) << 16) + (SpiBuf(a+1) << 8) + SpiBuf(a)
        if (D > 1) then print "DAB Frequency[" i "]=" DabFreqs(i)
      next

      if (D > 1) then print "Num DAB frequencies:" NumDabFreqs
      next_state = 300
    '- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
    'select FREQUENCY
    'Refer to SI4689 application note 649: eg s5.1 p382

    case 300
      RETRY_TUNE: next_state = 301

      'Set property to select the requested output (digital/analogue)
      'Note: the radio chip supports either analogue output or digital output
      'but not both at the same time.
      '
      'Property=0x0800, 0x8001=DAC (analogue), 0x8002=I2S (digital). Can't do both
      i = 2 '2 = digital
      if ((CtrlVal(Sdigital) = 0) or (HP_out = 0) or (WM8804 = 0)) then i = 1 'analogue

      if (i <> AudioOut) then
        AudioOut = i
        SetProperty(&h08, &h00, &h80, AudioOut) 'set the chosen output
      else
        goto DOTUNE
      endif

    case 301
      'Tune to a frequency
      DOTUNE:
      SetMux (MUX_MUTE)
      pin(P_SI4689_CS) = 0

      if (RadioMode = MODE_AM) then
        'CMD=0x40 (AM_TUNE_FREQ)
        'ARG1=00 (Tune Mode=normal ASAP, Injection=Auto)
        'ARG2-3 (Frequency in kHz, LSB then MSB)
        'ARG4-5 (Antenna tuning capacitor if non-zero, 0=automatic)
        spi write 6,&H40,&H00, Frequency and 255, Frequency >> 8, AntCap and 255, AntCap >> 8
        if (D > 2) then print "*40,00," Frequency "," AntCap " AM tune"

      elseif (RadioMode = MODE_FM) then
        'CMD=0x30 (FM_TUNE_FREQ)
        'ARG1=00 (DirTune=MPS, Tune Mode=normal ASAP, Injection=Auto)
        'ARG2-3 (Frequency in kHz, LSB then MSB)
        'ARG4-5 (Antenna tuning capacitor if non-zero, 0=automatic)
        i = Frequency / 10
        spi write 6,&H30,&H00, i and 255, i >> 8, AntCap and 255, AntCap >> 8
        if (D > 2) then print "*30,00," Frequency "," AntCap " FM tune"

      else 'It must be DAB
        'Need to discover which index in DabFreqs corresponds with the selected channel
        i = DabFreqToIndex(Frequency)

        'Select the DAB index we just discovered
        'CMD=0xB0 (DAB_TUNE_FREQ)
        'ARG1=00 (Injection = auto)
        'ARG2=Frequency Index
        'ARG3=00 (Fixed - not configurable)
        'ARG4-5= (Antenna tuning capacitor if non-zero, 0=automatic)
        spi write 6,&HB0,&H00,i,0, AntCap and 255, AntCap >> 8
        if (D > 2) then print "*B0,00," Frequency ",0," AntCap " DAB tune"
        NumDabServices = 0
        LastDabServList = -1
      endif

      pin(P_SI4689_CS) = 1
      CTSwait = 500
      StatusBits=1 '1=Seek/Tune Complete bit
      if (D > 0) then print "Tuning to " str$(Frequency) " kHz (" str$(NewServiceID) "," str$(NewCompID) ")"
      ExpectTune = 1
      if (RadioMode <> MODE_DAB) then next_state = 400

      'skip over next state if we are doing an AntCap search
      'don't need any DLS packets while searching
      if (CapSearch > 0) then next_state = 400

    case 302
      'property=0xB400 (DAB_XPAD_ENABLE)
      'ARG=0005 (Enable DLS packets, MOT slideshow)
      'Note: needs time for data ensemble to become available, check in RSQ loop
      SetProperty(&HB4, &H00, &H00, &H05)
      next_state = 400

    '- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
    'READ RADIO SIGNAL QUALITY STATUS
    'This state may be entered from an earlier state (cold boot or change frequency).
    'It may also be entered from state 0 - just state poll during normal radio playing
    'Refer to SI4689 application note 649: eg s5.1 p381

    case 400
      PollWait=0 'counter pacing out auto-calling this state (to fetch signal status)

      ''ExpectTune' is set from either a tune command (state 300) or AM/FM seek command
      'We must first check if the tune worked. Retuning can happen by manually selecting
      'new channel, or during a scan/seek operation
      if (ExpectTune <> 0) then
        ExpectTune = 0

        'Check bit 1 of last status read - Set to 1 when seek/tune is complete
        'If zero, the seek is still taking place
        if ((SpiBuf(0) and 1) = 1) then
          'Bit0: 0=seek down, 1=seek up
          'Bit1: 0=single seek, 1=full scan
          'Bit6: 1=Seek Completed
          'Bit7: 1=Seek Requested
          if ((Seeking and 128) <> 0) then Seeking=(Seeking and 3) or 64

        else
          'We have arrived via either states 1008 or 1107. Need to RESTART the seek
          if ((Seeking and 128) <> 0) then
            if (RadioMode = MODE_AM) then goto AM_SEEK
            if (RadioMode = MODE_FM) then goto FM_SEEK
	    'fall through if seeking DAB - not done in HW for DAB
          endif

          'DAB seeking is by jumping to next DAB frequency by a SW config
          'At this point, seek/tune is not complete and we are not in a seek state
          if (D > 0) then print "Failed seek/tune. Retrying"
          goto RETRY_TUNE
        endif

        'At this point, we have a seek/tune complete flag which needs to be cleared
        '(below) and we can prepare to release the mute. The mute release below is a
        'further protection against thumps / cracks in the analogue output.
        if ((MuteDowncount <= 0) and (RadioMode <> MODE_DAB)) then MuteDowncount = MUTE_RELEASE
      endif

      'Request the radio status (and clear any seek/tune interrupts)
      'Different status read messages required during normal times,
      'and during the antenna capacitance search
      if (CapSearch = 0) then
        pin(P_SI4689_CS) = 0

        if (RadioMode = MODE_AM) then
          'CMD=0x42 (AM_RSQ_STATUS)
          'ARG1=03 (Cancel the seek + send Seek/Tune Ack)
          spi write 2,&H42,&H03
          if (D > 2) then print "*42,03 AM Status"
          Reply_Size = 17
          next_state = 410

        elseif (RadioMode = MODE_FM) then
          'CMD=0x32 (FM_RSQ_STATUS)
          'ARG1=03 (Cancel the seek + send Seek/Tune Ack)
          spi write 2,&H32,&H03
          if (D > 2) then print "*32,03 FM Status"
          Reply_Size = 22
          next_state = 430

        else 'It must be DAB
          'CMD=0xB2 (DAB_DIGRAD_STATUS)
          'ARG1=01 (Send Seek/Tune Ack)
          spi write 2,&HB2,&H01
          if (D > 2) then print "*B2,01 DAB status"
          Reply_Size = 23
          next_state = 450
        endif

        pin(P_SI4689_CS) = 1
        CTSwait = 30
        SpiWait = 1
      else
        'We are in the state machine searching for optimal AntCap
        next_state = IntState
      endif

    case 410 'AM status from CMD=0x42
      'Receive the status messages initiated above. The response bytes are in SpiBuf()
      '[0] bit 7 = CLEAR TO Send next command
      '[0] bit 6 = Command error (ie a bug in this software)
      '[0] bit 5 = Digital radio link change interrupt (digital radio ensemble)
      '[0] bit 4 = An enabled data component of a digital services needs attention
      '[0] bit 3 = Received signal quality outside defined limits
      '[0] bit 1 = Auto controlled features has crossed a limit
      '[0] bit 0 = Seek / Tune complete
      '[1] bit 5 = New event related to digital radio
      '[1] bit 0 = New interrupt related to HD radio
      '[4] bit 3 = FIC decoder encountered unrecoverable errors
      '[4] bit 2 = Change in ensemble acquisition state
      '[4] bit 1 = RSSI below DAB low threshold
      '[4] bit 0 = RSSI above DAB high threshold
      ReadFreq = (SpiBuf(7) << 8) + SpiBuf(6)
      FreqOffset = 2 * Conv8bitSignedInt(SpiBuf(8))
      SigStrength = Conv8bitSignedInt(SpiBuf(9))
      SNR = Conv8bitSignedInt(SpiBuf(10))
      M = Conv8bitSignedInt(SpiBuf(11)) 'AM=Modulation Index
      ReadAntCap = (SpiBuf(13) << 8) + SpiBuf(12)

      NewStatus = 1
      Stereo = 0
      Quality = -1
      FibErr = -1
      CNR = -1

      if ((SigStrength > AmMinStren) and (SNR > AmSNR)) then
        Rstate = Rplaying
      else
        Rstate = Rtuned
      endif

      goto SET_VOLUME

    case 430 'FM status from CMD=0x32
      ReadFreq = (SpiBuf(7) * 2560) + (SpiBuf(6)*10)
      FreqOffset = 2 * Conv8bitSignedInt(SpiBuf(8))
      SigStrength = Conv8bitSignedInt(SpiBuf(9))
      SNR = Conv8bitSignedInt(SpiBuf(10))
      M = Conv8bitSignedInt(SpiBuf(11)) 'FM=Multipath indication
      ReadAntCap = (SpiBuf(13) << 8) + SpiBuf(12)

      if ((SigStrength > FmMinStren) and (SNR > FmSNR)) then
        Rstate = Rplaying
      else
        Rstate = Rtuned
      endif

      'Send next status request message to understand stereo status
      'CMD=0x33 (FM_ACF_STATUS) automatically controlled features
      'ARG1=01 (Clear any ACF Ack)
      pin(P_SI4689_CS) = 0
      spi write 2,&H33,&H01
      pin(P_SI4689_CS) = 1
      if (D > 2) then print "*33,01 FM ACF status"
      Reply_Size = 11
      CTSwait = 9
      SpiWait = 1

    case 431
      'Receive the additional FM status message initiated above
      '[8] bit 7 set when FM stereo pilot received,
      Stereo = 0
      if ((SpiBuf(8) and 128) <> 0) then Stereo=1
      NewStatus = 1
      Quality = -1
      FibErr = -1
      CNR = -1
      goto SET_VOLUME

    case 450 'DAB status from CMD=0xB2
      SigStrength = Conv8bitSignedInt(SpiBuf(6))
      SNR = Conv8bitSignedInt(SpiBuf(7))
      Quality = Conv8bitSignedInt(SpiBuf(8))
      CNR = Conv8bitSignedInt(SpiBuf(9))
      FibErr = (SpiBuf(11) << 8) + SpiBuf(10)
      ReadFreq = (SpiBuf(14) << 16) + (SpiBuf(13) << 8) + SpiBuf(12)
      FreqOffset = Conv8bitSignedInt(SpiBuf(17))
      M = Conv8bitSignedInt(SpiBuf(22)) 'DAB=Fast Detect
      ReadAntCap = (SpiBuf(19) << 8) + SpiBuf(18)
      NewStatus = 1

      'Check for DAB valid bit and acceptable quality
      if ((SpiBuf(5) and 1) = 0) then
        Status="No signal"
        Stereo = 0
        if ((Seeking and 128) <> 0) then Seeking=(Seeking and 3) or 64
        Rstate = Runtuned
        goto Exit_State_Machine 'next_state = IntState
      endif

      Stereo = 1
      if (Rstate = Runtuned) then Rstate = Rtuned

      if (Quality < 25) then
        if (D > 1) then print "Low DAB signal quality:" Quality
        Status="Poor Signal"

      elseif ((SpiBuf(5) and 13) = 13) then
        Status="Errors"

      elseif ((SpiBuf(5) and 13) = 5) then
        Status="Signal OK"

      endif

      if ((ServiceID < 0) or (Rstate <> Rplaying)) then 'don't read time if no service!
        next_state = IntState 'Return to idle

      else
        'Note: the value of SECONDS which is returned is always incorrect (30) ignore it
        'CMD=0xBC (DAB_GET_TIME)
        'ARG1=00 (0 = local time)
        pin(P_SI4689_CS) = 0
        spi write 2,&HBC,0
        pin(P_SI4689_CS) = 1
        if (D > 2) then print "*BC,00 DAB get time"
        Reply_Size = 11
        CTSwait = 9
        SpiWait = 1
      endif

    case 451
      a = (SpiBuf(5) << 8) + SpiBuf(4)
      b = SpiBuf(6)
      c = SpiBuf(7)
      e = SpiBuf(8)
      n = SpiBuf(9)
      'id = SpiBuf(10) 'ignore seconds - it is always wrong

      if ((c > 0) and (c < 32) and (b > 0) and (b < 13) and (a > 2019) and (a < 2100)) then
        String2 = str$(c,2,0,"0") + "/" + str$(b,2,0,"0") + "/" + str$(a)
      else
        String2 = ""
      endif

      'be careful of bit errors causing maths issues:
      if ((e >= 0) and (e < 24) and (n >= 0) and (n < 60)) then
        SL = str$(e,2,0,"0") + ":" + str$(n,2,0,"0")
	String1 = SL
        if (String2 <> "") then String1 = String2 + " " + SL
      else
        SL = ""
        String1 = String2
      endif

      if (D > 2) then print String1

      if (CtrlVal(RLN9) <> String1) then
        CtrlVal(RLN9) = String1 'display the time

        'Set the internal clock every now and then
        if (String2 <> "") then DATE$ = String2
        if (SL <> "") then TIME$ = SL
      endif

      goto SET_VOLUME

    case 452 'requires a unique state because it is referred to by IR remote routine
    SET_VOLUME:
      SetMute (0) 'unmute
      if (D > 1) then print "Audio mute off"
      next_state = 453

    case 453
      i = CtrlVal(BVolume)

      'Check presence of headphones
      if (pin(P_HEADPHONE) = 0) then
        'if headphones plugged in, set volume to half nominal level - to avoid deafness
        i = i>>1

        if (HP_out = 1) then
          HP_out = 0
          if (D > 0) then print "Headphones plugged in"

          'Turn off amp when headphones plugged in
          'NOTE: the amp has a volume setting. But we will ignore this and use the volume
          'setting of the radio. All we need to do is turn the amp on and off. It will
          'power up at its default volume which is 2/3 max which we will just use as-is
          pin(P_PAM8407_ENAB) = 0 'set to 0 to turn off the amp when headphones in
          pin(P_PAM8407_UP) = 1
          pin(P_PAM8407_DN) = 1
	  goto RETUNING
        endif

      elseif (HP_out = 0) then
        'headphones are removed
        HP_out = 1
        if (D > 0) then print "Headphones removed"
        pin(P_PAM8407_ENAB) = 1 'set to 1 to turn on amp when headphones removed

        'if the digital output is active, cause a 'frequency change' to same frequency
        'so digital interface enabled
	RETUNING:
        if ((DigitalMode <> 0) and (NewFrequency < AmMIN)) then
          if (D > 0) then print "Retuning"
          NewFrequency = Frequency
          NewServiceID = ServiceID
          NewCompID = CompID
        endif

      endif

      SetVolume (i)

      if (D > 1) then print "Analogue volume set to " i
      next_state = IntState 'Return to idle

    '- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
    'SEEK the next station up or down
    'Seeking:
    'Bit0: 0=seek down, 1=seek up
    'Bit1: 0=single seek, 1=full scan
    'Bit6: 1=Seek Completed
    'Bit7: 1=Seek Requested
    '
    'The Si4689 seek function appears to be 'sometimes' unreliable in that it will
    'seek but return with the same frequency. So when starting a seek, firstly nudge
    'the frequency one step, and then issue the seek

    case 1000
      if (D > 0) then print "Initiating AM seek"

      'Can't seek below bottom or top of band
      if (((Seeking and 1) = 0) and (Frequency <= AmMIN)) then goto NO_SEEK
      if (((Seeking and 1) = 1) and (Frequency >= AmMAX)) then goto NO_SEEK

      AM_SEEK:
      SetMux (MUX_MUTE)

      'CMD=0x41 (AM_SEEK_START)
      'ARG1=00 (start normal ASAP)
      'ARG2=08 (0=seek down, 1=seek up)
      'ARG3=00 (Fixed - not configurable)
      'ARG4,5=0000 (auto detect antenna capacitance)
      pin(P_SI4689_CS) = 0
      spi write 6,&H41,&H00,(Seeking and 1) << 1,0,0,0
      pin(P_SI4689_CS) = 1
      if (D > 2) then print "*41,00," hex$((Seeking and 1) << 1,2) ",0,0,0 AM seek start"

      CTSwait = 2000 'Allow potentially long time for seek to complete
      StatusBits = 1 '1=Seek/Tune Complete bit
      ExpectTune = 1
      SeekStart = Frequency
      next_state = 400

    '- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
    case 1100
      if (D > 0) then print "Initiating FM seek"

      'Can't seek below bottom or top of band
      if (((Seeking and 1) = 0) and (Frequency <= FmMIN)) then goto NO_SEEK

      if (((Seeking and 1) = 1) and (Frequency >= FmMAX)) then
        NO_SEEK:
	Seeking = 64
        goto Exit_State_Machine 'next_state = IntState
      endif

      FM_SEEK:
      SetMux (MUX_MUTE)

      'CMD=0x31 (FM_SEEK_START)
      'ARG1=00 (start normal ASAP)
      'ARG2=0x (0=seek down, 1=seek up)
      'ARG3=00 (Fixed - not configurable)
      'ARG4,5 (Antenna tuning capacitor if non-zero, 0=automatic)
      pin(P_SI4689_CS) = 0
      spi write 6,&H31,&H00,(Seeking and 1) << 1,0,0,0
      pin(P_SI4689_CS) = 1
      if (D > 2) then print "*31,00," hex$((Seeking and 1) << 1,2) ",0 FM seek start"

      CTSwait = 2000 'Allow potentially long time for seek to complete
      StatusBits = 1 '1=Seek/Tune Complete bit
      ExpectTune = 1
      SeekStart = Frequency
      next_state = 400

    '- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
    'Used for the varactor tuning algorithm described in AN851 page 17
    'Used prior to tuning the radio to known frequency of local FM/DAB station
    'VhfSw=0 means switch open (FM), VhfSw=1 means switch closed (DAB)

    case 10000
      if (RadioMode = MODE_FM) then
        i = FmSwitch
      else
        i = DabSwitch
      endif

      'property=0x1712 (DAB/FM_TUNE_FE_CFG)
      'ARG (Open or Close VHF Switch)
      SetProperty(&H17, &H12, &H00, i) '00 means switch open, 01 means switch closed

    case 10001
      'apply HW mute so we don't hear anything as we deliberately screw up received
      'signal strength
      SetMux (MUX_MUTE)
      SetMute (3) 'mute
      next_state = IntState

    '- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
    'This is used in the AntCap search

    case 11000
      'CMD=0xE5 (TEST_GET_RSSI)
      'ARG1=00 (Fixed - not configurable)
      pin(P_SI4689_CS) = 0
      spi write 2,&HE5,&H00
      pin(P_SI4689_CS) = 1
      if (D > 2) then print "*E5,00 Test Get RSSI"
      Reply_Size = 6
      CTSwait = 9
      SpiWait = 1

    case 11001
      'Read the signal level requested above - it is in a fixed-point binary format
      SigLevel = Conv8bitSignedInt(SpiBuf(5)) + (SpiBuf(4) / 256.0)
      if (D > 1) then print "Signal Level:" SigLevel "dBuV at " Frequency "kHz"
      next_state = IntState

    '- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
    'Read one or more consecutive property values back from the radio
    'The values will appear as 16 bit words starting at SpiBuf(4)+SpiBuf(5)

    case 20000
      'CMD=0x14 (GET_PROPERTY)
      'ARG1=count (number of properties to read)
      'ARG2,3=property id (which property to retrieve)
      pin(P_SI4689_CS) = 0
      spi write 4,&H14, NumProp, PropID and 255, PropID >> 8
      pin(P_SI4689_CS) = 1
      if (D > 2) then print "*14," hex$(NumProp,2) "," hex$(PropID,4) " Get Property"
      Reply_Size = 4 + (NumProp << 1)
      CTSwait = 9
      SpiWait = 1

    case 20001
      'Read out the results from above
      if (D > 1) then
        for i = 0 to (NumProp-1)
          print "Property " hex$(PropID+i,4) "==" hex$(SpiBuf(5+(i<<1)),2) hex$(SpiBuf(4+(i<<1)),2)
        next
      endif

      Exit_State_Machine: next_state = IntState

    '- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
    'Interrupt handler for DAB
    case 60000
      IntState=0

      if ((IntMask and 1) <> 0) then 'Digital Service Interrupt
        'Get digital service data
        'CMD=0x84 (DAB_GET_DIGITAL_SERVICE_DATA)
        'ARG1=0x01 (Fetch the status only (non destructive read), and clear the interrupt)
        pin(P_SI4689_CS) = 0
        spi write 2,&H84,1
        pin(P_SI4689_CS) = 1
        if (D > 2) then print "*84,01 DAB get digital service data"

        'Read only 24 bytes - tells the size of the full string, which must be read again
        Reply_Size = 24
        CTSwait = 9
        SpiWait = 1

        IntMask = IntMask and &HFE 'turn off the int bit so we don't process it again
      else
        goto RdsInt
      endif

    case 60001 'Digital Service Interrupt
      'Receive the status message initiated above. The response bytes are in SpiBuf()
      'The response we just received may have additional data associated with it
      i = (SpiBuf(19) << 8) + SpiBuf(18)
      a = (SpiBuf(21) << 8) + SpiBuf(20)
      b = (SpiBuf(23) << 8) + SpiBuf(22)
      c = (SpiBuf(11) << 24) + (SpiBuf(10) << 16) + (SpiBuf(9) << 8) + SpiBuf(8)
      e = (SpiBuf(15) << 24) + (SpiBuf(14) << 16) + (SpiBuf(13) << 8) + SpiBuf(12)

      if (D > 1) then
        print "Digital Service Interrupt"
        print " Service ID:" c "/" hex$(c,8)
        print " Component ID:" e "/" hex$(e,8)
        print " Buf Count:" SpiBuf(5)

        '0=playing normally, 1=stopped, 2=Overflow, 3=new object, 4=errors
        print " Service State:" hex$(SpiBuf(6),2)

        '0x0x=std data, 0x4x=non DLS pad, 0x8x=PDS pad, 0xCx=reserved
        print " Byte 7:" hex$(SpiBuf(7),2)
        print " UA type:" hex$(SpiBuf(17),2) hex$(SpiBuf(16),2)
        print " Byte Count:" i
        print " Seg Num:" a
        print " Num Segs:" b
      endif

      'Now we know size, re-read the entire message
      String1 = ""
      Read_Radio_Reply(24+i)
      Rstate = Rplaying
      a = 0

      'Refer to table 20/21 in AN649 p433
      if ((SpiBuf(7) and 192) = 128) then
        if (D > 1) then print " Selected via byte 7"
        a = 1

      elseif (((SpiBuf(24) = 0) or (SpiBuf(24) = 128)) and (SpiBuf(25) = 0)) then
        if (D > 1) then print " Selected via initial bytes"
        a = 1

      endif

      if (a <> 0) then
        if (i > 256) then i=256 'Maximum string size is 256
        String1 = ""

        for c = 24 to 23+i
          String1 = String1 + clean_chr(SpiBuf(c))
        next

        String1 = truncate_string (String1)
        if (D > 1) then print "Dab data: " String1
        display_string (String1)
      endif
      goto RdsInt

    case 60100 'RDS Interrupt
    RdsInt:
      if ((IntMask and 2) <> 0) then 'RDS Interrupt
      DECODE_MORE:
        'CMD=0x34 (FM_RDS_STATUS)
        'ARG1=0x01 = Clear RDS interrupt
	'     0x02 = clear anything stored in the fifo
	'     0x04 = Only grab status and return last valid data, don't decrement fifo
        pin(P_SI4689_CS) = 0
        spi write 2,&H34, 1
        pin(P_SI4689_CS) = 1

        if (D > 2) then print "*34,01 FM RDS Status"
        Reply_Size = 20
        CTSwait = 9
        SpiWait = 1
        IntMask = IntMask and &HFD 'turn off the int bit so we don't process it again
        next_state = 60101

      else
        goto DigEvInt
      endif

    case 60101
      'There are four separate pairs of bytes the chip data calls groups A-D. A and B
      'contain status info. C and D contain text. But there are several different formats
      'transmitted in an interleaved way. We need to identify what the current set
      'contains each message. There is also a forward error correction mechanism
      'implemented that allows some errors to be repaired (if only a few bits wrong) and
      'the probable number of errors for each of the four groups is indicated in byte 11
      'of the message response. If only a few errors are allegedly corrected, we will
      'accept and display that information. Ignore if more. Each message can contain 2 or
      '4 bytes of ascii, so multiple consecutive messages need to be integrated to make
      'a complete message. This gives rise to a state machine necessary to reassemble
      'the full messages from two or four byte chunks. Explanation provided in AN243
      '(which doesn't apply to the Si4689, but is still helpful)
      '
      'Refer also to U.S. RBDS Standard 1998 Specification of the radio broadcast data
      '(RDS) System
      if (D > 0) then
        print "--------------------------------"
        print "FM RDS interrupt"
        print "0, 1, 3    -> " hex$(SpiBuf(0),2) " " hex$(SpiBuf(1),2) " " hex$(SpiBuf(3),2)
        print "4, 5, 6, 10-> " hex$(SpiBuf(4),2) " " hex$(SpiBuf(5),2) " " hex$(SpiBuf(6),2) " " hex$(SpiBuf(10),2)
      endif

      if (SpiBuf(10) > 0) then 'This is the count of the number of waiting messages
        'including this one. i.e. 0 means empty. If non-zero, Blocks A-D contain the
        'oldest entry in the FIFO and [10] will decrement by one on the next poll,
        'assuming no new data has been recevied in the interim. The error byte is
        'formated as four groups of 2 bits. The two bit binary pattern 00 means no
        'errors in that block. The two bit binary pattern 01 means one or two errors
        'that were corrected. The other two binary patterns mean more corrected or
        'uncorrected errors.

        'GT = Group Type Code (stored in id)
        'address/index (stored in e)
        id = SpiBuf(15) >> 4
        e = SpiBuf(14) and 31 '0..31

        if (D > 0) then
          if ((SpiBuf(5) and 1) <> 0) then print " RDS FIFO overrun"
          if ((SpiBuf(5) and 2) <> 0) then print " RDS in sync"
          print " Rx Error flags=" hex$(SpiBuf(11), 2)

          if ((SpiBuf(5) and 16) <> 0) then 'TP and PTY valid
            print " TP=" (SpiBuf(6) >> 5) and 1
            print " PTY=" hex$(SpiBuf(6) and 31, 2)
          endif

          if ((SpiBuf(5) and 8) <> 0) then 'PI valid
            print " PI=" hex$((SpiBuf(9) << 8) + SpiBuf(8), 4)
          endif

          print " Group=" hex$(id, 2)
          print " Addr=" hex$(e, 2)
          print " Block A: " hex$(SpiBuf(13),2) hex$(SpiBuf(12),2)
          print " Block B: " hex$(SpiBuf(15),2) hex$(SpiBuf(14),2)
          print " Block C: " hex$(SpiBuf(17),2) hex$(SpiBuf(16),2) " " chr$(39) clean_chr(SpiBuf(17)) clean_chr(SpiBuf(16)) chr$(39)
          print " Block D: " hex$(SpiBuf(19),2) hex$(SpiBuf(18),2) " " chr$(39) clean_chr(SpiBuf(19)) clean_chr(SpiBuf(18)) chr$(39)
        endif

        'group (id) 0 contains 'basic information' - should remain constant for the
        'radio station. Only accept station identifier string data if there were
        'apparently zero errors in that data. The data is in groups B and D
        if ((id = 0) and ((SpiBuf(11) and &H33) = 0)) then
          String1 = clean_chr(SpiBuf(19)) + clean_chr(SpiBuf(18)) 'String comes in 4x2 chunks

          'Same as what we had previously? This checks for changes in string
          'For example, ABC Classic FM sends three strings in succession - 'ABC', 'Classic', 'FM'
          if ((ProgName(e and 3) <> "") and (ProgName(e and 3) <> String1)) then
            if (D > 0) then print " RDS group 0 snippet " e and 3 " change"

            for i = 0 to 3
              ProgName(i) = ""
            next
          endif

          ProgName(e and 3) = String1
          if (D > 0) then print " RDS group 0 snippet " e and 3 "=" chr$(39) String1 chr$(39)

          'Reassemble station name and display on line 1
          String1 = ProgName(0) + ProgName(1) + ProgName(2) + ProgName(3)

          'only proceed if we have received error-free snippets for ALL four segments
          if (len(String1) = 8) then
            String1 = truncate_string (String1) 'remove leading and trailing spaces
            if (D > 0) then print " RDS station name:" String1

            'does this string already exist in the RdsNameSet?
            for i = 0 to (MAX_SET-1)
              if (RdsNameSet(i) = String1) then exit for
            next

            if (i >= MAX_SET) then 'We don't already have this string
              if (RdsNames < MAX_SET) then
                RdsNameSet(RdsNames) = String1
                RdsNames = RdsNames + 1
              endif
            endif

           'Determine the concatenation of the strings. Concatenate in order UNLESS
           'one of them ends in the letters "FM" or "fm"
            for i = 0 to (MAX_SET-1)
              if ((len(RdsNameSet(i)) >= 2) and (lcase$(right$(RdsNameSet(i),2)) = "fm")) then exit for
            next

            String1 = ""
            j = 0

            'If none of the strings end in the letters 'FM', present strings in order
            if (i < MAX_SET) then j = i+1

            for i = 0 to (MAX_SET-1)
              if (RdsNameSet((i+j) mod MAX_SET) <> "") then
                if (String1 <> "") then String1 = String1 + " "
                String1 = String1 + RdsNameSet((i+j) mod MAX_SET)
              endif
            next

            'The maximum length of the combined string cannot exceed NameLen characters
            String1 = left$(String1, 16)

            'Do we already have a station name in the preset file?
            String2 = MatchName (Frequency, String1, -1, -1, 0)

            if (CtrlVal(RLN1) <> String2) then CtrlVal(RLN1) = String2

            'If the RDS string is different than the configured station name, display
            if ((String1 <> String2) and (CtrlVal(RLN8) <> String1)) then
              CtrlVal(RLN8) = String1
            endif
          endif

        elseif ((id = 2) and ((SpiBuf(11) and &H15) = 0)) then
          'group 2 contains 'radiotext' as two groups of 64 chars - only one is
          'active at a time. Tolerate a few corrected errors for radiotext.
          i = e and 16 'will be 0 if new section is a, or 16 if new section is b

          'Has the active group changed? start assembling the next string
	  ' i will be either 0 or 16
          if (i <> ActiveProgData) then

            TryToDispRDS (i+1) 'non-zero param forces string blanking
            ActiveProgData = i
          endif

          String1 = clean_chr(SpiBuf(17)) + clean_chr(SpiBuf(16)) + clean_chr(SpiBuf(19)) + clean_chr(SpiBuf(18))
          ProgData(e) = String1
          if (D > 0) then print " RDS group 2 snippet " e "=" chr$(39) String1 chr$(39)

        elseif ((id = 4) and ((e and 16) = 0) and ((SpiBuf(11) and &H3f) = 0)) then
          'group 4A contains TIME. Don't accept any errors in groups B, C, or D!
          'The algorithm is a mess and a kludge. It is described in the RBDS reference
          'section 3.1.5.6 and Annex G.

          'Extract the time
          'j=hours, n=minutes, c=timezone offset
          j = ((SpiBuf(16) and 1) << 4) + ((SpiBuf(19) and &hf0) >> 4)
          n = ((SpiBuf(19) and 15) << 2) + ((SpiBuf(18) and &Hc0) >> 6)
	  c = SpiBuf(18) and 31

          'The sign of the offset is in bit5 of SpiBuf(18)
          if ((SpiBuf(18) and 32) <> 0) then c = -c

          'Add the time zone offset to the specified time
	  'The offset is specified in chunks of half an hour
	  'I found that the ABC sets timezone to 0, but other stations set real TZ
	  id = (j * 60) + n + (c * 30)

          'We need to do some carrying and borrowing when the adjusted local time
          'crosses midnight
          if (id < 0) then
            id = id + 1440 'add one day (in minutes)
            c = -1

          elseif (id >= 1440) then
            id = id - 1440 'remove one day
            c = 1

          else
            c = 0

          endif

          'recalculate the hours and minutes after we adjusted to the local TZ
          j = id / 60
          n = id mod 60

          'be careful of bit errors causing maths issues:
	  if ((j >= 0) and (j < 24) and (n >= 0) and (n < 60)) then
            SL = str$(j,2,0,"0") + ":" + str$(n,2,0,"0")
          else
	    SL = ""
          endif

          'j=Modified Julian Day, b=Year, n=Month, e=Day
          'c is the carry resulting from timezone offset adjustment.
          j = ((SpiBuf(14) and 3) << 15) + (SpiBuf(17) << 7) + (SpiBuf(16) >> 1) + c

          b = int((j - 15078.2) / 365.25) 'What an awesome piece of computer science
          n = int((j - 14956.1 - int(b * 365.25)) / 30.6001) 'like the kludge?
          e = j - 14956 - int(b * 365.25) - int(n * 30.6001) 'I didn't write it.

          'the algorithm defines something called k, which I store in id
          id = 0 'I mean, 'id' is about as meaningful as 'k' in this awesome kludge
	  if ((n = 14) or (n = 15)) then id = 1

	  b = b + id + 1900
	  n = n - 1 - (id * 12)

          String1 = ""

          'Convert the date/month/year to a string, avoiding data errors (if any)
          if ((e > 0) and (e < 32) and (n > 0) and (n < 13) and (b > 2019) and (b < 2100)) then
            String2 = str$(e,2,0,"0") + "/" + str$(n,2,0,"0") + "/" + str$(b)
	    String1 = String2

            if (SL <> "") then String1 = String2 + " " + SL
          endif

          if (D > 0) then print " RDS group 4A. Julian Day=" j ", Time=" String1

          if (CtrlVal(RLN9) <> String1) then
            CtrlVal(RLN9) = String1 'display the time

            'Set the internal clock every now and then
            if (String2 <> "") then DATE$ = String2
            if (SL <> "") then TIME$ = SL
          endif
        endif

        'Is there more RDS data queued up to decode? Loop back if more.
        if (SpiBuf(10) > 1) then goto DECODE_MORE
        TryToDispRDS (0) 'after all messages loaded, try and display what we got
      endif

      goto DigEvInt

    case 60200 'Digital Event Interrupt
    DigEvInt:
      if ((IntMask and 4) <> 0) then 'Digital Event Interrupt
        'Get the DAB event status when interrupt tells us it is ready
        'CMD=0xB3 (DAB_GET_EVENT_STATUS)
        'ARG1=01 (Clear EVENT int)
        pin(P_SI4689_CS) = 0
        spi write 2,&HB3,&H01
        pin(P_SI4689_CS) = 1
        if (D > 2) then print "*B3,01 DAB get event status"
        if (D > 1) then print "Waiting for DAB service list"
        Reply_Size = 8
        CTSwait = 9
        SpiWait = 1

        IntMask = IntMask and &HFB 'turn off the int bit so we don't process it again
        next_state = 60201
      else
        next_state = IntState 'no other kinds of interrupt to process here
      endif

    case 60201 'get the service list identifier, and only get list if ID is new
      i = (SpiBuf(7) << 8) + SpiBuf(6)

      'Check that we have a new service list
      if (((SpiBuf(5) and 1) = 0) or (LastDabServList = i)) then
        if (D > 1) then print "No new service list"
        goto Exit_State_Machine 'next_state = IntState
      endif

      if (D > 1) then print "Received DabServList=" i " (Last was " LastDabServList ")"
      LastDabServList = i

      'CMD=0x80 (GET_DIGITAL_SERVICE_LIST)
      'ARG1=00 (Get Audio service list)
      pin(P_SI4689_CS) = 0
      spi write 2,&H80,&H00
      pin(P_SI4689_CS) = 1
      if (D > 2) then print "*80,00 Get audio service list"

      'Read only a short 6 bytes - tells us the actual size of the full string
      Reply_Size = 6
      CTSwait = 9
      SpiWait = 1

    case 60202
      i = 6 + (SpiBuf(5) << 8) + SpiBuf(4)

      if (i >= READ_REPLY_BUFSIZE) then
        if (D > 1) then print "Digital service list string size:" i
        clear
        run
      endif

      'Now we know # entries, re-read the entire message
      Read_Radio_Reply(i)

      'parse and decode the dab service list
      n = SpiBuf(8) 'this is the number of potential services in this list
      if (D > 1) then print "Num Dab Services in list:" n

      'Table format:
      '[0-3]= service ID
      '[4]= Service Info 1 (ignored)
      '[5]= Service Info 2
      '[6]= Service Info 3
      '[7]= padding (ignored)
      '[8-23]= Service Label / name of this service
      '[24-25]= Component ID
      '[26]= Component info
      '[27]= Valid flags
      'Refer to AN649 table 14 + EN 300 401 standard to understand format
      'Table components can arrive in a different order every time the radio is booted!
      a = 12 'this will be an offset into a variable length table
      NumDabServices = 0

      for i = 0 to (n - 1)
        id = SpiBuf(a) + (SpiBuf(a+1)<<8) + (SpiBuf(a+2)<<16) + (SpiBuf(a+3)<<24)
        e = SpiBuf(a+4) 'Service info #1 field, bit0=1 means digital service (like EPG)
        b = SpiBuf(a+5) and 15 'Number of components
        SL = ""

        for c = 8 to 23 '16 characters
          'the MMbasic file read command has trouble with unprintable characters
          if ((SpiBuf(a+c) < 32) or (SpiBuf(a+c) > 127)) then exit for
          SL = SL + CHR$(SpiBuf(a+c))
        next

        SL = truncate_string (SL) 'remove leading and trailing spaces

        if (D > 1) then
          print i ": Service ID:" id "/0x" hex$(id) "=" chr$(34) SL chr$(34)
          print "    Conditional Access:" SpiBuf(a+5)>>4
          print "    # Components:" b
          print "    Service Info 1:0x" hex$(e,2)
          print "    Service Info 3:0x" hex$(SpiBuf(a+6),2)
        endif

        a = a + 24 'point to the component field(s) - this is a variable length

        for c = 1 to b
          comp = SpiBuf(a) + (SpiBuf(a+1)<<8)

          if (D > 1) then
            print "    Component ID[" c "]:" comp "/0x" hex$(comp)
            print "    Component info[" c "]:0x" hex$(SpiBuf(a+2),2) hex$(SpiBuf(a+3),2)
          endif

          a = a + 4 'ignore the second two bytes

          'If we want to keep this record, store it in the array
          'if its a data service like an EPG - we're not interested
          if (((e and 1) = 0) and (SL <> "") and (NumDabServices < MAX_DAB_SERVICES)) then
            String2 = SL

            if (b > 1) then
              String2 = left$(SL,NameLen-1) + str$(c)
            endif

            DabServiceName(NumDabServices) = String2
            DabServiceID(NumDabServices) = id
            DabComponentID(NumDabServices) = comp
            NumDabServices = NumDabServices + 1

            'Do we already have a station name in the preset file? MatchName will add it
            String1 = MatchName (Frequency, String2, id, comp, 0)
          endif
        next
      next

      if (D > 1) then print "Keeping " NumDabServices " service descriptions"

      'If we are ready to start a service, can we find a match? If so, start the service
      if ((NewServiceID >= 0) and (NewCompID >= 0)) then
        for i = 0 to (NumDabServices - 1)
          if ((NewServiceID = DabServiceID(i)) and (NewCompID = DabComponentID(i))) then exit for
        next

        if (i < NumDabServices) then
          DabID=i
          if (D > 0) then print "Starting digital service ID=" NewServiceID " component=" NewCompID
          ServiceID = NewServiceID
          CompID = NewCompID
          NewServiceID = -1
          NewCompID = -1

          'prepare the separated hex values
          id = ServiceID and 255
          a = (ServiceID >> 8) and 255
          b = (ServiceID >> 16) and 255
          c = (ServiceID >> 24) and 255

          comp = CompID and 255
          e = (CompID >> 8) and 255
          n = (CompID >> 16) and 255
          i = (CompID >> 24) and 255

          'CMD=0x81 (DAB_START_DIGITAL_SERVICE)
          'ARG1-3=0 (Start audio service)
          'ARG4-7= Service ID
          'ARG8-11=Component ID
          pin(P_SI4689_CS) = 0
          spi write 12,&H81,0,0,0,id,a,b,c,comp,e,n,i
          pin(P_SI4689_CS) = 1
          if (D > 2) then print "*81,0,0,0," hex$(ServiceID,8) "," hex$(CompID,8) " DAB start digital service"
          CTSwait = 9
          SpiWait = 1

          Rstate = Rselected
          MuteDowncount = MUTE_RELEASE
	  State = 452
	  exit sub
        endif
      endif

      next_state = IntState
  end select

  'Jump to the next state
  State = next_state
end sub

'-----------------------------------------------------
'Manage scanning for the preset file, and seeking up and down
'
'The 'Seeking' variable is used as follows:
'Bit0: 0=seek down, 1=seek up
'Bit1: 0=single seek, 1=full scan
'Bit6: 1=Seek Completed
'Bit7: 1=Seek Requested

sub ProcessScan
  local integer i, x

  if ((D > 0) and (DABwait = 0)) then
    print "Seek ack Freq:" ReadFreq " SNR:" SNR " SigStrength:" SigStrength " min:" Min_SigStren " CNR:" CNR " Qual:" Quality
  endif

  Frequency = ReadFreq

  'bit1=0 means this scan was a one-off - ie a one off seek up/down. We can now exit
  if ((Seeking and 2) = 0) then
    Seeking=0
    exit sub
  endif

  'To reach here, we are in the middle of a full scan
  'State=0 to reach here
  'Did we arrive at a frequency where signal strength is OK and we want to remember it?
  if ((SigStrength >= Min_SigStren) and ((RadioMode = MODE_DAB) or (SNR > Min_SNR))) then
    if (RadioMode = MODE_DAB) then
      'DAB takes longer to sync and read service names
      if (DABwait <= 0) then
        DABwait=1
        exit sub

      elseif (DABwait < 2000) then 'have we have waited long enough for names to come in?
        exit sub
      endif

      'we have now waited ten seconds - if NumDabServices is still zero, give up
      DABwait = 0

      if (NumDabServices > 0) then
        for i = 0 to (NumDabServices - 1)
          x = AddPreset (Frequency, DabServiceName(i), DabServiceID(i), DabComponentID(i), 0)
        next
      endif

    else 'AM or FM
      x = AddPreset (Frequency, "", -1, -1, 0)
    endif
  endif

  'Prepare in case none of the if statements below are taken
  Seeking = &H83 'Seek requested + full scan + seek up
  AntCap = 0 'set the default (auto) AntCap

  if (RadioMode = MODE_AM) then State = 1000 'AM seek
  if (RadioMode = MODE_FM) then State = 1100 'FM seek

  'Reached end of AM or FM bands yet?
  if (Frequency = AmMAX) then 'reached top of AM band
    NewFrequency = AFmMIN
    CapWait = 0

  elseif (Frequency = FmMAX) then 'reached top of FM band
    NewFrequency = ADabMIN 'Dummy frequency to force DAB mode
    CapWait = 0
    State = 300 'For DAB, the seeks are manual frequency selections

  elseif ((NumDabFreqs > 0) and (Frequency >= DabFreqs(0))) then 'doing DAB
    'DAB is different than AM and FM - the chip doesn't scan for us
    'we need to select freq manually

    DabFreqIndex = DabFreqToIndex (Frequency) + 1 'choose the next index
    if (DabFreqIndex >= NumDabFreqs) then goto FINSCAN
    Frequency = DabFreqs(DabFreqIndex)

    if (Frequency > ADabMAX) then
      'reached the end of the DAB band - we can compact and rewrite the preset file
      FINSCAN:
      CompactPresets
      WritePresetFile
      ReadPresetFile

      if (D > 0) then print "Scan complete. " NumServices " services in preset table"
      Seeking = 0 'stop the complete scan by turing off bit 2
      CapWait = 0
      Change_Smode = Xstatus

      'What to do now? lets just load the first radio station
      NewFrequency = Pfreq(0)
      NewServiceID = Pid(0)
      NewCompID = Pcomp(0)
      State = 0

    else 'choose the next DAB channel frequency in the list of potentials
      NewFrequency = DabFreqs(DabFreqIndex)
      DABwait = 0 'we are waiting on a DAB channel
      CapWait = 0
      State = 300 'For DAB, the seeks are manual frequency selections
    endif

  else
    MuteDowncount = MUTE_RELEASE << 1 'remute
  endif
end sub

'-----------------------------------------------------
'Manage search for optimal antenna capacitance value
'Tuning algorithm described in AN851 page 17
'The application note describes a process using an expensive piece of test gear. As I
'can't afford that, I have implemented a poor-man's substitute.... I am using radio
'stations themselves as the test sources. The purpose of the AntCap algorithm described
'in the application note, and implemented below, is to try and get the radio chip to
'configure an antenna capacitance value that maximises signal strength. The radio has an
'in-built algorithm to calculate the antcap value, but you, - the developer - need to
'tell (config) the radio chip with the Y intercept and slope of the straight-line
'approximation of the function that represents the AntCap value as a function of tuned
'frequency. So this algorithm below will try and find the AntCap value that maximises
'RxStrength for the currently tuned frequency. You - the designer - need to take as many
'readings at different frequencies as you can, then plot them on a graph (eg using
'microsoft excel) and then calculate the Y intercept and slope of the line of best fit
'to programme into the radio chip. I have found that I could achieve a boost of around
'3.5dB signal strength by optimising the AntCap settings compared with the defaults used
'in the origianl SC software. This happened to help *a lot* for me because I live in a
'fringe signal area, and the extra power made all the difference in the world to a
'usable and an unreliable DAB signal.
'
'BTW: be patient while this algorithm runs - it is somewhat slow and takes minutes each
'frequency. It is taking many readings, and then averaging both across the target
'frequency, and also dragging in the values from surrounding frequencies. The documented
'algorithm takes way fewer readings and doesn't drag in the surrounding frequencies, but
'then again, my poor-mans implementation uses radio stations in a fringe reception area,
'and I found I needed way more averaging to reduce the noise in the results. Doing this
'seemed to work very well for me and improved performance. Your mileage may vary.
'
'Capsearch is another state machine variable
'CapSearch=1 - instruction to commence search at current Frequency
'CapSearch=2 - dummy state used to wait for touch
'CapSearch=3 - Set AntCap value in hardware and initialise
'CapSearch=4 - Set inductance switch
'CapSearch=5 - Set the new capacitance value
'CapSearch=6 - Delay a little to let hardware settle
'CapSearch=7 - Average RSSI using TEST_GET_RSSI
'CapSearch=10 - dummy state used to wait for touch

'Refer AN649 p26 for AntCap to capacitance formula:
'FM and DAB: C=(AntCap-1)*250fF (an fF is a femto farad - 1000 times smaller than a pF)
'AM: C=(AntCap-1)*142fF [from AN649 p269]

const MAX_SAMPLES=20 'Take 20 samples of our real radio station and average them

dim integer highest_antcap

function What_Cap (x as integer) as float
  if (RadioMode = MODE_AM) then
    What_Cap = (x - 1) * 142.0/1000.0
  else
    What_Cap = (x - 1) * 250.0/1000.0
  endif
end function

sub ProcessCap
  static integer i, bestcap, of, oi, oc
  static float bin, maxsig, f1, f2, f3, f4

  PollWait=0

  select case CapSearch
    case 1
      Change_Smode = Xcap
      ClearConsole

      'display welcome message to appear on the console screen
      CtrlVal(LN1) = "Antenna Capacitance Search: " + str$(Frequency) + " kHz"
      CtrlVal(LN2) = "Refer to Silicon Labs App Note AN851 Appendix A"
      CtrlVal(LN3) = "for more details about the algorithm."

      CtrlVal(LN5) = "Run this function for a range of stations"
      CtrlVal(LN6) = "across the FM and DAB bands and note results."
      CtrlVal(LN7) = "The optimal AntCap values will be automatically"
      CtrlVal(LN8) = "saved to the presets. Edit the presets and"
      CtrlVal(LN9) = "interpolate any missing AntCap values."

      CtrlVal(LN11) = "Scanning will take several minutes to complete."

      CtrlVal(LN13) = "Touch screen to start."
      CapSearch = 2

    'CASE 2 'Wait for touch - not coded into in this select statement!
    'The state is set to 3 by AR1 touch in touchdown interrupt routine

    case 3 'set the frequency, and wait for new firmware to load (if required)
      ClearConsole
      WriteMsg ("Commencing AntCap search...")

      if (RadioMode = MODE_AM) then
        highest_antcap = 4096 'Not much value in trying to do AM, it works OK as is
      else
        highest_antcap = 128
      endif

      maxsig = -1
      bestcap = -1
      of = Frequency 'the tune will default to the current AntCap setting
      oi = ServiceID
      oc = CompID
      f1 = 0
      f2 = 0
      f3 = 0
      f4 = 0

      'Tune to the current frequency, and if DAB, don't select a service ID
      NewFrequency = of
      NewServiceID = -1
      NewCompID = -1
      RadioMode = -1 'to force a firmware reload
      CapSearch = 4

    case 4 'set inductance switch
      State = 10000 'Set VHF inductance and mute
      AntCap = 1 'we want to scan from AntCap=1
      CapSearch = 5

    case 5 'set cap
    do_setcap:
      bin = 0.0
      i = 0 'our sample counter
      CapWait = -200 'Wait 1s for first sample
      CapSearch = 6
      State = 300 'Issue set frequency command (to set antcap) and read the RSSI status

    case 6 'waiting to settle
    do_wait:
      if (CapWait > 30) then
        State = 11000 'Request a signal level reading
        CapSearch = 7
      endif

    case 7
      bin = bin + SigLevel
      i = i + 1

      if (i < MAX_SAMPLES) then
        CapWait = 0 'Wait and then read another sample
        CapSearch = 6
        goto do_wait
      endif

      bin = bin / MAX_SAMPLES
      String1 = "AntCap=" + str$(AntCap) + " (" + str$(What_Cap(AntCap),0,2) + "pF), Av SigStren=" + str$(bin,0,1) + "dBuV"
      WriteMsg (String1)
      f1 = (bin + f1 + f2 + f3 + f4) / 5 'take average of last five frequencies

      if (f1 > maxsig) then
        maxsig = f1
        bestcap = AntCap - 3 'We are averaging across five caps, middle is three back
      endif

      if (AntCap < highest_antcap) then
        f1 = f2
	f2 = f3
	f3 = f4
	f4 = bin
        AntCap = AntCap + 1
        goto do_setcap
      endif

      AntCap = bestcap + 1

      'Can we apply this new AntCap setting to any existing preset?
      for i = 0 to (NumPresets - 1)
        if (Pfreq(i) = of) then
          Pcap(i) = AntCap
          Pwait = 1 'Set timer to rewrite the presets table
        endif
      next

      'create a new Preset record, if we didn't find one already
      'This new record will hold the measured AntCap value for this frequency
      if ((Pwait < 1) and (NumPresets < MAX_PRESETS)) then Pwait = AddPreset (of, "", oi, oc, AntCap)

      'We now have the maximum capacitance and signal level at that capacitance
      CapSearch = 10 'a dummy value for the AR1 gui change
      ClearConsole
      WriteMsg ("Antenna Capacitance Search: " + str$(of) + " kHz")
      WriteMsg ("")
      WriteMsg ("Optimal AntCap value=" + str$(AntCap))
      WriteMsg ("Optimal Capacitance=" + str$(What_Cap(bestcap),0,2) + " pF")
      WriteMsg ("Optimal Signal Strength=" + str$(maxsig,0,1) + " dBuV")
      CtrlVal(LN13) = "Touch screen to return to setup."

      'Retune the station, selecting the optimal AntCap value just calculated
      'This will also unmute the radio
      NewFrequency = of
      NewServiceID = oi
      NewCompID = oc
      RadioMode = -1 'to force a firmware reload
  end select
end sub

'-----------------------------------------------------
'THIS IS THE MAIN PROGRAMME
ReadDefaultFile
ReadPresetFile

'Here is the main loop that runs in the foreground and just spins around all day. The
'timing for the activities within the do-loop is generated from the tick interrupt but
'the work is all done in the foreground from this loop. The GUI mechanics run from
'interrupts, but navigating the menu structure is done by a state machine run from the
'do-loop below.

DO
  if (DoPowerAlarm > 0) then
    if (Pwait > 0) then WritePresetFile 'emergency write out the presets! ASAP!

    'display alarm
    WriteErr ("Power Drop")
    DoPowerAlarm = 0 'note: the gui functions get restored when MuteDowncount expires
  endif

  '- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  if (Change_Smode > 0) then ProcessScreenChange

  '- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  'The exit timer is a function to automatically fall back from one of the special
  'function screens (like setup, or choose a preset, etc). These screens also display
  'a dim 'X' gadget at the top left. Pressing the gadget does the same thing
  'The GuiCounter starts at zero and counts up. It resets to zero each new touch.
  if ((ExitTimer > 0) and (GuiCounter > ExitTimer)) then ProcessExitTimer

  '- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  'If a radio status update has been received, display that here

  if (NewStatus <> 0) then
    'These funny RLNx string arrangements below reduce flickering on the LCD
    String1 = MatchName (ReadFreq, CtrlVal(RLN1), ServiceID, CompID, 1)
    if (CtrlVal(RLN1) <> String1) then CtrlVal(RLN1) = String1

    String1 = str$(ReadFreq) + " kHz"
    if (CtrlVal(RLN2) <> String1) then CtrlVal(RLN2) = String1

    if (maSNR < -90) then maSNR = SNR else maSNR = (MA_MULT! * maSNR) + (MA_INV_MULT! * SNR)
    if (maCNR < -90) then maCNR = CNR else maCNR = (MA_MULT! * maCNR) + (MA_INV_MULT! * CNR)
    if (maSig < 0) then maSig = SigStrength else maSig = (MA_MULT! * maSig) + (MA_INV_MULT! * SigStrength)
    if (maQual < 0) then maQual = Quality else maQual = (MA_MULT! * maQual) + (MA_INV_MULT! * Quality)

    String1 = "Signal:" + str$(maSig, -1, 1)

    if (RadioMode = MODE_DAB) then
      String1 = String1 + "dBuV    CNR:" + str$(maCNR, -1, 1) + "dB    Quality:" + str$(maQual,0,0)
    else
      String1 = String1 + "dBuV    SNR:" + str$(maSNR, -1, 1) + "dB"
    endif

    'display the stats - but at a lower rate to reduce screen flicker
    if (((Rstate <> Rplaying) or (DispRefresh > 8)) and (CtrlVal(RLN3) <> String1)) then
      CtrlVal(RLN3) = String1
      DispRefresh = 0
    else
      DispRefresh = DispRefresh + 1
    endif

    if (D > 1) then
      print
      print "Freq:" ReadFreq ", SNR:" SNR ", SigStrength:" SigStrength ", CNR:" CNR ", Qual:" Quality ", Foffset:" FreqOffset;
      print ", DAB Status:" Status ", DAB Err:" FibErr ", M:" M ", Stereo:" Stereo ", AntCap:" ReadAntCap ", Seeking:" Seeking
    endif

    NewStatus = 0

    'If a Toslink framer was detected, read status and see what it says
    'if (WM8804 <> 0) then
    '  pin(P_WM8804_CSB) = 0
    '  spi write 1, &H8C 'Register 12. Bit7 set means read command
    '  spi read 1, SpiBuf()
    '  pin(P_WM8804_CSB) = 1

      'The normal value that reads back is 0x40 (UNLOCK??)
      'bit 0 = Audio_N (1=invalid PCM samples)
      'bit 1 = PCM_N (1 = sync code detected, not PCM)
      'bit 5/4= PLL freqency 10=48 kHz
      'bit 6= UNLOCK (1 = not locked to incoming stream)
    'endif

    if (Smode = Xconsole) then Change_Smode = Xstatus 'Flip to GUI if currently console
  endif
  '- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  'The config is stored in Config0.csv
  if (Dwait > 6000) then WriteDefaultFile 'saved 30 seconds after last delta

  '- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  'Save an update of the presets file 30 seconds after the last change
  if (Pwait > 6000) then '200 ticks per second. 6k=30 seconds
    CompactPresets
    WritePresetFile
    ReadPresetFile
  endif

  '- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  'Run the radio state machine every timer tick
  if (SpiWait > 0) then CheckSpi

  '- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  'If we are not in an idle state, then don't process user input (below)
  if (State <> 0) then continue do

  '- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  'Process Infrared remote control messages here - state=0 so we can change volume etc
  if (IrDev <> 0) then ProcessIR

  '- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  'ButtonDn and ButtonUp are set by the associated interrupt service routines. Don't want
  'to process these events in the interrupt, because that would take too long and the
  'GUI would judder under load. So we process here instead, outside of the ISR
  if (ButtonDn <> 0) then ProcessButtonDown
  if (ButtonUp <> 0) then ProcessButtonUp

  '- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  'If we are in a SCAN (Seek) process, check if a new station has been discovered
  'Bit 6 is set when a station has been discovered (or not discovered)
  if ((Seeking and 64) = 64) then ProcessScan

  '- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  'If we are in a Antenna Capacitance search process, process now
  if (CapSearch <> 0) then
    ProcessCap

  elseif (PollWait > 50) then
    'If nothing else is in progress, then re-fetch stats for current station every 250ms
    State=400 'If idle, grab fresh radio stats

  endif

  '- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  'Trigger a scan here, which will be further processed by ProcessScan above

  if ((Seeking = 0) and (DoScan > 0)) then
    if (D > 0) then print "Scan commencing"
    DoScan = 0
    NumServices = 0
    NewFrequency = AmMIN
    Seeking = &H83
    ServiceID = -1
    CompID = -1
    Change_Smode = Xscan
    DABwait = 0
    AntCap = 0
    ClearStatus
  endif

  '- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  'Manage the digital mode changes (the Si4689 supports either digital output mode
  'or analogue output mode, but not both modes together)

  if (DigitalMode <> CtrlVal(Sdigital)) then
    DigitalMode = CtrlVal(Sdigital)

    'Changes of mode are not immediately recognised by the radio chip.
    'Radio won't do anything until new frequency selected.
    'Force a change by choosing the same one we are on now.
    if (NewFrequency < AmMIN) then
      NewFrequency = Frequency
      NewServiceID = ServiceID
      NewCompID = CompID
    endif

    if (D > 0) then print "Changed digital output to " DigitalMode
  endif
LOOP
end
'-----------------------------------------------------
'Si4689 loader.bin file

CFunction LOADERBIN      'not an actual Cfunction, but an array to read data direct from flash
  00000000               'offset in 32bit words, set to zero to start below
  'data starts here
  FF000010 00000000 00000000 12345678 00000000 00000000 6CCB84FD 00000001
  DEADBEEF 00001648 00000004 00000004 00000000 50200544 00000000 00000000
  00000000 00000000 00000000 00000000 00000000 076FCF88 8100006F 5000D800
  5000E048 5000E150 50002A00 502867AC 5028656C A0002E00 50002A80 502896A0
  FFF0FF0C A0001E00 4FFFFF80 502895E8 50000040 502892F8 50000050 50000060
  FFFF0000 00001003 502008A8 502008C8 50200874 502007E0 502009F0 50200660
  5000FE00 90200403 502008FC 50286778 50002A50 50002200 501FFFFF 50260000
  A0000E00 502897D4 50008E00 A003F300 FFFFEFFF A002BE00 50002534 500027E0
  5000E134 5000E130 502860A8 50285198 000FFFFF FFF00000 50286594 5000E050
  0000ED0C 0000DEC0 0000C0DE 50286658 5000E000 00000BB7 0000C350 A000BE00
  5000E0B0 5028A570 03000000 9FFFFE00 00001000 A00020C0 000F00F3 A00020D0
  A00020E0 A00020F0 A0001004 00000800 A000C080 5000E0A8 A000C090 0000FFFF
  A000C0F0 A000C010 A000C070 50201544 A000C100 A00010F0 A0001044 A0003100
  A0001060 A0001080 A00011F0 00C00000 00400000 00004E20 5028A6A4 C3500000
  00008808 0001FFE3 A0001008 EFFC37FF EFFC3FFF A000C0E0 00060000 00040000
  A000C110 A000C120 A000C060 00020000 5028A580 02000000 5000E020 50284C50
  5000E030 5000E140 A003DE00 000103F7 000103E7 00001017 00001037 81000011
  5000E000 030361A8 004B000B 00042002 000040C7 500E0000 00000000 00000000
  00000000 4649423C 5244485F 4345535F 0A3E4E54 49422F3C 44485F46 45535F52
  3E4E5443 0000000A 21000042 5000E048 8000050A 50200400 21004136 20C03202
  15229200 820020C0 98971422 0020C00E C01522B2 22A20020 F01AB714 C6000065
  0000FFF6 21004136 20C03202 14223200 C0A03320 23520020 0005521A 420020C0
  441B1A23 0C0020C0 321B0C4A 03811E23 30930B32 08E09339 2B452600 26465526
  75265065 6085265A 1587980C 6A952664 1597B90C CFA0A26E 811D15A7 1A0C3204
  0C0008E0 0002C61A 228203BD E004AD23 0A0C0008 1D004E25 8203BDF0 04AD2B22
  060008E0 03BDFFF7 AD242282 0008E004 BDFFF3C6 25228203 08E004AD FFF08600
  E0262282 EE460008 272282FF 060008E0 2282FFEC 0008E028 82FFE9C6 08E02922
  FFE78600 E02A2282 E5460008 000000FF A2004136 052180A0 0020C032 A0C42292
  20C02099 32068100 82C46292 88AC0808 20C01A0C CC229200 C02099A0 62920020
  320781CC 08E0DA0C C0EC7C00 22B20020 10BBC0CC B20020C0 F01DCC62 0C004136
  3203814A 08E00B0C 41170C00 09313208 32065132 92F2A062 05210805 C0F99C32
  22B20020 20BB70CC B20020C0 20C0CC62 9023A200 6010AA40 20C020AA 9063A200
  C10111E5 0CC2320A 35EC3789 B250A0A2 0B8104A0 320CC132 A10008E0 0D81320C
  320EB132 A20008E0 A0B260A0 320B8104 E0320CC1 0CA10008 320D8132 E0320FB1
  0A4C0008 0B814B0C 320CC132 C10008E0 11A13210 3202B132 920020C0 99C08422
  2099A010 A10020C0 16C13217 3215D132 F13214E1 62923213 32128184 F2286B82
  6BE22B6B 236BD225 A2246BC2 0592266B C0F99C08 22E20020 20EE70CC E20020C0
  20C0CC62 9023D200 6010DD40 20C020DD 9063D200 C1321AB1 18A1321C 32199132
  92321B81 4A0C5C6A 0C0008E0 3203814A 08E01B0C 00F01D00 81004136 20C03205
  D8288200 66348080 0FA50238 00006500 0000F01D 0C006136 3205B115 820020C0
  1D31D82B 34808032 0C2E3826 3203814A 08E00B0C 32048100 08E01A0C 3202A100
  520020C0 090C166A 0C0020C0 186A921B 0C320381 0008E04A 020CF01D 42D22392
  23C2CD23 321EF1D1 0C094CF6 90620CED 0106832D 904E0C00 BF47932E 321F8105
  5002B847 AEE22022 3220910F D20020C0 DDE0D82B 0020C010 C2D86BD2 20C0F0A1
  D82BA200 C020AAC0 6BA20020 0020C0D8 C09E6952 69520020 0020C096 65966952
  0AF10009 890FF232 8105EF57 08E03221 814A0C00 0B0C3203 C80008E0 A201B811
  2291D023 9D098D32 F2043D0A FF8002A5 86A0E211 FF40FFEA EAA1E211 0FA0FFEA
  00F01D00 81004136 0882320A 471A0C89 2D0C3D68 C03220B1 2BC20020 20CCD098
  C20020C0 20C0986B 982B9200 C02099A0 07810020 986B9232 820008E0 23F100A1
  0020C032 80201FE2 20C020EE 205FE200 0000F01D 21004136 2481320A 89022232
  27322541 20C00EE2 8C243200 C0103380 64320020 00F01D8C 7C004136 32054105
  320533B6 0086FDC3 00A03200 820020C0 8080D824 33286634 C220D330 26A103C2
  3227B132 1600DC25 20C004BA D824B200 C010BB50 64B20020 C03A0CD8 24920020
  2099A0D8 920020C0 F01DD864 B20020C0 BB50D824 0020C010 0CD864B2 0020C07A
  A0D82492 20C02099 D8649200 0C320481 0008E01A 1DC1F01D F32CC232 81FF4C16
  1A0C3204 1D0008E0 000000F0 B6004136 02821433 FFA0A200 920B98A7 02B20102
  0299A702 81151BA7 28A1322A 3229B132 0B99090C 03BD0A99 08E002AD 00F01D00
  31004136 05213202 0020C032 20D82222 22803420 0020C011 1D306322 000000F0
  0C008136 269387F8 0C60A0A2 320B814B 08E001CD 8101AD00 C1B2320D 0008E010
  90904198 8169CCF4 02AD322B 1D0008E0 000000F0 A8004136 201D0C12 6B98A0BA
  923205C1 20C00009 982C8200 89F48080 E8BA1BAB 660A0C52 42D90479 26000106
  32D90149 AD01BBE7 322CE10B 98A09A20 322DB169 C01099E0 2C820020 1088B080
  C0208890 6C820020 A942F880 B8AFDC12 A732C802 190C069B 004622D9 26229800
  1C260819 322E8105 0C0008E0 3203814A 08E01B0C 00F01D00 0C004136 3203814A
  08E00B0C 31040C00 20C03202 1423B200 20C0BB1B 1463B200 A20020C0 20C01923
  14239200 C00539A7 63420020 0020C014 9C1723C2 2212663C C0FFB365 63420020
  0020C017 06166342 20C00004 1623D200 B1E57D8C 0020C0FF 0C166342 3203814A
  08E01B0C 00F01D00 B6004136 2FA17033 00686532 410002A2 3A163202 023AE606
  82371AE6 7816F0CA EFCA9216 B2169916 2BA7F3A0 F0A0C205 D255AAC7 DAD0F4A0
  16CD16C0 E0FEA0E2 4E16C0EA FFA0F20D 16C0FAF0 020C113F 0C000506 0DB837E8
  C3C2B23B 0016A5FD A20020C0 120C3064 56007BE5 04810082 01A0A232 1D0008E0
  37A90CF0 B23BE7B9 65FDC3C2 F7060014 02B3F6FF C1002146 02B23230 0102D202
  D011BB80 9BC720BB 0FC2B273 92090282 02F20402 F1C3D205 E20E02C2 CC800D02
  20CCE011 800C02E2 CCE011CC 0B02E220 E011CC80 CCD020CC 0602E263 D010AFD2
  EE8080DA 20EEF011 F211EE80 EE900A02 03029220 8011FF80 FF8011EE 08028220
  9011FF80 FF8020EE 07028220 8011FF80 AFF020FF 00296520 D4860A2D 46120CFF
  73B6FFD3 3231E13F F20202D2 DD800102 20DDF011 A22D9DE7 02B20602 11AA8005
  B220AAB0 AA800402 20AAB011 800302B2 AAB011AA 003B2520 ABA01B0C 460A2D93
  0000FFC3 46120C00 32D1FFC1 0202C232 800102E2 CCE011CC 0F9CD720 0C003365
  93ADA01D B9860A2D 000000FF B786120C E5A23BFF B5060017 0202A2FF 800102B2
  AAB011AA 00102520 A20020C0 AF063064 FDC3C2FF 0E0C0D0C B20202A2 AA800102
  20AAB011 AA40B23B 001E2511 00FFA786 BD006136 7C04CD03 0C32CCFA 0003464D
  C208A0D2 C3B2FCC4 0061A204 2F510368 AD01A932 00706505 7A2505AD 41030C00
  20C0321D B6D5C800 2146023C 08235600 56F324D2 05AD07CD 16008425 0578FE3A
  01ADD28C 81A0C750 5CB83233 08E09CC8 5622F600 503226A1 27B1A0D7 052DC232
  65092DD2 3AA0009A 0CAAAC20 C0E5B91B 05F10020 D82F9232 99A00A7C 0020C010
  C0D86F92 2FE20020 80380CD8 20C020EE D86FE200 C20004C6 DC16F324 01A0A200
  8101A092 E5993204 AD0008E0 00822505 A8FFDB86 A7B28C01 04810916 E01A0C32
  01A80008 F01D0A2D BC004136 32345102 26341226 B34C3622 42361237 124701A1
  02A18235 92341287 129701A2 02A2A233 B23212A7 12B703A2 04A2C231 0C0312C7
  22F01D02 F01D0C05 1D001522 020522F0 1522F01D 22F01D03 F01D0305 1D040522
  080522F0 0522F01D 22F01D09 F01D0515 0C004136 81BA4C17 A1B23234 3235D101
  E201A2C2 A2F202A2 52B90C03 45160012 01126207 1516224B 551526FF 97442526
  15A73B15 2615B72F E71D15C7 15F71415 04A2320B 62439537 F3460C48 055862FF
  62FFF1C6 F0460948 084862FF 62FFEEC6 ED460348 643060FF 06035832 4862FFEB
  FFE98604 620546F6 E7460248 06070CFF BD67FFE6 32364105 0C04B467 FFE28607
  06005862 072DFFE1 0000F01D 0C008136 810B0CDA 61293203 032D5169 043D6178
  4DF8C262 E0025D05 31490008 41292169 06BD14AC 01AD2169 F97CC38B 99323381
  0008E001 6A1731A8 B851C80B 041BC701 2C46020C 32376100 0C068316 FFA08204
  52000522 128701C5 01A0A250 C1001665 0CC23234 0020C008 C2322F91 B0705849
  0020C0F5 705949B2 20C041A8 5A49A200 720020C0 20C05B49 5C492200 E505A0A2
  20C00044 CC26D200 C0086D07 26E20020 F6EE07CC 070014E5 771BF9EA 1BFFA082
  9B934744 68073188 BD61A83B E541C803 20C0001B CC269200 C0086907 26A20020
  F6EA07CC C38B21B8 F97CA14B 99323381 0008E011 11C81A0C 020C01B8 B0C0BBC0
  0046832A 0C120C00 320381DA 08E01B0C 00F01D00 E5004136 1A0C000E 0C000B65
  0013252A A0000DE5 288C7480 F01D020C 0A251A0C 32349100 92322FA1 20C00C09
  584A9200 3A651A0C 32379100 B20020C0 6B07CC29 0020C008 07CC29C2 0A65F6EC
  F9EA0700 F01D120C A5004136 1A0C0009 0C000625 000DE52A A00008A5 288C7480
  F01D020C 04E51A0C 3234C100 C2322F91 20C0090C 5849C200 C0F5B020 49B20020
  41A82059 A20020C0 20C05A49 5B492200 33A54A0C 32379100 D20020C0 6D07CC29
  0020C008 07CC29E2 03A5F6EE F9EA0700 F01D120C 0C004136 91480C6A 8A20322F
  0020C093 49821A0C 00302558 C0323791 29B20020 086B07CC C20020C0 EC07CC29
  00F01DF6 21004136 2CB13237 32383132 B0322DC1 20C010B3 8022A200 B010AAC0
  20C020AA 8062A200 0CA0C392 0020C058 1B0C1A0C 65584982 20C0002E CC22C200
  C0086C07 22D20020 F6ED07CC 220020C0 F01D0003 0C004136 322F8119 920020C0
  20C05848 222A0C00 28255948 32379100 B20020C0 6B07CC29 0020C008 07CC29C2
  F825F6EC F9EA07FF 0000F01D A1004136 03BD3234 C1040A92 C9CC322F C0030A82
  4C820020 C64A0C58 20C00003 584C9200 20C00A0C 5C4CA200 80205A0C 3237D1F5
  820020C0 F820594C 0020C041 C05A4CF2 4C220020 322CE15B E0322DF1 20C010E4
  802DC200 E010CCF0 20C020CC 806DC200 1D0022A5 000000F0 0C004136 318C5C0B
  3981322F E003AD32 3AB10008 323BC132 910020C0 6CB2323C B13A0CA2 A0C2323E
  9932A9F2 323DA142 A1004F25 3EB1323F F2A0C232 A1004E65 3EB13240 F2A0C232
  A1004DA5 3EB13241 F1A0C232 A5004CE5 42A10006 3243B132 A1004E25 2CB13244
  3245C132 A1004B65 47B13246 E57C0C32 48A1004A 0C3B0C32 004A251C B23248A1
  0C0C70A0 A1004965 42C83249 0B3247B1 0048A5CC B2324AA1 1C0C73A0 CD0047E5
  81DA0C03 4BB1321B 0008E032 1C324CA1 A50C0CCB F01D0046 A1004136 10B1324D
  A50C0C32 4EA10045 3243B132 310046E5 13823223 32342120 A1376887 A4B2324F
  00A4C200 81004365 A3A23207 0008E0E8 B13250A1 43C13243 00422532 3C324FA1
  A50C1C0B 51A10041 0CCB0C32 0040E50C A1001586 53B13252 3254C132 B2003FE5
  55C10012 B71A0C32 AAF00B3C 11BBF011 46F6BCB7 1A0C0000 0B325681 0E42929A
  E03257A1 DAB20008 41BCB008 B211BBF0 A2C20D42 74B0B000 C020BBC0 53B20020
  3258A190 A20020C0 4CA18053 3259B132 D20202C2 E1C00E02 11DD2004 F004C0C0
  DDE011EE 20CCD020 1D003925 000000F0 A1004136 3EB1323D 258C0C32 40A10038
  323EB132 37658C0C 3241A100 0C323EB1 0036A58C B1323FA1 8C0C323E A10035E5
  43B1325A 00376532 0381DA0C E00B0C32 5BB10008 3220A132 B20020C0 5C91886A
  0020C032 1D886A92 000000F0 A1004136 47B1325D 650C0C32 4CA10032 325EB132
  A5325FC1 02CD0031 7C3260A1 0030E5FB 0C3261A1 651C0C1B F01D0030 A1004136
  47B1325D 650C0C32 62A1002F 3247B132 2EA50C0C 324CA100 C1325EB1 2DE53263
  3260A100 C280FB7C 20C3C001 A1002CE5 1B0C3261 2C651C0C 00F01D00 61004136
  A0823238 0D384780 06AD03BD CD326481 0008E004 0E0C063D 0ECD02DD 390020C0
  E9E2E9F2 0020C0D2 C0106242 62E20020 0020C012 C01362E2 62E20020 1562E214
  0C3202A1 1AA97649 B20020C0 AA4B1A2A F8071367 C0FF50F2 1B051BF7 4B5DB9CC
  C9F03DDD C102E932 42C9323C C03237A1 42B80020 BB0B12E9 B03210C1 20C0F4B0
  842A9200 B01099C0 20C02099 921B0C00 0381846A E0DA0C32 F01D0008 C0004136
  D2880020 6856E298 0A39560A A20020C0 2A561222 0020C004 881022B2 378BB6F2
  20C00888 11628200 1FF8F2F8 E80020C0 1262F2F2 F2E9EE8B D20020C0 CDD21022
  0020C0F8 C01062D2 22C20020 917CCC12 20C03265 12629200 A20020C0 1C0C1222
  C004EA16 22820020 0020C012 12C84238 20633380 5CC8A0CC A20020C0 03BD1122
  C0FFC0A5 22B20020 C0BB3A11 62B20020 0020C011 301222A2 20C0C0AA 1262A200
  920020C0 991B1322 920020C0 F01D1362 20C0F01D 1DD2C900 000000F0 C0004136
  D2880020 0C0338B6 0CF01D02 320381DA 08E00B0C C0130C00 12A80020 99A70298
  0020C009 0B0C22C8 0C833BC0 320381DA 08E01B0C C063CC00 D2980020 C0CB29B6
  D2A80020 3AB6120C 1D020C03 00F01DF0 0C004136 920B0CDA 03811522 92991B32
  08E01562 A8030C00 1B32B802 B702A9AA 0239013A B80020C0 A19B8C22 EBE5322F
  0020C0FF DA0C2239 0C320381 0008E01B 20C03A0C B8D29800 2429F6E2 20C01BAC
  1322D200 C20020C0 BCD71422 0020C00E C01322F2 22E20020 F03EF714 A90020C0
  0020C0D2 1866D288 0020C015 C01322B2 22920020 B72C0C14 20C00439 C0D2C900
  D2D80020 C0102D66 22F20020 1522E214 C0043EF7 D2A90020 0000F01D D1004136
  3BE1323A 0020C032 C0A26ED2 12B80020 203237C1 20C0A0DB 982CC200 C0C00A0C
  889DC9F4 87BB1B32 0BAD01BB A90020C0 0020C012 02F81288 9F87190C 0020C007
  01462299 322FA100 C0FFDF25 22A20020 C0AA1B14 62A20020 00F01D14 40004136
  F77C1063 C0307370 02580020 60105570 20C02055 1D025900 000000F0 C0004136
  02390020 0000F01D 29006136 AC013911 326631F5 E2322971 C7C210C3 0C079810
  26E9AC06 29667F19 A805DD15 8101B811 04CD3267 E10008E0 69C13268 56050C32
  11A8FDA5 2AA2190C A2020C27 29A0F8CA D8F01D83 3A0F0C17 E2B6572D 0BFDFF56
  B29DCA55 02820004 1B441B00 B2DD1B22 17D90049 A81F1B87 8101B811 69C13267
  0008E032 C13268E1 290C3269 1F0C0799 17D90D0C 26000086 661B04BD 0CFFEE46
  C607A91A 0F0CFFF9 060C17D8 B6572DEA F8AF568D 661B550B 820002B2 441B0004
  2D081B87 D90D0C0E 00040617 191C221B 17D9DD1B 0C059D97 A92A0C1F FFF34607
  41004136 0482320A 04848089 C216D816 052C0FAE 6A31023C 3205A132 B20020C0
  BBC0D82A 0020C010 C0D86AB2 2A920020 209920D8 920020C0 20C0D86A 88635200
  1C320781 0008E0EA 220020C0 07818863 E0EA1C32 A2920008 0020C030 81886392
  EA1C3207 920008E0 20C030A3 88639200 1C320781 0008E0EA C0326B91 63920020
  32078184 08E0EA1C 326C9100 920020C0 07818463 E0EA1C32 A1920008 0020C030
  81886392 EA1C3207 910008E0 2321326D 0020C032 81385292 EA1C3207 910008E0
  20C0326E 38529200 1C320781 0008E0EA C03FA392 52920020 32078130 08E0EA1C
  1CA19200 920020C0 07812052 E0EA1C32 A1920008 0020C010 81886392 EA1C3207
  C00008E0 52520020 32078120 08E0EA1C 08A19200 920020C0 07816852 E0EA1C32
  991C0008 920020C0 07815852 2CA1A232 920008E0 A1B28A04 A2A9CC00 20C01BA1
  7052A200 B0000206 20C020B9 7052B200 1C320781 0008E0EA 20A1DD7C 0020C032
  D0982AC2 20C010CC 986AC200 20C0EB7C 982A9200 C01099B0 6A920020 00F01D98
  00000000
End CFunction
