#include "timedate.h"
//for reference to cdTimer
#include "util.h"

const char daysOfWeek[9][4]={"SUN","MON","TUE","WED","THU","FRI","SAT","RPT","ALM"};  //zero = Sunday, extras align with alarm settings
const char daysInMonth[12]={31,28,31,30,31,30,31,31,30,31,30,31};
const char daysInMonthLY[12]={31,29,31,30,31,30,31,31,30,31,30,31};
const char monthsOfYear[12][4]={"JAN","FEB","MAR","APR","MAY","JUN","JUL","AUG","SEP","OCT","NOV","DEC"};   //zero-based

calDate_t getYMD(unsigned int n){    //put values into tdYear,tdMonth,tdDate,tdDay
    calDate_t r;
    r.year=0;   //2000=0
    r.month=0;  //Jan=0
    r.day=(n+5)%7;  //1/1/2000 was a Saturday
    while(n>(365+isLeapYear(r.year))){
        n=n-365;
        n=n-isLeapYear(r.year);
        r.year=r.year+1;
    }
    if(isLeapYear(r.year)){
        while(n>daysInMonthLY[r.month]){
            n=n-daysInMonthLY[r.month];
            r.month=r.month+1;
        }
    }else{
        while(n>daysInMonth[r.month]){
            n=n-daysInMonth[r.month];
            r.month=r.month+1;
        }        
    }
    r.date=(unsigned char)n;        //n should be much less than 255 by now
    return r;
}

unsigned int getSerial(calDate_t d){   //return serial, ignore day of week
    unsigned int n;
    n=d.date;
    if(isLeapYear(d.year)){
        while(d.month>0){
            n=n+daysInMonthLY[d.month-1];  //add days from previous month
            d.month=d.month-1;
        }
    }else{
        while(d.month>0){
            n=n+daysInMonth[d.month-1];  //add days from previous month
            d.month=d.month-1;
        }        
    }
    while(d.year>0){
        n=n+365;
        n=n+isLeapYear(d.year-1);    //leap day accrues from previous year
        d.year=d.year-1;
    }
    return n;
}

char isLeapYear(char n){
    //if(n==0){return 1;}         //2000 is a leap year (this is handled below)
    if(n==100){return 0;}       //2100 is not a leap year
    //if(n==200){return 0;}       //2200 is not a leap year and will not be reached before overflow
    if((n&3)==0){return 1;}     //other years that are multiples of 4 are leap years
    return 0;
}

timeAndDate_t fixTOD(timeAndDate_t t){
    while(t.time.second>59){
        t.time.second=t.time.second-60;
        t.time.minute=t.time.minute+1;
    }
    while(t.time.minute>59){
        t.time.minute=t.time.minute-60;
        t.time.hour=t.time.hour+1;                
    }
    while(t.time.hour>23){
        t.time.hour=t.time.hour-24;
        t.serial=t.serial+1;        
    }
    return t;
}

timeOfDay_t timeFromGPS(const char* s){ //get time from GPS HHMMSS.000 string
    timeOfDay_t r;
    r.hour=s[1]-'0';
    r.hour=r.hour+(s[0]-'0')*10;
    r.minute=s[3]-'0';
    r.minute=r.minute+(s[2]-'0')*10;
    r.second=s[5]-'0';
    r.second=r.second+(s[4]-'0')*10;
    //ignore fractions
    //error state:
    if((r.hour>23)||(r.minute>59)||(r.second>59)){
        r.hour=0xFF;
        r.minute=0xFF;
        r.second=0xFF;
    }
    return r;
}

calDate_t datefromGPS(const char* s){  //get date from GPS DDMMYY string
    calDate_t r;
    r.date=s[1]-'0';
    r.date=r.date+(s[0]-'0')*10;
    r.month=s[3]-'0';
    r.month=r.month+(s[2]-'0')*10;
    r.month=r.month-1;          //convert to zero based
    r.year=s[5]-'0';
    r.year=r.year+(s[4]-'0')*10;
    if(r.year<20){r.year=r.year+100;}
    if((r.date>31)||(r.month>11)){
        r.date=0;
        r.month=0;
        r.year=0;
    }
    return r;
}

void setDayOfWeek(calDate_t* t){   //update day of week from DD/MM/YY
    unsigned int n;
    n=getSerial(*t);
    t->day=(n+5)%7;  //1/1/2000 was a Saturday (ie 31/12/1999 = day 0 was a 5=Friday)
}

timeAndDate_t fixTZoffset(timeAndDate_t in, signed char min15){   //adjust in by 15min blocks
    //adjust to make it easier to apply negative offsets
    in.serial=in.serial-1;
    in.time.hour=in.time.hour+24;
    while(min15<0){
        in.time.hour=in.time.hour-1;
        min15=min15+4;
    }
    while(min15>3){
        in.time.hour=in.time.hour+1;
        min15=min15-4;
    }
    in.time.minute=in.time.minute+((unsigned char)(min15*15));    //min should be much less than 255 by now
    return fixTOD(in);
}

char countDowncdTimer(void){
    if(cdTimer.second>0){
        cdTimer.second=cdTimer.second-1;
        return 0;
    }else{
        cdTimer.second=59;
        if(cdTimer.minute>0){
            cdTimer.minute=cdTimer.minute-1;
            return 0;
        }else{
            cdTimer.minute=59;
            if(cdTimer.hour>0){
                cdTimer.hour=cdTimer.hour-1;
                return 0;
            }else{
                cdTimer.second=0;
                cdTimer.minute=0;
                cdTimer.hour=0;
                return 1;
            }
        }
    }    
}

char countDown(timeOfDay_t* t){ //take one second off and handle borrow, returns true on expiry (0:0:0)
    if(t->second>0){
        t->second=t->second-1;
        return 0;
    }else{
        t->second=59;
        if(t->minute>0){
            t->minute=t->minute-1;
            return 0;
        }else{
            t->minute=59;
            if(t->hour>0){
                t->hour=t->hour-1;
                return 0;
            }else{
                t->second=0;
                t->minute=0;
                t->hour=0;
                return 1;
            }
        }
    }
}

calDate_t getNextDOW(calDate_t t0, dayOfWeek_t d){    //find next date matching the day-of-week d on/after t0, eg 0=> Fri 1st becomes Sun 3rd
    setDayOfWeek(&t0);      //update this to match actual date
    d=d+7-t0.day;
    while(d>6){d=d-7;}
    t0=getYMD(getSerial(t0)+d);   //getYMD also sets day which should match d
    return t0;
}

unsigned int getDSTevent(calDate_t t0, DSTevent_t e){   //find date of next DST event after t0
    unsigned int s0= getSerial(t0);
    unsigned int s1;
    calDate_t t1=e.date;
    t1.year=t0.year;    
    t1=getNextDOW(t1,e.day);
    s1=getSerial(t1);
    if(s1>s0){ //same year, but later date is a good match
        return s1;
    }else{          //otherwise, it will be next years
        t1=e.date;  //reset this
        t1.year=t0.year+1;
        t1=getNextDOW(t1,e.day);
        s1=getSerial(t1);
        return s1;
    }
}

DSTstatus_t getDSTstatus(unsigned int d, DSTevent_t start, DSTevent_t end){ //check date d against start and end
    unsigned int s,e;
    if(getDSTevent(getYMD(d-1),start)==d){return DST_STARTS;}
    if(getDSTevent(getYMD(d-1),end)==d){return DST_ENDS;}    
    s=getDSTevent(getYMD(d),start);
    e=getDSTevent(getYMD(d),end);
    if(s>e){
        return DST_ACTIVE;
    }else{
        return DST_INACTIVE;
    }
}

