 /*
 * File:   uart.c
 * Author: Tim Blythman
 * Demo code for ATtiny816 on breakout board
 * UART
 * Created on 15 October 2018, 10:29 AM
 */

//UART

#include "uart.h"
#include <avr/interrupt.h>
//default with 20MHz clock and 6 divider
#define F_PER_CLK (3333333L)

void uartInit(unsigned long baud){       //start
    uart_rx_buffer_head=0;                          //reset rx buffer
    uart_rx_buffer_tail=0;
    //PORTA.DIRSET = PIN1_bm;                         //set alt TX as output 
    //PORTMUX.CTRLB |= PORTMUX_USART0_bm;             //set alt port active (TXD on PA1[17], RXD on PA2[18])
    PORTB.DIRSET = PIN2_bm;                         //set default TX as output    
    PORTMUX.CTRLB &= ~PORTMUX_USART0_bm;            //set default port as active (TXD on PB2[9], RXD on PB3[8])
    if(baud==0){return;}                            //avoid /0
    di();                                           // disable all interrupts
    USART0.CTRLB = 0;                               //reset USART
    USART0.CTRLB |= USART_RXEN_bm;                  // rx on
    USART0.CTRLB |= USART_TXEN_bm;                  // tx on 
    USART0.CTRLB |= USART_RXMODE_NORMAL_gc;         //normal rx baud mode (16 samples/bit)
    USART0.CTRLC = 0;
    USART0.CTRLC |= USART_CMODE_ASYNCHRONOUS_gc;    //async uart
    USART0.CTRLC |= USART_PMODE_DISABLED_gc;        //no parity
    USART0.CTRLC |= USART_SBMODE_1BIT_gc;           //1 stop bit
    USART0.CTRLC |= USART_CHSIZE_8BIT_gc;           //8 data bits
    USART0.BAUD=(F_PER_CLK*4+(baud/2))/(baud);      //set baud
    USART0.CTRLA &= ~USART_TXCIE_bm;                //disable tx complete interrupt
    USART0.CTRLA &= ~USART_DREIE_bm;                //disable tx data register empty interrupt
    USART0.CTRLA |= USART_RXCIE_bm;                 //enable rx complete interrupt
    ei();                                           // enable all interrupts   
}

void uartDeinit(){                       //shut down UART
    USART0.CTRLA &= ~USART_TXCIE_bm;                //disable tx complete interrupt
    USART0.CTRLA &= ~USART_DREIE_bm;                //disable tx data register empty interrupt
    USART0.CTRLA &= ~USART_RXCIE_bm;                //disable rx complete interrupt    
    USART0.CTRLB = 0;                               //reset USART
}

char uartSend(char c){                                  //returns number sent
    while((USART0.STATUS & USART_DREIF_bm) == 0){}      //wait till data register empty
    USART0.TXDATAL=c;                                   //assume 8 bits only
    return 1;    
}

char uartPrint(char* c){                                //send char array
    char n=0;
    while(*c){
        uartSend(*c++);
        n++;
    }
    return n;                       //characters written
}

char uartReceive(){                                     //take a char from rx buffer                  
  char c;
  if(uart_rx_buffer_head==uart_rx_buffer_tail){
      return -1;                                        //empty
  }
  c=uart_rx_buffer[uart_rx_buffer_tail];
  uart_rx_buffer_tail=(uart_rx_buffer_tail+1)%UART_BUFFER_SIZE;
  return c;
}

unsigned char uartAvailable(){                                    //bytes available
    return (uart_rx_buffer_head+UART_BUFFER_SIZE-uart_rx_buffer_tail)%UART_BUFFER_SIZE;
}

char uartSendInt(int n){                                //print a decimal int
    char c=0;               //number of c's sent
    char zf=0;              //leading zero suppression
    int m,t;                //multiplier, current digit
    if(n==0){uartSend('0');return 1;}   //avoid printing nothing
    if(n<0){uartSend('-');c++;n=-n;}
    for(m=10000;m>0;m=m/10){
        t=n/m;
        if(t||zf){zf=1;uartSend(t+'0');c++;}
        n=n-(m*t);
    }
    return c;
}

ISR(USART0_RXC_vect){
    if(USART0.STATUS & USART_RXCIF_bm){                                             //USART RX interrupt
        char b;
        b=USART0.RXDATAL;                                                           //RXCIF flag cleared by read
        if(((uart_rx_buffer_head+1)%UART_BUFFER_SIZE)!=uart_rx_buffer_tail){        //if space available in rx
            uart_rx_buffer[uart_rx_buffer_head]=b;                                  //store it
            uart_rx_buffer_head=(uart_rx_buffer_head+1)%UART_BUFFER_SIZE;           //update position    
        }
    }
}
