/*
 * File:   main.c
 * Author: Tim Blythman
 * Demo code for ATtiny816 on breakout board
 *
 * APIs:
 * UART- see uart.h
 * DAC- see DAC.h
 * Touch- see touch.h - uses shared capacitance method on analog pins
 * IO- see io.h
 * Time- see time.h
 * 
 * Configuration bit setup- see config.h - currently commented out so no changes occur (factory default)
 * 
 */

#include "config.h"
#include <xc.h>
#include <stdlib.h>
#include "DAC.h"
#include "uart.h"
#include "touch.h"
#include "io.h"
#include "time.h"

// Use these Enums for DAC reference:
//    VREF_DAC0REFSEL_0V55_gc
//    VREF_DAC0REFSEL_1V1_gc
//    VREF_DAC0REFSEL_2V5_gc
//    VREF_DAC0REFSEL_4V34_gc
//    VREF_DAC0REFSEL_1V5_gc

//these defines for LEDs on board:
//LED1 is connected to DAC output
#define LED2 &PORTC,0
#define LED3 &PORTC,1
#define LED4 &PORTC,2
#define LED5 &PORTC,3

//defines for touch pads
#define TOUCH1 ANALOG_PORTA4
#define TOUCH2 ANALOG_PORTA5
#define TOUCH3 ANALOG_PORTA7
#define TOUCH4 ANALOG_PORTB1

//calibration baselines and threshold for touch
#define BASELINE1 690
#define BASELINE2 730
#define BASELINE3 740
#define BASELINE4 740
#define THRESHOLD 60

int getSlider();                            //prototype for function below main

void main(void) {
    int DACvalue=0;
    int ADCresult;
    int sliderValue=0;
    DACinit(VREF_DAC0REFSEL_4V34_gc);       //init DAC with 4.3V reference
    uartInit(9600);    
    setDir(LED2,OUTPUT);
    setDir(LED3,OUTPUT);
    setDir(LED4,OUTPUT);
    setDir(LED5,OUTPUT);
    timeInit(0);                            //need to init to use timeDelay
    uartPrint("Raw analog values:\r\n");
    while(1){
        ADCresult=getAnalogTouch(TOUCH1);   //read touch1
        uartSendInt(ADCresult);
        if(ADCresult>BASELINE1+THRESHOLD){  //set LED based on threshold
            setPin(LED2,HIGH);
        }else{
            setPin(LED2,LOW);
        }
        uartSend(',');
        ADCresult=getAnalogTouch(TOUCH2);   //read touch2
        uartSendInt(ADCresult);
        if(ADCresult>BASELINE2+THRESHOLD){  //set LED based on threshold
            setPin(LED3,HIGH);
        }else{
            setPin(LED3,LOW);
        }
        uartSend(',');
        ADCresult=getAnalogTouch(TOUCH3);   //read touch3
        uartSendInt(ADCresult);
        if(ADCresult>BASELINE4+THRESHOLD){  //set LED based on threshold
            setPin(LED4,HIGH);
        }else{
            setPin(LED4,LOW);
        }
        uartSend(',');
        ADCresult=getAnalogTouch(TOUCH4);   //read touch4
        uartSendInt(ADCresult);
        if(ADCresult>BASELINE4+THRESHOLD){  //set LED based on threshold
            setPin(LED5,HIGH);
        }else{
            setPin(LED5,LOW);
        }
        uartSend(',');
        sliderValue=getSlider();            //get linear slider value
        uartSendInt(sliderValue);           //display to serial port
        uartSend('\r');
        uartSend('\n');
        if(sliderValue){DACvalue=sliderValue*2+60;}  //scale to suit DAC
        DACset(DACvalue);                            //set DAC
        if(DACvalue){DACvalue=DACvalue-5;}           //fade to off
        timeDelay(100);
    }
}

int getSlider(){            //get position on slider
    //use weighted average based on touch intensity and position
    int x[4]={20,40,60,80};             //x position
    int i[4];                           //intensities
    int iTotal=0;
    i[0]=getAnalogTouch(TOUCH1)-BASELINE1;
    if(i[0]<0){i[0]=0;}
    i[1]=getAnalogTouch(TOUCH2)-BASELINE2;
    if(i[1]<0){i[1]=0;}
    i[2]=getAnalogTouch(TOUCH3)-BASELINE3;
    if(i[2]<0){i[2]=0;}
    i[3]=getAnalogTouch(TOUCH4)-BASELINE4;
    if(i[3]<0){i[3]=0;}
    iTotal=i[0]+i[1]+i[2]+i[3];
    if(iTotal<THRESHOLD){return 0;}                             //no touch- only check 1x THRESHOLD, because only one or two active at a time
    return ((x[0]*i[0]+x[1]*i[1]+x[2]*i[2]+x[3]*i[3])/iTotal);  //20-80 for valid touch on slider
}