
#include <xc.h>
#include <string.h>
#include "PWM.h"

/*
 Channels 0 & 1 use HWPWM with a shared frequency between 732Hz and 120kHz and 7-bit to 10-bit duty cycles. They can be synchronised.
 Channels 2 & 3 use either SWPWM1 or SWPWM2. SWPWM1 is used from 30Hz to 2kHz (mode a) or 11Hz to 29Hz (mode b). SWPWM2 is used from 0.1Hz to 10Hz. If both channels use the same SWPWM, they must use the same frequency. In that case, they can be synchronised.
   
 HWPWM uses TMR2 and PWM1/PWM2 with precaler values of 1 (46875Hz-120kHz), 4 (11719-46874Hz), 16 (2930-11718Hz) or 64 (732-2929Hz)
 SWPWM1a uses TMR1 with a clock source of Fosc/4, prescaler values of 1 (184-2kHz), 2 (92-183Hz), 4 (46-91Hz) or 8 (23-45Hz)
 SWPWM1b uses TMR1 with a clock source of LFINTOSC and a prescaler value of 1 (1-22Hz)
 SWPWM2 uses TMR0 to generate interrupts at 1ms, then uses a three-entry table to determine when to update the ports (0.1-10Hz)
*/

unsigned char PWM_enabled;

// Minimum = 732, maximum = 120000
void HWPWM_SetFreq(unsigned long Hz) {
    PIE1bits.TMR2IE = 1;
    T2CON = 0;
    if( Hz >= 46875 ) {
        T2CON = 4;
        PR2 = 11999999UL / Hz;
        TMR2 = 0;
    } else if( Hz > 11718 ) {
        T2CON = 5;
        PR2 = 3000000UL / Hz;
        TMR2 = 0;
    } else if( Hz > 2929 ) {
        T2CON = 6;
        PR2 = 750000UL / Hz;
        TMR2 = 0;
    } else if( Hz > 731 ) {
        T2CON = 7;
        PR2 = 187391UL / Hz;
        TMR2 = 0;
    }
}

unsigned char pwm1_duty[2], pwm2_duty[2];

static void _HWPWM_SetDuty(unsigned char* to_here, unsigned char duty100) {
    unsigned short duty_val = (unsigned short)duty100 * (PR2+1) / 25;
    INTCONbits.GIE = 0;
    to_here[0] = duty_val << 6;
    to_here[1] = duty_val >> 2;
    INTCONbits.GIE = 1;
}

void HWPWM_SetDuty1(unsigned char duty100) {
    TRISCbits.TRISC5 = 0;
    if( duty100 > 0 && PWM_enabled ) {
        _HWPWM_SetDuty(pwm1_duty, duty100);
        PWM1CON = 0xC0;
    } else {
        PWM1CON = 0;
    }
}

void HWPWM_SetDuty2(unsigned char duty100) {
    TRISCbits.TRISC6 = 0;
    if( duty100 > 0 && PWM_enabled ) {
        _HWPWM_SetDuty(pwm2_duty, duty100);
        PWM2CON = 0xC0;
    } else {
        PWM2CON = 0;
    }
}

/*
  SWPWM1a uses TMR1 with a clock source of Fosc/4, prescaler values of 1 (184-2kHz), 2 (92-183Hz), 4 (46-91Hz) or 8 (23-45Hz)
  SWPWM1b uses TMR1 with a clock source of LFINTOSC and a prescaler value of 1 (1-22Hz)
*/
// Minimum = 1, maximum = 2000
unsigned short PR1;
unsigned char SWPWM_masks[3], SWPWM_new_masks[3];
unsigned short SWPWM_delays[4], SWPWM_new_delays[3];
unsigned char SWPWM1_duty[2], SWPWM1_state;
volatile unsigned char SWPWM1_update;
void SWPWM1_SetFreq(unsigned short Hz) {
    PIE1bits.TMR1IE = 0;

    if( Hz >= 184 ) {
        PR1 = 12000000UL / Hz;
        T1CON = 0x01;
    } else if( Hz >= 92 ) {
        PR1 = 6000000UL / Hz;
        T1CON = 0x11;
    } else if( Hz >= 46 ) {
        PR1 = 3000000UL / Hz;
        T1CON = 0x21;
    } else if( Hz >= 23 ) {
        PR1 = 1500000UL / Hz;
        T1CON = 0x31;
    } else {
        PR1 = 32000 / Hz;
        T1CON = 0xC1;
    }
    TMR1 = 0;
    ANSELCbits.ANSC3 = 0;
    TRISC &= ~((1<<3)|(1<<4));
    LATC &= ~((1<<3)|(1<<4));
    SWPWM1_UpdateDuty();

    PIE1bits.TMR1IE = 1;
}

void SWPWM1_HandleInterrupt() {
    TMR1 = SWPWM_delays[SWPWM1_state];
    if( SWPWM1_state == 0 ) {
        if( PWM_enabled )
            LATC |= SWPWM_masks[0];
    } else {
        LATC &= SWPWM_masks[SWPWM1_state];
    }
    ++SWPWM1_state;
    if( !SWPWM_delays[SWPWM1_state] ) {
        if( SWPWM1_update ) {
            SWPWM_masks[0] = SWPWM_new_masks[0];
            SWPWM_masks[1] = SWPWM_new_masks[1];
            SWPWM_masks[2] = SWPWM_new_masks[2];
            SWPWM_delays[0] = SWPWM_new_delays[0];
            SWPWM_delays[1] = SWPWM_new_delays[1];
            SWPWM_delays[2] = SWPWM_new_delays[2];
            SWPWM1_update = 0;
        }
        SWPWM1_state = 0;
    }
}

void SWPWM1_UpdateDuty() {
    unsigned short delays[2];
    unsigned char pins[2];
    SWPWM1_update = 0;
    delays[0] = (unsigned long)SWPWM1_duty[0] * PR1 / 100;
    delays[1] = (unsigned long)SWPWM1_duty[1] * PR1 / 100;

    if( delays[0] > delays[1] ) {
        delays[0] ^= delays[1];
        delays[1] ^= delays[0];
        delays[0] ^= delays[1];
        pins[0] = (1<<4);
        pins[1] = (1<<3);
    } else {
        pins[0] = (1<<3);
        pins[1] = (1<<4);
    }

    SWPWM_new_delays[2] = 0;
    if( delays[0] < delays[1] ) {
        if( delays[0] == 0 ) {
            LATC &= ~pins[0];
            SWPWM_new_masks[0] = pins[1];
            SWPWM_new_masks[1] = ~pins[1];
            SWPWM_new_delays[0] = -delays[1];
            SWPWM_new_delays[1] = delays[1] - PR1;
        } else {
            SWPWM_new_masks[0] = (1<<3)|(1<<4);
            SWPWM_new_masks[1] = ~pins[0];
            SWPWM_new_masks[2] = ~pins[1];
            SWPWM_new_delays[0] = -delays[0];
            SWPWM_new_delays[1] = delays[0] - delays[1];
            SWPWM_new_delays[2] = delays[1] - PR1;
        }
    } else {
        if( delays[0] == 0 ) {
            LATC &= ~((1<<3)|(1<<4));
            SWPWM_new_masks[0] = 0;
            SWPWM_new_delays[0] = PR1/2;
            SWPWM_new_delays[1] = PR1 - SWPWM_new_delays[0];
        } else {
            SWPWM_new_masks[0] = (1<<3)|(1<<4);
            SWPWM_new_masks[1] = ~((1<<3)|(1<<4));
            SWPWM_new_delays[0] = -delays[0];
            SWPWM_new_delays[1] = delays[0] - PR1;
        }
    }
    SWPWM1_update = 1;
}

unsigned short PWM2_tick_period, PWM2_counter, PWM2_duty[2];

void SWPWM2_Handle_msTick() {
    if( ++PWM2_counter >= PWM2_tick_period ) {
        PWM2_counter = 0;
        if( PWM_enabled ) {
            if( PWM2_duty[0] )
                LATCbits.LATC3 = 1;
            if( PWM2_duty[1] )
                LATCbits.LATC4 = 1;
        }
    } else {
        if( PWM2_counter == PWM2_duty[0] )
            LATCbits.LATC3 = 0;
        if( PWM2_counter == PWM2_duty[1] )
            LATCbits.LATC4 = 0;
    }
}

void SWPWM2_SetFreq(unsigned char dHz) {
    if( dHz )
        PWM2_tick_period = 10000 / dHz;
    else
        PWM2_tick_period = 10000;
    PWM2_counter = 0;
    PWM2_duty[0] = 0;
    PWM2_duty[1] = 0;
}

void SWPWM2_SetDuty(unsigned char duty100, unsigned char which) {
    unsigned short ticks = (unsigned long)PWM2_tick_period * duty100 / 100;
    PWM2_duty[which] = ticks;
    if( !ticks )
        LATCbits.LATC3 = 0;
}

void PWM_Enable(unsigned char enable) {
    if( enable != PWM_enabled ) {
        INTCONbits.GIE = 0;
        PIE1bits.TMR1IE = enable;
        PIE1bits.TMR2IE = enable;
        if( !enable ) {
            LATC &= ~((1<<3)|(1<<4)|(1<<5)|(1<<6));
            PWM1CON = 0;
            PWM2CON = 0;
        }
        PWM_enabled = enable;
        INTCONbits.GIE = 1;
    }
}
