
/* Pin mapping:
 *  1               : VCC (5V)
 *  2 (RA5)         : LED drive (active low)
 *  3 (RA4/AN3)     : external disable (pull low)
 *  4               : MCLR
 *  5 (RC5/PWM1)    : Out1
 *  6 (RC4)         : Out4
 *  7 (RC3/AN7)     : Out3
 *  8 (RC6/AN8/PWM2): Out3
 *  9 (RC7/AN9)     : Battery voltage sense (22k/10k divider)
 * 10 (RB7)         : Battery sense/3.3V reference enable (active high)
 * 11 (RB6)         : USB voltage sense (via 100k resistor)
 * 12 (RB5/AN11)    : TS4 (4.7k pull-up)
 * 13 (RB4/AN10)    : TS3 (4.7k pull-up)
 * 14 (RC3/AN6)     : TS2 (4.7k pull-up)
 * 15 (RC1/AN5)     : TS1 (4.7k pull-up)
 * 16 (RC0/AN4/VREF): 3.3V reference input
 * 17               : VUSB
 * 18               : PGC
 * 19               : PGD
 * 20               : GND
*/

/*
 * Test string:  restore WDTUMMgy6AM8AAoABQAZSwBxAgAAZIAAAAAAABQegAAAAAAAFB7gLgAyAHECAABkgQAAAAAAMkuBAAAAAAAySwAAgAHQBwEAAGSCAAAAAAAyS4IAAAAAADJLAACAAdAHAQAAZIMAAAAAADJLgwAAAAAAMksAAIABEQMA
 * another test: restore vDRwMCwz6AN4AAoACgAySyBOAAAZZIMAAAAAACMygAAAAAAAIzLIMgAyIE4AAABkgQAAAAAAMkuBAAAAAAAySwAAIAHQBwEAAGSCAAAAAAAyS4IAAAAAADJLAACAAdAHAQAAZIMAAAAAADJLgwAAAAAAMksAAIABbjwA
 * good config:  restore vDRwMCwz6AMIBwoAWgAySyBOAAAZZIMAAAAAACMygAAAAAAAIzLIMgAyIE4AAABkgQAAAAAAMkuBAAAAAAAySwAAIAHQBwEAAGSCAAAAAAAyS4IAAAAAADJLAACAAdAHAQAAZIMAAAAAADJLgwAAAAAAMksAAIABbjwA
 */

#include "system.h"
#include "PWM.h"
#include <stdio.h>
#include <ctype.h>

#include "app_device_cdc_basic.h"
#include "app_led_usb_status.h"

#include "usb.h"
#include "usb_device.h"
#include "usb_device_cdc.h"

#include "config.h"
#include "flash.h"
#include "therm.h"
#include "pwmlogic.h"
#include "dumprestore.h"
#include "strings.h"

active_config_t active_config;
overrides_t overrides;
status_t status;

static uint8_t readBuffer[128];//CDC_DATA_OUT_EP_SIZE];
static char cmdBuffer[255];
unsigned char cmdBufferPos;

extern void delay_us10(unsigned char us10);

unsigned short ReadADC(unsigned char channel) {
    ADCON0 = (channel << 2) | 3;
    delay_us10(50);
    ADCON0bits.GO = 1;
    while( ADCON0bits.GO )
        ;
    return (((unsigned short)ADRESH)<<8) | ADRESL;
}

static void copy_config_to_from_flash(unsigned char to) {
    unsigned char i;
    unsigned add = HEFLASH_START;
    char* buf = (char*)active_config.buf;
    for( i = 0; i < 4; ++i ) {
        if( to )
            HEFLASH_writeBlock(add, buf, 32);
        else
            HEFLASH_readBlock(buf, add, 32);
        add += 32;
        buf += 32;
    }
}

static void multiply_short_by_10(unsigned short* val) {
    *val *= 10;
}

static char* print_mV(char* dest, signed short val, unsigned short div_by, signed char num_dp) {
    unsigned short dig_val;
    if( val < 0 ) {
        *dest++ = '-';
        val = -val;
    }
    dig_val = div_by;
    while( dig_val < 10000 && dig_val < val ) {
        unsigned short temp = dig_val;
        multiply_short_by_10(&temp);
        if( temp <= val )
            dig_val = temp;
        else
            break;
    }
    while(1) {
        unsigned char dig = val / dig_val;
        *dest++ = '0' + (char)dig;
        if( dig_val == 1 )
            break;
        val -= dig * dig_val;
        if( dig_val == div_by && num_dp )
            *dest++ = '.';
        if( dig_val < div_by && --num_dp <= 0 )
            break;
        dig_val /= 10;
    }
    return dest;
}

static unsigned long get_long_Hz(unsigned char which) {
    return ((unsigned long*)&active_config.cfg.outputs[which].PWM_Hz)[0] & 0xFFFFFUL;
}
unsigned char SWPWM2_override;
static void update_PWM_freqs() {
    unsigned long lval = get_long_Hz(0);
    if( !lval )
        lval = get_long_Hz(1);
    HWPWM_SetFreq(lval);

    SWPWM2_override = 0;
    if( active_config.cfg.outputs[2].PWM_Hz.num >= 10 ) {
        SWPWM1_SetFreq(active_config.cfg.outputs[2].PWM_Hz.num);
        if( active_config.cfg.outputs[3].PWM_Hz.num && active_config.cfg.outputs[3].PWM_Hz.num != active_config.cfg.outputs[2].PWM_Hz.num )
            SWPWM2_override = 1;
    } else if( active_config.cfg.outputs[3].PWM_Hz.num >= 10 ) {
        SWPWM1_SetFreq(active_config.cfg.outputs[3].PWM_Hz.num);
    } else {
        SWPWM1_SetFreq(0);
    }

    if( active_config.cfg.outputs[2].PWM_Hz.num > 0 && active_config.cfg.outputs[2].PWM_Hz.num < 10 )
        SWPWM2_SetFreq(active_config.cfg.outputs[2].PWM_Hz.num);
    else if( active_config.cfg.outputs[3].PWM_Hz.num > 0 && (active_config.cfg.outputs[3].PWM_Hz.num < 10 || SWPWM2_override) )
        SWPWM2_SetFreq(active_config.cfg.outputs[3].PWM_Hz.num);
    else
        SWPWM2_SetFreq(0);
}

static unsigned char read_decimal_str(char** ptr, unsigned short* to_here, unsigned char num_dp) {
    char* start;
    unsigned char temp;
    unsigned short val = 0;

    start = *ptr;

    while( 1 ) {
        temp = **ptr - '0';
        if( temp > 9 || val > 6552 )
            break;
        multiply_short_by_10(&val);
        val += temp;
        ++(*ptr);
    }
    if( *ptr == start )
        return 0;
    temp = num_dp;
    while( temp ) {
        multiply_short_by_10(&val);
        --temp;
    }
    if( **ptr == '.' ) {
        unsigned short dec = 0;
        ++(*ptr);
        start = *ptr;
        while( 1 ) {
            temp = **ptr - '0';
            if( temp > 9 || dec > 6552 )
                break;
            multiply_short_by_10(&dec);
            dec += temp;
            ++(*ptr);
            if( --num_dp == 0 )
                break;
        }
        if( *ptr == start )
            return 0;
        while( num_dp ) {
            multiply_short_by_10(&dec);
            --num_dp;
        }
        val += dec;
    }

    *to_here = val;
    return 1;
}

static void reset_overrides() {
    overrides.supply_mV = 0;
    overrides.temp_dC[0] = 0x8000;
    overrides.temp_dC[1] = 0x8000;
    overrides.temp_dC[2] = 0x8000;
    overrides.temp_dC[3] = 0x8000;
}

extern unsigned char duties[4];
unsigned char PWM_enabled;
static void ParseCommand() {
    char* ptr = cmdBuffer;
    unsigned char i;

    if( !FLASH_cmp_string_caseinsensitive(&ptr, STR_show_status) || cmdBuffer[0] == '\0' ) {
        FLASH_read_string(&ptr, STR__r_nSupply__);
        ptr = print_mV(ptr, status.supply_mV, 1000, 2);
        *ptr++ = 'V';
        if( overrides.supply_mV )
            FLASH_read_string(&ptr, STR___OVERRIDE_);
        FLASH_read_string(&ptr, STR____);

        if( status.cur_mode == mode_sleep ) {
            FLASH_read_string(&ptr, STR__SLEEP_);
        } else if( status.cur_mode == mode_active ) {
            FLASH_read_string(&ptr, STR__ACTIVE_);
        } else {
            FLASH_read_string(&ptr, STR__COOLDOWN_);
        }
        ptr = print_mV(ptr, status.mode_timer, 1, 0);
        FLASH_read_string(&ptr, STR_s__r_n);

        for( i = 0; i < 4; ++i ) {
            FLASH_read_string(&ptr, STR__In_);
            ptr = print_mV(ptr, i+1, 1, 0);
            if( !(active_config.cfg.input_config1&(1<<i)) ) {
                FLASH_read_string(&ptr, STR___disabled_r_n);
            } else {
                FLASH_read_string(&ptr, STR___);
                ptr = print_mV(ptr, status.input_temp_dC[i], 100, 2);
                if( overrides.temp_dC[i] != 0x8000 ) {
                    FLASH_read_string(&ptr, STR_C__OVERRIDE__r_n);
                } else {
                    FLASH_read_string(&ptr, STR_C_r_n);
                }
            }
        }
        for( i = 0; i < 4; ++i ) {
            FLASH_read_string(&ptr, STR_Out_);
            ptr = print_mV(ptr, i+1, 1, 0);
            if( active_config.cfg.outputs[i].disabled ) {
                FLASH_read_string(&ptr, STR___disabled_r_n);
            } else {
                FLASH_read_string(&ptr, STR___);
                ptr = print_mV(ptr, duties[i], 1, 0);
                FLASH_read_string(&ptr, STR_pct_);
                if( duties[i] > 0 && duties[i] < 100 ) {
                    if( i < 2 && get_long_Hz(i) >= 32768 ) {
                        ptr = print_mV(ptr, get_long_Hz(i) / 10, 1, 0);
                        *ptr++ = '0';
                    } else {
                        ptr = print_mV(ptr, active_config.cfg.outputs[i].PWM_Hz.num, 1, 0);
                        if( active_config.cfg.outputs[i].PWM_Hz.denom > 1 ) {
                            *ptr++ = '/';
                            ptr = print_mV(ptr, active_config.cfg.outputs[i].PWM_Hz.denom, 1, 0);
                        }
                    }
                    FLASH_read_string(&ptr, STR_Hz);
                }
                if( !PWM_enabled ) {
                    FLASH_read_string(&ptr, STR___OFF_);
                }
                FLASH_read_string(&ptr, STR__r_n);
            }
        }
    } else if( !FLASH_cmp_string_caseinsensitive(&ptr, STR_dump) ) {
        FLASH_read_string(&ptr, STR__r_nrestore_);
        ptr += encode_base64(active_config.buf, ptr, sizeof(active_config.cfg));
        FLASH_read_string(&ptr, STR__r_n);
    } else if( !FLASH_cmp_string_caseinsensitive(&ptr, STR_restore_) ) {
        unsigned char temp = decode_base64(ptr, cmdBuffer);
        if( temp == sizeof(active_config.cfg) ) {
            mymemcpy((uint8_t*)&active_config.cfg, (uint8_t*)cmdBuffer, sizeof(active_config.cfg));
            update_PWM_freqs();
            goto good;
//            FLASH_read_string(&ptr, STR__r_nOK_r_n);
        } else {
            goto err;
//            FLASH_read_string(&ptr, STR__r_nError_r_n);
        }
    } else if( !FLASH_cmp_string_caseinsensitive(&ptr, STR_override_) ) {
        if( !FLASH_cmp_string_caseinsensitive(&ptr, STR_clear) ) {
            reset_overrides();
            goto good;
        } else if( !FLASH_cmp_string_caseinsensitive(&ptr, STR_supply_) ) {
            unsigned short temp;
            if( read_decimal_str(&ptr, &temp, 3) && tolower(ptr[0]) == 'v' && ptr[1] == '\0' ) {
                overrides.supply_mV = temp;
                goto good;
            }
        } else if( tolower(ptr[0]) == 't' && tolower(ptr[1]) == 's' ) {
            unsigned char ts;
            ptr += 2;
            ts = ptr[0] - '1';
            if( ts < 4 && ptr[1] == ' ' ) {
                unsigned short temp;
                unsigned char neg = 0;
                ptr += 2;
                if( *ptr == '-' ) {
                    neg = 1;
                    ++ptr;
                }
                if( read_decimal_str(&ptr, &temp, 2) && tolower(ptr[0]) == 'c' && ptr[1] == '\0' ) {
                    overrides.temp_dC[ts] = neg ? -(signed short)temp : temp;
                good:
                    ptr = cmdBuffer;
                    FLASH_read_string(&ptr, STR__r_nOK_r_n);
                    goto good2;
                }
            }
        }
    err:
        ptr = cmdBuffer;
        FLASH_read_string(&ptr, STR__r_nError_r_n);
    } else if( !FLASH_cmp_string_caseinsensitive(&ptr, STR_save) ) {
        copy_config_to_from_flash(1);
        goto good;
//        FLASH_read_string(&ptr, STR__r_nOK_r_n);
    } else if( !FLASH_cmp_string_caseinsensitive(&ptr, STR_revert) ) {
        copy_config_to_from_flash(0);
        update_PWM_freqs();
        goto good;
//        FLASH_read_string(&ptr, STR__r_nOK_r_n);
    } else {
        FLASH_read_string(&ptr, STR__r_nUnknown_command__r_n);
    }
good2:
    putUSBUSART(cmdBuffer, ptr-cmdBuffer);
}

unsigned short batt_mV;
unsigned short last_ms_timer, this_ms_timer, last_raw_ms_timer, this_raw_ms_timer;

//static unsigned char check_usb_present() {
//}

static inline void SWPWM1_SetDuty1(unsigned char duty100) {
    SWPWM1_duty[0] = duty100;
    SWPWM1_UpdateDuty();
}

static inline void SWPWM1_SetDuty2(unsigned char duty100) {
    SWPWM1_duty[1] = duty100;
    SWPWM1_UpdateDuty();
}

static void UpdateBattmV() {
    unsigned char i;
    // this acts like an IIR filter, smoothing out battery voltage readings
    batt_mV >>= 1;
    // plus we do some averaging
    for( i = 0; i < 8; ++i )
        batt_mV += ReadADC(9);
//    batt_mV += ReadADC(9) - batt_mV >> 4;
//    batt_mV = (ReadADC(9) + ReadADC(9) + ReadADC(9) + ReadADC(9)) * 4;
//    batt_mV = (ReadADC(9) + ReadADC(9)) * 8;
//    batt_mV = ReadADC(9) * 16;
}

MAIN_RETURN main(void) {
    unsigned char i;
    unsigned char no_usb, sleep_delay;
#if defined(USE_INTERNAL_OSC)
    //Make sure to turn on active clock tuning for USB full speed
    //operation from the INTOSC
    OSCCON = 0xFC;  //HFINTOSC @ 16MHz, 3X PLL, PLL enabled
    ACTCON = 0x90;  //Active clock tuning enabled for USB
#endif
    LATB = 0;
    LATC = 0;
    // RB6 = USB supply detection

    INTCONbits.PEIE = 1; // enable peripheral interrupts - general interrupts enabled by following function call
    copy_config_to_from_flash(0);
    reset_overrides();
    update_PWM_freqs();

    ADCON1 = 2|16|/*32|*/64|128;
    TRISAbits.TRISA5 = 0;
    TRISBbits.TRISB7 = 0;
    LATBbits.LATB7 = 1;

    while( 1 ) {
        LATAbits.LATA5 = 1;
        while( 1 ) {
            i = 8;
            do {
                TRISBbits.TRISB6 = 0;
                TRISBbits.TRISB6 = 1;
                _delay(1);
                if( !PORTBbits.RB6 )
                    goto NoUSB;
            } while( --i );
            break;
        NoUSB:
            UpdateBattmV();
            //batt_mV = ReadADC(9) * 16;
            if( batt_mV >= active_config.cfg.supply.on_mV )
                break;

            ADCON0 = 0;
            LATBbits.LATB7 = 0;
            WDTCON = 0x1B; // Watchdog timer enabled, 8s interval
            SLEEP();
            LATBbits.LATB7 = 1;
            ++status.mode_timer;

        }

        LATAbits.LATA5 = 0;
        InitSensors();
        USBDeviceInit();
        USBDeviceAttach();

        status.cur_mode = mode_active;
        status.mode_timer = 0;

        no_usb = 0;
        while(1)
        {
            CLRWDT();
            WDTCON = 0x0D; // Watchdog timer enabled, 64ms interval

            if( no_usb == 255 && status.cur_mode == mode_sleep )
                break;

            #if defined(USB_POLLING)
                // Interrupt or polling method.  If using polling, must call
                // this function periodically.  This function will take care
                // of processing and responding to SETUP transactions
                // (such as during the enumeration process when you first
                // plug in).  USB hosts require that USB devices should accept
                // and process SETUP packets in a timely fashion.  Therefore,
                // when using polling, this function should be called
                // regularly (such as once every 1.8ms or faster** [see
                // inline code comments in usb_device.c for explanation when
                // "or faster" applies])  In most cases, the USBDeviceTasks()
                // function does not take very long to execute (ex: <100
                // instruction cycles) before it returns.
                USBDeviceTasks();
            #endif

            UpdateSensorTemps();
            UpdateBattmV();
            PWM_Enable(batt_mV >= (PWM_enabled ? 4500 : 5000));
            if( overrides.supply_mV )
                status.supply_mV = overrides.supply_mV;
            else
                status.supply_mV = batt_mV;
            if( status.supply_mV >= active_config.cfg.supply.on_mV ) {
                if( status.cur_mode != mode_active ) {
                    if( status.cur_mode == mode_cooldown )
                        status.mode_timer = active_config.cfg.supply.cooldown_delay;
                    else
                        status.mode_timer = 0;
                    status.cur_mode = mode_active;
                }
                sleep_delay = 0;
            } else if( status.supply_mV < active_config.cfg.supply.off_mV ) {
                if( status.cur_mode != mode_sleep && (++sleep_delay & 128) ) {
                    status.cur_mode = mode_sleep;
                    status.mode_timer = 0;
                }
            } else if( status.supply_mV < active_config.cfg.supply.cooldown_min_mV ) {
                if( status.cur_mode == mode_active ) {
                    if( status.mode_timer >= active_config.cfg.supply.cooldown_delay ) {
                        status.cur_mode = mode_cooldown;
                        status.mode_timer = 0;
                    } else if( status.mode_timer < active_config.cfg.supply.cooldown_min_on_time ) {
                        status.cur_mode = mode_sleep;
                        status.mode_timer = 0;
                    }
                } else if( status.cur_mode == mode_cooldown && status.mode_timer >= active_config.cfg.supply.cooldown_s && status.cur_mode != mode_sleep ) {
                    status.cur_mode = mode_sleep;
                    status.mode_timer = 0;
                }
            } else {
                sleep_delay = 0;
            }

            this_ms_timer = get_ms_timer();
            if( this_ms_timer < last_ms_timer )
                ++status.mode_timer;
            last_ms_timer = this_ms_timer;

            this_raw_ms_timer = get_raw_ms_timer();
            update_duty_cycles(this_raw_ms_timer - last_raw_ms_timer);
            last_raw_ms_timer = this_raw_ms_timer;
            if( active_config.cfg.outputs[0].PWM_Hz.num || active_config.cfg.outputs[0].PWM_Hz.denom )
                HWPWM_SetDuty2(duties[0]); // these outputs are swapped on the board so we re-swap them in software
            if( active_config.cfg.outputs[1].PWM_Hz.num || active_config.cfg.outputs[1].PWM_Hz.denom )
                HWPWM_SetDuty1(duties[1]); // these outputs are swapped on the board so we re-swap them in software
            if( active_config.cfg.outputs[2].PWM_Hz.num )
                if( active_config.cfg.outputs[2].PWM_Hz.num >= 10 )
                    SWPWM1_SetDuty2(duties[2]); // these outputs are swapped on the board so we re-swap them in software
                else
                    SWPWM2_SetDuty(duties[2], 1);
            if( active_config.cfg.outputs[3].PWM_Hz.num )
                if( active_config.cfg.outputs[3].PWM_Hz.num >= 10 && !SWPWM2_override )
                    SWPWM1_SetDuty1(duties[3]); // these outputs are swapped on the board so we re-swap them in software
                else
                    SWPWM2_SetDuty(duties[3], 0);

            if( USBGetDeviceState() >= CONFIGURED_STATE && !USBIsDeviceSuspended() ) {
                no_usb = 0;
                if( !PWM_enabled )
                    LATAbits.LATA5 = (get_ms_timer() < 500);
                else
                    LATAbits.LATA5 = 0;

                if( USBUSARTIsTxTrfReady() == true ) {
                    uint8_t i, parsed;
                    uint8_t numBytesRead;

                    numBytesRead = getsUSBUSART(readBuffer, sizeof(readBuffer));
                    parsed = 0;
                    for( i = 0; i < numBytesRead; ++i ) {
//                        if( cmdBufferPos == sizeof(cmdBuffer) ) {
//                            mymemcpy/*memmove*/(cmdBuffer, cmdBuffer+1, sizeof(cmdBuffer)-1);
//                            --cmdBufferPos;
//                        }
                        if( readBuffer[i] == '\b' ) {
                            if( cmdBufferPos )
                                --cmdBufferPos;
                        } else if( readBuffer[i] != '\n' ) {
                            cmdBuffer[cmdBufferPos] = (char)readBuffer[i];
                            if( cmdBuffer[cmdBufferPos] == '\r' ) {
                                cmdBuffer[cmdBufferPos] = '\0';
                                ParseCommand();
                                ++parsed;
                                cmdBufferPos = 0;
                            } else {
                                if( ++cmdBufferPos == sizeof(cmdBuffer) )
                                    cmdBufferPos = 0;
                            }
                        }
                    }
                    if( !parsed ) {
                        putUSBUSART(readBuffer, numBytesRead);
                    }
                }
                CDCTxService();
            } else if( no_usb < 255 ) {
                ++no_usb;
            }
        }

        USBDeviceDetach();
        DeinitSensors();
    }
}
