/*
* File: Flash.c
*
* Self-Write Flash support functions
*
* Author: Lucio Di Jasio
*
* Created on August 28, 2013
*/
#include "flash.h"
#include <ctype.h>

static unsigned FLASH_read (unsigned address) {
    // 1. load the address pointers
    PMADR = address;
    PMCON1bits.CFGS = 0; //select the Flash address space
    PMCON1bits.RD = 1; //next operation will be a read
    NOP();
    NOP();
    // 2. return value read
    return PMDAT;
} //FLASH_read


/**
* unlock Flash Sequence
*/
static void _unlock (void) {
    #asm
        BANKSEL PMCON2
        MOVLW 0x55
        MOVWF PMCON2 & 0x7F
        MOVLW 0xAA
        MOVWF PMCON2 & 0x7F
        BSF PMCON1 & 0x7F,1 ; set WR bit
        NOP
        NOP
    #endasm
}//unlock

static void FLASH_write (unsigned address, unsigned data) {
    // 1. disable interrupts (remember setting)
//    char temp = INTCONbits.GIE;
//    INTCONbits.GIE = 0;
    // 2. load the address pointers
    PMADR = address;
    PMDAT = data;
    PMCON1bits.FREE = 0; // next operation will be a write
    // 3. perform unlock sequence
    _unlock();
    // 4. restore interrupts
//    if (temp)
//        INTCONbits.GIE = 1;
}//FLASH_write

/*
* File: HEFlash.c
*
* High Endurance Flash - EEPROM emulation and support routines
*
* Author: Lucio Di Jasio
*
* Created on August 28, 2013
*/

/******************************************************************************
* High Endurance Flash functions
*/
char HEFLASH_writeBlock (unsigned add, char* data, char count) {
    // 1. obtain absolute address in HE FLASH row
    INTCONbits.GIE = 0;
    PMCON1bits.WREN = 1; // enable Flash memory write/erase
    PMCON1bits.CFGS = 0; // select the Flash address space
    // 2. check input parameters
//    if ((count > FLASH_ROWSIZE)||(radd >= HEFLASH_MAXROWS))
//        return -1;//return parameter error
    // 3. erase the entire row
    PMADR = add;
    PMCON1bits.FREE = 1; // next operation will be an erase
    _unlock();
    // 4. fill the latches with data
    PMCON1bits.LWLO = 1; // 1 = latch, 0 = write row
    while (--count) {
        //load data in latches without writing
        FLASH_write (add++, (unsigned) *data++);
    }
    PMCON1bits.LWLO = 0; // 1 = latch, 0 = write row
    // no delay here!!!
    // 5. last byte of data -> write
    FLASH_write (add, (unsigned) *data);
    // NOTE: 2ms typ. delay here!!!
    // 6. return success
    PMCON1bits.WREN = 0; // disable Flash memory write/erase
    INTCONbits.GIE = 1;
    return PMCON1bits.WRERR; //0 success, 1 = write error
} //HEFLASH_writeBlock

void HEFLASH_readBlock (char *buffer, unsigned add, char count) {
    // 1. obtain absolute address in HE FLASH row
    INTCONbits.GIE = 0;
    // 2. check input parameters
//    if ((count > FLASH_ROWSIZE)||(radd >= HEFLASH_MAXROWS))
//        return -1;
    // 3. read content
    while (count) {
        *buffer++ = (char) FLASH_read (add++);
        count--;
    }
    // 4. success
    INTCONbits.GIE = 1;
//    return 0;
} //HEFLASH_readBlock

void FLASH_read_string(char** ptr, unsigned address) {
    unsigned short val;
    unsigned char val2;
//    char temp = INTCONbits.GIE;
    INTCONbits.GIE = 0;

    while( (val = FLASH_read(address)) ) {
        val2 = val&127;
        *(*ptr)++ = val2;
        val2 = (val>>7)&127;
        if( !val2 )
            break;
        *(*ptr)++ = val2;
        ++address;
    }
    *(*ptr) = '\0';
//    if( temp )
    INTCONbits.GIE = 1;
}

signed char FLASH_cmp_string(char** ptr, unsigned address) {
    unsigned short val;
    char* old = *ptr;
    char ret = 0, last, chr;
//    char temp = INTCONbits.GIE;
    INTCONbits.GIE = 0;

    while( (val = FLASH_read(address)) ) {
        last = val&127;
        ret = **ptr - last;
        if( ret )
            break;
        ++(*ptr);

        chr = (val>>7)&127;
        if( !chr )
            break;
        last = chr;
        ret = **ptr - last;
        if( ret )
            break;
        ++(*ptr);

        ++address;
    }
//    if( temp )
    INTCONbits.GIE = 1;

    if( last != ' ' && **ptr )
        ret = **ptr - last;
    if( ret || last != ' ' )
        *ptr = old;
    return ret;
}

static char tolowerdiff(char a, char b) {
    return tolower(a) - tolower(b);
}

signed char FLASH_cmp_string_caseinsensitive(char** ptr, unsigned address) {
    unsigned short val;
    char* old = *ptr;
    char ret = 0, last, chr;
//    char temp = INTCONbits.GIE;
    INTCONbits.GIE = 0;

    while( (val = FLASH_read(address)) ) {
        last = val&127;
        ret = tolowerdiff(**ptr, last);
        if( ret )
            break;
        ++(*ptr);

        chr = (val>>7)&127;
        if( !chr )
            break;
        last = chr;
        ret = tolowerdiff(**ptr, last);
        if( ret )
            break;
        ++(*ptr);

        ++address;
    }
//    if( temp )
    INTCONbits.GIE = 1;

    if( last != ' ' && **ptr )
        ret = tolowerdiff(**ptr, last);
    if( ret || last != ' ' )
        *ptr = old;
    return ret;
}
