/*  
 * USB CDC-ACM Demo
 *
 * This file may be used by anyone for any purpose and may be used as a
 * starting point making your own application using M-Stack.
 *
 * It is worth noting that M-Stack itself is not under the same license as
 * this file.
 *
 * M-Stack is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  For details, see sections 7, 8, and 9
 * of the Apache License, version 2.0 which apply to this file.  If you have
 * purchased a commercial license for this software from Signal 11 Software,
 * your commerical license superceeds the information in this header.
 *
 * Alan Ott
 * Signal 11 Software
 * 2014-05-12
 
 Modified Tim Blythman 2018 to behave as SPI bridge
 V1 pinout: C0 D0
            A4 DI
            A5 CK
            C5 LT
 
 V2 pinout: NA D0 (keep using C0 in software)
            C2 DI
            C3 CK
            C1 LT
 
 Default- does twinkle/chase pattern unless USB enumerated usb_is_configured()
 Hex SPI mode 9600 baud
 Binary SPI mode 19200 baud
 I2C without pullups 38400 baud
 I2C will send start/stop based on lines being sent, or Start (S), Stop (P) can be sent manually.
  
 */
#define _XTAL_FREQ 16000000
#include "usb.h"
#include <xc.h>
#include <string.h>
#include "usb_config.h"
#include "usb_ch9.h"
#include "usb_cdc.h"
#include "hardware.h"

#ifdef MULTI_CLASS_DEVICE
static uint8_t cdc_interfaces[] = { 0 };
#endif

#define BUFSIZE 128
#define I2CMODE 1
#define SPIMODE 3

#define SDAHI {TRISCbits.TRISC2=1;NOP();}
#define SDALO {TRISCbits.TRISC2=0;LATCbits.LATC2=0;}
#define SCLHI {TRISCbits.TRISC3=1;NOP();}
#define SCLLO {TRISCbits.TRISC3=0;LATCbits.LATC3=0;}

char txbuf[BUFSIZE]="";
char rxbuf[BUFSIZE]="";
char IOmode=0;
char I2CstartSent=0;

int txptr=0;
int rxptr=0;
char I2Cnybblecount;    //need to send/check ack bit after LSB of byte

uint8_t clocknybble(uint8_t n);
uint8_t clockbit(uint8_t n);
uint8_t clockbyte(uint8_t n);
char dec2hex(int8_t n);
int8_t hex2dec(char n);
void IOsetmode(char mode);
uint8_t I2Cbit(uint8_t n);
uint8_t I2Cnybble(uint8_t n);
void I2Cstart();
void I2Cstop();

static struct cdc_line_coding line_coding =
{
	115200,
	CDC_CHAR_FORMAT_1_STOP_BIT,
	CDC_PARITY_NONE,
	8,
};

int main(void)
{
    int i;
    int8_t j;
    char c;
    unsigned long t,t0=0;           //tick counter
    unsigned long lastt=0;       //clock tick counter    
    hardware_init();

#ifdef MULTI_CLASS_DEVICE
	cdc_set_interface_list(cdc_interfaces, sizeof(cdc_interfaces));
#endif
	usb_init();

    TRISA=0;        //IO setup, all outputs, low
    TRISC=0;        
    TRISCbits.TRISC0=1;//except C0
    LATA=0;
    LATC=0;
    ANSELA=0;
    ANSELC=0;
	srand(42);
    while (1) {
        if (!usb_is_configured()){          //default mode if not enumerated
            t=t0>>14;               //gives ~ 2Hz increment on t
            t0++;            
            if((t>2)&(t!=lastt)){           //slight delay to avoid flickering on hotplug, don't do anything unless t changes
                if(t<20){                   //turn on all for first 10s
                    LATCbits.LATC1=0;       //CS low
                    clockbyte(0xFF);        //all on
                    clockbyte(0xFF);        //all on
                    LATCbits.LATC1=1;       //CS high            
                }else if(t<600){            
                    LATCbits.LATC1=0;       //CS low
                    clockbyte(rand());      //twinkle
                    clocknybble(rand()&15);
                    clockbit(rand()&1);     //output 13 bits to avoid byte align
                    LATCbits.LATC1=1;       //CS high                                
                }else if(t<1200){            
                    if(t&7){
                        LATCbits.LATC1=0;       //CS low
                        clockbit(t&4);          //marching dots, 3 off, 4 on
                        LATCbits.LATC1=1;       //CS high                                                        
                    }
                }else{
                    t0=20L<<14;              //reset to after start
                }
            }
            lastt=t;
        }else{
            if(line_coding.dwDTERate<10000){ //slow mode (HEX)
                IOsetmode(SPIMODE);
                    for(i=0;i<rxptr;i++){       //do something to received data
                    j=hex2dec(rxbuf[i]);
                    if(j>-1){              //hex data
                        txbuf[txptr++]=dec2hex(clocknybble(j));
                        if(txptr>BUFSIZE-4){txptr=BUFSIZE-4;}       //clip
                    }else{
                        switch(rxbuf[i]){
                            case 'T': LATAbits.LATA4=1;break;
                            case 't': LATAbits.LATA4=0;break;
                            case 'U': LATAbits.LATA5=1;break;
                            case 'u': LATAbits.LATA5=0;break;
                            case 'V': LATCbits.LATC1=1;break;
                            case 'v': LATCbits.LATC1=0;break;
                            case 'W': LATCbits.LATC2=1;break;
                            case 'w': LATCbits.LATC2=0;break;
                            case 'X': LATCbits.LATC3=1;break;
                            case 'x': LATCbits.LATC3=0;break;
                            case 'Y': LATCbits.LATC4=1;break;
                            case 'y': LATCbits.LATC4=0;break;
                            case 'Z': LATCbits.LATC5=1;break;
                            case 'z': LATCbits.LATC5=0;break;
                        }
                    }
                }        
                rxptr=0;        //empty rx buffer        
                if(txptr){      //send line of tx if we have data
                    txbuf[txptr]='\r';
                    txbuf[txptr+1]='\n';
                    txptr=txptr+2;
                }
            }else if(line_coding.dwDTERate<20000){      //fast mode - binary
                IOsetmode(SPIMODE);
                LATCbits.LATC1=0;//CS low
                for(i=0;i<rxptr;i++){  
                    txbuf[txptr++]=clockbyte(rxbuf[i]);         //read and write byte
                    if(txptr>BUFSIZE-2){txptr=BUFSIZE-2;}       //clip
                }
                LATCbits.LATC1=1;//CS high
                rxptr=0;        //empty buffer        
            }else{            //above 20kbaud=>I2C
                IOsetmode(I2CMODE);      //I2C with/without pullups C2 data, C3 clock
                for(i=0;i<rxptr;i++){       //do something to received data
                    char jtemp;
                    j=hex2dec(rxbuf[i]);
                    if(j>-1){              //hex data
                        if(I2CstartSent==0){
                            I2Cstart();
                            I2CstartSent=1;
                        }
                        jtemp=I2Cnybble(j);
                        txbuf[txptr++]=dec2hex(jtemp&0xF);
                        if(I2Cnybblecount&1){
                            if(jtemp&0x10){
                                txbuf[txptr++]='N';//nack
                            }else{
                                txbuf[txptr++]='K';//ack                            
                            }
                        }
                        if(txptr>BUFSIZE-4){txptr=BUFSIZE-4;}       //clip
                        I2Cnybblecount++;
                    }else{
                        switch(rxbuf[i]){
                            case 'S':
                            case 's':
                                I2Cstart();
                                I2Cnybblecount=0;
                                I2CstartSent=1;
                                break;
                            case 'P':
                            case 'p':
                                I2Cstop();
                                txbuf[txptr]='\r';
                                txbuf[txptr+1]='\n';
                                txptr=txptr+2;
                                I2CstartSent=0;
                                break;
                            case 10:
                            case 13:
                                I2Cstop();                                
                                I2Cnybblecount=0;
                                txbuf[txptr]='\r';
                                txbuf[txptr+1]='\n';
                                txptr=txptr+2;
                                I2CstartSent=0;
                                break;                            
                        }                                    
                    }
                }
                rxptr=0;        //empty rx buffer        
            }            
        }
        
		/* Send data to the PC */
		// needs to be populated by user code
        
        if (usb_is_configured() &&
		    !usb_in_endpoint_halted(2) &&
		    !usb_in_endpoint_busy(2)) {
			int i,k;
			unsigned char *buf = usb_get_in_buffer(2);
            k=txptr;
            if(k){
                if(k>32){k=32;}     //limit data sent            
                for (i = 0; i < k; i++) {buf[i] = txbuf[i];}
                for(i=0;i<txptr;i++){txbuf[i]=txbuf[i+k];}  //move buffer up
                txptr=txptr-k;
                usb_send_in_buffer(2, k);
            }
		}

		/* Handle data received from the host */
		if (usb_is_configured() &&
		    !usb_out_endpoint_halted(2) &&
		    usb_out_endpoint_has_data(2)) {
			const unsigned char *out_buf;
			size_t out_buf_len;
			int i,k;
			out_buf_len = usb_get_out_buffer(2, &out_buf);
            k=rxptr+out_buf_len;
            if(k>BUFSIZE-2){k=BUFSIZE-2;}
            for(i=rxptr;i<k;i++){rxbuf[i]=out_buf[i-rxptr];}
            rxptr=k;
            usb_arm_out_endpoint(2);
		}
    }
}

void IOsetmode(char mode){     //I2C needs input on C2
    switch(mode){
        case I2CMODE:
            if(IOmode!=I2CMODE){
                IOmode=I2CMODE;
                TRISCbits.TRISC3=1;//C3 float hi
                TRISCbits.TRISC2=1;//C2, start floating hi
                LATCbits.LATC2=0;       //set low, toggle TRIS to float high
                I2CstartSent=0;         //clear I2C states
                I2Cnybblecount;
            }
            break;
        case SPIMODE:
            if(IOmode!=SPIMODE){
                IOmode=SPIMODE;
                TRISCbits.TRISC3=0;
                LATCbits.LATC3=0;       //low
                TRISCbits.TRISC2=0;//C2, output
                LATCbits.LATC2=0;       //set low
            }
            break;            
    }
}

uint8_t I2Cbit(uint8_t n){      //send one bit, return bit
    char b=0;
    if(n!=0){
        SDAHI
    }else{
        SDALO
    }
    NOP();
    NOP();
    SCLHI
    NOP();
    NOP();
    b=PORTCbits.RC2;        //read bit
    SCLLO
    return b;
}

uint8_t I2Cnybble(uint8_t n){     //send byte,return read byte + ACK
    char b=0;
    b=(I2Cbit(n&8)<<3);
    b=b+(I2Cbit(n&4)<<2);
    b=b+(I2Cbit(n&2)<<1);
    b=b+(I2Cbit(n&1));
    if(I2Cnybblecount&1){        //on odd nybbles, send/receive ack
        if(I2Cbit(1)){
            b=b|0x10;
        }
    }
    return b;
}

void I2Cstart(){                 //send start condition
    SDAHI          //allow repeated start
    NOP();
    SDALO
    NOP();
    SCLLO
}

void I2Cstop(){                  //send stop condition
    SDALO
    NOP();
    SCLHI
    NOP();
    SDAHI
}

uint8_t clockbyte(uint8_t n){
    uint8_t r=0;
    r=clockbit(n&128);
    r=(r<<1)|clockbit(n&64);
    r=(r<<1)|clockbit(n&32);
    r=(r<<1)|clockbit(n&16);
    r=(r<<1)|clockbit(n&8);
    r=(r<<1)|clockbit(n&4);
    r=(r<<1)|clockbit(n&2);
    r=(r<<1)|clockbit(n&1);    
    return r;
}

uint8_t clocknybble(uint8_t n){
    uint8_t r=0;
    r=clockbit(n&8);
    r=(r<<1)|clockbit(n&4);
    r=(r<<1)|clockbit(n&2);
    r=(r<<1)|clockbit(n&1);    
    return r;
}

uint8_t clockbit(uint8_t n){
    uint8_t r=0;
    if(n!=0){
        LATCbits.LATC2=1;        //set bit
    }else{
        LATCbits.LATC2=0;                
    }
    r=PORTCbits.RC0;
    LATCbits.LATC3=1;            //toggle clock
    LATCbits.LATC3=0;
    return r;
}

char dec2hex(int8_t n){
    n=n&15;
    if(n<10){
        return n+'0';
    }else{
        return n-10+'A';
    }
}

int8_t hex2dec(char n){
    if((n>='0')&&(n<='9')){return n-'0';}
    if((n>='a')&&(n<='f')){return n+10-'a';}
    if((n>='A')&&(n<='F')){return n+10-'A';}
    return -1;
}

/* Callbacks. These function names are set in usb_config.h. */
void app_set_configuration_callback(uint8_t configuration)
{

}

uint16_t app_get_device_status_callback()
{
	return 0x0000;
}

void app_endpoint_halt_callback(uint8_t endpoint, bool halted)
{

}

int8_t app_set_interface_callback(uint8_t interface, uint8_t alt_setting)
{
	return 0;
}

int8_t app_get_interface_callback(uint8_t interface)
{
	return 0;
}

void app_out_transaction_callback(uint8_t endpoint)
{

}

void app_in_transaction_complete_callback(uint8_t endpoint)
{

}

int8_t app_unknown_setup_request_callback(const struct setup_packet *setup)
{
	/* To use the CDC device class, have a handler for unknown setup
	 * requests and call process_cdc_setup_request() (as shown here),
	 * which will check if the setup request is CDC-related, and will
	 * call the CDC application callbacks defined in usb_cdc.h. For
	 * composite devices containing other device classes, make sure
	 * MULTI_CLASS_DEVICE is defined in usb_config.h and call all
	 * appropriate device class setup request functions here.
	 */
	return process_cdc_setup_request(setup);
}

int16_t app_unknown_get_descriptor_callback(const struct setup_packet *pkt, const void **descriptor)
{
	return -1;
}

void app_start_of_frame_callback(void)
{

}

void app_usb_reset_callback(void)
{

}

/* CDC Callbacks. See usb_cdc.h for documentation. */

int8_t app_send_encapsulated_command(uint8_t interface, uint16_t length)
{
	return -1;
}

int16_t app_get_encapsulated_response(uint8_t interface,
                                      uint16_t length, const void **report,
                                      usb_ep0_data_stage_callback *callback,
                                      void **context)
{
	return -1;
}

int8_t app_set_comm_feature_callback(uint8_t interface,
                                     bool idle_setting,
                                     bool data_multiplexed_state)
{
	return -1;
}

int8_t app_clear_comm_feature_callback(uint8_t interface,
                                       bool idle_setting,
                                       bool data_multiplexed_state)
{
	return -1;
}

int8_t app_get_comm_feature_callback(uint8_t interface,
                                     bool *idle_setting,
                                     bool *data_multiplexed_state)
{
	return -1;
}

int8_t app_set_line_coding_callback(uint8_t interface,
                                    const struct cdc_line_coding *coding)
{
	line_coding = *coding;
	return 0;
}

int8_t app_get_line_coding_callback(uint8_t interface,
                                    struct cdc_line_coding *coding)
{
	/* This is where baud rate, data, stop, and parity bits are set. */
	*coding = line_coding;
	return 0;
}

int8_t app_set_control_line_state_callback(uint8_t interface,
                                           bool dtr, bool dts)
{
	return 0;
}

int8_t app_send_break_callback(uint8_t interface, uint16_t duration)
{
	return 0;
}


#ifdef _PIC14E
void interrupt isr()
{
	usb_service();
}
#elif _PIC18

#ifdef __XC8
void interrupt high_priority isr()
{
	usb_service();
}
#elif _PICC18
#error need to make ISR
#endif

#endif
