//default pin configuration
#define MAINS_SENSE A1
#define VIN_SENSE A3
#define MAINS_ISOLATE 8
#define INVERTER_CONTROL 2
#define INVERTER_SENSE 4
#define BATTERY_SENSE A2
#include "TimerOne.h"

#define SAMPLECOUNT 256
volatile unsigned int mainsSamplePtr=0;
volatile unsigned int mainsSamples[SAMPLECOUNT];
unsigned int mainsSampleBackup[SAMPLECOUNT];

void setup() {
  pinMode(MAINS_ISOLATE,OUTPUT);
  digitalWrite(MAINS_ISOLATE,LOW);
  Serial.begin(115200);
  inverterSetup();                        //set IOs for inverter control
  Timer1.initialize(1000);                //set for 1kHz sample rate = 20 samples per mains cycle
  Timer1.attachInterrupt(getSamples); 
}

void loop() {
  static int s=0;
  float mainsFromRMS,mainsFromPP;     //variables for testing status
  saveSamples();                                          //transfer samples from live to backup array
  mainsFromRMS=getMainsFromRMS();                         //get mains voltages
  mainsFromPP=getMainsFromPP();
  noInterrupts();                                   //interrupts off for analog readings
  Serial.println();
  Serial.print("Battery Sense:             ");
  Serial.println(analogReadOversample(BATTERY_SENSE));
  Serial.print("VIN Sense:                 ");
  Serial.println(analogReadOversample(VIN_SENSE));
  Serial.print("Mains Sense:               ");
  Serial.println(analogRead(MAINS_SENSE));
  Serial.print("Mains RMS (before scaling):");
  Serial.println(mainsFromRMS);
  Serial.print("Mains P-P (before scaling):");
  Serial.println(mainsFromPP);
  interrupts();                                     //interrupts back on
  if(s==0){
    Serial.print("Inverter turn off:");
    if(inverter(0)){
      Serial.println("OK");
    }else{
      Serial.println("Fail");
    }
    s=1;
  }else{
    Serial.print("Inverter turn on:");
    if(inverter(1)){
      Serial.println("OK");
    }else{
      Serial.println("Fail");
    }
    s=0;    
  }
  delay(5000);
}

void inverterSetup(){                                     //set up IO pins
  digitalWrite(INVERTER_CONTROL,LOW);
  pinMode(INVERTER_CONTROL,OUTPUT);
  pinMode(INVERTER_SENSE,INPUT_PULLUP);  
}

int inverter(int s){                                      //control inverter, report back
  int stat;
  stat=inverterRun();
  if(s==stat){return 2;}                                  //inverter already at that state
  if(s){                                                  //to turn on
    digitalWrite(INVERTER_CONTROL,HIGH);                  //pulse the trigger and wait until LED comes on
    for(int i=0;i<10;i++){
      delay(100);
      if(digitalRead(INVERTER_SENSE)==0){                 //it's on
        digitalWrite(INVERTER_CONTROL,LOW);               //drop output
        return 1;                                         //all good
      }
    }
    digitalWrite(INVERTER_CONTROL,LOW);                   //drop output
  }else{                                                  //to turn off
    digitalWrite(INVERTER_CONTROL,HIGH);                  //toggle high
    delay(100);    
    digitalWrite(INVERTER_CONTROL,LOW);                   //toggle low
    delay(100);                                           //wait to change state
    if(digitalRead(INVERTER_SENSE)==1){return 1;}         //it's off, all good
  }
  return 0;                                               //no response, fail
}

int inverterRun(){                                        //check status
  return digitalRead(INVERTER_SENSE)?0:1;
}

void getSamples(){                                        //ISR for samples, uses ~34% CPU time
  digitalWrite(13,HIGH);                                  //flag time in ISR on digital pin                              
  mainsSamples[mainsSamplePtr++]=analogRead(MAINS_SENSE); //read mains
  mainsSamplePtr=mainsSamplePtr&(SAMPLECOUNT-1);
  digitalWrite(13,LOW);                                   //leaving ISR
}

float getMainsFromPP(){                                   //estimated RMS voltage from P-P reading
  float v;
  v=(getMainsMax()-getMainsMin());            //find P-P voltage (unscaled for test)
  v=v*0.353553391;                                        //P-P=>RMS by multiplying by 1/(2sqrt(2))
  return v;
}

float getMainsFromRMS(){                                  //RMS calculation over sample period
  int m=getMainsMean();                                   //get mean
  long s=0;
  float r;
  for(int i=0;i<SAMPLECOUNT;i++){
    s=s+(mainsSampleBackup[i]-m)*(mainsSampleBackup[i]-m);//add up squares
  }
  r=s;                                                    //convert to float
  r=r/SAMPLECOUNT;                                        //average
  return sqrt(r);                             //get actual value (returned unscaled for test)
}

void saveSamples(){                                 //save current buffer
  noInterrupts();                                   //interrupts off so we get a consistent set of samples
  for(int i=0;i<SAMPLECOUNT;i++){
      mainsSampleBackup[i]=mainsSamples[i];         //copy
  }
  interrupts();                                     //interrupts back on
}

int getMainsMin(){                                        //work out minimum of samples
  unsigned int a=1023;
  for(int i=0;i<SAMPLECOUNT;i++){
      if(mainsSampleBackup[i]<a){a=mainsSampleBackup[i];}
  }
  return a;
}

int getMainsMax(){                                        //work out maximum of samples
  unsigned int a=0;
  for(int i=0;i<SAMPLECOUNT;i++){
      if(mainsSampleBackup[i]>a){a=mainsSampleBackup[i];}
  }
  return a;
}

int getMainsMean(){                                       //work out mean (needed for RMS calculations)
  return 480;     //nominal zero point for 10k/10k/75k network
  long a=0;
  for(int i=0;i<SAMPLECOUNT;i++){
    a=a+mainsSampleBackup[i];
  }
  return a/SAMPLECOUNT;
}

float analogReadOversample(int pin){
  float t=0;
  analogRead(pin);          //allow sample and hold to charge
  for(int i=0;i<SAMPLECOUNT;i++){
    t=t+analogRead(pin);   
  }
  return t/SAMPLECOUNT;
}

