#include <EEPROM.h>

//voltage conversion scaling factors
#define VIN_SCALE (0.053856)
#define BATTERY_SCALE (0.053856)
#define MAINS_SCALE (2.709)

//thresholds for triggers
#define BATTERY_CRITICAL (23.0)
#define BATTERY_MIN (25.0)
#define BATTERY_OK (27.0)
#define VIN_MIN (11.0)
#define VIN_OK (11.5)
#define MAINS_MIN (180.0)
#define MAINS_DB (20.0)
#define MAINS_MAX (260.0)
#define VIN_CRITICAL (10.5)

//timeouts
#define REPORTINTERVAL 250
#define MAINS_DELAY 10000
#define VIN_DELAY 5000
#define BATTERY_CRITICAL_DELAY 5000

//programmable parameters to load from EEPROM
float vin_scale;
float battery_scale;
float mains_scale;
float battery_critical;
float battery_min;
float battery_ok;
float vin_min;
float vin_ok;
float mains_min;
float mains_db;
float mains_max;
float mains_delay;
float vin_delay;
float battery_critical_delay;
float vin_critical;

char eepromid[]="SC11106181";
char checkid[12]="";

char menuItems[15][24]={
  "VIN_SCALE             :",
  "BATTERY_SCALE         :",
  "MAINS_SCALE           :",
  "BATTERY_CRITICAL      :",
  "BATTERY_MIN           :",
  "BATTERY_OK            :",
  "VIN_MIN               :",
  "VIN_OK                :",
  "MAINS_MIN             :",
  "MAINS_DB              :",
  "MAINS_MAX             :",
  "MAINS_DELAY           :",
  "VIN_DELAY             :",
  "BATTERY_CRITICAL_DELAY:",
  "VIN_CRITICAL          :"
};

//for setup menus
#define BUFFERLENGTH 20
int menuSet=-1;                           //menu setting, -1= not setting, 0>= setting item n
char nBuffer[BUFFERLENGTH+2]="";


void setup() {
  Serial.begin(115200);
  if(loadFromEEPROM()){                   //load settings from EEPROM
    Serial.println(F("Load from EEPROM OK"));    
  }else{
    Serial.println(F("EEPROM Error, defaults loaded and saved"));
  }
}

void loop() {
  doMenu();
}

int saveToEEPROM(){                                       //save values to EEPROM
  EEPROM.put(0,vin_scale);
  EEPROM.put(4,battery_scale);
  EEPROM.put(8,mains_scale);
  EEPROM.put(12,battery_critical);
  EEPROM.put(16,battery_min);
  EEPROM.put(20,battery_ok);
  EEPROM.put(24,vin_min);
  EEPROM.put(28,vin_ok);
  EEPROM.put(32,mains_min);
  EEPROM.put(36,mains_db);
  EEPROM.put(40,mains_max);
  EEPROM.put(44,mains_delay);
  EEPROM.put(48,vin_delay);
  EEPROM.put(52,battery_critical_delay);
  EEPROM.put(56,vin_critical);
}

int loadFromEEPROM(){                                       //load from EEPROM, returns 0 if defaults had to be loaded due to error
  readEEPROMid(checkid,strlen(eepromid));       //read id flag
  if(!strmatch(checkid,eepromid)){
    Serial.println(F("EEPROM signature not found, loading and saving defaults"));
    loadDefaults();                   //set values to default
    saveToEEPROM();                   //save these
    writeEEPROMid(eepromid);          //set id flag in EEPROM
  }
  EEPROM.get(0,vin_scale);
  EEPROM.get(4,battery_scale);
  EEPROM.get(8,mains_scale);
  EEPROM.get(12,battery_critical);
  EEPROM.get(16,battery_min);
  EEPROM.get(20,battery_ok);
  EEPROM.get(24,vin_min);
  EEPROM.get(28,vin_ok);
  EEPROM.get(32,mains_min);
  EEPROM.get(36,mains_db);
  EEPROM.get(40,mains_max);
  EEPROM.get(44,mains_delay);
  EEPROM.get(48,vin_delay);
  EEPROM.get(52,battery_critical_delay);
  EEPROM.get(56,vin_critical);
#ifdef DEBUG
  printCurrent();
#endif  
//if there's a problem with EEPROM values
  if(vin_scale<=0){loadDefaults();saveToEEPROM();return 0;}
  if(battery_scale<=0){loadDefaults();saveToEEPROM();return 0;}
  if(battery_critical<0){loadDefaults();saveToEEPROM();return 0;}
  if(battery_min<0){loadDefaults();saveToEEPROM();return 0;}
  if(battery_ok<battery_min){loadDefaults();saveToEEPROM();return 0;}
  if(vin_min<0){loadDefaults();saveToEEPROM();return 0;}
  if(vin_ok<vin_min){loadDefaults();saveToEEPROM();return 0;}
  if(mains_db<0){loadDefaults();saveToEEPROM();return 0;}
  if(mains_min<0){loadDefaults();saveToEEPROM();return 0;}
  if(mains_max<2*mains_db){loadDefaults();saveToEEPROM();return 0;}
  if(mains_delay<0){loadDefaults();saveToEEPROM();return 0;}
  if(vin_delay<0){loadDefaults();saveToEEPROM();return 0;}
  if(battery_critical_delay<0){loadDefaults();saveToEEPROM();return 0;}
  if(vin_critical<0){loadDefaults();saveToEEPROM();return 0;}
  return 1;
}

void loadDefaults(){                                    //for when EEPROM fails (or is blank)
  vin_scale=VIN_SCALE;
  battery_scale=BATTERY_SCALE;
  mains_scale=MAINS_SCALE;
  battery_critical=BATTERY_CRITICAL;
  battery_min=BATTERY_MIN;
  battery_ok=BATTERY_OK;
  vin_min=VIN_MIN;
  vin_ok=VIN_OK;
  mains_min=MAINS_MIN;
  mains_db=MAINS_DB;
  mains_max=MAINS_MAX;
  mains_delay=MAINS_DELAY;
  vin_delay=VIN_DELAY;
  battery_critical_delay=BATTERY_CRITICAL_DELAY;
  vin_critical=VIN_CRITICAL;
#ifdef DEBUG
  printCurrent();
  Serial.println(F("Loaded from defaults"));
#endif
  }

float getValue(int n){
  switch(n){
    case(0):return vin_scale;break;
    case(1):return battery_scale;break;
    case(2):return mains_scale;break;
    case(3):return battery_critical;break;
    case(4):return battery_min;break;
    case(5):return battery_ok;break;
    case(6):return vin_min;break;
    case(7):return vin_ok;break;
    case(8):return mains_min;break;
    case(9):return mains_db;break;
    case(10):return mains_max;break;
    case(11):return mains_delay;break;
    case(12):return vin_delay;break;
    case(13):return battery_critical_delay;break;
    case(14):return vin_critical;break;
  }  
}

void setValue(int n,float val){
  switch(n){
    case(0):vin_scale=val;break;
    case(1):battery_scale=val;break;
    case(2):mains_scale=val;break;
    case(3):battery_critical=val;break;
    case(4):battery_min=val;break;
    case(5):battery_ok=val;break;
    case(6):vin_min=val;break;
    case(7):vin_ok=val;break;
    case(8):mains_min=val;break;
    case(9):mains_db=val;break;
    case(10):mains_max=val;break;
    case(11):mains_delay=val;break;
    case(12):vin_delay=val;break;
    case(13):battery_critical_delay=val;break;
    case(14):vin_critical=val;break;
  }
}

void doMenu(){                                      //get serial input and process
  float num;
  while(Serial.available()){                        //no input => nothing to do
    int a=Serial.read();
    if(menuSet==-1){                                //not setting a parameter
      if((a>='A')&&(a<='O')){
        menuSet=a-'A';                              //start setting A-O
        Serial.println(F("Enter Value for:"));
        Serial.print(menuItems[menuSet]);        
        Serial.println(getValue(menuSet),7);
        a=0;
//        debug=0;                                    //turn debug data off during setting
      }
//      if(a=='~'){debug=!debug;}                     //toggle debug mode with ~
      if(a=='?'){printHelp();}                      //Print commands
      if(a=='s'){
        saveToEEPROM();
        Serial.println(F("Saved to EEPROM"));
      }
      if(a=='l'){
        loadFromEEPROM();
        Serial.println(F("Loaded from EEPROM"));
      }
      if(a=='d'){
        loadDefaults();
      }
      if(a=='p'){
        printCurrent();
      }
    }
    if(menuSet>-1){                                 //in the middle of setting a parameter
      if(a>31){                                     //ascii character
        if(strlen(nBuffer)<BUFFERLENGTH){
          nBuffer[strlen(nBuffer)+1]=0;
          nBuffer[strlen(nBuffer)]=a;               //add typed character to buffer           
          Serial.write(a);                          //echo
        }        
      }else{
        if(a==8){                                   //backspace
          if(strlen(nBuffer)>0){
            nBuffer[strlen(nBuffer)-1]=0;
            Serial.write(8);                        //echo backspace
            }
        }
        if(a==13){
          if(strlen(nBuffer)>0){                    //ignore first CR on systems that have line entry (eg Arduino SM)
            num=atof(nBuffer);                      //convert to float
            Serial.println(num,7);                  //display number
            if((num>0)||(num==0&&nBuffer[0]=='0'&&nBuffer[1]==0)){
              Serial.println(F("Valid"));           //check if valid number
              setValue(menuSet,num);                //and put in variable
            }else{
              Serial.println(F("Not Valid"));
            }              
            Serial.print(menuItems[menuSet]);        
            Serial.println(getValue(menuSet),7);
            nBuffer[0]=0;                           //empty string
            menuSet=-1;                             //ready for next
          }
        }
      }
    }
  }
}

void printHelp(){
  Serial.println(F("UPS SETUP"));
  Serial.println(F("?     This Help"));
  Serial.println(F("~     Toggle voltage status output on/off"));
  Serial.println(F("A-O   Enter parameter, followed by number and enter"));
  Serial.println(F("s     Save current to EEPROM"));
  Serial.println(F("l     Load from EEPROM"));
  Serial.println(F("d     Load from defaults"));
  Serial.println(F("p     Print current parameters"));
  Serial.println();
}

void printCurrent(){
  Serial.println(F("Current Values:"));
  Serial.print(F("A:"));Serial.print(menuItems[0]);Serial.println(vin_scale,7);
  Serial.print(F("B:"));Serial.print(menuItems[1]);Serial.println(battery_scale,7);
  Serial.print(F("C:"));Serial.print(menuItems[2]);Serial.println(mains_scale,7);
  Serial.print(F("D:"));Serial.print(menuItems[3]);Serial.println(battery_critical,7);
  Serial.print(F("E:"));Serial.print(menuItems[4]);Serial.println(battery_min,7);
  Serial.print(F("F:"));Serial.print(menuItems[5]);Serial.println(battery_ok,7);
  Serial.print(F("G:"));Serial.print(menuItems[6]);Serial.println(vin_min,7);
  Serial.print(F("H:"));Serial.print(menuItems[7]);Serial.println(vin_ok,7);
  Serial.print(F("I:"));Serial.print(menuItems[8]);Serial.println(mains_min,7);
  Serial.print(F("J:"));Serial.print(menuItems[9]);Serial.println(mains_db,7);
  Serial.print(F("K:"));Serial.print(menuItems[10]);Serial.println(mains_max,7);
  Serial.print(F("L:"));Serial.print(menuItems[11]);Serial.println(mains_delay,7);
  Serial.print(F("M:"));Serial.print(menuItems[12]);Serial.println(vin_delay,7);
  Serial.print(F("N:"));Serial.print(menuItems[13]);Serial.println(battery_critical_delay,7);
  Serial.print(F("O:"));Serial.print(menuItems[14]);Serial.println(vin_critical,7);
}

void writeEEPROMid(char* id){                       //write id to end of eeprom
  int i;
  unsigned int s=EEPROM.length();
  unsigned int n=strlen(id);
  if(n>s-2){return;}                //fail if it won't fit
  for(i=0;i<n+1;i++){
    EEPROM.write(s-n-1+i,id[i]);    
  }  
}

void readEEPROMid(char* id,unsigned int idsize){      //read idsize+null bytes into id buffer
  unsigned int i=0;
  byte b=1;                 //to start in while loop
  unsigned int s=EEPROM.length();
  unsigned int p=s-idsize-1;
  if(p<0){return;}
  while(b&&(p<s)){
    b=EEPROM.read(p);
    id[i]=b;
    i++;
    p++;
  }  
}

bool strmatch(char* x, char* y){   //true if strings match
  unsigned int n;
  n=strlen(x);
  if(n!=strlen(y)){return 0;}       //can't match if different lengths
  for(int i=0;i<n;i++){
    if(x[i]!=y[i]){return 0;}      //different at some point
  }
  return 1;                        //all match
}
