// Arduino based UPS
// 24V inverter backed by Lithium Batteries
// Arduino detects mains via isolating/reducing transformer
// switches between mains and inverter as necessary using relays
// state machine implemented for mains/inverter switching
// monitors battery voltage, its own VIN
// controls/monitors inverter via opto-isolated controls
// TimerOne interrupt samples mains at 1kHz, giving 20 samples per cycle
// buffer is copied before calculations to avoid aliasing artifacts
// calculates mains voltage based on P-P and RMS values
// GREEN LED: solid when on mains, flashing when mains is out of range
// YELLOW LED: solid when on inverter, flashing if inverter fault (ie inverter not running)
// RED LED: flashes faster with lower battery voltage, solid on below critical
// Settings/thresholds all in EEPROM
// Text/serial menu- use ? for help (only if APCPROTOCOL is not defined)
// Handles shutdown on VIN off- powers down inverter and engages mains transformer to prevent bootloop on mains
// define DEBUG for serial messages/setup
// define APCPROTOCOL to enable APC compatible output on serial port

//#define DEBUG
#ifndef DEBUG
#define APCPROTOCOL
#endif

#include <EEPROM.h>
#include "TimerOne.h"
#include "Wire.h"
#define I2C_ADDR  0x20  // 0x20 is the address with all jumpers removed
#include <avr/wdt.h>

char eepromid[]="SC11106181";
char checkid[12]="";

//voltage conversion scaling factors
#define VIN_SCALE (0.053856)
#define BATTERY_SCALE (0.053856)
#define MAINS_SCALE (2.709)

//IO pins
#define MAINS_SENSE A1
#define VIN_SENSE A3
#define MAINS_ISOLATE 8
#define PIEZO_PIN 9
#define INVERTER_CONTROL 2
#define INVERTER_SENSE 4
#define BATTERY_SENSE A2

//relay output bits
#define INVERTER_CONNECT 1
#define INVERTER_CHANGEOVER 2
#define MAINS_CONNECT 4
#define GREEN_LED 8
#define AMBER_LED 16
#define RED_LED 32
#define ALL_OFF 0

//state machine states
#define OUTPUT_OFF 0
#define OUTPUT_TO_MAINS 1
#define OUTPUT_ON_MAINS 2
#define OUTPUT_TO_INVERTER 3
#define OUTPUT_ON_INVERTER 4

//thresholds for triggers
#define BATTERY_CRITICAL (23.0)
#define BATTERY_MIN (25.0)
#define BATTERY_OK (27.0)
#define VIN_MIN (11.0)
#define VIN_OK (11.5)
#define MAINS_MIN (180.0)
#define MAINS_DB (20.0)
#define MAINS_MAX (260.0)
#define VIN_CRITICAL (10.5)

//timeouts
#define REPORTINTERVAL 250
#define MAINS_DELAY 10000
#define VIN_DELAY 5000
#define BATTERY_CRITICAL_DELAY 5000
#define SHUTDOWN_COMMAND_DELAY (60000L)

//samples
#define SAMPLECOUNT 32
#define PIEZOFREQ 1000
//nominal zero point for mains divider 10k-(10k|75k) => 480
#define MAINSZEROPOINT 480

int outputState=OUTPUT_OFF;
int shutDown=0;                           //set to 1 to shutdown
int ledState=0;                           //LEDs off to start
int ledFlash=0;                           //^with ledState to give animated effects
int relayDriver=0;                        //all off
int mainsmin,mainsmax,mainsmean;          //temp storage

unsigned long mainsTimer=0;               //how long has mains been OK?
unsigned long vinTimer=0;                 //how long since VIN dropped below threshold?
unsigned long batteryCriticalTimer=0;     //how long since battery went critical?

//rolling sample buffers
volatile unsigned int mainsSamples[SAMPLECOUNT];
volatile unsigned int batterySamples[SAMPLECOUNT];
volatile unsigned int vinSamples[SAMPLECOUNT];
unsigned int mainsSampleBackup[SAMPLECOUNT];

//battery and VIN are sampled alternately to reduce time spent in interrupts
volatile int mainsSamplePtr=0;
volatile int dcSamplePtr=0;
volatile bool samplephase=false;

//status flags
bool vinOK=0,batteryOK=0,mainsOK=0;

//debug status
bool debug=0;

//programmable parameters to load from EEPROM
float vin_scale;
float battery_scale;
float mains_scale;
float battery_critical;
float battery_min;
float battery_ok;
float vin_min;
float vin_ok;
float mains_min;
float mains_db;
float mains_max;
float mains_delay;
float vin_delay;
float battery_critical_delay;
float vin_critical;

char menuItems[15][24]={
  "VIN_SCALE             :",
  "BATTERY_SCALE         :",
  "MAINS_SCALE           :",
  "BATTERY_CRITICAL      :",
  "BATTERY_MIN           :",
  "BATTERY_OK            :",
  "VIN_MIN               :",
  "VIN_OK                :",
  "MAINS_MIN             :",
  "MAINS_DB              :",
  "MAINS_MAX             :",
  "MAINS_DELAY           :",
  "VIN_DELAY             :",
  "BATTERY_CRITICAL_DELAY:",
  "VIN_CRITICAL          :"
};

//for setup menus
#define BUFFERLENGTH 20
int menuSet=-1;                           //menu setting, -1= not setting, 0>= setting item n
char nBuffer[BUFFERLENGTH+2]="";

//for variable frequency flashing
#define MIN_HALFCYCLE 50
#define MAX_HALFCYCLE 1000

//status buffers for APC protocol
char apcGoutput='O';                      //default='O' for no change
float apcMoutput=-1;                      //maximum line voltage
float apcNoutput=999;                     //minimum line voltage
bool apcKStrigger=0;
unsigned long apcKStime=0;                //flag and timer for K/S timed shutdowns
bool apcUtrigger=0;                       //simulate line fail
int apcZcount=0;                          //need at least two 'Z' to trigger shutdown

void setup() {
  int iStatus;                            //temp variable for inverter status
#ifdef DEBUG
  Serial.begin(115200);
#endif
#ifdef APCPROTOCOL
  Serial.begin(2400);
#endif
  if(loadFromEEPROM()){                   //load settings from EEPROM
#ifdef DEBUG
    Serial.println(F("Load from EEPROM OK"));    
  }else{
    Serial.println(F("EEPROM Error, defaults loaded and saved"));
#endif
  }
  pinMode(13,OUTPUT);                     //use D13 to indicate time spent in interrupt handler
  pinMode(PIEZO_PIN,OUTPUT);              //for tones
  for(int i=0;i<SAMPLECOUNT;i++){         //clear buffers
    mainsSamples[i]=0;
    batterySamples[i]=0;
    vinSamples[i]=0;
  }
  inverterSetup();                        //set IOs for inverter control
  relay8init();                           //init relay driver
  Timer1.initialize(1000);                //set for 1kHz sample rate = 20 samples per mains cycle
  Timer1.attachInterrupt(getSamples); 
  digitalWrite(MAINS_ISOLATE,LOW);        //pin for disconnecting transformer from VIN
  pinMode(MAINS_ISOLATE,OUTPUT);
  pinMode(VIN_SENSE,INPUT);
  pinMode(MAINS_SENSE,INPUT);
  iStatus=inverter(1);
#ifdef DEBUG
  Serial.print(F("Inverter turn on:")); 
  Serial.println(iStatus);                //turn on inverter at boot so it's ready
#endif
  while(millis()<100){}                   //wait to get some samples before running
}

void loop() {                             //loop cycles are ~1ms if no printing occurs
  float meanVin,meanBattery,mainsFromRMS,mainsFromPP;     //variables for testing status
  saveSamples();                                          //transfer samples from live to backup array
  mainsFromRMS=getMainsFromRMS();                         //get mains voltages
  mainsFromPP=getMainsFromPP();
  meanVin=getMeanVIN();                                   //get DC voltages
  meanBattery=getMeanBattery();
  if(apcMoutput<mainsFromRMS){apcMoutput=mainsFromRMS;}   //maximum voltage
  if(apcNoutput>mainsFromRMS){apcNoutput=mainsFromRMS;}   //minimum voltage
  if((mainsFromRMS<mains_min)||(mainsFromPP<mains_min)){  //if mains is low
#ifdef DEBUG
    if(mainsOK){
      Serial.println(mainsmin);
      Serial.println(mainsmax);
      Serial.println(mainsFromRMS);                       //debug data
      Serial.println(mainsFromPP);
      dumpMainsSamples();
      Serial.println(F("MIN reached"));
    }
#endif
    mainsOK=0;                                            //flag it
    mainsTimer=millis();                                  //set timer
    apcGoutput='L';                                       //output was low
  }
  if((mainsFromRMS>mains_max)||(mainsFromPP>mains_max)){  //if mains is high
#ifdef DEBUG
    if(mainsOK){
      Serial.println(mainsmin);
      Serial.println(mainsmax);
      Serial.println(mainsFromRMS);                       //debug data
      Serial.println(mainsFromPP);
      dumpMainsSamples();
      Serial.println(F("MAX reached"));
    }
#endif
    mainsOK=0;                                            //flag it
    mainsTimer=millis();                                  //set timer
    apcGoutput='H';                                       //output was high
  }
  if((mainsFromRMS>(mains_min+mains_db))&&(mainsFromPP>(mains_min+mains_db))&&(mainsFromRMS<(mains_max-mains_db))&&(mainsFromPP<(mains_max-mains_db))){
#ifdef DEBUG
    if(!mainsOK){                                         //mains is back in range
      Serial.println(mainsmin);
      Serial.println(mainsmax);
      Serial.println(mainsFromRMS);                       //debug data
      Serial.println(mainsFromPP);
      dumpMainsSamples();
      Serial.println(F("In Bounds"));
    }
#endif
    mainsOK=1;                                            //set flag
  }
  if(meanVin<vin_critical){                               //VIN low- VIN has been switched off
#ifdef DEBUG
    Serial.println(F("VIN low, VIN fail or powerdown"));
    Serial.println(meanVin);
#endif
    shutDown=2;                                           //set shutdown flag for state machine, 2 means now
  }
  if(apcKStrigger&&(millis()-apcKStime>battery_critical_delay)){// apc K or S command triggering shutdown
    shutDown=1;                                           //set shutdown flag for state machine    
    apcKStrigger=0;
  }
  outputStateMachine();                                   //state machine to change modes
  setDrivers();                                         
  if(!mainsOK){                                           //alert user, mains fault
      ledFlash|=GREEN_LED;
  }else{
      ledFlash&=~GREEN_LED;
  }  
  if(!inverterRun()){                                     //is inverter running? (it should be all the time)
    ledFlash|=AMBER_LED;                                  //alert user, inverter fault
  }else{
    ledFlash&=~AMBER_LED;        
  }
  if(meanBattery<battery_min){                            //battery below lower threshold
#ifdef APCPROTOCOL
    if(batteryOK){
      if(Serial.available()){parseAPC();}                 //avoid sending data if host has sent command
      Serial.println("%");                                //APC battery low message
    }
#endif    
    batteryOK=0;
  }
  if(meanBattery>battery_ok){                             //battery above upper threshold
#ifdef APCPROTOCOL
    if(!batteryOK){
      if(Serial.available()){parseAPC();}                 //avoid sending data if host has sent command
      Serial.println("+");                                //APC battery OK message
    }
#endif    
    batteryOK=1;
  }
  if(meanBattery>battery_critical){batteryCriticalTimer=millis();}      //reset timer
  flashRate(meanBattery,battery_critical,battery_ok);                   //set RED led flash rate based on status
  if(millis()-batteryCriticalTimer>battery_critical_delay){             //battery has been v.low for a while
    if(outputState==OUTPUT_ON_INVERTER){                                //batteries v.low and running on inverter => shut down
#ifdef DEBUG
      Serial.println(F("Battery Critical, can't run inverter"));
#endif
      shutDown=1;                                         //set shutdown flag for state machine
    }
  }
#ifdef DEBUG
  doMenu();
  if(debug){reportStatus();}                              //only outputs if interval is elapsed
#endif
#ifdef APCPROTOCOL
  parseAPC();                                             //parse any incoming commands
#endif
  if(millis()>2000){digitalWrite(MAINS_ISOLATE,HIGH);}    //switch out alternate transformer suppply after startup
}

void getSamples(){                                        //ISR for samples, uses ~34% CPU time
  digitalWrite(13,HIGH);                                  //flag time in ISR on digital pin                              
  mainsSamples[mainsSamplePtr++]=analogRead(MAINS_SENSE); //read mains
  mainsSamplePtr=mainsSamplePtr&(SAMPLECOUNT-1);
  if(samplephase){
    batterySamples[dcSamplePtr++]=analogRead(BATTERY_SENSE);//read battery
    dcSamplePtr=dcSamplePtr&(SAMPLECOUNT-1);
    samplephase=false;    
  }else{
    vinSamples[dcSamplePtr]=analogRead(VIN_SENSE);         //or VIN on alternate cycles
    samplephase=true;        
  }
  digitalWrite(13,LOW);                                   //leaving ISR
}

void outputStateMachine(){
  static unsigned long lineFailAlertTime=0;
#ifdef APCPROTOCOL
  if((outputState==OUTPUT_ON_INVERTER)&&(lineFailAlertTime!=millis()/30000)){
    lineFailAlertTime=millis()/30000;
    if(Serial.available()){parseAPC();}                   //avoid sending data if host has sent command
    Serial.println("!");                                  //line fail
    tone(PIEZO_PIN,PIEZOFREQ,200);                        //short tone on shutdown
  }
#endif
  if(shutDown){                                           //shutting down
#ifdef DEBUG
    Serial.println(F("Critical Fault, shutting down"));    
#endif
#ifdef APCPROTOCOL
    if(Serial.available()){parseAPC();}                   //avoid sending data if host has sent command
    Serial.println("*");                                  //APC shutdown message
#endif
    digitalWrite(MAINS_ISOLATE,LOW);                      //pin for disconnecting transformer from VIN, run from this to avoid boot loop if mains on
    if(shutDown!=2){                                      //skip delay on VIN fail, Z command
      for(int i=0;i<(SHUTDOWN_COMMAND_DELAY/3000);i++){   //give urgent warning for a minute, delay shutdown to allow things to happen gracefully
        tone(PIEZO_PIN,PIEZOFREQ,2000);                   //long tone on shutdown
        parseAPC();                                       //check if command received to restart
        delay(3000);                                      //let tone sound
      }
    }
    relayDriver=ALL_OFF;                                  //disconnect all
    ledState=ALL_OFF;                                     //LEDs off
    ledFlash=ALL_OFF;
    setDrivers();                                         //set
    inverter(0);                                          //shut down inverter
    while(1){                                             //stay shut down until reset
      parseAPC();                                         //or command received to start
    }
  }
  switch(outputState){                                    //based on state
    case(OUTPUT_OFF):
#ifdef DEBUG
      Serial.println(F("Starting- try mains"));           //boot message
#endif
      ledState=GREEN_LED|AMBER_LED;                       //switching outputs
      setDrivers();                                       //set LEDs
      outputState=OUTPUT_TO_MAINS;                        //next time we'll switch to mains
      break;
    case(OUTPUT_TO_MAINS):                                //transition to mains
      relayDriver=ALL_OFF;                                //disconnect all to prevent bridging
      setDrivers();      
      delay(10);                                          //wait
      relayDriver=MAINS_CONNECT;                          //connect mains supply
      ledState=GREEN_LED;                                 //running on mains
      setDrivers();      
#ifdef DEBUG
      Serial.println(F("Switched to mains"));
#endif
      outputState=OUTPUT_ON_MAINS;
#ifdef APCPROTOCOL
      if(Serial.available()){parseAPC();}                 //avoid sending data if host has sent command
      Serial.println("$");                                //recover from line fail
#endif
      if(shutDown){outputState=OUTPUT_OFF;}               //catch flag change
      break;
    case(OUTPUT_ON_MAINS):                                //running on mains
      if((!mainsOK)||(apcUtrigger)){                      //mains fault or simulate mains fault
        apcUtrigger=0;                                    //clear simulate flag
        if(inverter(1)){                                  //start inverter if not running
          outputState=OUTPUT_TO_INVERTER;                 //start transition
          ledState=GREEN_LED|AMBER_LED;                   //transition LEDs
          setDrivers();                      
        }else{
          ledState=GREEN_LED|RED_LED;                     //mains is not good, but inverter not running, show fault status, don't change yet
          setDrivers();                                
        }
      }      
      if(shutDown){outputState=OUTPUT_OFF;}               //catch flag change
      break;
    case(OUTPUT_TO_INVERTER):
      relayDriver=ALL_OFF;                                //disconnect all to prevent bridging
      setDrivers();      
      delay(10);                                          //wait
      relayDriver=INVERTER_CONNECT|INVERTER_CHANGEOVER;   //connect inverter and changeover relay
      ledState=AMBER_LED;                                 //running on inverter
      setDrivers();            
#ifdef DEBUG
      Serial.println(F("Switched to inverter"));
#endif
      outputState=OUTPUT_ON_INVERTER;
#ifdef APCPROTOCOL
      if(Serial.available()){parseAPC();}                 //avoid sending data if host has sent command
      Serial.println("!");                                //line fail
      lineFailAlertTime=millis()/30000;                   //timer
#endif
      if(shutDown){outputState=OUTPUT_OFF;}               //catch flag change
      break;
    case(OUTPUT_ON_INVERTER):                             //running on inverter
      if(mainsOK&&(millis()-mainsTimer>mains_delay)){     //mains stable for time
        outputState=OUTPUT_TO_MAINS;                      //start transition
        ledState=GREEN_LED|AMBER_LED;                     //transition status
        setDrivers();              
      }
      if(shutDown){outputState=OUTPUT_OFF;}               //catch flag change
      break;
    default:
      outputState=OUTPUT_OFF;                             //unknown state, return to known state
      break;    
  }
}

void relay8init(){
  Wire.begin();                                           // Wake up I2C bus
  Wire.beginTransmission(I2C_ADDR);
  Wire.write(0x00);                                       // IODIRA register
  Wire.write(0x00);                                       // Set all of bank A to outputs
  Wire.endTransmission();
  sendValueToLatch(0);                                    //all off
}

void sendValueToLatch(int latchValue){
  Wire.beginTransmission(I2C_ADDR);
  Wire.write(0x12);                                       // Select GPIOA
  Wire.write(latchValue);                                 // Send value to bank A
  Wire.endTransmission();
}

int inverter(int s){                                      //control inverter, report back
  int stat;
  stat=inverterRun();
  if(s==stat){return 2;}                                  //inverter already at that state
  if(s){                                                  //to turn on
    digitalWrite(INVERTER_CONTROL,HIGH);                  //pulse the trigger and wait until LED comes on
    for(int i=0;i<10;i++){
      delay(100);
      if(digitalRead(INVERTER_SENSE)==0){                 //it's on
        digitalWrite(INVERTER_CONTROL,LOW);               //drop output
        return 1;                                         //all good
      }
    }
    digitalWrite(INVERTER_CONTROL,LOW);                   //drop output
  }else{                                                  //to turn off
    digitalWrite(INVERTER_CONTROL,HIGH);                  //toggle high
    delay(100);    
    digitalWrite(INVERTER_CONTROL,LOW);                   //toggle low
    delay(100);                                           //wait to change state
    if(digitalRead(INVERTER_SENSE)==1){return 1;}         //it's off, all good
  }
  return 0;                                               //no response, fail
}

void inverterSetup(){                                     //set up IO pins
  digitalWrite(INVERTER_CONTROL,LOW);
  pinMode(INVERTER_CONTROL,OUTPUT);
  pinMode(INVERTER_SENSE,INPUT_PULLUP);  
}

int inverterRun(){                                        //check status
  return digitalRead(INVERTER_SENSE)?0:1;
}

void setDrivers(){                                        //combine status, LEDs and flashing
  if((millis()/500)&3){                                   //&3 gives 75%/25% duty cycle
    sendValueToLatch(ledState|relayDriver);
  }else{
    sendValueToLatch((ledState^ledFlash)|relayDriver);
  }  
}

int getMainsMin(){                                        //work out minimum of samples
  unsigned int a=1023;
  for(int i=0;i<SAMPLECOUNT;i++){
      if(mainsSampleBackup[i]<a){a=mainsSampleBackup[i];}
  }
  mainsmin=a;
  return a;
}

int getMainsMax(){                                        //work out maximum of samples
  unsigned int a=0;
  for(int i=0;i<SAMPLECOUNT;i++){
      if(mainsSampleBackup[i]>a){a=mainsSampleBackup[i];}
  }
  mainsmax=a;
  return a;
}

int getMainsMean(){                                       //work out mean (needed for RMS calculations)
  return MAINSZEROPOINT;     //nominal zero point for 10k/10k/75k network
  //long method taking average
  /*
  long a=0;
  for(int i=0;i<SAMPLECOUNT;i++){
    a=a+mainsSampleBackup[i];
  }
  mainsmean=a/SAMPLECOUNT;
  return a/SAMPLECOUNT;
  */
}

float getMeanVIN(){                                       //rolling average of VIN
  unsigned int a=0;
  for(int i=0;i<SAMPLECOUNT;i++){
    a=a+vinSamples[i];
  }
  return (a*vin_scale)/SAMPLECOUNT;  
}

float getMeanBattery(){                                   //rolling average of battery
  unsigned int a=0;
  for(int i=0;i<SAMPLECOUNT;i++){
    a=a+batterySamples[i];
  }
  return (a*battery_scale)/SAMPLECOUNT;  
}

void reportStatus(){                                      //debugging only
  static unsigned long t=millis()/REPORTINTERVAL;
  unsigned long newt;
  newt=millis()/REPORTINTERVAL;
  if(newt==t){return;}                                    //if we've ticked over to next multiple of REPORTINTERVAL
  t=newt;
  if(!debug){return;}
  Serial.print(F("\nMains from P-P:"));
  Serial.println(getMainsFromPP(),2);
  Serial.print(F("Mains from RMS:"));
  Serial.println(getMainsFromRMS(),2);
  Serial.print(F("Battery       :"));
  Serial.println(getMeanBattery(),2);
  Serial.print(F("VIN           :"));
  Serial.println(getMeanVIN(),2);  
}

float getMainsFromPP(){                                   //estimated RMS voltage from P-P reading
  float v;
  v=(getMainsMax()-getMainsMin())*mains_scale;            //find P-P voltage
  v=v*0.353553391;                                        //P-P=>RMS by multiplying by 1/(2sqrt(2))
  return v;
}

float getMainsFromRMS(){                                  //RMS calculation over sample period
  int m=getMainsMean();                                   //get mean
  long s=0;
  float r;
  for(int i=0;i<SAMPLECOUNT;i++){
    s=s+(mainsSampleBackup[i]-m)*(mainsSampleBackup[i]-m);//add up squares
  }
  r=s;                                                    //convert to float
  r=r/SAMPLECOUNT;                                        //average
  return sqrt(r)*mains_scale;                             //get actual value
}

int saveToEEPROM(){                                       //save values to EEPROM
  EEPROM.put(0,vin_scale);
  EEPROM.put(4,battery_scale);
  EEPROM.put(8,mains_scale);
  EEPROM.put(12,battery_critical);
  EEPROM.put(16,battery_min);
  EEPROM.put(20,battery_ok);
  EEPROM.put(24,vin_min);
  EEPROM.put(28,vin_ok);
  EEPROM.put(32,mains_min);
  EEPROM.put(36,mains_db);
  EEPROM.put(40,mains_max);
  EEPROM.put(44,mains_delay);
  EEPROM.put(48,vin_delay);
  EEPROM.put(52,battery_critical_delay);
  EEPROM.put(56,vin_critical);
}

int loadFromEEPROM(){                                       //load from EEPROM, returns 0 if defaults had to be loaded due to error
  readEEPROMid(checkid,strlen(eepromid));       //read id flag
  if(!strmatch(checkid,eepromid)){
    Serial.println(F("EEPROM signature not found, loading and saving defaults"));
    loadDefaults();                   //set values to default
    saveToEEPROM();                   //save these
    writeEEPROMid(eepromid);          //set id flag in EEPROM
  }
  EEPROM.get(0,vin_scale);
  EEPROM.get(4,battery_scale);
  EEPROM.get(8,mains_scale);
  EEPROM.get(12,battery_critical);
  EEPROM.get(16,battery_min);
  EEPROM.get(20,battery_ok);
  EEPROM.get(24,vin_min);
  EEPROM.get(28,vin_ok);
  EEPROM.get(32,mains_min);
  EEPROM.get(36,mains_db);
  EEPROM.get(40,mains_max);
  EEPROM.get(44,mains_delay);
  EEPROM.get(48,vin_delay);
  EEPROM.get(52,battery_critical_delay);
  EEPROM.get(56,vin_critical);
#ifdef DEBUG
  printCurrent();
#endif  
//if there's a problem with EEPROM values
  if(vin_scale<=0){loadDefaults();saveToEEPROM();return 0;}
  if(battery_scale<=0){loadDefaults();saveToEEPROM();return 0;}
  if(battery_critical<0){loadDefaults();saveToEEPROM();return 0;}
  if(battery_min<0){loadDefaults();saveToEEPROM();return 0;}
  if(battery_ok<battery_min){loadDefaults();saveToEEPROM();return 0;}
  if(vin_min<0){loadDefaults();saveToEEPROM();return 0;}
  if(vin_ok<vin_min){loadDefaults();saveToEEPROM();return 0;}
  if(mains_db<0){loadDefaults();saveToEEPROM();return 0;}
  if(mains_min<0){loadDefaults();saveToEEPROM();return 0;}
  if(mains_max<2*mains_db){loadDefaults();saveToEEPROM();return 0;}
  if(mains_delay<0){loadDefaults();saveToEEPROM();return 0;}
  if(vin_delay<0){loadDefaults();saveToEEPROM();return 0;}
  if(battery_critical_delay<0){loadDefaults();saveToEEPROM();return 0;}
  if(vin_critical<0){loadDefaults();saveToEEPROM();return 0;}
  return 1;
}

void loadDefaults(){                                    //for when EEPROM fails (or is blank)
  vin_scale=VIN_SCALE;
  battery_scale=BATTERY_SCALE;
  mains_scale=MAINS_SCALE;
  battery_critical=BATTERY_CRITICAL;
  battery_min=BATTERY_MIN;
  battery_ok=BATTERY_OK;
  vin_min=VIN_MIN;
  vin_ok=VIN_OK;
  mains_min=MAINS_MIN;
  mains_db=MAINS_DB;
  mains_max=MAINS_MAX;
  mains_delay=MAINS_DELAY;
  vin_delay=VIN_DELAY;
  battery_critical_delay=BATTERY_CRITICAL_DELAY;
  vin_critical=VIN_CRITICAL;
#ifdef DEBUG
  printCurrent();
  Serial.println(F("Loaded from defaults"));
#endif
  }

float getValue(int n){
  switch(n){
    case(0):return vin_scale;break;
    case(1):return battery_scale;break;
    case(2):return mains_scale;break;
    case(3):return battery_critical;break;
    case(4):return battery_min;break;
    case(5):return battery_ok;break;
    case(6):return vin_min;break;
    case(7):return vin_ok;break;
    case(8):return mains_min;break;
    case(9):return mains_db;break;
    case(10):return mains_max;break;
    case(11):return mains_delay;break;
    case(12):return vin_delay;break;
    case(13):return battery_critical_delay;break;
    case(14):return vin_critical;break;
  }  
}

void setValue(int n,float val){
  switch(n){
    case(0):vin_scale=val;break;
    case(1):battery_scale=val;break;
    case(2):mains_scale=val;break;
    case(3):battery_critical=val;break;
    case(4):battery_min=val;break;
    case(5):battery_ok=val;break;
    case(6):vin_min=val;break;
    case(7):vin_ok=val;break;
    case(8):mains_min=val;break;
    case(9):mains_db=val;break;
    case(10):mains_max=val;break;
    case(11):mains_delay=val;break;
    case(12):vin_delay=val;break;
    case(13):battery_critical_delay=val;break;
    case(14):vin_critical=val;break;
  }
}

void dumpMainsSamples(){                            //dump mains sample buffers
  Serial.println(F("---"));
  for(int i=0;i<SAMPLECOUNT;i++){
    Serial.println(mainsSampleBackup[i]);
  }
  Serial.println(F("---"));
}

void saveSamples(){                                 //save current buffer
  noInterrupts();                                   //interrupts off so we get a consistent set of samples
  for(int i=0;i<SAMPLECOUNT;i++){
      mainsSampleBackup[i]=mainsSamples[i];         //copy
  }
  interrupts();                                     //interrupts back on
}

void doMenu(){                                      //get serial input and process
  float num;
  while(Serial.available()){                        //no input => nothing to do
    int a=Serial.read();
    if(menuSet==-1){                                //not setting a parameter
      if((a>='A')&&(a<='O')){
        menuSet=a-'A';                              //start setting A-O
        Serial.println(F("Enter Value for:"));
        Serial.print(menuItems[menuSet]);        
        Serial.println(getValue(menuSet),7);
        a=0;
        debug=0;                                    //turn debug data off during setting
      }
      if(a=='~'){debug=!debug;}                     //toggle debug mode with ~
      if(a=='?'){printHelp();}                      //Print commands
      if(a=='s'){
        saveToEEPROM();
        Serial.println(F("Saved to EEPROM"));
      }
      if(a=='l'){
        loadFromEEPROM();
        Serial.println(F("Loaded from EEPROM"));
      }
      if(a=='d'){
        loadDefaults();
      }
      if(a=='p'){
        printCurrent();
      }
    }
    if(menuSet>-1){                                 //in the middle of setting a parameter
      if(a>31){                                     //ascii character
        if(strlen(nBuffer)<BUFFERLENGTH){
          nBuffer[strlen(nBuffer)+1]=0;
          nBuffer[strlen(nBuffer)]=a;               //add typed character to buffer           
          Serial.write(a);                          //echo
        }        
      }else{
        if(a==8){                                   //backspace
          if(strlen(nBuffer)>0){
            nBuffer[strlen(nBuffer)-1]=0;
            Serial.write(8);                        //echo backspace
            }
        }
        if(a==13){
          if(strlen(nBuffer)>0){                    //ignore first CR on systems that have line entry (eg Arduino SM)
            num=atof(nBuffer);                      //convert to float
            Serial.println(num,7);                  //display number
            if((num>0)||(num==0&&nBuffer[0]=='0'&&nBuffer[1]==0)){
              Serial.println(F("Valid"));           //check if valid number
              setValue(menuSet,num);                //and put in variable
            }else{
              Serial.println(F("Not Valid"));
            }              
            Serial.print(menuItems[menuSet]);        
            Serial.println(getValue(menuSet),7);
            nBuffer[0]=0;                           //empty string
            menuSet=-1;                             //ready for next
          }
        }
      }
    }
  }
}

void printHelp(){
  Serial.println(F("UPS SETUP"));
  Serial.println(F("?     This Help"));
  Serial.println(F("~     Toggle voltage status output on/off"));
  Serial.println(F("A-O   Enter parameter, followed by number and enter"));
  Serial.println(F("s     Save current to EEPROM"));
  Serial.println(F("l     Load from EEPROM"));
  Serial.println(F("d     Load from defaults"));
  Serial.println(F("p     Print current parameters"));
  Serial.println();
}

void printCurrent(){
  Serial.println(F("Current Values:"));
  Serial.print(F("A:"));Serial.print(menuItems[0]);Serial.println(vin_scale,7);
  Serial.print(F("B:"));Serial.print(menuItems[1]);Serial.println(battery_scale,7);
  Serial.print(F("C:"));Serial.print(menuItems[2]);Serial.println(mains_scale,7);
  Serial.print(F("D:"));Serial.print(menuItems[3]);Serial.println(battery_critical,7);
  Serial.print(F("E:"));Serial.print(menuItems[4]);Serial.println(battery_min,7);
  Serial.print(F("F:"));Serial.print(menuItems[5]);Serial.println(battery_ok,7);
  Serial.print(F("G:"));Serial.print(menuItems[6]);Serial.println(vin_min,7);
  Serial.print(F("H:"));Serial.print(menuItems[7]);Serial.println(vin_ok,7);
  Serial.print(F("I:"));Serial.print(menuItems[8]);Serial.println(mains_min,7);
  Serial.print(F("J:"));Serial.print(menuItems[9]);Serial.println(mains_db,7);
  Serial.print(F("K:"));Serial.print(menuItems[10]);Serial.println(mains_max,7);
  Serial.print(F("L:"));Serial.print(menuItems[11]);Serial.println(mains_delay,7);
  Serial.print(F("M:"));Serial.print(menuItems[12]);Serial.println(vin_delay,7);
  Serial.print(F("N:"));Serial.print(menuItems[13]);Serial.println(battery_critical_delay,7);
  Serial.print(F("O:"));Serial.print(menuItems[14]);Serial.println(vin_critical,7);
}

void flashRate(float in,float lower,float upper){
  static unsigned long t=millis();
  float val=map(in,lower,upper,MIN_HALFCYCLE,MAX_HALFCYCLE);//map to cycle time range
  if(val<MIN_HALFCYCLE){ledState|=RED_LED;return;}          //solid on below minimum
  if(val>MAX_HALFCYCLE){ledState&=~RED_LED;return;}         //solid off above maximum
  if(millis()-t>val){                                       //toggle on overflow
    ledState^=RED_LED;
    t=millis();
  }    
}

int getBatteryPC(){           //battery percent for 'f' command, done manually as map is implemented as long
  float p;                    //calculate with floats to maintain resolution
  float meanBattery=getMeanBattery();
  if(meanBattery<battery_critical){return 0;}
  if(meanBattery>battery_ok){return 100;}
  p=(meanBattery-battery_critical)*100/(battery_ok-battery_critical);       //no need to check for battery_critical=battery_ok (DIV/0), as this should be caught by eeprom loading  
  return p;
}

int parseAPC(){                               //buffer may be none or more characters, depending if multibyte command is given
  int n=Serial.available();
  int b=Serial.peek();
  if(n<1){return 0;}                          //nothing to process
  switch(b){
    //deal with non-implemented cases-ignore
    case(26):                                 //Capability string (not implemented)
    case('A'):                                //Front panel test
    case('C'):                                //Internal temperature (not implemented)
    case('D'):                                //Runtime calibration
    case('E'):                                //Automatic selftest intervals
    case('F'):                                //Line frequency, Hz
    case('I'):                                //Measure-UPS: Alarm enable
    case('J'):                                //Measure-UPS: Alarm status
    case('O'):                                //Output voltage
    case('P'):                                //Power load %
    case('R'):                                //Turn dumb
    case('W'):                                //Self test battery
    case('X'):                                //Self-test battery results
    case('b'):         //Firmware revision
    case('c'):         //UPS local id
    case('e'):         //Return threshold
    case('g'):         //Nominal battery voltage
    case('h'):         //Measure-UPS: Ambient humidity
    case('i'):         //Measure-UPS: Dry contacts
    case('j'):         //Estimated runtime
    case('k'):         //Alarm delay
    case('l'):         //Low transfer voltage
    case('m'):         //Manufacturing date
    case('n'):         //Serial number
    case('o'):         //On-battery voltage
    case('p'):         //Shutdown grace delay
    case('q'):         //Low battery warning
    case('r'):         //Wakeup delay
    case('s'):         //Sensitivity
    case('u'):         //Upper transfer voltage
    case('v'):         //Measure-UPS: Firmware
    case('t'):         //Measure-UPS: Ambient temperature
    case('x'):         //Last battery change
    case('y'):         //Copyright notice
    case('z'):         //Reset to factory settings
    case('-'):         //Capability cycle
    case('@'):         //Shutdown and return
    case('0'):         //Abort shutdown
    case('~'):         //Register #1
    case('\''):        //Register #2
    case('7'):         //Dip switch positions
    case('8'):         //Register #3
    case('9'):         //Line quality
    case('>'):         //Battery packs
    case('['):         //Measure-UPS: Upper temp limit
    case(']'):         //Measure-UPS: Lower temp limit
    case('{'):         //Measure-UPS: Upper humidity limit
    case('}'):         //Measure-UPS: Lower humidity limit
    case('^'):         //Run in bypass mode
    case('<'):         //Number of bad battery packs
    case('/'):         //Load current
    case('\\'):        //Apparent load power
    case(22):          //Output voltage selection
    case(12):          //Front panel language
    case('w'):         //Run time conservation
      Serial.println("NA");
      Serial.read();                          //purge-not implemented
      break;
    case(1):                                  //Model string
      Serial.println("Silicon Chip LiFePO UPS");
      Serial.read();                          //purge-done with command
      break;
    case(14):                                 //Turn on UPS- will only work if mains is available
      //force reset
      reboot();      
      Serial.read();                          //purge-done with command
      break;
    case('B'):                                //Battery voltage
      Serial.println(getMeanBattery(),2);
      Serial.read();                          //purge-done with command
      break;          
    case('G'):                                //Cause of transfer
      Serial.write(apcGoutput); 
      Serial.println();
      Serial.read();                          //purge-done with command
      break;          
    case('K'):                                //Shutdown with grace period
      apcKStrigger=1;
      apcKStime=millis();
      Serial.println("OK");
      Serial.read();                          //purge-done with command
      break;      
    case('L'):                                //Input line voltage
      Serial.println(getMainsFromRMS(),1);
      Serial.read();                          //purge-done with command
      break;          
    case('M'):                                //Maximum line voltage
      Serial.println(apcMoutput,1);
      apcMoutput=getMainsFromRMS();           //reset maximum
      Serial.read();                          //purge-done with command
      break;          
    case('N'):                                //Minimum line voltage
      Serial.println(apcNoutput,1);
      apcNoutput=getMainsFromRMS();           //reset minimum
      Serial.read();                          //purge-done with command
      break;          
    case('Q'):                                //Status flags      
      Serial.print(((outputState==OUTPUT_ON_MAINS)?0x08:0x10)|(batteryOK?0:0x40),HEX);  //binary data sent as ascii hex
      Serial.println();
      Serial.read();                          //purge-done with command
      break;          
    case('S'):                                //Soft shutdown
      if(outputState==OUTPUT_ON_INVERTER){    //only if on inverter
        apcKStrigger=1;
        apcKStime=millis();
        Serial.println("OK");
      }else{
        Serial.println("NA");
      }
      Serial.read();                          //purge-done with command
      apcGoutput='S';                         //last change was manual
      break;      
    case('U'):                                //Simulate power failure
      apcUtrigger=1;                          //set flag
      mainsTimer=millis();
      Serial.read();                          //purge-done with command
      break;      
    case('V'):                                //Firmware revision
      Serial.println("IWI");
      Serial.read();                          //purge-done with command
      break;      
    case('Y'):                                //Enter smart mode
      Serial.println("SM");
      Serial.read();                          //purge-done with command
      break;      
    case('Z'):                                //Shutdown immediately
      apcZcount++;
      if(apcZcount>1){
        shutDown=2;                           //trigger shutdown
        apcZcount=0;        
      }
      Serial.read();                          //purge-done with command
      break;      
    case('a'):                                //Protocol info (version.alerts.commands)
      Serial.println("3.!$%+*.\001BGKLMNQSUVYZaf");
      Serial.read();                          //purge-done with command
      break;          
    case('f'):         //Battery level
      int t;
      t=getBatteryPC();
      Serial.write(((t/100)%10)+'0');
      Serial.write(((t/10)%10)+'0');
      Serial.write((t%10)+'0');
      Serial.println();
      Serial.read();                          //purge-done with command
      break;          
    default: 
      Serial.read();                          //unknown command
      return -2;
  }
  return 0;             //all good
}

void reboot() {
  wdt_disable();
  wdt_enable(WDTO_15MS);
  while (1) {}
}

void writeEEPROMid(char* id){                       //write id to end of eeprom
  int i;
  unsigned int s=EEPROM.length();
  unsigned int n=strlen(id);
  if(n>s-2){return;}                //fail if it won't fit
  for(i=0;i<n+1;i++){
    EEPROM.write(s-n-1+i,id[i]);    
  }  
}

void readEEPROMid(char* id,unsigned int idsize){      //read idsize+null bytes into id buffer
  unsigned int i=0;
  byte b=1;                 //to start in while loop
  unsigned int s=EEPROM.length();
  unsigned int p=s-idsize-1;
  if(p<0){return;}
  while(b&&(p<s)){
    b=EEPROM.read(p);
    id[i]=b;
    i++;
    p++;
  }  
}

bool strmatch(char* x, char* y){   //true if strings match
  unsigned int n;
  n=strlen(x);
  if(n!=strlen(y)){return 0;}       //can't match if different lengths
  for(int i=0;i<n;i++){
    if(x[i]!=y[i]){return 0;}      //different at some point
  }
  return 1;                        //all match
}
